"""
STC API - High-level interface for Seigr Toolset Crypto

Provides convenient functions for encryption, hashing, and key derivation
using the complete STC architecture (CEL, PHE, CKE, DSF, PCF, STATE).

v0.2.0 - Enhanced with entropy amplification, multi-path hashing,
         persistence vector obfuscation, and self-auditing
"""

from typing import Union, Optional, Dict, Any, Tuple
import numpy as np
import json

from core.cel import ContinuousEntropyLattice, initialize_cel
from core.phe import ProbabilisticHashingEngine, create_phe
from core.cke import ContextualKeyEmergence, create_cke
from core.dsf import DataStateFolding, create_dsf
from core.pcf import PolymorphicCryptographicFlow, create_pcf
from core.state import StateManager, create_state_manager
from core.state.metadata_utils import (
    encrypt_metadata,
    decrypt_metadata,
    inject_decoy_vectors,
    extract_real_vector
)
from utils.tlv_format import (
    serialize_metadata_tlv,
    deserialize_metadata_tlv,
    detect_metadata_version,
    METADATA_VERSION_JSON,
    METADATA_VERSION_TLV
)

# Version constant
STC_VERSION = "0.2.0"


class STCContext:
    """
    Complete STC context managing all cryptographic components
    
    This is the main interface for using STC functionality.
    """
    
    def __init__(
        self,
        seed: Union[str, bytes, int],
        lattice_size: int = 128,  # Reduced from 256 for better performance
        depth: int = 6,  # Reduced from 8 for better performance
        morph_interval: int = 100
    ):
        """
        Initialize STC context
        
        Args:
            seed: Seed for deterministic initialization
            lattice_size: CEL lattice dimension (default 128, was 256)
            depth: CEL lattice depth (default 6, was 8)
            morph_interval: PCF morph interval
        """
        self.seed = seed
        
        # Initialize all components
        self.cel = initialize_cel(seed, lattice_size, depth)
        self.phe = create_phe()
        self.cke = create_cke()
        self.dsf = create_dsf()
        self.pcf = create_pcf(morph_interval)
        self.state_manager = create_state_manager()
        
        # Sync components
        self._sync_components()
        
    def _sync_components(self) -> None:
        """Synchronize all components"""
        # Get CEL snapshot
        cel_snapshot = self.cel.snapshot()
        
        # Bind to PHE and PCF
        self.phe.map_entropy(cel_snapshot)
        self.pcf.bind(cel_snapshot)
    
    def encrypt(
        self,
        data: Union[str, bytes],
        context_data: Optional[Dict[str, Any]] = None,
        password: Optional[str] = None,
        use_decoys: bool = True,  # Now enabled by default in v0.2.1!
        num_decoys: int = 3
    ) -> Tuple[bytes, Union[bytes, Dict[str, Any]]]:
        """
        Encrypt data using complete STC pipeline (v0.2.1)
        Enhanced with varint compression and decoy vector support
        
        Args:
            data: Data to encrypt
            context_data: Optional additional context
            password: Password for metadata encryption (if None, uses seed)
            use_decoys: Whether to inject decoy vectors (NOW WORKING in v0.2.1!)
            num_decoys: Number of decoy snapshots (3-5)
            
        Returns:
            Tuple of (encrypted_bytes, metadata_bytes)
        """
        # Convert string to bytes
        if isinstance(data, str):
            data_bytes = data.encode('utf-8')
            original_length = len(data_bytes)
            was_string = True
        else:
            data_bytes = data
            original_length = len(data_bytes)
            was_string = False
        
        # Use seed as password if not provided
        if password is None:
            if isinstance(self.seed, str):
                password = self.seed
            elif isinstance(self.seed, bytes):
                password = self.seed.decode('utf-8', errors='replace')
            else:
                password = str(self.seed)
        
        # Update CEL with operation context
        operation_context = {
            'data': data_bytes,
            'operation': 'encrypt',
            'parameters': context_data or {}
        }
        self.cel.update(operation_context)
        
        # Get fresh CEL snapshot
        cel_snapshot = self.cel.snapshot()
        
        # Generate probabilistic hash of data
        phe_output = self.phe.digest(data_bytes, context_data)
        
        # Derive encryption key
        cke_context = {
            'cel_snapshot': cel_snapshot,
            'phe_output': phe_output,
            'operation': 'encrypt',
            'seed': self.seed,
            'data_size': len(data_bytes)
        }
        key_vector = self.cke.derive(cke_context)
        
        # Encrypt using DSF
        encrypted = self.dsf.fold(data_bytes, key_vector, cel_snapshot)
        
        # Cycle PCF
        self.pcf.cycle()
        
        # Discard ephemeral key
        self.cke.discard()
        
        # Prepare metadata with v0.2.0 enhancements
        metadata = {
            'original_length': original_length,
            'was_string': was_string,
            'phe_hash': phe_output,
            'cel_snapshot': cel_snapshot,
            'stc_version': STC_VERSION
        }
        
        # Encrypt metadata with differential encoding
        encrypted_metadata = encrypt_metadata(
            metadata,
            password,
            use_differential=False,  # Disabled: produces more data than full lattice
            seed=self.seed
        )
        
        # Inject decoy vectors if requested
        if use_decoys:
            encrypted_metadata = inject_decoy_vectors(
                encrypted_metadata,
                password,
                num_decoys=num_decoys
            )
        
        # Serialize to TLV binary format
        if use_decoys:
            # Store obfuscated structure
            metadata_bytes = serialize_metadata_tlv({
                'obfuscated': True,
                'vectors': encrypted_metadata['vectors'],
                'num_vectors': encrypted_metadata['num_vectors']
            })
        else:
            metadata_bytes = serialize_metadata_tlv(encrypted_metadata)
        
        return encrypted, metadata_bytes
    
    def decrypt(
        self,
        encrypted_data: bytes,
        metadata: Union[Dict[str, Any], bytes],
        context_data: Optional[Dict[str, Any]] = None,
        password: Optional[str] = None
    ) -> Union[str, bytes]:
        """
        Decrypt data using STC pipeline (v0.2.0)
        Enhanced with metadata decryption and version detection
        
        Args:
            encrypted_data: Encrypted bytes
            metadata: Metadata from encryption (dict or TLV bytes)
            context_data: Optional additional context
            password: Password for metadata decryption (if None, uses seed)
            
        Returns:
            Decrypted data (string or bytes based on metadata)
            
        Raises:
            ValueError: If metadata version is incompatible
        """
        # Use seed as password if not provided
        if password is None:
            if isinstance(self.seed, str):
                password = self.seed
            elif isinstance(self.seed, bytes):
                password = self.seed.decode('utf-8', errors='replace')
            else:
                password = str(self.seed)
        
        # Detect and handle metadata version
        if isinstance(metadata, bytes):
            version = detect_metadata_version(metadata)
            
            if version == METADATA_VERSION_JSON:
                # v0.1.x JSON format
                raise ValueError(
                    "This data was encrypted with STC v0.1.x. "
                    "Please use the migration utility to convert to v0.2.0 format."
                )
            
            # Deserialize TLV
            metadata_dict = deserialize_metadata_tlv(metadata)
            
            # Check if obfuscated with decoys
            if metadata_dict.get('obfuscated'):
                # Extract real vector from decoys
                obfuscated_data = {
                    'vectors': metadata_dict['vectors'],
                    'num_vectors': metadata_dict['num_vectors']
                }
                encrypted_metadata = extract_real_vector(obfuscated_data, password)
                # Decrypt metadata
                metadata = decrypt_metadata(encrypted_metadata, password, seed=self.seed)
            elif 'encrypted_metadata' in metadata_dict:
                # Metadata is encrypted (not obfuscated)
                metadata = decrypt_metadata(metadata_dict, password, seed=self.seed)
            else:
                # Metadata is already in plain form (backward compat or direct TLV)
                metadata = metadata_dict
        
        elif isinstance(metadata, dict):
            # Check for v0.1.x plain dict (has 'phe_hash' as hex string)
            if isinstance(metadata.get('phe_hash'), str):
                raise ValueError(
                    "This data was encrypted with STC v0.1.x. "
                    "Please use the migration utility to convert to v0.2.0 format."
                )
            # Already decrypted metadata (v0.2.0)
            pass
        
        # Reconstruct PHE hash
        phe_output = metadata['phe_hash']
        if isinstance(phe_output, str):
            phe_output = bytes.fromhex(phe_output)
        
        # Use embedded CEL snapshot from encryption for exact reconstruction
        cel_snapshot = metadata['cel_snapshot']
        
        # Derive decryption key (same process as encryption)
        cke_context = {
            'cel_snapshot': cel_snapshot,
            'phe_output': phe_output,
            'operation': 'encrypt',  # Use same operation for key derivation
            'seed': self.seed,
            'data_size': metadata['original_length']
        }
        key_vector = self.cke.derive(cke_context)
        
        # Decrypt using DSF
        decrypted = self.dsf.unfold(
            encrypted_data,
            key_vector,
            cel_snapshot,
            original_length=metadata['original_length']
        )
        
        # Discard ephemeral key
        self.cke.discard()
        
        # Convert back to string if original was string
        if metadata.get('was_string', False):
            return decrypted.decode('utf-8')
        
        return decrypted
    
    def hash(
        self,
        data: Union[str, bytes],
        context_data: Optional[Dict[str, Any]] = None
    ) -> bytes:
        """
        Generate probabilistic hash
        
        Args:
            data: Data to hash
            context_data: Optional context
            
        Returns:
            Hash bytes
        """
        # Update CEL
        if isinstance(data, str):
            data_bytes = data.encode('utf-8')
        else:
            data_bytes = data
        
        operation_context = {
            'data': data_bytes,
            'operation': 'hash',
            'parameters': context_data or {}
        }
        self.cel.update(operation_context)
        
        # Sync PHE
        cel_snapshot = self.cel.snapshot()
        self.phe.map_entropy(cel_snapshot)
        
        # Generate hash
        hash_result = self.phe.digest(data_bytes, context_data)
        
        # Cycle PCF
        self.pcf.cycle()
        
        return hash_result
    
    def derive_key(
        self,
        length: int = 32,
        context_data: Optional[Dict[str, Any]] = None
    ) -> bytes:
        """
        Derive ephemeral key
        
        Args:
            length: Key length in bytes
            context_data: Optional context
            
        Returns:
            Derived key bytes
        """
        # Update CEL
        operation_context = {
            'operation': 'derive_key',
            'parameters': context_data or {}
        }
        self.cel.update(operation_context)
        
        # Get CEL snapshot
        cel_snapshot = self.cel.snapshot()
        
        # Derive key
        cke_context = {
            'cel_snapshot': cel_snapshot,
            'operation': 'derive_key',
            'seed': self.seed,
        }
        if context_data:
            cke_context.update(context_data)
        
        key_vector = self.cke.derive(cke_context, key_length=length)
        key_bytes = bytes(key_vector)
        
        # Discard after extraction
        self.cke.discard()
        
        # Cycle PCF
        self.pcf.cycle()
        
        return key_bytes
    
    def save_state(self, filepath: Optional[str] = None) -> Dict[str, Any]:
        """
        Save complete context state
        
        Args:
            filepath: Optional file path to save to
            
        Returns:
            State dictionary
        """
        state = self.state_manager.save({
            'cel': self.cel,
            'pcf': self.pcf,
            'seed': self.seed,
            'metadata': {
                'lattice_size': self.cel.lattice_size,
                'depth': self.cel.depth,
                'morph_interval': self.pcf.morph_interval,
            }
        })
        
        if filepath:
            self.state_manager.save_to_file(state, filepath)
        
        return state
    
    def get_status(self) -> str:
        """
        Get human-readable status
        
        Returns:
            Status string
        """
        lines = [
            "=== STC Context Status ===",
            "",
            f"CEL Operation Count: {self.cel.operation_count}",
            f"CEL State Version: {self.cel.state_version}",
            f"PHE Operation Count: {self.phe.operation_count}",
            f"CKE Derivation Count: {self.cke.derivation_count}",
            f"DSF Operation Count: {self.dsf.operation_count}",
            "",
            "PCF Status:",
            self.pcf.describe(),
        ]
        
        return "\n".join(lines)


# Convenience functions

def initialize(
    seed: Union[str, bytes, int],
    lattice_size: int = 256,
    depth: int = 8,
    morph_interval: int = 100
) -> STCContext:
    """
    Initialize STC context
    
    Args:
        seed: Seed for initialization
        lattice_size: CEL lattice size
        depth: CEL depth
        morph_interval: PCF morph interval
        
    Returns:
        Initialized STCContext
    """
    return STCContext(seed, lattice_size, depth, morph_interval)


def encrypt(
    data: Union[str, bytes],
    context: STCContext,
    context_data: Optional[Dict[str, Any]] = None
) -> Tuple[bytes, Dict[str, Any]]:
    """
    Encrypt data
    
    Args:
        data: Data to encrypt
        context: STC context
        context_data: Optional additional context
        
    Returns:
        Tuple of (encrypted_bytes, metadata)
    """
    return context.encrypt(data, context_data)


def decrypt(
    encrypted_data: bytes,
    metadata: Dict[str, Any],
    context: STCContext,
    context_data: Optional[Dict[str, Any]] = None
) -> Union[str, bytes]:
    """
    Decrypt data
    
    Args:
        encrypted_data: Encrypted bytes
        metadata: Metadata from encryption
        context: STC context
        context_data: Optional additional context
        
    Returns:
        Decrypted data
    """
    return context.decrypt(encrypted_data, metadata, context_data)


def hash_data(
    data: Union[str, bytes],
    context: STCContext,
    context_data: Optional[Dict[str, Any]] = None
) -> bytes:
    """
    Generate probabilistic hash
    
    Args:
        data: Data to hash
        context: STC context
        context_data: Optional context
        
    Returns:
        Hash bytes
    """
    return context.hash(data, context_data)


def quick_encrypt(data: Union[str, bytes], seed: Union[str, bytes, int]) -> Tuple[bytes, Dict[str, Any], STCContext]:
    """
    Quick encryption with new context
    
    Args:
        data: Data to encrypt
        seed: Seed for context
        
    Returns:
        Tuple of (encrypted_bytes, metadata, context)
    """
    context = initialize(seed)
    encrypted, metadata = context.encrypt(data)
    return encrypted, metadata, context


def quick_decrypt(
    encrypted_data: bytes,
    metadata: Dict[str, Any],
    seed: Union[str, bytes, int]
) -> Union[str, bytes]:
    """
    Quick decryption with new context
    
    Reconstructs CEL state from embedded snapshot in metadata, enabling
    deterministic decryption without manual state tracking.
    
    Per DSF instructions: "quick_decrypt() must regenerate the CEL lattice 
    deterministically using CEL snapshot embedded in ciphertext"
    
    Args:
        encrypted_data: Encrypted bytes
        metadata: Metadata from encryption (includes cel_snapshot)
        seed: Seed (must match encryption seed)
        
    Returns:
        Decrypted data
    """
    context = initialize(seed)
    
    # The metadata now contains the full CEL snapshot from encryption
    # No need to fast-forward - we use the exact state from encryption
    # This ensures deterministic reconstruction per DSF requirements
    
    return context.decrypt(encrypted_data, metadata)
