"""
Example: Password manager using STC v0.2.0
Demonstrates password-based credential storage with MAC verification
"""

import sys
import os
# Add parent directory to path for imports
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '../..')))

import json
from datetime import datetime
from interfaces.api.stc_api import STCContext


class PasswordManager:
    """
    Simple password manager using STC v0.2.0 with password-based encryption
    Features MAC verification and encrypted metadata
    """
    
    def __init__(self, master_password: str):
        """
        Initialize password manager
        
        Args:
            master_password: Master password for encryption/decryption
        """
        self.master_password = master_password
        self.vault = {}
        # Use a unique seed derived from master password for additional entropy
        self.context = STCContext(f"vault-{master_password}")
    
    def add_credential(self, service: str, username: str, password: str):
        """
        Add credential to vault
        
        Args:
            service: Service name
            username: Username
            password: Password
        """
        # Create credential entry
        credential = {
            'service': service,
            'username': username,
            'password': password,
            'created': datetime.now().isoformat(),
            'modified': datetime.now().isoformat()
        }
        
        # Convert to JSON
        credential_json = json.dumps(credential)
        
        # Encrypt with password-based encryption (v0.2.0)
        # Encrypt password
        encrypted, metadata = self.context.encrypt(
            password,
            password=self.master_password
            # use_decoys=True by default in v0.2.1 (decoys now supported!)
        )
        
        # Store in vault (both encrypted and metadata are bytes)
        self.vault[service] = {
            'encrypted': encrypted.hex(),  # Store as hex string for JSON compatibility
            'metadata': metadata.hex()      # Metadata is binary TLV format
        }
        
        print(f"[OK] Credential for '{service}' added (encrypted with MAC)")
    
    def get_credential(self, service: str) -> dict:
        """
        Retrieve credential from vault
        
        Args:
            service: Service name
            
        Returns:
            Credential dictionary
            
        Raises:
            ValueError: If service not found or MAC verification fails
        """
        if service not in self.vault:
            raise ValueError(f"No credential found for '{service}'")
        
        # Get encrypted data
        entry = self.vault[service]
        encrypted = bytes.fromhex(entry['encrypted'])
        metadata = bytes.fromhex(entry['metadata'])
        
        # Decrypt with MAC verification (will raise ValueError if wrong password)
        try:
            credential_json = self.context.decrypt(
                encrypted, 
                metadata, 
                password=self.master_password
            )
        except ValueError as e:
            raise ValueError(f"Failed to decrypt '{service}': {e}")
        
        # Parse JSON
        credential = json.loads(credential_json)
        
        return credential
    
    def display_credential(self, service: str, show_password: bool = False):
        """
        Display credential (with password masked by default)
        
        Args:
            service: Service name
            show_password: Whether to show actual password (default: False)
        """
        credential = self.get_credential(service)
        print(f"\nCredential for '{credential['service']}':")
        print(f"  Username: {credential['username']}")
        # Always mask password unless explicitly requested
        masked_password = '*' * 12  # Fixed length to avoid leaking password length
        print(f"  Password: {masked_password}")
        print(f"  Created: {credential['created']}")
    
    def list_services(self):
        """List all stored services"""
        return list(self.vault.keys())
    
    def save_vault(self, filepath: str):
        """
        Save vault to file (vault is already in serializable format)
        
        Args:
            filepath: File path
        """
        with open(filepath, 'w') as f:
            json.dump(self.vault, f, indent=2)
        print(f"[OK] Vault saved to {filepath}")
    
    def load_vault(self, filepath: str):
        """
        Load vault from file
        
        Args:
            filepath: File path
        """
        with open(filepath, 'r') as f:
            self.vault = json.load(f)
        print(f"[OK] Vault loaded from {filepath} ({len(self.vault)} credentials)")


if __name__ == '__main__':
    print("=" * 70)
    print("STC v0.2.0 Password Manager Example")
    print("=" * 70)
    
    # Create password manager
    # NOTE: In production, use a strong master password and secure input method
    master_password = "example-master-password-change-in-production"
    pm = PasswordManager(master_password)
    
    # Add credentials (using example data - not real credentials)
    print("\nAdding example credentials...")
    pm.add_credential('github.com', 'user@example.com', 'github-pass-123')
    pm.add_credential('aws.amazon.com', 'admin', 'aws-key-456')
    pm.add_credential('database', 'db_admin', 'db-pass-789')
    
    print(f"\nStored services: {pm.list_services()}")
    
    # Retrieve and display credential (password masked by default)
    print("\nRetrieving GitHub credential (password masked):")
    pm.display_credential('github.com')
    
    # Show that password can be retrieved when needed (but not logged)
    github_cred = pm.get_credential('github.com')
    print(f"\n[OK] Password retrieved successfully (length: {len(github_cred['password'])} chars)")
    
    # Save vault
    print("\nSaving vault to encrypted storage...")
    pm.save_vault('password_vault.json')
    
    # Create new manager and load vault
    print("\nCreating new password manager and loading vault...")
    pm2 = PasswordManager(master_password)
    pm2.load_vault('password_vault.json')
    
    # Verify decryption works
    print("\nVerifying vault integrity (MAC verification):")
    for service in pm2.list_services():
        cred = pm2.get_credential(service)
        print(f"  [OK] {service}: {cred['username']} (MAC verified)")
    
    # Test wrong password protection
    print("\nTesting wrong password protection...")
    try:
        wrong_pm = PasswordManager("wrong-password")
        wrong_pm.load_vault('password_vault.json')
        wrong_pm.get_credential('github.com')
        print("  [FAIL] SECURITY FAILURE: Wrong password accepted!")
    except ValueError as e:
        print(f"  [OK] Wrong password correctly rejected (MAC verification)")
    
    print("\n" + "=" * 70)
    print("[OK] Password manager working correctly!")
    print("[OK] All credentials encrypted with MAC protection!")
    print("=" * 70)
    
    print("\nSecurity Features (v0.2.0):")
    print("  [OK] Password-based encryption with MAC verification")
    print("  [OK] Metadata encrypted with ephemeral keys")
    print("  [OK] Binary TLV format for compact storage")
    print("  [OK] Tamper detection via MAC")
    print("  [OK] Wrong password rejection")
    
    print("\nSecurity Best Practices:")
    print("  - Your master password is the ONLY key - memorize it securely")
    print("  - No cloud services, no third parties - complete data sovereignty")
    print("  - Backup your vault file separately from your master password")
    print("  - Set proper file permissions on vault file (chmod 600 on Unix/Linux)")
    print("  - If you lose your master password, your data is unrecoverable")
    print("  - This is the tradeoff for true cryptographic sovereignty")
