"""
Example: Configuration file encryption using STC v0.2.0
Demonstrates password-based config encryption with MAC verification
"""

import sys
import os
# Add parent directory to path for imports
sys.path.insert(0, os.path.abspath(os.path.join(os.path.dirname(__file__), '../..')))

import json
from interfaces.api.stc_api import STCContext


def encrypt_config_file(config_data: dict, password: str, output_file: str):
    """
    Encrypt configuration data with password protection (v0.2.0)
    
    Args:
        config_data: Configuration dictionary
        password: Encryption password
        output_file: Output file path
    """
    # Convert config to JSON
    config_json = json.dumps(config_data, indent=2)
    
    # Initialize STC context
    context = STCContext(f"config-{password}")
    
    # Encrypt with password-based encryption
    encrypted, metadata = context.encrypt(
        config_json,
        password=password
        # use_decoys=True by default in v0.2.1
    )
    
    # Save encrypted data (binary)
    with open(output_file, 'wb') as f:
        f.write(encrypted)
    
    # Save metadata (binary TLV format)
    metadata_file = output_file + '.meta'
    with open(metadata_file, 'wb') as f:
        f.write(metadata)
    
    print(f"[OK] Configuration encrypted and saved to {output_file}")
    print(f"[OK] Metadata (with MAC) saved to {metadata_file}")
    print(f"  Encrypted size: {len(encrypted)} bytes")
    print(f"  Metadata size: {len(metadata)} bytes")


def decrypt_config_file(encrypted_file: str, password: str) -> dict:
    """
    Decrypt configuration file with MAC verification (v0.2.0)
    
    Args:
        encrypted_file: Encrypted file path
        password: Decryption password
        
    Returns:
        Configuration dictionary
        
    Raises:
        ValueError: If wrong password or tampering detected
    """
    # Load encrypted data
    with open(encrypted_file, 'rb') as f:
        encrypted = f.read()
    
    # Load metadata
    metadata_file = encrypted_file + '.meta'
    with open(metadata_file, 'rb') as f:
        metadata = f.read()
    
    # Initialize context with same seed pattern
    context = STCContext(f"config-{password}")
    
    # Decrypt with MAC verification
    config_json = context.decrypt(encrypted, metadata, password=password)
    
    # Parse JSON
    config_data = json.loads(config_json)
    
    return config_data


if __name__ == '__main__':
    print("=" * 70)
    print("STC v0.2.0 Configuration Encryption Example")
    print("=" * 70)
    
    # Example configuration (using placeholder values - not real credentials)
    config = {
        'database': {
            'host': 'localhost',
            'port': 5432,
            'username': 'admin',
            'password': 'example-db-password'  # Placeholder
        },
        'api_keys': {
            'service_a': 'sk-aaaaa-example-key-aaaaa',  # Placeholder
            'service_b': 'sk-bbbbb-example-key-bbbbb'   # Placeholder
        },
        'settings': {
            'debug': False,
            'max_connections': 100,
            'timeout': 30
        }
    }
    
    # Encryption password (should be stored securely)
    password = "example-config-password-change-in-production"
    
    # Encrypt configuration
    print("\nEncrypting configuration with password protection...")
    encrypt_config_file(config, password, 'config.enc')
    
    # Decrypt configuration
    print("\nDecrypting configuration (with MAC verification)...")
    try:
        decrypted_config = decrypt_config_file('config.enc', password)
        print("[OK] MAC verification passed")
    except ValueError as e:
        print(f"[FAIL] Decryption failed: {e}")
        exit(1)
    
    # Verify decryption (without logging sensitive data)
    print("\nVerifying decrypted configuration:")
    print(f"  [OK] Database host: {decrypted_config['database']['host']}")
    print(f"  [OK] Database port: {decrypted_config['database']['port']}")
    print(f"  [OK] Database username: {decrypted_config['database']['username']}")
    print(f"  [OK] Database password: {'*' * 16} (decrypted successfully)")
    print(f"  [OK] API keys: {len(decrypted_config['api_keys'])} keys loaded")
    print(f"  [OK] Settings: {len(decrypted_config['settings'])} settings loaded")
    
    # Verify integrity
    assert config == decrypted_config
    print("\n[OK] Configuration matches original perfectly")
    
    # Test wrong password protection
    print("\nTesting wrong password protection...")
    try:
        decrypt_config_file('config.enc', 'wrong-password')
        print("[FAIL] SECURITY FAILURE: Wrong password accepted!")
    except ValueError:
        print("[OK] Wrong password correctly rejected (MAC verification)")
    
    # Test tampering detection
    print("\nTesting tamper detection...")
    try:
        # Load and tamper with encrypted data
        with open('config.enc', 'rb') as f:
            tampered = bytearray(f.read())
        tampered[10] ^= 0xFF  # Flip bits
        
        with open('config.enc.tampered', 'wb') as f:
            f.write(tampered)
        
        # Copy metadata
        with open('config.enc.meta', 'rb') as f:
            meta = f.read()
        with open('config.enc.tampered.meta', 'wb') as f:
            f.write(meta)
        
        # Try to decrypt tampered data
        decrypt_config_file('config.enc.tampered', password)
        print("[FAIL] SECURITY FAILURE: Tampering not detected!")
    except (ValueError, Exception):
        print("[OK] Tampering correctly detected")
    
    print("\n" + "=" * 70)
    print("[OK] Configuration encryption working correctly!")
    print("=" * 70)
    
    print("\nSecurity Features (v0.2.0):")
    print("  [OK] Password-based encryption")
    print("  [OK] MAC verification prevents tampering")
    print("  [OK] Metadata encrypted with ephemeral keys")
    print("  [OK] Wrong password rejection")
    print("  [OK] Binary TLV format for compact storage")
    
    print("\nSecurity Best Practices:")
    print("  - Store your password securely (environment variable, key vault, etc.)")
    print("  - Set proper file permissions (chmod 600 on Unix/Linux)")
    print("  - Keep backups of both encrypted files and password (separately)")
    print("  - No cloud services required - complete data sovereignty")
    print("  - If you lose your password, your data is unrecoverable")
