# Usage Guide - STC v0.2.0

Complete guide for using STC v0.2.0 with password-based encryption, MAC verification, and binary TLV format.

## Quick Start

### Password-Based Encryption (Recommended)

```python
from interfaces.api.stc_api import STCContext

# Initialize context with seed
ctx = STCContext('my-unique-seed')

# Encrypt with password (includes MAC for tamper detection)
encrypted, metadata = ctx.encrypt(
    "Secret message",
    password="strong_password"
)

# Decrypt with password (MAC automatically verified)
try:
    decrypted = ctx.decrypt(encrypted, metadata, password="strong_password")
    print(decrypted)  # "Secret message"
except ValueError:
    print("Wrong password or data tampered!")
```

### Basic Encryption (No Password)

```python
from interfaces.api.stc_api import STCContext

# Initialize context (seed used as password)
ctx = STCContext('my-seed')

# Encrypt without explicit password
encrypted, metadata = ctx.encrypt("Sensitive data")

# Decrypt (seed used as password)
decrypted = ctx.decrypt(encrypted, metadata)
```

## Advanced Features

### Custom Lattice Parameters

```python
# Smaller lattice = faster but less security
ctx = STCContext(
    seed="my-seed",
    lattice_size=64,   # Default: 128 (range: 16-256)
    depth=4,           # Default: 6 (range: 2-8)
    morph_interval=50  # Default: 100 (PCF morphing frequency)
)

# Encrypt with custom parameters
encrypted, metadata = ctx.encrypt("data", password="pw")
```

**Performance vs Security Trade-off:**
- `lattice_size=64, depth=4`: ~0.5s encryption, 196KB metadata (fast, moderate security)
- `lattice_size=128, depth=6`: ~2.3s encryption, 786KB metadata (balanced - **default**)
- `lattice_size=256, depth=8`: ~10s encryption, 4MB metadata (slow, maximum security)

### Binary Data Encryption

```python
# Works with bytes
binary_data = b'\x00\x01\x02\xFF\xFE\xFD'
encrypted, metadata = ctx.encrypt(binary_data, password="pw")
decrypted = ctx.decrypt(encrypted, metadata, password="pw")

assert decrypted == binary_data
```

### Quick API (One-Liners)

```python
from interfaces.api import stc_api

# Quick encrypt
encrypted, metadata, context = stc_api.quick_encrypt(
    "sensitive data",
    seed="my-seed"
)

# Quick decrypt
decrypted = stc_api.quick_decrypt(
    encrypted,
    metadata,
    seed="my-seed"
)
```

## Probabilistic Hashing

### Basic Hashing

```python
ctx = STCContext('hash-seed')

# Each hash is different due to CEL evolution
hash1 = ctx.hash("same data")
hash2 = ctx.hash("same data")

assert hash1 != hash2  # Different hashes (probabilistic)
assert len(hash1) == 32  # 32-byte output
```

### Key Derivation

```python
ctx = STCContext('my-seed')

# Derive encryption keys
key_32 = ctx.derive_key(length=32)  # 32 bytes
key_64 = ctx.derive_key(length=64)  # 64 bytes

# Each call produces different key (CEL evolves)
key2 = ctx.derive_key(length=32)
assert key_32 != key2
```

## State Management

### Save/Load Context State

```python
ctx = STCContext('my-seed')

# Perform operations
ctx.hash("data1")
ctx.hash("data2")

# Save state
state = ctx.save_state()

# State contains CEL, PHE, PCF snapshots
print(f"State keys: {state.keys()}")

# Load state later
ctx2 = STCContext('my-seed')
ctx2.load_state(state)
```

### Context Status

```python
status = ctx.get_status()
print(status)
# Output:
# === STC Context Status ===
# CEL Operation Count: 10
# CEL State Version: 5
# CEL Lattice Size: 128x128x6
# PCF Status: Active
# PCF Morph Interval: 100
# ==========================
```

## Persistent Storage

### Saving Encrypted Files (v0.2.0)

```python
import json

# Encrypt data
encrypted, metadata = ctx.encrypt("sensitive data", password="pw")

# Save encrypted data (binary)
with open('data.enc', 'wb') as f:
    f.write(encrypted)

# Save metadata (binary TLV format)
with open('data.enc.meta', 'wb') as f:
    f.write(metadata)

print(f"Encrypted: {len(encrypted)} bytes")
print(f"Metadata: {len(metadata)} bytes (~786KB)")
```

### Loading Encrypted Files

```python
# Load files
with open('data.enc', 'rb') as f:
    encrypted = f.read()

with open('data.enc.meta', 'rb') as f:
    metadata = f.read()

# Decrypt
ctx = STCContext('my-seed')
decrypted = ctx.decrypt(encrypted, metadata, password="pw")
```

## Security Features (v0.2.0)

### MAC Verification

```python
# Encrypt with password
encrypted, metadata = ctx.encrypt("data", password="correct_pw")

# Wrong password raises ValueError
try:
    ctx.decrypt(encrypted, metadata, password="wrong_pw")
except ValueError as e:
    print(f"✓ Wrong password rejected: {e}")

# Tampered data raises ValueError
tampered = bytearray(encrypted)
tampered[10] ^= 0xFF  # Flip bits

try:
    ctx.decrypt(bytes(tampered), metadata, password="correct_pw")
except ValueError as e:
    print(f"✓ Tampering detected: {e}")
```

### Metadata Encryption

```python
# Metadata is encrypted with ephemeral key derived from password
encrypted, metadata = ctx.encrypt("data", password="pw")

# Metadata is binary TLV format (not human-readable)
print(f"Metadata type: {type(metadata)}")  # <class 'bytes'>

# Attempting to deserialize without password fails
from utils.tlv_format import deserialize_metadata_tlv
try:
    plain_meta = deserialize_metadata_tlv(metadata)
    # This succeeds but data is encrypted inside
    print(f"Encrypted metadata keys: {plain_meta.keys()}")
except Exception as e:
    print(f"Cannot parse: {e}")
```

## Practical Examples

### Password Manager

```python
import json
from datetime import datetime

class PasswordVault:
    def __init__(self, master_password: str):
        self.ctx = STCContext(f"vault-{master_password}")
        self.password = master_password
        self.vault = {}
    
    def store(self, service: str, username: str, password: str):
        data = json.dumps({
            'service': service,
            'username': username,
            'password': password,
            'created': datetime.now().isoformat()
        })
        
        enc, meta = self.ctx.encrypt(data, password=self.password)
        self.vault[service] = {
            'encrypted': enc.hex(),
            'metadata': meta.hex()
        }
    
    def retrieve(self, service: str) -> dict:
        entry = self.vault[service]
        enc = bytes.fromhex(entry['encrypted'])
        meta = bytes.fromhex(entry['metadata'])
        
        data = self.ctx.decrypt(enc, meta, password=self.password)
        return json.loads(data)

# Usage
vault = PasswordVault("master_password_123")
vault.store("github.com", "user@example.com", "gh_token_xyz")
creds = vault.retrieve("github.com")
print(f"Username: {creds['username']}")
```

### Configuration Encryption

```python
import json

def encrypt_config(config: dict, password: str, output: str):
    """Encrypt config file with password protection"""
    ctx = STCContext(f"config-{password}")
    
    config_json = json.dumps(config, indent=2)
    encrypted, metadata = ctx.encrypt(config_json, password=password)
    
    # Save both as binary
    with open(output, 'wb') as f:
        f.write(encrypted)
    with open(f"{output}.meta", 'wb') as f:
        f.write(metadata)

def decrypt_config(input_file: str, password: str) -> dict:
    """Decrypt config file with MAC verification"""
    ctx = STCContext(f"config-{password}")
    
    with open(input_file, 'rb') as f:
        encrypted = f.read()
    with open(f"{input_file}.meta", 'rb') as f:
        metadata = f.read()
    
    config_json = ctx.decrypt(encrypted, metadata, password=password)
    return json.loads(config_json)

# Usage
config = {
    'database': {'host': 'localhost', 'password': 'db_pw'},
    'api_key': 'sk-xxxxx'
}

encrypt_config(config, "config_password", "app.enc")
loaded = decrypt_config("app.enc", "config_password")
```

## Error Handling

### Common Errors

**Wrong Password:**
```python
try:
    decrypted = ctx.decrypt(encrypted, metadata, password="wrong")
except ValueError as e:
    # "MAC verification failed" or "Decryption failed"
    print(f"Authentication error: {e}")
```

**Corrupted Data:**
```python
try:
    decrypted = ctx.decrypt(b"corrupted", metadata, password="pw")
except (ValueError, Exception) as e:
    print(f"Data corruption: {e}")
```

**Invalid Parameters:**
```python
try:
    ctx = STCContext('seed', lattice_size=1000)  # Too large
except ValueError as e:
    print(f"Invalid lattice size: {e}")
```

## Performance Optimization

### Reuse Contexts

```python
# ❌ Slow - recreates CEL each time
for item in items:
    ctx = STCContext('seed')  # CEL initialization is expensive
    encrypted, meta = ctx.encrypt(item, password="pw")

# ✓ Fast - reuse context
ctx = STCContext('seed')  # Once
for item in items:
    encrypted, meta = ctx.encrypt(item, password="pw")
```

### Batch Encryption

```python
# ❌ High overhead - one metadata per item
for item in items:
    enc, meta = ctx.encrypt(item, password="pw")
    # Each metadata is ~786KB

# ✓ Lower overhead - one metadata for all
batch = json.dumps(items)
encrypted, metadata = ctx.encrypt(batch, password="pw")
# Only one 786KB metadata for entire batch
```

### Choose Appropriate Lattice Size

```python
# For non-critical data (logs, cache)
fast_ctx = STCContext('seed', lattice_size=64, depth=4)

# For critical data (passwords, keys)
secure_ctx = STCContext('seed', lattice_size=128, depth=6)  # Default

# For maximum security (cold storage)
paranoid_ctx = STCContext('seed', lattice_size=256, depth=8)
```

## Security Best Practices

### Password Management

```python
# ✓ Good - high-entropy password
import getpass
password = getpass.getpass("Enter master password: ")
ctx = STCContext('unique-seed')
encrypted, metadata = ctx.encrypt(data, password=password)

# ❌ Bad - hardcoded password
encrypted, metadata = ctx.encrypt(data, password="password123")

# ❌ Bad - password in source code
PASSWORD = "secret"  # Never do this
```

### Seed Selection

```python
# ✓ Good - unique per user/application
user_id = "alice@example.com"
ctx = STCContext(f"app-v1-{user_id}")

# ✓ Good - user-provided seed
seed = input("Enter encryption seed: ")
ctx = STCContext(seed)

# ❌ Bad - generic seed
ctx = STCContext("default")  # Same for all users!
```

### Data Sanitization

```python
# ✓ Good - don't log sensitive data
encrypted, metadata = ctx.encrypt(password, password=master_pw)
print(f"Encrypted {len(encrypted)} bytes")  # Safe

# ❌ Bad - logging plaintext
print(f"Encrypting password: {password}")  # Never log plaintext!

# ✓ Good - clear sensitive data after use
password = "sensitive"
encrypted, metadata = ctx.encrypt(password, password=master_pw)
password = None  # Clear reference
```

### Metadata Protection

```python
# Metadata contains encrypted CEL state
# Treat it as sensitive as encrypted data

# ✓ Good - protect both equally
os.chmod("data.enc", 0o600)
os.chmod("data.enc.meta", 0o600)

# ❌ Bad - expose metadata
# Attacker might analyze CEL patterns
```

## Testing

### Verify Encryption

```python
def test_encryption():
    ctx = STCContext('test-seed')
    data = "test data"
    password = "test_pw"
    
    # Encrypt
    encrypted, metadata = ctx.encrypt(data, password=password)
    
    # Verify encrypted data differs from plaintext
    assert encrypted != data.encode()
    
    # Verify metadata is present
    assert len(metadata) > 0
    
    # Decrypt
    decrypted = ctx.decrypt(encrypted, metadata, password=password)
    
    # Verify round-trip
    assert decrypted == data
    
    print("✓ Encryption test passed")

test_encryption()
```

### Test Different Data Types

```python
test_cases = [
    ("string", "simple text"),
    ("unicode", "Hello 世界 🌍"),
    ("bytes", b'\x00\xFF\xFE\xFD'),
    ("large", "x" * 10000),
    ("empty", ""),
]

ctx = STCContext('test')

for name, data in test_cases:
    enc, meta = ctx.encrypt(data, password="pw")
    dec = ctx.decrypt(enc, meta, password="pw")
    assert dec == data
    print(f"✓ {name}")
```

## Migration from v0.1.x

### API Changes

```python
# v0.1.x
from seigrtc.interfaces.api import stc_api
context = stc_api.initialize(seed="seed")
encrypted, metadata = context.encrypt("data")

# v0.2.0
from interfaces.api.stc_api import STCContext
context = STCContext('seed')
encrypted, metadata = context.encrypt("data", password="pw")
```

### Metadata Format

```python
# v0.1.x: JSON dict with numpy arrays
metadata = {
    'cel_snapshot': {'lattice': [[...]], ...},
    'phe_hash': '...',
    ...
}

# v0.2.0: Binary TLV format
metadata = b'\x01\x00\x00\x00...'  # bytes

# Auto-detection works
ctx.decrypt(encrypted, old_json_metadata)  # Works
ctx.decrypt(encrypted, new_tlv_metadata)   # Works
```

## Limitations

- **Metadata Size**: ~786 KB constant overhead
- **Performance**: ~2.3s for small messages (encrypt + decrypt)
- **Decoy Vectors**: Not yet supported (use_decoys must be False)
- **Streaming**: Not supported (load entire file into memory)

See [CHANGELOG.md](../CHANGELOG.md) for version history and [PERFORMANCE_OPTIMIZATIONS.md](../PERFORMANCE_OPTIMIZATIONS.md) for optimization details.


## Basic Encryption/Decryption

### Simple String Encryption

```python
from interfaces.api import stc_api

# Initialize context
context = stc_api.initialize(seed="my-secret-seed")

# Encrypt
plaintext = "This is sensitive data"
encrypted, metadata = context.encrypt(plaintext)

# Decrypt
decrypted = context.decrypt(encrypted, metadata)

assert decrypted == plaintext
```

### Binary Data Encryption

```python
# Works with bytes too
binary_data = b'\x00\x01\x02\x03\xFF\xFE\xFD'
encrypted, metadata = context.encrypt(binary_data)
decrypted = context.decrypt(encrypted, metadata)

assert decrypted == binary_data
```

## Quick API (One-Time Operations)

### Quick Encrypt/Decrypt

```python
from interfaces.api import stc_api

# Encrypt without managing context
encrypted, metadata, context = stc_api.quick_encrypt(
    data="sensitive message",
    seed="my-seed"
)

# Decrypt - context reconstructed from metadata
decrypted = stc_api.quick_decrypt(
    encrypted_data=encrypted,
    metadata=metadata,
    seed="my-seed"
)
```

**Use Case**: One-time encryption where you don't need to reuse the context.

## Probabilistic Hashing

### Basic Hashing

```python
context = stc_api.initialize(seed="hash-seed")

# Hash changes over time
hash1 = context.hash("same data")
hash2 = context.hash("same data")

assert hash1 != hash2  # Different due to CEL evolution
```

### Password Verification

For password verification, freeze CEL state:

```python
# Store this snapshot when setting password
context = stc_api.initialize(seed="user-specific-seed")
password_hash = context.hash("user-password")
cel_snapshot = context.cel.get_snapshot()

# Later, for verification:
verify_context = stc_api.initialize(seed="user-specific-seed")
verify_context.cel.restore_from_snapshot(cel_snapshot)
candidate_hash = verify_context.hash("candidate-password")

if candidate_hash == password_hash:
    print("Password correct")
```

## Context-Aware Operations

### Using Context Data

```python
context = stc_api.initialize(seed="main-seed")

# Context influences key derivation
encrypted1, meta1 = context.encrypt(
    data="secret",
    context_data={"user": "alice", "role": "admin"}
)

# Different context = different ciphertext
encrypted2, meta2 = context.encrypt(
    data="secret",
    context_data={"user": "bob", "role": "user"}
)

assert encrypted1 != encrypted2

# Must use same context for decryption
decrypted1 = context.decrypt(encrypted1, meta1, 
    context_data={"user": "alice", "role": "admin"})

assert decrypted1 == "secret"
```

## Persistent Storage

### Saving Encrypted Data

```python
import json

# Encrypt
encrypted, metadata = context.encrypt("data to store")

# Metadata contains NumPy arrays - need custom serialization
def serialize_metadata(meta):
    """Convert metadata for JSON storage"""
    result = meta.copy()
    # CEL snapshot already contains nested lists (handled by StateManager)
    # PHE hash is bytes - convert to hex
    if 'phe_hash' in result:
        result['phe_hash'] = result['phe_hash'].hex()
    return result

# Save to file
with open('encrypted.json', 'w') as f:
    json.dump({
        'encrypted': encrypted.hex(),
        'metadata': serialize_metadata(metadata)
    }, f)
```

### Loading Encrypted Data

```python
# Load from file
with open('encrypted.json', 'r') as f:
    data = json.load(f)

# Deserialize
encrypted = bytes.fromhex(data['encrypted'])
metadata = data['metadata']
metadata['phe_hash'] = bytes.fromhex(metadata['phe_hash'])

# Decrypt with same seed
decrypted = stc_api.quick_decrypt(
    encrypted_data=encrypted,
    metadata=metadata,
    seed="my-secret-seed"
)
```

## Password Manager Example

See `examples/password_manager/password_manager.py` for complete implementation.

### Key Features

```python
class PasswordManager:
    def __init__(self, master_password: str):
        self.context = stc_api.initialize(master_password)
        self.vault = {}
    
    def store_credential(self, service: str, username: str, password: str):
        credential = {
            'service': service,
            'username': username,
            'password': password,
            'created': datetime.now().isoformat()
        }
        encrypted, metadata = self.context.encrypt(json.dumps(credential))
        self.vault[service] = {
            'encrypted': encrypted,
            'metadata': metadata
        }
    
    def get_credential(self, service: str) -> dict:
        entry = self.vault[service]
        decrypted = self.context.decrypt(
            entry['encrypted'],
            entry['metadata']
        )
        return json.loads(decrypted)
```

## Configuration Encryption Example

See `examples/config_encryption/config_example.py` for complete implementation.

### Encrypting Config Files

```python
import json

# Load config
with open('config.json', 'r') as f:
    config = json.load(f)

# Encrypt entire config
context = stc_api.initialize(seed="config-encryption-key")
config_json = json.dumps(config)
encrypted, metadata = context.encrypt(config_json)

# Save encrypted config
with open('config.enc', 'wb') as f:
    f.write(encrypted)

# Save metadata separately
with open('config.enc.meta', 'w') as f:
    json.dump(serialize_metadata(metadata), f)
```

### Loading Encrypted Config

```python
# Load encrypted files
with open('config.enc', 'rb') as f:
    encrypted = f.read()

with open('config.enc.meta', 'r') as f:
    metadata = json.load(f)
    metadata['phe_hash'] = bytes.fromhex(metadata['phe_hash'])

# Decrypt
decrypted = stc_api.quick_decrypt(
    encrypted_data=encrypted,
    metadata=metadata,
    seed="config-encryption-key"
)

config = json.loads(decrypted)
```

## Advanced: Manual CEL Control

### Controlling CEL Updates

```python
context = stc_api.initialize(seed="test")

# Manual update timing
import time

context.cel.update()  # First update
time.sleep(0.001)     # Wait 1ms
context.cel.update()  # Second update - different timing delta

# Get entropy at specific state
entropy = context.cel.get_entropy(32)
```

### Saving/Restoring CEL State

```python
# Save CEL state
snapshot = context.cel.get_snapshot()

# Do some operations
context.encrypt("data1")
context.encrypt("data2")

# Restore to saved state
context.cel.restore_from_snapshot(snapshot)

# CEL is now in same state as before operations
```

## Error Handling

### Common Issues

**Wrong seed for decryption:**
```python
encrypted, metadata, ctx = stc_api.quick_encrypt("data", seed="seed1")

# This will produce garbage:
decrypted = stc_api.quick_decrypt(encrypted, metadata, seed="wrong-seed")
# No error raised - decryption "succeeds" but returns corrupted data
```

**Missing metadata fields:**
```python
try:
    decrypted = context.decrypt(encrypted, {})  # Empty metadata
except KeyError as e:
    print(f"Missing metadata field: {e}")
```

**Type errors:**
```python
try:
    encrypted = context.encrypt(12345)  # Not str or bytes
except TypeError as e:
    print(f"Invalid type: {e}")
```

## Performance Tips

### Reuse Contexts

Don't create new contexts for each operation:

```python
# Bad - slow
for item in items:
    ctx = stc_api.initialize(seed="same-seed")  # Reinitializes CEL each time
    encrypted, meta = ctx.encrypt(item)

# Good - fast
context = stc_api.initialize(seed="same-seed")  # Once
for item in items:
    encrypted, meta = context.encrypt(item)
```

### Minimize Metadata Storage

Metadata is ~10-20 KB. For bulk encryption:

```python
# Bad - store metadata for each item
for item in items:
    enc, meta = context.encrypt(item)
    save(enc, meta)  # Each item gets 10-20 KB overhead

# Better - batch encrypt
batch = json.dumps(items)
encrypted, metadata = context.encrypt(batch)
save(encrypted, metadata)  # One metadata dict for all items
```

### Monitor Memory Usage

```python
import sys

context = stc_api.initialize(seed="test", lattice_size=256, depth=8)

# CEL lattice size
lattice_bytes = context.cel.lattice.nbytes
print(f"CEL memory: {lattice_bytes / 1024 / 1024:.2f} MB")

# Metadata size
metadata_size = len(json.dumps(serialize_metadata(metadata)))
print(f"Metadata: {metadata_size / 1024:.2f} KB")
```

## Security Best Practices

### Seed Management

```python
# Good - user-provided, high-entropy seed
seed = input("Enter master password: ")  # User memorizes this
context = stc_api.initialize(seed=seed)

# Bad - hardcoded seed
context = stc_api.initialize(seed="password123")  # Never do this

# Bad - low-entropy seed
import random
seed = str(random.randint(0, 999))  # Only 1000 possibilities
```

### Data Sanitization

```python
# Don't log sensitive data
encrypted, metadata = context.encrypt(password)
print(f"Encrypted password: {encrypted.hex()}")  # Safe - encrypted
print(f"Password: {password}")  # NEVER do this

# Don't expose decrypted data unnecessarily
decrypted = context.decrypt(encrypted, metadata)
# Use decrypted immediately, don't store in variables longer than needed
```

### Metadata Protection

```python
# Metadata contains CEL state - don't expose publicly
# If attacker gets metadata, they see CEL snapshot
# Combine with seed protection:

# Store metadata with same protection as encrypted data
save_encrypted_file(encrypted, metadata)  # Both protected
```

## Testing Encryption

### Verify Round-Trip

```python
def test_encryption():
    data = "test data"
    context = stc_api.initialize(seed="test-seed")
    
    encrypted, metadata = context.encrypt(data)
    decrypted = context.decrypt(encrypted, metadata)
    
    assert decrypted == data, "Round-trip failed"
    assert encrypted != data.encode(), "Data not encrypted"
    print("✓ Encryption test passed")

test_encryption()
```

### Test Different Data Types

```python
test_cases = [
    "simple string",
    "unicode: 你好世界 🌍",
    b'\x00\x01\x02\xFF\xFE\xFD',
    "a" * 10000,  # Large data
    "",  # Empty string
]

context = stc_api.initialize(seed="test")

for data in test_cases:
    encrypted, metadata = context.encrypt(data)
    decrypted = context.decrypt(encrypted, metadata)
    assert decrypted == data
    print(f"✓ Passed: {type(data).__name__}")
```

## Limitations and Constraints

### Data Size

```python
# No hard limit, but performance degrades
# Recommended: < 1 MB per operation
# Larger data: split into chunks

def encrypt_large(data: bytes, chunk_size: int = 1024*100):
    chunks = [data[i:i+chunk_size] for i in range(0, len(data), chunk_size)]
    encrypted_chunks = []
    metadatas = []
    
    for chunk in chunks:
        enc, meta = context.encrypt(chunk)
        encrypted_chunks.append(enc)
        metadatas.append(meta)
    
    return encrypted_chunks, metadatas
```

### Seed Constraints

- Must be deterministic for same decryption
- Length doesn't matter (hashed internally)
- Empty seed is valid but insecure

### Thread Safety

```python
# Not thread-safe - use separate contexts
import threading

def worker(thread_id):
    # Create context per thread
    context = stc_api.initialize(seed=f"thread-{thread_id}")
    encrypted, meta = context.encrypt(f"data-{thread_id}")

threads = [threading.Thread(target=worker, args=(i,)) for i in range(10)]
for t in threads:
    t.start()
for t in threads:
    t.join()
```

## CLI Usage

```bash
# Encrypt file
stc encrypt input.txt --seed "my-seed" --output encrypted.bin

# Decrypt file
stc decrypt encrypted.bin --seed "my-seed" --metadata meta.json --output decrypted.txt

# Hash data
stc hash "data to hash" --seed "hash-seed"
```

See `interfaces/cli/stc_cli.py` for complete CLI documentation.
