# Core Modules

Detailed specifications for STC core modules.

## Continuous Entropy Lattice (CEL)

**File**: `core/cel/cel.py`

### Purpose

Provides self-evolving entropy source without external randomness.

### Implementation Details

#### Lattice Structure
- **Type**: 3D NumPy array (`np.ndarray`)
- **Shape**: `(size, size, depth)` - default `(256, 256, 8)`
- **Data Type**: `uint64`
- **Total Cells**: 524,288 for default configuration
- **Memory**: ~4 MB for default lattice

#### Initialization

1. Seed conversion to integer:
   - String → UTF-8 bytes → integer via `int.from_bytes()`
   - Bytes → integer directly
   - Integer → used as-is

2. Lattice generation:
   ```python
   # For each cell (i, j, k):
   base = (seed + i * size + j) % PRIME
   exponent = (k + 1) % PRIME
   lattice[i, j, k] = pow(base, exponent, PRIME)
   ```
   where `PRIME = 65521` (largest prime < 2^16)

3. Initial diffusion:
   - 3 rounds of non-linear diffusion
   - Prevents zero-valued cells
   - Spreads seed influence across lattice

#### State Evolution

Triggered by `update()` call:

1. Calculate time delta:
   ```python
   current_time = time.perf_counter()
   delta_t = current_time - last_update_time
   time_factor = int((delta_t * 1e9) % PRIME)
   ```

2. Apply non-linear diffusion:
   - Each cell influenced by 6 neighbors (±x, ±y, ±z)
   - Weighted combination: `(sum_neighbors * time_factor) % PRIME`
   - 1 round per update

3. Increment update counter

#### Entropy Extraction

`get_entropy(length)` method:

1. Flatten lattice to 1D array
2. Take first `length` values
3. Return as uint64 NumPy array
4. Call `update()` after extraction

### Constants

```python
PRIME = 65521           # Modulus for all operations
DEFAULT_SIZE = 256      # Lattice dimension
DEFAULT_DEPTH = 8       # Lattice depth layers
DIFFUSION_ROUNDS = 3    # Initial diffusion rounds
```

### Thread Safety

**Not thread-safe**. Concurrent `update()` calls cause race conditions on:
- `lattice` array
- `last_update_time`
- `update_count`

---

## Probabilistic Hashing Engine (PHE)

**File**: `core/phe/phe.py`

### Purpose

Generates hashes influenced by CEL state, producing different outputs over time for same input.

### Implementation Details

#### Hash Generation Process

1. Update CEL state:
   ```python
   self.cel.update()
   ```

2. Convert input to bytes:
   - String → UTF-8 bytes
   - Bytes → as-is

3. Get CEL entropy:
   ```python
   entropy = self.cel.get_entropy(32)  # 32 uint64 values
   ```

4. Combine data with entropy:
   ```python
   for i, byte in enumerate(data_bytes):
       combined = (byte + entropy[i % 32]) % 256
       hash_value.append(combined)
   ```

5. Apply context if provided:
   ```python
   if context:
       context_bytes = str(context).encode()
       for i, ctx_byte in enumerate(context_bytes):
           hash_value[i % 32] ^= ctx_byte
   ```

6. Return first 32 bytes

### Properties

- **Output Size**: Always 32 bytes
- **Determinism**: Same input + same CEL state = same hash
- **Time Variance**: Different CEL state → different hash
- **Collision Resistance**: Depends on CEL entropy quality

### Use Cases

- Password verification (with static CEL state)
- Data fingerprinting
- Context-aware hashing
- Not suitable for: blockchain, Merkle trees, or applications requiring deterministic hashing

---

## Contextual Key Emergence (CKE)

**File**: `core/cke/cke.py`

### Purpose

Derives encryption keys from CEL state and optional context data.

### Implementation Details

#### Key Derivation Process

1. Hash context data (if provided):
   ```python
   if context_data:
       context_hash = self.phe.hash(str(context_data))
   ```

2. Get CEL entropy:
   ```python
   base_entropy = self.cel.get_entropy(key_length)
   ```

3. Combine entropy with context:
   ```python
   for i in range(key_length):
       if context_data:
           key_vector[i] = (base_entropy[i] + context_hash[i]) % PRIME
       else:
           key_vector[i] = base_entropy[i] % PRIME
   ```

4. Return key as NumPy array copy

### Key Properties

- **Length**: Configurable (default 32 bytes)
- **Type**: `np.ndarray` of `uint64`
- **Range**: Values in [0, 65520] (mod PRIME)
- **Ephemeral**: Not stored, regenerated on demand

### Important Notes

**Fixed in v0.1.0**: `derive()` now returns `key_vector.copy()` instead of reference. This prevents external modification of internal state.

---

## Data-State Folding (DSF)

**File**: `core/dsf/dsf.py`

### Purpose

Encrypts data via multidimensional tensor transformations.

### Implementation Details

#### Encryption: fold()

1. **Tensor Preparation**:
   ```python
   # Calculate tensor dimensions
   total_size = len(data)
   height = int(np.ceil(np.sqrt(total_size)))
   width = height
   
   # Pad data to fit tensor
   padded_size = height * width
   padded_data = data + b'\x00' * (padded_size - total_size)
   
   # Reshape to 2D tensor
   tensor = np.frombuffer(padded_data, dtype=np.uint8).reshape(height, width)
   ```

2. **Apply 5 Folding Strategies**:

   **a. Rotation** (circular shift):
   ```python
   shift = int(key[0] % height)
   tensor = np.roll(tensor, shift, axis=0)
   tensor = np.roll(tensor, shift, axis=1)
   ```
   
   **b. Permutation** (row/column shuffle):
   ```python
   seed = int(key[1])
   rng = np.random.RandomState(seed)
   row_order = rng.permutation(height)
   col_order = rng.permutation(width)
   tensor = tensor[row_order, :]
   tensor = tensor[:, col_order]
   ```
   
   **c. Compression** (modular mixing):
   ```python
   for i in range(height):
       for j in range(width):
           tensor[i, j] = (tensor[i, j] + key[(i+j) % len(key)]) % 256
   ```
   
   **d. Diffusion** (non-linear mixing):
   ```python
   tensor = math_primitives.non_linear_diffusion(tensor, rounds=3)
   ```
   
   **e. Entropy Weighting** (CEL integration):
   ```python
   cel_entropy = self.cel.get_entropy(height * width)
   entropy_weights = (cel_entropy % 256).reshape(height, width)
   tensor = ((tensor.astype(np.uint16) * entropy_weights) % 256).astype(np.uint8)
   ```

3. **Flatten and Return**:
   ```python
   return tensor.flatten().tobytes()
   ```

#### Decryption: unfold()

Reverses operations in exact opposite order:

1. Reshape to tensor
2. Reverse entropy weighting (modular inverse)
3. Reverse diffusion
4. Reverse compression
5. Reverse permutation (inverse order)
6. Reverse rotation (negative shift)
7. Trim to original length

### Critical Requirements

- **Integer-Only Operations**: All arithmetic uses modulo 256 or PRIME
- **No Rounding**: Eliminated in v0.1.0 to ensure perfect reversibility
- **CEL State Match**: Decryption requires exact CEL state from encryption
- **Key Match**: Same key must be used for fold/unfold

### Performance

- **Fold Time**: ~50-200 ms for <10 KB data
- **Memory**: ~2x input size during processing
- **Overhead**: Metadata ~10-20 KB (CEL snapshot)

---

## Polymorphic Cryptographic Flow (PCF)

**File**: `core/pcf/pcf.py`

### Purpose

Tracks operation count and provides morphing state for algorithm adaptation.

### Implementation Details

#### State Tracking

```python
class PolymorphicCryptographicFlow:
    def __init__(self, morph_interval: int = 100):
        self.operation_count = 0
        self.morph_interval = morph_interval
        self.current_morph = 0
```

#### Methods

**update()**:
```python
def update(self):
    self.operation_count += 1
    if self.operation_count % self.morph_interval == 0:
        self.current_morph += 1
```

**get_morph_state()**:
```python
def get_morph_state() -> int:
    return self.current_morph
```

### Current Usage

In v0.1.0, PCF is **initialized but not actively used** in encryption/decryption logic. It's present for future algorithm morphing implementations.

### Future Use Cases

- Algorithm selection based on morph state
- Dynamic folding strategy changes
- Adaptive security parameter adjustment

---

## State Management

**File**: `core/state/state.py`

### Purpose

Serializes and deserializes CEL state for metadata storage.

### Implementation Details

#### Serialization

```python
@staticmethod
def serialize_state(cel: ContinuousEntropyLattice) -> Dict[str, Any]:
    snapshot = cel.get_snapshot()
    return {
        'lattice': snapshot['lattice'].tolist(),  # NumPy array → Python list
        'size': snapshot['size'],
        'depth': snapshot['depth'],
        'seed': snapshot['seed'],
        'update_count': snapshot['update_count']
    }
```

#### Deserialization

```python
@staticmethod
def deserialize_state(state_dict: Dict[str, Any]) -> ContinuousEntropyLattice:
    # Create new CEL with same parameters
    cel = ContinuousEntropyLattice(
        seed=state_dict['seed'],
        size=state_dict['size'],
        depth=state_dict['depth']
    )
    
    # Restore lattice from list
    lattice_array = np.array(state_dict['lattice'], dtype=np.uint64)
    
    # Restore snapshot
    cel.restore_from_snapshot({
        'lattice': lattice_array,
        'size': state_dict['size'],
        'depth': state_dict['depth'],
        'seed': state_dict['seed'],
        'update_count': state_dict['update_count']
    })
    
    return cel
```

### JSON Compatibility

- Converts NumPy arrays to nested Python lists
- All numeric types remain JSON-compatible
- Enables metadata storage in JSON files

### Size Considerations

Default lattice (256×256×8):
- Raw size: 524,288 uint64 values
- JSON size: ~10-15 KB (compressed with integer serialization)
- Dominates metadata size

---

## Mathematical Primitives

**File**: `utils/math_primitives.py`

### Functions

#### modular_exponentiation
```python
def modular_exponentiation(base: int, exponent: int, modulus: int) -> int
```

Fast modular exponentiation using binary method.

**Usage**: CEL lattice initialization

**Example**:
```python
result = modular_exponentiation(5, 100, 65521)  # 5^100 mod 65521
```

#### modular_inverse
```python
def modular_inverse(a: int, m: int) -> int
```

Computes modular multiplicative inverse using Extended Euclidean Algorithm.

**Usage**: DSF entropy weight reversal

**Example**:
```python
inv = modular_inverse(7, 65521)  # 7^-1 mod 65521
```

#### non_linear_diffusion
```python
def non_linear_diffusion(matrix: np.ndarray, rounds: int = 1) -> np.ndarray
```

Applies cellular automaton-like diffusion to 2D matrix.

**Process**:
1. For each cell, compute weighted sum of neighbors
2. Apply modular arithmetic
3. Repeat for specified rounds

**Fixed in v0.1.0**: Now uses separate result matrix per iteration to avoid in-place modification bugs.

#### tensor_permutation
```python
def tensor_permutation(tensor: np.ndarray, seed: int) -> Tuple[np.ndarray, np.ndarray]
```

Deterministic tensor shuffling.

**Returns**: (permuted_tensor, inverse_permutation_indices)

**Usage**: DSF permutation strategy

#### safe_index
```python
def safe_index(tensor: np.ndarray, indices: Tuple[int, ...]) -> Any
```

Bounds-checked array indexing.

**Returns**: Value at index or 0 if out of bounds

---

## Dependencies Between Modules

```
CEL (base entropy source)
 ├─► PHE (hashing)
 │    └─► CKE (key derivation)
 │         └─► DSF (encryption/decryption)
 ├─► CKE (key derivation)
 │    └─► DSF
 └─► DSF (direct entropy access)

PCF (independent)
STATE (operates on CEL)
UTILS (pure functions, no dependencies)
```

## Version History

### v0.1.0 Changes

1. **CEL**: Fixed initialization (use PRIME=65521 instead of 2^16)
2. **CKE**: Fixed derive() to return copy instead of reference
3. **DSF**: Replaced floating-point rotations with np.roll()
4. **DSF**: Removed all rounding operations
5. **Utils**: Fixed non_linear_diffusion in-place modification bug

All modules now use pure integer arithmetic for perfect reversibility.
