# STC Architecture

## Overview

STC is a Python library implementing a post-classical cryptographic system. The architecture consists of six core modules that work together to provide encryption, hashing, and key derivation without using traditional cryptographic primitives.

## System Components

```
┌─────────────────────────────────────────────────────┐
│                  Application Layer                  │
│            (User Code / Examples / CLI)             │
└─────────────────────────────────────────────────────┘
                         │
                         ▼
┌─────────────────────────────────────────────────────┐
│                   API Interface                     │
│         interfaces/api/stc_api.py                   │
│  - initialize()  - encrypt()   - decrypt()          │
│  - quick_encrypt()  - quick_decrypt()               │
└─────────────────────────────────────────────────────┘
                         │
        ┌────────────────┼────────────────┐
        ▼                ▼                ▼
┌──────────────┐  ┌──────────────┐  ┌──────────────┐
│     CEL      │  │     PHE      │  │     CKE      │
│  Entropy     │  │   Hashing    │  │   Key Gen    │
│  Lattice     │  │   Engine     │  │  Derivation  │
└──────────────┘  └──────────────┘  └──────────────┘
        │                │                │
        └────────────────┼────────────────┘
                         ▼
                  ┌──────────────┐
                  │     DSF      │
                  │  Data-State  │
                  │   Folding    │
                  └──────────────┘
                         │
        ┌────────────────┼────────────────┐
        ▼                ▼                ▼
┌──────────────┐  ┌──────────────┐  ┌──────────────┐
│     PCF      │  │    STATE     │  │    UTILS     │
│  Polymorphic │  │  Management  │  │Math Prims    │
│     Flow     │  │  Persistence │  │              │
└──────────────┘  └──────────────┘  └──────────────┘
```

## Core Modules

### 1. Continuous Entropy Lattice (CEL)
- **Location**: `core/cel/cel.py`
- **Class**: `ContinuousEntropyLattice`
- **Purpose**: Self-evolving entropy source
- **Dependencies**: NumPy, `utils.math_primitives`

#### Functionality
- Initializes a 3D lattice (default: 256×256×8) from a seed
- Evolves state using timing deltas between operations
- Provides entropy via `get_entropy()` method
- Updates internal state with `update()` method

#### State Evolution
- Uses `time.perf_counter()` for microsecond-precision timing
- Applies non-linear diffusion to lattice cells
- Modular arithmetic prevents overflow (mod 65521)

### 2. Probabilistic Hashing Engine (PHE)
- **Location**: `core/phe/phe.py`
- **Class**: `ProbabilisticHashingEngine`
- **Purpose**: Non-deterministic hashing
- **Dependencies**: CEL, `utils.math_primitives`

#### Functionality
- Generates hashes influenced by CEL state
- Same input produces different hashes over time
- Returns 32-byte hash values
- Integrates CEL entropy into hash computation

### 3. Contextual Key Emergence (CKE)
- **Location**: `core/cke/cke.py`
- **Class**: `ContextualKeyEmergence`
- **Purpose**: Key derivation from context
- **Dependencies**: CEL, PHE

#### Functionality
- Derives encryption keys from CEL state and context data
- Returns 32-byte key vectors
- Keys are ephemeral (not stored)
- Context data influences key generation

### 4. Data-State Folding (DSF)
- **Location**: `core/dsf/dsf.py`
- **Class**: `DataStateFolding`
- **Purpose**: Encryption via multidimensional transformations
- **Dependencies**: CEL, CKE, `utils.math_primitives`

#### Encryption Process
1. Reshape data into 2D tensor
2. Apply 5 folding strategies sequentially:
   - Rotation (circular shift via `np.roll`)
   - Permutation (row/column shuffling)
   - Compression (modular arithmetic)
   - Diffusion (non-linear mixing)
   - Entropy weighting (CEL integration)
3. Flatten tensor back to bytes

#### Decryption Process
- Reverses folding operations in reverse order
- Requires exact CEL state (embedded in metadata)
- All operations use integer arithmetic only

### 5. Polymorphic Cryptographic Flow (PCF)
- **Location**: `core/pcf/pcf.py`
- **Class**: `PolymorphicCryptographicFlow`
- **Purpose**: Dynamic algorithm selection
- **Dependencies**: None

#### Functionality
- Tracks operation count
- Morphs behavior every N operations (default: 100)
- Returns current morph state
- Updates flow state on each operation

### 6. State Management
- **Location**: `core/state/state.py`
- **Class**: `StateManager`
- **Purpose**: State serialization/deserialization
- **Dependencies**: NumPy (for array conversion)

#### Functionality
- Serializes CEL snapshots to JSON-compatible format
- Restores CEL state from metadata
- Handles NumPy array conversion to Python lists
- Enables deterministic decryption

## Data Flow

### Encryption Flow
```
Input Data (bytes/str)
    │
    ▼
Convert to bytes
    │
    ▼
CEL.update() ────────────┐
    │                    │ (timing entropy)
    ▼                    │
CKE.derive() ◄───────────┘
    │
    ▼
DSF.fold()
    │  ├─ Rotation
    │  ├─ Permutation
    │  ├─ Compression
    │  ├─ Diffusion
    │  └─ Entropy weighting
    ▼
Encrypted bytes + Metadata
```

### Decryption Flow
```
Encrypted bytes + Metadata
    │
    ▼
Extract CEL snapshot from metadata
    │
    ▼
Restore CEL state
    │
    ▼
CKE.derive() (same key)
    │
    ▼
DSF.unfold()
    │  ├─ Reverse entropy weighting
    │  ├─ Reverse diffusion
    │  ├─ Reverse compression
    │  ├─ Reverse permutation
    │  └─ Reverse rotation
    ▼
Original data
```

## Mathematical Primitives

Location: `utils/math_primitives.py`

### Functions
- `modular_exponentiation(base, exp, mod)` - Secure exponentiation
- `modular_inverse(a, m)` - Extended Euclidean algorithm
- `non_linear_diffusion(matrix, rounds)` - Cellular automaton-like mixing
- `tensor_permutation(tensor, seed)` - Deterministic shuffling
- `tensor_rotation(tensor, angle)` - **Deprecated** (was floating-point)
- `safe_index(tensor, indices)` - Bounds-checked array access

### Key Properties
- All operations use integer arithmetic
- No floating-point operations (ensures reversibility)
- Modular arithmetic prevents overflow
- Deterministic given same inputs

## Context Object

The `STCContext` class (`interfaces/api/stc_api.py`) maintains:
- CEL instance
- PHE instance  
- CKE instance
- DSF instance
- PCF instance
- STATE instance

Methods:
- `encrypt(data, context_data)` → `(encrypted_bytes, metadata)`
- `decrypt(encrypted_data, metadata, context_data)` → `original_data`
- `hash(data, context_data)` → `hash_bytes`

## Metadata Structure

Encryption produces metadata dict containing:
```python
{
    'original_length': int,        # Original data length
    'was_string': bool,            # True if input was string
    'phe_hash': bytes,             # Probabilistic hash of original
    'cel_snapshot': {              # Complete CEL state
        'lattice': list,           # Lattice as nested lists
        'size': int,               # Lattice size
        'depth': int,              # Lattice depth
        'seed': int,               # Original seed
        'update_count': int        # Number of updates
    }
}
```

## Design Constraints

### What STC Does NOT Use
- XOR operations
- AES, DES, or any block cipher
- RSA, ECC, or public-key cryptography
- SHA, BLAKE, or traditional hash functions
- Random number generators (random, secrets, os.urandom)
- External entropy sources

### What STC Uses Instead
- Modular arithmetic (mod prime)
- Array transformations (np.roll, permutations)
- Timing-based entropy (perf_counter deltas)
- State evolution (cellular automaton patterns)
- Integer-only operations

## Performance Characteristics

- **Encryption Speed**: ~100-500 ms for small data (<10 KB)
- **Memory Usage**: ~2-8 MB for CEL lattice
- **Key Size**: 32 bytes (256 bits)
- **Hash Size**: 32 bytes (256 bits)
- **Metadata Size**: ~10-20 KB (CEL snapshot included)

## Security Considerations

### Strengths
- No known classical cryptanalysis applies
- CEL state evolution adds temporal uniqueness
- Integer-only operations prevent side-channel timing attacks
- No key storage (keys derived on-demand)

### Limitations
- Alpha status: not peer-reviewed
- No formal security proofs
- CEL snapshot in metadata reveals internal state
- Timing entropy may be insufficient on virtualized systems
- Not quantum-resistant (uses modular arithmetic)

## Thread Safety

Current implementation is **not thread-safe**:
- CEL state mutations during concurrent operations
- PCF morph counter shared across instances
- No locks or synchronization primitives

For concurrent use, create separate `STCContext` instances per thread.

## Version Information

- **Current Version**: 0.1.0
- **Python Requirement**: 3.9+
- **NumPy Requirement**: 1.24.0+
- **Status**: Alpha (research-grade)
