# API Reference - STC v0.2.0

Complete API documentation for Seigr Toolset Crypto v0.2.0

## interfaces.api.stc_api

### STCContext

Main context class for STC operations with password-based encryption and MAC verification.

#### Constructor

```python
STCContext(
    seed: Union[str, bytes, int],
    lattice_size: int = 128,
    depth: int = 6,
    morph_interval: int = 100
)
```

**Parameters:**
- `seed`: Initialization seed (determines initial CEL state)
- `lattice_size`: CEL lattice dimension (default: 128, range: 16-256)
- `depth`: CEL lattice depth (default: 6, range: 2-8)
- `morph_interval`: PCF morphing frequency (default: 100 operations)

**Returns:** `STCContext` instance

**Example:**
```python
from interfaces.api.stc_api import STCContext

# Default parameters (balanced)
ctx = STCContext('my-seed')

# Custom parameters (faster, smaller metadata)
ctx_fast = STCContext('my-seed', lattice_size=64, depth=4)

# Maximum security (slower, larger metadata)
ctx_secure = STCContext('my-seed', lattice_size=256, depth=8)
```

---

#### encrypt()

```python
encrypt(
    data: Union[str, bytes],
    context_data: Optional[Dict[str, Any]] = None,
    password: Optional[str] = None,
    use_decoys: bool = False,
    num_decoys: int = 3
) -> Tuple[bytes, bytes]
```

Encrypt data with password-based protection and MAC.

**Parameters:**
- `data`: Data to encrypt (string or bytes)
- `context_data`: Optional additional context for key derivation
- `password`: Encryption password (if None, uses seed)
- `use_decoys`: Enable decoy vectors (NOT YET SUPPORTED - must be False)
- `num_decoys`: Number of decoy snapshots (3-5)

**Returns:** Tuple of `(encrypted_bytes, metadata_bytes)`
- `encrypted_bytes`: Encrypted data (binary)
- `metadata_bytes`: Binary TLV format metadata (~786KB)

**Raises:**
- `TypeError`: If data is not str or bytes
- `ValueError`: If use_decoys=True (not yet implemented)

**Example:**
```python
ctx = STCContext('my-seed')

# With password
encrypted, metadata = ctx.encrypt(
    "Secret message",
    password="strong_password"
)

# Without password (uses seed)
encrypted, metadata = ctx.encrypt("Data")

# Binary data
encrypted, metadata = ctx.encrypt(
    b'\x00\x01\x02\xFF',
    password="pw"
)
```

---

#### decrypt()

```python
decrypt(
    encrypted_data: bytes,
    metadata: Union[Dict[str, Any], bytes],
    context_data: Optional[Dict[str, Any]] = None,
    password: Optional[str] = None
) -> Union[str, bytes]
```

Decrypt data with automatic MAC verification.

**Parameters:**
- `encrypted_data`: Encrypted bytes from encrypt()
- `metadata`: Metadata from encrypt() (binary TLV or dict for v0.1.x compat)
- `context_data`: Optional context (must match encryption)
- `password`: Decryption password (if None, uses seed)

**Returns:** Decrypted data (string if was_string=True, else bytes)

**Raises:**
- `ValueError`: If MAC verification fails (wrong password or tampering)
- `KeyError`: If required metadata fields missing
- `Exception`: If decryption fails for other reasons

**Example:**
```python
ctx = STCContext('my-seed')

# With password
try:
    decrypted = ctx.decrypt(
        encrypted,
        metadata,
        password="strong_password"
    )
except ValueError:
    print("Wrong password or data tampered!")

# Without password (uses seed)
decrypted = ctx.decrypt(encrypted, metadata)

# Backward compatible with v0.1.x JSON metadata
decrypted = ctx.decrypt(encrypted, old_json_metadata)
```

---

#### hash()

```python
hash(
    data: Union[str, bytes],
    context_data: Optional[Dict[str, Any]] = None
) -> bytes
```

Generate probabilistic hash (CEL-driven, changes over time).

**Parameters:**
- `data`: Data to hash
- `context_data`: Optional additional context

**Returns:** 32-byte hash value

**Note:** Each call produces different hash due to CEL evolution. Not suitable for password verification unless CEL state is frozen.

**Example:**
```python
ctx = STCContext('hash-seed')

hash1 = ctx.hash("data")
hash2 = ctx.hash("data")

assert hash1 != hash2  # Different due to CEL evolution
assert len(hash1) == 32
```

---

#### derive_key()

```python
derive_key(
    length: int = 32,
    context_data: Optional[Dict[str, Any]] = None
) -> bytes
```

Derive cryptographic key using CKE.

**Parameters:**
- `length`: Key length in bytes (default: 32)
- `context_data`: Optional additional context

**Returns:** Derived key (bytes of specified length)

**Example:**
```python
ctx = STCContext('my-seed')

# Derive 32-byte key
key = ctx.derive_key()

# Derive 64-byte key
key_64 = ctx.derive_key(length=64)

# With context
key_ctx = ctx.derive_key(context_data={'user': 'alice'})
```

---

#### save_state()

```python
save_state() -> Dict[str, Any]
```

Save current context state (CEL, PHE, PCF).

**Returns:** Dict containing all component states

**Example:**
```python
ctx = STCContext('my-seed')

# Perform operations
ctx.hash("data1")
ctx.hash("data2")

# Save state
state = ctx.save_state()

# State keys
print(state.keys())  # dict_keys(['cel_state', 'phe_state', 'pcf_state'])
```

---

#### load_state()

```python
load_state(state: Dict[str, Any]) -> None
```

Load previously saved context state.

**Parameters:**
- `state`: State dict from save_state()

**Example:**
```python
# Save state
state = ctx.save_state()

# Create new context and load state
ctx2 = STCContext('my-seed')
ctx2.load_state(state)

# ctx2 now has same state as ctx
```

---

#### get_status()

```python
get_status() -> str
```

Get human-readable context status.

**Returns:** Formatted status string

**Example:**
```python
ctx = STCContext('my-seed')
print(ctx.get_status())

# Output:
# === STC Context Status ===
# CEL Operation Count: 10
# CEL State Version: 5
# CEL Lattice Size: 128x128x6
# PCF Status: Active
# PCF Morph Interval: 100
# ==========================
```

---

### Module Functions

#### initialize()

```python
initialize(
    seed: Union[str, bytes, int],
    lattice_size: int = 128,
    depth: int = 6,
    morph_interval: int = 100
) -> STCContext
```

Initialize STC context (convenience function).

**Parameters:** Same as `STCContext.__init__`

**Returns:** `STCContext` instance

**Example:**
```python
from interfaces.api import stc_api

ctx = stc_api.initialize(seed="my-seed")
```

---

#### quick_encrypt()

```python
quick_encrypt(
    data: Union[str, bytes],
    seed: Union[str, bytes, int],
    password: Optional[str] = None,
    lattice_size: int = 128,
    depth: int = 6
) -> Tuple[bytes, bytes, STCContext]
```

One-liner encryption (creates context, encrypts, returns all).

**Parameters:**
- `data`: Data to encrypt
- `seed`: Encryption seed
- `password`: Optional password (if None, uses seed)
- `lattice_size`: CEL lattice size (default: 128)
- `depth`: CEL depth (default: 6)

**Returns:** Tuple of `(encrypted, metadata, context)`

**Example:**
```python
from interfaces.api import stc_api

encrypted, metadata, ctx = stc_api.quick_encrypt(
    "Secret data",
    seed="my-seed",
    password="strong_pw"
)
```

---

#### quick_decrypt()

```python
quick_decrypt(
    encrypted_data: bytes,
    metadata: Union[Dict[str, Any], bytes],
    seed: Union[str, bytes, int],
    password: Optional[str] = None
) -> Union[str, bytes]
```

One-liner decryption (creates context from metadata, decrypts).

**Parameters:**
- `encrypted_data`: Encrypted bytes
- `metadata`: Metadata from encryption
- `seed`: Decryption seed (must match encryption)
- `password`: Optional password (must match encryption)

**Returns:** Decrypted data

**Raises:** `ValueError` if MAC verification fails

**Example:**
```python
from interfaces.api import stc_api

decrypted = stc_api.quick_decrypt(
    encrypted,
    metadata,
    seed="my-seed",
    password="strong_pw"
)
```

---

## core.cel

### ContinuousEntropyLattice

Self-evolving 3D entropy lattice with timing-based entropy.

#### Constructor

```python
ContinuousEntropyLattice(
    lattice_size: int = 256,
    depth: int = 8
)
```

**Parameters:**
- `lattice_size`: Lattice dimension (NxN)
- `depth`: Lattice depth (Z dimension)

**Example:**
```python
from core.cel import ContinuousEntropyLattice

cel = ContinuousEntropyLattice(lattice_size=128, depth=6)
cel.init('my-seed')
```

---

#### init()

```python
init(seed: Union[str, bytes, int]) -> None
```

Initialize lattice from seed.

**Parameters:**
- `seed`: Initialization seed

---

#### update()

```python
update(context: Optional[Dict[str, Any]] = None) -> None
```

Evolve lattice state (mixes timing, memory, and context entropy).

**Parameters:**
- `context`: Optional context data to mix in

---

#### snapshot()

```python
snapshot() -> Dict[str, Any]
```

Take snapshot of current lattice state.

**Returns:** Dict with lattice array, operation count, etc.

---

#### restore_snapshot()

```python
restore_snapshot(snapshot: Dict[str, Any]) -> None
```

Restore lattice from snapshot.

**Parameters:**
- `snapshot`: Snapshot dict from snapshot()

---

#### extract_entropy()

```python
extract_entropy(
    length: int,
    context: Optional[str] = None
) -> np.ndarray
```

Extract entropy from lattice.

**Parameters:**
- `length`: Number of entropy values to extract
- `context`: Optional context for extraction

**Returns:** NumPy array of entropy values

---

## core.phe

### ProbabilisticHashingEngine

Multi-path probabilistic hashing with CEL integration.

#### Constructor

```python
ProbabilisticHashingEngine()
```

**Example:**
```python
from core.phe import ProbabilisticHashingEngine

phe = ProbabilisticHashingEngine()
```

---

#### digest()

```python
digest(
    data: Union[str, bytes],
    context: Optional[Dict[str, Any]] = None
) -> bytes
```

Generate multi-path probabilistic hash.

**Parameters:**
- `data`: Data to hash
- `context`: Optional context

**Returns:** 32-byte hash

**Example:**
```python
phe = ProbabilisticHashingEngine()
hash_value = phe.digest(b"data to hash")
```

---

#### map_entropy()

```python
map_entropy(cel_snapshot: Dict[str, Any]) -> None
```

Bind PHE to CEL snapshot.

**Parameters:**
- `cel_snapshot`: CEL snapshot from ContinuousEntropyLattice.snapshot()

---

#### trace()

```python
trace() -> Dict[str, Any]
```

Get PHE execution trace.

**Returns:** Dict with operation count, path history, etc.

---

## core.cke

### ContextualKeyEmergence

Ephemeral key generation from context intersections.

#### Constructor

```python
ContextualKeyEmergence()
```

---

#### derive()

```python
derive(
    length: int = 32,
    context: Optional[Dict[str, Any]] = None,
    cel_snapshot: Optional[Dict[str, Any]] = None,
    phe_hash: Optional[bytes] = None
) -> bytes
```

Derive ephemeral key.

**Parameters:**
- `length`: Key length in bytes
- `context`: Optional context data
- `cel_snapshot`: Optional CEL snapshot
- `phe_hash`: Optional PHE hash

**Returns:** Derived key (bytes)

---

## utils.tlv_format

### serialize_metadata_tlv()

```python
serialize_metadata_tlv(metadata: Dict[str, Any]) -> bytes
```

Serialize metadata to binary TLV format.

**Parameters:**
- `metadata`: Metadata dict

**Returns:** Binary TLV bytes

---

### deserialize_metadata_tlv()

```python
deserialize_metadata_tlv(tlv_bytes: bytes) -> Dict[str, Any]
```

Deserialize TLV bytes to metadata dict.

**Parameters:**
- `tlv_bytes`: Binary TLV data

**Returns:** Metadata dict

**Raises:**
- `ValueError`: If TLV format invalid
- `Exception`: If version unsupported

---

## Performance Metrics (v0.2.0)

Default parameters (`lattice_size=128, depth=6`):
- **Encryption**: ~1.3s for small messages
- **Decryption**: ~0.9s for small messages
- **Metadata Size**: ~786 KB (constant)
- **Hash Generation**: ~0.05s per hash

See [PERFORMANCE_OPTIMIZATIONS.md](../PERFORMANCE_OPTIMIZATIONS.md) for detailed benchmarks.

---

## Version Compatibility

### v0.2.0 Features

- Password-based encryption with MAC verification
- Binary TLV metadata format (~786 KB vs 4 MB)
- Metadata encryption with ephemeral keys
- Automatic version detection (v0.1.x JSON vs v0.2.0 TLV)
- 76x performance improvement over v0.2.0-alpha

### Backward Compatibility

```python
# v0.1.x JSON metadata still supported
ctx = STCContext('seed')
decrypted = ctx.decrypt(encrypted, old_json_metadata)

# v0.2.0 TLV metadata
decrypted = ctx.decrypt(encrypted, new_tlv_metadata)

# Auto-detection handles both
```

---

## Error Codes

| Error | Cause | Solution |
|-------|-------|----------|
| `ValueError: MAC verification failed` | Wrong password or data tampered | Check password, verify data integrity |
| `KeyError: 'vectors'` | Trying to use decoy vectors | Set `use_decoys=False` |
| `ValueError: Invalid lattice size` | lattice_size out of range (16-256) | Use valid range |
| `TypeError: data must be str or bytes` | Invalid data type | Convert to str or bytes |

---

## See Also

- [Usage Guide](usage-guide.md) - Practical examples and patterns
- [Architecture](architecture.md) - System design and components
- [CHANGELOG](../CHANGELOG.md) - Version history
- [PERFORMANCE_OPTIMIZATIONS](../PERFORMANCE_OPTIMIZATIONS.md) - Optimization details
