# Seigr Toolset Crypto (STC)

[![Sponsor Seigr-lab](https://img.shields.io/badge/Sponsor-Seigr--lab-forestgreen?style=flat&logo=github)](https://github.com/sponsors/Seigr-lab)
[![Version](https://img.shields.io/badge/version-0.2.1-blue)](https://github.com/Seigr-lab/SeigrToolsetCrypto/releases)
[![License](https://img.shields.io/badge/license-ANTI--CAPITALIST-red)](LICENSE)

**Post-classical cryptographic engine with entropy-regenerative architecture**

## Overview

STC v0.2.1 is a research-grade cryptographic system that rejects traditional symmetric/asymmetric paradigms. Instead of XOR-based mixing, static keys, and classical block ciphers, it implements:

- **Continuous Entropy Lattice (CEL)** - Self-evolving entropy field regenerated from computational deltas
- **Probabilistic Hashing Engine (PHE)** - Multi-path hashing with CEL-driven path selection
- **Contextual Key Emergence (CKE)** - Ephemeral keys reconstructed from context intersections
- **Data-State Folding (DSF)** - Encryption via multidimensional tensor folding
- **Polymorphic Cryptographic Flow (PCF)** - Dynamic algorithmic morphing
- **State Management** - Deterministic reconstruction with varint compression

## Architecture

```
core/
├── cel/    # Continuous Entropy Lattice
├── phe/    # Probabilistic Hashing Engine
├── cke/    # Contextual Key Emergence
├── dsf/    # Data-State Folding
├── pcf/    # Polymorphic Cryptographic Flow
└── state/  # State persistence and reconstruction

interfaces/
├── api/    # Programmatic interface
├── cli/    # Command-line tools
└── bindings/  # Future cross-language bindings

utils/      # Mathematical primitives + TLV varint encoding
tests/      # Validation and integrity checks
```

## ⚡ Performance (v0.2.1)

- **Encryption**: ~0.6s for small messages (2x faster than v0.2.0)
- **Decryption**: ~0.5s for small messages  
- **Metadata Size**: ~276 KB without decoys, ~414 KB with 3 decoys (65% smaller than v0.2.0)
- **Total Throughput**: ~1.2s for encrypt+decrypt cycle

> **Version History**:
> - v0.2.1: 1.95x faster than v0.2.0, 65% smaller metadata via varint encoding
> - v0.2.0: 76x faster than v0.1.0 via systematic optimizations
> 
> See [PERFORMANCE_OPTIMIZATIONS.md](PERFORMANCE_OPTIMIZATIONS.md) for optimization details and [CHANGELOG.md](CHANGELOG.md) for version history.

## Installation

### From PyPI (Recommended)

```bash
pip install seigr-toolset-crypto==0.2.1
```

### From GitHub Release

Download the latest release from [Releases](https://github.com/Seigr-lab/SeigrToolsetCrypto/releases):

```bash
# Install from wheel (recommended)
pip install seigr_toolset_crypto-0.2.1-py3-none-any.whl

# Or install from source tarball
pip install seigr_toolset_crypto-0.2.1.tar.gz
```

### From Source (Development)

```bash
git clone https://github.com/Seigr-lab/SeigrToolsetCrypto.git
cd SeigrToolsetCrypto
pip install -e .
```

### Requirements

- Python 3.9+
- NumPy 1.24.0+

## Quick Start

### Password-Based Encryption (v0.2.0)

```python
from interfaces.api.stc_api import STCContext

# Initialize context with seed
ctx = STCContext('my-unique-seed')

# Encrypt with password
encrypted, metadata = ctx.encrypt(
    "Secret message",
    password="strong_password"
)

# Decrypt with same password
decrypted = ctx.decrypt(encrypted, metadata, password="strong_password")
print(decrypted)  # "Secret message"
```

### Basic API (No Password)

```python
from interfaces.api import stc_api

# Initialize STC context
context = stc_api.initialize(seed="your-seed-phrase")

# Encrypt data (uses seed as password)
encrypted, metadata = context.encrypt("sensitive information")

# Decrypt data
decrypted = context.decrypt(encrypted, metadata)
print(decrypted)  # "sensitive information"

# Generate probabilistic hash
hash_result = context.hash("data to hash")
```

### Quick API (One-liners)

```python
from interfaces.api import stc_api

# Quick encrypt - returns encrypted data, metadata, and context
encrypted, metadata, context = stc_api.quick_encrypt(
    "sensitive data", 
    seed="your-seed"
)

# Quick decrypt - reconstructs context from metadata
decrypted = stc_api.quick_decrypt(
    encrypted, 
    metadata, 
    seed="your-seed"
)
```

## Usage

### Advanced: Custom Parameters

### Advanced: Custom Parameters

```python
from interfaces.api.stc_api import STCContext

# Custom lattice parameters (smaller = faster, less security)
context = STCContext(
    seed="your-seed",
    lattice_size=128,  # Default: 128 (was 256 in v0.1.x)
    depth=6,           # Default: 6 (was 8 in v0.1.x)
    morph_interval=100 # Polymorphic flow morphing interval
)

# Encrypt with custom context
encrypted, metadata = context.encrypt(
    "data",
    password="password123"
)

# Derive keys
key = context.derive_key(length=32)

# Hash data
hash_value = context.hash("data")
```

### State Management

```python
# Save context state
state = context.save_state()

# Load state (for resuming)
context.load_state(state)

# Get context status
status = context.get_status()
print(status)
```

## Features

### v0.2.0 Enhancements

- ✅ **Password-based encryption** with MAC verification
- ✅ **Metadata encryption** using ephemeral keys
- ✅ **Binary TLV format** for compact metadata storage
- ✅ **CEL entropy amplification** with timing chains and multi-tier feedback
- ✅ **PHE multi-path hashing** (3-5 parallel paths, CEL-driven)
- ✅ **Performance optimizations** (76x speedup from v0.2.0-alpha)
- ✅ **Entropy quality auditing** and collision monitoring
- ✅ **Backward compatibility** with v0.1.x JSON format

### Known Limitations

- **Metadata size**: ~786 KB constant overhead (reduced from 4 MB in alpha)
- **Decoy vectors**: Not yet supported (TLV serialization pending)
- **Performance**: 2.3s for small messages (acceptable for security-critical use)

See [CHANGELOG.md](CHANGELOG.md) for detailed version history and [PERFORMANCE_OPTIMIZATIONS.md](PERFORMANCE_OPTIMIZATIONS.md) for optimization details.

## Principles

1. **No legacy cryptography** - No XOR, no substitution-permutation networks, no block ciphers
2. **No external randomness** - All entropy from internal computation (timing deltas, state evolution)
3. **Deterministic from seed** - Same seed produces same initial state; state evolves with CEL dynamics
4. **Context-sensitive polymorphism** - Behavior adapts to operation chains
5. **Ephemeral keys** - Keys emerge and discard, never persist
6. **Self-sovereign security** - No cloud dependencies, no external services, full user control

## Examples

See `examples/` directory for practical demonstrations:

- **`password_manager/`** - Secure credential storage with password-based encryption (v0.2.0)
- **`config_encryption/`** - Configuration file encryption with metadata persistence
- **`validation/`** - Example validation scripts

Run examples:

```bash
cd examples/password_manager
python password_manager.py

cd examples/config_encryption
python config_example.py
```

## Testing

Run the full test suite:

```bash
# Run all tests
python -m unittest discover tests/

# Run specific test modules
python -m unittest tests.test_cel
python -m unittest tests.test_phe
python -m unittest tests.test_integration
```

All 37 tests pass in v0.2.0.

## Development Status

**v0.2.0** - Production-ready with known limitations (see above)

### Roadmap

- **v0.2.1**: Variable-length integer encoding (metadata → ~100-200 KB)
- **v0.2.2**: Decoy vector TLV serialization
- **v0.3.0**: Streaming support for large files, migration utilities

## Contributing

Contributions welcome! Please:
1. Fork the repository
2. Create a feature branch
3. Add tests for new features
4. Submit a pull request

See [docs/](docs/) for architecture and API documentation.

## License

ANTI-CAPITALIST SOFTWARE LICENSE (v 1.4) - See LICENSE file for details

---

## Citation

If you use STC in research, please cite:

```bibtex
@software{seigr_toolset_crypto,
  title = {Seigr Toolset Crypto: Post-Classical Cryptographic Engine},
  author = {Seigr-lab},
  year = {2025},
  version = {0.2.0},
  url = {https://github.com/Seigr-lab/SeigrToolsetCrypto}
}
```
