"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connections = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const peer_1 = require("./peer");
/**
 * Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 *
 * @stability stable
 */
class Connections {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ConnectionsProps(props);
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    /**
     * @stability stable
     */
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * Add a security group to the list of security groups managed by this object.
     *
     * @stability stable
     */
    addSecurityGroup(...securityGroups) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ISecurityGroup(securityGroups);
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * Allow connections to the peer on the given port.
     *
     * @stability stable
     */
    allowTo(other, portRange, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow connections from the peer on the given port.
     *
     * @stability stable
     */
    allowFrom(other, portRange, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * Allow hosts inside the security group to connect to each other on the given port.
     *
     * @stability stable
     */
    allowInternally(portRange, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * Allow to all IPv4 ranges.
     *
     * @stability stable
     */
    allowToAnyIpv4(portRange, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow from any IPv4 ranges.
     *
     * @stability stable
     */
    allowFromAnyIpv4(portRange, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_Port(portRange);
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * Allow connections from the peer on our default port.
     *
     * Even if the peer has a default port, we will always use our default port.
     *
     * @stability stable
     */
    allowDefaultPortFrom(other, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * Allow hosts inside the security group to connect to each other.
     *
     * @stability stable
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * Allow default connections from all IPv4 ranges.
     *
     * @stability stable
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * Allow connections to the security group on their default port.
     *
     * @stability stable
     */
    allowToDefaultPort(other, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call allowToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * Allow connections from the peer on our default port.
     *
     * Even if the peer has a default port, we will always use our default port.
     *
     * @stability stable
     */
    allowDefaultPortTo(other, description) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_IConnectable(other);
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
_a = JSII_RTTI_SYMBOL_1;
Connections[_a] = { fqn: "@aws-cdk/aws-ec2.Connections", version: "1.143.0" };
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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