"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientVpnEndpoint = exports.ClientVpnUserBasedAuthentication = exports.ClientVpnSessionTimeout = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const logs = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const client_vpn_authorization_rule_1 = require("./client-vpn-authorization-rule");
const client_vpn_route_1 = require("./client-vpn-route");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const network_util_1 = require("./network-util");
const security_group_1 = require("./security-group");
/**
 * Maximum VPN session duration time.
 *
 * @stability stable
 */
var ClientVpnSessionTimeout;
(function (ClientVpnSessionTimeout) {
    ClientVpnSessionTimeout[ClientVpnSessionTimeout["EIGHT_HOURS"] = 8] = "EIGHT_HOURS";
    ClientVpnSessionTimeout[ClientVpnSessionTimeout["TEN_HOURS"] = 10] = "TEN_HOURS";
    ClientVpnSessionTimeout[ClientVpnSessionTimeout["TWELVE_HOURS"] = 12] = "TWELVE_HOURS";
    ClientVpnSessionTimeout[ClientVpnSessionTimeout["TWENTY_FOUR_HOURS"] = 24] = "TWENTY_FOUR_HOURS";
})(ClientVpnSessionTimeout = exports.ClientVpnSessionTimeout || (exports.ClientVpnSessionTimeout = {}));
/**
 * User-based authentication for a client VPN endpoint.
 *
 * @stability stable
 */
class ClientVpnUserBasedAuthentication {
    /**
     * Active Directory authentication.
     *
     * @stability stable
     */
    static activeDirectory(directoryId) {
        return new ActiveDirectoryAuthentication(directoryId);
    }
    /**
     * Federated authentication.
     *
     * @stability stable
     */
    static federated(samlProvider, selfServiceSamlProvider) {
        return new FederatedAuthentication(samlProvider, selfServiceSamlProvider);
    }
}
exports.ClientVpnUserBasedAuthentication = ClientVpnUserBasedAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientVpnUserBasedAuthentication[_a] = { fqn: "@aws-cdk/aws-ec2.ClientVpnUserBasedAuthentication", version: "1.143.0" };
/**
 * Active Directory authentication
 */
class ActiveDirectoryAuthentication extends ClientVpnUserBasedAuthentication {
    constructor(directoryId) {
        super();
        this.directoryId = directoryId;
    }
    render() {
        return {
            type: 'directory-service-authentication',
            activeDirectory: { directoryId: this.directoryId },
        };
    }
}
/**
 * Federated authentication
 */
class FederatedAuthentication extends ClientVpnUserBasedAuthentication {
    constructor(samlProvider, selfServiceSamlProvider) {
        super();
        this.samlProvider = samlProvider;
        this.selfServiceSamlProvider = selfServiceSamlProvider;
    }
    render() {
        var _c;
        return {
            type: 'federated-authentication',
            federatedAuthentication: {
                samlProviderArn: this.samlProvider.samlProviderArn,
                selfServiceSamlProviderArn: (_c = this.selfServiceSamlProvider) === null || _c === void 0 ? void 0 : _c.samlProviderArn,
            },
        };
    }
}
/**
 * A client VPN connnection.
 *
 * @stability stable
 */
class ClientVpnEndpoint extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this._targetNetworksAssociated = new core_1.ConcreteDependable();
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnEndpointProps(props);
        if (!core_1.Token.isUnresolved(props.vpc.vpcCidrBlock)) {
            const clientCidr = new network_util_1.CidrBlock(props.cidr);
            const vpcCidr = new network_util_1.CidrBlock(props.vpc.vpcCidrBlock);
            if (vpcCidr.containsCidr(clientCidr)) {
                throw new Error('The client CIDR cannot overlap with the local CIDR of the VPC');
            }
        }
        if (props.dnsServers && props.dnsServers.length > 2) {
            throw new Error('A client VPN endpoint can have up to two DNS servers');
        }
        if (props.logging == false && (props.logGroup || props.logStream)) {
            throw new Error('Cannot specify `logGroup` or `logStream` when logging is disabled');
        }
        if (props.clientConnectionHandler
            && !core_1.Token.isUnresolved(props.clientConnectionHandler.functionName)
            && !props.clientConnectionHandler.functionName.startsWith('AWSClientVPN-')) {
            throw new Error('The name of the Lambda function must begin with the `AWSClientVPN-` prefix');
        }
        if (props.clientLoginBanner
            && !core_1.Token.isUnresolved(props.clientLoginBanner)
            && props.clientLoginBanner.length > 1400) {
            throw new Error(`The maximum length for the client login banner is 1400, got ${props.clientLoginBanner.length}`);
        }
        const logging = (_c = props.logging) !== null && _c !== void 0 ? _c : true;
        const logGroup = logging
            ? (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup') : undefined;
        const securityGroups = (_e = props.securityGroups) !== null && _e !== void 0 ? _e : [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.connections = new connections_1.Connections({ securityGroups });
        const endpoint = new ec2_generated_1.CfnClientVpnEndpoint(this, 'Resource', {
            authenticationOptions: renderAuthenticationOptions(props.clientCertificateArn, props.userBasedAuthentication),
            clientCidrBlock: props.cidr,
            clientConnectOptions: props.clientConnectionHandler
                ? {
                    enabled: true,
                    lambdaFunctionArn: props.clientConnectionHandler.functionArn,
                }
                : undefined,
            connectionLogOptions: {
                enabled: logging,
                cloudwatchLogGroup: logGroup === null || logGroup === void 0 ? void 0 : logGroup.logGroupName,
                cloudwatchLogStream: (_f = props.logStream) === null || _f === void 0 ? void 0 : _f.logStreamName,
            },
            description: props.description,
            dnsServers: props.dnsServers,
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            selfServicePortal: booleanToEnabledDisabled(props.selfServicePortal),
            serverCertificateArn: props.serverCertificateArn,
            splitTunnel: props.splitTunnel,
            transportProtocol: props.transportProtocol,
            vpcId: props.vpc.vpcId,
            vpnPort: props.port,
            sessionTimeoutHours: props.sessionTimeout,
            clientLoginBannerOptions: props.clientLoginBanner
                ? {
                    enabled: true,
                    bannerText: props.clientLoginBanner,
                }
                : undefined,
        });
        this.endpointId = endpoint.ref;
        if (props.userBasedAuthentication && ((_g = props.selfServicePortal) !== null && _g !== void 0 ? _g : true)) {
            // Output self-service portal URL
            new core_1.CfnOutput(this, 'SelfServicePortalUrl', {
                value: `https://self-service.clientvpn.amazonaws.com/endpoints/${this.endpointId}`,
            });
        }
        // Associate subnets
        const subnetIds = props.vpc.selectSubnets(props.vpcSubnets).subnetIds;
        if (core_1.Token.isUnresolved(subnetIds)) {
            throw new Error('Cannot associate subnets when VPC are imported from parameters or exports containing lists of subnet IDs.');
        }
        for (const [idx, subnetId] of Object.entries(subnetIds)) {
            this._targetNetworksAssociated.add(new ec2_generated_1.CfnClientVpnTargetNetworkAssociation(this, `Association${idx}`, {
                clientVpnEndpointId: this.endpointId,
                subnetId,
            }));
        }
        this.targetNetworksAssociated = this._targetNetworksAssociated;
        if ((_h = props.authorizeAllUsersToVpcCidr) !== null && _h !== void 0 ? _h : true) {
            this.addAuthorizationRule('AuthorizeAll', {
                cidr: props.vpc.vpcCidrBlock,
            });
        }
    }
    /**
     * Import an existing client VPN endpoint.
     *
     * @stability stable
     */
    static fromEndpointAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnEndpointAttributes(attrs);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.endpointId = attrs.endpointId;
                this.connections = new connections_1.Connections({ securityGroups: attrs.securityGroups });
                this.targetNetworksAssociated = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an authorization rule to this endpoint.
     *
     * @stability stable
     */
    addAuthorizationRule(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnAuthorizationRuleOptions(props);
        return new client_vpn_authorization_rule_1.ClientVpnAuthorizationRule(this, id, {
            ...props,
            clientVpnEndpoint: this,
        });
    }
    /**
     * Adds a route to this endpoint.
     *
     * @stability stable
     */
    addRoute(id, props) {
        jsiiDeprecationWarnings._aws_cdk_aws_ec2_ClientVpnRouteOptions(props);
        return new client_vpn_route_1.ClientVpnRoute(this, id, {
            ...props,
            clientVpnEndpoint: this,
        });
    }
}
exports.ClientVpnEndpoint = ClientVpnEndpoint;
_b = JSII_RTTI_SYMBOL_1;
ClientVpnEndpoint[_b] = { fqn: "@aws-cdk/aws-ec2.ClientVpnEndpoint", version: "1.143.0" };
function renderAuthenticationOptions(clientCertificateArn, userBasedAuthentication) {
    const authenticationOptions = [];
    if (clientCertificateArn) {
        authenticationOptions.push({
            type: 'certificate-authentication',
            mutualAuthentication: {
                clientRootCertificateChainArn: clientCertificateArn,
            },
        });
    }
    if (userBasedAuthentication) {
        authenticationOptions.push(userBasedAuthentication.render());
    }
    if (authenticationOptions.length === 0) {
        throw new Error('A client VPN endpoint must use at least one authentication option');
    }
    return authenticationOptions;
}
function booleanToEnabledDisabled(val) {
    switch (val) {
        case undefined:
            return undefined;
        case true:
            return 'enabled';
        case false:
            return 'disabled';
    }
}
//# sourceMappingURL=data:application/json;base64,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