# Cadence

A minimalist rhythm analyzer. So far, it's a simplified version of [Prosodic](https://github.com/quadrismegistus/prosodic) for English and Finnish poetic scansion. It's more thorough (returns all parses, not just the non-harmonically bounded ones) but slower.

## Quickstart

You can also run this quickstart as an [interactive notebook on Colab](https://colab.research.google.com/github/quadrismegistus/cadence/blob/main/README.ipynb).

### Setup

Install:

```
pip install -U git+https://github.com/quadrismegistus/cadence
```

Import in python:

```python
# Import
import cadence as cd
```


### Load text


```python
# grab a string of a poem or text

milton="""
OF Mans First Disobedience, and the Fruit
Of that Forbidden Tree, whose mortal tast
Brought Death into the World, and all our woe
"""
```


```python
# scan text (syllabify, stress), return as dataframe

txtdf = cd.scan(milton)
txtdf
```



<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>is_funcword</th>
      <th>is_heavy</th>
      <th>is_light</th>
      <th>is_peak</th>
      <th>is_stressed</th>
      <th>is_syll</th>
      <th>is_trough</th>
      <th>is_unstressed</th>
      <th>prom_strength</th>
      <th>prom_stress</th>
      <th>prom_weight</th>
    </tr>
    <tr>
      <th>stanza_i</th>
      <th>line_i</th>
      <th>line_str</th>
      <th>word_i</th>
      <th>word_str</th>
      <th>word_ipa_i</th>
      <th>word_ipa</th>
      <th>syll_i</th>
      <th>syll_str</th>
      <th>syll_ipa</th>
      <th>syll_stress</th>
      <th>syll_weight</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th rowspan="44" valign="top">1</th>
      <th rowspan="12" valign="top">1</th>
      <th rowspan="12" valign="top">OF Mans First Disobedience, and the Fruit</th>
      <th>1</th>
      <th>OF</th>
      <th>1</th>
      <th>ʌv</th>
      <th>1</th>
      <th>OF</th>
      <th>ʌv</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>Mans</th>
      <th>1</th>
      <th>'mænz</th>
      <th>1</th>
      <th>Mans</th>
      <th>'mænz</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>First</th>
      <th>1</th>
      <th>'fɛːst</th>
      <th>1</th>
      <th>First</th>
      <th>'fɛːst</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="5" valign="top">4</th>
      <th rowspan="5" valign="top">Disobedience</th>
      <th rowspan="5" valign="top">1</th>
      <th rowspan="5" valign="top">`dɪ.sə.'biː.diː.əns</th>
      <th>1</th>
      <th>Di</th>
      <th>`dɪ</th>
      <th>S</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>0.5</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>so</th>
      <th>sə</th>
      <th>U</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>be</th>
      <th>'biː</th>
      <th>P</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>dien</th>
      <th>diː</th>
      <th>U</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>ce</th>
      <th>əns</th>
      <th>U</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>,</th>
      <th>0</th>
      <th></th>
      <th>0</th>
      <th>,</th>
      <th></th>
      <th></th>
      <th></th>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0.0</td>
      <td>NaN</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th>6</th>
      <th>and</th>
      <th>1</th>
      <th>ænd</th>
      <th>1</th>
      <th>and</th>
      <th>ænd</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>7</th>
      <th>the</th>
      <th>1</th>
      <th>ðə</th>
      <th>1</th>
      <th>the</th>
      <th>ðə</th>
      <th>U</th>
      <th>L</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>Fruit</th>
      <th>1</th>
      <th>'fruːt</th>
      <th>1</th>
      <th>Fruit</th>
      <th>'fruːt</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="16" valign="top">2</th>
      <th rowspan="16" valign="top">Of that Forbidden Tree, whose mortal tast</th>
      <th>1</th>
      <th>Of</th>
      <th>1</th>
      <th>ʌv</th>
      <th>1</th>
      <th>Of</th>
      <th>ʌv</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">2</th>
      <th rowspan="2" valign="top">that</th>
      <th>1</th>
      <th>'ðæt</th>
      <th>1</th>
      <th>that</th>
      <th>'ðæt</th>
      <th>P</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1.0</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>ðət</th>
      <th>1</th>
      <th>that</th>
      <th>ðət</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="6" valign="top">3</th>
      <th rowspan="6" valign="top">Forbidden</th>
      <th rowspan="3" valign="top">1</th>
      <th rowspan="3" valign="top">'fɔːr.bɪ.dən</th>
      <th>1</th>
      <th>For</th>
      <th>'fɔːr</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>bid</th>
      <th>bɪ</th>
      <th>U</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1.0</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>den</th>
      <th>dən</th>
      <th>U</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="3" valign="top">2</th>
      <th rowspan="3" valign="top">fɛːr.'bɪ.dən</th>
      <th>1</th>
      <th>For</th>
      <th>fɛːr</th>
      <th>U</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>bid</th>
      <th>'bɪ</th>
      <th>P</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>den</th>
      <th>dən</th>
      <th>U</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>Tree</th>
      <th>1</th>
      <th>'triː</th>
      <th>1</th>
      <th>Tree</th>
      <th>'triː</th>
      <th>P</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>,</th>
      <th>0</th>
      <th></th>
      <th>0</th>
      <th>,</th>
      <th></th>
      <th></th>
      <th></th>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1.0</td>
      <td>NaN</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">6</th>
      <th rowspan="2" valign="top">whose</th>
      <th>1</th>
      <th>'huːz</th>
      <th>1</th>
      <th>whose</th>
      <th>'huːz</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>huːz</th>
      <th>1</th>
      <th>whose</th>
      <th>huːz</th>
      <th>U</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">7</th>
      <th rowspan="2" valign="top">mortal</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">'mɔːr.təl</th>
      <th>1</th>
      <th>mor</th>
      <th>'mɔːr</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>tal</th>
      <th>təl</th>
      <th>U</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>tast</th>
      <th>1</th>
      <th>'teɪst</th>
      <th>1</th>
      <th>tast</th>
      <th>'teɪst</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="16" valign="top">3</th>
      <th rowspan="16" valign="top">Brought Death into the World, and all our woe</th>
      <th>1</th>
      <th>Brought</th>
      <th>1</th>
      <th>'brɔːt</th>
      <th>1</th>
      <th>Brought</th>
      <th>'brɔːt</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>Death</th>
      <th>1</th>
      <th>'dɛθ</th>
      <th>1</th>
      <th>Death</th>
      <th>'dɛθ</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="6" valign="top">3</th>
      <th rowspan="6" valign="top">into</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">ɪn.'tuː</th>
      <th>1</th>
      <th>in</th>
      <th>ɪn</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>to</th>
      <th>'tuː</th>
      <th>P</th>
      <th>L</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">2</th>
      <th rowspan="2" valign="top">'ɪn.tuː</th>
      <th>1</th>
      <th>in</th>
      <th>'ɪn</th>
      <th>P</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>to</th>
      <th>tuː</th>
      <th>U</th>
      <th>L</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">3</th>
      <th rowspan="2" valign="top">ɪn.tʌ</th>
      <th>1</th>
      <th>in</th>
      <th>ɪn</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>to</th>
      <th>tʌ</th>
      <th>U</th>
      <th>L</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>the</th>
      <th>1</th>
      <th>ðə</th>
      <th>1</th>
      <th>the</th>
      <th>ðə</th>
      <th>U</th>
      <th>L</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>World</th>
      <th>1</th>
      <th>'wɛːld</th>
      <th>1</th>
      <th>World</th>
      <th>'wɛːld</th>
      <th>P</th>
      <th>H</th>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>6</th>
      <th>,</th>
      <th>0</th>
      <th></th>
      <th>0</th>
      <th>,</th>
      <th></th>
      <th></th>
      <th></th>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>NaN</td>
    </tr>
    <tr>
      <th>7</th>
      <th>and</th>
      <th>1</th>
      <th>ænd</th>
      <th>1</th>
      <th>and</th>
      <th>ænd</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">8</th>
      <th rowspan="2" valign="top">all</th>
      <th>1</th>
      <th>'ɔːl</th>
      <th>1</th>
      <th>all</th>
      <th>'ɔːl</th>
      <th>P</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>ɔːl</th>
      <th>1</th>
      <th>all</th>
      <th>ɔːl</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>9</th>
      <th>our</th>
      <th>1</th>
      <th>aʊr</th>
      <th>1</th>
      <th>our</th>
      <th>aʊr</th>
      <th>U</th>
      <th>H</th>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>10</th>
      <th>woe</th>
      <th>1</th>
      <th>'woʊ</th>
      <th>1</th>
      <th>woe</th>
      <th>'woʊ</th>
      <th>P</th>
      <th>L</th>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
  </tbody>
</table>
</div>



### Metrical parsing


```python
# Parse metrically, return as dataframe by syllable, sorted from best to worst parses

parses_bysyll = cd.parse(txtdf)
parses_bysyll
```

    Metrically parsing lines (+word IPA combinations) [x7]: 100%|██████████| 15/15 [00:12<00:00,  1.24it/s]






<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>*total</th>
      <th>*clash</th>
      <th>*s/unstressed</th>
      <th>*w-res</th>
      <th>*f-res</th>
      <th>*w/peak</th>
      <th>*lapse</th>
      <th>*w/stressed</th>
      <th>combo_num_syll</th>
      <th>is_funcword</th>
      <th>is_heavy</th>
      <th>is_light</th>
      <th>is_peak</th>
      <th>is_s</th>
      <th>is_stressed</th>
      <th>is_syll</th>
      <th>is_trough</th>
      <th>is_unstressed</th>
      <th>is_w</th>
      <th>parse_num_pos</th>
      <th>parse_num_syll</th>
      <th>prom_strength</th>
      <th>prom_stress</th>
      <th>prom_weight</th>
    </tr>
    <tr>
      <th>stanza_i</th>
      <th>line_i</th>
      <th>line_str</th>
      <th>parse_rank</th>
      <th>combo_i</th>
      <th>combo_stress</th>
      <th>combo_ipa</th>
      <th>parse_i</th>
      <th>parse</th>
      <th>parse_str</th>
      <th>parse_pos_i</th>
      <th>parse_pos</th>
      <th>word_i</th>
      <th>word_str</th>
      <th>word_ipa_i</th>
      <th>word_ipa</th>
      <th>syll_i</th>
      <th>combo_syll_i</th>
      <th>syll_str</th>
      <th>syll_ipa</th>
      <th>syll_stress</th>
      <th>syll_weight</th>
      <th>parse_syll_i</th>
      <th>parse_syll</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th rowspan="11" valign="top">1</th>
      <th rowspan="5" valign="top">1</th>
      <th rowspan="5" valign="top">OF Mans First Disobedience, and the Fruit</th>
      <th rowspan="5" valign="top">1</th>
      <th rowspan="5" valign="top">0</th>
      <th rowspan="5" valign="top">UPPSUPUUUUP</th>
      <th rowspan="5" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th rowspan="5" valign="top">5</th>
      <th rowspan="5" valign="top">wswswswwsws</th>
      <th rowspan="5" valign="top">of|MANS|first|DI|so|BE|dien.ce|AND|the|FRUIT</th>
      <th>0</th>
      <th>w</th>
      <th>1</th>
      <th>OF</th>
      <th>1</th>
      <th>ʌv</th>
      <th>1</th>
      <th>0</th>
      <th>OF</th>
      <th>ʌv</th>
      <th>U</th>
      <th>H</th>
      <th>0</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>1</th>
      <th>s</th>
      <th>2</th>
      <th>Mans</th>
      <th>1</th>
      <th>'mænz</th>
      <th>1</th>
      <th>1</th>
      <th>Mans</th>
      <th>'mænz</th>
      <th>P</th>
      <th>H</th>
      <th>1</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>w</th>
      <th>3</th>
      <th>First</th>
      <th>1</th>
      <th>'fɛːst</th>
      <th>1</th>
      <th>2</th>
      <th>First</th>
      <th>'fɛːst</th>
      <th>P</th>
      <th>H</th>
      <th>2</th>
      <th>w</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>s</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>1</th>
      <th>3</th>
      <th>Di</th>
      <th>`dɪ</th>
      <th>S</th>
      <th>L</th>
      <th>3</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.5</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>w</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>2</th>
      <th>4</th>
      <th>so</th>
      <th>sə</th>
      <th>U</th>
      <th>L</th>
      <th>4</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
    </tr>
    <tr>
      <th rowspan="5" valign="top">3</th>
      <th rowspan="5" valign="top">Brought Death into the World, and all our woe</th>
      <th rowspan="5" valign="top">1068</th>
      <th rowspan="5" valign="top">3</th>
      <th rowspan="5" valign="top">PPPUUPUUUP</th>
      <th rowspan="5" valign="top">'brɔːt 'dɛθ 'ɪn.tuː ðə 'wɛːld ænd ɔːl aʊr 'woʊ</th>
      <th rowspan="5" valign="top">143</th>
      <th rowspan="5" valign="top">swwsswwssw</th>
      <th rowspan="5" valign="top">BROUGHT|death.in|TO.THE|world.and|ALL.OUR|woe</th>
      <th rowspan="2" valign="top">3</th>
      <th rowspan="2" valign="top">ww</th>
      <th>5</th>
      <th>World</th>
      <th>1</th>
      <th>'wɛːld</th>
      <th>1</th>
      <th>5</th>
      <th>World</th>
      <th>'wɛːld</th>
      <th>P</th>
      <th>H</th>
      <th>5</th>
      <th>w</th>
      <td>2.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>10</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>6</td>
      <td>10</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>7</th>
      <th>and</th>
      <th>1</th>
      <th>ænd</th>
      <th>1</th>
      <th>6</th>
      <th>and</th>
      <th>ænd</th>
      <th>U</th>
      <th>H</th>
      <th>6</th>
      <th>w</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>6</td>
      <td>10</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">4</th>
      <th rowspan="2" valign="top">ss</th>
      <th>8</th>
      <th>all</th>
      <th>2</th>
      <th>ɔːl</th>
      <th>1</th>
      <th>7</th>
      <th>all</th>
      <th>ɔːl</th>
      <th>U</th>
      <th>H</th>
      <th>7</th>
      <th>s</th>
      <td>4.0</td>
      <td>0</td>
      <td>1</td>
      <td>1.0</td>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>10</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>6</td>
      <td>10</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>9</th>
      <th>our</th>
      <th>1</th>
      <th>aʊr</th>
      <th>1</th>
      <th>8</th>
      <th>our</th>
      <th>aʊr</th>
      <th>U</th>
      <th>H</th>
      <th>8</th>
      <th>s</th>
      <td>4.0</td>
      <td>0</td>
      <td>1</td>
      <td>1.0</td>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>10</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>6</td>
      <td>10</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>w</th>
      <th>10</th>
      <th>woe</th>
      <th>1</th>
      <th>'woʊ</th>
      <th>1</th>
      <th>9</th>
      <th>woe</th>
      <th>'woʊ</th>
      <th>P</th>
      <th>L</th>
      <th>9</th>
      <th>w</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>10</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>6</td>
      <td>10</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
  </tbody>
</table>
<p>28088 rows × 24 columns</p>
</div>




```python
# Summarize syllable-level parses up to lines (summing violation counts, etc)

parses_byline = cd.to_lines(parses_bysyll)      # set keyword `agg` to `np.mean` to average instead
parses_byline
```



<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>*total</th>
      <th>*clash</th>
      <th>*s/unstressed</th>
      <th>*w-res</th>
      <th>*f-res</th>
      <th>*w/stressed</th>
      <th>*lapse</th>
      <th>*w/peak</th>
      <th>combo_num_syll</th>
      <th>is_funcword</th>
      <th>is_heavy</th>
      <th>is_light</th>
      <th>is_peak</th>
      <th>is_s</th>
      <th>is_stressed</th>
      <th>is_syll</th>
      <th>is_trough</th>
      <th>is_unstressed</th>
      <th>is_w</th>
      <th>parse_num_pos</th>
      <th>parse_num_syll</th>
      <th>prom_strength</th>
      <th>prom_stress</th>
      <th>prom_weight</th>
    </tr>
    <tr>
      <th>stanza_i</th>
      <th>line_i</th>
      <th>line_str</th>
      <th>parse_rank</th>
      <th>combo_i</th>
      <th>combo_stress</th>
      <th>combo_ipa</th>
      <th>parse_i</th>
      <th>parse</th>
      <th>parse_str</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th rowspan="11" valign="top">1</th>
      <th rowspan="5" valign="top">1</th>
      <th rowspan="5" valign="top">OF Mans First Disobedience, and the Fruit</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">0</th>
      <th rowspan="2" valign="top">UPPSUPUUUUP</th>
      <th rowspan="2" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>5</th>
      <th>wswswswwsws</th>
      <th>of|MANS|first|DI|so|BE|dien.ce|AND|the|FRUIT</th>
      <td>3.0</td>
      <td>0</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>wswswsswsws</th>
      <th>of|MANS|first|DI|so|BE.DIEN|ce|AND|the|FRUIT</th>
      <td>3.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>6</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>5</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">3</th>
      <th rowspan="2" valign="top">0</th>
      <th rowspan="2" valign="top">UPPSUPUUUUP</th>
      <th rowspan="2" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>2</th>
      <th>wswswswswws</th>
      <th>of|MANS|first|DI|so|BE|dien|CE|and.the|FRUIT</th>
      <td>4.0</td>
      <td>0</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>13</th>
      <th>wswswwswsws</th>
      <th>of|MANS|first|DI|so.be|DIEN|ce|AND|the|FRUIT</th>
      <td>4.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>0</th>
      <th>UPPSUPUUUUP</th>
      <th>ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>0</th>
      <th>wswswswswsw</th>
      <th>of|MANS|first|DI|so|BE|dien|CE|and|THE|fruit</th>
      <td>5.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>0</td>
      <td>1</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>11</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <th>...</th>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
      <td>...</td>
    </tr>
    <tr>
      <th rowspan="5" valign="top">3</th>
      <th rowspan="5" valign="top">Brought Death into the World, and all our woe</th>
      <th rowspan="4" valign="top">1061</th>
      <th rowspan="4" valign="top">5</th>
      <th rowspan="4" valign="top">PPUUUPUUUP</th>
      <th rowspan="4" valign="top">'brɔːt 'dɛθ ɪn.tʌ ðə 'wɛːld ænd ɔːl aʊr 'woʊ</th>
      <th>65</th>
      <th>wwswsswssw</th>
      <th>brought.death|IN|to|THE.WORLD|and|ALL.OUR|woe</th>
      <td>21.0</td>
      <td>1</td>
      <td>4</td>
      <td>4.0</td>
      <td>6.0</td>
      <td>3</td>
      <td>3</td>
      <td>0</td>
      <td>10</td>
      <td>6.0</td>
      <td>7</td>
      <td>3</td>
      <td>0</td>
      <td>5</td>
      <td>4</td>
      <td>10</td>
      <td>0</td>
      <td>6</td>
      <td>5</td>
      <td>7</td>
      <td>10</td>
      <td>0.0</td>
      <td>4.0</td>
      <td>7.0</td>
    </tr>
    <tr>
      <th>88</th>
      <th>wwsswwssww</th>
      <th>brought.death|IN.TO|the.world|AND.ALL|our.woe</th>
      <td>21.0</td>
      <td>1</td>
      <td>4</td>
      <td>1.0</td>
      <td>7.0</td>
      <td>4</td>
      <td>4</td>
      <td>0</td>
      <td>10</td>
      <td>6.0</td>
      <td>7</td>
      <td>3</td>
      <td>0</td>
      <td>4</td>
      <td>4</td>
      <td>10</td>
      <td>0</td>
      <td>6</td>
      <td>6</td>
      <td>5</td>
      <td>10</td>
      <td>0.0</td>
      <td>4.0</td>
      <td>7.0</td>
    </tr>
    <tr>
      <th>143</th>
      <th>swwsswwssw</th>
      <th>BROUGHT|death.in|TO.THE|world.and|ALL.OUR|woe</th>
      <td>21.0</td>
      <td>0</td>
      <td>4</td>
      <td>3.0</td>
      <td>7.0</td>
      <td>3</td>
      <td>4</td>
      <td>0</td>
      <td>10</td>
      <td>6.0</td>
      <td>7</td>
      <td>3</td>
      <td>0</td>
      <td>5</td>
      <td>4</td>
      <td>10</td>
      <td>0</td>
      <td>6</td>
      <td>5</td>
      <td>6</td>
      <td>10</td>
      <td>0.0</td>
      <td>4.0</td>
      <td>7.0</td>
    </tr>
    <tr>
      <th>164</th>
      <th>sswsswwssw</th>
      <th>BROUGHT.DEATH|in|TO.THE|world.and|ALL.OUR|woe</th>
      <td>21.0</td>
      <td>1</td>
      <td>4</td>
      <td>3.0</td>
      <td>7.0</td>
      <td>2</td>
      <td>4</td>
      <td>0</td>
      <td>10</td>
      <td>6.0</td>
      <td>7</td>
      <td>3</td>
      <td>0</td>
      <td>6</td>
      <td>4</td>
      <td>10</td>
      <td>0</td>
      <td>6</td>
      <td>4</td>
      <td>6</td>
      <td>10</td>
      <td>0.0</td>
      <td>4.0</td>
      <td>7.0</td>
    </tr>
    <tr>
      <th>1068</th>
      <th>3</th>
      <th>PPPUUPUUUP</th>
      <th>'brɔːt 'dɛθ 'ɪn.tuː ðə 'wɛːld ænd ɔːl aʊr 'woʊ</th>
      <th>143</th>
      <th>swwsswwssw</th>
      <th>BROUGHT|death.in|TO.THE|world.and|ALL.OUR|woe</th>
      <td>22.0</td>
      <td>1</td>
      <td>4</td>
      <td>3.0</td>
      <td>7.0</td>
      <td>4</td>
      <td>3</td>
      <td>0</td>
      <td>10</td>
      <td>6.0</td>
      <td>7</td>
      <td>3</td>
      <td>0</td>
      <td>5</td>
      <td>5</td>
      <td>10</td>
      <td>0</td>
      <td>5</td>
      <td>5</td>
      <td>6</td>
      <td>10</td>
      <td>0.0</td>
      <td>5.0</td>
      <td>7.0</td>
    </tr>
  </tbody>
</table>
<p>2780 rows × 24 columns</p>
</div>



### Metrical analysis


```python
# Show best parse(s) for each line (parses can be tied for first)

parses_byline.query('parse_rank==1')
```



<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>*total</th>
      <th>*clash</th>
      <th>*s/unstressed</th>
      <th>*w-res</th>
      <th>*f-res</th>
      <th>*w/stressed</th>
      <th>*lapse</th>
      <th>*w/peak</th>
      <th>combo_num_syll</th>
      <th>is_funcword</th>
      <th>is_heavy</th>
      <th>is_light</th>
      <th>is_peak</th>
      <th>is_s</th>
      <th>is_stressed</th>
      <th>is_syll</th>
      <th>is_trough</th>
      <th>is_unstressed</th>
      <th>is_w</th>
      <th>parse_num_pos</th>
      <th>parse_num_syll</th>
      <th>prom_strength</th>
      <th>prom_stress</th>
      <th>prom_weight</th>
    </tr>
    <tr>
      <th>stanza_i</th>
      <th>line_i</th>
      <th>line_str</th>
      <th>parse_rank</th>
      <th>combo_i</th>
      <th>combo_stress</th>
      <th>combo_ipa</th>
      <th>parse_i</th>
      <th>parse</th>
      <th>parse_str</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th rowspan="4" valign="top">1</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">OF Mans First Disobedience, and the Fruit</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">0</th>
      <th rowspan="2" valign="top">UPPSUPUUUUP</th>
      <th rowspan="2" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>5</th>
      <th>wswswswwsws</th>
      <th>of|MANS|first|DI|so|BE|dien.ce|AND|the|FRUIT</th>
      <td>3.0</td>
      <td>0</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>wswswsswsws</th>
      <th>of|MANS|first|DI|so|BE.DIEN|ce|AND|the|FRUIT</th>
      <td>3.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>6</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>5</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>Of that Forbidden Tree, whose mortal tast</th>
      <th>1</th>
      <th>3</th>
      <th>UPUPUPUPUP</th>
      <th>ʌv 'ðæt fɛːr.'bɪ.dən 'triː huːz 'mɔːr.təl 'teɪst</th>
      <th>0</th>
      <th>wswswswsws</th>
      <th>of|THAT|for|BID|den|TREE|whose|MOR|tal|TAST</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>10</td>
      <td>2.0</td>
      <td>8</td>
      <td>2</td>
      <td>2</td>
      <td>5</td>
      <td>5</td>
      <td>10</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>10</td>
      <td>10</td>
      <td>2.0</td>
      <td>5.0</td>
      <td>8.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>Brought Death into the World, and all our woe</th>
      <th>1</th>
      <th>0</th>
      <th>PPUPUPUPUP</th>
      <th>'brɔːt 'dɛθ ɪn.'tuː ðə 'wɛːld ænd 'ɔːl aʊr 'woʊ</th>
      <th>0</th>
      <th>wswswswsws</th>
      <th>brought|DEATH|in|TO|the|WORLD|and|ALL|our|WOE</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>6.0</td>
      <td>7</td>
      <td>3</td>
      <td>0</td>
      <td>5</td>
      <td>6</td>
      <td>10</td>
      <td>0</td>
      <td>4</td>
      <td>5</td>
      <td>10</td>
      <td>10</td>
      <td>0.0</td>
      <td>6.0</td>
      <td>7.0</td>
    </tr>
  </tbody>
</table>
</div>




```python
# Show best parse(s) for line 1, syllable-level information

parses_bysyll.query('line_i==1 and parse_rank==1')
```



<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>*total</th>
      <th>*clash</th>
      <th>*s/unstressed</th>
      <th>*w-res</th>
      <th>*f-res</th>
      <th>*w/peak</th>
      <th>*lapse</th>
      <th>*w/stressed</th>
      <th>combo_num_syll</th>
      <th>is_funcword</th>
      <th>is_heavy</th>
      <th>is_light</th>
      <th>is_peak</th>
      <th>is_s</th>
      <th>is_stressed</th>
      <th>is_syll</th>
      <th>is_trough</th>
      <th>is_unstressed</th>
      <th>is_w</th>
      <th>parse_num_pos</th>
      <th>parse_num_syll</th>
      <th>prom_strength</th>
      <th>prom_stress</th>
      <th>prom_weight</th>
    </tr>
    <tr>
      <th>stanza_i</th>
      <th>line_i</th>
      <th>line_str</th>
      <th>parse_rank</th>
      <th>combo_i</th>
      <th>combo_stress</th>
      <th>combo_ipa</th>
      <th>parse_i</th>
      <th>parse</th>
      <th>parse_str</th>
      <th>parse_pos_i</th>
      <th>parse_pos</th>
      <th>word_i</th>
      <th>word_str</th>
      <th>word_ipa_i</th>
      <th>word_ipa</th>
      <th>syll_i</th>
      <th>combo_syll_i</th>
      <th>syll_str</th>
      <th>syll_ipa</th>
      <th>syll_stress</th>
      <th>syll_weight</th>
      <th>parse_syll_i</th>
      <th>parse_syll</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th rowspan="22" valign="top">1</th>
      <th rowspan="22" valign="top">1</th>
      <th rowspan="22" valign="top">OF Mans First Disobedience, and the Fruit</th>
      <th rowspan="22" valign="top">1</th>
      <th rowspan="22" valign="top">0</th>
      <th rowspan="22" valign="top">UPPSUPUUUUP</th>
      <th rowspan="22" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th rowspan="11" valign="top">5</th>
      <th rowspan="11" valign="top">wswswswwsws</th>
      <th rowspan="11" valign="top">of|MANS|first|DI|so|BE|dien.ce|AND|the|FRUIT</th>
      <th>0</th>
      <th>w</th>
      <th>1</th>
      <th>OF</th>
      <th>1</th>
      <th>ʌv</th>
      <th>1</th>
      <th>0</th>
      <th>OF</th>
      <th>ʌv</th>
      <th>U</th>
      <th>H</th>
      <th>0</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>1</th>
      <th>s</th>
      <th>2</th>
      <th>Mans</th>
      <th>1</th>
      <th>'mænz</th>
      <th>1</th>
      <th>1</th>
      <th>Mans</th>
      <th>'mænz</th>
      <th>P</th>
      <th>H</th>
      <th>1</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>w</th>
      <th>3</th>
      <th>First</th>
      <th>1</th>
      <th>'fɛːst</th>
      <th>1</th>
      <th>2</th>
      <th>First</th>
      <th>'fɛːst</th>
      <th>P</th>
      <th>H</th>
      <th>2</th>
      <th>w</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>s</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>1</th>
      <th>3</th>
      <th>Di</th>
      <th>`dɪ</th>
      <th>S</th>
      <th>L</th>
      <th>3</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.5</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>w</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>2</th>
      <th>4</th>
      <th>so</th>
      <th>sə</th>
      <th>U</th>
      <th>L</th>
      <th>4</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>s</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>3</th>
      <th>5</th>
      <th>be</th>
      <th>'biː</th>
      <th>P</th>
      <th>L</th>
      <th>5</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">6</th>
      <th rowspan="2" valign="top">ww</th>
      <th rowspan="2" valign="top">4</th>
      <th rowspan="2" valign="top">Disobedience</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">`dɪ.sə.'biː.diː.əns</th>
      <th>4</th>
      <th>6</th>
      <th>dien</th>
      <th>diː</th>
      <th>U</th>
      <th>L</th>
      <th>6</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>5</th>
      <th>7</th>
      <th>ce</th>
      <th>əns</th>
      <th>U</th>
      <th>H</th>
      <th>7</th>
      <th>w</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>7</th>
      <th>s</th>
      <th>6</th>
      <th>and</th>
      <th>1</th>
      <th>ænd</th>
      <th>1</th>
      <th>8</th>
      <th>and</th>
      <th>ænd</th>
      <th>U</th>
      <th>H</th>
      <th>8</th>
      <th>s</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>w</th>
      <th>7</th>
      <th>the</th>
      <th>1</th>
      <th>ðə</th>
      <th>1</th>
      <th>9</th>
      <th>the</th>
      <th>ðə</th>
      <th>U</th>
      <th>L</th>
      <th>9</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>9</th>
      <th>s</th>
      <th>8</th>
      <th>Fruit</th>
      <th>1</th>
      <th>'fruːt</th>
      <th>1</th>
      <th>10</th>
      <th>Fruit</th>
      <th>'fruːt</th>
      <th>P</th>
      <th>H</th>
      <th>10</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th rowspan="11" valign="top">8</th>
      <th rowspan="11" valign="top">wswswsswsws</th>
      <th rowspan="11" valign="top">of|MANS|first|DI|so|BE.DIEN|ce|AND|the|FRUIT</th>
      <th>0</th>
      <th>w</th>
      <th>1</th>
      <th>OF</th>
      <th>1</th>
      <th>ʌv</th>
      <th>1</th>
      <th>0</th>
      <th>OF</th>
      <th>ʌv</th>
      <th>U</th>
      <th>H</th>
      <th>0</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>1</th>
      <th>s</th>
      <th>2</th>
      <th>Mans</th>
      <th>1</th>
      <th>'mænz</th>
      <th>1</th>
      <th>1</th>
      <th>Mans</th>
      <th>'mænz</th>
      <th>P</th>
      <th>H</th>
      <th>1</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>2</th>
      <th>w</th>
      <th>3</th>
      <th>First</th>
      <th>1</th>
      <th>'fɛːst</th>
      <th>1</th>
      <th>2</th>
      <th>First</th>
      <th>'fɛːst</th>
      <th>P</th>
      <th>H</th>
      <th>2</th>
      <th>w</th>
      <td>1.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>3</th>
      <th>s</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>1</th>
      <th>3</th>
      <th>Di</th>
      <th>`dɪ</th>
      <th>S</th>
      <th>L</th>
      <th>3</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.5</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>w</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>2</th>
      <th>4</th>
      <th>so</th>
      <th>sə</th>
      <th>U</th>
      <th>L</th>
      <th>4</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">5</th>
      <th rowspan="2" valign="top">ss</th>
      <th rowspan="2" valign="top">4</th>
      <th rowspan="2" valign="top">Disobedience</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">`dɪ.sə.'biː.diː.əns</th>
      <th>3</th>
      <th>5</th>
      <th>be</th>
      <th>'biː</th>
      <th>P</th>
      <th>L</th>
      <th>5</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>1.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>4</th>
      <th>6</th>
      <th>dien</th>
      <th>diː</th>
      <th>U</th>
      <th>L</th>
      <th>6</th>
      <th>s</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>NaN</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>6</th>
      <th>w</th>
      <th>4</th>
      <th>Disobedience</th>
      <th>1</th>
      <th>`dɪ.sə.'biː.diː.əns</th>
      <th>5</th>
      <th>7</th>
      <th>ce</th>
      <th>əns</th>
      <th>U</th>
      <th>H</th>
      <th>7</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>7</th>
      <th>s</th>
      <th>6</th>
      <th>and</th>
      <th>1</th>
      <th>ænd</th>
      <th>1</th>
      <th>8</th>
      <th>and</th>
      <th>ænd</th>
      <th>U</th>
      <th>H</th>
      <th>8</th>
      <th>s</th>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>0.0</td>
      <td>1.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>w</th>
      <th>7</th>
      <th>the</th>
      <th>1</th>
      <th>ðə</th>
      <th>1</th>
      <th>9</th>
      <th>the</th>
      <th>ðə</th>
      <th>U</th>
      <th>L</th>
      <th>9</th>
      <th>w</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>1.0</td>
      <td>0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>10</td>
      <td>11</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>0.0</td>
    </tr>
    <tr>
      <th>9</th>
      <th>s</th>
      <th>8</th>
      <th>Fruit</th>
      <th>1</th>
      <th>'fruːt</th>
      <th>1</th>
      <th>10</th>
      <th>Fruit</th>
      <th>'fruːt</th>
      <th>P</th>
      <th>H</th>
      <th>10</th>
      <th>s</th>
      <td>0.0</td>
      <td>0</td>
      <td>0</td>
      <td>NaN</td>
      <td>NaN</td>
      <td>0</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>10</td>
      <td>11</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>1.0</td>
    </tr>
  </tbody>
</table>
</div>




```python
# Top 10 parses for first line
parses_byline.query('line_i==1 and parse_rank<=10')
```



<table border="1" class="dataframe">
  <thead>
    <tr style="text-align: right;">
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th>*total</th>
      <th>*clash</th>
      <th>*s/unstressed</th>
      <th>*w-res</th>
      <th>*f-res</th>
      <th>*w/stressed</th>
      <th>*lapse</th>
      <th>*w/peak</th>
      <th>combo_num_syll</th>
      <th>is_funcword</th>
      <th>is_heavy</th>
      <th>is_light</th>
      <th>is_peak</th>
      <th>is_s</th>
      <th>is_stressed</th>
      <th>is_syll</th>
      <th>is_trough</th>
      <th>is_unstressed</th>
      <th>is_w</th>
      <th>parse_num_pos</th>
      <th>parse_num_syll</th>
      <th>prom_strength</th>
      <th>prom_stress</th>
      <th>prom_weight</th>
    </tr>
    <tr>
      <th>stanza_i</th>
      <th>line_i</th>
      <th>line_str</th>
      <th>parse_rank</th>
      <th>combo_i</th>
      <th>combo_stress</th>
      <th>combo_ipa</th>
      <th>parse_i</th>
      <th>parse</th>
      <th>parse_str</th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
      <th></th>
    </tr>
  </thead>
  <tbody>
    <tr>
      <th rowspan="11" valign="top">1</th>
      <th rowspan="11" valign="top">1</th>
      <th rowspan="11" valign="top">OF Mans First Disobedience, and the Fruit</th>
      <th rowspan="2" valign="top">1</th>
      <th rowspan="2" valign="top">0</th>
      <th rowspan="2" valign="top">UPPSUPUUUUP</th>
      <th rowspan="2" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>5</th>
      <th>wswswswwsws</th>
      <th>of|MANS|first|DI|so|BE|dien.ce|AND|the|FRUIT</th>
      <td>3.0</td>
      <td>0</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>8</th>
      <th>wswswsswsws</th>
      <th>of|MANS|first|DI|so|BE.DIEN|ce|AND|the|FRUIT</th>
      <td>3.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>6</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>5</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th rowspan="2" valign="top">3</th>
      <th rowspan="2" valign="top">0</th>
      <th rowspan="2" valign="top">UPPSUPUUUUP</th>
      <th rowspan="2" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>2</th>
      <th>wswswswswws</th>
      <th>of|MANS|first|DI|so|BE|dien|CE|and.the|FRUIT</th>
      <td>4.0</td>
      <td>0</td>
      <td>1</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>1</td>
      <td>1</td>
      <td>1</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>13</th>
      <th>wswswwswsws</th>
      <th>of|MANS|first|DI|so.be|DIEN|ce|AND|the|FRUIT</th>
      <td>4.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th rowspan="4" valign="top">5</th>
      <th rowspan="4" valign="top">0</th>
      <th rowspan="4" valign="top">UPPSUPUUUUP</th>
      <th rowspan="4" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>0</th>
      <th>wswswswswsw</th>
      <th>of|MANS|first|DI|so|BE|dien|CE|and|THE|fruit</th>
      <td>5.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>0</td>
      <td>1</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>11</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>21</th>
      <th>wswsswswsws</th>
      <th>of|MANS|first|DI.SO|be|DIEN|ce|AND|the|FRUIT</th>
      <td>5.0</td>
      <td>0</td>
      <td>3</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>6</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>5</td>
      <td>10</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>170</th>
      <th>swswwswwsws</th>
      <th>OF|mans|FIRST|di.so|BE|dien.ce|AND|the|FRUIT</th>
      <td>5.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>9</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>173</th>
      <th>swswwsswsws</th>
      <th>OF|mans|FIRST|di.so|BE.DIEN|ce|AND|the|FRUIT</th>
      <td>5.0</td>
      <td>0</td>
      <td>3</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>0</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>6</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>5</td>
      <td>9</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th rowspan="3" valign="top">9</th>
      <th rowspan="3" valign="top">0</th>
      <th rowspan="3" valign="top">UPPSUPUUUUP</th>
      <th rowspan="3" valign="top">ʌv 'mænz 'fɛːst `dɪ.sə.'biː.diː.əns ænd ðə 'fruːt</th>
      <th>6</th>
      <th>wswswswwsww</th>
      <th>of|MANS|first|DI|so|BE|dien.ce|AND|the.fruit</th>
      <td>6.0</td>
      <td>0</td>
      <td>1</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>2</td>
      <td>2</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>4</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>7</td>
      <td>9</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>9</th>
      <th>wswswsswsww</th>
      <th>of|MANS|first|DI|so|BE.DIEN|ce|AND|the.fruit</th>
      <td>6.0</td>
      <td>0</td>
      <td>2</td>
      <td>0.0</td>
      <td>1.0</td>
      <td>2</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>5</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>6</td>
      <td>9</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
    <tr>
      <th>162</th>
      <th>swswsswwsws</th>
      <th>OF|mans|FIRST|di|SO.BE|dien.ce|AND|the|FRUIT</th>
      <td>6.0</td>
      <td>0</td>
      <td>3</td>
      <td>0.0</td>
      <td>0.0</td>
      <td>2</td>
      <td>1</td>
      <td>0</td>
      <td>11</td>
      <td>3.0</td>
      <td>6</td>
      <td>5</td>
      <td>1</td>
      <td>6</td>
      <td>5</td>
      <td>11</td>
      <td>3</td>
      <td>6</td>
      <td>5</td>
      <td>9</td>
      <td>11</td>
      <td>1.0</td>
      <td>4.5</td>
      <td>6.0</td>
    </tr>
  </tbody>
</table>
</div>


