# Copyright 2020-2021 SECO Mind S.r.l.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


from datetime import datetime
from os import path

from cryptography import x509
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives import serialization
from cryptography.hazmat.primitives.asymmetric import ec
from cryptography.x509.oid import NameOID


def generate_csr(realm: str, device_id: str, crypto_store_dir: str) -> bytes:
    """
    Utility function that generate the csr for the device

    Parameters
    ----------
    realm: str
        The Astarte realm where the device will be registered
    device_id: str
        The device ID
    crypto_store_dir: str
        Path to the folder where crypto information is stored

    Returns
    -------
    bytes
        The device certificate signing request file
    """
    key = None
    # Do we need to generate a keypair?
    if not path.exists(path.join(crypto_store_dir, "device.key")):
        # Generate our key
        key = ec.generate_private_key(curve=ec.SECP256R1(), backend=default_backend())
        # Write our key to disk for safe keeping
        with open(path.join(crypto_store_dir, "device.key"), "wb") as f:
            f.write(
                key.private_bytes(
                    encoding=serialization.Encoding.PEM,
                    format=serialization.PrivateFormat.TraditionalOpenSSL,
                    encryption_algorithm=serialization.NoEncryption(),
                )
            )
    else:
        # Load the key
        with open(path.join(crypto_store_dir, "device.key"), "rb") as key_file:
            key = serialization.load_pem_private_key(
                key_file.read(), password=None, backend=default_backend()
            )

    csr = (
        x509.CertificateSigningRequestBuilder()
        .subject_name(
            x509.Name(
                [
                    # Provide various details about who we are.
                    x509.NameAttribute(NameOID.ORGANIZATION_NAME, "Devices"),
                    x509.NameAttribute(NameOID.COMMON_NAME, f"{realm}/{device_id}"),
                    # Sign the CSR with our private key.
                ]
            )
        )
        .sign(key, hashes.SHA256(), default_backend())
    )

    # Return the CSR
    return csr.public_bytes(serialization.Encoding.PEM)


def import_device_certificate(client_crt: str, crypto_store_dir: str) -> None:
    certificate = x509.load_pem_x509_certificate(client_crt.encode("ascii"), default_backend())

    # Store the certificate
    with open(path.join(crypto_store_dir, "device.crt"), "wb") as f:
        f.write(certificate.public_bytes(encoding=serialization.Encoding.PEM))


def device_has_certificate(crypto_store_dir: str) -> bool:
    """
    Utility function that checks if a certificate is present for the device

    Parameters
    ----------
    crypto_store_dir: str
        Path to the folder where crypto information is stored

    Returns
    -------
    bool
        True if the certificate is present, False otherwise

    """
    cert_path = path.join(crypto_store_dir, "device.crt")
    key_path = path.join(crypto_store_dir, "device.key")

    return (
        path.exists(cert_path) and path.exists(key_path) and certificate_is_valid(crypto_store_dir)
    )


def certificate_is_valid(crypto_store_dir: str) -> bool:
    """
    Utility function that checks the certificate validity

    Parameters
    ----------
    crypto_store_dir: str
        Path to the folder where crypto information are stored

    Returns
    -------
    bool
        True if the certificate is valid, False otherwise.

    """
    cert_path = path.join(crypto_store_dir, "device.crt")
    with open(cert_path, "r", encoding="utf-8") as file:
        data = file.read()
    if data:
        try:
            certificate = x509.load_pem_x509_certificate(data.encode("ascii"), default_backend())
        except ValueError:
            return False
        return certificate.not_valid_before < datetime.utcnow() < certificate.not_valid_after
    return False
