"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const index_1 = require("./index");
const USAGE = 'Usage: node manual.ts key_pair [max_azs]';
const args = process.argv.slice(2);
if (args.length < 1) {
    process.stderr.write('Insufficient number of arguments were provided.\n');
    process.stderr.write(`${USAGE}\n`);
    process.exit(1);
}
const keyPair = args[0];
let maxAzs;
if (args.length > 1) {
    maxAzs = Number(args[1]);
    if (Number.isNaN(maxAzs)) {
        process.stderr.write('[max_azs] argument could not be converted to a number.\n');
        process.exit(1);
    }
}
else {
    maxAzs = 2;
}
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, 'Stack', {
    env: {
        account: process.env.CDK_DEPLOY_ACCOUNT || process.env.CDK_DEFAULT_ACCOUNT,
        region: process.env.CDK_DEPLOY_REGION || process.env.CDK_DEFAULT_REGION,
    },
});
const natProvider = new index_1.NatAsgProvider({
    instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.T2, aws_cdk_lib_1.aws_ec2.InstanceSize.MICRO),
    trafficDirection: aws_cdk_lib_1.aws_ec2.NatTrafficDirection.NONE,
    keyPair: keyPair,
});
const vpc = new aws_cdk_lib_1.aws_ec2.Vpc(stack, 'Vpc', {
    cidr: '10.0.0.0/16',
    maxAzs: maxAzs,
    natGatewayProvider: natProvider,
    subnetConfiguration: [
        {
            name: 'Public',
            subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PUBLIC,
            cidrMask: 24,
        },
        {
            name: 'PrivateWithNat',
            subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
            cidrMask: 24,
        },
    ],
});
const { subnets } = vpc.selectSubnets({
    subnetGroupName: 'PrivateWithNat',
});
for (const subnet of subnets) {
    const instance = new aws_cdk_lib_1.aws_ec2.Instance(subnet, 'Instance', {
        instanceType: aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.T2, aws_cdk_lib_1.aws_ec2.InstanceSize.MICRO),
        machineImage: new aws_cdk_lib_1.aws_ec2.AmazonLinuxImage({
            cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.X86_64,
            edition: aws_cdk_lib_1.aws_ec2.AmazonLinuxEdition.STANDARD,
            generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            storage: aws_cdk_lib_1.aws_ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            virtualization: aws_cdk_lib_1.aws_ec2.AmazonLinuxVirt.HVM,
        }),
        vpc: vpc,
        allowAllOutbound: false,
        keyName: keyPair,
        vpcSubnets: {
            subnets: [subnet],
        },
    });
    instance.connections.allowFrom(natProvider, aws_cdk_lib_1.aws_ec2.Port.tcp(22));
    instance.connections.allowToAnyIpv4(aws_cdk_lib_1.aws_ec2.Port.allIcmp());
    const subnetPeer = aws_cdk_lib_1.aws_ec2.Peer.ipv4(subnet.ipv4CidrBlock);
    natProvider.connections.allowFrom(subnetPeer, aws_cdk_lib_1.aws_ec2.Port.tcp(80));
    natProvider.connections.allowFrom(subnetPeer, aws_cdk_lib_1.aws_ec2.Port.tcp(443));
    natProvider.connections.allowFrom(subnetPeer, aws_cdk_lib_1.aws_ec2.Port.allIcmp());
    natProvider.connections.allowTo(subnetPeer, aws_cdk_lib_1.aws_ec2.Port.tcp(22));
}
natProvider.connections.allowFrom(aws_cdk_lib_1.aws_ec2.Peer.ipv4('0.0.0.0/0'), aws_cdk_lib_1.aws_ec2.Port.tcp(22));
const defaultPeer = aws_cdk_lib_1.aws_ec2.Peer.ipv4('0.0.0.0/0');
natProvider.connections.allowTo(defaultPeer, aws_cdk_lib_1.aws_ec2.Port.tcp(80));
natProvider.connections.allowTo(defaultPeer, aws_cdk_lib_1.aws_ec2.Port.tcp(443));
natProvider.connections.allowToAnyIpv4(aws_cdk_lib_1.aws_ec2.Port.allIcmp());
app.synth();
//# sourceMappingURL=data:application/json;base64,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