"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NatAsgProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const CFG_SCRIPT_NAME = 'cfg.sh';
const CFG_SCRIPT_PATH = path.join(__dirname, '..', 'resources', CFG_SCRIPT_NAME);
/**
 * `NatAsgProvider` is a NAT provider that places each NAT instance in its own
 * auto scaling group to improve fault tolerance and availability.
 *
 * `NatAsgProvider` extends `NatProvider`:
 * @see https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.NatProvider.html
 */
class NatAsgProvider extends aws_cdk_lib_1.aws_ec2.NatProvider {
    /**
     *
     * @param props Configuration properties.
     */
    constructor(props) {
        super();
        this.props = props;
        this.netIfIds = new PrefSet();
    }
    configureNat(opts) {
        const idPrefix = 'NatAsg';
        const instType = this.props.instanceType ??
            aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.BURSTABLE2, aws_cdk_lib_1.aws_ec2.InstanceSize.MICRO);
        const traffDir = this.props.trafficDirection ??
            aws_cdk_lib_1.aws_ec2.NatTrafficDirection.INBOUND_AND_OUTBOUND;
        this.sg = this.props.securityGroup ??
            new aws_cdk_lib_1.aws_ec2.SecurityGroup(opts.vpc, `${idPrefix}SecurityGroup`, {
                vpc: opts.vpc,
                description: 'Security group for the NAT instances',
                allowAllOutbound: allowAllOutbound(traffDir),
            });
        this.conns = new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: [this.sg],
        });
        if (allowAllInbound(traffDir)) {
            this.conns.allowFromAnyIpv4(aws_cdk_lib_1.aws_ec2.Port.allTraffic());
        }
        const cfgScriptAsset = new aws_cdk_lib_1.aws_s3_assets.Asset(opts.vpc, `${idPrefix}CfgScript`, {
            path: CFG_SCRIPT_PATH,
        });
        const cfgScriptTargetPath = path.join('/tmp', CFG_SCRIPT_NAME);
        const cfgScriptInitFile = aws_cdk_lib_1.aws_ec2.InitFile.fromExistingAsset(cfgScriptTargetPath, cfgScriptAsset, {
            mode: '000755',
            owner: 'root',
            group: 'root',
        });
        const machImg = new aws_cdk_lib_1.aws_ec2.AmazonLinuxImage({
            cpuType: aws_cdk_lib_1.aws_ec2.AmazonLinuxCpuType.X86_64,
            edition: aws_cdk_lib_1.aws_ec2.AmazonLinuxEdition.STANDARD,
            generation: aws_cdk_lib_1.aws_ec2.AmazonLinuxGeneration.AMAZON_LINUX_2,
            storage: aws_cdk_lib_1.aws_ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
            virtualization: aws_cdk_lib_1.aws_ec2.AmazonLinuxVirt.HVM,
        });
        const attachNetIfPolicyStmt = new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: ['ec2:AttachNetworkInterface'],
        });
        for (const subnet of opts.natSubnets) {
            const netIf = new aws_cdk_lib_1.aws_ec2.CfnNetworkInterface(subnet, `${idPrefix}NetworkInterface`, {
                subnetId: subnet.subnetId,
                description: 'Network interface for a NAT instance',
                groupSet: [this.sg.securityGroupId],
                sourceDestCheck: false,
            });
            const eip = new aws_cdk_lib_1.aws_ec2.CfnEIP(subnet, `${idPrefix}Eip`, {
                domain: 'vpc',
            });
            new aws_cdk_lib_1.aws_ec2.CfnEIPAssociation(subnet, `${idPrefix}EipAssoc`, {
                allocationId: eip.attrAllocationId,
                networkInterfaceId: netIf.ref,
            });
            const udata = aws_cdk_lib_1.aws_ec2.UserData.forLinux();
            udata.addCommands('yum install -y aws-cfn-bootstrap');
            const role = new aws_cdk_lib_1.aws_iam.Role(subnet, `${idPrefix}Role`, {
                assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            });
            role.addToPolicy(attachNetIfPolicyStmt);
            const instProf = new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(subnet, `${idPrefix}InstanceProfile`, {
                roles: [role.roleName],
            });
            const launchTmpl = new aws_cdk_lib_1.aws_ec2.CfnLaunchTemplate(subnet, `${idPrefix}LaunchTemplate`, {
                launchTemplateData: {
                    creditSpecification: { cpuCredits: 'standard' },
                    imageId: machImg.getImage(subnet).imageId,
                    instanceType: instType.toString(),
                    iamInstanceProfile: { arn: instProf.attrArn },
                    keyName: this.props.keyPair,
                    networkInterfaces: [{
                            deviceIndex: 0,
                            associatePublicIpAddress: true,
                            deleteOnTermination: true,
                            groups: [this.sg.securityGroupId],
                        }],
                    userData: aws_cdk_lib_1.Lazy.string({
                        produce: () => {
                            return aws_cdk_lib_1.Fn.base64(udata.render());
                        },
                    }),
                },
            });
            const asg = new aws_cdk_lib_1.aws_autoscaling.CfnAutoScalingGroup(subnet, `${idPrefix}Asg`, {
                maxSize: '1',
                minSize: '1',
                availabilityZones: [subnet.availabilityZone],
                launchTemplate: {
                    version: launchTmpl.attrLatestVersionNumber,
                    launchTemplateId: launchTmpl.ref,
                },
                vpcZoneIdentifier: [subnet.subnetId],
            });
            asg.cfnOptions.creationPolicy = {
                resourceSignal: {
                    count: 1,
                    timeout: 'PT10M',
                },
            };
            asg.cfnOptions.updatePolicy = {
                autoScalingRollingUpdate: {
                    pauseTime: 'PT10M',
                    suspendProcesses: [
                        'HealthCheck',
                        'ReplaceUnhealthy',
                        'AZRebalance',
                        'AlarmNotification',
                        'ScheduledActions',
                    ],
                    waitOnResourceSignals: true,
                },
            };
            const cfnInit = aws_cdk_lib_1.aws_ec2.CloudFormationInit.fromElements(cfgScriptInitFile, aws_cdk_lib_1.aws_ec2.InitCommand.argvCommand([`${cfgScriptTargetPath}`, netIf.ref]));
            cfnInit.attach(launchTmpl, {
                instanceRole: role,
                platform: aws_cdk_lib_1.aws_ec2.OperatingSystemType.LINUX,
                userData: udata,
                signalResource: asg,
            });
            this.netIfIds.add(subnet.availabilityZone, netIf.ref);
        }
        for (const s of opts.privateSubnets) {
            this.configureSubnet(s);
        }
    }
    configureSubnet(subnet) {
        subnet.addRoute('DefaultRoute', {
            routerType: aws_cdk_lib_1.aws_ec2.RouterType.NETWORK_INTERFACE,
            routerId: this.netIfIds.pick(subnet.availabilityZone),
            enablesInternetConnectivity: true,
        });
    }
    get configuredGateways() {
        return this.netIfIds.values().map(x => ({ az: x[0], gatewayId: x[1] }));
    }
    /**
     * The network connections associated with the security group of the NAT instances.
     *
     * @see https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.Connections.html
     */
    get connections() {
        if (this.conns === undefined) {
            throw new Error("Pass the NatAsgProvider to a VPC before accessing \'connections\'");
        }
        return this.conns;
    }
    /**
     * The security group associated with the NAT instances.
     *
     * @see https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_ec2.ISecurityGroup.html
     */
    get securityGroup() {
        if (this.sg === undefined) {
            throw new Error("Pass the NatAsgProvider to a VPC before accessing \'securityGroup\'");
        }
        return this.sg;
    }
}
exports.NatAsgProvider = NatAsgProvider;
_a = JSII_RTTI_SYMBOL_1;
NatAsgProvider[_a] = { fqn: "cdk-nat-asg-provider.NatAsgProvider", version: "0.0.4" };
function allowAllOutbound(d) {
    return d === aws_cdk_lib_1.aws_ec2.NatTrafficDirection.INBOUND_AND_OUTBOUND ||
        d === aws_cdk_lib_1.aws_ec2.NatTrafficDirection.OUTBOUND_ONLY;
}
function allowAllInbound(d) {
    return d === aws_cdk_lib_1.aws_ec2.NatTrafficDirection.INBOUND_AND_OUTBOUND;
}
/**
 * PrefSet is a set of values where each value may be associated with a
 * preference key. A value may be retrieved by providing a preference
 * key. If there's no value associated with the preference key, then a
 * value is selected in a round-robin fashion.
 */
class PrefSet {
    constructor() {
        this.prefVals = {};
        this.vals = new Array();
        this.next = 0;
    }
    add(pref, val) {
        this.prefVals[pref] = val;
        this.vals.push([pref, val]);
    }
    pick(pref) {
        if (this.vals.length === 0) {
            throw new Error('Picking an empty set');
        }
        if (pref in this.prefVals) {
            return this.prefVals[pref];
        }
        this.next = this.next % this.vals.length;
        const picked = this.vals[this.next][1];
        this.next++;
        return picked;
    }
    values() {
        return this.vals;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSw2QkFBNkI7QUFFN0IsNkNBT3FCO0FBRXJCLE1BQU0sZUFBZSxHQUFHLFFBQVEsQ0FBQztBQUNqQyxNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsV0FBVyxFQUFFLGVBQWUsQ0FBQyxDQUFDO0FBaURqRjs7Ozs7O0dBTUc7QUFDSCxNQUFhLGNBQWUsU0FBUSxxQkFBRyxDQUFDLFdBQVc7SUFNakQ7OztPQUdHO0lBQ0gsWUFBNkIsS0FBMEI7UUFDckQsS0FBSyxFQUFFLENBQUM7UUFEbUIsVUFBSyxHQUFMLEtBQUssQ0FBcUI7UUFUL0MsYUFBUSxHQUFHLElBQUksT0FBTyxFQUFVLENBQUM7SUFXekMsQ0FBQztJQUVNLFlBQVksQ0FBQyxJQUE2QjtRQUMvQyxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFFMUIsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZO1lBQ3RDLHFCQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxxQkFBRyxDQUFDLGFBQWEsQ0FBQyxVQUFVLEVBQUUscUJBQUcsQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFNUUsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0I7WUFDMUMscUJBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxvQkFBb0IsQ0FBQztRQUUvQyxJQUFJLENBQUMsRUFBRSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYTtZQUNoQyxJQUFJLHFCQUFHLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxRQUFRLGVBQWUsRUFBRTtnQkFDMUQsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2dCQUNiLFdBQVcsRUFBRSxzQ0FBc0M7Z0JBQ25ELGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLFFBQVEsQ0FBQzthQUM3QyxDQUFDLENBQUM7UUFFTCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUkscUJBQUcsQ0FBQyxXQUFXLENBQUM7WUFDL0IsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztTQUMxQixDQUFDLENBQUM7UUFFSCxJQUFJLGVBQWUsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUM3QixJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLHFCQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDcEQ7UUFFRCxNQUFNLGNBQWMsR0FBRyxJQUFJLDJCQUFRLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxRQUFRLFdBQVcsRUFBRTtZQUMxRSxJQUFJLEVBQUUsZUFBZTtTQUN0QixDQUFDLENBQUM7UUFFSCxNQUFNLG1CQUFtQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBRS9ELE1BQU0saUJBQWlCLEdBQUcscUJBQUcsQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUMsbUJBQW1CLEVBQUUsY0FBYyxFQUFFO1lBQzVGLElBQUksRUFBRSxRQUFRO1lBQ2QsS0FBSyxFQUFFLE1BQU07WUFDYixLQUFLLEVBQUUsTUFBTTtTQUNkLENBQUMsQ0FBQztRQUVILE1BQU0sT0FBTyxHQUFHLElBQUkscUJBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztZQUN2QyxPQUFPLEVBQUUscUJBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNO1lBQ3RDLE9BQU8sRUFBRSxxQkFBRyxDQUFDLGtCQUFrQixDQUFDLFFBQVE7WUFDeEMsVUFBVSxFQUFFLHFCQUFHLENBQUMscUJBQXFCLENBQUMsY0FBYztZQUNwRCxPQUFPLEVBQUUscUJBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxlQUFlO1lBQy9DLGNBQWMsRUFBRSxxQkFBRyxDQUFDLGVBQWUsQ0FBQyxHQUFHO1NBQ3hDLENBQUMsQ0FBQztRQUVILE1BQU0scUJBQXFCLEdBQUcsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztZQUNwRCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFLENBQUMsNEJBQTRCLENBQUM7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsS0FBSyxNQUFNLE1BQU0sSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ3BDLE1BQU0sS0FBSyxHQUFHLElBQUkscUJBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLEVBQUUsR0FBRyxRQUFRLGtCQUFrQixFQUFFO2dCQUMvRSxRQUFRLEVBQUUsTUFBTSxDQUFDLFFBQVE7Z0JBQ3pCLFdBQVcsRUFBRSxzQ0FBc0M7Z0JBQ25ELFFBQVEsRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDO2dCQUNuQyxlQUFlLEVBQUUsS0FBSzthQUN2QixDQUFDLENBQUM7WUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLHFCQUFHLENBQUMsTUFBTSxDQUFDLE1BQU0sRUFBRSxHQUFHLFFBQVEsS0FBSyxFQUFFO2dCQUNuRCxNQUFNLEVBQUUsS0FBSzthQUNkLENBQUMsQ0FBQztZQUNILElBQUkscUJBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLEVBQUUsR0FBRyxRQUFRLFVBQVUsRUFBRTtnQkFDdkQsWUFBWSxFQUFFLEdBQUcsQ0FBQyxnQkFBZ0I7Z0JBQ2xDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxHQUFHO2FBQzlCLENBQUMsQ0FBQztZQUVILE1BQU0sS0FBSyxHQUFHLHFCQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDO1lBQ3RDLEtBQUssQ0FBQyxXQUFXLENBQUMsa0NBQWtDLENBQUMsQ0FBQztZQUV0RCxNQUFNLElBQUksR0FBRyxJQUFJLHFCQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxHQUFHLFFBQVEsTUFBTSxFQUFFO2dCQUNuRCxTQUFTLEVBQUUsSUFBSSxxQkFBRyxDQUFDLGdCQUFnQixDQUFDLG1CQUFtQixDQUFDO2FBQ3pELENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxXQUFXLENBQUMscUJBQXFCLENBQUMsQ0FBQztZQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLHFCQUFHLENBQUMsa0JBQWtCLENBQUMsTUFBTSxFQUFFLEdBQUcsUUFBUSxpQkFBaUIsRUFBRTtnQkFDaEYsS0FBSyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQzthQUN2QixDQUFDLENBQUM7WUFFSCxNQUFNLFVBQVUsR0FBRyxJQUFJLHFCQUFHLENBQUMsaUJBQWlCLENBQUMsTUFBTSxFQUFFLEdBQUcsUUFBUSxnQkFBZ0IsRUFBRTtnQkFDaEYsa0JBQWtCLEVBQUU7b0JBQ2xCLG1CQUFtQixFQUFFLEVBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRTtvQkFDL0MsT0FBTyxFQUFFLE9BQU8sQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTztvQkFDekMsWUFBWSxFQUFFLFFBQVEsQ0FBQyxRQUFRLEVBQUU7b0JBQ2pDLGtCQUFrQixFQUFFLEVBQUUsR0FBRyxFQUFFLFFBQVEsQ0FBQyxPQUFPLEVBQUU7b0JBQzdDLE9BQU8sRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU87b0JBQzNCLGlCQUFpQixFQUFFLENBQUM7NEJBQ2xCLFdBQVcsRUFBRSxDQUFDOzRCQUNkLHdCQUF3QixFQUFFLElBQUk7NEJBQzlCLG1CQUFtQixFQUFFLElBQUk7NEJBQ3pCLE1BQU0sRUFBRSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsZUFBZSxDQUFDO3lCQUNsQyxDQUFDO29CQUNGLFFBQVEsRUFBRSxrQkFBSSxDQUFDLE1BQU0sQ0FBQzt3QkFDcEIsT0FBTyxFQUFFLEdBQUcsRUFBRTs0QkFDWixPQUFPLGdCQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO3dCQUNuQyxDQUFDO3FCQUNGLENBQUM7aUJBQ0g7YUFDRixDQUFDLENBQUM7WUFFSCxNQUFNLEdBQUcsR0FBRyxJQUFJLDZCQUFXLENBQUMsbUJBQW1CLENBQUMsTUFBTSxFQUFFLEdBQUcsUUFBUSxLQUFLLEVBQUU7Z0JBQ3hFLE9BQU8sRUFBRSxHQUFHO2dCQUNaLE9BQU8sRUFBRSxHQUFHO2dCQUNaLGlCQUFpQixFQUFFLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDO2dCQUM1QyxjQUFjLEVBQUU7b0JBQ2QsT0FBTyxFQUFFLFVBQVUsQ0FBQyx1QkFBdUI7b0JBQzNDLGdCQUFnQixFQUFFLFVBQVUsQ0FBQyxHQUFHO2lCQUNqQztnQkFDRCxpQkFBaUIsRUFBRSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUM7YUFDckMsQ0FBQyxDQUFDO1lBQ0gsR0FBRyxDQUFDLFVBQVUsQ0FBQyxjQUFjLEdBQUc7Z0JBQzlCLGNBQWMsRUFBRTtvQkFDZCxLQUFLLEVBQUUsQ0FBQztvQkFDUixPQUFPLEVBQUUsT0FBTztpQkFDakI7YUFDRixDQUFDO1lBQ0YsR0FBRyxDQUFDLFVBQVUsQ0FBQyxZQUFZLEdBQUc7Z0JBQzVCLHdCQUF3QixFQUFFO29CQUN4QixTQUFTLEVBQUUsT0FBTztvQkFDbEIsZ0JBQWdCLEVBQUU7d0JBQ2hCLGFBQWE7d0JBQ2Isa0JBQWtCO3dCQUNsQixhQUFhO3dCQUNiLG1CQUFtQjt3QkFDbkIsa0JBQWtCO3FCQUNuQjtvQkFDRCxxQkFBcUIsRUFBRSxJQUFJO2lCQUM1QjthQUNGLENBQUM7WUFFRixNQUFNLE9BQU8sR0FBRyxxQkFBRyxDQUFDLGtCQUFrQixDQUFDLFlBQVksQ0FDakQsaUJBQWlCLEVBQ2pCLHFCQUFHLENBQUMsV0FBVyxDQUFDLFdBQVcsQ0FBQyxDQUFDLEdBQUcsbUJBQW1CLEVBQUUsRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FDbkUsQ0FBQztZQUNGLE9BQU8sQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO2dCQUN6QixZQUFZLEVBQUUsSUFBSTtnQkFDbEIsUUFBUSxFQUFFLHFCQUFHLENBQUMsbUJBQW1CLENBQUMsS0FBSztnQkFDdkMsUUFBUSxFQUFFLEtBQUs7Z0JBQ2YsY0FBYyxFQUFFLEdBQUc7YUFDcEIsQ0FBQyxDQUFDO1lBRUgsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN2RDtRQUVELEtBQUssTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUNuQyxJQUFJLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3pCO0lBQ0gsQ0FBQztJQUVNLGVBQWUsQ0FBQyxNQUF5QjtRQUM5QyxNQUFNLENBQUMsUUFBUSxDQUFDLGNBQWMsRUFBRTtZQUM5QixVQUFVLEVBQUUscUJBQUcsQ0FBQyxVQUFVLENBQUMsaUJBQWlCO1lBQzVDLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsZ0JBQWdCLENBQUM7WUFDckQsMkJBQTJCLEVBQUUsSUFBSTtTQUNsQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsSUFBVyxrQkFBa0I7UUFDM0IsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLFdBQVc7UUFDcEIsSUFBSSxJQUFJLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRSxDQUFDLENBQUM7U0FDdEY7UUFDRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUM7SUFDcEIsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLGFBQWE7UUFDdEIsSUFBSSxJQUFJLENBQUMsRUFBRSxLQUFLLFNBQVMsRUFBRTtZQUN6QixNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7U0FDeEY7UUFDRCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUM7SUFDakIsQ0FBQzs7QUFsTUgsd0NBbU1DOzs7QUFFRCxTQUFTLGdCQUFnQixDQUFDLENBQTBCO0lBQ2xELE9BQU8sQ0FBQyxLQUFLLHFCQUFHLENBQUMsbUJBQW1CLENBQUMsb0JBQW9CO1FBQ3ZELENBQUMsS0FBSyxxQkFBRyxDQUFDLG1CQUFtQixDQUFDLGFBQWEsQ0FBQztBQUNoRCxDQUFDO0FBRUQsU0FBUyxlQUFlLENBQUMsQ0FBMEI7SUFDakQsT0FBTyxDQUFDLEtBQUsscUJBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxvQkFBb0IsQ0FBQztBQUM1RCxDQUFDO0FBRUQ7Ozs7O0dBS0c7QUFDSCxNQUFNLE9BQU87SUFBYjtRQUNtQixhQUFRLEdBQXNCLEVBQUUsQ0FBQztRQUNqQyxTQUFJLEdBQUcsSUFBSSxLQUFLLEVBQWUsQ0FBQztRQUN6QyxTQUFJLEdBQUcsQ0FBQyxDQUFDO0lBMEJuQixDQUFDO0lBeEJRLEdBQUcsQ0FBQyxJQUFZLEVBQUUsR0FBTTtRQUM3QixJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQztRQUMxQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQzlCLENBQUM7SUFFTSxJQUFJLENBQUMsSUFBWTtRQUN0QixJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMxQixNQUFNLElBQUksS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUM7U0FDekM7UUFFRCxJQUFJLElBQUksSUFBSSxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ3pCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUM1QjtRQUVELElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztRQUN6QyxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFWixPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDO0lBRU0sTUFBTTtRQUNYLE9BQU8sSUFBSSxDQUFDLElBQUksQ0FBQztJQUNuQixDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuXG5pbXBvcnQge1xuICBhd3NfZWMyIGFzIGVjMixcbiAgYXdzX2lhbSBhcyBpYW0sXG4gIGF3c19hdXRvc2NhbGluZyBhcyBhdXRvc2NhbGluZyxcbiAgYXdzX3MzX2Fzc2V0cyBhcyBzM2Fzc2V0cyxcbiAgRm4sXG4gIExhenksXG59IGZyb20gJ2F3cy1jZGstbGliJztcblxuY29uc3QgQ0ZHX1NDUklQVF9OQU1FID0gJ2NmZy5zaCc7XG5jb25zdCBDRkdfU0NSSVBUX1BBVEggPSBwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4nLCAncmVzb3VyY2VzJywgQ0ZHX1NDUklQVF9OQU1FKTtcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRvIGNvbmZpZ3VyZSBgTmF0QXNnUHJvdmlkZXJgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIE5hdEFzZ1Byb3ZpZGVyUHJvcHMge1xuICAvKipcbiAgICogVGhlIEVDMiBpbnN0YW5jZSB0eXBlIG9mIHRoZSBOQVQgaW5zdGFuY2VzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jZGsvYXBpL3YyL2RvY3MvYXdzLWNkay1saWIuYXdzX2VjMi5JbnN0YW5jZVR5cGUuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdFZhbHVlIHQyLm1pY3JvXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBlYzIuSW5zdGFuY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgYWxsb3dlZCB0cmFmZmljIGRpcmVjdGlvbnMgdGhyb3VnaCB0aGUgTkFUIGluc3RhbmNlcy5cbiAgICpcbiAgICogSWYgeW91IHNldCB0aGlzIHRvIGEgdmFsdWUgb3RoZXIgdGhhblxuICAgKiBgZWMyLk5hdFRyYWZmaWNEaXJlY3Rpb24uSU5CT1VORF9BTkRfT1VUQk9VTkRgLCB5b3UgbXVzdFxuICAgKiBjb25maWd1cmUgdGhlIHNlY3VyaXR5IGdyb3VwIGZvciB0aGUgTkFUIGluc3RhbmNlcyBlaXRoZXIgYnkgcHJvdmlkaW5nXG4gICAqIGEgZnVsbHkgY29uZmlndXJlZCBzZWN1cml0eSBncm91cCB0aHJvdWdoIHRoZSBgc2VjdXJpdHlHcm91cGAgcHJvcGVydHlcbiAgICogb3IgYnkgdXNpbmcgdGhlIGBOYXRBc2dQcm92aWRlcmAgb2JqZWN0J3MgYHNlY3VyaXR5R3JvdXBgIG9yXG4gICAqIGBjb25uZWN0aW9uc2AgcHJvcGVydGllcyBhZnRlciBwYXNzaW5nIHRoZSBgTmF0QXNnUHJvdmlkZXJgIG9iamVjdCB0byBhXG4gICAqIGBWcGNgIG9iamVjdC5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS92Mi9kb2NzL2F3cy1jZGstbGliLmF3c19lYzIuTmF0VHJhZmZpY0RpcmVjdGlvbi5odG1sXG4gICAqXG4gICAqIEBkZWZhdWx0VmFsdWUgYGF3cy1jZGstbGliLmF3c19lYzIuTmF0VHJhZmZpY0RpcmVjdGlvbi5JTkJPVU5EX0FORF9PVVRCT1VORGBcbiAgICovXG4gIHJlYWRvbmx5IHRyYWZmaWNEaXJlY3Rpb24/OiBlYzIuTmF0VHJhZmZpY0RpcmVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIFNTSCBrZXkgcGFpciBncmFudGluZyBhY2Nlc3MgdG8gdGhlIE5BVCBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0VDMi9sYXRlc3QvVXNlckd1aWRlL2VjMi1rZXktcGFpcnMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkga2V5UGFpcj86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGdyb3VwIGFzc29jaWF0ZWQgd2l0aCB0aGUgTkFUIGluc3RhbmNlcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS92Mi9kb2NzL2F3cy1jZGstbGliLmF3c19lYzIuSVNlY3VyaXR5R3JvdXAuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdFZhbHVlIEEgc2VjdXJpdHkgZ3JvdXAgd2lsbCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbn1cblxuLyoqXG4gKiBgTmF0QXNnUHJvdmlkZXJgIGlzIGEgTkFUIHByb3ZpZGVyIHRoYXQgcGxhY2VzIGVhY2ggTkFUIGluc3RhbmNlIGluIGl0cyBvd25cbiAqIGF1dG8gc2NhbGluZyBncm91cCB0byBpbXByb3ZlIGZhdWx0IHRvbGVyYW5jZSBhbmQgYXZhaWxhYmlsaXR5LlxuICpcbiAqIGBOYXRBc2dQcm92aWRlcmAgZXh0ZW5kcyBgTmF0UHJvdmlkZXJgOlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS92Mi9kb2NzL2F3cy1jZGstbGliLmF3c19lYzIuTmF0UHJvdmlkZXIuaHRtbFxuICovXG5leHBvcnQgY2xhc3MgTmF0QXNnUHJvdmlkZXIgZXh0ZW5kcyBlYzIuTmF0UHJvdmlkZXIgaW1wbGVtZW50cyBlYzIuSUNvbm5lY3RhYmxlIHtcbiAgcHJpdmF0ZSBuZXRJZklkcyA9IG5ldyBQcmVmU2V0PHN0cmluZz4oKTtcblxuICBwcml2YXRlIGNvbm5zPzogZWMyLkNvbm5lY3Rpb25zO1xuICBwcml2YXRlIHNnPzogZWMyLklTZWN1cml0eUdyb3VwO1xuXG4gIC8qKlxuICAgKlxuICAgKiBAcGFyYW0gcHJvcHMgQ29uZmlndXJhdGlvbiBwcm9wZXJ0aWVzLlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBwcm9wczogTmF0QXNnUHJvdmlkZXJQcm9wcykge1xuICAgIHN1cGVyKCk7XG4gIH1cblxuICBwdWJsaWMgY29uZmlndXJlTmF0KG9wdHM6IGVjMi5Db25maWd1cmVOYXRPcHRpb25zKSB7XG4gICAgY29uc3QgaWRQcmVmaXggPSAnTmF0QXNnJztcblxuICAgIGNvbnN0IGluc3RUeXBlID0gdGhpcy5wcm9wcy5pbnN0YW5jZVR5cGUgPz9cbiAgICAgIGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuQlVSU1RBQkxFMiwgZWMyLkluc3RhbmNlU2l6ZS5NSUNSTyk7XG5cbiAgICBjb25zdCB0cmFmZkRpciA9IHRoaXMucHJvcHMudHJhZmZpY0RpcmVjdGlvbiA/P1xuICAgICAgZWMyLk5hdFRyYWZmaWNEaXJlY3Rpb24uSU5CT1VORF9BTkRfT1VUQk9VTkQ7XG5cbiAgICB0aGlzLnNnID0gdGhpcy5wcm9wcy5zZWN1cml0eUdyb3VwID8/XG4gICAgICBuZXcgZWMyLlNlY3VyaXR5R3JvdXAob3B0cy52cGMsIGAke2lkUHJlZml4fVNlY3VyaXR5R3JvdXBgLCB7XG4gICAgICAgIHZwYzogb3B0cy52cGMsXG4gICAgICAgIGRlc2NyaXB0aW9uOiAnU2VjdXJpdHkgZ3JvdXAgZm9yIHRoZSBOQVQgaW5zdGFuY2VzJyxcbiAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogYWxsb3dBbGxPdXRib3VuZCh0cmFmZkRpciksXG4gICAgICB9KTtcblxuICAgIHRoaXMuY29ubnMgPSBuZXcgZWMyLkNvbm5lY3Rpb25zKHtcbiAgICAgIHNlY3VyaXR5R3JvdXBzOiBbdGhpcy5zZ10sXG4gICAgfSk7XG5cbiAgICBpZiAoYWxsb3dBbGxJbmJvdW5kKHRyYWZmRGlyKSkge1xuICAgICAgdGhpcy5jb25ucy5hbGxvd0Zyb21BbnlJcHY0KGVjMi5Qb3J0LmFsbFRyYWZmaWMoKSk7XG4gICAgfVxuXG4gICAgY29uc3QgY2ZnU2NyaXB0QXNzZXQgPSBuZXcgczNhc3NldHMuQXNzZXQob3B0cy52cGMsIGAke2lkUHJlZml4fUNmZ1NjcmlwdGAsIHtcbiAgICAgIHBhdGg6IENGR19TQ1JJUFRfUEFUSCxcbiAgICB9KTtcblxuICAgIGNvbnN0IGNmZ1NjcmlwdFRhcmdldFBhdGggPSBwYXRoLmpvaW4oJy90bXAnLCBDRkdfU0NSSVBUX05BTUUpO1xuXG4gICAgY29uc3QgY2ZnU2NyaXB0SW5pdEZpbGUgPSBlYzIuSW5pdEZpbGUuZnJvbUV4aXN0aW5nQXNzZXQoY2ZnU2NyaXB0VGFyZ2V0UGF0aCwgY2ZnU2NyaXB0QXNzZXQsIHtcbiAgICAgIG1vZGU6ICcwMDA3NTUnLFxuICAgICAgb3duZXI6ICdyb290JyxcbiAgICAgIGdyb3VwOiAncm9vdCcsXG4gICAgfSk7XG5cbiAgICBjb25zdCBtYWNoSW1nID0gbmV3IGVjMi5BbWF6b25MaW51eEltYWdlKHtcbiAgICAgIGNwdVR5cGU6IGVjMi5BbWF6b25MaW51eENwdVR5cGUuWDg2XzY0LFxuICAgICAgZWRpdGlvbjogZWMyLkFtYXpvbkxpbnV4RWRpdGlvbi5TVEFOREFSRCxcbiAgICAgIGdlbmVyYXRpb246IGVjMi5BbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIsXG4gICAgICBzdG9yYWdlOiBlYzIuQW1hem9uTGludXhTdG9yYWdlLkdFTkVSQUxfUFVSUE9TRSxcbiAgICAgIHZpcnR1YWxpemF0aW9uOiBlYzIuQW1hem9uTGludXhWaXJ0LkhWTSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGF0dGFjaE5ldElmUG9saWN5U3RtdCA9IG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICBhY3Rpb25zOiBbJ2VjMjpBdHRhY2hOZXR3b3JrSW50ZXJmYWNlJ10sXG4gICAgfSk7XG5cbiAgICBmb3IgKGNvbnN0IHN1Ym5ldCBvZiBvcHRzLm5hdFN1Ym5ldHMpIHtcbiAgICAgIGNvbnN0IG5ldElmID0gbmV3IGVjMi5DZm5OZXR3b3JrSW50ZXJmYWNlKHN1Ym5ldCwgYCR7aWRQcmVmaXh9TmV0d29ya0ludGVyZmFjZWAsIHtcbiAgICAgICAgc3VibmV0SWQ6IHN1Ym5ldC5zdWJuZXRJZCxcbiAgICAgICAgZGVzY3JpcHRpb246ICdOZXR3b3JrIGludGVyZmFjZSBmb3IgYSBOQVQgaW5zdGFuY2UnLFxuICAgICAgICBncm91cFNldDogW3RoaXMuc2cuc2VjdXJpdHlHcm91cElkXSxcbiAgICAgICAgc291cmNlRGVzdENoZWNrOiBmYWxzZSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBlaXAgPSBuZXcgZWMyLkNmbkVJUChzdWJuZXQsIGAke2lkUHJlZml4fUVpcGAsIHtcbiAgICAgICAgZG9tYWluOiAndnBjJyxcbiAgICAgIH0pO1xuICAgICAgbmV3IGVjMi5DZm5FSVBBc3NvY2lhdGlvbihzdWJuZXQsIGAke2lkUHJlZml4fUVpcEFzc29jYCwge1xuICAgICAgICBhbGxvY2F0aW9uSWQ6IGVpcC5hdHRyQWxsb2NhdGlvbklkLFxuICAgICAgICBuZXR3b3JrSW50ZXJmYWNlSWQ6IG5ldElmLnJlZixcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCB1ZGF0YSA9IGVjMi5Vc2VyRGF0YS5mb3JMaW51eCgpO1xuICAgICAgdWRhdGEuYWRkQ29tbWFuZHMoJ3l1bSBpbnN0YWxsIC15IGF3cy1jZm4tYm9vdHN0cmFwJyk7XG5cbiAgICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3VibmV0LCBgJHtpZFByZWZpeH1Sb2xlYCwge1xuICAgICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnZWMyLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIH0pO1xuICAgICAgcm9sZS5hZGRUb1BvbGljeShhdHRhY2hOZXRJZlBvbGljeVN0bXQpO1xuICAgICAgY29uc3QgaW5zdFByb2YgPSBuZXcgaWFtLkNmbkluc3RhbmNlUHJvZmlsZShzdWJuZXQsIGAke2lkUHJlZml4fUluc3RhbmNlUHJvZmlsZWAsIHtcbiAgICAgICAgcm9sZXM6IFtyb2xlLnJvbGVOYW1lXSxcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCBsYXVuY2hUbXBsID0gbmV3IGVjMi5DZm5MYXVuY2hUZW1wbGF0ZShzdWJuZXQsIGAke2lkUHJlZml4fUxhdW5jaFRlbXBsYXRlYCwge1xuICAgICAgICBsYXVuY2hUZW1wbGF0ZURhdGE6IHtcbiAgICAgICAgICBjcmVkaXRTcGVjaWZpY2F0aW9uOiB7IGNwdUNyZWRpdHM6ICdzdGFuZGFyZCcgfSxcbiAgICAgICAgICBpbWFnZUlkOiBtYWNoSW1nLmdldEltYWdlKHN1Ym5ldCkuaW1hZ2VJZCxcbiAgICAgICAgICBpbnN0YW5jZVR5cGU6IGluc3RUeXBlLnRvU3RyaW5nKCksXG4gICAgICAgICAgaWFtSW5zdGFuY2VQcm9maWxlOiB7IGFybjogaW5zdFByb2YuYXR0ckFybiB9LFxuICAgICAgICAgIGtleU5hbWU6IHRoaXMucHJvcHMua2V5UGFpcixcbiAgICAgICAgICBuZXR3b3JrSW50ZXJmYWNlczogW3tcbiAgICAgICAgICAgIGRldmljZUluZGV4OiAwLFxuICAgICAgICAgICAgYXNzb2NpYXRlUHVibGljSXBBZGRyZXNzOiB0cnVlLFxuICAgICAgICAgICAgZGVsZXRlT25UZXJtaW5hdGlvbjogdHJ1ZSxcbiAgICAgICAgICAgIGdyb3VwczogW3RoaXMuc2cuc2VjdXJpdHlHcm91cElkXSxcbiAgICAgICAgICB9XSxcbiAgICAgICAgICB1c2VyRGF0YTogTGF6eS5zdHJpbmcoe1xuICAgICAgICAgICAgcHJvZHVjZTogKCkgPT4ge1xuICAgICAgICAgICAgICByZXR1cm4gRm4uYmFzZTY0KHVkYXRhLnJlbmRlcigpKTtcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSksXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgY29uc3QgYXNnID0gbmV3IGF1dG9zY2FsaW5nLkNmbkF1dG9TY2FsaW5nR3JvdXAoc3VibmV0LCBgJHtpZFByZWZpeH1Bc2dgLCB7XG4gICAgICAgIG1heFNpemU6ICcxJyxcbiAgICAgICAgbWluU2l6ZTogJzEnLFxuICAgICAgICBhdmFpbGFiaWxpdHlab25lczogW3N1Ym5ldC5hdmFpbGFiaWxpdHlab25lXSxcbiAgICAgICAgbGF1bmNoVGVtcGxhdGU6IHtcbiAgICAgICAgICB2ZXJzaW9uOiBsYXVuY2hUbXBsLmF0dHJMYXRlc3RWZXJzaW9uTnVtYmVyLFxuICAgICAgICAgIGxhdW5jaFRlbXBsYXRlSWQ6IGxhdW5jaFRtcGwucmVmLFxuICAgICAgICB9LFxuICAgICAgICB2cGNab25lSWRlbnRpZmllcjogW3N1Ym5ldC5zdWJuZXRJZF0sXG4gICAgICB9KTtcbiAgICAgIGFzZy5jZm5PcHRpb25zLmNyZWF0aW9uUG9saWN5ID0ge1xuICAgICAgICByZXNvdXJjZVNpZ25hbDoge1xuICAgICAgICAgIGNvdW50OiAxLFxuICAgICAgICAgIHRpbWVvdXQ6ICdQVDEwTScsXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgICAgYXNnLmNmbk9wdGlvbnMudXBkYXRlUG9saWN5ID0ge1xuICAgICAgICBhdXRvU2NhbGluZ1JvbGxpbmdVcGRhdGU6IHtcbiAgICAgICAgICBwYXVzZVRpbWU6ICdQVDEwTScsXG4gICAgICAgICAgc3VzcGVuZFByb2Nlc3NlczogW1xuICAgICAgICAgICAgJ0hlYWx0aENoZWNrJyxcbiAgICAgICAgICAgICdSZXBsYWNlVW5oZWFsdGh5JyxcbiAgICAgICAgICAgICdBWlJlYmFsYW5jZScsXG4gICAgICAgICAgICAnQWxhcm1Ob3RpZmljYXRpb24nLFxuICAgICAgICAgICAgJ1NjaGVkdWxlZEFjdGlvbnMnLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgd2FpdE9uUmVzb3VyY2VTaWduYWxzOiB0cnVlLFxuICAgICAgICB9LFxuICAgICAgfTtcblxuICAgICAgY29uc3QgY2ZuSW5pdCA9IGVjMi5DbG91ZEZvcm1hdGlvbkluaXQuZnJvbUVsZW1lbnRzKFxuICAgICAgICBjZmdTY3JpcHRJbml0RmlsZSxcbiAgICAgICAgZWMyLkluaXRDb21tYW5kLmFyZ3ZDb21tYW5kKFtgJHtjZmdTY3JpcHRUYXJnZXRQYXRofWAsIG5ldElmLnJlZl0pLFxuICAgICAgKTtcbiAgICAgIGNmbkluaXQuYXR0YWNoKGxhdW5jaFRtcGwsIHtcbiAgICAgICAgaW5zdGFuY2VSb2xlOiByb2xlLFxuICAgICAgICBwbGF0Zm9ybTogZWMyLk9wZXJhdGluZ1N5c3RlbVR5cGUuTElOVVgsXG4gICAgICAgIHVzZXJEYXRhOiB1ZGF0YSxcbiAgICAgICAgc2lnbmFsUmVzb3VyY2U6IGFzZyxcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLm5ldElmSWRzLmFkZChzdWJuZXQuYXZhaWxhYmlsaXR5Wm9uZSwgbmV0SWYucmVmKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IHMgb2Ygb3B0cy5wcml2YXRlU3VibmV0cykge1xuICAgICAgdGhpcy5jb25maWd1cmVTdWJuZXQocyk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGNvbmZpZ3VyZVN1Ym5ldChzdWJuZXQ6IGVjMi5Qcml2YXRlU3VibmV0KSB7XG4gICAgc3VibmV0LmFkZFJvdXRlKCdEZWZhdWx0Um91dGUnLCB7XG4gICAgICByb3V0ZXJUeXBlOiBlYzIuUm91dGVyVHlwZS5ORVRXT1JLX0lOVEVSRkFDRSxcbiAgICAgIHJvdXRlcklkOiB0aGlzLm5ldElmSWRzLnBpY2soc3VibmV0LmF2YWlsYWJpbGl0eVpvbmUpLFxuICAgICAgZW5hYmxlc0ludGVybmV0Q29ubmVjdGl2aXR5OiB0cnVlLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIGdldCBjb25maWd1cmVkR2F0ZXdheXMoKTogZWMyLkdhdGV3YXlDb25maWdbXSB7XG4gICAgcmV0dXJuIHRoaXMubmV0SWZJZHMudmFsdWVzKCkubWFwKHggPT4gKHsgYXo6IHhbMF0sIGdhdGV3YXlJZDogeFsxXSB9KSk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIG5ldHdvcmsgY29ubmVjdGlvbnMgYXNzb2NpYXRlZCB3aXRoIHRoZSBzZWN1cml0eSBncm91cCBvZiB0aGUgTkFUIGluc3RhbmNlcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2FwaS92Mi9kb2NzL2F3cy1jZGstbGliLmF3c19lYzIuQ29ubmVjdGlvbnMuaHRtbFxuICAgKi9cbiAgcHVibGljIGdldCBjb25uZWN0aW9ucygpOiBlYzIuQ29ubmVjdGlvbnMge1xuICAgIGlmICh0aGlzLmNvbm5zID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIlBhc3MgdGhlIE5hdEFzZ1Byb3ZpZGVyIHRvIGEgVlBDIGJlZm9yZSBhY2Nlc3NpbmcgXFwnY29ubmVjdGlvbnNcXCdcIik7XG4gICAgfVxuICAgIHJldHVybiB0aGlzLmNvbm5zO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBzZWN1cml0eSBncm91cCBhc3NvY2lhdGVkIHdpdGggdGhlIE5BVCBpbnN0YW5jZXMuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nkay9hcGkvdjIvZG9jcy9hd3MtY2RrLWxpYi5hd3NfZWMyLklTZWN1cml0eUdyb3VwLmh0bWxcbiAgICovXG4gIHB1YmxpYyBnZXQgc2VjdXJpdHlHcm91cCgpOiBlYzIuSVNlY3VyaXR5R3JvdXAge1xuICAgIGlmICh0aGlzLnNnID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihcIlBhc3MgdGhlIE5hdEFzZ1Byb3ZpZGVyIHRvIGEgVlBDIGJlZm9yZSBhY2Nlc3NpbmcgXFwnc2VjdXJpdHlHcm91cFxcJ1wiKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuc2c7XG4gIH1cbn1cblxuZnVuY3Rpb24gYWxsb3dBbGxPdXRib3VuZChkOiBlYzIuTmF0VHJhZmZpY0RpcmVjdGlvbik6IGJvb2xlYW4ge1xuICByZXR1cm4gZCA9PT0gZWMyLk5hdFRyYWZmaWNEaXJlY3Rpb24uSU5CT1VORF9BTkRfT1VUQk9VTkQgfHxcbiAgICBkID09PSBlYzIuTmF0VHJhZmZpY0RpcmVjdGlvbi5PVVRCT1VORF9PTkxZO1xufVxuXG5mdW5jdGlvbiBhbGxvd0FsbEluYm91bmQoZDogZWMyLk5hdFRyYWZmaWNEaXJlY3Rpb24pOiBib29sZWFuIHtcbiAgcmV0dXJuIGQgPT09IGVjMi5OYXRUcmFmZmljRGlyZWN0aW9uLklOQk9VTkRfQU5EX09VVEJPVU5EO1xufVxuXG4vKipcbiAqIFByZWZTZXQgaXMgYSBzZXQgb2YgdmFsdWVzIHdoZXJlIGVhY2ggdmFsdWUgbWF5IGJlIGFzc29jaWF0ZWQgd2l0aCBhXG4gKiBwcmVmZXJlbmNlIGtleS4gQSB2YWx1ZSBtYXkgYmUgcmV0cmlldmVkIGJ5IHByb3ZpZGluZyBhIHByZWZlcmVuY2VcbiAqIGtleS4gSWYgdGhlcmUncyBubyB2YWx1ZSBhc3NvY2lhdGVkIHdpdGggdGhlIHByZWZlcmVuY2Uga2V5LCB0aGVuIGFcbiAqIHZhbHVlIGlzIHNlbGVjdGVkIGluIGEgcm91bmQtcm9iaW4gZmFzaGlvbi5cbiAqL1xuY2xhc3MgUHJlZlNldDxBPiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgcHJlZlZhbHM6IFJlY29yZDxzdHJpbmcsIEE+ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgdmFscyA9IG5ldyBBcnJheTxbc3RyaW5nLCBBXT4oKTtcbiAgcHJpdmF0ZSBuZXh0ID0gMDtcblxuICBwdWJsaWMgYWRkKHByZWY6IHN0cmluZywgdmFsOiBBKSB7XG4gICAgdGhpcy5wcmVmVmFsc1twcmVmXSA9IHZhbDtcbiAgICB0aGlzLnZhbHMucHVzaChbcHJlZiwgdmFsXSk7XG4gIH1cblxuICBwdWJsaWMgcGljayhwcmVmOiBzdHJpbmcpOiBBIHtcbiAgICBpZiAodGhpcy52YWxzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdQaWNraW5nIGFuIGVtcHR5IHNldCcpO1xuICAgIH1cblxuICAgIGlmIChwcmVmIGluIHRoaXMucHJlZlZhbHMpIHtcbiAgICAgIHJldHVybiB0aGlzLnByZWZWYWxzW3ByZWZdO1xuICAgIH1cblxuICAgIHRoaXMubmV4dCA9IHRoaXMubmV4dCAlIHRoaXMudmFscy5sZW5ndGg7XG4gICAgY29uc3QgcGlja2VkID0gdGhpcy52YWxzW3RoaXMubmV4dF1bMV07XG4gICAgdGhpcy5uZXh0Kys7XG5cbiAgICByZXR1cm4gcGlja2VkO1xuICB9XG5cbiAgcHVibGljIHZhbHVlcygpOiBBcnJheTxbc3RyaW5nLCBBXT4ge1xuICAgIHJldHVybiB0aGlzLnZhbHM7XG4gIH1cbn0iXX0=