from pathlib import Path

from harbor.llms.base import BaseLLM, LLMResponse
from harbor.models.agent.rollout_detail import RolloutDetail


class Chat:
    def __init__(self, model: BaseLLM):
        self._model = model
        self._messages = []
        self._cumulative_input_tokens = 0
        self._cumulative_output_tokens = 0
        self._cumulative_cache_tokens = 0
        self._cumulative_cost = 0.0
        self._completion_token_ids_list: list[list[int]] = []
        self._logprobs_list: list[list[float]] = []

    @property
    def total_input_tokens(self) -> int:
        return self._cumulative_input_tokens

    @property
    def total_output_tokens(self) -> int:
        return self._cumulative_output_tokens

    @property
    def total_cache_tokens(self) -> int:
        return self._cumulative_cache_tokens

    @property
    def total_cost(self) -> float:
        return self._cumulative_cost

    @property
    def messages(self) -> list:
        return self._messages

    @property
    def rollout_details(self) -> list[RolloutDetail]:
        """Get detailed rollout information from all chat interactions.

        Returns:
            List containing a single RolloutDetail representing the complete
            linear chat history, or an empty list if no rollout data was collected.

            TODO: consider multiple rollout details for non-linear chat histories, e.g.
            subagents, summarization, etc.
        """
        if not self._completion_token_ids_list and not self._logprobs_list:
            return []

        rollout_detail: RolloutDetail = {}

        if self._completion_token_ids_list:
            rollout_detail["completion_token_ids"] = self._completion_token_ids_list

        if self._logprobs_list:
            rollout_detail["logprobs"] = self._logprobs_list

        return [rollout_detail]

    async def chat(
        self,
        prompt: str,
        logging_path: Path | None = None,
        **kwargs,
    ) -> LLMResponse:
        llm_response: LLMResponse = await self._model.call(
            prompt=prompt,
            message_history=self._messages,
            logging_path=logging_path,
            **kwargs,
        )

        # Get token usage and cost from the LLM response
        usage = llm_response.usage
        if usage is not None:
            self._cumulative_input_tokens += usage.prompt_tokens
            self._cumulative_output_tokens += usage.completion_tokens
            self._cumulative_cache_tokens += usage.cache_tokens
            self._cumulative_cost += usage.cost_usd

        # Accumulate rollout details from the response
        self._accumulate_rollout_details(llm_response)

        self._messages.extend(
            [
                {"role": "user", "content": prompt},
                {"role": "assistant", "content": llm_response.content},
            ]
        )
        return llm_response

    def _accumulate_rollout_details(self, llm_response: LLMResponse) -> None:
        """Accumulate rollout details from an LLM response.

        Args:
            llm_response: The LLM response containing token IDs and logprobs
        """
        # Accumulate token IDs per turn
        if llm_response.completion_token_ids:
            # Store completion token IDs for this turn
            self._completion_token_ids_list.append(llm_response.completion_token_ids)

            # Store logprobs for this turn (if available)
            if llm_response.logprobs:
                self._logprobs_list.append(llm_response.logprobs)
