from pathlib import Path
from typing import Annotated

from typer import Option, Typer

traces_app = Typer(no_args_is_help=True)


@traces_app.command("export")
def export(
    path: Annotated[
        Path,
        Option(
            "--path",
            "-p",
            help="Path to a trial dir or a root containing trials recursively",
        ),
    ],
    recursive: Annotated[
        bool,
        Option(
            "--recursive/--no-recursive",
            help="Search recursively for trials under path",
        ),
    ] = True,
    episodes: Annotated[
        str,
        Option(
            "--episodes",
            help="Export all episodes or only the last episode per trial (all|last)",
        ),
    ] = "all",
    to_sharegpt: Annotated[
        bool,
        Option(
            "--sharegpt/--no-sharegpt",
            help="Also emit ShareGPT-formatted conversations column",
        ),
    ] = False,
    push: Annotated[
        bool,
        Option(
            "--push/--no-push", help="Push dataset to Hugging Face Hub after export"
        ),
    ] = False,
    repo_id: Annotated[
        str | None,
        Option(
            "--repo",
            help="Target HF repo id (org/name) when --push is set",
            show_default=False,
        ),
    ] = None,
    verbose: Annotated[
        bool,
        Option("--verbose/--no-verbose", help="Print discovery details for debugging"),
    ] = False,
    filter: Annotated[
        str | None,
        Option(
            "--filter",
            help="Filter trials by result: success|failure|all (default all)",
            show_default=False,
        ),
    ] = None,
):
    from harbor.utils.traces_utils import export_traces as _export_traces

    if push and not repo_id:
        raise ValueError("--push requires --repo <org/name>")

    if episodes not in ("all", "last"):
        raise ValueError("--episodes must be one of: all, last")

    if filter and filter not in ("all", "success", "failure"):
        raise ValueError("--filter must be one of: success, failure, all")

    ds = _export_traces(
        root=path,
        recursive=recursive,
        episodes=episodes,
        to_sharegpt=to_sharegpt,
        repo_id=repo_id,
        push=push,
        verbose=verbose,
        success_filter=(None if (not filter or filter == "all") else filter),
    )
    print(f"Exported {len(ds)} rows from {path}")
