"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("aws-cdk-lib");
const events = require("aws-cdk-lib/aws-events");
const sns = require("aws-cdk-lib/aws-sns");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
const lib_1 = require("../lib");
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventbridgeToSns(stack, 'test', props);
}
function deployStackWithNewEventBus(stack) {
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: { eventBusName: 'test' }
    };
    return new lib_1.EventbridgeToSns(stack, 'test-neweventbus', props);
}
test('check if the event rule has permission/policy in place in sns for it to be able to publish to the topic', () => {
    const stack = new cdk.Stack();
    const testConstruct = deployNewStack(stack);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SNS::TopicPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        StringEquals: {
                            "AWS:SourceOwner": {
                                Ref: "AWS::AccountId"
                            }
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::Join": [
                                "",
                                [
                                    "arn:",
                                    {
                                        Ref: "AWS::Partition"
                                    },
                                    ":iam::",
                                    {
                                        Ref: "AWS::AccountId"
                                    },
                                    ":root"
                                ]
                            ]
                        }
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "TopicOwnerOnlyAccess"
                },
                {
                    Action: [
                        "SNS:Publish",
                        "SNS:RemovePermission",
                        "SNS:SetTopicAttributes",
                        "SNS:DeleteTopic",
                        "SNS:ListSubscriptionsByTopic",
                        "SNS:GetTopicAttributes",
                        "SNS:Receive",
                        "SNS:AddPermission",
                        "SNS:Subscribe"
                    ],
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: {
                        AWS: "*"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "HttpsOnly"
                },
                {
                    Action: "sns:Publish",
                    Effect: "Allow",
                    Principal: {
                        Service: "events.amazonaws.com"
                    },
                    Resource: {
                        Ref: "testSnsTopic42942701"
                    },
                    Sid: "2"
                }
            ],
            Version: "2012-10-17"
        },
        Topics: [
            {
                Ref: "testSnsTopic42942701"
            }
        ]
    });
});
test('check events rule properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "testSnsTopic42942701"
                },
                Id: {
                    "Fn::Join": [
                        "",
                        [
                            "Defaulttest-",
                            {
                                "Fn::Select": [
                                    2,
                                    {
                                        "Fn::Split": [
                                            "/",
                                            {
                                                Ref: "AWS::StackId"
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    expect(construct.eventsRule).toBeDefined();
    expect(construct.snsTopic).toBeDefined();
    expect(construct.encryptionKey).toBeDefined();
});
test('check the sns topic properties', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testEncryptionKeyB55BFDBC",
                "Arn"
            ]
        }
    });
});
test('check the sns topic properties with existing KMS key', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKey: key
    };
    new lib_1.EventbridgeToSns(stack, 'test-events-rule-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    template.hasResourceProperties('AWS::KMS::Key', {
        Description: "my-key",
        EnableKeyRotation: true
    });
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const construct = deployStackWithNewEventBus(stack);
    expect(construct.eventsRule).toBeDefined();
    expect(construct.snsTopic).toBeDefined();
    expect(construct.encryptionKey).toBeDefined();
    expect(construct.eventBus).toBeDefined();
    // Check whether eventbus exists
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::Events::EventBus', 1);
});
test('Confirm CheckEventBridgeProps is being called', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {},
        existingEventBusInterface: new events.EventBus(stack, `test-existing-new-eventbus`, { eventBusName: 'test' })
    };
    const app = () => {
        new lib_1.EventbridgeToSns(stack, 'test-eventbridge-sns', props);
    };
    expect(app).toThrowError('Error - Either provide existingEventBusInterface or eventBusProps, but not both.\n');
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const props = {
        eventBusProps: {
            eventBusName: 'testcustomeventbus'
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new lib_1.EventbridgeToSns(stack, 'test-new-eventbridge-sns', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Events::EventBus', {
        Name: 'testcustomeventbus'
    });
});
test('Topic is encrypted when key is provided on topicProps.masterKey prop', () => {
    const stack = new cdk.Stack();
    const key = defaults.buildEncryptionKey(stack, {
        description: 'my-key'
    });
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        topicProps: {
            masterKey: key
        }
    };
    new lib_1.EventbridgeToSns(stack, 'test-events-rule-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "EncryptionKey1B843E66",
                "Arn"
            ]
        }
    });
    template.hasResourceProperties('AWS::KMS::Key', {
        Description: "my-key",
        EnableKeyRotation: true
    });
});
test('Topic is encrypted when keyProps are provided', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        encryptionKeyProps: {
            description: 'my-key'
        }
    };
    new lib_1.EventbridgeToSns(stack, 'test-events-rule-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "testeventsrulesqsEncryptionKey19AB0C02",
                "Arn"
            ]
        }
    });
    template.hasResourceProperties('AWS::KMS::Key', {
        Description: "my-key",
        EnableKeyRotation: true
    });
});
test('Topic is encrypted with AWS-managed KMS key when enableEncryptionWithCustomerManagedKey property is false', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        },
        enableEncryptionWithCustomerManagedKey: false
    };
    new lib_1.EventbridgeToSns(stack, 'test-events-rule-sqs', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition"
                    },
                    ":kms:",
                    {
                        Ref: "AWS::Region"
                    },
                    ":",
                    {
                        Ref: "AWS::AccountId"
                    },
                    ":alias/aws/sns"
                ]
            ]
        }
    });
});
test('Properties correctly set when unencrypted existing topic is provided', () => {
    const stack = new cdk.Stack();
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const props = {
        existingTopicObj,
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    const testConstruct = new lib_1.EventbridgeToSns(stack, 'test', props);
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).not.toBeDefined();
});
test('Confirm CheckSnsProps is being called', () => {
    const stack = new cdk.Stack();
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const props = {
        existingTopicObj,
        topicProps: {},
        eventRuleProps: {
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    const app = () => {
        new lib_1.EventbridgeToSns(stack, 'test', props);
    };
    expect(app).toThrowError("Error - Either provide topicProps or existingTopicObj, but not both.\n");
});
//# sourceMappingURL=data:application/json;base64,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