import unittest

from .propagation import atmospheric_attenuation

p618_att_tot_table = """
51,5	-0,14	0,03138298	1	14,25	31,07699124  1	0,65	0	1	1,21279072
41,9	12,49	0,04612299	1	14,25	40,23203600  1	0,65	0	1	1,10400153
33,94	18,43	0,00000000	1	14,25	46,35969261  1	0,65	0	1	0,82781253
51,5	-0,14	0,03138298	1	14,25	31,07699124  1	0,65	0	0,1	2,90152327
41,9	12,49	0,04612299	1	14,25	40,23203600  1	0,65	0	0,1	3,17165505
33,94	18,43	0,00000000	1	14,25	46,35969261  1	0,65	0	0,1	2,31053122
51,5	-0,14	0,03138298	1	14,25	31,07699124  1	0,65	0	0,01	7,50726532
41,9	12,49	0,04612299	1	14,25	40,23203600  1	0,65	0	0,01	8,69315393
33,94	18,43	0,00000000	1	14,25	46,35969261  1	0,65	0	0,01	6,33097642
51,5	-0,14	0,03138298	1	14,25	31,07699124  1	0,65	0	0,001	15,58184351
41,9	12,49	0,04612299	1	14,25	40,23203600  1	0,65	0	0,001	18,12446882
33,94	18,43	0,00000000	1	14,25	46,35969261  1	0,65	0	0,001	13,34528655
51,5	-0,14	0,03138298	1	29	31,07699124  1	0,65	0	1	4,83682548
41,9	12,49	0,04612299	1	29	40,23203600  1	0,65	0	1	4,57030291
33,94	18,43	0,00000000	1	29	46,35969261  1	0,65	0	1	3,37970773
51,5	-0,14	0,03138298	1	29	31,07699124  1	0,65	0	0,1	11,19917055
41,9	12,49	0,04612299	1	29	40,23203600  1	0,65	0	0,1	12,47579603
33,94	18,43	0,00000000	1	29	46,35969261  1	0,65	0	0,1	9,22353663
51,5	-0,14	0,03138298	1	29	31,07699124  1	0,65	0	0,01	26,07175081
41,9	12,49	0,04612299	1	29	40,23203600  1	0,65	0	0,01	30,48600187
33,94	18,43	0,00000000	1	29	46,35969261  1	0,65	0	0,01	22,66125273
51,5	-0,14	0,03138298	1	29	31,07699124  1	0,65	0	0,001	47,80872386
41,9	12,49	0,04612299	1	29	40,23203600  1	0,65	0	0,001	55,98834728
33,94	18,43	0,00000000	1	29	46,35969261  1	0,65	0	0,001	42,07849210
22,9	-43,23	0,00000000	-100	14,25	22,27833468  1	0,65	0	1	2,74415451
25,78	-80,22	0,00861728	-100	14,25	52,67898486  1	0,65	0	1	2,21165713
22,9	-43,23	0,00000000	-100	14,25	22,27833468  1	0,65	0	0,1	9,28165675
25,78	-80,22	0,00861728	-100	14,25	52,67898486  1	0,65	0	0,1	7,06719080
22,9	-43,23	0,00000000	-100	14,25	22,27833468  1	0,65	0	0,01	19,95415823
25,78	-80,22	0,00861728	-100	14,25	52,67898486  1	0,65	0	0,01	17,19826055
22,9	-43,23	0,00000000	-100	14,25	22,27833468  1	0,65	0	0,001	30,86510308
25,78	-80,22	0,00861728	-100	14,25	52,67898486  1	0,65	0	0,001	30,68736787
22,9	-43,23	0,00000000	-100	29	22,27833468  1	0,65	0	1	10,59851911
25,78	-80,22	0,00861728	-100	29	52,67898486  1	0,65	0	1	9,64018288
22,9	-43,23	0,00000000	-100	29	22,27833468  1	0,65	0	0,1	33,09186000
25,78	-80,22	0,00861728	-100	29	52,67898486  1	0,65	0	0,1	28,54673650
22,9	-43,23	0,00000000	-100	29	22,27833468  1	0,65	0	0,01	63,40311131
25,78	-80,22	0,00861728	-100	29	52,67898486  1	0,65	0	0,01	61,45823575
22,9	-43,23	0,00000000	-100	29	22,27833468  1	0,65	0	0,001	87,33730346
25,78	-80,22	0,00861728	-100	29	52,67898486  1	0,65	0	0,001	96,37217508
28,717	77,3	0,20938370	100	14,25	48,24117054  1	0,65	90	1	2,25793670
3,133	101,7	0,05125146	100	14,25	85,80459566  1	0,65	90	1	2,82423339
9,05	38,7	2,53986188	100	14,25	20,14335809  1	0,65	90	1	1,96132622
28,717	77,3	0,20938370	100	14,25	48,24117054  1	0,65	90	0,1	6,46781701
3,133	101,7	0,05125146	100	14,25	85,80459566  1	0,65	90	0,1	11,82080134
9,05	38,7	2,53986188	100	14,25	20,14335809  1	0,65	90	0,1	6,80772618
28,717	77,3	0,20938370	100	14,25	48,24117054  1	0,65	90	0,01	15,85241845
3,133	101,7	0,05125146	100	14,25	85,80459566  1	0,65	90	0,01	22,43075823
9,05	38,7	2,53986188	100	14,25	20,14335809  1	0,65	90	0,01	13,22187874
28,717	77,3	0,20938370	100	14,25	48,24117054  1	0,65	90	0,001	29,20774522
3,133	101,7	0,05125146	100	14,25	85,80459566  1	0,65	90	0,001	29,63331804
9,05	38,7	2,53986188	100	14,25	20,14335809  1	0,65	90	0,001	18,32188141
28,717	77,3	0,20938370	100	29	48,24117054  1	0,65	90	1	9,72324587
3,133	101,7	0,05125146	100	29	85,80459566  1	0,65	90	1	13,41799688
9,05	38,7	2,53986188	100	29	20,14335809  1	0,65	90	1	7,00833145
28,717	77,3	0,20938370	100	29	48,24117054  1	0,65	90	0,1	26,06108561
3,133	101,7	0,05125146	100	29	85,80459566  1	0,65	90	0,1	52,02324991
9,05	38,7	2,53986188	100	29	20,14335809  1	0,65	90	0,1	22,53547444
28,717	77,3	0,20938370	100	29	48,24117054  1	0,65	90	0,01	56,66860081
3,133	101,7	0,05125146	100	29	85,80459566  1	0,65	90	0,01	86,58264540
9,05	38,7	2,53986188	100	29	20,14335809  1	0,65	90	0,01	39,27455494
28,717	77,3	0,20938370	100	29	48,24117054  1	0,65	90	0,001	91,79204756
3,133	101,7	0,05125146	100	29	85,80459566  1	0,65	90	0,001	99,87574945
9,05	38,7	2,53986188	100	29	20,14335809  1	0,65	90	0,001	48,93930866
"""


class TestPropagation(unittest.TestCase):

    def test_rain_attenuation(self):
        # Test the examples from the ITU-R validation spreadsheet entitled
        # "Validation examples for Study Group 3 Earth-Space propagation
        # prediction methods, Version: 5.1 (P), ITU Radio communication Study
        # Groups.". More specifically, test from the "P618-13 Att_Tot" tab of
        # the spreadsheet, which focuses on the total attenuation.
        for line in p618_att_tot_table.splitlines()[1:]:
            # Parse the columns of the above table
            (rx_lat, rx_lng, _, _, f_ghz, elev, antenna_diameter,
             antenna_efficiency, pol_skew, unavailability,
             expected_attn_db) = [
                 float(x.replace(',', '.')) for x in line.split()
             ]
            # Check the computation
            f_hz = f_ghz * 1e9
            availability = (100 - unavailability)
            attn_db = atmospheric_attenuation(rx_lat, rx_lng, elev, pol_skew,
                                              f_hz, availability,
                                              antenna_diameter,
                                              antenna_efficiency)
            self.assertAlmostEqual(attn_db, expected_attn_db, delta=0.1)
