# From Wikipedia:
#
# Mazovia encoding is used under DOS to represent Polish texts.
# Basically it is code page 437 with some positions filled with Polish letters.
#
# See https://pl.wikipedia.org/wiki/Mazovia_(kod)

import codecs

class MazoviaCodec(codecs.Codec):
    def encode(self, input_, errors='strict'):
        return codecs.charmap_encode(input_, errors, encoding_table)

    def decode(self, input_, errors='strict'):
        return codecs.charmap_decode(input_, errors, decoding_table)

def lookup(name):
    if name != 'mazovia':
        return None
    return codecs.CodecInfo(
        name='mazovia',
        encode=MazoviaCodec().encode,
        decode=MazoviaCodec().decode,
    )

def register():
  codecs.register(lookup)

decoding_table = ''.join(chr(c) for c in range(128)) + \
(
    '\u00C7'  # 0x80 'Ç' LATIN CAPITAL LETTER C WITH CEDILLA
    '\u00FC'  # 0x81 'ü' LATIN SMALL LETTER U WITH DIAERESIS
    '\u00E9'  # 0x82 'é' LATIN SMALL LETTER E WITH ACUTE
    '\u00E2'  # 0x83 'â' LATIN SMALL LETTER A WITH CIRCUMFLEX
    '\u00E4'  # 0x84 'ä' LATIN SMALL LETTER A WITH DIAERESIS
    '\u00E0'  # 0x85 'à' LATIN SMALL LETTER A WITH GRAVE
    '\u0105'  # 0x86 'ą' LATIN SMALL LETTER A WITH OGONEK
    '\u00E7'  # 0x87 'ç' LATIN SMALL LETTER C WITH CEDILLA
    '\u00EA'  # 0x88 'ê' LATIN SMALL LETTER E WITH CIRCUMFLEX
    '\u00EB'  # 0x89 'ë' LATIN SMALL LETTER E WITH DIAERESIS
    '\u00E8'  # 0x8A 'è' LATIN SMALL LETTER E WITH GRAVE
    '\u00EF'  # 0x8B 'ï' LATIN SMALL LETTER I WITH DIAERESIS
    '\u00EE'  # 0x8C 'î' LATIN SMALL LETTER I WITH CIRCUMFLEX
    '\u0107'  # 0x8D 'ć' LATIN SMALL LETTER C WITH ACUTE
    '\u00C4'  # 0x8E 'Ä' LATIN CAPITAL LETTER A WITH DIAERESIS
    '\u0104'  # 0x8F 'Ą' LATIN CAPITAL LETTER A WITH OGONEK
    '\u0118'  # 0x90 'Ę' LATIN CAPITAL LETTER E WITH OGONEK
    '\u0119'  # 0x91 'ę' LATIN SMALL LETTER E WITH OGONEK
    '\u0142'  # 0x92 'ł' LATIN SMALL LETTER L WITH STROKE
    '\u00F4'  # 0x93 'ô' LATIN SMALL LETTER O WITH CIRCUMFLEX
    '\u00F6'  # 0x94 'ö' LATIN SMALL LETTER O WITH DIAERESIS
    '\u0106'  # 0x95 'Ć' LATIN CAPITAL LETTER C WITH ACUTE
    '\u00FB'  # 0x96 'û' LATIN SMALL LETTER U WITH CIRCUMFLEX
    '\u00F9'  # 0x97 'ù' LATIN SMALL LETTER U WITH GRAVE
    '\u015A'  # 0x98 'Ś' LATIN CAPITAL LETTER S WITH ACUTE
    '\u00D6'  # 0x99 'Ö' LATIN CAPITAL LETTER O WITH DIAERESIS
    '\u00DC'  # 0x9A 'Ü' LATIN CAPITAL LETTER U WITH DIAERESIS
    '\u00A2'  # 0x9B '¢' CENT SIGN
    '\u0141'  # 0x9C 'Ł' LATIN CAPITAL LETTER L WITH STROKE
    '\u00A5'  # 0x9D '¥' YEN SIGN
    '\u015B'  # 0x9E 'ś' LATIN SMALL LETTER S WITH ACUTE
    '\u0192'  # 0x9F 'ƒ' LATIN SMALL LETTER F WITH HOOK
    '\u0179'  # 0xA0 'Ź' LATIN CAPITAL LETTER Z WITH ACUTE
    '\u017B'  # 0xA1 'Ż' LATIN CAPITAL LETTER Z WITH DOT ABOVE
    '\u00F3'  # 0xA2 'ó' LATIN SMALL LETTER O WITH ACUTE
    '\u00D3'  # 0xA3 'Ó' LATIN CAPITAL LETTER O WITH ACUTE
    '\u0144'  # 0xA4 'ń' LATIN SMALL LETTER N WITH ACUTE
    '\u0143'  # 0xA5 'Ń' LATIN CAPITAL LETTER N WITH ACUTE
    '\u017A'  # 0xA6 'ź' LATIN SMALL LETTER Z WITH ACUTE
    '\u017C'  # 0xA7 'ż' LATIN SMALL LETTER Z WITH DOT ABOVE
    '\u00BF'  # 0xA8 '¿' INVERTED QUESTION MARK
    '\u2310'  # 0xA9 '⌐' REVERSED NOT SIGN
    '\u00AC'  # 0xAA '¬' NOT SIGN
    '\u00BD'  # 0xAB '½' VULGAR FRACTION ONE HALF
    '\u00BC'  # 0xAC '¼' VULGAR FRACTION ONE QUARTER
    '\u00A1'  # 0xAD '¡' INVERTED EXCLAMATION MARK
    '\u00AB'  # 0xAE '«' LEFT-POINTING DOUBLE ANGLE QUOTATION MARK
    '\u00BB'  # 0xAF '»' RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
    '\u2591'  # 0xB0 '░' LIGHT SHADE
    '\u2592'  # 0xB1 '▒' MEDIUM SHADE
    '\u2593'  # 0xB2 '▓' DARK SHADE
    '\u2502'  # 0xB3 '│' BOX DRAWINGS LIGHT VERTICAL
    '\u2524'  # 0xB4 '┤' BOX DRAWINGS LIGHT VERTICAL AND LEFT
    '\u2561'  # 0xB5 '╡' BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE
    '\u2562'  # 0xB6 '╢' BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE
    '\u2556'  # 0xB7 '╖' BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE
    '\u2555'  # 0xB8 '╕' BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE
    '\u2563'  # 0xB9 '╣' BOX DRAWINGS DOUBLE VERTICAL AND LEFT
    '\u2551'  # 0xBA '║' BOX DRAWINGS DOUBLE VERTICAL
    '\u2557'  # 0xBB '╗' BOX DRAWINGS DOUBLE DOWN AND LEFT
    '\u255D'  # 0xBC '╝' BOX DRAWINGS DOUBLE UP AND LEFT
    '\u255C'  # 0xBD '╜' BOX DRAWINGS UP DOUBLE AND LEFT SINGLE
    '\u255B'  # 0xBE '╛' BOX DRAWINGS UP SINGLE AND LEFT DOUBLE
    '\u2510'  # 0xBF '┐' BOX DRAWINGS LIGHT DOWN AND LEFT
    '\u2514'  # 0xC0 '└' BOX DRAWINGS LIGHT UP AND RIGHT
    '\u2534'  # 0xC1 '┴' BOX DRAWINGS LIGHT UP AND HORIZONTAL
    '\u252C'  # 0xC2 '┬' BOX DRAWINGS LIGHT DOWN AND HORIZONTAL
    '\u251C'  # 0xC3 '├' BOX DRAWINGS LIGHT VERTICAL AND RIGHT
    '\u2500'  # 0xC4 '─' BOX DRAWINGS LIGHT HORIZONTAL
    '\u253C'  # 0xC5 '┼' BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL
    '\u255E'  # 0xC6 '╞' BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE
    '\u255F'  # 0xC7 '╟' BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE
    '\u255A'  # 0xC8 '╚' BOX DRAWINGS DOUBLE UP AND RIGHT
    '\u2554'  # 0xC9 '╔' BOX DRAWINGS DOUBLE DOWN AND RIGHT
    '\u2569'  # 0xCA '╩' BOX DRAWINGS DOUBLE UP AND HORIZONTAL
    '\u2566'  # 0xCB '╦' BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL
    '\u2560'  # 0xCC '╠' BOX DRAWINGS DOUBLE VERTICAL AND RIGHT
    '\u2550'  # 0xCD '═' BOX DRAWINGS DOUBLE HORIZONTAL
    '\u256C'  # 0xCE '╬' BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL
    '\u2567'  # 0xCF '╧' BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE
    '\u2568'  # 0xD0 '╨' BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE
    '\u2564'  # 0xD1 '╤' BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE
    '\u2565'  # 0xD2 '╥' BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE
    '\u2559'  # 0xD3 '╙' BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE
    '\u2558'  # 0xD4 '╘' BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE
    '\u2552'  # 0xD5 '╒' BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE
    '\u2553'  # 0xD6 '╓' BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE
    '\u256B'  # 0xD7 '╫' BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE
    '\u256A'  # 0xD8 '╪' BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE
    '\u2518'  # 0xD9 '┘' BOX DRAWINGS LIGHT UP AND LEFT
    '\u250C'  # 0xDA '┌' BOX DRAWINGS LIGHT DOWN AND RIGHT
    '\u2588'  # 0xDB '█' FULL BLOCK
    '\u2584'  # 0xDC '▄' LOWER HALF BLOCK
    '\u258C'  # 0xDD '▌' LEFT HALF BLOCK
    '\u2590'  # 0xDE '▐' RIGHT HALF BLOCK
    '\u2580'  # 0xDF '▀' UPPER HALF BLOCK
    '\u03B1'  # 0xE0 'α' GREEK SMALL LETTER ALPHA
    '\u00DF'  # 0xE1 'ß' LATIN SMALL LETTER SHARP S
    '\u0393'  # 0xE2 'Γ' GREEK CAPITAL LETTER GAMMA
    '\u03C0'  # 0xE3 'π' GREEK SMALL LETTER PI
    '\u03A3'  # 0xE4 'Σ' GREEK CAPITAL LETTER SIGMA
    '\u03C3'  # 0xE5 'σ' GREEK SMALL LETTER SIGMA
    '\u00B5'  # 0xE6 'µ' MICRO SIGN
    '\u03C4'  # 0xE7 'τ' GREEK SMALL LETTER TAU
    '\u03A6'  # 0xE8 'Φ' GREEK CAPITAL LETTER PHI
    '\u0398'  # 0xE9 'Θ' GREEK CAPITAL LETTER THETA
    '\u03A9'  # 0xEA 'Ω' GREEK CAPITAL LETTER OMEGA
    '\u03B4'  # 0xEB 'δ' GREEK SMALL LETTER DELTA
    '\u221E'  # 0xEC '∞' INFINITY
    '\u03C6'  # 0xED 'φ' GREEK SMALL LETTER PHI
    '\u03B5'  # 0xEE 'ε' GREEK SMALL LETTER EPSILON
    '\u2229'  # 0xEF '∩' INTERSECTION
    '\u2261'  # 0xF0 '≡' IDENTICAL TO
    '\u00B1'  # 0xF1 '±' PLUS-MINUS SIGN
    '\u2265'  # 0xF2 '≥' GREATER-THAN OR EQUAL TO
    '\u2264'  # 0xF3 '≤' LESS-THAN OR EQUAL TO
    '\u2320'  # 0xF4 '⌠' TOP HALF INTEGRAL
    '\u2321'  # 0xF5 '⌡' BOTTOM HALF INTEGRAL
    '\u00F7'  # 0xF6 '÷' DIVISION SIGN
    '\u2248'  # 0xF7 '≈' ALMOST EQUAL TO
    '\u00B0'  # 0xF8 '°' DEGREE SIGN
    '\u2219'  # 0xF9 '∙' BULLET OPERATOR
    '\u00B7'  # 0xFA '·' MIDDLE DOT
    '\u221A'  # 0xFB '√' SQUARE ROOT
    '\u207F'  # 0xFC 'ⁿ' SUPERSCRIPT LATIN SMALL LETTER N
    '\u00B2'  # 0xFD '²' SUPERSCRIPT TWO
    '\u25A0'  # 0xFE '■' BLACK SQUARE
    '\u00A0'  # 0xFF 'NBSP' NO-BREAK SPACE
)

encoding_table=codecs.charmap_build(decoding_table)

test_string = b'za\xA7\xA2\x92\x8D g\x91\x9El\x86 ja\xA6\xA4, ZA\xA1\xA3\x9C\x95 G\x90\x98L\x8F JA\xA0\xA5!'

