# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

from dingtalk.client.api.base import DingTalkBaseAPI


class TbDingDing(DingTalkBaseAPI):
    """
    钉钉API
    """

    def dingtalk_corp_message_corpconversation_sendmock(
            self,
            microapp_agent_id,
            to_user,
            to_party,
            message_type,
            message
    ):
        """
        企业会话消息发送
        测试接入top是否有问题
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27851

        :param microapp_agent_id: 微应用agentId
        :param to_user: 消息接收者userid列表
        :param to_party: 消息接收者部门列表
        :param message_type: 消息类型
        :param message: 消息体
        """
        return self._top_request(
            "dingtalk.corp.message.corpconversation.sendmock",
            {
                "microapp_agent_id": microapp_agent_id,
                "to_user": to_user,
                "to_party": to_party,
                "message_type": message_type,
                "message": message
            }
        )

    def dingtalk_corp_emp_search(
            self,
            keyword='',
            offset='',
            size=''
    ):
        """
        企业通讯录员工搜索
        钉钉企业通讯录搜索接口, 支持翻页
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27910

        :param keyword: 搜索关键字
        :param offset: 偏移量
        :param size: 请求数量
        """
        return self._top_request(
            "dingtalk.corp.emp.search",
            {
                "keyword": keyword,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_corp_ext_add(
            self,
            contact
    ):
        """
        添加企业外部联系人
        钉钉企业外部通讯录, 添加外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28177

        :param contact: 外部联系人信息
        """
        return self._top_request(
            "dingtalk.corp.ext.add",
            {
                "contact": contact
            },
            result_processor=lambda x: x["userid"]
        )

    def dingtalk_corp_ext_list(
            self,
            size='20',
            offset='0'
    ):
        """
        外部联系人列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28178

        :param size: 分页大小, 最大100
        :param offset: 偏移位置
        """
        return self._top_request(
            "dingtalk.corp.ext.list",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_corp_ext_listlabelgroups(
            self,
            size='20',
            offset='0'
    ):
        """
        标签列表
        拉取标签列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28179

        :param size: 分页大小,最大100
        :param offset: 偏移位置
        """
        return self._top_request(
            "dingtalk.corp.ext.listlabelgroups",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_corp_encryption_key_list(
            self
    ):
        """
        企业密钥列表
        查询企业密钥列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28386

        """
        return self._top_request(
            "dingtalk.corp.encryption.key.list"
        )

    def dingtalk_corp_conference_details_query(
            self,
            since_time='',
            limit='',
            caller_user_id='',
            member_user_id=''
    ):
        """
        钉钉企业电话会议详情记录查询
        查询企业电话会议详情记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28418

        :param since_time: 查询起始时间
        :param limit: 查询个数，上限100
        :param caller_user_id: 主叫userId
        :param member_user_id: 成员userId
        """
        return self._top_request(
            "dingtalk.corp.conference.details.query",
            {
                "since_time": since_time,
                "limit": limit,
                "caller_user_id": caller_user_id,
                "member_user_id": member_user_id
            }
        )

    def dingtalk_corp_ext_update(
            self,
            contact
    ):
        """
        更新外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28531

        :param contact: 外部联系人信息
        """
        return self._top_request(
            "dingtalk.corp.ext.update",
            {
                "contact": contact
            }
        )

    def dingtalk_corp_conversation_corpconversion_listmember(
            self,
            open_conversation_id,
            offset,
            count
    ):
        """
        获取企业群群成员接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28620

        :param open_conversation_id: 群组id
        :param offset: 群成员列表偏移量
        :param count: 本次请求获取群成员的大小，最大为100
        """
        return self._top_request(
            "dingtalk.corp.conversation.corpconversion.listmember",
            {
                "open_conversation_id": open_conversation_id,
                "offset": offset,
                "count": count
            }
        )

    def dingtalk_corp_conversation_corpconversion_getconversation(
            self,
            open_conversation_id
    ):
        """
        获取企业群基本信息
        获取企业群基本信息接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28621

        :param open_conversation_id: 群组ID
        """
        return self._top_request(
            "dingtalk.corp.conversation.corpconversion.getconversation",
            {
                "open_conversation_id": open_conversation_id
            }
        )

    def dingtalk_corp_message_corpconversation_asyncsend(
            self,
            msgtype,
            agent_id,
            msgcontent,
            userid_list='',
            dept_id_list='',
            to_all_user='false'
    ):
        """
        企业会话消息异步发送
        企业会话消息异步发送接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28915

        :param msgtype: 消息类型,如text、file、oa等，具体见文档
        :param agent_id: 微应用的id
        :param msgcontent: 与msgtype对应的消息体，具体见文档
        :param userid_list: 接收者的用户userid列表
        :param dept_id_list: 接收者的部门id列表
        :param to_all_user: 是否发送给企业全部用户
        """
        return self._top_request(
            "dingtalk.corp.message.corpconversation.asyncsend",
            {
                "msgtype": msgtype,
                "agent_id": agent_id,
                "msgcontent": msgcontent,
                "userid_list": userid_list,
                "dept_id_list": dept_id_list,
                "to_all_user": to_all_user
            }
        )

    def dingtalk_corp_message_corpconversation_getsendprogress(
            self,
            agent_id,
            task_id
    ):
        """
        获取异步发送企业会话消息的发送进度
        获取异步发送企业会话消息的进度
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28917

        :param agent_id: 发送消息时使用的微应用的id
        :param task_id: 发送消息时钉钉返回的任务id
        """
        return self._top_request(
            "dingtalk.corp.message.corpconversation.getsendprogress",
            {
                "agent_id": agent_id,
                "task_id": task_id
            }
        )

    def dingtalk_corp_message_corpconversation_getsendresult(
            self,
            agent_id='',
            task_id=''
    ):
        """
        获取异步向企业会话发送消息的结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28918

        :param agent_id: 微应用的agentid
        :param task_id: 异步任务的id
        """
        return self._top_request(
            "dingtalk.corp.message.corpconversation.getsendresult",
            {
                "agent_id": agent_id,
                "task_id": task_id
            }
        )

    def dingtalk_corp_message_corpconversation_asyncsendbycode(
            self,
            msgtype,
            agent_id,
            msgcontent,
            code,
            user_id_list='',
            dept_id_list='',
            to_all_user='false'
    ):
        """
        通过用户授权码异步向企业会话发送消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28919

        :param msgtype: 消息类型,如text、file、oa等，具体见文档
        :param agent_id: 微应用的id
        :param msgcontent: 与msgtype对应的消息体，具体见文档
        :param code: 用户操作产生的授权码
        :param user_id_list: 接收者的用户userid列表
        :param dept_id_list: 接收者的部门id列表
        :param to_all_user: 是否发送给企业全部用户
        """
        return self._top_request(
            "dingtalk.corp.message.corpconversation.asyncsendbycode",
            {
                "msgtype": msgtype,
                "agent_id": agent_id,
                "msgcontent": msgcontent,
                "code": code,
                "user_id_list": user_id_list,
                "dept_id_list": dept_id_list,
                "to_all_user": to_all_user
            }
        )

    def dingtalk_corp_chatbot_createorgbot(
            self,
            create_chat_bot_model
    ):
        """
        isv为企业创建企业机器人
        给ISV提供为企业创建机器人的接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28964

        :param create_chat_bot_model: 创建机器人modle
        """
        return self._top_request(
            "dingtalk.corp.chatbot.createorgbot",
            {
                "create_chat_bot_model": create_chat_bot_model
            }
        )

    def dingtalk_isv_call_getuserlist(
            self,
            start='0',
            offset='200'
    ):
        """
        dingtalk.isv.call.getuserlist
        删除ISV套件对应的企业下的可以主动调用接口发起免费电话的员工
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29002

        :param start: 游标开始值
        :param offset: 批量值
        """
        return self._top_request(
            "dingtalk.isv.call.getuserlist",
            {
                "start": start,
                "offset": offset
            }
        )

    def dingtalk_isv_call_calluser(
            self,
            staff_id,
            authed_corp_id,
            authed_staff_id
    ):
        """
        主叫方发起免费电话给授权企业下的授权范围内的人员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29003

        :param staff_id: isv套件所属企业下的员工userid
        :param authed_corp_id: 授权isv套件企业的corpid
        :param authed_staff_id: 授权isv套件企业的员工userid
        """
        return self._top_request(
            "dingtalk.isv.call.calluser",
            {
                "staff_id": staff_id,
                "authed_corp_id": authed_corp_id,
                "authed_staff_id": authed_staff_id
            }
        )

    def dingtalk_isv_call_setuserlist(
            self,
            staff_id_list
    ):
        """
        设置isv发起免费电话的主叫白名单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29004

        :param staff_id_list: 套件所所属企业免费电话主叫人员工号列表
        """
        return self._top_request(
            "dingtalk.isv.call.setuserlist",
            {
                "staff_id_list": staff_id_list
            }
        )

    def dingtalk_isv_call_removeuserlist(
            self,
            staff_id_list
    ):
        """
        删除isv免费电话员工名单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29005

        :param staff_id_list: 要删除的员工userid列表
        """
        return self._top_request(
            "dingtalk.isv.call.removeuserlist",
            {
                "staff_id_list": staff_id_list
            }
        )

    def dingtalk_corp_chatbot_updateorgbot(
            self,
            icon,
            name,
            chatbot_id
    ):
        """
        修改企业机器人
        小蜜isv修改机器人名字头像接口。(接口只给小蜜用，机器人应用会白名单处理)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29054

        :param icon: 头像的mediaId
        :param name: 机器人名字
        :param chatbot_id: 创建时返回的机器人Id
        """
        return self._top_request(
            "dingtalk.corp.chatbot.updateorgbot",
            {
                "icon": icon,
                "name": name,
                "chatbot_id": chatbot_id
            }
        )

    def dingtalk_corp_chatbot_listorgbot(
            self,
            agent_id,
            type
    ):
        """
        机器人查询接口
        小蜜isv查询给企业创建的机器人接口。(接口只给小蜜用，机器人应用会白名单处理)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29055

        :param agent_id: 微应用id
        :param type: 钉钉分配的类型
        """
        return self._top_request(
            "dingtalk.corp.chatbot.listorgbot",
            {
                "agent_id": agent_id,
                "type": type
            }
        )

    def dingtalk_corp_chatbot_listorgbotbytypeandbottype(
            self,
            type,
            bot_type
    ):
        """
        通过机器人type和botType查询机器人接口
        小蜜isv查询给企业创建的机器人接口。(接口只给小蜜用，机器人应用会白名单处理)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29081

        :param type: 机器人类型(钉钉分配)
        :param bot_type: 2-企业对内机器人，3-企业对外机器人
        """
        return self._top_request(
            "dingtalk.corp.chatbot.listorgbotbytypeandbottype",
            {
                "type": type,
                "bot_type": bot_type
            }
        )

    def dingtalk_smartwork_attends_listschedule(
            self,
            work_date,
            offset='0',
            size='200'
    ):
        """
        考勤排班信息按天全量查询接口
        按天查询企业考勤排班全量信息，使用分页功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29082

        :param work_date: 排班时间，只取年月日部分
        :param offset: 偏移位置
        :param size: 分页大小，最大200
        """
        return self._top_request(
            "dingtalk.smartwork.attends.listschedule",
            {
                "work_date": work_date,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_smartwork_attends_getsimplegroups(
            self,
            offset='0',
            size='10'
    ):
        """
        获取考勤组列表详情
        获取公司自身的考勤组列表详情信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29083

        :param offset: 偏移位置
        :param size: 分页大小，最大10
        """
        return self._top_request(
            "dingtalk.smartwork.attends.getsimplegroups",
            {
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_corp_role_simplelist(
            self,
            role_id,
            size='20',
            offset='0'
    ):
        """
        获取角色的员工列表
        获取企业中角色下的员工列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29204

        :param role_id: 角色ID
        :param size: 分页大小
        :param offset: 分页偏移
        """
        return self._top_request(
            "dingtalk.corp.role.simplelist",
            {
                "role_id": role_id,
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_corp_role_list(
            self,
            size='20',
            offset='0'
    ):
        """
        获取企业角色列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29205

        :param size: 分页大小
        :param offset: 分页偏移
        """
        return self._top_request(
            "dingtalk.corp.role.list",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_smartwork_bpms_processinstance_create(
            self,
            process_code,
            originator_user_id,
            dept_id,
            approvers,
            form_component_values,
            agent_id='',
            cc_list='',
            cc_position=''
    ):
        """
        发起审批实例
        企业或isv调用该api，来发起一个审批实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29498

        :param process_code: 审批流的唯一码
        :param originator_user_id: 审批实例发起人的userid
        :param dept_id: 发起人所在的部门
        :param approvers: 审批人userid列表
        :param form_component_values: 审批流表单参数
        :param agent_id: 企业微应用标识
        :param cc_list: 抄送人userid列表
        :param cc_position: 抄送时间,分为（START,FINISH,START_FINISH）
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.processinstance.create",
            {
                "process_code": process_code,
                "originator_user_id": originator_user_id,
                "dept_id": dept_id,
                "approvers": approvers,
                "form_component_values": form_component_values,
                "agent_id": agent_id,
                "cc_list": cc_list,
                "cc_position": cc_position
            }
        )

    def dingtalk_smartwork_checkin_record_get(
            self,
            userid_list,
            start_time,
            end_time,
            cursor,
            size
    ):
        """
        获取多个用户的签到记录
        查询多个用户一段时间范围内的签到记录，只给企业调用，ISV无法调用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29538

        :param userid_list: 需要查询的用户列表
        :param start_time: 起始时间,单位毫秒
        :param end_time: 截止时间，单位毫秒。如果是取1个人的数据，时间范围最大到10天，如果是取多个人的数据，时间范围最大1天。
        :param cursor: 分页查询的游标，最开始可以传0
        :param size: 分页查询的每页大小，最大100
        """
        return self._top_request(
            "dingtalk.smartwork.checkin.record.get",
            {
                "userid_list": userid_list,
                "start_time": start_time,
                "end_time": end_time,
                "cursor": cursor,
                "size": size
            }
        )

    def dingtalk_smartwork_bpms_process_sync(
            self,
            agent_id,
            src_process_code,
            target_process_code,
            biz_category_id='',
            process_name=''
    ):
        """
        更新审批流
        ISV调用该接口，可以更新对应授权企业的审批单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29766

        :param agent_id: 企业微应用标识
        :param src_process_code: 源审批流的唯一码
        :param target_process_code: 目标审批流的唯一码
        :param biz_category_id: 业务分类标识（建议采用JAVA包名的命名方式,如:com.alibaba）
        :param process_name: 审批流名称
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.process.sync",
            {
                "agent_id": agent_id,
                "src_process_code": src_process_code,
                "target_process_code": target_process_code,
                "biz_category_id": biz_category_id,
                "process_name": process_name
            }
        )

    def dingtalk_smartwork_bpms_processinstance_list(
            self,
            process_code,
            start_time,
            end_time='',
            size='10',
            cursor='0',
            userid_list=''
    ):
        """
        获取审批实例列表
        企业可以根据审批流的唯一标识，分页获取该审批流对应的审批实例。只能取到权限范围内的相关部门的审批实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29833

        :param process_code: 流程模板唯一标识，可在oa后台编辑审批表单部分查询
        :param start_time: 审批实例开始时间，毫秒级
        :param end_time: 审批实例结束时间，毫秒级，默认取当前值
        :param size: 分页参数，每页大小，最多传10
        :param cursor: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值
        :param userid_list: 发起人用户id列表
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.processinstance.list",
            {
                "process_code": process_code,
                "start_time": start_time,
                "end_time": end_time,
                "size": size,
                "cursor": cursor,
                "userid_list": userid_list
            }
        )

    def dingtalk_corp_role_deleterole(
            self,
            role_id
    ):
        """
        删除角色信息
        企业在做企业内部组织结构角色管理的时候，如果需要删除该企业下某个角色信息，可以调用该接口。调用的前提是该角色下面的所有员工都已经被删除该角色
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29977

        :param role_id: 角色id
        """
        return self._top_request(
            "dingtalk.corp.role.deleterole",
            {
                "role_id": role_id
            }
        )

    def dingtalk_corp_role_getrolegroup(
            self,
            group_id
    ):
        """
        获取角色组信息
        该接口通过groupId参数可以获取该角色组详细信息以及下面所有关联的角色的信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29978

        :param group_id: 角色组的Id
        """
        return self._top_request(
            "dingtalk.corp.role.getrolegroup",
            {
                "group_id": group_id
            }
        )

    def dingtalk_corp_role_addrolesforemps(
            self,
            rolelid_list,
            userid_list
    ):
        """
        批量为员工增加角色信息
        企业在做企业员工管理的时候，需要对部分员工进行角色分类，该接口可以批量为员工增加角色信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29979

        :param rolelid_list: 角色id list
        :param userid_list: 员工id list
        """
        return self._top_request(
            "dingtalk.corp.role.addrolesforemps",
            {
                "rolelid_list": rolelid_list,
                "userid_list": userid_list
            }
        )

    def dingtalk_corp_role_removerolesforemps(
            self,
            roleid_list,
            userid_list
    ):
        """
        批量删除员工角的色信息
        企业在做企业员工管理的时候，需要对部分员工进行角色分类，该接口可以批量删除员工的角色信息。 角色在钉钉的组织结构里面就是标签的意思，你可以批量为一批用户添加一批角色信息（dingtalk.corp.role.addrolesforemps），那么调用该接口就可以批量删除已经存在的角色和员工对应关系，角色和员工是多对多的关系。参考代码如下： req.setRolelidList('1,2,3,4,5'); // 已经存在的角色id列表 req.setUseridList('a,b,c,d,e'); // 用户的id列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29980

        :param roleid_list: 角色标签id
        :param userid_list: 用户userId
        """
        return self._top_request(
            "dingtalk.corp.role.removerolesforemps",
            {
                "roleid_list": roleid_list,
                "userid_list": userid_list
            }
        )

    def dingtalk_corp_invoice_gettitle(
            self
    ):
        """
        获取企业开票抬头信息
        该接口通过orgId参数可以获取该企业的发票抬头信息。
        注意：ISV必须先加入白名单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30318

        """
        return self._top_request(
            "dingtalk.corp.invoice.gettitle"
        )

    def dingtalk_corp_device_manage_get(
            self,
            device_service_id,
            device_id
    ):
        """
        获取单设备详情
        ISV或者企业通过deviceId来获取设备详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30500

        :param device_service_id: 设备服务商ID
        :param device_id: 设备ID
        """
        return self._top_request(
            "dingtalk.corp.device.manage.get",
            {
                "device_service_id": device_service_id,
                "device_id": device_id
            }
        )

    def dingtalk_corp_device_manage_querylist(
            self,
            device_service_id,
            cursor,
            size
    ):
        """
        拉取企业下某类设备列表
        此接口用于展示企业已经绑定的设备列表(分页接口)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30501

        :param device_service_id: 设备服务商ID
        :param cursor: 分页拉取设备列表的游标，首次拉取可传Null或者0
        :param size: 单次请求的大小，最大不超过20
        """
        return self._top_request(
            "dingtalk.corp.device.manage.querylist",
            {
                "device_service_id": device_service_id,
                "cursor": cursor,
                "size": size
            }
        )

    def dingtalk_corp_device_nick_update(
            self,
            device_service_id,
            device_id,
            new_nick
    ):
        """
        更改设备昵称
        修改企业绑定的智能硬件对应的昵称
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30514

        :param device_service_id: 设备服务商ID
        :param device_id: 设备ID
        :param new_nick: 设备新昵称
        """
        return self._top_request(
            "dingtalk.corp.device.nick.update",
            {
                "device_service_id": device_service_id,
                "device_id": device_id,
                "new_nick": new_nick
            }
        )

    def dingtalk_corp_device_manage_unbind(
            self,
            device_service_id,
            device_id
    ):
        """
        设备解绑
        此接口用于解绑已经绑定到xx企业的某个设备
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30530

        :param device_service_id: 设备服务商ID
        :param device_id: 设备ID
        """
        return self._top_request(
            "dingtalk.corp.device.manage.unbind",
            {
                "device_service_id": device_service_id,
                "device_id": device_id
            }
        )

    def dingtalk_corp_user_personainfo_get(
            self,
            userid
    ):
        """
        用户画像
        用户画像,包括用户性别，工作行业属性等
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31620

        :param userid: 员工id
        """
        return self._top_request(
            "dingtalk.corp.user.personainfo.get",
            {
                "userid": userid
            }
        )

    def dingtalk_corp_report_list(
            self,
            start_time,
            end_time,
            cursor,
            size,
            template_name='',
            userid=''
    ):
        """
        查询企业员工发出的日志列表
        企业可以根据员工userid或者日志模板名称，分页获取员工一段时间范围内在【日志】微应用发送的日志详细信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31998

        :param start_time: 查询起始时间
        :param end_time: 查询截止时间
        :param cursor: 查询游标，初始传入0，后续从上一次的返回值中获取
        :param size: 每页数据量
        :param template_name: 要查询的模板名称
        :param userid: 员工的userid
        """
        return self._top_request(
            "dingtalk.corp.report.list",
            {
                "start_time": start_time,
                "end_time": end_time,
                "cursor": cursor,
                "size": size,
                "template_name": template_name,
                "userid": userid
            }
        )

    def dingtalk_corp_chatbot_addchatbotinstance(
            self,
            chatbot_id,
            open_conversation_id,
            name='',
            icon_media_id=''
    ):
        """
        向群添加机器人接口
        企业应用，向自己的企业群，添加已安装的企业机器人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32058

        :param chatbot_id: 机器人id，由钉钉事先分配
        :param open_conversation_id: 创建群时返回的openConvsationId
        :param name: 机器人名字(如果为空，默认是机器人安装时的名字)
        :param icon_media_id: 机器人头像(如果为空，默认是机器人安装时的头像)
        """
        return self._top_request(
            "dingtalk.corp.chatbot.addchatbotinstance",
            {
                "chatbot_id": chatbot_id,
                "open_conversation_id": open_conversation_id,
                "name": name,
                "icon_media_id": icon_media_id
            }
        )

    def dingtalk_smartwork_attends_getleaveapproveduration(
            self,
            userid,
            from_date,
            to_date
    ):
        """
        计算请假时长
        钉钉考勤微应用，提供了排班的功能，企业管理员可以设置排班规则，该接口可以自动根据排班规则统计出每个员工的请假时长，进而可以与企业自有的请假／财务系统对接，进行工资统计，如果您的企业使用了钉钉考勤并希望依赖考勤系统自动计算员工请假时长，可选择使用此接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32125

        :param userid: 员工在企业内的UserID，企业用来唯一标识用户的字段。
        :param from_date: 请假开始时间
        :param to_date: 请假结束时间
        """
        return self._top_request(
            "dingtalk.smartwork.attends.getleaveapproveduration",
            {
                "userid": userid,
                "from_date": from_date,
                "to_date": to_date
            }
        )

    def dingtalk_smartwork_attends_getusergroup(
            self,
            userid
    ):
        """
        获取用户考勤组
        在钉钉考勤微应用中，考勤组是一类具有相同的班次、考勤位置等考勤规则的人或部门的组合，一个企业中的一个人只能属于一个考勤组。如果您的企业使用了钉钉考勤并希望获取员工的考勤组信息，可选择使用此接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32308

        :param userid: 员工在企业内的UserID，企业用来唯一标识用户的字段。
        """
        return self._top_request(
            "dingtalk.smartwork.attends.getusergroup",
            {
                "userid": userid
            }
        )

    def dingtalk_corp_calendar_create(
            self,
            create_vo=None
    ):
        """
        创建日程
        通过此接口可以把企业员工的待办事项导入到钉钉日历并在钉钉日历中展示, 支持任务, 会议,审批,普通日程等.
        该接口处于内部灰度阶段, 申请使用请邮件联系: zhaoting.yht@alibaba-inc.com
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32840

        :param create_vo: 创建日程实体
        """
        return self._top_request(
            "dingtalk.corp.calendar.create",
            {
                "create_vo": create_vo
            }
        )

    def dingtalk_corp_ding_create(
            self,
            creator_userid,
            receiver_userids,
            remind_type,
            remind_time,
            text_content,
            attachment=None
    ):
        """
        发DING通知
        通过此接口发DING通知给企业内部员工, 支持短信DING和应用内DING.
        该接口正在灰度内测中, 暂不对外开放
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32841

        :param creator_userid: 发送者工号
        :param receiver_userids: 接收者工号列表
        :param remind_type: 提醒类型:1-应用内;2-短信
        :param remind_time: 发送时间(单位:毫秒)
        :param text_content: 通知内容
        :param attachment: 附件内容
        """
        return self._top_request(
            "dingtalk.corp.ding.create",
            {
                "creator_userid": creator_userid,
                "receiver_userids": receiver_userids,
                "remind_type": remind_type,
                "remind_time": remind_time,
                "text_content": text_content,
                "attachment": attachment
            }
        )

    def dingtalk_smartwork_bpms_processinstance_get(
            self,
            process_instance_id
    ):
        """
        获取单个审批实例详情
        根据审批实例id，获取审批实例详情，详情包括审批表单信息、操作记录列表、操作人、抄送人、审批任务列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32977

        :param process_instance_id: 审批实例id
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.processinstance.get",
            {
                "process_instance_id": process_instance_id
            }
        )

    def dingtalk_corp_chatbot_install(
            self,
            chatbot_vo
    ):
        """
        ISV给企业安装机器人
        企业安装微应用后，ISV可通过本接口给企业上架一个企业机器人。目前灰度几个ISV，机器人应用会白名单处理
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33009

        :param chatbot_vo: 安装的机器人信息
        """
        return self._top_request(
            "dingtalk.corp.chatbot.install",
            {
                "chatbot_vo": chatbot_vo
            }
        )

    def dingtalk_corp_chatbot_updatebychatbotid(
            self,
            update_type,
            icon='',
            name='',
            chatbot_id='',
            preview_media_id='',
            description='',
            breif=''
    ):
        """
        ISV变更企业已安装机器人的名字或头像
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33010

        :param update_type: 更新名字或头像时是否更新群里已添加机器人的名字或头像。         * 0-不更新群里机器人名字或头像         * 1-更新群里机器人名字或头像
        :param icon: 机器人头像
        :param name: 机器人名字
        :param chatbot_id: 机器人id(钉钉分配)
        :param preview_media_id: 机器人预览图
        :param description: 机器人详细描述
        :param breif: 机器人简单描述
        """
        return self._top_request(
            "dingtalk.corp.chatbot.updatebychatbotid",
            {
                "update_type": update_type,
                "icon": icon,
                "name": name,
                "chatbot_id": chatbot_id,
                "preview_media_id": preview_media_id,
                "description": description,
                "breif": breif
            }
        )

    def dingtalk_corp_chatbot_listbychatbotids(
            self,
            chatbot_ids
    ):
        """
        isv查询企业已安装的企业机器人
        ISV根据chatbotId查询给企业已安装的机器人信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33011

        :param chatbot_ids: chatbotId列表
        """
        return self._top_request(
            "dingtalk.corp.chatbot.listbychatbotids",
            {
                "chatbot_ids": chatbot_ids
            }
        )

    def dingtalk_corp_extcontact_create(
            self,
            contact
    ):
        """
        外部联系人添加
        添加企业外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33400

        :param contact: 外部联系人信息
        """
        return self._top_request(
            "dingtalk.corp.extcontact.create",
            {
                "contact": contact
            }
        )

    def dingtalk_corp_smartdevice_hasface(
            self,
            userid_list
    ):
        """
        查询企业员工是否已录入人脸
        开通人脸识别的企业中，会存在部分用户录入照片，此接口用于批量查看员工是否已录入照片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33401

        :param userid_list: 查询用userid列表
        """
        return self._top_request(
            "dingtalk.corp.smartdevice.hasface",
            {
                "userid_list": userid_list
            }
        )

    def dingtalk_corp_extcontact_list(
            self,
            size='20',
            offset='0'
    ):
        """
        外部联系人列表
        获取企业外部联系人列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33402

        :param size: 分页大小, 最大100
        :param offset: 偏移位置
        """
        return self._top_request(
            "dingtalk.corp.extcontact.list",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_corp_extcontact_update(
            self,
            contact
    ):
        """
        外部联系人更新
        更新企业外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33403

        :param contact: 外部联系人信息
        """
        return self._top_request(
            "dingtalk.corp.extcontact.update",
            {
                "contact": contact
            }
        )

    def dingtalk_corp_extcontact_get(
            self,
            user_id
    ):
        """
        外部联系人详情
        获取企业外部联系人详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33404

        :param user_id: userId
        """
        return self._top_request(
            "dingtalk.corp.extcontact.get",
            {
                "user_id": user_id
            }
        )

    def dingtalk_corp_extcontact_delete(
            self,
            userid
    ):
        """
        外部联系人删除
        删除企业外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33405

        :param userid: userId
        """
        return self._top_request(
            "dingtalk.corp.extcontact.delete",
            {
                "userid": userid
            }
        )

    def dingtalk_corp_extcontact_listlabelgroups(
            self,
            size='20',
            offset='0'
    ):
        """
        外部联系人标签列表
        获取企业外部联系人标签列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33406

        :param size: 分页大小,最大100
        :param offset: 偏移位置
        """
        return self._top_request(
            "dingtalk.corp.extcontact.listlabelgroups",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_corp_device_manage_hasbinddevice(
            self,
            device_service_id
    ):
        """
        查询是否绑定某个设备产品
        允许开发者查询企业是否绑定了某个类型的钉钉智能硬件设备
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33410

        :param device_service_id: 设备产品类型 产品编码：M1：9 C1：14 M2：15 D1：24
        """
        return self._top_request(
            "dingtalk.corp.device.manage.hasbinddevice",
            {
                "device_service_id": device_service_id
            }
        )

    def dingtalk_corp_smartdevice_addface(
            self,
            face_vo
    ):
        """
        向企业员工添加人脸识别照片
        为用户添加识别照片，以及指定在终端上识别的有效期，指定用户类型对应识别提示语
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33411

        :param face_vo: 识别用户数据
        """
        return self._top_request(
            "dingtalk.corp.smartdevice.addface",
            {
                "face_vo": face_vo
            }
        )

    def dingtalk_corp_smartdevice_addrecognizenotify(
            self,
            notify_vo
    ):
        """
        添加用户识别成功后的通知
        当M2成功识别指定用户后，如需向指定用户发消息通知，使用些接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33417

        :param notify_vo: 通知数据
        """
        return self._top_request(
            "dingtalk.corp.smartdevice.addrecognizenotify",
            {
                "notify_vo": notify_vo
            }
        )

    def dingtalk_corp_smartdevice_receptionist_pushinfo(
            self,
            microapp_agent_id,
            desc_template,
            desc_content
    ):
        """
        智能前台消息推送
        智能前台开放部分功能区块，区块内的显示信息由isv基于企业不同推送展示消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33418

        :param microapp_agent_id: 微应用agentID
        :param desc_template: 智能前台信息展示模板ID，需要向智能硬件团队申请
        :param desc_content: 展示模板需要的变量数据
        """
        return self._top_request(
            "dingtalk.corp.smartdevice.receptionist.pushinfo",
            {
                "microapp_agent_id": microapp_agent_id,
                "desc_template": desc_template,
                "desc_content": desc_content
            }
        )

    def dingtalk_corp_hrm_employee_get(
            self,
            userid
    ):
        """
        获取智能人事员工花名册详细数据
        获取智能人事指定员工花名册详细数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33453

        :param userid: 查询用户userid
        """
        return self._top_request(
            "dingtalk.corp.hrm.employee.get",
            {
                "userid": userid
            }
        )

    def dingtalk_smartwork_blackboard_listtopten(
            self,
            userid
    ):
        """
        列出用户的公告列表
        列出用户当前有权限看到的10条公告，可用于在企业自定义工作首页进行公告轮播展示
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33463

        :param userid: 用户id
        """
        return self._top_request(
            "dingtalk.smartwork.blackboard.listtopten",
            {
                "userid": userid
            }
        )

    def dingtalk_corp_health_stepinfo_list(
            self,
            type,
            object_id,
            stat_dates
    ):
        """
        获取个人或部门钉钉运动步数
        查询企业用户或部门每天的钉钉运动步数，最多可以查询31天的数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33538

        :param type: 0表示取用户步数，1表示取部门步数
        :param object_id: 可以传入用户userid或者部门id
        :param stat_dates: 时间列表，注意时间格式是YYYYMMDD
        """
        return self._top_request(
            "dingtalk.corp.health.stepinfo.list",
            {
                "type": type,
                "object_id": object_id,
                "stat_dates": stat_dates
            }
        )

    def dingtalk_isv_blazers_generatecode(
            self,
            biz_id,
            ext
    ):
        """
        isv客户接入钉钉
        约定的isv通过此接口提供其业务对象唯一标示及相关信息，钉钉返回一个引流页并提供一个token，token后续供钉钉用于识别该业务对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34004

        :param biz_id: 业务对象唯一标示
        :param ext: 具体业务场景下约定的数据，格式：Map<String,String>
        """
        return self._top_request(
            "dingtalk.isv.blazers.generatecode",
            {
                "biz_id": biz_id,
                "ext": ext
            }
        )

    def dingtalk_corp_blazers_getbinddata(
            self
    ):
        """
        isv客户绑定数据
        获取绑定信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34005

        """
        return self._top_request(
            "dingtalk.corp.blazers.getbinddata"
        )

    def dingtalk_corp_blazers_getbizid(
            self
    ):
        """
        isv客户获取bizId
        获取bizId
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34006

        """
        return self._top_request(
            "dingtalk.corp.blazers.getbizid"
        )

    def dingtalk_corp_health_stepinfo_listbyuserid(
            self,
            userids,
            stat_date
    ):
        """
        批量查询多个用户的钉钉运动步数
        根据用户列表和时间列表，批量查询钉钉运动步数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34390

        :param userids: 员工userid列表，最多传50个
        :param stat_date: 时间，注意时间格式是YYMMDD
        """
        return self._top_request(
            "dingtalk.corp.health.stepinfo.listbyuserid",
            {
                "userids": userids,
                "stat_date": stat_date
            }
        )

    def dingtalk_corp_health_stepinfo_getuserstatus(
            self,
            userid
    ):
        """
        查询用户是否开启了钉钉运动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34391

        :param userid: 用户id
        """
        return self._top_request(
            "dingtalk.corp.health.stepinfo.getuserstatus",
            {
                "userid": userid
            }
        )

    def dingtalk_corp_hrm_employee_delemployeedismissionandhandover(
            self,
            dismission_info_with_hand_over,
            op_userid
    ):
        """
        确认离职并指定交接人，会从通讯录删除
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34825

        :param dismission_info_with_hand_over: 确认离职对象
        :param op_userid: 操作人userid
        """
        return self._top_request(
            "dingtalk.corp.hrm.employee.delemployeedismissionandhandover",
            {
                "dismission_info_with_hand_over": dismission_info_with_hand_over,
                "op_userid": op_userid
            }
        )

    def dingtalk_corp_hrm_employee_getdismissionlist(
            self,
            current,
            page_size,
            op_userid
    ):
        """
        获取离职人员信息
        智能人事开放接口-查询离职人员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34826

        :param current: 第几页，从1开始
        :param page_size: 一页多少数据，在1-100之间
        :param op_userid: 操作人userid
        """
        return self._top_request(
            "dingtalk.corp.hrm.employee.getdismissionlist",
            {
                "current": current,
                "page_size": page_size,
                "op_userid": op_userid
            }
        )

    def dingtalk_corp_dingindex_get(
            self,
            stat_dates
    ):
        """
        获取企业钉钉指数
        查询企业的日钉钉指数和月平均钉钉指数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34828

        :param stat_dates: 统计日期
        """
        return self._top_request(
            "dingtalk.corp.dingindex.get",
            {
                "stat_dates": stat_dates
            }
        )

    def dingtalk_corp_hrm_employee_modjobinfo(
            self,
            op_userid,
            hrm_api_job_model
    ):
        """
        更新员工工作信息
        钉钉智能人事开放接口-更新员工工作信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34829

        :param op_userid: 操作人userid，必须是拥有被操作人操作权限的管理员userid
        :param hrm_api_job_model: 员工信息对象，被操作人userid是必填，其他信息选填，填写则更新
        """
        return self._top_request(
            "dingtalk.corp.hrm.employee.modjobinfo",
            {
                "op_userid": op_userid,
                "hrm_api_job_model": hrm_api_job_model
            }
        )

    def dingtalk_corp_ding_receiverstatus_list(
            self,
            ding_id,
            page_size,
            page_no,
            confirmed_status=''
    ):
        """
        ding接收者状态列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34893

        :param ding_id: dingid
        :param page_size: 每页显示数量，最大值50
        :param page_no: 分页页码，从1开始
        :param confirmed_status: 确认状态，三种情况：不传表示查所有；传0表示查未确认状态；传1表示查已经确认状态；
        """
        return self._top_request(
            "dingtalk.corp.ding.receiverstatus.list",
            {
                "ding_id": ding_id,
                "page_size": page_size,
                "page_no": page_no,
                "confirmed_status": confirmed_status
            }
        )

    def dingtalk_corp_hrm_employee_setuserworkdata(
            self,
            op_userid,
            hrm_api_user_data_model
    ):
        """
        更新用户绩效数据
        钉钉智能人事开放接口-更新员工绩效信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35667

        :param op_userid: 操作人userid，必须是拥有被操作人操作权限的管理员userid
        :param hrm_api_user_data_model: 员工信息对象，被操作人userid是必填
        """
        return self._top_request(
            "dingtalk.corp.hrm.employee.setuserworkdata",
            {
                "op_userid": op_userid,
                "hrm_api_user_data_model": hrm_api_user_data_model
            }
        )

    def dingtalk_smartwork_bpms_processinstance_updatevariables(
            self,
            process_instance_id,
            variables,
            remark=''
    ):
        """
        更新审批实例的表单值
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35764

        :param process_instance_id: 审批实例id
        :param variables: 表单参数列表
        :param remark: 评论
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.processinstance.updatevariables",
            {
                "process_instance_id": process_instance_id,
                "variables": variables,
                "remark": remark
            }
        )

    def dingtalk_smartwork_bpms_processinstance_getwithform(
            self,
            process_instance_id
    ):
        """
        获取审批实例的详情和表单信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35765

        :param process_instance_id: 审批实例id
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.processinstance.getwithform",
            {
                "process_instance_id": process_instance_id
            }
        )

    def dingtalk_corp_ding_task_create(
            self,
            task_send_v_o
    ):
        """
        ding任务创建接口
        ding任务创建。不过会有一些限制，只能发送文本任务，不能设置抄送人，只能有一个任务执行人。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35778

        :param task_send_v_o: 任务对外接口
        """
        return self._top_request(
            "dingtalk.corp.ding.task.create",
            {
                "task_send_v_o": task_send_v_o
            }
        )

    def dingtalk_smartwork_bpms_process_getbybiztype(
            self,
            biz_type
    ):
        """
        根据审批套件标识获取审批模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35826

        :param biz_type: 套件开发时与审批约定的业务标识
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.process.getbybiztype",
            {
                "biz_type": biz_type
            }
        )

    def dingtalk_smartwork_bpms_process_getvisible(
            self,
            userid,
            process_code_list
    ):
        """
        检测用户是否有审批模板的可用权限
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35827

        :param userid: 员工ID
        :param process_code_list: 流程模板唯一标识，可在oa后台编辑审批表单部分查询
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.process.getvisible",
            {
                "userid": userid,
                "process_code_list": process_code_list
            }
        )

    def dingtalk_smartwork_bpms_processinstanceid_list(
            self,
            process_code,
            start_time,
            end_time='',
            cursor='',
            size='',
            userid_list=''
    ):
        """
        获取审批实例ID列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35878

        :param process_code: 流程模板唯一标识，可在oa后台编辑审批表单部分查询
        :param start_time: 审批实例开始时间，毫秒级
        :param end_time: 审批实例结束时间，毫秒级，默认取当前值
        :param cursor: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值
        :param size: 分页参数，每页大小，最多传20
        :param userid_list: 发起人用户id列表
        """
        return self._top_request(
            "dingtalk.smartwork.bpms.processinstanceid.list",
            {
                "process_code": process_code,
                "start_time": start_time,
                "end_time": end_time,
                "cursor": cursor,
                "size": size,
                "userid_list": userid_list
            }
        )

    def dingtalk_corp_deptgroup_syncuser(
            self,
            userid='',
            dept_id=''
    ):
        """
        单个成员的部门群同步
        从部门成员同步到部门群
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35919

        :param userid: 用户id
        :param dept_id: 部门id
        """
        return self._top_request(
            "dingtalk.corp.deptgroup.syncuser",
            {
                "userid": userid,
                "dept_id": dept_id
            }
        )

    def dingtalk_corp_liveness_get(
            self
    ):
        """
        获取企业活跃度
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36176

        """
        return self._top_request(
            "dingtalk.corp.liveness.get"
        )

    def dingtalk_corp_blazers_unbind(
            self
    ):
        """
        完全解绑
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36499

        """
        return self._top_request(
            "dingtalk.corp.blazers.unbind"
        )

    def dingtalk_corp_blazers_removemapping(
            self,
            biz_id=''
    ):
        """
        移除绑定关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36500

        :param biz_id: 商户唯一标识
        """
        return self._top_request(
            "dingtalk.corp.blazers.removemapping",
            {
                "biz_id": biz_id
            }
        )

    def dingtalk_corp_search_corpcontact_baseinfo(
            self,
            query,
            offset,
            size
    ):
        """
        搜索企业通讯录基本接口
        根据姓名和工号搜索企业员工
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36791

        :param query: 搜索词,长度大于2开始搜
        :param offset: 开始位置,从0开始
        :param size: 拉取个数上限100
        """
        return self._top_request(
            "dingtalk.corp.search.corpcontact.baseinfo",
            {
                "query": query,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_corp_smartdevice_getface(
            self,
            userid=''
    ):
        """
        获取用户授权底图照片
        获取指定企业员工的识别底图照片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36860

        :param userid: 员工ID
        """
        return self._top_request(
            "dingtalk.corp.smartdevice.getface",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_process_listbyuserid(
            self,
            offset,
            size,
            userid=''
    ):
        """
        根据用户id获取可见审批模板列表
        根据userid分页获取用户可见的审批模板列表，每次最多获取100个模板。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36861

        :param offset: 分页游标，从0开始。根据返回结果里的next_cursor是否为空来判断是否还有下一页，且再次调用时offset设置成next_cursor的值
        :param size: 分页大小，最大可设置成100
        :param userid: 用户id, 不传表示查询企业下所有审批模板
        """
        return self._top_request(
            "dingtalk.oapi.process.listbyuserid",
            {
                "offset": offset,
                "size": size,
                "userid": userid
            }
        )

    def dingtalk_oapi_report_template_listbyuserid(
            self,
            userid='',
            offset='0',
            size='100'
    ):
        """
        根据用户id获取可见的日志模板列表
        根据用户userId获取当前企业下可见的日志模板列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36909

        :param userid: 员工userId, 不传递表示获取所有日志模板
        :param offset: 分页游标，从0开始。根据返回结果里的next_cursor是否为空来判断是否还有下一页，且再次调用时offset设置成next_cursor的值
        :param size: 分页大小，最大可设置成100
        """
        return self._top_request(
            "dingtalk.oapi.report.template.listbyuserid",
            {
                "userid": userid,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_test_test(
            self,
            input=''
    ):
        """
        test
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36910

        :param input: 1
        """
        return self._top_request(
            "dingtalk.oapi.test.test",
            {
                "input": input
            }
        )

    def dingtalk_oapi_attendance_listschedule(
            self,
            work_date,
            offset='0',
            size='200'
    ):
        """
        考勤排班信息按天全量查询接口
        按天查询企业考勤排班全量信息，使用分页功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36979

        :param work_date: 排班时间，只取年月日部分
        :param offset: 偏移位置
        :param size: 分页大小，最大200
        """
        return self._top_request(
            "dingtalk.oapi.attendance.listschedule",
            {
                "workDate": work_date,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_role_list(
            self,
            size='20',
            offset='0'
    ):
        """
        获取企业角色列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36980

        :param size: 分页大小
        :param offset: 分页偏移
        """
        return self._top_request(
            "dingtalk.oapi.role.list",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_oapi_extcontact_create(
            self,
            contact
    ):
        """
        添加企业外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36981

        :param contact: 外部联系人信息
        """
        return self._top_request(
            "dingtalk.oapi.extcontact.create",
            {
                "contact": contact
            }
        )

    def dingtalk_oapi_role_simplelist(
            self,
            role_id,
            size='20',
            offset='0'
    ):
        """
        获取角色的员工列表
        获取企业中角色下的员工列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36982

        :param role_id: 角色ID
        :param size: 分页大小
        :param offset: 分页偏移
        """
        return self._top_request(
            "dingtalk.oapi.role.simplelist",
            {
                "role_id": role_id,
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_oapi_attendance_getsimplegroups(
            self,
            offset='0',
            size='10'
    ):
        """
        获取考勤组列表详情
        获取公司自身的考勤组列表详情信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36983

        :param offset: 偏移位置
        :param size: 分页大小，最大10
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getsimplegroups",
            {
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_attendance_getleaveapproveduration(
            self,
            userid,
            from_date,
            to_date
    ):
        """
        计算请假时长
        钉钉考勤微应用，提供了排班的功能，企业管理员可以设置排班规则，该接口可以自动根据排班规则统计出每个员工的请假时长，进而可以与企业自有的请假／财务系统对接，进行工资统计，如果您的企业使用了钉钉考勤并希望依赖考勤系统自动计算员工请假时长，可选择使用此接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36984

        :param userid: 员工在企业内的UserID，企业用来唯一标识用户的字段。
        :param from_date: 请假开始时间
        :param to_date: 请假结束时间
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getleaveapproveduration",
            {
                "userid": userid,
                "from_date": from_date,
                "to_date": to_date
            }
        )

    def dingtalk_oapi_role_addrolesforemps(
            self,
            role_ids,
            user_ids
    ):
        """
        批量为员工增加角色信息
        企业在做企业员工管理的时候，需要对部分员工进行角色分类，该接口可以批量为员工增加角色信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36985

        :param role_ids: 角色id list
        :param user_ids: 员工id list
        """
        return self._top_request(
            "dingtalk.oapi.role.addrolesforemps",
            {
                "roleIds": role_ids,
                "userIds": user_ids
            }
        )

    def dingtalk_oapi_extcontact_get(
            self,
            user_id
    ):
        """
        获取企业外部联系人详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36986

        :param user_id: userId
        """
        return self._top_request(
            "dingtalk.oapi.extcontact.get",
            {
                "user_id": user_id
            }
        )

    def dingtalk_oapi_attendance_getusergroup(
            self,
            userid
    ):
        """
        获取用户考勤组
        在钉钉考勤微应用中，考勤组是一类具有相同的班次、考勤位置等考勤规则的人或部门的组合，一个企业中的一个人只能属于一个考勤组。如果您的企业使用了钉钉考勤并希望获取员工的考勤组信息，可选择使用此接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36987

        :param userid: 员工在企业内的UserID，企业用来唯一标识用户的字段。
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getusergroup",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_role_removerolesforemps(
            self,
            role_ids,
            user_ids
    ):
        """
        批量删除员工角的色信息
        企业在做企业员工管理的时候，需要对部分员工进行角色分类，该接口可以批量删除员工的角色信息。 角色在钉钉的组织结构里面就是标签的意思，你可以批量为一批用户添加一批角色信息（dingtalk.corp.role.addrolesforemps），那么调用该接口就可以批量删除已经存在的角色和员工对应关系，角色和员工是多对多的关系。参考代码如下： req.setRolelidList('1,2,3,4,5'); // 已经存在的角色id列表 req.setUseridList('a,b,c,d,e'); // 用户的id列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36991

        :param role_ids: 角色标签id
        :param user_ids: 用户userId
        """
        return self._top_request(
            "dingtalk.oapi.role.removerolesforemps",
            {
                "roleIds": role_ids,
                "userIds": user_ids
            }
        )

    def dingtalk_oapi_role_deleterole(
            self,
            role_id
    ):
        """
        删除角色信息
        企业在做企业内部组织结构角色管理的时候，如果需要删除该企业下某个角色信息，可以调用该接口。调用的前提是该角色下面的所有员工都已经被删除该角色
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36992

        :param role_id: 角色id
        """
        return self._top_request(
            "dingtalk.oapi.role.deleterole",
            {
                "role_id": role_id
            }
        )

    def dingtalk_oapi_role_getrolegroup(
            self,
            group_id
    ):
        """
        获取角色组信息
        该接口通过groupId参数可以获取该角色组详细信息以及下面所有关联的角色的信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36997

        :param group_id: 角色组的Id
        """
        return self._top_request(
            "dingtalk.oapi.role.getrolegroup",
            {
                "group_id": group_id
            }
        )

    def dingtalk_oapi_checkin_record_get(
            self,
            userid_list,
            start_time,
            end_time,
            cursor,
            size
    ):
        """
        获取多个用户的签到记录
        查询多个用户一段时间范围内的签到记录，只给企业调用，ISV无法调用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37000

        :param userid_list: 需要查询的用户列表
        :param start_time: 起始时间,单位毫秒
        :param end_time: 截止时间，单位毫秒。如果是取1个人的数据，时间范围最大到10天，如果是取多个人的数据，时间范围最大1天。
        :param cursor: 分页查询的游标，最开始可以传0
        :param size: 分页查询的每页大小，最大100
        """
        return self._top_request(
            "dingtalk.oapi.checkin.record.get",
            {
                "userid_list": userid_list,
                "start_time": start_time,
                "end_time": end_time,
                "cursor": cursor,
                "size": size
            }
        )

    def dingtalk_oapi_extcontact_update(
            self,
            contact
    ):
        """
        更新企业外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37001

        :param contact: 外部联系人信息
        """
        return self._top_request(
            "dingtalk.oapi.extcontact.update",
            {
                "contact": contact
            }
        )

    def dingtalk_oapi_extcontact_delete(
            self,
            user_id
    ):
        """
        删除企业外部联系人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37002

        :param user_id: userId
        """
        return self._top_request(
            "dingtalk.oapi.extcontact.delete",
            {
                "user_id": user_id
            }
        )

    def dingtalk_oapi_process_sync(
            self,
            agent_id,
            src_process_code,
            target_process_code,
            biz_category_id='',
            process_name=''
    ):
        """
        更新审批流
        ISV调用该接口，可以更新对应授权企业的审批单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37003

        :param agent_id: 企业微应用标识
        :param src_process_code: 源审批流的唯一码
        :param target_process_code: 目标审批流的唯一码
        :param biz_category_id: 业务分类标识（建议采用JAVA包名的命名方式,如:com.alibaba）
        :param process_name: 审批流名称
        """
        return self._top_request(
            "dingtalk.oapi.process.sync",
            {
                "agent_id": agent_id,
                "src_process_code": src_process_code,
                "target_process_code": target_process_code,
                "biz_category_id": biz_category_id,
                "process_name": process_name
            }
        )

    def dingtalk_oapi_extcontact_list(
            self,
            size='20',
            offset='0'
    ):
        """
        获取企业外部联系人列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37004

        :param size: 分页大小, 最大100
        :param offset: 偏移位置
        """
        return self._top_request(
            "dingtalk.oapi.extcontact.list",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_oapi_extcontact_listlabelgroups(
            self,
            size='20',
            offset='0'
    ):
        """
        获取企业外部联系人标签列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37005

        :param size: 分页大小,最大100
        :param offset: 偏移位置
        """
        return self._top_request(
            "dingtalk.oapi.extcontact.listlabelgroups",
            {
                "size": size,
                "offset": offset
            }
        )

    def dingtalk_oapi_processinstance_create(
            self,
            process_code,
            originator_user_id,
            dept_id,
            form_component_values,
            agent_id='',
            approvers='',
            cc_list='',
            cc_position='',
            approvers_v2=None
    ):
        """
        发起审批实例
        企业或isv调用该api，来发起一个审批实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37006

        :param process_code: 审批流的唯一码
        :param originator_user_id: 审批实例发起人的userid
        :param dept_id: 发起人所在的部门
        :param form_component_values: 审批流表单参数
        :param agent_id: 企业微应用标识
        :param approvers: 审批人userid列表
        :param cc_list: 抄送人userid列表
        :param cc_position: 抄送时间,分为（START,FINISH,START_FINISH）
        :param approvers_v2: 审批人列表，支持会签/或签，优先级高于approvers变量
        """
        return self._top_request(
            "dingtalk.oapi.processinstance.create",
            {
                "process_code": process_code,
                "originator_user_id": originator_user_id,
                "dept_id": dept_id,
                "form_component_values": form_component_values,
                "agent_id": agent_id,
                "approvers": approvers,
                "cc_list": cc_list,
                "cc_position": cc_position,
                "approvers_v2": approvers_v2
            }
        )

    def dingtalk_oapi_processinstance_list(
            self,
            process_code,
            start_time,
            end_time='',
            size='10',
            cursor='0',
            userid_list=''
    ):
        """
        获取审批实例列表
        企业可以根据审批流的唯一标识，分页获取该审批流对应的审批实例。只能取到权限范围内的相关部门的审批实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37007

        :param process_code: 流程模板唯一标识，可在oa后台编辑审批表单部分查询
        :param start_time: 审批实例开始时间，毫秒级
        :param end_time: 审批实例结束时间，毫秒级，默认取当前值
        :param size: 分页参数，每页大小，最多传10
        :param cursor: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值
        :param userid_list: 发起人用户id列表
        """
        return self._top_request(
            "dingtalk.oapi.processinstance.list",
            {
                "process_code": process_code,
                "start_time": start_time,
                "end_time": end_time,
                "size": size,
                "cursor": cursor,
                "userid_list": userid_list
            }
        )

    def dingtalk_oapi_processinstance_get(
            self,
            process_instance_id
    ):
        """
        获取单个审批实例详情
        根据审批实例id，获取审批实例详情，详情包括审批表单信息、操作记录列表、操作人、抄送人、审批任务列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37008

        :param process_instance_id: 审批实例id
        """
        return self._top_request(
            "dingtalk.oapi.processinstance.get",
            {
                "process_instance_id": process_instance_id
            }
        )

    def dingtalk_oapi_report_list(
            self,
            start_time,
            end_time,
            cursor,
            size,
            template_name='',
            userid=''
    ):
        """
        查询企业员工发出的日志列表
        企业可以根据员工userid或者日志模板名称，分页获取员工一段时间范围内在【日志】微应用发送的日志详细信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37009

        :param start_time: 查询起始时间
        :param end_time: 查询截止时间
        :param cursor: 查询游标，初始传入0，后续从上一次的返回值中获取
        :param size: 每页数据量
        :param template_name: 要查询的模板名称
        :param userid: 员工的userid
        """
        return self._top_request(
            "dingtalk.oapi.report.list",
            {
                "start_time": start_time,
                "end_time": end_time,
                "cursor": cursor,
                "size": size,
                "template_name": template_name,
                "userid": userid
            }
        )

    def dingtalk_oapi_blackboard_listtopten(
            self,
            userid
    ):
        """
        列出用户的公告列表
        列出用户当前有权限看到的10条公告，可用于在企业自定义工作首页进行公告轮播展示
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37010

        :param userid: 用户id
        """
        return self._top_request(
            "dingtalk.oapi.blackboard.listtopten",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_health_stepinfo_list(
            self,
            type,
            object_id,
            stat_dates
    ):
        """
        获取个人或部门钉钉运动步数
        查询企业用户或部门每天的钉钉运动步数，最多可以查询31天的数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37011

        :param type: 0表示取用户步数，1表示取部门步数
        :param object_id: 可以传入用户userid或者部门id
        :param stat_dates: 时间列表，注意时间格式是YYYYMMDD
        """
        return self._top_request(
            "dingtalk.oapi.health.stepinfo.list",
            {
                "type": type,
                "object_id": object_id,
                "stat_dates": stat_dates
            }
        )

    def dingtalk_oapi_health_stepinfo_listbyuserid(
            self,
            userids,
            stat_date
    ):
        """
        批量查询多个用户的钉钉运动步数
        根据用户列表和时间列表，批量查询钉钉运动步数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37012

        :param userids: 员工userid列表，最多传50个
        :param stat_date: 时间，注意时间格式是YYMMDD
        """
        return self._top_request(
            "dingtalk.oapi.health.stepinfo.listbyuserid",
            {
                "userids": userids,
                "stat_date": stat_date
            }
        )

    def dingtalk_oapi_health_stepinfo_getuserstatus(
            self,
            userid
    ):
        """
        查询用户是否开启了钉钉运动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37013

        :param userid: 用户id
        """
        return self._top_request(
            "dingtalk.oapi.health.stepinfo.getuserstatus",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_process_copy(
            self,
            agent_id,
            process_code,
            biz_category_id='',
            process_name='',
            description='',
            copy_type='1'
    ):
        """
        复制审批流
        审批接口开放中的第一步，ISV调用此接口把它自身的审批流复制都授权企业中
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37014

        :param agent_id: 企业微应用标识
        :param process_code: 审批流的唯一码
        :param biz_category_id: 业务分类标识（建议采用JAVA包名的命名方式，）
        :param process_name: 审批流名称
        :param description: 审批流描述
        :param copy_type: 复制类型，1 不包含流程信息，2 包含流程信息且审批中员工可见。默认为1
        """
        return self._top_request(
            "dingtalk.oapi.process.copy",
            {
                "agent_id": agent_id,
                "process_code": process_code,
                "biz_category_id": biz_category_id,
                "process_name": process_name,
                "description": description,
                "copy_type": copy_type
            }
        )

    def dingtalk_oapi_message_corpconversation_asyncsendbycode(
            self,
            msgtype,
            agent_id,
            msgcontent,
            code,
            user_id_list='',
            dept_id_list='',
            to_all_user='false'
    ):
        """
        通过用户授权码异步向企业会话发送消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37019

        :param msgtype: 消息类型,如text、file、oa等，具体见文档
        :param agent_id: 微应用的id
        :param msgcontent: 与msgtype对应的消息体，具体见文档
        :param code: 用户操作产生的授权码
        :param user_id_list: 接收者的用户userid列表
        :param dept_id_list: 接收者的部门id列表
        :param to_all_user: 是否发送给企业全部用户
        """
        return self._top_request(
            "dingtalk.oapi.message.corpconversation.asyncsendbycode",
            {
                "msgtype": msgtype,
                "agent_id": agent_id,
                "msgcontent": msgcontent,
                "code": code,
                "user_id_list": user_id_list,
                "dept_id_list": dept_id_list,
                "to_all_user": to_all_user
            }
        )

    def dingtalk_oapi_message_corpconversation_asyncsend(
            self,
            msgtype,
            agent_id,
            msgcontent,
            userid_list='',
            dept_id_list='',
            to_all_user='false'
    ):
        """
        企业会话消息异步发送
        企业会话消息异步发送接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37020

        :param msgtype: 消息类型,如text、file、oa等，具体见文档
        :param agent_id: 微应用的id
        :param msgcontent: 与msgtype对应的消息体，具体见文档
        :param userid_list: 接收者的用户userid列表
        :param dept_id_list: 接收者的部门id列表
        :param to_all_user: 是否发送给企业全部用户
        """
        return self._top_request(
            "dingtalk.oapi.message.corpconversation.asyncsend",
            {
                "msgtype": msgtype,
                "agent_id": agent_id,
                "msgcontent": msgcontent,
                "userid_list": userid_list,
                "dept_id_list": dept_id_list,
                "to_all_user": to_all_user
            }
        )

    def dingtalk_corp_hrm_employee_addresumerecord(
            self,
            userid='',
            title='',
            content='',
            k_v_content='',
            phone_url='',
            pc_url='',
            web_url='',
            record_time_stamp=''
    ):
        """
        新增成长记录
        企业员工的成长记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37022

        :param userid: 被操作人userid
        :param title: 成长记录title
        :param content: 成长记录第一条内容
        :param k_v_content: 厂长记录kv展示内容：json格式，顺序展示
        :param phone_url: 手机端url
        :param pc_url: pc端url
        :param web_url: webOA后台url
        :param record_time_stamp: 20180428 零点零分零秒
        """
        return self._top_request(
            "dingtalk.corp.hrm.employee.addresumerecord",
            {
                "userid": userid,
                "title": title,
                "content": content,
                "k_v_content": k_v_content,
                "phone_url": phone_url,
                "pc_url": pc_url,
                "web_url": web_url,
                "record_time_stamp": record_time_stamp
            }
        )

    def dingtalk_oapi_message_corpconversation_getsendprogress(
            self,
            agent_id,
            task_id
    ):
        """
        获取异步发送企业会话消息的发送进度
        获取异步发送企业会话消息的进度
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37023

        :param agent_id: 发送消息时使用的微应用的id
        :param task_id: 发送消息时钉钉返回的任务id
        """
        return self._top_request(
            "dingtalk.oapi.message.corpconversation.getsendprogress",
            {
                "agent_id": agent_id,
                "task_id": task_id
            }
        )

    def dingtalk_oapi_message_corpconversation_getsendresult(
            self,
            agent_id='',
            task_id=''
    ):
        """
        获取异步向企业会话发送消息的结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37024

        :param agent_id: 微应用的agentid
        :param task_id: 异步任务的id
        """
        return self._top_request(
            "dingtalk.oapi.message.corpconversation.getsendresult",
            {
                "agent_id": agent_id,
                "task_id": task_id
            }
        )

    def dingtalk_oapi_media_upload(
            self,
            media,
            type=''
    ):
        """
        上传媒体文件
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37051

        :param media: form-data中媒体文件标识，有filename、filelength、content-type等信息
        :param type: 媒体文件类型，分别有图片（image）、语音（voice）、普通文件(file)
        """
        return self._top_request(
            "dingtalk.oapi.media.upload",
            {
                "media": media,
                "type": type
            }
        )

    def dingtalk_oapi_user_getuserinfo(
            self,
            code=''
    ):
        """
        通过免登码获取用户userid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37091

        :param code: requestAuthCode接口中获取的CODE
        """
        return self._top_request(
            "dingtalk.oapi.user.getuserinfo",
            {
                "code": code
            }
        )

    def dingtalk_oapi_attendance_list(
            self,
            work_date_from='',
            work_date_to='',
            user_id_list='',
            offset='',
            limit='',
            is_i18n=''
    ):
        """
        考勤打卡数据开放
        该接口仅限企业接入使用，用于返回企业内员工的实际打卡结果。比如，企业给一个员工设定的排班是上午9点和下午6点各打一次卡，即使员工在这期间打了多次，该接口也只会返回两条记录，包括上午的打卡结果和下午的打卡结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37092

        :param work_date_from: 查询考勤打卡记录的起始工作日
        :param work_date_to: 查询考勤打卡记录的结束工作日
        :param user_id_list: 员工在企业内的UserID列表，企业用来唯一标识用户的字段
        :param offset: 表示获取考勤数据的起始点，第一次传0，如果还有多余数据，下次获取传的offset值为之前的offset+limit
        :param limit: 表示获取考勤数据的条数，最大不能超过50条
        :param is_i18n: 是否国际化
        """
        return self._top_request(
            "dingtalk.oapi.attendance.list",
            {
                "workDateFrom": work_date_from,
                "workDateTo": work_date_to,
                "userIdList": user_id_list,
                "offset": offset,
                "limit": limit,
                "isI18n": is_i18n
            }
        )

    def dingtalk_oapi_attendance_listRecord(
            self,
            user_ids,
            check_date_from,
            check_date_to,
            is_i18n=''
    ):
        """
        考勤打卡记录开放
        该接口仅限企业接入使用，用于返回企业内员工的实际打卡记录。比如，企业给一个员工设定的排班是上午9点和下午6点各打一次卡，但是员工在这期间打了多次，该接口会把所有的打卡记录都返回。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37094

        :param user_ids: 企业内的员工id列表，最多不能超过50个
        :param check_date_from: 查询考勤打卡记录的结束工作日。注意，起始与结束工作日最多相隔7天
        :param check_date_to: 查询考勤打卡记录的结束工作日。注意，起始与结束工作日最多相隔7天
        :param is_i18n: 是否国际化
        """
        return self._top_request(
            "dingtalk.oapi.attendance.listRecord",
            {
                "userIds": user_ids,
                "checkDateFrom": check_date_from,
                "checkDateTo": check_date_to,
                "isI18n": is_i18n
            }
        )

    def dingtalk_oapi_service_set_corp_ipwhitelist(
            self,
            auth_corpid='',
            ip_whitelist=''
    ):
        """
        ISV为授权方的企业单独设置IP白名单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37095

        :param auth_corpid: 授权方corpid
        :param ip_whitelist: 要为其设置的IP白名单,格式支持IP段,用星号表示,如【5.6.*.*】,代表从【5.6.0.*】到【5.6.255.*】的任意IP,在第三段设为星号时,将忽略第四段的值,注意:仅支持后两段设置为星号
        """
        return self._top_request(
            "dingtalk.oapi.service.set_corp_ipwhitelist",
            {
                "auth_corpid": auth_corpid,
                "ip_whitelist": ip_whitelist
            }
        )

    def dingtalk_oapi_service_reauth_corp(
            self,
            app_id='',
            corpid_list=''
    ):
        """
        重新授权未激活应用的企业
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37096

        :param app_id: 套件下的微应用ID
        :param corpid_list: 未激活的corpid列表
        """
        return self._top_request(
            "dingtalk.oapi.service.reauth_corp",
            {
                "app_id": app_id,
                "corpid_list": corpid_list
            }
        )

    def dingtalk_oapi_chat_send(
            self,
            chatid='',
            action_card=None,
            oa=None,
            voice=None,
            file=None,
            image=None,
            link=None,
            text=None,
            msgtype='',
            markdown=None,
            msg=None
    ):
        """
        发送群消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37099

        :param chatid: 群会话id
        :param action_card: actionCard消息
        :param oa: OA消息
        :param voice: 语音消息
        :param file: 文件消息
        :param image: 图片消息
        :param link: 链接消息
        :param text: 文本消息
        :param msgtype: 消息类型
        :param markdown: markdown消息
        :param msg: 消息格式
        """
        return self._top_request(
            "dingtalk.oapi.chat.send",
            {
                "chatid": chatid,
                "action_card": action_card,
                "oa": oa,
                "voice": voice,
                "file": file,
                "image": image,
                "link": link,
                "text": text,
                "msgtype": msgtype,
                "markdown": markdown,
                "msg": msg
            }
        )

    def dingtalk_oapi_chat_getReadList(
            self,
            message_id='',
            cursor='',
            size=''
    ):
        """
        查询群消息已读人员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37100

        :param message_id: 发送群消息接口返回的加密消息id
        :param cursor: 分页查询的游标，第一次可以传0，后续传返回结果中的next_cursor的值。当返回结果中，没有next_cursor时，表示没有后续的数据了，可以结束调用
        :param size: 分页查询的大小，最大可以传100
        """
        return self._top_request(
            "dingtalk.oapi.chat.getReadList",
            {
                "messageId": message_id,
                "cursor": cursor,
                "size": size
            }
        )

    def dingtalk_oapi_chat_update(
            self,
            chatid='',
            name='',
            owner='',
            owner_type='',
            add_useridlist='',
            del_useridlist='',
            add_extidlist='',
            del_extidlist='',
            icon='',
            is_ban='',
            searchable='',
            validation_type='',
            mention_all_authority='',
            management_type='',
            chat_banned_type='',
            show_history_type=''
    ):
        """
        修改群会话
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37102

        :param chatid: 群会话id
        :param name: 群名称
        :param owner: 群主的userId
        :param owner_type: 群主类型，emp：企业员工，ext：外部联系人
        :param add_useridlist: 添加成员列表
        :param del_useridlist: 删除成员列表
        :param add_extidlist: 添加外部联系人成员列表
        :param del_extidlist: 删除外部联系人成员列表
        :param icon: 群头像mediaId
        :param is_ban: 是否禁言
        :param searchable: 群可搜索，0-默认，不可搜索，1-可搜索
        :param validation_type: 入群验证，0：不入群验证（默认） 1：入群验证
        :param mention_all_authority: @all 权限，0-默认，所有人，1-仅群主可@all
        :param management_type: 管理类型，0-默认，所有人可管理，1-仅群主可管理
        :param chat_banned_type: 群禁言，0-默认，不禁言，1-全员禁言
        :param show_history_type: 新成员可查看聊天历史 0否 1是
        """
        return self._top_request(
            "dingtalk.oapi.chat.update",
            {
                "chatid": chatid,
                "name": name,
                "owner": owner,
                "ownerType": owner_type,
                "add_useridlist": add_useridlist,
                "del_useridlist": del_useridlist,
                "add_extidlist": add_extidlist,
                "del_extidlist": del_extidlist,
                "icon": icon,
                "isBan": is_ban,
                "searchable": searchable,
                "validationType": validation_type,
                "mentionAllAuthority": mention_all_authority,
                "managementType": management_type,
                "chatBannedType": chat_banned_type,
                "showHistoryType": show_history_type
            }
        )

    def dingtalk_oapi_department_list_parent_depts(
            self,
            user_id=''
    ):
        """
        查询指定用户的所有上级父部门路径
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37103

        :param user_id: 用户userId
        """
        return self._top_request(
            "dingtalk.oapi.department.list_parent_depts",
            {
                "userId": user_id
            }
        )

    def dingtalk_oapi_department_list_parent_depts_by_dept(
            self,
            id=''
    ):
        """
        查询部门的所有上级父部门路径
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37104

        :param id: 部门id
        """
        return self._top_request(
            "dingtalk.oapi.department.list_parent_depts_by_dept",
            {
                "id": id
            }
        )

    def dingtalk_oapi_chat_create(
            self,
            name='',
            owner='',
            useridlist='',
            owner_type='',
            conversation_tag='',
            show_history_type='',
            extidlist='',
            searchable='',
            validation_type='',
            mention_all_authority='',
            management_type='',
            chat_banned_type='',
            icon=''
    ):
        """
        创建群会话
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37105

        :param name: 群名称
        :param owner: 群主的userId
        :param useridlist: 群成员userId列表
        :param owner_type: 群主类型，emp：企业员工，ext：外部联系人
        :param conversation_tag: 群类型，2：企业群，0：普通群
        :param show_history_type: 新成员可查看100条聊天历史消息的类型，1：可查看，默认或0：不可查看
        :param extidlist: 外部联系人成员列表
        :param searchable: 群可搜索，0-默认，不可搜索，1-可搜索
        :param validation_type: 入群验证，0：不入群验证（默认） 1：入群验证
        :param mention_all_authority: @all 权限，0-默认，所有人，1-仅群主可@all
        :param management_type: 管理类型，0-默认，所有人可管理，1-仅群主可管理
        :param chat_banned_type: 群禁言，0-默认，不禁言，1-全员禁言
        :param icon: 群头像资源id
        """
        return self._top_request(
            "dingtalk.oapi.chat.create",
            {
                "name": name,
                "owner": owner,
                "useridlist": useridlist,
                "ownerType": owner_type,
                "conversationTag": conversation_tag,
                "showHistoryType": show_history_type,
                "extidlist": extidlist,
                "searchable": searchable,
                "validationType": validation_type,
                "mentionAllAuthority": mention_all_authority,
                "managementType": management_type,
                "chatBannedType": chat_banned_type,
                "icon": icon
            }
        )

    def dingtalk_oapi_department_update(
            self,
            id='',
            org_dept_owner='',
            outer_permit_users='',
            outer_permit_depts='',
            outer_dept='',
            user_perimits='',
            dept_perimits='',
            dept_hiding='',
            dept_manager_userid_list='',
            auto_add_user='',
            create_dept_group='',
            order='',
            parentid='',
            lang='',
            name='',
            source_identifier='',
            user_permits='',
            dept_permits='',
            outer_dept_only_self='',
            group_contain_sub_dept='',
            group_contain_outer_dept='',
            group_contain_hidden_dept=''
    ):
        """
        更新部门
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37106

        :param id: 部门id
        :param org_dept_owner: 企业群群主
        :param outer_permit_users: 本部门的员工仅可见员工自己为true时，可以配置额外可见人员，值为userid组成的的字符串，使用|符号进行分割。总数不能超过200。
        :param outer_permit_depts: 本部门的员工仅可见员工自己为true时，可以配置额外可见部门，值为部门id组成的的字符串，使用|符号进行分割。总数不能超过200。
        :param outer_dept: 是否本部门的员工仅可见员工自己, 为true时，本部门员工默认只能看到员工自己
        :param user_perimits: 可以查看指定隐藏部门的其他人员列表，如果部门隐藏，则此值生效，取值为其他的人员userid组成的的字符串，使用| 符号进行分割。总数不能超过200。
        :param dept_perimits: 可以查看指定隐藏部门的其他部门列表，如果部门隐藏，则此值生效，取值为其他的部门id组成的的字符串，使用 | 符号进行分割。总数不能超过200。
        :param dept_hiding: 是否隐藏部门, true表示隐藏, false表示显示
        :param dept_manager_userid_list: 部门的主管列表,取值为由主管的userid组成的字符串，不同的userid使用’| 符号进行分割
        :param auto_add_user: 如果有新人加入部门是否会自动加入部门群
        :param create_dept_group: 是否创建一个关联此部门的企业群
        :param order: 在父部门中的次序值。order值小的排序靠前
        :param parentid: 父部门id。根部门id为1
        :param lang: 通讯录语言(默认zh_CN另外支持en_US)
        :param name: 部门名称。长度限制为1~64个字符。不允许包含字符‘-’‘，’以及‘,’。
        :param source_identifier: 部门标识字段，开发者可用该字段来唯一标识一个部门，并与钉钉外部通讯录里的部门做映射
        :param user_permits: 可以查看指定隐藏部门的其他人员列表，如果部门隐藏，则此值生效，取值为其他的人员userid组成的的字符串，使用| 符号进行分割。总数不能超过200。
        :param dept_permits: 可以查看指定隐藏部门的其他部门列表，如果部门隐藏，则此值生效，取值为其他的部门id组成的的字符串，使用 | 符号进行分割。总数不能超过200。
        :param outer_dept_only_self: 是否只能看到所在部门及下级部门通讯录
        :param group_contain_sub_dept: 本门群是否包含子部门
        :param group_contain_outer_dept: 部门群是否包含外包部门
        :param group_contain_hidden_dept: 部门群是否包含隐藏部门
        """
        return self._top_request(
            "dingtalk.oapi.department.update",
            {
                "id": id,
                "orgDeptOwner": org_dept_owner,
                "outerPermitUsers": outer_permit_users,
                "outerPermitDepts": outer_permit_depts,
                "outerDept": outer_dept,
                "userPerimits": user_perimits,
                "deptPerimits": dept_perimits,
                "deptHiding": dept_hiding,
                "deptManagerUseridList": dept_manager_userid_list,
                "autoAddUser": auto_add_user,
                "createDeptGroup": create_dept_group,
                "order": order,
                "parentid": parentid,
                "lang": lang,
                "name": name,
                "sourceIdentifier": source_identifier,
                "userPermits": user_permits,
                "deptPermits": dept_permits,
                "outerDeptOnlySelf": outer_dept_only_self,
                "groupContainSubDept": group_contain_sub_dept,
                "groupContainOuterDept": group_contain_outer_dept,
                "groupContainHiddenDept": group_contain_hidden_dept
            }
        )

    def dingtalk_oapi_department_list(
            self,
            lang='',
            fetch_child='',
            id=''
    ):
        """
        获取部门列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37107

        :param lang: 通讯录语言(默认zh_CN，未来会支持en_US)
        :param fetch_child: 是否递归部门的全部子部门，ISV微应用固定传递false。
        :param id: 父部门id(如果不传，默认部门为根部门，根部门ID为1)
        """
        return self._top_request(
            "dingtalk.oapi.department.list",
            {
                "lang": lang,
                "fetch_child": fetch_child,
                "id": id
            }
        )

    def dingtalk_oapi_user_getUseridByUnionid(
            self,
            unionid=''
    ):
        """
        根据unionid获取成员的userid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37108

        :param unionid: 用户在当前钉钉开放平台账号范围内的唯一标识，同一个钉钉开放平台账号可以包含多个开放应用，同时也包含ISV的套件应用及企业应用
        """
        return self._top_request(
            "dingtalk.oapi.user.getUseridByUnionid",
            {
                "unionid": unionid
            }
        )

    def dingtalk_oapi_user_can_access_microapp(
            self,
            app_id='',
            user_id=''
    ):
        """
        获取管理员的微应用管理权限
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37109

        :param app_id: 微应用id
        :param user_id: 员工唯一标识ID
        """
        return self._top_request(
            "dingtalk.oapi.user.can_access_microapp",
            {
                "appId": app_id,
                "userId": user_id
            }
        )

    def dingtalk_oapi_user_get_admin(
            self
    ):
        """
        获取管理员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37110

        """
        return self._top_request(
            "dingtalk.oapi.user.get_admin"
        )

    def dingtalk_oapi_user_list(
            self,
            lang='',
            department_id='',
            offset='',
            size='',
            order=''
    ):
        """
        获取部门成员（详情）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37111

        :param lang: 通讯录语言(默认zh_CN另外支持en_US)
        :param department_id: 获取的部门id
        :param offset: 支持分页查询，与size参数同时设置时才生效，此参数代表偏移量
        :param size: 支持分页查询，与offset参数同时设置时才生效，此参数代表分页大小，最大100
        :param order: 支持分页查询，部门成员的排序规则，默认不传是按自定义排序；entry_asc代表按照进入部门的时间升序，entry_desc代表按照进入部门的时间降序，modify_asc代表按照部门信息修改时间升序，modify_desc代表按照部门信息修改时间降序，custom代表用户定义(未定义时按照拼音)排序
        """
        return self._top_request(
            "dingtalk.oapi.user.list",
            {
                "lang": lang,
                "department_id": department_id,
                "offset": offset,
                "size": size,
                "order": order
            }
        )

    def dingtalk_oapi_user_simplelist(
            self,
            lang='',
            department_id='',
            offset='',
            size='',
            order=''
    ):
        """
        获取部门成员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37113

        :param lang: 通讯录语言(默认zh_CN另外支持en_US)
        :param department_id: 获取的部门id
        :param offset: 支持分页查询，与size参数同时设置时才生效，此参数代表偏移量
        :param size: 支持分页查询，与offset参数同时设置时才生效，此参数代表分页大小，最大100
        :param order: 支持分页查询，部门成员的排序规则，默认不传是按自定义排序；entry_asc代表按照进入部门的时间升序，entry_desc代表按照进入部门的时间降序，modify_asc代表按照部门信息修改时间升序，modify_desc代表按照部门信息修改时间降序，custom代表用户定义(未定义时按照拼音)排序
        """
        return self._top_request(
            "dingtalk.oapi.user.simplelist",
            {
                "lang": lang,
                "department_id": department_id,
                "offset": offset,
                "size": size,
                "order": order
            }
        )

    def dingtalk_oapi_user_batchdelete(
            self,
            useridlist=''
    ):
        """
        批量删除成员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37115

        :param useridlist: 员工UserID列表。列表长度在1到20之间
        """
        return self._top_request(
            "dingtalk.oapi.user.batchdelete",
            {
                "useridlist": useridlist
            }
        )

    def dingtalk_oapi_user_delete(
            self,
            userid=''
    ):
        """
        删除成员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37116

        :param userid: 员工唯一标识ID（不可修改）
        """
        return self._top_request(
            "dingtalk.oapi.user.delete",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_user_create(
            self,
            userid='',
            is_senior='',
            is_hide='',
            jobnumber='',
            email='',
            remark='',
            work_place='',
            tel='',
            mobile='',
            position='',
            department='',
            name='',
            extattr='',
            org_email='',
            order_in_depts='',
            hired_date=''
    ):
        """
        创建成员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37117

        :param userid: 员工唯一标识ID（不可修改），企业内必须唯一。长度为1~64个字符，如果不传，服务器将自动生成一个userid
        :param is_senior: 是否高管模式，true表示是，false表示不是。开启后，手机号码对所有员工隐藏。普通员工无法对其发DING、发起钉钉免费商务电话。高管之间不受影响。
        :param is_hide: 是否号码隐藏, true表示隐藏, false表示不隐藏。隐藏手机号后，手机号在个人资料页隐藏，但仍可对其发DING、发起钉钉免费商务电话。
        :param jobnumber: 员工工号。对应显示到OA后台和客户端个人资料的工号栏目。长度为0~64个字符
        :param email: 邮箱。长度为0~64个字符。企业内必须唯一，不可重复
        :param remark: 备注，长度为0~1000个字符
        :param work_place: 办公地点，长度为0~50个字符
        :param tel: 分机号，长度为0~50个字符，企业内必须唯一，不可重复
        :param mobile: 手机号码，企业内必须唯一，不可重复
        :param position: 职位信息。长度为0~64个字符
        :param department: 数组类型，数组里面值为整型，成员所属部门id列表
        :param name: 成员名称。长度为1~64个字符
        :param extattr: 扩展属性，可以设置多种属性(但手机上最多只能显示10个扩展属性，具体显示哪些属性，请到OA管理后台->设置->通讯录信息设置和OA管理后台->设置->手机端显示信息设置)
        :param org_email: 员工的企业邮箱，员工的企业邮箱已开通，才能增加此字段， 否则会报错
        :param order_in_depts: 在对应的部门中的排序, Map结构的json字符串, key是部门的Id, value是人员在这个部门的排序值
        :param hired_date: 入职时间
        """
        return self._top_request(
            "dingtalk.oapi.user.create",
            {
                "userid": userid,
                "isSenior": is_senior,
                "isHide": is_hide,
                "jobnumber": jobnumber,
                "email": email,
                "remark": remark,
                "workPlace": work_place,
                "tel": tel,
                "mobile": mobile,
                "position": position,
                "department": department,
                "name": name,
                "extattr": extattr,
                "orgEmail": org_email,
                "orderInDepts": order_in_depts,
                "hiredDate": hired_date
            }
        )

    def dingtalk_oapi_department_list_ids(
            self,
            id=''
    ):
        """
        获取子部门ID列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37118

        :param id: 部门id
        """
        return self._top_request(
            "dingtalk.oapi.department.list_ids",
            {
                "id": id
            }
        )

    def dingtalk_oapi_user_update(
            self,
            userid='',
            name='',
            work_place='',
            remark='',
            is_senior='',
            org_email='',
            manager_userid='',
            tel='',
            order_in_depts='',
            mobile='',
            department='',
            email='',
            position='',
            extattr='',
            jobnumber='',
            is_hide='',
            lang='',
            hired_date=''
    ):
        """
        更新用户详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37119

        :param userid: 用户id
        :param name: 名字
        :param work_place: 工作地点
        :param remark: 备注
        :param is_senior: 是否高管模式
        :param org_email: 公司邮箱
        :param manager_userid: 主管
        :param tel: 分机号，长度为0~50个字符
        :param order_in_depts: 实际是Map的序列化字符串
        :param mobile: 手机号
        :param department: 部门列表
        :param email: 邮箱
        :param position: 职位
        :param extattr: 扩展属性
        :param jobnumber: 工号
        :param is_hide: 是否号码隐藏
        :param lang: 通讯录语言(默认zh_CN另外支持en_US)
        :param hired_date: 入职时间
        """
        return self._top_request(
            "dingtalk.oapi.user.update",
            {
                "userid": userid,
                "name": name,
                "workPlace": work_place,
                "remark": remark,
                "isSenior": is_senior,
                "orgEmail": org_email,
                "managerUserid": manager_userid,
                "tel": tel,
                "orderInDepts": order_in_depts,
                "mobile": mobile,
                "department": department,
                "email": email,
                "position": position,
                "extattr": extattr,
                "jobnumber": jobnumber,
                "isHide": is_hide,
                "lang": lang,
                "hiredDate": hired_date
            }
        )

    def dingtalk_oapi_user_get(
            self,
            userid
    ):
        """
        获取用户详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37120

        :param userid: userid
        """
        return self._top_request(
            "dingtalk.oapi.user.get",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_sso_getuserinfo(
            self,
            code='',
            access_token=''
    ):
        """
        通过CODE换取微应用管理员的身份信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37121

        :param code: 再次强调，此token不同于一般的accesstoken，需要调用获取微应用管理员免登需要的AccessToken
        :param access_token: 通过Oauth认证给URL带上的CODE
        """
        return self._top_request(
            "dingtalk.oapi.sso.getuserinfo",
            {
                "code": code,
                "access_token": access_token
            }
        )

    def dingtalk_oapi_chat_get(
            self,
            chatid=''
    ):
        """
        获取群会话
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37139

        :param chatid: 群会话的id
        """
        return self._top_request(
            "dingtalk.oapi.chat.get",
            {
                "chatid": chatid
            }
        )

    def dingtalk_oapi_department_get(
            self,
            id='',
            lang=''
    ):
        """
        获取部门详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37140

        :param id: 部门id
        :param lang: 通讯录语言(默认zh_CN，未来会支持en_US)
        """
        return self._top_request(
            "dingtalk.oapi.department.get",
            {
                "id": id,
                "lang": lang
            }
        )

    def dingtalk_oapi_department_delete(
            self,
            id=''
    ):
        """
        删除部门
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37141

        :param id: 部门id
        """
        return self._top_request(
            "dingtalk.oapi.department.delete",
            {
                "id": id
            }
        )

    def dingtalk_oapi_department_create(
            self,
            parentid='',
            parent_balance_first='',
            share_balance='',
            outer_permit_users='',
            outer_permit_depts='',
            outer_dept='',
            user_perimits='',
            dept_perimits='',
            dept_hiding='',
            create_dept_group='',
            order='',
            name='',
            source_identifier='',
            dept_permits='',
            user_permits='',
            outer_dept_only_self=''
    ):
        """
        创建部门
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37142

        :param parentid: 父部门id。根部门id为1
        :param parent_balance_first: 是否优先使用父部门的预算
        :param share_balance: 是否共享预算
        :param outer_permit_users: 本部门的员工仅可见员工自己为true时，可以配置额外可见人员，值为userid组成的的字符串，使用|符号进行分割。总数不能超过200。
        :param outer_permit_depts: 本部门的员工仅可见员工自己为true时，可以配置额外可见部门，值为部门id组成的的字符串，使用|符号进行分割。总数不能超过200。
        :param outer_dept: 是否本部门的员工仅可见员工自己, 为true时，本部门员工默认只能看到员工自己
        :param user_perimits: 已废弃
        :param dept_perimits: 已废弃
        :param dept_hiding: 是否隐藏部门, true表示隐藏, false表示显示
        :param create_dept_group: 是否创建一个关联此部门的企业群，默认为false
        :param order: 在父部门中的次序值。order值小的排序靠前
        :param name: 部门名称。长度限制为1~64个字符。不允许包含字符‘-’‘，’以及‘,’
        :param source_identifier: 部门标识字段，开发者可用该字段来唯一标识一个部门，并与钉钉外部通讯录里的部门做映射
        :param dept_permits: 可以查看指定隐藏部门的其他部门列表，如果部门隐藏，则此值生效，取值为其他的部门id组成的的字符串，使用 | 符号进行分割。总数不能超过200。
        :param user_permits: 可以查看指定隐藏部门的其他人员列表，如果部门隐藏，则此值生效，取值为其他的人员userid组成的的字符串，使用| 符号进行分割。总数不能超过200。
        :param outer_dept_only_self: 是否只能看到所在部门及下级部门通讯录
        """
        return self._top_request(
            "dingtalk.oapi.department.create",
            {
                "parentid": parentid,
                "parentBalanceFirst": parent_balance_first,
                "shareBalance": share_balance,
                "outerPermitUsers": outer_permit_users,
                "outerPermitDepts": outer_permit_depts,
                "outerDept": outer_dept,
                "userPerimits": user_perimits,
                "deptPerimits": dept_perimits,
                "deptHiding": dept_hiding,
                "createDeptGroup": create_dept_group,
                "order": order,
                "name": name,
                "sourceIdentifier": source_identifier,
                "deptPermits": dept_permits,
                "userPermits": user_permits,
                "outerDeptOnlySelf": outer_dept_only_self
            }
        )

    def dingtalk_oapi_gettoken(
            self,
            corpid='',
            corpsecret='',
            appkey='',
            appsecret=''
    ):
        """
        获取企业token
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37143

        :param corpid: corpid
        :param corpsecret: corpsecret
        :param appkey: appkey
        :param appsecret: appsecret
        """
        return self._top_request(
            "dingtalk.oapi.gettoken",
            {
                "corpid": corpid,
                "corpsecret": corpsecret,
                "appkey": appkey,
                "appsecret": appsecret
            }
        )

    def dingtalk_oapi_call_back_delete_call_back(
            self
    ):
        """
        删除事件回调接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37145

        """
        return self._top_request(
            "dingtalk.oapi.call_back.delete_call_back"
        )

    def dingtalk_oapi_call_back_update_call_back(
            self,
            call_back_tag='',
            aes_key='',
            token='',
            url=''
    ):
        """
        更新事件回调接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37146

        :param call_back_tag: 需要监听的事件类型，有20种，“user_add_org”, “user_modify_org”, “user_leave_org”,“org_admin_add”, “org_admin_remove”, “org_dept_create”, “org_dept_modify”, “org_dept_remove”, “org_remove”, “chat_add_member”, “chat_remove_member”, “chat_quit”, “chat_update_owner”, “chat_update_title”, “chat_disband”, “chat_disband_microapp”, “check_in”,“bpms_task_change”,“bpms_instance_change”,,“label_user_change”,“label_conf_add”, “label_conf_modify”,“label_conf_del”,
        :param aes_key: 数据加密密钥。用于回调数据的加密，长度固定为43个字符，从a-z, A-Z, 0-9共62个字符中选取,您可以随机生成，ISV(服务提供商)推荐使用注册套件时填写的EncodingAESKey
        :param token: 加解密需要用到的token，ISV(服务提供商)推荐使用注册套件时填写的token，普通企业可以随机填写
        :param url: 更新事件回调接口
        """
        return self._top_request(
            "dingtalk.oapi.call_back.update_call_back",
            {
                "call_back_tag": call_back_tag,
                "aes_key": aes_key,
                "token": token,
                "url": url
            }
        )

    def dingtalk_oapi_call_back_get_call_back(
            self
    ):
        """
        查询事件回调接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37147

        """
        return self._top_request(
            "dingtalk.oapi.call_back.get_call_back"
        )

    def dingtalk_oapi_sns_get_sns_token(
            self,
            persistent_code='',
            openid=''
    ):
        """
        获取用户授权的SNS_TOKEN
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37148

        :param persistent_code: 用户授权给钉钉开放应用的持久授权码
        :param openid: 用户的openid
        """
        return self._top_request(
            "dingtalk.oapi.sns.get_sns_token",
            {
                "persistent_code": persistent_code,
                "openid": openid
            }
        )

    def dingtalk_oapi_service_get_unactive_corp(
            self,
            app_id=''
    ):
        """
        获取应用未激活的企业列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37149

        :param app_id: 套件下的微应用ID
        """
        return self._top_request(
            "dingtalk.oapi.service.get_unactive_corp",
            {
                "app_id": app_id
            }
        )

    def dingtalk_oapi_service_get_agent(
            self,
            agentid='',
            permanent_code='',
            auth_corpid='',
            suite_key=''
    ):
        """
        获取企业的应用信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37150

        :param agentid: agentid
        :param permanent_code: permanent_code
        :param auth_corpid: auth_corpid
        :param suite_key: suite_key
        """
        return self._top_request(
            "dingtalk.oapi.service.get_agent",
            {
                "agentid": agentid,
                "permanent_code": permanent_code,
                "auth_corpid": auth_corpid,
                "suite_key": suite_key
            }
        )

    def dingtalk_oapi_service_get_auth_info(
            self,
            suite_key='',
            auth_corpid=''
    ):
        """
        获取企业基本信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37151

        :param suite_key: 套件key
        :param auth_corpid: 授权方corpid
        """
        return self._top_request(
            "dingtalk.oapi.service.get_auth_info",
            {
                "suite_key": suite_key,
                "auth_corpid": auth_corpid
            }
        )

    def dingtalk_oapi_service_get_corp_token(
            self,
            auth_corpid='',
            permanent_code=''
    ):
        """
        获取企业授权的凭证
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37152

        :param auth_corpid: 授权方corpid
        :param permanent_code: 永久授权码，通过get_permanent_code获取
        """
        return self._top_request(
            "dingtalk.oapi.service.get_corp_token",
            {
                "auth_corpid": auth_corpid,
                "permanent_code": permanent_code
            }
        )

    def dingtalk_oapi_service_activate_suite(
            self,
            suite_key='',
            auth_corpid='',
            permanent_code=''
    ):
        """
        激活套件
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37153

        :param suite_key: 套件key
        :param auth_corpid: 授权方corpid
        :param permanent_code: 永久授权码，从get_permanent_code接口中获取
        """
        return self._top_request(
            "dingtalk.oapi.service.activate_suite",
            {
                "suite_key": suite_key,
                "auth_corpid": auth_corpid,
                "permanent_code": permanent_code
            }
        )

    def dingtalk_oapi_service_get_permanent_code(
            self,
            tmp_auth_code=''
    ):
        """
        获取企业授权的永久授权码
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37154

        :param tmp_auth_code: 回调接口（tmp_auth_code）获取的临时授权码
        """
        return self._top_request(
            "dingtalk.oapi.service.get_permanent_code",
            {
                "tmp_auth_code": tmp_auth_code
            }
        )

    def dingtalk_oapi_service_get_suite_token(
            self,
            suite_key='',
            suite_secret='',
            suite_ticket=''
    ):
        """
        获取应用套件令牌Token
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37155

        :param suite_key: 套件key，开发者后台创建套件后获取
        :param suite_secret: 套件secret，开发者后台创建套件后获取
        :param suite_ticket: 钉钉推送的ticket
        """
        return self._top_request(
            "dingtalk.oapi.service.get_suite_token",
            {
                "suite_key": suite_key,
                "suite_secret": suite_secret,
                "suite_ticket": suite_ticket
            }
        )

    def dingtalk_oapi_call_back_get_call_back_failed_result(
            self
    ):
        """
        获取回调失败的结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37156

        """
        return self._top_request(
            "dingtalk.oapi.call_back.get_call_back_failed_result"
        )

    def dingtalk_oapi_call_back_register_call_back(
            self,
            call_back_tag='',
            token='',
            aes_key='',
            url=''
    ):
        """
        注册事件回调接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37157

        :param call_back_tag: 需要监听的事件类型，有20种，“user_add_org”, “user_modify_org”, “user_leave_org”,“org_admin_add”, “org_admin_remove”, “org_dept_create”, “org_dept_modify”, “org_dept_remove”, “org_remove”,“label_user_change”, “label_conf_add”, “label_conf_modify”,“label_conf_del”,“org_change”, “chat_add_member”, “chat_remove_member”, “chat_quit”, “chat_update_owner”, “chat_update_title”, “chat_disband”, “chat_disband_microapp”,“check_in”,“bpms_task_change”,“bpms_instance_change”
        :param token: 加解密需要用到的token，ISV(服务提供商)推荐使用注册套件时填写的token，普通企业可以随机填写
        :param aes_key: 数据加密密钥。用于回调数据的加密，长度固定为43个字符，从a-z, A-Z, 0-9共62个字符中选取,您可以随机生成，ISV(服务提供商)推荐使用注册套件时填写的EncodingAESKey
        :param url: 接收事件回调的url
        """
        return self._top_request(
            "dingtalk.oapi.call_back.register_call_back",
            {
                "call_back_tag": call_back_tag,
                "token": token,
                "aes_key": aes_key,
                "url": url
            }
        )

    def dingtalk_oapi_user_get_org_user_count(
            self,
            only_active=''
    ):
        """
        获取企业员工人数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37158

        :param only_active: 0：包含未激活钉钉的人员数量 1：不包含未激活钉钉的人员数量
        """
        return self._top_request(
            "dingtalk.oapi.user.get_org_user_count",
            {
                "onlyActive": only_active
            }
        )

    def dingtalk_oapi_message_send_to_conversation(
            self,
            sender='',
            markdown=None,
            oa=None,
            voice=None,
            file=None,
            link=None,
            image=None,
            text=None,
            msgtype='',
            cid='',
            action_card=None,
            msg=None
    ):
        """
        发送普通消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37159

        :param sender: 消息发送者员工ID
        :param markdown: markdown消息
        :param oa: OA消息
        :param voice: voice消息
        :param file: file消息
        :param link: link消息
        :param image: image消息
        :param text: text消息
        :param msgtype: text
        :param cid: 群消息或者个人聊天会话Id，(通过JSAPI之pickConversation接口唤起联系人界面选择之后即可拿到会话ID，之后您可以使用获取到的cid调用此接口）
        :param action_card: ActionCard消息
        :param msg: 消息内容
        """
        return self._top_request(
            "dingtalk.oapi.message.send_to_conversation",
            {
                "sender": sender,
                "markdown": markdown,
                "oa": oa,
                "voice": voice,
                "file": file,
                "link": link,
                "image": image,
                "text": text,
                "msgtype": msgtype,
                "cid": cid,
                "action_card": action_card,
                "msg": msg
            }
        )

    def dingtalk_oapi_sns_getuserinfo(
            self,
            sns_token=''
    ):
        """
        获取用户授权的个人信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37160

        :param sns_token: 用户授权给开放应用的token
        """
        return self._top_request(
            "dingtalk.oapi.sns.getuserinfo",
            {
                "sns_token": sns_token
            }
        )

    def dingtalk_oapi_sns_get_persistent_code(
            self,
            tmp_auth_code=''
    ):
        """
        获取用户授权的持久授权码
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37161

        :param tmp_auth_code: 用户授权给钉钉开放应用的临时授权码
        """
        return self._top_request(
            "dingtalk.oapi.sns.get_persistent_code",
            {
                "tmp_auth_code": tmp_auth_code
            }
        )

    def dingtalk_oapi_microapp_create(
            self,
            app_icon='',
            app_desc='',
            homepage_url='',
            pc_homepage_url='',
            omp_link='',
            app_name=''
    ):
        """
        创建微应用
        企业开发者通过接口创建微应用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37162

        :param app_icon: 微应用的图标。需要调用上传接口将图标上传到钉钉服务器后获取到的mediaId
        :param app_desc: 钉钉测试微应用
        :param homepage_url: 微应用的移动端主页，必须以http开头或https开头
        :param pc_homepage_url: 微应用的PC端主页，必须以http开头或https开头，如果不为空则必须与homepageUrl的域名一致
        :param omp_link: 微应用的OA后台管理主页，必须以http开头或https开头。微应用后台管理员免登 开发
        :param app_name: 微应用的名称。长度限制为1~10个字符
        """
        return self._top_request(
            "dingtalk.oapi.microapp.create",
            {
                "appIcon": app_icon,
                "appDesc": app_desc,
                "homepageUrl": homepage_url,
                "pcHomepageUrl": pc_homepage_url,
                "ompLink": omp_link,
                "appName": app_name
            }
        )

    def dingtalk_oapi_microapp_update(
            self,
            app_name='',
            app_icon='',
            app_desc='',
            homepage_url='',
            pc_homepage_url='',
            omp_link='',
            agent_id=''
    ):
        """
        更新微应用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37163

        :param app_name: 微应用的名称。长度限制为1~10个字符
        :param app_icon: 微应用的图标。需要调用上传接口将图标上传到钉钉服务器后获取到的mediaId
        :param app_desc: 微应用的描述。长度限制为1~20个字符
        :param homepage_url: 微应用的移动端主页，必须以http开头或https开头
        :param pc_homepage_url: 微应用的PC端主页，必须以http开头或https开头，如果不为空则必须与homepageUrl的域名一致
        :param omp_link: 微应用的OA后台管理主页，必须以http开头或https开头
        :param agent_id: 微应用实例化id
        """
        return self._top_request(
            "dingtalk.oapi.microapp.update",
            {
                "appName": app_name,
                "appIcon": app_icon,
                "appDesc": app_desc,
                "homepageUrl": homepage_url,
                "pcHomepageUrl": pc_homepage_url,
                "ompLink": omp_link,
                "agentId": agent_id
            }
        )

    def dingtalk_oapi_microapp_list(
            self
    ):
        """
        列出微应用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37164

        """
        return self._top_request(
            "dingtalk.oapi.microapp.list"
        )

    def dingtalk_oapi_checkin_record(
            self,
            department_id='',
            end_time='',
            start_time='',
            offset='',
            size='',
            order=''
    ):
        """
        获得签到数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37165

        :param department_id: 部门id（1 表示根部门）
        :param end_time: 开始时间，精确到毫秒，注意字段的位数 例：1520956800000
        :param start_time: 结束时间，精确到毫秒，注意字段的位数 例：1520956800000（默认为当前时间）
        :param offset: 支持分页查询，与size 参数同时设置时才生效，此参数代表偏移量，从0 开始
        :param size: 支持分页查询，与offset 参数同时设置时才生效，此参数代表分页大小，最大100
        :param order: 排序，asc 为正序，desc 为倒序
        """
        return self._top_request(
            "dingtalk.oapi.checkin.record",
            {
                "department_id": department_id,
                "end_time": end_time,
                "start_time": start_time,
                "offset": offset,
                "size": size,
                "order": order
            }
        )

    def dingtalk_oapi_microapp_delete(
            self,
            agent_id=''
    ):
        """
        删除微应用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37166

        :param agent_id: 微应用实例化id，企业只能删除自建微应用
        """
        return self._top_request(
            "dingtalk.oapi.microapp.delete",
            {
                "agentId": agent_id
            }
        )

    def dingtalk_oapi_microapp_list_by_userid(
            self,
            userid=''
    ):
        """
        列出员工可见的微应用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37167

        :param userid: 员工userid
        """
        return self._top_request(
            "dingtalk.oapi.microapp.list_by_userid",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_microapp_visible_scopes(
            self,
            agent_id=''
    ):
        """
        获取企业设置的微应用可见范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37168

        :param agent_id: 微应用实例化id
        """
        return self._top_request(
            "dingtalk.oapi.microapp.visible_scopes",
            {
                "agentId": agent_id
            }
        )

    def dingtalk_oapi_microapp_set_visible_scopes(
            self,
            user_visible_scopes='',
            dept_visible_scopes='',
            is_hidden='',
            agent_id=''
    ):
        """
        设置微应用的可见范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37169

        :param user_visible_scopes: 设置可见的员工id列表，格式为JSON数组
        :param dept_visible_scopes: 设置可见的部门id列表，格式为JSON数组
        :param is_hidden: 是否仅限管理员可见，true代表仅限管理员可见
        :param agent_id: 微应用实例化id
        """
        return self._top_request(
            "dingtalk.oapi.microapp.set_visible_scopes",
            {
                "userVisibleScopes": user_visible_scopes,
                "deptVisibleScopes": dept_visible_scopes,
                "isHidden": is_hidden,
                "agentId": agent_id
            }
        )

    def dingtalk_oapi_microapp_rule_get_rule_list(
            self,
            userid='',
            agent_id=''
    ):
        """
        微应用规则绑定服务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37170

        :param userid: 用户在企业内的唯一标识
        :param agent_id: 微应用在企业内的id
        """
        return self._top_request(
            "dingtalk.oapi.microapp.rule.get_rule_list",
            {
                "userid": userid,
                "agentId": agent_id
            }
        )

    def dingtalk_oapi_microapp_rule_get_user_total(
            self,
            agent_id='',
            rule_id_list=''
    ):
        """
        获取规则绑定的用户数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37171

        :param agent_id: 规则所属的微应用agentId
        :param rule_id_list: 需要查询的规则id集合
        """
        return self._top_request(
            "dingtalk.oapi.microapp.rule.get_user_total",
            {
                "agentId": agent_id,
                "ruleIdList": rule_id_list
            }
        )

    def dingtalk_oapi_sso_gettoken(
            self,
            corpid='',
            corpsecret=''
    ):
        """
        获取微应用后台免登的SsoToken
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37172

        :param corpid: 企业Id
        :param corpsecret: 这里必须填写专属的SSOSecret
        """
        return self._top_request(
            "dingtalk.oapi.sso.gettoken",
            {
                "corpid": corpid,
                "corpsecret": corpsecret
            }
        )

    def dingtalk_oapi_get_jsapi_ticket(
            self
    ):
        """
        获取jsapi_ticket
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37173

        """
        return self._top_request(
            "dingtalk.oapi.get_jsapi_ticket"
        )

    def dingtalk_oapi_microapp_rule_delete(
            self,
            agent_id='',
            rule_id=''
    ):
        """
        删除规则
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37174

        :param agent_id: 规则所属的微应用agentId
        :param rule_id: 被删除的规则id
        """
        return self._top_request(
            "dingtalk.oapi.microapp.rule.delete",
            {
                "agentId": agent_id,
                "ruleId": rule_id
            }
        )

    def dingtalk_oapi_sns_gettoken(
            self,
            appid='',
            appsecret=''
    ):
        """
        获取钉钉开放应用的ACCESS_TOKEN
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37175

        :param appid: 由钉钉开放平台提供给开放应用的唯一标识
        :param appsecret: 由钉钉开放平台提供的密钥
        """
        return self._top_request(
            "dingtalk.oapi.sns.gettoken",
            {
                "appid": appid,
                "appsecret": appsecret
            }
        )

    def dingtalk_oapi_auth_scopes(
            self
    ):
        """
        获取CorpSecret授权范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37176

        """
        return self._top_request(
            "dingtalk.oapi.auth.scopes"
        )

    def dingtalk_oapi_process_gettodonum(
            self,
            userid
    ):
        """
        获取待我审批数量
        获取用户待审批数量
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37188

        :param userid: 用户id
        """
        return self._top_request(
            "dingtalk.oapi.process.gettodonum",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_report_getunreadcount(
            self,
            userid=''
    ):
        """
        查询企业员工的日志未读数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37189

        :param userid: 员工id
        """
        return self._top_request(
            "dingtalk.oapi.report.getunreadcount",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_smartdevice_visitor_addvisitor(
            self,
            visitor_vo
    ):
        """
        添加企业访客邀约记录信息
        ISV向钉钉智能硬件添加授权企业的访客邀约记录，以实现与智能识别设备的打通流程
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37221

        :param visitor_vo: 访客预约模型
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.visitor.addvisitor",
            {
                "visitor_vo": visitor_vo
            }
        )

    def dingtalk_oapi_smartdevice_visitor_editvisitor(
            self,
            reservation_id,
            visitor_vo
    ):
        """
        编辑企业访客预约数据
        ISV向钉钉智能硬件同步授权企业的访问预约记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37222

        :param reservation_id: 预约编号
        :param visitor_vo: 预约数据
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.visitor.editvisitor",
            {
                "reservation_id": reservation_id,
                "visitor_vo": visitor_vo
            }
        )

    def dingtalk_oapi_robot_send(
            self,
            msgtype,
            text=None,
            at=None,
            link=None,
            markdown=None,
            action_card=None,
            feed_card=None
    ):
        """
        自定义机器人发送消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37246

        :param msgtype: 消息类型
        :param text: text类型
        :param at: 被@人的手机号
        :param link: 消息类型，此时固定为:link
        :param markdown: 此消息类型为固定markdown
        :param action_card: 此消息类型为固定actionCard
        :param feed_card: 此消息类型为固定feedCard
        """
        return self._top_request(
            "dingtalk.oapi.robot.send",
            {
                "msgtype": msgtype,
                "text": text,
                "at": at,
                "link": link,
                "markdown": markdown,
                "actionCard": action_card,
                "feedCard": feed_card
            }
        )

    def dingtalk_oapi_calendar_create(
            self,
            create_vo=None
    ):
        """
        创建日程
        通过此接口可以把企业员工的待办事项导入到钉钉日历并在钉钉日历中展示, 支持任务, 会议,审批,普通日程等.
        该接口处于内部灰度阶段, 申请使用请邮件联系:wusuan.yby@alibaba-inc.com
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37264

        :param create_vo: 创建日程实体
        """
        return self._top_request(
            "dingtalk.oapi.calendar.create",
            {
                "create_vo": create_vo
            }
        )

    def dingtalk_oapi_ding_create(
            self,
            creator_userid,
            receiver_userids,
            remind_type,
            remind_time,
            text_content,
            attachment=None
    ):
        """
        发DING通知
        通过此接口发DING通知给企业内部员工, 支持短信DING和应用内DING.
        该接口正在灰度内测中, 暂不对外开放
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37267

        :param creator_userid: 发送者工号
        :param receiver_userids: 接收者工号列表
        :param remind_type: 提醒类型:1-应用内;2-短信
        :param remind_time: 发送时间(单位:毫秒)
        :param text_content: 通知内容
        :param attachment: 附件内容
        """
        return self._top_request(
            "dingtalk.oapi.ding.create",
            {
                "creator_userid": creator_userid,
                "receiver_userids": receiver_userids,
                "remind_type": remind_type,
                "remind_time": remind_time,
                "text_content": text_content,
                "attachment": attachment
            }
        )

    def dingtalk_oapi_chat_updategroupnick(
            self,
            userid,
            chatid,
            group_nick
    ):
        """
        设置群昵称
        设置群成员的群昵称
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37291

        :param userid: 用户userid
        :param chatid: chatid
        :param group_nick: 群昵称
        """
        return self._top_request(
            "dingtalk.oapi.chat.updategroupnick",
            {
                "userid": userid,
                "chatid": chatid,
                "group_nick": group_nick
            }
        )

    def dingtalk_oapi_attendance_getupdatedata(
            self,
            userid,
            work_date
    ):
        """
        获取考勤更新数据
        增加这个api的目的是为了给外部企业提供考勤数据变更，提升外部企业的考勤统计数据的准确度。外部企业只能通过入参获取数据，属于只读接口。
        接口的入参是 查询用户corpId，userid以及查询日期，这些参数是我们开放平台推送给用户的变更。
        返回的结果是三个list结果：分别是打卡流水记录 打卡结果 以及 审批列表 都是被查询用户在查询日期内的数据。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37307

        :param userid: 用户id
        :param work_date: 工作日
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getupdatedata",
            {
                "userid": userid,
                "work_date": work_date
            }
        )

    def dingtalk_oapi_cspace_add_to_single_chat(
            self,
            file_name='',
            media_id='',
            userid='',
            agent_id=''
    ):
        """
        发送文件给指定用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37310

        :param file_name: 文件名(需包含含扩展名),需要utf-8 urlEncode
        :param media_id: 调用钉盘上传文件接口得到的mediaid,需要utf-8 urlEncode
        :param userid: 文件发送者微应用的agentId
        :param agent_id: 文件接收人的userid
        """
        return self._top_request(
            "dingtalk.oapi.cspace.add_to_single_chat",
            {
                "file_name": file_name,
                "media_id": media_id,
                "userid": userid,
                "agent_id": agent_id
            }
        )

    def dingtalk_oapi_cspace_add(
            self,
            agent_id='',
            code='',
            media_id='',
            folder_id='',
            space_id='',
            name='',
            overwrite=''
    ):
        """
        新增文件到用户钉盘
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37324

        :param agent_id: 微应用的agentId
        :param code: 如果是微应用，code值为微应用免登授权码,如果是服务窗应用，code值为服务窗免登授权码。code为临时授权码，只能消费一次，下次请求需要重新获取新的code。
        :param media_id: 调用钉盘上传文件接口得到的mediaid, 需要utf-8 urlEncode
        :param folder_id: 调用云盘选择控件后获取的用户钉盘空间ID
        :param space_id: 调用云盘选择控件后获取的用户钉盘文件夹ID
        :param name: 上传文件的名称，不能包含非法字符，需要utf-8 urlEncode
        :param overwrite: 遇到同名文件是否覆盖，若不覆盖，则会自动重命名本次新增的文件，默认为false
        """
        return self._top_request(
            "dingtalk.oapi.cspace.add",
            {
                "agent_id": agent_id,
                "code": code,
                "media_id": media_id,
                "folder_id": folder_id,
                "space_id": space_id,
                "name": name,
                "overwrite": overwrite
            }
        )

    def dingtalk_oapi_cspace_get_custom_space(
            self,
            domain='',
            agent_id=''
    ):
        """
        获取企业下的自定义空间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37325

        :param domain: 企业调用时传入，需要为10个字节以内的字符串，仅可包含字母和数字，大小写不敏感
        :param agent_id: ISV调用时传入，微应用agentId
        """
        return self._top_request(
            "dingtalk.oapi.cspace.get_custom_space",
            {
                "domain": domain,
                "agent_id": agent_id
            }
        )

    def dingtalk_oapi_cspace_grant_custom_space(
            self,
            agent_id='',
            domain='',
            type='',
            userid='',
            path='',
            fileids='',
            duration=''
    ):
        """
        授权用户访问企业下的自定义空间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37326

        :param agent_id: ISV调用时传入，授权访问指定微应用的自定义空间
        :param domain: 权限类型，目前支持上传和下载，上传请传add，下载请传download
        :param type: 企业调用时传入，授权访问该domain的自定义空间
        :param userid: 企业用户userid
        :param path: 授权访问的路径，如授权访问所有文件传“/”，授权访问/doc文件夹传“/doc/” 需要utf-8 urlEncode
        :param fileids: 授权访问的文件id列表，id之间用英文逗号隔开，如“fileId1,fileId2”
        :param duration: 权限有效时间，有效范围为0~3600秒，超出此范围或不传默认为30秒
        """
        return self._top_request(
            "dingtalk.oapi.cspace.grant_custom_space",
            {
                "agent_id": agent_id,
                "domain": domain,
                "type": type,
                "userid": userid,
                "path": path,
                "fileids": fileids,
                "duration": duration
            }
        )

    def dingtalk_oapi_file_upload_transaction(
            self,
            agent_id='',
            file_size='',
            chunk_numbers='',
            upload_id=''
    ):
        """
        开启文件上传事务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37327

        :param agent_id: 微应用的agentId
        :param file_size: 文件大小
        :param chunk_numbers: 文件总块数
        :param upload_id: 上传事务id 需要utf-8 urlEncode
        """
        return self._top_request(
            "dingtalk.oapi.file.upload.transaction",
            {
                "agent_id": agent_id,
                "file_size": file_size,
                "chunk_numbers": chunk_numbers,
                "upload_id": upload_id
            }
        )

    def dingtalk_oapi_file_upload_chunk(
            self,
            agent_id='',
            upload_id='',
            chunk_sequence='',
            file=''
    ):
        """
        上传文件块
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37328

        :param agent_id: 微应用的agentId
        :param upload_id: 上传事务id 需要utf-8 urlEncode
        :param chunk_sequence: 文件块号，从1开始计数
        :param file: 文件内容
        """
        return self._top_request(
            "dingtalk.oapi.file.upload.chunk",
            {
                "agent_id": agent_id,
                "upload_id": upload_id,
                "chunk_sequence": chunk_sequence,
                "file": file
            }
        )

    def dingtalk_oapi_file_upload_single(
            self,
            agent_id='',
            file_size='',
            file=''
    ):
        """
        单步文件上传
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37329

        :param agent_id: 微应用的agentId
        :param file_size: 文件大小
        :param file: 文件内容
        """
        return self._top_request(
            "dingtalk.oapi.file.upload.single",
            {
                "agent_id": agent_id,
                "file_size": file_size,
                "file": file
            }
        )

    def dingtalk_oapi_org_setoaurl(
            self,
            oa_url,
            oa_title=''
    ):
        """
        设置企业工作台首页
        企业可使用该接口来设置工作台首页地址，效果类似于在oa管理后台手动设置工作台地址
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37360

        :param oa_url: 工作台首页地址，必须是https开头
        :param oa_title: 工作台名称，认证企业才能设置
        """
        return self._top_request(
            "dingtalk.oapi.org.setoaurl",
            {
                "oa_url": oa_url,
                "oa_title": oa_title
            }
        )

    def dingtalk_oapi_call_removeuserlist(
            self,
            staff_id_list
    ):
        """
        删除isv免费电话员工名单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37370

        :param staff_id_list: 要删除的员工userid列表
        """
        return self._top_request(
            "dingtalk.oapi.call.removeuserlist",
            {
                "staff_id_list": staff_id_list
            }
        )

    def dingtalk_oapi_call_setuserlist(
            self,
            staff_id_list
    ):
        """
        设置isv发起免费电话的主叫白名单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37371

        :param staff_id_list: 套件所所属企业免费电话主叫人员工号列表
        """
        return self._top_request(
            "dingtalk.oapi.call.setuserlist",
            {
                "staff_id_list": staff_id_list
            }
        )

    def dingtalk_oapi_call_calluser(
            self,
            staff_id,
            authed_corp_id,
            authed_staff_id
    ):
        """
        主叫方发起免费电话给授权企业下的授权范围内的人员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37372

        :param staff_id: isv套件所属企业下的员工userid
        :param authed_corp_id: 授权isv套件企业的corpid
        :param authed_staff_id: 授权isv套件企业的员工userid
        """
        return self._top_request(
            "dingtalk.oapi.call.calluser",
            {
                "staff_id": staff_id,
                "authed_corp_id": authed_corp_id,
                "authed_staff_id": authed_staff_id
            }
        )

    def dingtalk_oapi_call_getuserlist(
            self,
            offset='0',
            size=''
    ):
        """
        查询可以主动调用接口发起免费电话的员工
        删除ISV套件对应的企业下的可以主动调用接口发起免费电话的员工
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37373

        :param offset: 偏移量
        :param size: size
        """
        return self._top_request(
            "dingtalk.oapi.call.getuserlist",
            {
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_org_setscreen(
            self,
            media_id,
            end_time,
            jump_url=''
    ):
        """
        企业设置开机闪屏动画信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37412

        :param media_id: 开机图片资源id，可以通过/media/upload接口上传图片获取
        :param end_time: yyyy-MM-dd显示截止时间
        :param jump_url: 点击开机图片跳转地址
        """
        return self._top_request(
            "dingtalk.oapi.org.setscreen",
            {
                "mediaId": media_id,
                "endTime": end_time,
                "jumpUrl": jump_url
            }
        )

    def dingtalk_oapi_smartdevice_removeface(
            self,
            userid
    ):
        """
        从企业删除员工人脸识别照片
        删除企业中员工的人脸识别照片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37428

        :param userid: 员工ID
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.removeface",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_sns_getuserinfo_bycode(
            self,
            tmp_auth_code=''
    ):
        """
        根据sns临时授权码获取用户信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37515

        :param tmp_auth_code: 登录的临时授权码
        """
        return self._top_request(
            "dingtalk.oapi.sns.getuserinfo_bycode",
            {
                "tmp_auth_code": tmp_auth_code
            }
        )

    def dingtalk_oapi_hrm_employee_getdismissionlist(
            self,
            current,
            page_size,
            op_userid
    ):
        """
        智能人事开放接口-查询离职人员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37535

        :param current: 第几页，从1开始
        :param page_size: 一页多少数据，在1-100之间
        :param op_userid: 操作人userid
        """
        return self._top_request(
            "dingtalk.oapi.hrm.employee.getdismissionlist",
            {
                "current": current,
                "page_size": page_size,
                "op_userid": op_userid
            }
        )

    def dingtalk_oapi_hrm_employee_get(
            self,
            userid
    ):
        """
        获取智能人事指定员工花名册详细数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37536

        :param userid: 查询用户userid
        """
        return self._top_request(
            "dingtalk.oapi.hrm.employee.get",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_hrm_employee_addresumerecord(
            self,
            userid='',
            title='',
            content='',
            k_v_content='',
            phone_url='',
            pc_url='',
            web_url='',
            record_timestamp=''
    ):
        """
        添加企业员工的成长记录
        企业员工的成长记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37577

        :param userid: 被操作人userid
        :param title: 成长记录title
        :param content: 成长记录第一条内容
        :param k_v_content: 厂长记录kv展示内容：json格式，顺序展示
        :param phone_url: 手机端url
        :param pc_url: pc端url
        :param web_url: webOA后台url
        :param record_timestamp: 20180428 零点零分零秒
        """
        return self._top_request(
            "dingtalk.oapi.hrm.employee.addresumerecord",
            {
                "userid": userid,
                "title": title,
                "content": content,
                "k_v_content": k_v_content,
                "phone_url": phone_url,
                "pc_url": pc_url,
                "web_url": web_url,
                "record_timestamp": record_timestamp
            }
        )

    def dingtalk_oapi_impaas_conversation_sendmessage(
            self,
            chatid,
            type,
            content
    ):
        """
        impaas发送群消息接口
        发送群消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37583

        :param chatid: 群id
        :param type: 1. 优惠券 2 系统消息
        :param content: 消息内容
        """
        return self._top_request(
            "dingtalk.oapi.impaas.conversation.sendmessage",
            {
                "chatid": chatid,
                "type": type,
                "content": content
            }
        )

    def dingtalk_oapi_impaas_conversation_create(
            self,
            channel,
            owner_userid,
            userid_list,
            name
    ):
        """
        impaas创建群接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37584

        :param channel: 渠道
        :param owner_userid: 群主员工id
        :param userid_list: 成员员工id列表
        :param name: 群名称
        """
        return self._top_request(
            "dingtalk.oapi.impaas.conversation.create",
            {
                "channel": channel,
                "owner_userid": owner_userid,
                "userid_list": userid_list,
                "name": name
            }
        )

    def dingtalk_oapi_impaas_conversation_modifymember(
            self,
            chatid,
            memberid_list,
            type,
            channel
    ):
        """
        修改群成员接口
        IMPAAS修改群成员接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37585

        :param chatid: 群ID
        :param memberid_list: 会员ID列表
        :param type: 1 添加 2 删除
        :param channel: 渠道
        """
        return self._top_request(
            "dingtalk.oapi.impaas.conversation.modifymember",
            {
                "chatid": chatid,
                "memberid_list": memberid_list,
                "type": type,
                "channel": channel
            }
        )

    def dingtalk_oapi_hrm_employee_updateresumerecord(
            self,
            userid,
            resume_id,
            title='',
            content='',
            k_v_content='',
            phone_url='',
            pc_url='',
            web_url='',
            record_timestamp=''
    ):
        """
        智能人事-更新成长记录
        更新成长记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37587

        :param userid: 被操作人员工userid
        :param resume_id: 成长记录唯一标识
        :param title: 成长记录title
        :param content: 成长记录第一条内容
        :param k_v_content: 成长记录kv展示内容：json格式，顺序展示
        :param phone_url: 手机端url
        :param pc_url: pc端url
        :param web_url: webOA后台url
        :param record_timestamp: 20180428 零点零分零秒
        """
        return self._top_request(
            "dingtalk.oapi.hrm.employee.updateresumerecord",
            {
                "userid": userid,
                "resume_id": resume_id,
                "title": title,
                "content": content,
                "k_v_content": k_v_content,
                "phone_url": phone_url,
                "pc_url": pc_url,
                "web_url": web_url,
                "record_timestamp": record_timestamp
            }
        )

    def dingtalk_oapi_hrm_employee_delresumerecord(
            self,
            userid,
            resume_id
    ):
        """
        智能人事-删除成长记录
        支持开放删除成长记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37588

        :param userid: 员工userid
        :param resume_id: 成长记录唯一标识
        """
        return self._top_request(
            "dingtalk.oapi.hrm.employee.delresumerecord",
            {
                "userid": userid,
                "resume_id": resume_id
            }
        )

    def dingtalk_oapi_role_getrole(
            self,
            role_id
    ):
        """
        获取角色详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37691

        :param role_id: 角色id
        """
        return self._top_request(
            "dingtalk.oapi.role.getrole",
            {
                "roleId": role_id
            }
        )

    def dingtalk_oapi_message_corpconversation_asyncsend_v2(
            self,
            agent_id,
            msg,
            userid_list='',
            dept_id_list='',
            to_all_user='false'
    ):
        """
        企业会话消息异步发送接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37692

        :param agent_id: 微应用的id
        :param msg: 消息体，具体见文档
        :param userid_list: 接收者的用户userid列表
        :param dept_id_list: 接收者的部门id列表
        :param to_all_user: 是否发送给企业全部用户
        """
        return self._top_request(
            "dingtalk.oapi.message.corpconversation.asyncsend_v2",
            {
                "agent_id": agent_id,
                "msg": msg,
                "userid_list": userid_list,
                "dept_id_list": dept_id_list,
                "to_all_user": to_all_user
            }
        )

    def dingtalk_oapi_certify_queryinfo(
            self,
            userid
    ):
        """
        查询实名认证信息
        提供查询实名认证信息（仅支持2018云栖大会）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38022

        :param userid: 员工userId，也称staffId
        """
        return self._top_request(
            "dingtalk.oapi.certify.queryinfo",
            {
                "userid": userid
            }
        )

    def dingtalk_ccoservice_servicegroup_get(
            self,
            open_group_id
    ):
        """
        查询服务群信息
        企业查询自己企业内服务群的信息，会把服务群的群主信息，dingtalkid，nick，真实姓名和群名称返回。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38055

        :param open_group_id: 服务群id
        """
        return self._top_request(
            "dingtalk.ccoservice.servicegroup.get",
            {
                "open_group_id": open_group_id
            }
        )

    def dingtalk_ccoservice_servicegroup_addmember(
            self,
            userid,
            open_group_id
    ):
        """
        把企业某个员工加入到服务群
        此API主要就是把某个企业的员工加入到此企业的某个服务群内
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38056

        :param userid: 企业员工id
        :param open_group_id: 服务群id
        """
        return self._top_request(
            "dingtalk.ccoservice.servicegroup.addmember",
            {
                "userid": userid,
                "open_group_id": open_group_id
            }
        )

    def dingtalk_oapi_microapp_addwithuserid(
            self,
            agent_id,
            userids
    ):
        """
        添加用户到微应用可见范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38097

        :param agent_id: 微应用实例化id，表示企业和微应用的唯一关系
        :param userids: 用户id列表，最多10个
        """
        return self._top_request(
            "dingtalk.oapi.microapp.addwithuserid",
            {
                "agentId": agent_id,
                "userids": userids
            }
        )

    def dingtalk_oapi_microapp_delwithuserid(
            self,
            agent_id,
            userids
    ):
        """
        去掉用户的微应用可见范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38098

        :param agent_id: 微应用实例化id，表示企业和微应用的唯一关系
        :param userids: 用户id列表，最多10个
        """
        return self._top_request(
            "dingtalk.oapi.microapp.delwithuserid",
            {
                "agentId": agent_id,
                "userids": userids
            }
        )

    def dingtalk_oapi_org_listshortcut(
            self
    ):
        """
        查询企业工作台快捷方式
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38101

        """
        return self._top_request(
            "dingtalk.oapi.org.listshortcut"
        )

    def dingtalk_oapi_org_setshortcut(
            self,
            agent_ids=''
    ):
        """
        设置企业工作台快捷方式列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38102

        :param agent_ids: 微应用实例id列表
        """
        return self._top_request(
            "dingtalk.oapi.org.setshortcut",
            {
                "agentIds": agent_ids
            }
        )

    def dingtalk_oapi_impaas_newretail_sendstaffmessage(
            self,
            msg_type='0',
            sender='',
            userid_list='',
            content=''
    ):
        """
        新零售发送单聊消息给员工
        新零售场景下，通过系统账号或员工账号向员工发送单聊账号
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38164

        :param msg_type: 0系统消息
        :param sender: 系统账号
        :param userid_list: 用账号列表
        :param content: 消息体
        """
        return self._top_request(
            "dingtalk.oapi.impaas.newretail.sendstaffmessage",
            {
                "msg_type": msg_type,
                "sender": sender,
                "userid_list": userid_list,
                "content": content
            }
        )

    def dingtalk_oapi_impaas_newretail_sendstaffgroupmessage(
            self,
            msg_type='0',
            sender='',
            chat_id='',
            content=''
    ):
        """
        新零售的系统账号发送群聊消息
        新零售场景下，先创建某企业的内部群。
        然后用特定的系统账号向群内发送群聊消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38183

        :param msg_type: 消息类型
        :param sender: 系统账号名
        :param chat_id: 加密后的群号
        :param content: 消息内容
        """
        return self._top_request(
            "dingtalk.oapi.impaas.newretail.sendstaffgroupmessage",
            {
                "msg_type": msg_type,
                "sender": sender,
                "chat_id": chat_id,
                "content": content
            }
        )

    def dingtalk_oapi_smartdevice_visitor_removevisitor(
            self,
            reservation_id
    ):
        """
        取消企业访客邀约记录信息
        ISV向钉钉智能硬件取消授权企业的访客邀约记录，以实现与智能识别设备的打通流程
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38184

        :param reservation_id: 预约编号
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.visitor.removevisitor",
            {
                "reservation_id": reservation_id
            }
        )

    def dingtalk_oapi_impaas_group_getbydeptid(
            self,
            dept_id
    ):
        """
        新零售场景下，获取企业全员群id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38514

        :param dept_id: 1企业全员群
        """
        return self._top_request(
            "dingtalk.oapi.impaas.group.getbydeptid",
            {
                "dept_id": dept_id
            }
        )

    def dingtalk_oapi_role_updaterole(
            self,
            role_id,
            role_name
    ):
        """
        更新角色
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38575

        :param role_id: 1
        :param role_name: 1
        """
        return self._top_request(
            "dingtalk.oapi.role.updaterole",
            {
                "roleId": role_id,
                "roleName": role_name
            }
        )

    def dingtalk_oapi_role_addrole(
            self,
            role_name,
            group_id
    ):
        """
        添加角色
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38576

        :param role_name: test
        :param group_id: test
        """
        return self._top_request(
            "dingtalk.oapi.role.addrole",
            {
                "roleName": role_name,
                "groupId": group_id
            }
        )

    def dingtalk_oapi_role_addrolegroup(
            self,
            name
    ):
        """
        创建角色组
        添加角色组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38577

        :param name: 名称
        """
        return self._top_request(
            "dingtalk.oapi.role.addrolegroup",
            {
                "name": name
            }
        )

    def dingtalk_oapi_impaas_relation_add(
            self,
            src_im_openid,
            dst_im_openid,
            is_double_way,
            begin_time,
            end_time
    ):
        """
        impaas关系导入
        二方关系导入钉钉
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38619

        :param src_im_openid: 发送者钉钉的openid
        :param dst_im_openid: 接收者钉钉的openid
        :param is_double_way: 是否双向关系
        :param begin_time: 时间戳精确到毫秒
        :param end_time: 时间戳精确到毫秒
        """
        return self._top_request(
            "dingtalk.oapi.impaas.relation.add",
            {
                "request": {
                    "src_im_openid": src_im_openid,
                    "dst_im_openid": dst_im_openid,
                    "is_double_way": is_double_way,
                    "begin_time": begin_time,
                    "end_time": end_time
                }
            }
        )

    def dingtalk_oapi_chat_transform(
            self,
            open_conversation_id
    ):
        """
        将conversationId转换成chatId
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38620

        :param open_conversation_id: 开放cid
        """
        return self._top_request(
            "dingtalk.oapi.chat.transform",
            {
                "open_conversation_id": open_conversation_id
            }
        )

    def dingtalk_oapi_impaas_user_addprofile(
            self,
            id,
            nick,
            channel,
            extension='',
            avatar_mediaid=''
    ):
        """
        二方账号导入
        二方渠道导入账号
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38621

        :param id: 账号id
        :param nick: 账号nick
        :param channel: 渠道
        :param extension: 附加信息
        :param avatar_mediaid: 头像mediaid
        """
        return self._top_request(
            "dingtalk.oapi.impaas.user.addprofile",
            {
                "request": {
                    "id": id,
                    "nick": nick,
                    "channel": channel,
                    "extension": extension,
                    "avatar_mediaid": avatar_mediaid
                }
            }
        )

    def dingtalk_oapi_impaas_relation_del(
            self,
            src_im_openid,
            dst_im_openid,
            is_double_way
    ):
        """
        二方关系删除
        删除二方关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38622

        :param src_im_openid: 发送者的im的openid
        :param dst_im_openid: 接受者的im的openid
        :param is_double_way: 是否双向关系
        """
        return self._top_request(
            "dingtalk.oapi.impaas.relation.del",
            {
                "request": {
                    "src_im_openid": src_im_openid,
                    "dst_im_openid": dst_im_openid,
                    "is_double_way": is_double_way
                }
            }
        )

    def dingtalk_oapi_impaas_groupmember_modifymemberinfo(
            self,
            chatid,
            member_info
    ):
        """
        IMPAAS修改群成员信息
        修改群成员信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38679

        :param chatid: 群ID
        :param member_info: 要修改的成员信息
        """
        return self._top_request(
            "dingtalk.oapi.impaas.groupmember.modifymemberinfo",
            {
                "request": {
                    "chatid": chatid,
                    "member_info": member_info
                }
            }
        )

    def dingtalk_oapi_impaas_group_dismiss(
            self,
            chatid
    ):
        """
        impaas解散群接口
        IMPAAS解散群接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38680

        :param chatid: 群ID
        """
        return self._top_request(
            "dingtalk.oapi.impaas.group.dismiss",
            {
                "request": {
                    "chatid": chatid
                }
            }
        )

    def dingtalk_oapi_impaas_groupmember_getmemberlist(
            self,
            chatid,
            offset='0',
            length='1000'
    ):
        """
        IMPAAS获取群成员列表
        获取群成员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38681

        :param chatid: 群ID
        :param offset: 群成员列表偏移
        :param length: 本次请求返回的群成员列表数量
        """
        return self._top_request(
            "dingtalk.oapi.impaas.groupmember.getmemberlist",
            {
                "request": {
                    "chatid": chatid,
                    "offset": offset,
                    "length": length
                }
            }
        )

    def dingtalk_oapi_impaas_conversaion_changegroupowner(
            self,
            chatid='',
            userid='',
            channel=''
    ):
        """
        impaas群名称更改
        盒马接入钉钉sdk，群主转交功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38711

        :param chatid: 钉钉会话id
        :param userid: 员工id
        :param channel: 应用channel
        """
        return self._top_request(
            "dingtalk.oapi.impaas.conversaion.changegroupowner",
            {
                "chatid": chatid,
                "userid": userid,
                "channel": channel
            }
        )

    def dingtalk_oapi_impaas_message_asyncsend(
            self,
            msg_content,
            msg_type,
            group_id='',
            msg_extension='',
            receiverid_list='',
            senderid=None,
            msg_feature='',
            xpn_model=None
    ):
        """
        impaas消息异步发送
        消息异步发送，返回taskid，根据taskid获取消息是否成功
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38716

        :param msg_content: 消息内容，根据msgtype不同，解析方式不同
        :param msg_type: 消息类型：text，image，user-defined
        :param group_id: 群ID
        :param msg_extension: 消息的可扩展字段，透传
        :param receiverid_list: 接受者，暂不支持，可不填
        :param senderid: 发送者，暂不支持，可不填
        :param msg_feature: 消息的特性：静默消息，系统消息
        :param xpn_model: 推送信息
        """
        return self._top_request(
            "dingtalk.oapi.impaas.message.asyncsend",
            {
                "request": {
                    "msg_content": msg_content,
                    "msg_type": msg_type,
                    "group_id": group_id,
                    "msg_extension": msg_extension,
                    "receiverid_list": receiverid_list,
                    "senderid": senderid,
                    "msg_feature": msg_feature,
                    "xpn_model": xpn_model
                }
            }
        )

    def dingtalk_oapi_alitrip_btrip_train_city_suggest(
            self,
            rq
    ):
        """
        火车票城市搜索
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38724

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.train.city.suggest",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_impaas_message_getmessagestatus(
            self,
            taskid,
            senderid=None
    ):
        """
        获取消息的发送状态
        获取消息发送的状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38725

        :param taskid: 异步返回的任务ID
        :param senderid: 账号信息
        """
        return self._top_request(
            "dingtalk.oapi.impaas.message.getmessagestatus",
            {
                "request": {
                    "taskid": taskid,
                    "senderid": senderid
                }
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_entity_delete(
            self,
            rq
    ):
        """
        删除成本中心人员信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38726

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.entity.delete",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_approval_new(
            self,
            rq
    ):
        """
        用户新建审批单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38730

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.approval.new",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_category_address_get(
            self,
            rq
    ):
        """
        获取类目预定页跳转地址
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38731

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.category.address.get",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_ccoservice_servicegroup_isignoreproblemcheck(
            self,
            open_conversation_id,
            dingtalk_id
    ):
        """
        是否忽略问题模型判断
        接口描述：是否忽略问题模型判断，返回值：true: 可以忽略; false: 不能忽略,走正常逻辑;
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38761

        :param open_conversation_id: 群id
        :param dingtalk_id: 用户dingtalk加密id
        """
        return self._top_request(
            "dingtalk.oapi.ccoservice.servicegroup.isignoreproblemcheck",
            {
                "open_conversation_id": open_conversation_id,
                "dingtalk_id": dingtalk_id
            }
        )

    def dingtalk_oapi_ccoservice_servicegroup_updateservicetime(
            self,
            open_conversation_id,
            start_time,
            end_time,
            time_type=''
    ):
        """
        修改服务群入群文案时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38776

        :param open_conversation_id: 群加密id
        :param start_time: 服务开始时间
        :param end_time: 服务结束时间
        :param time_type: 日期类型,0-工作日;1-每日;99-端上不显示
        """
        return self._top_request(
            "dingtalk.oapi.ccoservice.servicegroup.updateservicetime",
            {
                "open_conversation_id": open_conversation_id,
                "start_time": start_time,
                "end_time": end_time,
                "time_type": time_type
            }
        )

    def dingtalk_oapi_alitrip_btrip_apply_search(
            self,
            rq
    ):
        """
        搜索企业审批单数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38777

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.apply.search",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_apply_get(
            self,
            rq
    ):
        """
        获取单个申请单的详情数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38778

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.apply.get",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_transfer(
            self,
            rq
    ):
        """
        商旅成本中心转换为外部成本中心
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38803

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.transfer",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_invoice_search(
            self,
            rq
    ):
        """
        差旅申请用户搜索可用发票列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38806

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.invoice.search",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_flight_order_search(
            self,
            rq
    ):
        """
        获取企业机票订单数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38807

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.flight.order.search",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_train_order_search(
            self,
            rq
    ):
        """
        获取企业火车票订单数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38808

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.train.order.search",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_entity_add(
            self,
            rq
    ):
        """
        增加成本中心人员信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38809

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.entity.add",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_hotel_order_search(
            self,
            rq
    ):
        """
        企业获取商旅酒店订单数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38810

        :param rq: rq
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.hotel.order.search",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_entity_set(
            self,
            rq=None
    ):
        """
        设置成本中心人员信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38836

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.entity.set",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_delete(
            self,
            rq
    ):
        """
        删除成本中心
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38837

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.delete",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_modify(
            self,
            rq
    ):
        """
        修改成本中心基本信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38838

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.modify",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_new(
            self,
            rq
    ):
        """
        新建成本中心
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38839

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.new",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_approval_update(
            self,
            rq
    ):
        """
        更新审批单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38840

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.approval.update",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_cost_center_query(
            self,
            rq
    ):
        """
        查询成本中心
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38841

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.cost.center.query",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_impaas_groupmember_modify(
            self,
            modify_type,
            member_list,
            channel,
            chatid
    ):
        """
        修改群成员
        修改群成员列表，支持成员增删。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38860

        :param modify_type: 该参数表示本次请求的操作类型，“1”表示添加成员，“2”表示删除成员。
        :param member_list: 待操作成员列表
        :param channel: 接入方channel信息，该值由接入方接入IMPaaS平台时，向IMPaaS平台申请，例如“hema”“eleme”等。
        :param chatid: 群ID，由创建群接口返回。
        """
        return self._top_request(
            "dingtalk.oapi.impaas.groupmember.modify",
            {
                "request": {
                    "modify_type": modify_type,
                    "member_list": member_list,
                    "channel": channel,
                    "chatid": chatid
                }
            }
        )

    def dingtalk_oapi_impaas_group_modify(
            self,
            group_owner,
            group_name,
            chatid,
            show_history_type=''
    ):
        """
        impaas群信息修改
        修改群信息：群名称、群主等；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38861

        :param group_owner: 修改后的群主，若为空或与当前群主相同，则不会对群主进行变更操作。
        :param group_name: 修改后的群名称
        :param chatid: 群ID，由创建群接口返回
        :param show_history_type: 新人进群是否能查看最近100条记录。0:不能；1:可以查看最近100条记录；不填默认为0
        """
        return self._top_request(
            "dingtalk.oapi.impaas.group.modify",
            {
                "request": {
                    "group_owner": group_owner,
                    "group_name": group_name,
                    "chatid": chatid,
                    "show_history_type": show_history_type
                }
            }
        )

    def dingtalk_oapi_alitrip_btrip_vehicle_order_search(
            self,
            rq
    ):
        """
        用车订单查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38876

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.vehicle.order.search",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_alitrip_btrip_flight_city_suggest(
            self,
            rq
    ):
        """
        机票城市搜索
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38877

        :param rq: 请求对象
        """
        return self._top_request(
            "dingtalk.oapi.alitrip.btrip.flight.city.suggest",
            {
                "rq": rq
            }
        )

    def dingtalk_oapi_attendance_isopensmartreport(
            self
    ):
        """
        判断是否开启考勤智能报表
        考勤智能报表是考勤为了满足企业个性化的统计需求而开发的一套智能的可编程的报表系统，企业可以通过简单勾选或者高级编程模式来定制自己的统计规则，通过智能报表可以非常方便地输出便于计算薪酬结果的统计数据。该接口可以通过企业corpId来判断企业是否开启了考勤智能报表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38905

        """
        return self._top_request(
            "dingtalk.oapi.attendance.isopensmartreport"
        )

    def dingtalk_oapi_attendance_getattcolumns(
            self
    ):
        """
        获取企业考勤报表列
        该接口用于获取企业智能考勤报表中的列信息，通过获取列信息中的id值，ISV可以根据列的id查询考勤智能报表中该列的统计数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38907

        """
        return self._top_request(
            "dingtalk.oapi.attendance.getattcolumns"
        )

    def dingtalk_oapi_attendance_getcolumnval(
            self,
            userid='',
            column_id_list='',
            from_date='',
            to_date=''
    ):
        """
        获取智能考勤报表的列值
        该接口用于获取钉钉智能考勤报表的列值数据，其中包含了一定时间段内报表某一列的所有数据，以及相关的列信息，可以供指定的ISV进行使用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38908

        :param userid: 用户的userId
        :param column_id_list: 列id
        :param from_date: 开始时间
        :param to_date: 结束时间
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getcolumnval",
            {
                "userid": userid,
                "column_id_list": column_id_list,
                "from_date": from_date,
                "to_date": to_date
            }
        )

    def dingtalk_oapi_impaas_relation_get(
            self,
            src_im_openid,
            dst_im_openids
    ):
        """
        impaas关系查询
        impaas关系查询, 查询 sender im openid 跟 receiver im openid的关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38909

        :param src_im_openid: 发送者钉钉的openid
        :param dst_im_openids: 接收者钉钉的openid
        """
        return self._top_request(
            "dingtalk.oapi.impaas.relation.get",
            {
                "request": {
                    "src_im_openid": src_im_openid,
                    "dst_im_openids": dst_im_openids
                }
            }
        )

    def dingtalk_oapi_impaas_user_getlogintoken(
            self,
            id,
            channel
    ):
        """
        获取免登token
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38910

        :param id: 二方账号
        :param channel: 渠道类型
        """
        return self._top_request(
            "dingtalk.oapi.impaas.user.getlogintoken",
            {
                "request": {
                    "id": id,
                    "channel": channel
                }
            }
        )

    def dingtalk_oapi_user_getDeptMember(
            self,
            dept_id=''
    ):
        """
        根据部门id获取员工ID列表
        aa
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39028

        :param dept_id: a
        """
        return self._top_request(
            "dingtalk.oapi.user.getDeptMember",
            {
                "deptId": dept_id
            }
        )

    def dingtalk_oapi_workrecord_add(
            self,
            userid,
            create_time,
            title,
            url,
            form_item_list,
            originator_user_id='',
            source_name=''
    ):
        """
        新增待办事项
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39074

        :param userid: 用户id
        :param create_time: 待办时间。Unix时间戳
        :param title: 标题
        :param url: 待办跳转url
        :param form_item_list: 表单列表
        :param originator_user_id: manager7078
        :param source_name: 待办来源名称
        """
        return self._top_request(
            "dingtalk.oapi.workrecord.add",
            {
                "userid": userid,
                "create_time": create_time,
                "title": title,
                "url": url,
                "formItemList": form_item_list,
                "originator_user_id": originator_user_id,
                "source_name": source_name
            }
        )

    def dingtalk_oapi_workrecord_update(
            self,
            userid,
            record_id
    ):
        """
        更新待办事项状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39075

        :param userid: 用户id
        :param record_id: 待办事项唯一id
        """
        return self._top_request(
            "dingtalk.oapi.workrecord.update",
            {
                "userid": userid,
                "record_id": record_id
            }
        )

    def dingtalk_oapi_attendance_getleavetimebynames(
            self,
            userid='',
            leave_names='',
            from_date='',
            to_date=''
    ):
        """
        获取考勤智能报表的假期数据
        该接口用于根据字段名和用户id获取钉钉智能考勤报表的假期数据，其中包含了一定时间段内报表假期列的所有数据，由于假期列是一个动态列，因此需要根据假期名称获取数据，可以供指定的ISV进行使用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39271

        :param userid: 用户的userId
        :param leave_names: 列id
        :param from_date: 开始时间
        :param to_date: 结束时间
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getleavetimebynames",
            {
                "userid": userid,
                "leave_names": leave_names,
                "from_date": from_date,
                "to_date": to_date
            }
        )

    def dingtalk_oapi_ding_task_status_update(
            self,
            task_id='',
            task_status='',
            operator_userid=''
    ):
        """
        更新任务完成状态
        企业更新已经发送到用户的任务完成状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39343

        :param task_id: 任务ID
        :param task_status: 任务状态:1-未完成;2-完成
        :param operator_userid: 操作人id
        """
        return self._top_request(
            "dingtalk.oapi.ding.task.status.update",
            {
                "task_id": task_id,
                "task_status": task_status,
                "operator_userid": operator_userid
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_querydimission(
            self,
            offset,
            size
    ):
        """
        智能人事查询公司离职员工列表
        智能人事业务，提供企业/ISV分页查询公司离职员工id列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39418

        :param offset: 分页游标，从0开始。根据返回结果里的next_cursor是否为空来判断是否还有下一页，且再次调用时offset设置成next_cursor的值
        :param size: 分页大小，最大50
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.querydimission",
            {
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_listdimission(
            self,
            userid_list=''
    ):
        """
        批量获取员工离职信息
        根据传入的staffId列表，批量查询员工的离职信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39419

        :param userid_list: 员工id
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.listdimission",
            {
                "userid_list": userid_list
            }
        )

    def dingtalk_oapi_newretail_sendsms(
            self,
            smsmodule=None
    ):
        """
        天地会短信通知用户激活
        提供给天地会调用发短信，提醒用户激活钉钉
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39422

        :param smsmodule: 短信接受者信息
        """
        return self._top_request(
            "dingtalk.oapi.newretail.sendsms",
            {
                "smsmodule": smsmodule
            }
        )

    def dingtalk_oapi_ccoservice_entrance_sendnotify(
            self,
            app_id,
            userid,
            content
    ):
        """
        通过企业的系统账号给企业员工发送消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39429

        :param app_id: 微应用ID
        :param userid: 员工ID
        :param content: 文本的通知
        """
        return self._top_request(
            "dingtalk.oapi.ccoservice.entrance.sendnotify",
            {
                "app_id": app_id,
                "userid": userid,
                "content": content
            }
        )

    def dingtalk_oapi_impaas_user_modprofile(
            self,
            account_info,
            extension='',
            nick='',
            avatar_mediaid=''
    ):
        """
        修改profile信息
        修改profile信息，开放 nick， extension， avatarMediaId
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39433

        :param account_info: 账号信息
        :param extension: 附带信息
        :param nick: nick
        :param avatar_mediaid: 头像
        """
        return self._top_request(
            "dingtalk.oapi.impaas.user.modprofile",
            {
                "request": {
                    "account_info": account_info,
                    "extension": extension,
                    "nick": nick,
                    "avatar_mediaid": avatar_mediaid
                }
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_list(
            self,
            userid_list,
            field_filter_list=''
    ):
        """
        批量获取员工花名册字段信息
        智能人事业务，企业/ISV根据员工id批量访问员工花名册信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39434

        :param userid_list: 员工id列表
        :param field_filter_list: 需要获取的花名册字段信息
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.list",
            {
                "userid_list": userid_list,
                "field_filter_list": field_filter_list
            }
        )

    def dingtalk_oapi_processinstance_listids(
            self,
            process_code,
            start_time,
            end_time='',
            size='10',
            cursor='0',
            userid_list=''
    ):
        """
        分页获取审批实例id列表
        企业可以根据审批流的唯一标识，分页获取该审批流对应的审批实例id。只能取到权限范围内的相关部门的审批实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39471

        :param process_code: 流程模板唯一标识，可在oa后台编辑审批表单部分查询
        :param start_time: 审批实例开始时间，毫秒级
        :param end_time: 审批实例结束时间，毫秒级，默认取当前值
        :param size: 分页参数，每页大小，最多传10
        :param cursor: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值
        :param userid_list: 发起人用户id列表
        """
        return self._top_request(
            "dingtalk.oapi.processinstance.listids",
            {
                "process_code": process_code,
                "start_time": start_time,
                "end_time": end_time,
                "size": size,
                "cursor": cursor,
                "userid_list": userid_list
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_addpreentry(
            self,
            param
    ):
        """
        智能人事添加企业待入职员工
        智能人事添加待入职员工信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39500

        :param param: 添加待入职入参
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.addpreentry",
            {
                "param": param
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_querypreentry(
            self,
            offset,
            size
    ):
        """
        智能人事查询公司待入职员工列表
        智能人事业务，企业/ISV分页查询公司待入职员工id列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39501

        :param offset: 分页起始值，默认0开始
        :param size: 分页大小，最大50
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.querypreentry",
            {
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_queryonjob(
            self,
            status_list,
            offset,
            size
    ):
        """
        智能人事查询公司在职员工列表
        智能人事业务，提供企业/ISV按在职状态分页查询公司在职员工id列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39502

        :param status_list: 在职员工子状态筛选。2，试用期；3，正式；5，待离职；-1，无状态
        :param offset: 分页起始值，默认0开始
        :param size: 分页大小，最大50
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.queryonjob",
            {
                "status_list": status_list,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_smartdevice_visitor_sendnotify(
            self,
            visitor_notify_vo=None,
            reservation_id=''
    ):
        """
        为访客ISV提供发送通知的功能
        提供向被访者或访客接待人员发送通知信息的功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39519

        :param visitor_notify_vo: 访客通知模型
        :param reservation_id: 预约ID
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.visitor.sendnotify",
            {
                "visitor_notify_vo": visitor_notify_vo,
                "reservation_id": reservation_id
            }
        )

    def dingtalk_oapi_sns_send_msg(
            self,
            code='',
            msg=None
    ):
        """
        个人E应用消息推送
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39594

        :param code: form表单提交成功后获取的formId
        :param msg: 消息内容
        """
        return self._top_request(
            "dingtalk.oapi.sns.send_msg",
            {
                "code": code,
                "msg": msg
            }
        )

    def dingtalk_oapi_user_get_admin_scope(
            self,
            userid
    ):
        """
        查询管理员通讯录权限范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39683

        :param userid: 用户id
        """
        return self._top_request(
            "dingtalk.oapi.user.get_admin_scope",
            {
                "userid": userid
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_listcontact(
            self,
            userid_list
    ):
        """
        批量获取离职员工通讯录信息
        ISV批量获取离职员工的通讯录花名册字段信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39977

        :param userid_list: 员工id列表
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.listcontact",
            {
                "userid_list": userid_list
            }
        )

    def dingtalk_oapi_user_listbypage(
            self,
            offset,
            size,
            lang='',
            department_id='',
            order=''
    ):
        """
        获取部门成员(详情)
        获取部门成员（详情）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40282

        :param offset: 支持分页查询，与size参数同时设置时才生效，此参数代表偏移量
        :param size: 支持分页查询，与offset参数同时设置时才生效，此参数代表分页大小，最大100
        :param lang: 通讯录语言(默认zh_CN另外支持en_US)
        :param department_id: 获取的部门id
        :param order: 支持分页查询，部门成员的排序规则，默认不传是按自定义排序；entry_asc代表按照进入部门的时间升序，entry_desc代表按照进入部门的时间降序，modify_asc代表按照部门信息修改时间升序，modify_desc代表按照部门信息修改时间降序，custom代表用户定义(未定义时按照拼音)排序
        """
        return self._top_request(
            "dingtalk.oapi.user.listbypage",
            {
                "offset": offset,
                "size": size,
                "lang": lang,
                "department_id": department_id,
                "order": order
            }
        )

    def dingtalk_oapi_impaas_conversation_updateentranceid(
            self,
            entrance_id,
            accounts,
            uuid,
            channel,
            cid,
            extension=''
    ):
        """
        IMPaaS设置二级会话的入口id
        在二级会话功能中，部分场景需要修改会话入口。另外，数据初始化时，也需要该接口做一部分数据订正
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40563

        :param entrance_id: 入口id，数字
        :param accounts: 要设置的用户列表
        :param uuid: 该请求的唯一id，用于去重、打日志
        :param channel: 业务channel
        :param cid: 会话id
        :param extension: 扩展信息，可选
        """
        return self._top_request(
            "dingtalk.oapi.impaas.conversation.updateentranceid",
            {
                "request": {
                    "entrance_id": entrance_id,
                    "accounts": accounts,
                    "uuid": uuid,
                    "channel": channel,
                    "cid": cid,
                    "extension": extension
                }
            }
        )

    def dingtalk_oapi_attendance_getleavestatus(
            self,
            userid_list,
            start_time,
            end_time,
            offset,
            size
    ):
        """
        请假状态查询接口
        该接口用于查询指定企业下的指定用户在指定时间段内的请假状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40614

        :param userid_list: 待查询用户id列表，支持最多100个用户的批量查询
        :param start_time: 开始时间 ，时间戳，支持最多180天的查询
        :param end_time: 结束时间，时间戳，支持最多180天的查询
        :param offset: 分页偏移，非负整数
        :param size: 分页大小，正整数，最大20
        """
        return self._top_request(
            "dingtalk.oapi.attendance.getleavestatus",
            {
                "userid_list": userid_list,
                "start_time": start_time,
                "end_time": end_time,
                "offset": offset,
                "size": size
            }
        )

    def qimen_dingtalk_flow_form(
            self,
            corp_id,
            user_id,
            process_code,
            action,
            form=None
    ):
        """
        钉钉审批开放表单
        审批表单接入第三方企业数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40627

        :param corp_id: 钉钉企业ID
        :param user_id: 员工ID
        :param process_code: 审批模板唯一编码
        :param action: 表单渲染动作：RENDER, UPDATE, SUBMIT, 分别对应表单的初始化，内容更新以及提交前检查
        :param form: 表单数据
        """
        return self._top_request(
            "qimen.dingtalk.flow.form",
            {
                "corpId": corp_id,
                "userId": user_id,
                "processCode": process_code,
                "action": action,
                "form": form
            }
        )

    def dingtalk_oapi_workrecord_getbyuserid(
            self,
            userid,
            offset,
            limit,
            status
    ):
        """
        获取用户的待办事项
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40726

        :param userid: 用户唯一ID
        :param offset: 分页游标，从0开始，如返回结果中has_more为true，则表示还有数据，offset再传上一次的offset+limit
        :param limit: 分页大小，最多50
        :param status: 待办事项状态，0表示未完成，1表示完成
        """
        return self._top_request(
            "dingtalk.oapi.workrecord.getbyuserid",
            {
                "userid": userid,
                "offset": offset,
                "limit": limit,
                "status": status
            }
        )

    def dingtalk_oapi_process_template_list(
            self,
            userid='',
            offset='',
            size=''
    ):
        """
        分页获取企业下所有的流程模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41001

        :param userid: abc123
        :param offset: 游标属性
        :param size: 每页数量
        """
        return self._top_request(
            "dingtalk.oapi.process.template.list",
            {
                "userid": userid,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_process_property_update(
            self,
            userid='',
            process_code='',
            component_id='',
            props=None
    ):
        """
        修改用户指定审批流程模板的支付属性
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41002

        :param userid: 员工工号 企业唯一
        :param process_code: 模板code
        :param component_id: 控件id
        :param props: 控件属性集
        """
        return self._top_request(
            "dingtalk.oapi.process.property.update",
            {
                "userid": userid,
                "process_code": process_code,
                "component_id": component_id,
                "props": props
            }
        )

    def dingtalk_oapi_chatbot_pictureurl_get(
            self,
            download_code
    ):
        """
        下载机器人授权的图片
        通过downloadCode获取下载url
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41003

        :param download_code: 图片临时下载码
        """
        return self._top_request(
            "dingtalk.oapi.chatbot.pictureurl.get",
            {
                "download_code": download_code
            }
        )

    def dingtalk_oapi_conference_publish(
            self,
            conference_id,
            userid
    ):
        """
        智能会务发布
        钉钉智能会务，将会务信息发布到工作台，参会者可见
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41023

        :param conference_id: 会务id
        :param userid: 操作用户id
        """
        return self._top_request(
            "dingtalk.oapi.conference.publish",
            {
                "conference_id": conference_id,
                "userid": userid
            }
        )

    def dingtalk_oapi_conference_get(
            self,
            conference_id
    ):
        """
        智能会务获取会务基础信息
        钉钉智能会务，提供给ISV获取会务基础信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41024

        :param conference_id: 会务Id
        """
        return self._top_request(
            "dingtalk.oapi.conference.get",
            {
                "conference_id": conference_id
            }
        )

    def dingtalk_oapi_conference_unpublish(
            self,
            conference_id,
            userid
    ):
        """
        智能会务撤销发布
        钉钉智能会务，将会务信息从工作台tag中撤销，参会者不可见
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41025

        :param conference_id: 会务id
        :param userid: 操作用户id
        """
        return self._top_request(
            "dingtalk.oapi.conference.unpublish",
            {
                "conference_id": conference_id,
                "userid": userid
            }
        )

    def dingtalk_oapi_edu_guardian_list(
            self,
            class_id,
            page_no,
            page_size
    ):
        """
        查询班级下家长列表
        查询班级下家长列表信息，通过orgid，班级id查询家长的nick（如：小A的爸爸），非真实姓名，关系类别（妈妈、爸爸、其他亲属），学生staffid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41051

        :param class_id: 班级ID
        :param page_no: 分页页数
        :param page_size: 每页大小
        """
        return self._top_request(
            "dingtalk.oapi.edu.guardian.list",
            {
                "class_id": class_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def dingtalk_oapi_edu_guardian_get(
            self,
            class_id,
            guardian_userid
    ):
        """
        家长信息查询
        家长信息查询，查询家长信息，通过orgid，班级id查询家长的nick（如：小A的爸爸），非真实姓名，关系类别（妈妈、爸爸、其他亲属），学生staffid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41052

        :param class_id: 班级ID
        :param guardian_userid: 家长ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.guardian.get",
            {
                "class_id": class_id,
                "guardian_userid": guardian_userid
            }
        )

    def dingtalk_oapi_edu_student_get(
            self,
            class_id,
            student_userid
    ):
        """
        学生信息查询
        学生信息查询，通过orgid，学生id，班级id,查询学生的名称、所在校区id、年级id、学段id、学生的staffid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41053

        :param class_id: 班级ID
        :param student_userid: 学生ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.student.get",
            {
                "class_id": class_id,
                "student_userid": student_userid
            }
        )

    def dingtalk_oapi_edu_student_list(
            self,
            class_id,
            page_no,
            page_size
    ):
        """
        学生列表信息
        学生列表信息查询,查询学生的名称、所在的班级id、校区id、年级id、学段id、学生的staffid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41054

        :param class_id: 班级ID
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "dingtalk.oapi.edu.student.list",
            {
                "class_id": class_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def dingtalk_oapi_edu_class_get(
            self,
            class_id
    ):
        """
        班级信息查询
        班级信息查询，通过orgid、班级id查询班级名称、所在学段id、校区id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41055

        :param class_id: 班级ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.class.get",
            {
                "class_id": class_id
            }
        )

    def dingtalk_oapi_edu_class_list(
            self,
            grade_id,
            page_no,
            page_size
    ):
        """
        班级列表信息
        班级列表信息，通过orgid、班级id查询班级名称、所在学段id、校区id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41056

        :param grade_id: 年级ID
        :param page_no: 分页页数
        :param page_size: 分页每页大小
        """
        return self._top_request(
            "dingtalk.oapi.edu.class.list",
            {
                "grade_id": grade_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def dingtalk_oapi_edu_grade_get(
            self,
            grade_id
    ):
        """
        年级信息查询
        年级信息查询接口，通过组织id和年级id查询年级名称、年级的级别、学段id、校区id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41057

        :param grade_id: 年级ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.grade.get",
            {
                "grade_id": grade_id
            }
        )

    def dingtalk_oapi_edu_grade_list(
            self,
            period_id
    ):
        """
        年级列表查询
        年级列表查询，通过学段id查询其下面的年级信息列表，最多返回10条数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41058

        :param period_id: 学段ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.grade.list",
            {
                "period_id": period_id
            }
        )

    def dingtalk_oapi_edu_teacher_get(
            self,
            class_id,
            teacher_userid
    ):
        """
        查询老师信息
        查询老师信息，查询班级下老师列表，通过组织id和班级id查询老师的名称、班级id、staffid、是否为班主任、校区id信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41059

        :param class_id: 班级ID
        :param teacher_userid: 用户ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.teacher.get",
            {
                "class_id": class_id,
                "teacher_userid": teacher_userid
            }
        )

    def dingtalk_oapi_edu_campus_list(
            self
    ):
        """
        查询校区列表
        查询某个组织下面的学校信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41060

        """
        return self._top_request(
            "dingtalk.oapi.edu.campus.list"
        )

    def dingtalk_oapi_edu_campus_get(
            self,
            campus_id
    ):
        """
        校区信息查询
        查询校区基本信息接口，返回校区的名称和校区id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41061

        :param campus_id: 校区ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.campus.get",
            {
                "campus_id": campus_id
            }
        )

    def dingtalk_oapi_edu_period_list(
            self,
            campus_id
    ):
        """
        学段列表查询
        学段列表查询，每个校区仅有4个学段，幼儿园、小学、初中、高中，因此最多4条数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41062

        :param campus_id: 校区ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.period.list",
            {
                "campus_id": campus_id
            }
        )

    def dingtalk_oapi_edu_period_get(
            self,
            period_id
    ):
        """
        查询学段信息
        学段信息查询接口，通过orgid，学段id查询学段信息，返回学段名称、nick、校区id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41063

        :param period_id: 学段ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.period.get",
            {
                "period_id": period_id
            }
        )

    def dingtalk_oapi_edu_teacher_list(
            self,
            class_id
    ):
        """
        查询班级下老师列表
        查询班级下老师列表，通过组织id和班级id查询老师的名称、班级id、staffid、是否为班主任、校区id信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41064

        :param class_id: 班级ID
        """
        return self._top_request(
            "dingtalk.oapi.edu.teacher.list",
            {
                "class_id": class_id
            }
        )

    def dingtalk_oapi_corp_conversation_member_list(
            self,
            chat_id,
            offset,
            size
    ):
        """
        分页获取群成员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41168

        :param chat_id: 会话ID
        :param offset: 偏移位置，从0开始，后续传offset+size的值。member_list返回为空表示结束了。
        :param size: 请求数量
        """
        return self._top_request(
            "dingtalk.oapi.corp.conversation.member.list",
            {
                "chat_id": chat_id,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_faceauth_get(
            self,
            tmp_auth_code,
            app_biz_id
    ):
        """
        实名认证中的人脸扫描结果查询
        实名认证中的人脸扫描结果查询开放
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41270

        :param tmp_auth_code: 人脸扫描的授权码
        :param app_biz_id: 业务方定义的id
        """
        return self._top_request(
            "dingtalk.oapi.faceauth.get",
            {
                "tmp_auth_code": tmp_auth_code,
                "app_biz_id": app_biz_id
            }
        )

    def dingtalk_oapi_impaas_otoconversation_create(
            self,
            account_info_list=None,
            entrance_id_list='',
            uuid='',
            channel='',
            extension=''
    ):
        """
        创建单聊二级会话
        部分新零售场景需要支持二级会话，也就是把部分同类会话聚合在一起
        本接口给二方提供了创建二级单聊会话的能力
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41535

        :param account_info_list: 账号列表，size=2。第一个表示自己，第二个表示对方
        :param entrance_id_list: 入口id列表，size=2。普通会话填0，二级会话填entrnaceid
        :param uuid: 用于去重和追踪
        :param channel: channel名称
        :param extension: 扩展信息
        """
        return self._top_request(
            "dingtalk.oapi.impaas.otoconversation.create",
            {
                "request": {
                    "account_info_list": account_info_list,
                    "entrance_id_list": entrance_id_list,
                    "uuid": uuid,
                    "channel": channel,
                    "extension": extension
                }
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_member_update(
            self,
            biz_id,
            add_user_ids='',
            del_user_ids=''
    ):
        """
        人脸识别组人员更新
        更新人脸识别组的人员，如新增人员、移除人员
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41758

        :param biz_id: 业务id
        :param add_user_ids: 需新增的用户id列表
        :param del_user_ids: 需移除的用户id列表
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.member.update",
            {
                "biz_id": biz_id,
                "add_user_ids": add_user_ids,
                "del_user_ids": del_user_ids
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_member_list(
            self,
            biz_id,
            cursor,
            size
    ):
        """
        查询已在识别组的人员
        查询已在识别组的人员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41759

        :param biz_id: 业务id
        :param cursor: 游标，第一次传 <=0的值，后续传本调用的返回值
        :param size: 分页大小
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.member.list",
            {
                "biz_id": biz_id,
                "cursor": cursor,
                "size": size
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_device_update(
            self,
            biz_id,
            add_device_ids='',
            del_device_ids=''
    ):
        """
        更新识别组关联的人脸设备
        更新人脸识别组绑定是设备，如新增、移除关联的设备
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41760

        :param biz_id: 业务id
        :param add_device_ids: 需新增的设备id列表
        :param del_device_ids: 需移除的设备id列表
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.device.update",
            {
                "biz_id": biz_id,
                "add_device_ids": add_device_ids,
                "del_device_ids": del_device_ids
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_device_list(
            self,
            biz_id,
            size,
            mode,
            type,
            cursor
    ):
        """
        查询人脸设备列表
        查询人脸设备列表，可选择查询已关联设备列表、或企业已有设备的列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41761

        :param biz_id: 业务id
        :param size: 分页大小
        :param mode: 查询模式：all-企业所有设备；bound-本组已关联设备
        :param type: 需查询的设备类型
        :param cursor: 游标，第一次传 <=0的值，后续传本调用的返回值
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.device.list",
            {
                "biz_id": biz_id,
                "size": size,
                "mode": mode,
                "type": type,
                "cursor": cursor
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_get(
            self,
            biz_id
    ):
        """
        查询人脸识别组
        通过业务id查询识别组基础信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41762

        :param biz_id: 业务id
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.get",
            {
                "biz_id": biz_id
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_create(
            self,
            biz_id,
            title,
            start_time,
            end_time,
            status,
            greeting_msg='',
            bg_img_url=''
    ):
        """
        创建人脸识别组
        创建人脸识别组基础信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41763

        :param biz_id: 业务id【不区分大小写】：调用方内保证唯一即可，可容纳23个字符，推荐前三个字符表示业务编号，留19个字符存储业务的记录id
        :param title: 识别组的标题
        :param start_time: 开始时间
        :param end_time: 结束时间
        :param status: 识别组启用状态：1-已启用；2未启用；
        :param greeting_msg: 识别成功后的问候语
        :param bg_img_url: M2上的定制UI
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.create",
            {
                "biz_id": biz_id,
                "title": title,
                "start_time": start_time,
                "end_time": end_time,
                "status": status,
                "greeting_msg": greeting_msg,
                "bg_img_url": bg_img_url
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_update(
            self,
            biz_id,
            title,
            start_time,
            end_time,
            status,
            greeting_msg='',
            bg_img_url=''
    ):
        """
        更新识别组基础信息
        更新人脸识别组基础信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41764

        :param biz_id: 业务id
        :param title: 识别组的标题
        :param start_time: 开始时间
        :param end_time: 结束时间
        :param status: 识别组启用状态：1-已启用；2未启用；
        :param greeting_msg: 识别成功后的问候语
        :param bg_img_url: M2上的定制UI
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.update",
            {
                "biz_id": biz_id,
                "title": title,
                "start_time": start_time,
                "end_time": end_time,
                "status": status,
                "greeting_msg": greeting_msg,
                "bg_img_url": bg_img_url
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_enable(
            self,
            biz_id,
            enable,
            device_ids=''
    ):
        """
        启用/禁用M2人脸识别
        启用/禁用当前识别组的人脸识别功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41765

        :param biz_id: 业务id
        :param enable: true-启用识别；false-禁用识别
        :param device_ids: 设备id列表
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.enable",
            {
                "biz_id": biz_id,
                "enable": enable,
                "device_ids": device_ids
            }
        )

    def dingtalk_oapi_smartdevice_facegroup_removeall(
            self,
            biz_id
    ):
        """
        删除人脸识别组
        删除人脸识别组，并解绑关联数据（人员、M2绑定关系等）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41766

        :param biz_id: 业务id
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.facegroup.removeall",
            {
                "biz_id": biz_id
            }
        )

    def dingtalk_oapi_im_chatbot_delete(
            self,
            open_conversation_id,
            chatbot_user_id
    ):
        """
        删除会话内机器人实例
        根据开放的conversationId和机器人示例userId，删除会话内机器人实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41967

        :param open_conversation_id: 开放的会话conversationId
        :param chatbot_user_id: 开放的机器人userId
        """
        return self._top_request(
            "dingtalk.oapi.im.chatbot.delete",
            {
                "open_conversation_id": open_conversation_id,
                "chatbot_user_id": chatbot_user_id
            }
        )

    def dingtalk_oapi_im_chatbot_get(
            self,
            open_conversation_id
    ):
        """
        获取会话内机器人实例列表
        根据开放的conversationId，获取该会话内机器人实例列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41968

        :param open_conversation_id: 开放的会话conversationId
        """
        return self._top_request(
            "dingtalk.oapi.im.chatbot.get",
            {
                "open_conversation_id": open_conversation_id
            }
        )

    def dingtalk_oapi_im_chat_cid_convert(
            self,
            chat_id
    ):
        """
        会话chatId转conversationId
        根据会话chatId，查询其对应的conversationId
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41969

        :param chat_id: 开放的chatId
        """
        return self._top_request(
            "dingtalk.oapi.im.chat.cid.convert",
            {
                "chat_id": chat_id
            }
        )

    def dingtalk_oapi_im_chat_servicegroup_create(
            self,
            title,
            owner_userid,
            org_inner_group='false'
    ):
        """
        创建服务群
        创建一个服务群
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42043

        :param title: 群标题
        :param owner_userid: 群主在钉钉组织内的userid
        :param org_inner_group: 是否企业内部服务群
        """
        return self._top_request(
            "dingtalk.oapi.im.chat.servicegroup.create",
            {
                "title": title,
                "owner_userid": owner_userid,
                "org_inner_group": org_inner_group
            }
        )

    def dingtalk_oapi_im_chat_servicegroup_disband(
            self,
            chat_id
    ):
        """
        解散一个服务群
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42044

        :param chat_id: 开放的chatId
        """
        return self._top_request(
            "dingtalk.oapi.im.chat.servicegroup.disband",
            {
                "chat_id": chat_id
            }
        )

    def dingtalk_oapi_im_chat_servicegroup_member_query(
            self,
            chat_id,
            page_size,
            page_num,
            include_owner='0'
    ):
        """
        获取服务群成员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42045

        :param chat_id: 开放的chatId
        :param page_size: 每页大小，最大100
        :param page_num: 页码，从1开始
        :param include_owner: 0- 不包含群主，1-包含群主
        """
        return self._top_request(
            "dingtalk.oapi.im.chat.servicegroup.member.query",
            {
                "chat_id": chat_id,
                "page_size": page_size,
                "page_num": page_num,
                "include_owner": include_owner
            }
        )

    def dingtalk_oapi_im_chat_servicegroup_query(
            self,
            chat_id
    ):
        """
        获取服务群基本信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42046

        :param chat_id: 开放的chatId
        """
        return self._top_request(
            "dingtalk.oapi.im.chat.servicegroup.query",
            {
                "chat_id": chat_id
            }
        )

    def dingtalk_oapi_im_chat_servicegroup_member_update(
            self,
            member_dingtalk_ids,
            action,
            chat_id
    ):
        """
        更新群成员（移除、设为管理员）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42047

        :param member_dingtalk_ids: 成员的ID列表，英文逗号分隔
        :param action: 更新类型，REMOVE 移除、SET_ADMIN设为管理员、SET_NORMAL 设为普通成员
        :param chat_id: 开放的chatId
        """
        return self._top_request(
            "dingtalk.oapi.im.chat.servicegroup.member.update",
            {
                "member_dingtalk_ids": member_dingtalk_ids,
                "action": action,
                "chat_id": chat_id
            }
        )

    def dingtalk_oapi_smartdevice_bind_create(
            self,
            device_bind_req_vo
    ):
        """
        智能硬件建立绑定关系
        智能设备接入钉钉时，需要和组织建立绑定关系，此接口用于创建绑定关系。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42084

        :param device_bind_req_vo: 设备请求信息
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.bind.create",
            {
                "device_bind_req_vo": device_bind_req_vo
            }
        )

    def dingtalk_oapi_statistics_details(
            self,
            type=''
    ):
        """
        大客户企业开放数据
        对大客户开放其企业BI数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42090

        :param type: 业务类型，目前有employee,group, live
        """
        return self._top_request(
            "dingtalk.oapi.statistics.details",
            {
                "type": type
            }
        )

    def dingtalk_oapi_chatbot_message_send(
            self,
            chatbot_id,
            userid,
            message
    ):
        """
        机器人发送消息
        通过机器人单聊会话主动给企业内员工发送消息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42230

        :param chatbot_id: 企业机器人模板类型
        :param userid: 企业员工ID
        :param message: 消息内容,支持的消息类型详见：https://open-doc.dingtalk.com/microapp/serverapi2/qf2nxq#a-namesgw3aga%E6%B6%88%E6%81%AF%E7%B1%BB%E5%9E%8B%E5%8F%8A%E6%95%B0%E6%8D%AE%E6%A0%BC%E5%BC%8F
        """
        return self._top_request(
            "dingtalk.oapi.chatbot.message.send",
            {
                "chatbot_id": chatbot_id,
                "userid": userid,
                "message": message
            }
        )

    def dingtalk_oapi_attendance_corp_inviteactive_open(
            self,
            admin_name,
            admin_phone
    ):
        """
        喔趣企业开通邀1得1活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42470

        :param admin_name: 姓名
        :param admin_phone: 手机号
        """
        return self._top_request(
            "dingtalk.oapi.attendance.corp.inviteactive.open",
            {
                "admin_name": admin_name,
                "admin_phone": admin_phone
            }
        )

    def dingtalk_oapi_live_grouplive_statistics(
            self,
            live_uuid,
            cid='',
            open_id=''
    ):
        """
        查询内部群某一场直播的数据统计
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42136

        :param live_uuid: 直播uuid
        :param cid: 群id
        :param open_id: 用户id
        """
        return self._top_request(
            "dingtalk.oapi.live.grouplive.statistics",
            {
                "live_uuid": live_uuid,
                "cid": cid,
                "open_id": open_id
            }
        )

    def dingtalk_oapi_live_grouplive_list(
            self,
            cid,
            open_id='',
            to_time='',
            from_time=''
    ):
        """
        查询内部群直播的列表
        查询内部群的直播列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42137

        :param cid: 群id
        :param open_id: 用户id
        :param to_time: 开始时间ms
        :param from_time: 截止时间ms
        """
        return self._top_request(
            "dingtalk.oapi.live.grouplive.list",
            {
                "cid": cid,
                "open_id": open_id,
                "to_time": to_time,
                "from_time": from_time
            }
        )

    def dingtalk_oapi_process_form_get(
            self,
            process_code
    ):
        """
        获取表单schema
        ISV通过这个接口获取流程的表单schema信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42472

        :param process_code: 流程模板code
        """
        return self._top_request(
            "dingtalk.oapi.process.form.get",
            {
                "process_code": process_code
            }
        )

    def dingtalk_oapi_process_template_save(
            self,
            process_code,
            vm,
            font
    ):
        """
        自定义打印模板保存
        自定义打印模板文件保存
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42473

        :param process_code: 流程编码
        :param vm: vm文件
        :param font: 字体
        """
        return self._top_request(
            "dingtalk.oapi.process.template.save",
            {
                "process_code": process_code,
                "vm": vm,
                "font": font
            }
        )

    def dingtalk_oapi_calendar_list(
            self,
            user_id,
            calendar_folder_id='',
            time_min=None,
            i_cal_uid='',
            single_events='',
            page_token='',
            max_results='',
            time_max=None
    ):
        """
        日程查询
        该api是提供给企业和isv用的，查询企业内员工的日程数据。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42527

        :param user_id: 员工ID
        :param calendar_folder_id: 钉钉日历文件夹的对外id，默认是自己的默认文件夹
        :param time_min: 查询时间下限
        :param i_cal_uid: 日程跨域唯一id，用于唯一标识一组关联日程事件
        :param single_events: 是否需要展开循环日程
        :param page_token: 查询对应页，值有上一次请求返回的结果里对应nextPageToken
        :param max_results: 结果返回的最多数量，默认250，最多返回2500
        :param time_max: 查询时间上限
        """
        return self._top_request(
            "dingtalk.oapi.calendar.list",
            {
                "user_id": user_id,
                "calendar_folder_id": calendar_folder_id,
                "time_min": time_min,
                "i_cal_uid": i_cal_uid,
                "single_events": single_events,
                "page_token": page_token,
                "max_results": max_results,
                "time_max": time_max
            }
        )

    def dingtalk_oapi_live_playback(
            self,
            start_time='',
            page_size='',
            offset='',
            end_time=''
    ):
        """
        直播回放查询能力开放
        提供ISV开放回访获取能力：ISV通过该接口调用，根据直播的开始和结束时间，分页获取到所有的直播回放信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42652

        :param start_time: 直播开始时间
        :param page_size: 直播结束时间
        :param offset: 页面大小
        :param end_time: 偏移量
        """
        return self._top_request(
            "dingtalk.oapi.live.playback",
            {
                "request": {
                    "start_time": start_time,
                    "page_size": page_size,
                    "offset": offset,
                    "end_time": end_time
                }
            }
        )

    def dingtalk_oapi_live_create(
            self,
            userid,
            title,
            user_nick='',
            intro='',
            shared='',
            cover_url='',
            land_scape='',
            appt_begin_time='',
            appt_end_time='',
            pre_video_play_url=''
    ):
        """
        直播创建能力开放
        提供ISV开放直播创建能力：ISV通过该接口调用，提供直播的基本信息，在钉钉创建直播，并获取到直播的推流地址。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42653

        :param userid: 主播ID,必填
        :param title: 标题,必填
        :param user_nick: 别名,选填
        :param intro: 简介,选填
        :param shared: 直播可观看类型类型,必填: false 受限制的直播, true 公开的直播(默认)
        :param cover_url: 封面图,选填: 如果不填写, 则采用默认
        :param land_scape: 横竖屏,选填: false 竖屏, true 横屏(默认)
        :param appt_begin_time: 直播计划开始时间,选填: 如果不填写, 则取当前时间
        :param appt_end_time: 直播计划结束时间,选填
        :param pre_video_play_url: 预告视频Url,选填
        """
        return self._top_request(
            "dingtalk.oapi.live.create",
            {
                "request": {
                    "userid": userid,
                    "title": title,
                    "user_nick": user_nick,
                    "intro": intro,
                    "shared": shared,
                    "cover_url": cover_url,
                    "land_scape": land_scape,
                    "appt_begin_time": appt_begin_time,
                    "appt_end_time": appt_end_time,
                    "pre_video_play_url": pre_video_play_url
                }
            }
        )

    def dingtalk_oapi_live_query(
            self,
            uuid
    ):
        """
        直播查询能力开放
        提供ISV开放直播查询能力：ISV通过该接口调用，根据直播uuid获取直播的详细信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42654

        :param uuid: 直播UUID,必填
        """
        return self._top_request(
            "dingtalk.oapi.live.query",
            {
                "request": {
                    "uuid": uuid
                }
            }
        )

    def dingtalk_oapi_process_baseinfo_list(
            self,
            process_codes=''
    ):
        """
        查询模板基础信息列表
        查询企业下的模板基础信息，包括名称，图片，管理员等
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42744

        :param process_codes: 模板code列表
        """
        return self._top_request(
            "dingtalk.oapi.process.baseinfo.list",
            {
                "process_codes": process_codes
            }
        )

    def dingtalk_oapi_process_template_upgradeinfo_query(
            self,
            process_codes
    ):
        """
        获取模板升级信息
        获取模板升级信息,模板是否可升级
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42785

        :param process_codes: 流程编码List<String>类型
        """
        return self._top_request(
            "dingtalk.oapi.process.template.upgradeinfo.query",
            {
                "process_codes": process_codes
            }
        )

    def dingtalk_oapi_process_template_upgrade(
            self,
            process_code,
            form_component_id,
            userid,
            detail_component_id=''
    ):
        """
        审批模板升级
        升级企业审批模板中的金额类组件到新的支付套件
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42786

        :param process_code: 流程code
        :param form_component_id: 数组或金额类组件id
        :param userid: 其实是staffId
        :param detail_component_id: 明细组件id
        """
        return self._top_request(
            "dingtalk.oapi.process.template.upgrade",
            {
                "process_code": process_code,
                "form_component_id": form_component_id,
                "userid": userid,
                "detail_component_id": detail_component_id
            }
        )

    def dingtalk_oapi_dingpay_order_terminate(
            self,
            order_nos,
            operator,
            reason,
            extension=''
    ):
        """
        中止订单支付
        中止dingpay订单的支付
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42787

        :param order_nos: dingpay单号列表
        :param operator: 操作者员工号
        :param reason: 中止原因
        :param extension: 扩展信息
        """
        return self._top_request(
            "dingtalk.oapi.dingpay.order.terminate",
            {
                "order_nos": order_nos,
                "operator": operator,
                "reason": reason,
                "extension": extension
            }
        )

    def dingtalk_oapi_customize_conversation_update(
            self,
            userid,
            chat_id,
            extension_key,
            extension_value
    ):
        """
        定制化服务会话标配置
        对会话进行打标，打标之后，根据另外的配置，可以让客户端使用weex展示会话，而不是native展示
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42906

        :param userid: 员工账号
        :param chat_id: 会话id
        :param extension_key: extensionKey
        :param extension_value: extensionValue
        """
        return self._top_request(
            "dingtalk.oapi.customize.conversation.update",
            {
                "userid": userid,
                "chat_id": chat_id,
                "extension_key": extension_key,
                "extension_value": extension_value
            }
        )

    def dingtalk_oapi_smartwork_hrm_employee_unionexport(
            self,
            param
    ):
        """
        智能人事联合导出
        智能人事联合导出，ISV提供业务excel并制定需要填补的数据，由智能人事填补后，直接对用户提供下载服务。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42924

        :param param: 导出请求对象
        """
        return self._top_request(
            "dingtalk.oapi.smartwork.hrm.employee.unionexport",
            {
                "param": param
            }
        )

    def dingtalk_oapi_callback_failrecord_confirm(
            self,
            id_list
    ):
        """
        确认开放平台回调失败记录
        查询开放平台回调失败记录，提供此接口用于确认消费
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42935

        :param id_list: 失败记录id列表
        """
        return self._top_request(
            "dingtalk.oapi.callback.failrecord.confirm",
            {
                "id_list": id_list
            }
        )

    def dingtalk_oapi_callback_failrecord_list(
            self,
            req
    ):
        """
        查询回调失败记录
        提供给isv用于查询http回调失败记录，做补偿
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42936

        :param req: 请求参数
        """
        return self._top_request(
            "dingtalk.oapi.callback.failrecord.list",
            {
                "req": req
            }
        )

    def dingtalk_oapi_report_statistics(
            self,
            report_id
    ):
        """
        获取日志统计数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43020

        :param report_id: 日志id
        """
        return self._top_request(
            "dingtalk.oapi.report.statistics",
            {
                "report_id": report_id
            }
        )

    def dingtalk_oapi_attendance_vacation_record_list(
            self,
            op_userid,
            leave_code,
            userids,
            offset,
            size
    ):
        """
        查询假期消费记录
        根据企业或员工分页获取假期消费记录信息 每次返回50条数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43049

        :param op_userid: 操作者ID
        :param leave_code: 假期类型唯一标识
        :param userids: 待查询员工ID列表
        :param offset: 分页偏移(从0开始非负整数)
        :param size: 分页偏移(正整数 最大50)
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.record.list",
            {
                "op_userid": op_userid,
                "leave_code": leave_code,
                "userids": userids,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_attendance_vacation_quota_list(
            self,
            leave_code,
            op_userid,
            userids,
            offset,
            size
    ):
        """
        查询假期余额
        根据企业或员工分页获取假期余额信息 每次返回50条数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43050

        :param leave_code: 假期类型唯一标识
        :param op_userid: 操作者ID
        :param userids: 待查询的员工ID列表
        :param offset: 分页偏移(从0开始非负整数)
        :param size: 分页偏移(正整数 最大50)
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.quota.list",
            {
                "leave_code": leave_code,
                "op_userid": op_userid,
                "userids": userids,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_appstore_internal_order_get(
            self,
            biz_order_id
    ):
        """
        获取内购订单信息
        应用内购流程中，通过该接口查询内购订单信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43055

        :param biz_order_id: 内购商品订单号
        """
        return self._top_request(
            "dingtalk.oapi.appstore.internal.order.get",
            {
                "biz_order_id": biz_order_id
            }
        )

    def dingtalk_oapi_appstore_internal_skupage_get(
            self,
            goods_code,
            item_code='',
            callback_page='',
            extend_param=''
    ):
        """
        获取内购商品SKU详情页面地址
        应用内购流程中，通过该接口获取内购商品SKU页面地址
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43056

        :param goods_code: 内购商品码
        :param item_code: 内购商品规格码，如果设置了规格码，页面会默认选中该规格码
        :param callback_page: 回调页面(进行URLEncode处理)，微应用为页面URL，E应用为页面路径地址
        :param extend_param: 与callbackPage配合使用。当用户从SKU页面下单并支付成功后，会跳转回ISV页面，此时将此参数原样回传。主要用于用户页面引导等操作，不能作为权益开通凭证。
        """
        return self._top_request(
            "dingtalk.oapi.appstore.internal.skupage.get",
            {
                "goods_code": goods_code,
                "item_code": item_code,
                "callback_page": callback_page,
                "extend_param": extend_param
            }
        )

    def dingtalk_oapi_appstore_internal_unfinishedorder_list(
            self,
            page,
            page_size,
            item_code=''
    ):
        """
        获取ISV未完成处理订单
        应用内购流程中，通过该接口获取ISV未处理完成的内购订单列表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43057

        :param page: 分页查询页码，其实页码为1
        :param page_size: 分页查询每页大小，最大限制100
        :param item_code: 商品规格码
        """
        return self._top_request(
            "dingtalk.oapi.appstore.internal.unfinishedorder.list",
            {
                "page": page,
                "page_size": page_size,
                "item_code": item_code
            }
        )

    def dingtalk_oapi_appstore_internal_order_finish(
            self,
            biz_order_id
    ):
        """
        内购商品订单处理完成
        应用内购流程中，通过该接口回传ISV完成订单处理的状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43058

        :param biz_order_id: 内购订单号
        """
        return self._top_request(
            "dingtalk.oapi.appstore.internal.order.finish",
            {
                "biz_order_id": biz_order_id
            }
        )

    def dingtalk_oapi_appstore_internal_order_consume(
            self,
            biz_order_id,
            request_id,
            quantity,
            userid
    ):
        """
        应用内购商品核销
        应用内购流程中，通过该接口对内购订单进行核销
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43059

        :param biz_order_id: 内购商品订单号
        :param request_id: 核销请求Id，由ISV生成，用于请求幂等
        :param quantity: 订购商品核销数量
        :param userid: 员工在当前企业内的唯一标识，也称staffId
        """
        return self._top_request(
            "dingtalk.oapi.appstore.internal.order.consume",
            {
                "biz_order_id": biz_order_id,
                "request_id": request_id,
                "quantity": quantity,
                "userid": userid
            }
        )

    def dingtalk_oapi_attendance_vacation_quota_update(
            self,
            leave_quotas,
            op_userid
    ):
        """
        批量更新假期余额
        企业批量更新假期余额信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43085

        :param leave_quotas: 待更新的假期余额记录
        :param op_userid: 操作者ID
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.quota.update",
            {
                "leave_quotas": leave_quotas,
                "op_userid": op_userid
            }
        )

    def dingtalk_oapi_attendance_vacation_quota_init(
            self,
            op_userid,
            leave_quotas
    ):
        """
        初始化假期余额
        批量初始化假期余额 清空指定假期类型 指定员工所有假期余额 消费记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43086

        :param op_userid: 操作者ID
        :param leave_quotas: 待初始化的假期余额记录
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.quota.init",
            {
                "op_userid": op_userid,
                "leave_quotas": leave_quotas
            }
        )

    def dingtalk_oapi_attendance_vacation_type_update(
            self,
            op_userid,
            leave_code,
            leave_name='',
            leave_view_unit='',
            biz_type='',
            natural_day_leave='',
            hours_in_per_day='',
            extras=''
    ):
        """
        假期类型更新
        该接口用于更新指定企业下的指定假期类型的相关规则
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43092

        :param op_userid: 操作者ID
        :param leave_code: 假期类型唯一标识
        :param leave_name: 假期名称
        :param leave_view_unit: 请假单位，可以按照天半天或者小时请假。(day、halfday、hour其中一种)
        :param biz_type: 假期类型，普通假期或者加班转调休假期。(general_leave、lieu_leave其中一种)
        :param natural_day_leave: 是否按照自然日统计请假时长，当为false的时候，用户发起请假时候会根据用户在请假时间段内的排班情况来计算请假时长。
        :param hours_in_per_day: 每天折算的工作时长(百分之一 例如1天=10小时=1000)
        :param extras: 调休假有效期规则(validity_type:有效类型 absolute_time(绝对时间)、relative_time(相对时间)其中一种 validity_value:延长日期(当validity_type为absolute_time该值该值不为空且满足yy-mm格式 validity_type为relative_time该值为大于1的整数))
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.type.update",
            {
                "op_userid": op_userid,
                "leave_code": leave_code,
                "leave_name": leave_name,
                "leave_view_unit": leave_view_unit,
                "biz_type": biz_type,
                "natural_day_leave": natural_day_leave,
                "hours_in_per_day": hours_in_per_day,
                "extras": extras
            }
        )

    def dingtalk_oapi_attendance_vacation_type_list(
            self,
            op_userid
    ):
        """
        假期类型查询
        查询假期类型
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43093

        :param op_userid: 操作员ID
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.type.list",
            {
                "op_userid": op_userid
            }
        )

    def dingtalk_oapi_attendance_vacation_type_delete(
            self,
            leave_code,
            op_userid
    ):
        """
        假期类型删除接口
        删除指定的假期类型
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43094

        :param leave_code: 假期类型唯一标识
        :param op_userid: 操作员ID
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.type.delete",
            {
                "leave_code": leave_code,
                "op_userid": op_userid
            }
        )

    def dingtalk_oapi_attendance_vacation_type_create(
            self,
            leave_name,
            leave_view_unit,
            biz_type,
            natural_day_leave,
            op_userid,
            hours_in_per_day,
            extras=''
    ):
        """
        假期类型添加
        添加假期类型
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43095

        :param leave_name: 假期名称
        :param leave_view_unit: 请假单位，可以按照天半天或者小时请假。(day、halfday、hour其中一种)
        :param biz_type: 假期类型，普通假期或者加班转调休假期。(general_leave、lieu_leave其中一种)
        :param natural_day_leave: 是否按照自然日统计请假时长，当为false的时候，用户发起请假时候会根据用户在请假时间段内的排班情况来计算请假时长
        :param op_userid: 操作者ID
        :param hours_in_per_day: 每天折算的工作时长(百分之一 例如1天=10小时=1000)
        :param extras: 调休假有效期规则(validity_type:有效类型 absolute_time(绝对时间)、relative_time(相对时间)其中一种 validity_value:延长日期(当validity_type为absolute_time该值该值不为空且满足yy-mm格式 validity_type为relative_time该值为大于1的整数))
        """
        return self._top_request(
            "dingtalk.oapi.attendance.vacation.type.create",
            {
                "leave_name": leave_name,
                "leave_view_unit": leave_view_unit,
                "biz_type": biz_type,
                "natural_day_leave": natural_day_leave,
                "op_userid": op_userid,
                "hours_in_per_day": hours_in_per_day,
                "extras": extras
            }
        )

    def dingtalk_oapi_calendar_delete(
            self,
            userid='',
            calendar_id=''
    ):
        """
        日程删除
        删除一个指定的日程
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43141

        :param userid: 员工id
        :param calendar_id: 日程id
        """
        return self._top_request(
            "dingtalk.oapi.calendar.delete",
            {
                "userid": userid,
                "calendar_id": calendar_id
            }
        )

    def dingtalk_oapi_message_send_to_single_conversation(
            self,
            sender_userid,
            receiver_userid,
            msg
    ):
        """
        企业内个人之间发送单聊消息接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43230

        :param sender_userid: 发送者userId
        :param receiver_userid: 接收者userId
        :param msg: 推送消息内容
        """
        return self._top_request(
            "dingtalk.oapi.message.send_to_single_conversation",
            {
                "sender_userid": sender_userid,
                "receiver_userid": receiver_userid,
                "msg": msg
            }
        )

    def dingtalk_oapi_workbench_shortcut_delete(
            self,
            app_id,
            biz_no
    ):
        """
        删除企业已添加的三方快捷方式
        提供给ISV使用的用于删除已经添加到企业的应用快捷方式
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43265

        :param app_id: 应用ID
        :param biz_no: 系统交互唯一流水号(ISV维度下不可重复)
        """
        return self._top_request(
            "dingtalk.oapi.workbench.shortcut.delete",
            {
                "app_id": app_id,
                "biz_no": biz_no
            }
        )

    def dingtalk_oapi_workbench_shortcut_getguideuri(
            self,
            app_id
    ):
        """
        获取快捷方式添加到工作台引导页地址
        ISV将应用快捷方式添加到企业后会引导用户将该快捷方式添加到工作台,需要获取到工作台快捷方式引导页地址
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43266

        :param app_id: ISV微应用ID
        """
        return self._top_request(
            "dingtalk.oapi.workbench.shortcut.getguideuri",
            {
                "app_id": app_id
            }
        )

    def dingtalk_oapi_workbench_shortcut_list(
            self,
            app_id,
            page_size='',
            page_index=''
    ):
        """
        获取企业对指定应用已安装的快捷方式
        根据企业 + 应用信息等(分页)获取企业下对应该应用已经开通的快捷方式列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43267

        :param app_id: ISV微应用id
        :param page_size: 每页记录数
        :param page_index: 当前页码
        """
        return self._top_request(
            "dingtalk.oapi.workbench.shortcut.list",
            {
                "app_id": app_id,
                "page_size": page_size,
                "page_index": page_index
            }
        )

    def dingtalk_oapi_workbench_shortcut_add(
            self,
            shortcut_uri,
            icon,
            app_id,
            name,
            biz_no,
            pc_shortcut_uri=''
    ):
        """
        给开通应用企业添加快捷方式
        ISV给开通其应用的企业添加快捷方式
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43268

        :param shortcut_uri: 移动端快捷方式跳转地址(默认地址)
        :param icon: 图标Url
        :param app_id: 微应用ID
        :param name: 快捷方式名称
        :param biz_no: 系统交互唯一业务号,ISV企业下唯一
        :param pc_shortcut_uri: PC端快捷方式跳转地址
        """
        return self._top_request(
            "dingtalk.oapi.workbench.shortcut.add",
            {
                "shortcut_uri": shortcut_uri,
                "icon": icon,
                "app_id": app_id,
                "name": name,
                "biz_no": biz_no,
                "pc_shortcut_uri": pc_shortcut_uri
            }
        )

    def dingtalk_oapi_workbench_shortcut_update(
            self,
            app_id,
            biz_no,
            shortcut_uri='',
            icon='',
            name='',
            pc_shortcut_uri=''
    ):
        """
        更新企业已添加的快捷方式
        ISV对已经添加其应用快捷方式的企业做快捷方式做更新
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43269

        :param app_id: 应用ID
        :param biz_no: 系统交互唯一业务单号
        :param shortcut_uri: 快捷方式跳转地址(移动端地址-默认地址)
        :param icon: 图标Url
        :param name: 快捷方式名称
        :param pc_shortcut_uri: PC端快捷方式跳转地址
        """
        return self._top_request(
            "dingtalk.oapi.workbench.shortcut.update",
            {
                "app_id": app_id,
                "biz_no": biz_no,
                "shortcut_uri": shortcut_uri,
                "icon": icon,
                "name": name,
                "pc_shortcut_uri": pc_shortcut_uri
            }
        )

    def dingtalk_oapi_report_comment_list(
            self,
            report_id,
            offset='0',
            size='20'
    ):
        """
        获取日志评论详情
        分页获取评论详情，包括评论人userid、评论内容、评论时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43545

        :param report_id: 日志id
        :param offset: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值，默认值为0
        :param size: 分页参数，每页大小，最多传20，默认值为20
        """
        return self._top_request(
            "dingtalk.oapi.report.comment.list",
            {
                "report_id": report_id,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_report_receiver_list(
            self,
            report_id,
            offset='0',
            size='100'
    ):
        """
        获取日志分享人员列表
        获取日志的分享人员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43546

        :param report_id: 日志id
        :param offset: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值，默认值为0
        :param size: 分页参数，每页大小，最多传100，默认值为100
        """
        return self._top_request(
            "dingtalk.oapi.report.receiver.list",
            {
                "report_id": report_id,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_report_statistics_listbytype(
            self,
            report_id,
            type,
            offset='0',
            size='100'
    ):
        """
        根据类型获取日志相关人员列表
        分页获取日志相关人员列表，包括已读人员列表、评论人员列表、点赞人员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43547

        :param report_id: 日志id
        :param type: 查询类型 0:已读人员列表 1:评论人员列表 2:点赞人员列表
        :param offset: 分页查询的游标，最开始传0，后续传返回参数中的next_cursor值，默认值为0
        :param size: 分页参数，每页大小，最多传100，默认值为100
        """
        return self._top_request(
            "dingtalk.oapi.report.statistics.listbytype",
            {
                "report_id": report_id,
                "type": type,
                "offset": offset,
                "size": size
            }
        )

    def dingtalk_oapi_message_corpconversation_recall(
            self,
            agent_id,
            msg_task_id
    ):
        """
        撤回工作通知消息
        根据发送工作通知消息的taskId进行消息撤回操作
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43694

        :param agent_id: 发送工作通知的微应用agentId
        :param msg_task_id: 发送工作通知返回的taskId
        """
        return self._top_request(
            "dingtalk.oapi.message.corpconversation.recall",
            {
                "agent_id": agent_id,
                "msg_task_id": msg_task_id
            }
        )

    def dingtalk_oapi_processinstance_cspace_info(
            self,
            user_id,
            agent_id=''
    ):
        """
        查询审批钉盘空间信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43803

        :param user_id: 用户id
        :param agent_id: 企业应用标识(ISV调用必须设置)
        """
        return self._top_request(
            "dingtalk.oapi.processinstance.cspace.info",
            {
                "user_id": user_id,
                "agent_id": agent_id
            }
        )

    def dingtalk_oapi_process_save(
            self,
            save_process_request=None
    ):
        """
        保存审批模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43900

        :param save_process_request: 入参
        """
        return self._top_request(
            "dingtalk.oapi.process.save",
            {
                "saveProcessRequest": save_process_request
            }
        )

    def dingtalk_oapi_process_form_condition_list(
            self,
            process_code,
            agentid=''
    ):
        """
        查询已设置为条件的表单组件
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43904

        :param process_code: 审批模板id
        :param agentid: 应用id
        """
        return self._top_request(
            "dingtalk.oapi.process.form.condition.list",
            {
                "request": {
                    "process_code": process_code,
                    "agentid": agentid
                }
            }
        )

    def dingtalk_oapi_robot_message_getpushid(
            self
    ):
        """
        ISV获取机器人消息批量推送Id
        颁发给isv机器人消息批量推送id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44051

        """
        return self._top_request(
            "dingtalk.oapi.robot.message.getpushid"
        )

    def dingtalk_oapi_robot_message_statistics_listbypushid(
            self,
            page_size,
            page,
            push_id,
            conversation_ids
    ):
        """
        pushid维度的机器人消息批量推送统计数据获取接口
        ISV通过该接口可以获取pushid维度的机器人消息批量推送统计数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44078

        :param page_size: 分页大小
        :param page: 当前页码
        :param push_id: 机器人消息推送Id
        :param conversation_ids: 群Id列表
        """
        return self._top_request(
            "dingtalk.oapi.robot.message.statistics.listbypushid",
            {
                "page_size": page_size,
                "page": page,
                "push_id": push_id,
                "conversation_ids": conversation_ids
            }
        )

    def dingtalk_oapi_robot_message_statistics_list(
            self,
            page_size,
            page,
            push_ids
    ):
        """
        获取机器人消息批量推送统计数据
        ISV通过该接口可以获取机器人消息批量推送统计数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44079

        :param page_size: 分页大小
        :param page: 当前页码
        :param push_ids: 机器人消息推送Id列表
        """
        return self._top_request(
            "dingtalk.oapi.robot.message.statistics.list",
            {
                "page_size": page_size,
                "page": page,
                "push_ids": push_ids
            }
        )

    def dingtalk_oapi_robot_message_statistics_listbyconversationid(
            self,
            push_id,
            conversation_ids,
            page_size,
            page,
            read_status=''
    ):
        """
        conversionid维度的机器人消息批量推送统计数据获取接口
        ISV通过该接口可以获取conversationid维度的机器人消息批量推送统计数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44094

        :param push_id: 机器人消息推送Id
        :param conversation_ids: 群Id列表
        :param page_size: 分页大小
        :param page: 当前页码
        :param read_status: 已读状态
        """
        return self._top_request(
            "dingtalk.oapi.robot.message.statistics.listbyconversationid",
            {
                "push_id": push_id,
                "conversation_ids": conversation_ids,
                "page_size": page_size,
                "page": page,
                "read_status": read_status
            }
        )

    def dingtalk_oapi_smartdevice_atmachine_get_by_deptid(
            self,
            param
    ):
        """
        查询部门智能考勤机列表
        查询考勤机设备列表，可选择查询部门已关联的考勤机设备列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44096

        :param param: 查询智能考勤机列表参数模型
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.atmachine.get_by_deptid",
            {
                "param": param
            }
        )

    def dingtalk_oapi_smartdevice_atmachine_get_by_userid(
            self,
            param=None
    ):
        """
        查询员工智能考勤机列表
        允许isv查询员工智能考勤机列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44097

        :param param: 查询智能考勤机列表参数模型
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.atmachine.get_by_userid",
            {
                "param": param
            }
        )

    def dingtalk_oapi_smartdevice_atmachine_user_update(
            self,
            param=None
    ):
        """
        变更智能考勤机员工
        提供变更智能考勤机员工功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44098

        :param param: 变更智能考勤机员工参数模型
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.atmachine.user.update",
            {
                "param": param
            }
        )

    def dingtalk_oapi_relation_remark_modify(
            self,
            markers,
            markees
    ):
        """
        修改备注名
        给指定企业，批量修改备注名。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44214

        :param markers: 修改者的userid
        :param markees: 系统自动生成
        """
        return self._top_request(
            "dingtalk.oapi.relation.remark.modify",
            {
                "markers": markers,
                "markees": markees
            }
        )

    def dingtalk_oapi_process_workrecord_create(
            self,
            process_code,
            originator_user_id,
            form_component_values,
            url,
            agentid='',
            title=''
    ):
        """
        发起不带流程的审批实例
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44261

        :param process_code: 审批模板唯一码
        :param originator_user_id: 审批发起人
        :param form_component_values: 表单参数列表
        :param url: 实例跳转链接
        :param agentid: 应用id
        :param title: 实例标题
        """
        return self._top_request(
            "dingtalk.oapi.process.workrecord.create",
            {
                "request": {
                    "process_code": process_code,
                    "originator_user_id": originator_user_id,
                    "form_component_values": form_component_values,
                    "url": url,
                    "agentid": agentid,
                    "title": title
                }
            }
        )

    def dingtalk_oapi_process_workrecord_update(
            self,
            process_instance_id,
            status,
            result,
            agentid=''
    ):
        """
        同步待办实例状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44262

        :param process_instance_id: 实例id
        :param status: 实例状态，分为COMPLETED, TERMINATED
        :param result: 实例结果, 如果实例状态是COMPLETED，需要设置result，分为agree和refuse
        :param agentid: 应用id
        """
        return self._top_request(
            "dingtalk.oapi.process.workrecord.update",
            {
                "request": {
                    "process_instance_id": process_instance_id,
                    "status": status,
                    "result": result,
                    "agentid": agentid
                }
            }
        )

    def dingtalk_oapi_process_workrecord_task_create(
            self,
            process_instance_id,
            tasks,
            agentid='',
            activity_id=''
    ):
        """
        创建待办任务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44263

        :param process_instance_id: 实例id
        :param tasks: 任务列表
        :param agentid: 应用id
        :param activity_id: 节点id
        """
        return self._top_request(
            "dingtalk.oapi.process.workrecord.task.create",
            {
                "request": {
                    "process_instance_id": process_instance_id,
                    "tasks": tasks,
                    "agentid": agentid,
                    "activity_id": activity_id
                }
            }
        )

    def dingtalk_oapi_process_workrecord_task_update(
            self,
            process_instance_id,
            tasks,
            agentid=''
    ):
        """
        更新待办任务状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44264

        :param process_instance_id: 实例id
        :param tasks: 任务列表
        :param agentid: 应用id
        """
        return self._top_request(
            "dingtalk.oapi.process.workrecord.task.update",
            {
                "request": {
                    "process_instance_id": process_instance_id,
                    "tasks": tasks,
                    "agentid": agentid
                }
            }
        )

    def dingtalk_oapi_process_workrecord_forward_create(
            self,
            process_instance_id,
            userid_list,
            agentid=''
    ):
        """
        创建实例抄送人
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44265

        :param process_instance_id: 实例id
        :param userid_list: 抄送人id列表
        :param agentid: 应用id
        """
        return self._top_request(
            "dingtalk.oapi.process.workrecord.forward.create",
            {
                "request": {
                    "process_instance_id": process_instance_id,
                    "userid_list": userid_list,
                    "agentid": agentid
                }
            }
        )

    def dingtalk_oapi_process_workrecord_taskgroup_cancel(
            self,
            process_instance_id,
            activity_id,
            agentid=''
    ):
        """
        批量取消任务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44317

        :param process_instance_id: 实例id
        :param activity_id: 任务组id
        :param agentid: 应用id
        """
        return self._top_request(
            "dingtalk.oapi.process.workrecord.taskgroup.cancel",
            {
                "request": {
                    "process_instance_id": process_instance_id,
                    "activity_id": activity_id,
                    "agentid": agentid
                }
            }
        )

    def dingtalk_oapi_process_delete(
            self,
            process_code,
            agentid=''
    ):
        """
        删除模板
        删除创建的审批模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44349

        :param process_code: 流程code
        :param agentid: 微应用agentId，ISV必填
        """
        return self._top_request(
            "dingtalk.oapi.process.delete",
            {
                "request": {
                    "process_code": process_code,
                    "agentid": agentid
                }
            }
        )

    def dingtalk_oapi_smartdevice_applyoutid(
            self,
            dev_serv_id,
            sn
    ):
        """
        向钉钉申请智能硬件设备外部ID
        给硬件设备分配设备id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44397

        :param dev_serv_id: 设备类型id
        :param sn: 设备序列号
        """
        return self._top_request(
            "dingtalk.oapi.smartdevice.applyoutid",
            {
                "dev_serv_id": dev_serv_id,
                "sn": sn
            }
        )

    def dingtalk_oapi_material_article_list(
            self,
            unionid,
            page_size,
            page_start
    ):
        """
        查询文章列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44535

        :param unionid: 服务号的unionid
        :param page_size: 每页条数
        :param page_start: 页码
        """
        return self._top_request(
            "dingtalk.oapi.material.article.list",
            {
                "unionid": unionid,
                "page_size": page_size,
                "page_start": page_start
            }
        )

    def dingtalk_oapi_material_article_delete(
            self,
            unionid,
            article_id
    ):
        """
        删除文章
        服务窗删除文章（使得文章页不可访问）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44541

        :param unionid: 服务号的unionid
        :param article_id: 文章id
        """
        return self._top_request(
            "dingtalk.oapi.material.article.delete",
            {
                "unionid": unionid,
                "article_id": article_id
            }
        )

    def dingtalk_oapi_material_article_publish(
            self,
            unionid,
            article_id
    ):
        """
        文章发布接口
        服务窗发布文章成页面
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44542

        :param unionid: 服务号的unionid
        :param article_id: 文章id
        """
        return self._top_request(
            "dingtalk.oapi.material.article.publish",
            {
                "unionid": unionid,
                "article_id": article_id
            }
        )

    def dingtalk_oapi_material_article_get(
            self,
            unionid,
            article_id
    ):
        """
        获取文章
        获取单篇文章信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44543

        :param unionid: 服务号的unionid
        :param article_id: 文章id
        """
        return self._top_request(
            "dingtalk.oapi.material.article.get",
            {
                "unionid": unionid,
                "article_id": article_id
            }
        )

    def dingtalk_oapi_material_article_update(
            self,
            article,
            unionid
    ):
        """
        保存文章详情
        服务窗用于修改文章详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44544

        :param article: 文章对象
        :param unionid: 服务号的unionid
        """
        return self._top_request(
            "dingtalk.oapi.material.article.update",
            {
                "article": article,
                "unionid": unionid
            }
        )

    def dingtalk_oapi_material_article_add(
            self,
            article,
            unionid
    ):
        """
        新增文章
        服务窗新建文章使用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=44545

        :param article: 文章参数对象
        :param unionid: 服务号的unionid
        """
        return self._top_request(
            "dingtalk.oapi.material.article.add",
            {
                "article": article,
                "unionid": unionid
            }
        )


class TbYongHu(DingTalkBaseAPI):
    """
    用户API
    """

    def taobao_appstore_subscribe_get(
            self,
            nick
    ):
        """
        查询appstore应用订购关系
        查询appstore应用订购关系(对于新上架的多版本应用，建议使用taobao.vas.subscribe.get)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=285

        :param nick: 用户昵称
        """
        return self._top_request(
            "taobao.appstore.subscribe.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["user_subscribe"]
        )

    def taobao_user_buyer_get(
            self,
            fields
    ):
        """
        查询买家信息API
        查询买家信息API，只能买家类应用调用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21348

        :param fields: 只返回nick, avatar参数
        """
        return self._top_request(
            "taobao.user.buyer.get",
            {
                "fields": fields
            },
            result_processor=lambda x: x["user"]
        )

    def taobao_user_seller_get(
            self,
            fields
    ):
        """
        查询卖家用户信息
        查询卖家用户信息（只能查询有店铺的用户） 只能卖家类应用调用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21349

        :param fields: 需要返回的字段列表，可选值为返回示例值中的可以看到的字段
        """
        return self._top_request(
            "taobao.user.seller.get",
            {
                "fields": fields
            },
            result_processor=lambda x: x["user"]
        )

    def taobao_mixnick_get(
            self,
            nick
    ):
        """
        将用户名转换成混淆后的值
        将用户名转换成混淆后的值，为店铺前台模块授权改造提供数据订正接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21428

        :param nick: 用户名
        """
        return self._top_request(
            "taobao.mixnick.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["nick"]
        )

    def taobao_opensecurity_uid_get(
            self,
            tb_user_id
    ):
        """
        淘宝open security uid 获取接口
        根据明文 taobao user id 换取 app的 open_uid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24672

        :param tb_user_id: 淘宝用户ID
        """
        return self._top_request(
            "taobao.opensecurity.uid.get",
            {
                "tb_user_id": tb_user_id
            },
            result_processor=lambda x: x["open_uid"]
        )

    def taobao_opensecurity_isv_uid_get(
            self,
            open_uid
    ):
        """
        获取open security uid for isv
        根据 open_uid 获取 open_uid_isv 用于同一个 isv的多个app间数据关联
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24673

        :param open_uid: 基于Appkey生成的open security淘宝用户Id
        """
        return self._top_request(
            "taobao.opensecurity.isv.uid.get",
            {
                "open_uid": open_uid
            },
            result_processor=lambda x: x["open_uid_isv"]
        )

    def taobao_open_account_delete(
            self,
            open_account_ids='',
            isv_account_ids=''
    ):
        """
        OpenAccount删除数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24819

        :param open_account_ids: Open Account的id列表
        :param isv_account_ids: ISV自己账号的id列表
        """
        return self._top_request(
            "taobao.open.account.delete",
            {
                "open_account_ids": open_account_ids,
                "isv_account_ids": isv_account_ids
            },
            result_processor=lambda x: x["datas"]
        )

    def taobao_open_account_update(
            self,
            param_list=None
    ):
        """
        Open Account数据更新
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24820

        :param param_list: Open Account
        """
        return self._top_request(
            "taobao.open.account.update",
            {
                "param_list": param_list
            },
            result_processor=lambda x: x["datas"]
        )

    def taobao_open_account_create(
            self,
            param_list=None
    ):
        """
        Open Account导入数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24821

        :param param_list: Open Account的列表
        """
        return self._top_request(
            "taobao.open.account.create",
            {
                "param_list": param_list
            },
            result_processor=lambda x: x["datas"]
        )

    def taobao_open_account_list(
            self,
            open_account_ids='',
            isv_account_ids=''
    ):
        """
        OpenAccount账号信息查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24848

        :param open_account_ids: Open Account的id列表, 每次最多查询 20 个帐户
        :param isv_account_ids: ISV自己账号的id列表，isvAccountId和openAccountId二选一必填, 每次最多查询 20 个帐户
        """
        return self._top_request(
            "taobao.open.account.list",
            {
                "open_account_ids": open_account_ids,
                "isv_account_ids": isv_account_ids
            },
            result_processor=lambda x: x["datas"]
        )

    def taobao_open_account_search(
            self,
            query
    ):
        """
        open account数据搜索
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25157

        :param query: 基于阿里云OpenSearch服务，openSearch查询语法:https://help.aliyun.com/document_detail/29157.html，搜索服务能够基于id，loginId，mobile，email，isvAccountId，display_name,create_app_key,做搜索查询，示例中mobile可以基于模糊搜素，搜索135的账号，该搜索是分页返回，start表示开始行，hit表示一页返回值，最大500
        """
        return self._top_request(
            "taobao.open.account.search",
            {
                "query": query
            },
            result_processor=lambda x: x["data"]
        )

    def taobao_open_account_token_validate(
            self,
            param_token
    ):
        """
        open account token验证
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25270

        :param param_token: token
        """
        return self._top_request(
            "taobao.open.account.token.validate",
            {
                "param_token": param_token
            },
            result_processor=lambda x: x["data"]
        )

    def taobao_open_account_token_apply(
            self,
            token_timestamp='',
            open_account_id='',
            isv_account_id='',
            uuid='',
            login_state_expire_in='',
            ext=''
    ):
        """
        申请免登Open Account Token
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25271

        :param token_timestamp: 时间戳单位是毫秒
        :param open_account_id: open account id
        :param isv_account_id: isv自己账号的唯一id
        :param uuid: 用于防重放的唯一id
        :param login_state_expire_in: ISV APP的登录态时长单位是秒
        :param ext: 用于透传一些业务附加参数
        """
        return self._top_request(
            "taobao.open.account.token.apply",
            {
                "token_timestamp": token_timestamp,
                "open_account_id": open_account_id,
                "isv_account_id": isv_account_id,
                "uuid": uuid,
                "login_state_expire_in": login_state_expire_in,
                "ext": ext
            },
            result_processor=lambda x: x["data"]
        )

    def taobao_open_sms_sendvercode(
            self,
            send_ver_code_request
    ):
        """
        发送短信验证码
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25596

        :param send_ver_code_request: 发送验证码请求
        """
        return self._top_request(
            "taobao.open.sms.sendvercode",
            {
                "send_ver_code_request": send_ver_code_request
            }
        )

    def taobao_open_sms_checkvercode(
            self,
            check_ver_code_request
    ):
        """
        验证短信验证码
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25597

        :param check_ver_code_request: 验证验证码
        """
        return self._top_request(
            "taobao.open.sms.checkvercode",
            {
                "check_ver_code_request": check_ver_code_request
            }
        )

    def taobao_open_sms_sendmsg(
            self,
            send_message_request
    ):
        """
        发送短信
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25598

        :param send_message_request: 发送短信请求
        """
        return self._top_request(
            "taobao.open.sms.sendmsg",
            {
                "send_message_request": send_message_request
            }
        )

    def taobao_open_account_index_find(
            self,
            index_type='0',
            index_value=''
    ):
        """
        Open Account索引查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25869

        :param index_type: int MOBILE         = 1;int EMAIL          = 2;int ISV_ACCOUNT_ID = 3;int LOGIN_ID       = 4;int OPEN_ID        = 5;
        :param index_value: 具体值，当索引类型是 OPEN_ID 是，格式为 oauthPlatform|openId，即使用竖线分隔的组合值
        """
        return self._top_request(
            "taobao.open.account.index.find",
            {
                "index_type": index_type,
                "index_value": index_value
            }
        )

    def taobao_open_sms_rmdelaymsg(
            self,
            param_remove_delay_message_request=None
    ):
        """
        删除延迟消息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25966

        :param param_remove_delay_message_request: 删除延迟消息
        """
        return self._top_request(
            "taobao.open.sms.rmdelaymsg",
            {
                "param_remove_delay_message_request": param_remove_delay_message_request
            }
        )

    def taobao_data_wifi_put(
            self,
            wifi_data=''
    ):
        """
        第三方wifi设备商上传数据
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        阿里巴巴和第三方WIFI 数据提供商之间的数据交换
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25969

        :param wifi_data: wifi_data 正式的定义如下：   一、WIFI_DATA 是一个JSON 数组，其中每一个元素的格式是SENSOR_DATA。   二、SENSOR_DATA 是一个JSON 数组，其中：  	 a.第一个元素是版本号，当前有效的版本号是数字2。   	 b.第二个元素是探针的MAC 地址，在上述例子中是60:CD:A9:00:AB:7E。   	 c.第三个元素是探针探测到的移动设备MAC 地址，格式为USER_MAC_LIST。   	   1. USER_MAC_LIST 是一个JSON 数组，其中每一个元素的格式是USER_MAC_DATA。   	   2. USER_MAC_DATA 是一个JSON 数组，其中。   	 	  i.第一个元素是用户的MAC。   	 	 ii.第二个元素是探测到的信号强度，如果该值无法得到，需要填充默认值1。   	 	iii.第三个元素是用户的手机号码。如果该值无法得到，需要填充默认值0。  	 d.第四个元素是一个UNIX 时间戳。在上述例子中是1444398542000。需要注意 是从1970 年1 月1 日零点(UTC)到当前的毫秒数，不是秒数。
        """
        return self._top_request(
            "taobao.data.wifi.put",
            {
                "wifi_data": wifi_data
            },
            result_processor=lambda x: x["is_success"]
        )

    def alibaba_aliqin_flow_wallet_check_balance(
            self,
            grade_id
    ):
        """
        商家预存余额检查
        检查商家CRM预存余额是否足够进行活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25978

        :param grade_id: 检查金额档位id
        """
        return self._top_request(
            "alibaba.aliqin.flow.wallet.check.balance",
            {
                "grade_id": grade_id
            },
            result_processor=lambda x: x["value"]
        )

    def taobao_open_sms_batchsendmsg(
            self,
            params
    ):
        """
        批量发送短信
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26027

        :param params: 参数列表
        """
        return self._top_request(
            "taobao.open.sms.batchsendmsg",
            {
                "params": params
            }
        )

    def taobao_user_avatar_get(
            self,
            nick
    ):
        """
        淘宝用户头像查询
        根据混淆nick查询用户头像
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26303

        :param nick: 混淆nick
        """
        return self._top_request(
            "taobao.user.avatar.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["avatar"]
        )

    def tmall_service_settleadjustment_modify(
            self,
            param_settle_adjustment_request
    ):
        """
        修改结算调整单
        提供给服务商在对结算有异议时，发起结算调整单。
        通过说明调整单ID，调整费用值，调整原因进行结算调整单修改。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27436

        :param param_settle_adjustment_request: 结算调整单父节点
        """
        return self._top_request(
            "tmall.service.settleadjustment.modify",
            {
                "param_settle_adjustment_request": param_settle_adjustment_request
            }
        )

    def taobao_rdc_aligenius_account_validate(
            self
    ):
        """
        AG商家账号校验
        提供应对接AG的erp系统查询其旗下的商家是否为AG商家
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31159

        """
        return self._top_request(
            "taobao.rdc.aligenius.account.validate"
        )

    def tmall_fantasy_kindle_crowd(
            self
    ):
        """
        kindle人群信息返回
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        根据接口得到的当前用户信息，返回由算法计算得出的当前用户对应的人群信息（人群信息以1~6的数字标返回）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31237

        """
        return self._top_request(
            "tmall.fantasy.kindle.crowd"
        )

    def alibaba_interact_ui_video(
            self,
            unnamed=''
    ):
        """
        视频播放
        Weex页面播放视频
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31570

        :param unnamed: 仅作鉴权使用，没有实际数据传输
        """
        return self._top_request(
            "alibaba.interact.ui.video",
            {
                "unnamed": unnamed
            },
            result_processor=lambda x: x["unnamed"]
        )

    def taobao_messageaccount_messsage_mass_send(
            self,
            param=None
    ):
        """
        消息号开放-消息群发
        外部 isv 调用该进口来进行消息号消息的群发
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31908

        :param param: 参数
        """
        return self._top_request(
            "taobao.messageaccount.messsage.mass.send",
            {
                "param": param
            }
        )

    def taobao_messageaccount_messsage_reply(
            self,
            param0
    ):
        """
        消息号下行回复接口
        外部 isv 调用该进口来进行消息号消息的回复
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31931

        :param param0: 入参
        """
        return self._top_request(
            "taobao.messageaccount.messsage.reply",
            {
                "param0": param0
            }
        )

    def taobao_messageaccount_messsage_normal_send(
            self,
            param=None
    ):
        """
        下行普通消息
        消息号下行单个普通消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35382

        :param param: 下行普通消息
        """
        return self._top_request(
            "taobao.messageaccount.messsage.normal.send",
            {
                "param": param
            },
            result_processor=lambda x: x["model"]
        )

    def taobao_miniapp_messsage_normal_send(
            self,
            param=None
    ):
        """
        轻店铺下行普通消息给用户
        小程序下行单个普通消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38414

        :param param: 普通消息结构
        """
        return self._top_request(
            "taobao.miniapp.messsage.normal.send",
            {
                "param": param
            },
            result_processor=lambda x: x["model"]
        )

    def taobao_miniapp_messsage_reply(
            self,
            param=None
    ):
        """
        轻店铺下行回复接口
        外部 isv 调用该进口来进行轻店铺消息的回复
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38415

        :param param: 入参
        """
        return self._top_request(
            "taobao.miniapp.messsage.reply",
            {
                "param": param
            }
        )

    def wdk_hema_orderlist_get(
            self,
            name
    ):
        """
        盒马小程序测试服务端api01
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        盒马小程序服务端测试api01
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40389

        :param name: 名称
        """
        return self._top_request(
            "wdk.hema.orderlist.get",
            {
                "name": name
            }
        )

    def taobao_miniapp_userInfo_get(
            self
    ):
        """
        用户开放信息获取
        获取用户的 openId，snsNick（如果用户设置过的话），和加密头像链接
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41744

        """
        return self._top_request(
            "taobao.miniapp.userInfo.get"
        )

    def taobao_mixnick_change(
            self,
            src_mix_nick,
            src_appkey
    ):
        """
        新旧mixnick互转
        如果采用老的Appkey获取MixNick A’， 发现DB里 new MixNick为null，则使用新的Appkey调API来换取A’’；如果采用新的Appkey获取A’’，发现DB里不存在当前A’’ 的记录时，先用老Appkey调API得到A’ 查询是否存在A用户的记录，如果已经存在，则关联，否则新增一条MixNick为null的新用户记录。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24410

        :param src_mix_nick: 输入的混淆nick
        :param src_appkey: 输入的appkey
        """
        return self._top_request(
            "taobao.mixnick.change",
            {
                "src_mix_nick": src_mix_nick,
                "src_appkey": src_appkey
            }
        )


class TbLeiMu(DingTalkBaseAPI):
    """
    类目API
    """

    def taobao_itempropvalues_get(
            self,
            fields,
            cid,
            pvs='',
            type='1',
            attr_keys=''
    ):
        """
        获取标准类目属性值
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=13

        :param fields: 需要返回的字段。目前支持有：cid,pid,prop_name,vid,name,name_alias,status,sort_order
        :param cid: 叶子类目ID ,通过taobao.itemcats.get获得叶子类目ID
        :param pvs: 属性和属性值 id串，格式例如(pid1;pid2)或(pid1:vid1;pid2:vid2)或(pid1;pid2:vid2)
        :param type: 获取类目的类型：1代表集市、2代表天猫
        :param attr_keys: 属性的Key，支持多条，以“,”分隔
        """
        return self._top_request(
            "taobao.itempropvalues.get",
            {
                "fields": fields,
                "cid": cid,
                "pvs": pvs,
                "type": type,
                "attr_keys": attr_keys
            }
        )

    def taobao_itemprops_get(
            self,
            cid,
            fields='pid,name,must,multi,prop_values',
            pid='',
            parent_pid='',
            is_key_prop='',
            is_sale_prop='',
            is_color_prop='',
            is_enum_prop='',
            is_input_prop='',
            is_item_prop='',
            child_path='',
            type='1',
            attr_keys=''
    ):
        """
        获取标准商品类目属性
        通过设置必要的参数，来获取商品后台标准类目属性，以及这些属性里面详细的属性值prop_values。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=121

        :param cid: 叶子类目ID，如果只传cid，则只返回一级属性,通过taobao.itemcats.get获得叶子类目ID
        :param fields: 需要返回的字段列表，见：ItemProp，默认返回：pid, name, must, multi, prop_values
        :param pid: 属性id (取类目属性时，传pid，不用同时传PID和parent_pid)
        :param parent_pid: 父属性ID
        :param is_key_prop: 是否关键属性。可选值:true(是),false(否)
        :param is_sale_prop: 是否销售属性。可选值:true(是),false(否)
        :param is_color_prop: 是否颜色属性。可选值:true(是),false(否) (删除的属性不会匹配和返回这个条件)
        :param is_enum_prop: 是否枚举属性。可选值:true(是),false(否) (删除的属性不会匹配和返回这个条件)。如果返回true，属性值是下拉框选择输入，如果返回false，属性值是用户自行手工输入。
        :param is_input_prop: 在is_enum_prop是true的前提下，是否是卖家可以自行输入的属性（注：如果is_enum_prop返回false，该参数统一返回false）。可选值:true(是),false(否) (删除的属性不会匹配和返回这个条件)
        :param is_item_prop: 是否商品属性，这个属性只能放于发布商品时使用。可选值:true(是),false(否)
        :param child_path: 类目子属性路径,由该子属性上层的类目属性和类目属性值组成,格式pid:vid;pid:vid.取类目子属性需要传child_path,cid
        :param type: 获取类目的类型：1代表集市、2代表天猫
        :param attr_keys: 属性的Key，支持多条，以“,”分隔
        """
        return self._top_request(
            "taobao.itemprops.get",
            {
                "cid": cid,
                "fields": fields,
                "pid": pid,
                "parent_pid": parent_pid,
                "is_key_prop": is_key_prop,
                "is_sale_prop": is_sale_prop,
                "is_color_prop": is_color_prop,
                "is_enum_prop": is_enum_prop,
                "is_input_prop": is_input_prop,
                "is_item_prop": is_item_prop,
                "child_path": child_path,
                "type": type,
                "attr_keys": attr_keys
            }
        )

    def taobao_itemcats_get(
            self,
            cids='',
            fields='cid,parent_cid,name,is_parent',
            parent_cid=''
    ):
        """
        获取后台供卖家发布商品的标准商品类目
        获取后台供卖家发布商品的标准商品类目。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=122

        :param cids: 商品所属类目ID列表，用半角逗号(,)分隔 例如:(18957,19562,) (cids、parent_cid至少传一个)
        :param fields: 需要返回的字段列表，见ItemCat，默认返回：cid,parent_cid,name,is_parent；增量类目信息,根据fields传入的参数返回相应的结果。 features字段： 1、如果存在attr_key=freeze表示该类目被冻结了，attr_value=0,5，value可能存在2个值（也可能只有1个），用逗号分割，0表示禁编辑，5表示禁止发布
        :param parent_cid: 父商品类目 id，0表示根节点, 传输该参数返回所有子类目。 (cids、parent_cid至少传一个)
        """
        return self._top_request(
            "taobao.itemcats.get",
            {
                "cids": cids,
                "fields": fields,
                "parent_cid": parent_cid
            }
        )

    def taobao_itemcats_authorize_get(
            self,
            fields
    ):
        """
        查询商家被授权品牌列表和类目列表
        查询B商家被授权品牌列表、类目列表和 c 商家新品类目列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=161

        :param fields: 需要返回的字段。目前支持有：<br/>brand.vid, brand.name, <br/>item_cat.cid, item_cat.name, item_cat.status,item_cat.sort_order,item_cat.parent_cid,item_cat.is_parent,<br/>xinpin_item_cat.cid, <br/>xinpin_item_cat.name, <br/>xinpin_item_cat.status,<br/>xinpin_item_cat.sort_order,<br/>xinpin_item_cat.parent_cid,<br/>xinpin_item_cat.is_parent
        """
        return self._top_request(
            "taobao.itemcats.authorize.get",
            {
                "fields": fields
            },
            result_processor=lambda x: x["seller_authorize"]
        )

    def alibaba_wholesale_category_get(
            self
    ):
        """
        获取类目信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24365

        """
        return self._top_request(
            "alibaba.wholesale.category.get",
            result_processor=lambda x: x["wholesale_category_result"]
        )


class TbShangPin(DingTalkBaseAPI):
    """
    商品API
    """

    def taobao_product_get(
            self,
            fields,
            product_id='',
            cid='',
            props=''
    ):
        """
        获取一个产品的信息
        两种方式查看一个产品详细信息: <br/>传入product_id来查询 <br/>传入cid和props来查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=4

        :param fields: 需返回的字段列表.可选值:Product数据结构中的所有字段;多个字段之间用','分隔.
        :param product_id: Product的id.两种方式来查看一个产品:1.传入product_id来查询 2.传入cid和props来查询
        :param cid: 商品类目id.调用taobao.itemcats.get获取;必须是叶子类目id,如果没有传product_id,那么cid和props必须要传.
        :param props: 比如:诺基亚N73这个产品的关键属性列表就是:品牌:诺基亚;型号:N73,对应的PV值就是10005:10027;10006:29729.
        """
        return self._top_request(
            "taobao.product.get",
            {
                "fields": fields,
                "product_id": product_id,
                "cid": cid,
                "props": props
            },
            result_processor=lambda x: x["product"]
        )

    def taobao_products_search(
            self,
            fields,
            q='',
            cid='',
            props='',
            status='',
            page_no='',
            page_size='',
            vertical_market='',
            customer_props='',
            suite_items_str='',
            barcode_str='',
            market_id=''
    ):
        """
        搜索产品信息
        只有天猫商家发布商品时才需要用到，并非商品搜索api，当前暂不提供商品搜索api。<br/>二种方式搜索所有产品信息(二种至少传一种): <br/><br/>传入关键字q搜索<br/><br/>传入cid和props搜索<br/><br/>返回值支持:product_id,name,pic_path,cid,props,price,tsc<br/><br/>当用户指定了cid并且cid为垂直市场（3C电器城、鞋城）的类目id时，默认只返回小二确认<br/>的产品。如果用户没有指定cid，或cid为普通的类目，默认返回商家确认或小二确认的产<br/>品。如果用户自定了status字段，以指定的status类型为准。<br/><br/>新增一：<br/>   传入suite_items_str 按规格搜索套装产品。<br/>   返回字段增加suite_items_str,is_suite_effecitve支持。<br/><br/>新增二：<br/>   传入barcode_str 支持按条码搜索产品。<br/>   返回字段增加barcode_str。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=5

        :param fields: 需返回的字段列表.可选值:Product数据结构中的以下字段:product_id,name,pic_url,cid,props,price,tsc;多个字段之间用','分隔.新增字段status(product的当前状态)
        :param q: 搜索的关键词是用来搜索产品的title.　注:q,cid和props至少传入一个
        :param cid: 商品类目ID.调用taobao.itemcats.get获取.
        :param props: 属性,属性值的组合.格式:pid:vid;pid:vid;调用taobao.itemprops.get获取类目属性pid <br/>,再用taobao.itempropvalues.get取得vid.
        :param status: 想要获取的产品的状态列表，支持多个状态并列获取，多个状态之间用','分隔，最多同时指定5种状态。例如，只获取小二确认的spu传入'3',只要商家确认的传入'0'，既要小二确认又要商家确认的传入'0,3'。目前只支持者两种类型的状态搜索，输入其他状态无效。
        :param page_no: 页码.传入值为1代表第一页,传入值为2代表第二页,依此类推.默认返回的数据是从第一页开始.
        :param page_size: 每页条数.每页返回最多返回100条,默认值为40.
        :param vertical_market: 传入值为：3表示3C表示3C垂直市场产品，4表示鞋城垂直市场产品，8表示网游垂直市场产品。一次只能指定一种垂直市场类型
        :param customer_props: 用户自定义关键属性,结构：pid1:value1;pid2:value2，如果有型号，系列等子属性用: 隔开 例如：“20000:优衣库:型号:001;632501:1234”，表示“品牌:优衣库:型号:001;货号:1234”
        :param suite_items_str: 按关联产品规格specs搜索套装产品
        :param barcode_str: 按条码搜索产品信息,多个逗号隔开，不支持条码为全零的方式
        :param market_id: 市场ID，1为取C2C市场的产品信息， 2为取B2C市场的产品信息。  不填写此值则默认取C2C的产品信息。
        """
        return self._top_request(
            "taobao.products.search",
            {
                "fields": fields,
                "q": q,
                "cid": cid,
                "props": props,
                "status": status,
                "page_no": page_no,
                "page_size": page_size,
                "vertical_market": vertical_market,
                "customer_props": customer_props,
                "suite_items_str": suite_items_str,
                "barcode_str": barcode_str,
                "market_id": market_id
            }
        )

    def taobao_product_add(
            self,
            cid,
            image,
            native_unkeyprops='',
            outer_id='',
            props='',
            binds='',
            sale_props='',
            customer_props='',
            price='',
            name='',
            desc='',
            major='true',
            market_time='',
            property_alias=''
    ):
        """
        上传一个产品，不包括产品非主图和属性图片
        获取类目ID，必需是叶子类目ID；调用taobao.itemcats.get.v2获取 <br/>传入关键属性,结构:pid:vid;pid:vid.调用taobao.itemprops.get.v2获取pid,<br/>调用taobao.itempropvalues.get获取vid;如果碰到用户自定义属性,请用customer_props.<br/>新增：套装产品发布，目前支持单件多个即 A*2 形式的套装
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=6

        :param cid: 商品类目ID.调用taobao.itemcats.get获取;注意:必须是叶子类目 id.
        :param image: 产品主图片.最大1M,目前仅支持GIF,JPG.
        :param native_unkeyprops: native_unkeyprops
        :param outer_id: 外部产品ID
        :param props: 关键属性 结构:pid:vid;pid:vid.调用taobao.itemprops.get获取pid,调用taobao.itempropvalues.get获取vid;如果碰到用户自定义属性,请用customer_props.
        :param binds: 非关键属性结构:pid:vid;pid:vid.<br>
                      非关键属性<font color=red>不包含</font>关键属性、销售属性、用户自定义属性、商品属性;
                      <br>调用taobao.itemprops.get获取pid,调用taobao.itempropvalues.get获取vid.<br><font color=red>注:支持最大长度为512字节</font>
        :param sale_props: 销售属性结构:pid:vid;pid:vid.调用taobao.itemprops.get获取is_sale_prop＝true的pid,调用taobao.itempropvalues.get获取vid.
        :param customer_props: 用户自定义属性,结构：pid1:value1;pid2:value2，如果有型号，系列等子属性用: 隔开 例如：“20000:优衣库:型号:001;632501:1234”，表示“品牌:优衣库:型号:001;货号:1234”
                               <br><font color=red>注：包含所有自定义属性的传入</font>
        :param price: 产品市场价.精确到2位小数;单位为元.如：200.07
        :param name: 产品名称,最大30个字符.
        :param desc: 产品描述.最大不超过25000个字符
        :param major: 是不是主图
        :param market_time: 上市时间。目前只支持鞋城类目传入此参数
        :param property_alias: 销售属性值别名。格式为pid1:vid1:alias1;pid1:vid2:alia2。只有少数销售属性值支持传入别名，比如颜色和尺寸
        """
        return self._top_request(
            "taobao.product.add",
            {
                "cid": cid,
                "image": image,
                "native_unkeyprops": native_unkeyprops,
                "outer_id": outer_id,
                "props": props,
                "binds": binds,
                "sale_props": sale_props,
                "customer_props": customer_props,
                "price": price,
                "name": name,
                "desc": desc,
                "major": major,
                "market_time": market_time,
                "property_alias": property_alias
            },
            result_processor=lambda x: x["product"]
        )

    def taobao_product_img_upload(
            self,
            product_id,
            image,
            id='',
            position='0',
            is_major=''
    ):
        """
        上传单张产品非主图，如果需要传多张，可调多次
        1.传入产品ID <br/>2.传入图片内容 <br/>注意：图片最大为500K,只支持JPG,GIF格式,如果需要传多张，可调多次
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=7

        :param product_id: 产品ID.Product的id
        :param image: 图片内容.图片最大为500K,只支持JPG,GIF格式.
        :param id: 产品图片ID.修改图片时需要传入
        :param position: 图片序号
        :param is_major: 是否将该图片设为主图.可选值:true,false;默认值:false.
        """
        return self._top_request(
            "taobao.product.img.upload",
            {
                "product_id": product_id,
                "image": image,
                "id": id,
                "position": position,
                "is_major": is_major
            },
            result_processor=lambda x: x["product_img"]
        )

    def taobao_product_propimg_upload(
            self,
            product_id,
            props,
            image,
            id='',
            position='0'
    ):
        """
        上传单张产品属性图片，如果需要传多张，可调多次
        传入产品ID <br/>传入props,目前仅支持颜色属性.调用taobao.itemprops.get.v2取得颜色属性pid,<br/>再用taobao.itempropvalues.get取得vid;格式:pid:vid,只能传入一个颜色pid:vid串; <br/>传入图片内容 <br/>注意：图片最大为2M,只支持JPG,GIF,如果需要传多张，可调多次
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=8

        :param product_id: 产品ID.Product的id
        :param props: 属性串.目前仅支持颜色属性.调用taobao.itemprops.get获取类目属性,取得颜色属性pid,再用taobao.itempropvalues.get取得vid;格式:pid:vid,只能传入一个颜色pid:vid串;
        :param image: 图片内容.图片最大为2M,只支持JPG,GIF.
        :param id: 产品属性图片ID
        :param position: 图片序号
        """
        return self._top_request(
            "taobao.product.propimg.upload",
            {
                "product_id": product_id,
                "props": props,
                "image": image,
                "id": id,
                "position": position
            },
            result_processor=lambda x: x["product_prop_img"]
        )

    def taobao_product_update(
            self,
            product_id,
            outer_id='',
            binds='',
            sale_props='',
            price='',
            desc='',
            image='',
            name='',
            major='true',
            native_unkeyprops='native_unkeyprops'
    ):
        """
        修改一个产品，可以修改主图，不能修改子图片
        传入产品ID <br/>可修改字段：outer_id,binds,sale_props,name,price,desc,image <br/>注意：1.可以修改主图,不能修改子图片,主图最大500K,目前仅支持GIF,JPG<br/>      2.商城卖家产品发布24小时后不能作删除或修改操作
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=9

        :param product_id: 产品ID
        :param outer_id: 外部产品ID
        :param binds: 非关键属性.调用taobao.itemprops.get获取pid,调用taobao.itempropvalues.get获取vid;格式:pid:vid;pid:vid
        :param sale_props: 销售属性.调用taobao.itemprops.get获取pid,调用taobao.itempropvalues.get获取vid;格式:pid:vid;pid:vid
        :param price: 产品市场价.精确到2位小数;单位为元.如:200.07
        :param desc: 产品描述.最大不超过25000个字符
        :param image: 产品主图.最大500K,目前仅支持GIF,JPG
        :param name: 产品名称.最大不超过30个字符
        :param major: 是否是主图
        :param native_unkeyprops: 自定义非关键属性
        """
        return self._top_request(
            "taobao.product.update",
            {
                "product_id": product_id,
                "outer_id": outer_id,
                "binds": binds,
                "sale_props": sale_props,
                "price": price,
                "desc": desc,
                "image": image,
                "name": name,
                "major": major,
                "native_unkeyprops": native_unkeyprops
            },
            result_processor=lambda x: x["product"]
        )

    def taobao_products_get(
            self,
            fields,
            nick,
            page_no='',
            page_size=''
    ):
        """
        获取产品列表
        根据淘宝会员帐号搜索所有产品信息，只有天猫商家发布商品时才需要用到。 <br/>注意：支持分页，每页最多返回100条,默认值为40,页码从1开始，默认为第一页
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10

        :param fields: 需返回的字段列表.可选值:Product数据结构中的所有字段;多个字段之间用','分隔
        :param nick: 用户昵称
        :param page_no: 页码.传入值为1代表第一页,传入值为2代表第二页,依此类推.默认返回的数据是从第一页开始.
        :param page_size: 每页条数.每页返回最多返回100条,默认值为40
        """
        return self._top_request(
            "taobao.products.get",
            {
                "fields": fields,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["products"]
        )

    def taobao_items_onsale_get(
            self,
            fields,
            q='',
            cid='',
            seller_cids='',
            page_no='1',
            has_discount='',
            has_showcase='',
            order_by='list_time:desc',
            is_taobao='',
            is_ex='',
            page_size='40',
            start_modified='',
            end_modified='',
            is_cspu='',
            is_combine='',
            auction_type=''
    ):
        """
        获取当前会话用户出售中的商品列表
        获取当前用户作为卖家的出售中的商品列表，并能根据传入的搜索条件对出售中的商品列表进行过滤 <br/>只能获得商品的部分信息，商品的详细信息请通过taobao.item.get获取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=18

        :param fields: 需返回的字段列表。可选值：Item商品结构体中的以下字段： approve_status,num_iid,title,nick,type,cid,pic_url,num,props,valid_thru,list_time,price,has_discount,has_invoice,has_warranty,has_showcase,modified,delist_time,postage_id,seller_cids,outer_id,sold_quantity ；字段之间用“,”分隔。不支持其他字段，如果需要获取其他字段数据，调用taobao.item.seller.get 获取。
        :param q: 搜索字段。搜索商品的title。
        :param cid: 商品类目ID。ItemCat中的cid字段。可以通过taobao.itemcats.get取到
        :param seller_cids: 卖家店铺内自定义类目ID。多个之间用“,”分隔。可以根据taobao.sellercats.list.get获得.(<font color='red'>注：目前最多支持32个ID号传入</font>)
        :param page_no: 页码。取值范围:大于零的整数。默认值为1,即默认返回第一页数据。用此接口获取数据时，当翻页获取的条数（page_no*page_size）超过10万,为了保护后台搜索引擎，接口将报错。所以请大家尽可能的细化自己的搜索条件，例如根据修改时间分段获取商品
        :param has_discount: 是否参与会员折扣。可选值：true，false。默认不过滤该条件
        :param has_showcase: 是否橱窗推荐。 可选值：true，false。默认不过滤该条件
        :param order_by: 排序方式。格式为column:asc/desc ，column可选值:list_time(上架时间),delist_time(下架时间),num(商品数量)，modified(最近修改时间)，sold_quantity（商品销量）,;默认上架时间降序(即最新上架排在前面)。如按照上架时间降序排序方式为list_time:desc
        :param is_taobao: 商品是否在淘宝显示
        :param is_ex: 商品是否在外部网店显示
        :param page_size: 每页条数。取值范围:大于零的整数;最大值：200；默认值：40。用此接口获取数据时，当翻页获取的条数（page_no*page_size）超过2万,为了保护后台搜索引擎，接口将报错。所以请大家尽可能的细化自己的搜索条件，例如根据修改时间分段获取商品
        :param start_modified: 起始的修改时间
        :param end_modified: 结束的修改时间
        :param is_cspu: 是否挂接了达尔文标准产品体系
        :param is_combine: 组合商品
        :param auction_type: 商品类型：a-拍卖,b-一口价
        """
        return self._top_request(
            "taobao.items.onsale.get",
            {
                "fields": fields,
                "q": q,
                "cid": cid,
                "seller_cids": seller_cids,
                "page_no": page_no,
                "has_discount": has_discount,
                "has_showcase": has_showcase,
                "order_by": order_by,
                "is_taobao": is_taobao,
                "is_ex": is_ex,
                "page_size": page_size,
                "start_modified": start_modified,
                "end_modified": end_modified,
                "is_cspu": is_cspu,
                "is_combine": is_combine,
                "auction_type": auction_type
            }
        )

    def taobao_item_update(
            self,
            num_iid,
            input_custom_cpv='',
            cpv_memo='',
            sku_spec_ids='',
            sku_delivery_times='',
            sku_hd_length='',
            sku_hd_height='',
            sku_hd_lamp_quantity='',
            location_state='',
            location_city='',
            food_security_prd_license_no='',
            food_security_design_code='',
            food_security_factory='',
            food_security_factory_site='',
            food_security_contact='',
            food_security_mix='',
            food_security_plan_storage='',
            food_security_period='',
            food_security_food_additive='',
            food_security_supplier='',
            food_security_product_date_start='',
            food_security_product_date_end='',
            food_security_stock_date_start='',
            food_security_stock_date_end='',
            food_security_health_product_no='',
            locality_life_obs='',
            locality_life_version='',
            locality_life_packageid='',
            ms_payment_price='',
            ms_payment_voucher_price='',
            ms_payment_reference_price='',
            delivery_time_delivery_time='',
            delivery_time_delivery_time_type='',
            delivery_time_need_delivery_time='',
            cid='',
            props='',
            num='',
            price='',
            title='',
            desc='',
            post_fee='',
            express_fee='',
            ems_fee='',
            list_time='',
            increment='',
            image='',
            stuff_status='',
            auction_point='',
            property_alias='',
            seller_cids='',
            postage_id='',
            outer_id='',
            product_id='',
            pic_path='',
            auto_fill='',
            is_taobao='',
            is_ex='',
            is_3_d='',
            is_replace_sku='',
            lang='',
            has_discount='',
            has_showcase='',
            approve_status='',
            freight_payer='',
            valid_thru='',
            has_invoice='',
            has_warranty='',
            sell_promise='',
            cod_postage_id='',
            is_lightning_consignment='',
            weight='',
            is_xinpin='',
            sub_stock='0',
            item_size='',
            item_weight='',
            sell_point='',
            barcode='',
            newprepay='',
            qualification='',
            o2o_bind_service='',
            ignorewarning='',
            features='',
            after_sale_id='',
            change_prop='',
            desc_modules='',
            is_offline='',
            wireless_desc='',
            spu_confirm='',
            input_pids='',
            sku_quantities='',
            sku_prices='',
            sku_properties='',
            sku_outer_ids='',
            input_str='',
            sku_barcode='',
            video_id='',
            interactive_id='',
            lease_extends_info='',
            locality_life_expirydate='',
            locality_life_network_id='',
            locality_life_merchant='',
            locality_life_verification='',
            locality_life_refund_ratio='',
            locality_life_choose_logis='',
            locality_life_onsale_auto_refund_ratio='',
            locality_life_refundmafee='',
            locality_life_eticket='',
            empty_fields='',
            scenic_ticket_pay_way='',
            scenic_ticket_book_cost='',
            paimai_info_mode='',
            paimai_info_deposit='',
            paimai_info_interval='',
            paimai_info_reserve='',
            paimai_info_valid_hour='',
            paimai_info_valid_minute='',
            global_stock_type='',
            global_stock_country='',
            global_stock_delivery_place='',
            global_stock_tax_free_promise=''
    ):
        """
        更新商品信息
        根据传入的num_iid更新对应的商品的数据。
        传入的num_iid所对应的商品必须属于当前会话的用户
        商品的属性和sku的属性有包含的关系，商品的价格要位于sku的价格区间之中（例如，sku价格有5元、10元两种，那么商品的价格就需要大于等于5元，小于等于10元，否则更新商品会失败）
        商品的类目和商品的价格、sku的价格都有一定的相关性（具体的关系要通过类目属性查询接口获得）
        当关键属性值更新为“其他”的时候，需要输入input_pids和input_str商品才能更新成功。该接口不支持产品属性修改。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21

        :param num_iid: 商品数字ID，该参数必须
        :param input_custom_cpv: 针对当前商品的自定义属性值
        :param cpv_memo: 针对当前商品的标准属性值的补充说明，让买家更加了解商品信息减少交易纠纷
        :param sku_spec_ids: 此参数暂时不起作用
        :param sku_delivery_times: 此参数暂时不起作用
        :param sku_hd_length: 家装建材类目，商品SKU的长度，正整数，单位为cm，部分类目必选。 数据和SKU一一对应，用,分隔，如：20,30,30
        :param sku_hd_height: 家装建材类目，商品SKU的高度，单位为cm，部分类目必选。 天猫和淘宝格式不同。天猫：可选值为：'0-15', '15-25', '25-50', '50-60', '60-80', '80-120', '120-160', '160-200'。 数据和SKU一一对应，用,分隔，格式如：15-25,25-50,25-50。 淘宝：正整数，单位为cm,格式如：20,30,30
        :param sku_hd_lamp_quantity: 家装建材类目，商品SKU的灯头数量，正整数，大于等于3，部分类目必选。 数据和SKU一一对应，用,分隔，如：3,5,7
        :param location_state: 所在地省份。如浙江
        :param location_city: 所在地城市。如杭州
        :param food_security_prd_license_no: 生产许可证号
        :param food_security_design_code: 产品标准号
        :param food_security_factory: 厂名
        :param food_security_factory_site: 厂址
        :param food_security_contact: 厂家联系方式
        :param food_security_mix: 配料表
        :param food_security_plan_storage: 储藏方法
        :param food_security_period: 保质期，默认有单位，传入数字
        :param food_security_food_additive: 食品添加剂
        :param food_security_supplier: 供货商
        :param food_security_product_date_start: 生产开始日期，格式必须为yyyy-MM-dd
        :param food_security_product_date_end: 生产结束日期,格式必须为yyyy-MM-dd
        :param food_security_stock_date_start: 进货开始日期，要在生产日期之后，格式必须为yyyy-MM-dd
        :param food_security_stock_date_end: 进货结束日期，要在生产日期之后，格式必须为yyyy-MM-dd
        :param food_security_health_product_no: 健字号，保健品/膳食营养补充剂 这个类目下特有的信息，此类目下无需填写生产许可证编号（QS），如果填写了生产许可证编号（QS）将被忽略不保存；保存宝贝时，标题前会自动加上健字号产品名称一起作为宝贝标题；
        :param locality_life_obs: 预约门店是否支持门店自提,1:是
        :param locality_life_version: 电子凭证版本 新版电子凭证值:1
        :param locality_life_packageid: 新版电子凭证包id
        :param ms_payment_price: 订金
        :param ms_payment_voucher_price: 尾款可抵扣金额
        :param ms_payment_reference_price: 参考价
        :param delivery_time_delivery_time: 商品级别设置的发货时间。设置了商品级别的发货时间，相对发货时间，则填写相对发货时间的天数（大于3）；绝对发货时间，则填写yyyy-mm-dd格式，如2013-11-11
        :param delivery_time_delivery_time_type: 发货时间类型：绝对发货时间或者相对发货时间
        :param delivery_time_need_delivery_time: 设置是否使用发货时间，商品级别，sku级别
        :param cid: 叶子类目id
        :param props: 商品属性列表。格式:pid:vid;pid:vid。属性的pid调用taobao.itemprops.get取得，属性值的vid用taobao.itempropvalues.get取得vid。 如果该类目下面没有属性，可以不用填写。如果有属性，必选属性必填，其他非必选属性可以选择不填写.属性不能超过35对。所有属性加起来包括分割符不能超过549字节，单个属性没有限制。 如果有属性是可输入的话，则用字段input_str填入属性的值。
        :param num: 商品数量，取值范围:0-900000000的整数。且需要等于Sku所有数量的和 拍卖商品中增加拍只能为1，荷兰拍要在[2,500)范围内。
        :param price: 商品价格。取值范围:0-100000000;精确到2位小数;单位:元。如:200.07，表示:200元7分。需要在正确的价格区间内。 拍卖商品对应的起拍价。
        :param title: 宝贝标题. 不能超过30字符,受违禁词控制
        :param desc: 商品描述. 字数要大于5个字符，小于25000个字符 ，受违禁词控制
        :param post_fee: 平邮费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:5.07，表示:5元7分, 注:post_fee,express_fee,ems_fee需一起填写
        :param express_fee: 快递费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:15.07，表示:15元7分
        :param ems_fee: ems费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:25.07，表示:25元7分
        :param list_time: 上架时间。大于当前时间则宝贝会下架进入定时上架的宝贝中。
        :param increment: 加价(降价)幅度。如果为0，代表系统代理幅度。对于增价拍和荷兰拍来说是加价幅度，对于降价拍来说是降价幅度。
        :param image: 商品图片。类型:JPG,GIF;最大长度:3M
        :param stuff_status: 商品新旧程度。可选值:new（全新）,unused（闲置）,second（二手）。
        :param auction_point: 商品的积分返点比例。如：5 表示返点比例0.5%. 注意：返点比例必须是>0的整数，而且最大是90,即为9%.B商家在发布非虚拟商品时，返点必须是 5的倍数，即0.5%的倍数。其它是1的倍数，即0.1%的倍数。无名良品商家发布商品时，复用该字段记录积分宝返点比例，返点必须是对应类目的返点步长的整数倍，默认是5，即0.5%。注意此时该字段值依旧必须是>0的整数，注意此时该字段值依旧必须是>0的整数，最高值不超过500，即50%
        :param property_alias: 属性值别名。如pid:vid:别名;pid1:vid1:别名1， pid:属性id vid:值id。总长度不超过800个字符，如'123:333:你好'，引号内的是10个字符。
        :param seller_cids: 重新关联商品与店铺类目，结构:',cid1,cid2,...,'，如果店铺类目存在二级类目，必须传入子类目cids。
        :param postage_id: 宝贝所属的运费模板ID。取值范围：整数且必须是该卖家的运费模板的ID（可通过taobao.delivery.templates.get获得当前会话用户的所有邮费模板）
        :param outer_id: 商家编码
        :param product_id: 商品所属的产品ID(B商家发布商品需要用)
        :param pic_path: 商品主图需要关联的图片空间的相对url。这个url所对应的图片必须要属于当前用户。pic_path和image只需要传入一个,如果两个都传，默认选择pic_path
        :param auto_fill: 代充商品类型。只有少数类目下的商品可以标记上此字段，具体哪些类目可以上传可以通过taobao.itemcat.features.get获得。在代充商品的类目下，不传表示不标记商品类型（交易搜索中就不能通过标记搜到相关的交易了）。可选类型： no_mark(不做类型标记) time_card(点卡软件代充) fee_card(话费软件代充)
        :param is_taobao: 是否在淘宝上显示（如果传FALSE，则在淘宝主站无法显示该商品）
        :param is_ex: 是否在外店显示
        :param is_3_d: 是否是3D
        :param is_replace_sku: 是否替换sku
        :param lang: 商品文字的版本，繁体传入”zh_HK”，简体传入”zh_CN”
        :param has_discount: 支持会员打折。可选值:true,false;
        :param has_showcase: 橱窗推荐。可选值:true,false;
        :param approve_status: 商品上传后的状态。可选值:onsale（出售中）,instock（库中），如果同时更新商品状态为出售中及list_time为将来的时间，则商品还是处于定时上架的状态, 此时item.is_timing为true
        :param freight_payer: 运费承担方式。运费承担方式。可选值:seller（卖家承担）,buyer(买家承担);
        :param valid_thru: 有效期。可选值:7,14;单位:天;
        :param has_invoice: 是否有发票。可选值:true,false (商城卖家此字段必须为true)
        :param has_warranty: 是否有保修。可选值:true,false;
        :param sell_promise: 是否承诺退换货服务!虚拟商品无须设置此项!
        :param cod_postage_id: 货到付款运费模板ID该字段已经废弃，货到付款模板已经集成到运费模板中。
        :param is_lightning_consignment: 实物闪电发货。注意：在售的闪电发货产品不允许取消闪电发货，需要先下架商品才能取消闪电发货标记
        :param weight: 商品的重量(商超卖家专用字段)
        :param is_xinpin: 商品是否为新品。只有在当前类目开通新品,并且当前用户拥有该类目下发布新品权限时才能设置is_xinpin为true，否则设置true后会返回错误码:isv.invalid-permission:xinpin。同时只有一口价全新的宝贝才能设置为新品，否则会返回错误码：isv.invalid-parameter:xinpin。不设置参数就保持原有值。
        :param sub_stock: 商品是否支持拍下减库存:1支持;2取消支持(付款减库存);0(默认)不更改 集市卖家默认拍下减库存; 商城卖家默认付款减库存
        :param item_size: 表示商品的体积，如果需要使用按体积计费的运费模板，一定要设置这个值。该值的单位为立方米（m3），如果是其他单位，请转换成成立方米。该值支持两种格式的设置：格式1：bulk:3,单位为立方米(m3),表示直接设置为商品的体积。格式2：length:10;breadth:10;height:10，单位为米（m）。体积和长宽高都支持小数类型。在传入体积或长宽高时候，不能带单位。体积的单位默认为立方米（m3），长宽高的单位默认为米(m)在编辑的时候，如果需要删除体积属性，请设置该值为0，如bulk:0
        :param item_weight: 商品的重量，用于按重量计费的运费模板。注意：单位为kg。 只能传入数值类型（包含小数），不能带单位，单位默认为kg。 在编辑时候，如果需要在商品里删除重量的信息，就需要将值设置为0
        :param sell_point: 商品卖点信息，最长150个字符。天猫和集市都可用
        :param barcode: 商品条形码
        :param newprepay: 该宝贝是否支持【7天无理由退货】，卖家选择的值只是一个因素，最终以类目和选择的属性条件来确定是否支持7天。填入字符0，表示不支持；未填写或填人字符1，表示支持7天无理由退货；<br>注意：使用该API修改商品其它属性如标题title时，如需保持商品不支持7天无理由退货状态，该字段需传入0 。
        :param qualification: 商品资质信息
        :param o2o_bind_service: 汽车O2O绑定线下服务标记，如不为空，表示关联服务，否则，不关联服务。
        :param ignorewarning: 忽略警告提示.
        :param features: 宝贝特征值，格式为：【key1:value1;key2:value2;key3:value3;】，key和value用【:】分隔，key&value之间用【;】分隔，只有在Top支持的特征值才能保存到宝贝上，目前支持的Key列表为：mysize_tp,是指尺码库对应的key
        :param after_sale_id: 售后说明模板id
        :param change_prop: 基础色数据，淘宝不使用
        :param desc_modules: 已废弃
        :param is_offline: 是否是线下商品。1：线上商品（默认值）；2：线上或线下商品；3：线下商品。
        :param wireless_desc: 无线的宝贝描述
        :param spu_confirm: 手机类目spu 产品信息确认声明
        :param input_pids: 用户自行输入的类目属性ID串，结构：'pid1,pid2,pid3'，如：'20000'（表示品牌） 注：通常一个类目下用户可输入的关键属性不超过1个。
        :param sku_quantities: 更新的Sku的数量串，结构如：num1,num2,num3 如:2,3,4
        :param sku_prices: 更新的Sku的价格串，结构如：10.00,5.00,… 精确到2位小数;单位:元。如:200.07，表示:200元7分
        :param sku_properties: 更新的sku的属性串，调用taobao.itemprops.get获取。格式:pid1:vid;pid2:vid,多个sku属性之间用逗号分隔。该字段内的属性需要在props字段同时包含。如果新增商品包含了sku，则此字段一定要传入,字段长度要控制在512个字节以内。
        :param sku_outer_ids: Sku的外部id串，结构如：1234,1342,… sku_properties, sku_quantities, sku_prices, sku_outer_ids在输入数据时要一一对应，如果没有sku_outer_ids也要写上这个参数，入参是','(这个是两个sku的示列，逗号数应该是sku个数减1)；该参数最大长度是512个字节
        :param input_str: 用户自行输入的子属性名和属性值，结构:'父属性值;一级子属性名;一级子属性值;二级子属性名;自定义输入值,....',如：“耐克;耐克系列;科比系列;科比系列;2K5,Nike乔丹鞋;乔丹系列;乔丹鞋系列;乔丹鞋系列;json5”，多个自定义属性用','分割，input_str需要与input_pids一一对应，注：通常一个类目下用户可输入的关键属性不超过1个。所有属性别名加起来不能超过3999字节。此处不可以使用“其他”、“其它”和“其她”这三个词。
        :param sku_barcode: sku层面的条形码，多个SKU情况，与SKU价格库存格式类似，用逗号分隔
        :param video_id: 主图视频id
        :param interactive_id: 主图视频互动信息id，必须有主图视频id才能传互动信息id
        :param lease_extends_info: 淘宝租赁扩展信息
        :param locality_life_expirydate: 本地生活电子交易凭证业务，目前此字段只涉及到的信息为有效期;如果有效期为起止日期类型，此值为2012-08-06,2012-08-16如果有效期为【购买成功日 至】类型则格式为2012-08-16如果有效期为天数类型则格式为15
        :param locality_life_network_id: 网点ID,在参数empty_fields里设置locality_life.network_id可删除网点ID
        :param locality_life_merchant: 码商信息，格式为 码商id:nick
        :param locality_life_verification: 核销打款,1代表核销打款 0代表非核销打款; 在参数empty_fields里设置locality_life.verification可删除核销打款
        :param locality_life_refund_ratio: 退款比例，百分比%前的数字,1-100的正整数值; 在参数empty_fields里设置locality_life.refund_ratio可删除退款比例
        :param locality_life_choose_logis: 编辑电子凭证宝贝时候表示是否使用邮寄0: 代表不使用邮寄；1：代表使用邮寄；如果不设置这个值，代表不使用邮寄
        :param locality_life_onsale_auto_refund_ratio: 电子凭证售中自动退款比例，百分比%前的数字，介于1-100之间的整数
        :param locality_life_refundmafee: 退款码费承担方。发布电子凭证宝贝的时候会增加“退款码费承担方”配置项，可选填：(1)s（卖家承担） (2)b(买家承担)
        :param locality_life_eticket: 电子凭证业务属性，数据字典是: 1、is_card:1 (暂时不用) 2、consume_way:4 （1 串码 ，4 身份证）3、consume_midmnick ：(核销放行账号:用户id-用户名，支持多个，用逗号分隔,例如 1234-测试账号35,1345-测试账号56）4、market:eticket (电子凭证商品标记) 5、has_pos:1 (1 表示商品配置线下门店，在detail上进行展示 ，没有或者其他值只不展示)格式是: k1:v2;k2:v2;........ 如：has_pos:1;market:eticket;consume_midmnick:901409638-OPPO;consume_way:4
        :param empty_fields: 支持宝贝信息的删除,如需删除对应的食品安全信息中的储藏方法、保质期， 则应该设置此参数的值为：food_security.plan_storage,food_security.period; 各个参数的名称之间用【,】分割, 如果对应的参数有设置过值，即使在这个列表中，也不会被删除; 目前支持此功能的宝贝信息如下：食品安全信息所有字段、电子交易凭证字段（locality_life，locality_life.verification，locality_life.refund_ratio，locality_life.network_id ，locality_life.onsale_auto_refund_ratio）。支持对全球购宝贝信息的清除（字符串中包含global_stock）
        :param scenic_ticket_pay_way: 景区门票类宝贝编辑时候，当卖家签订了支付宝代扣协议时候，需要选择支付方式：全额支付和订金支付。当scenic_ticket_pay_way为1时表示全额支付，为2时表示订金支付
        :param scenic_ticket_book_cost: 景区门票在选择订金支付时候，需要交的预订费。传入的值是1到20之间的数值，小数点后最多可以保留两位（多余的部分将做四舍五入的处理）。这个数值表示的是预订费的比例，最终的预订费为 scenic_ticket_book_cost乘一口价除以100
        :param paimai_info_mode: 拍卖商品选择的拍卖类型，拍卖类型包括三种：增价拍(1)，荷兰拍(2)和降价拍(3)。
        :param paimai_info_deposit: 拍卖宝贝的保证金。对于增价拍和荷兰拍来说保证金有两种模式：淘宝默认模式（首次出价金额的10%），自定义固定保证金（固定冻结金额只能输入不超过30万的正整数），并且保证金只冻结1次。对于降价拍来说保证金只有淘宝默认的（竞拍价格的10% * 竞拍数量），并且每次出价都需要冻结保证金。对于拍卖宝贝来说，保证金是必须的，但是默认使用淘宝默认保证金模式，只有用户需要使用自定义固定保证金的时候才需要使用到这个参数。如果该参数不传或传入0则代表使用默认。
        :param paimai_info_interval: 降价拍宝贝的降价周期(分钟)。降价拍宝贝的价格每隔paimai_info.interval时间会下降一次increment。
        :param paimai_info_reserve: 降价拍宝贝的保留价。对于降价拍来说，paimai_info.reserve必须大于0，且小于price-increment，而且（price-paimai_info.reserve）/increment的计算结果必须为整数
        :param paimai_info_valid_hour: 自定义销售周期的小时数。拍卖宝贝可以自定义销售周期，这里指定销售周期的小时数。自定义销售周期的小时数。拍卖宝贝可以自定义销售周期，这里指定销售周期的小时数。注意，该参数只作为输入参数，不能通过taobao.item.get接口获取。
        :param paimai_info_valid_minute: 自定义销售周期的分钟数。拍卖宝贝可以自定义销售周期，这里是指定销售周期的分钟数。自定义销售周期的小时数。拍卖宝贝可以自定义销售周期，这里指定销售周期的小时数。注意，该参数只作为输入参数，不能通过taobao.item.get接口获取。
        :param global_stock_type: 全球购商品采购地（库存类型）全球购商品有两种库存类型：现货和代购 参数值为1时代表现货，值为2时代表代购。注意：使用时请与 全球购商品采购地（地区/国家）配合使用
        :param global_stock_country: 全球购商品采购地（地区/国家）,默认值只在全球购商品采购地（库存类型选择情况生效），地区国家值为（美国, 香港, 日本, 英国, 新西兰, 德国, 韩国, 荷兰, 澳洲, 法国, 意大利, 台湾, 澳门, 加拿大, 瑞士, 西班牙, 泰国, 新加坡, 马来西亚, 菲律宾, 其他）
        :param global_stock_delivery_place: 全球购商品发货地，发货地现在有两种类型：“国内”和“海外及港澳台”，参数值为1时代表“国内”，值为2时代表“海外及港澳台”
        :param global_stock_tax_free_promise: 全球购商品卖家包税承诺，当值为true时，代表卖家承诺包税。
        """
        return self._top_request(
            "taobao.item.update",
            {
                "num_iid": num_iid,
                "input_custom_cpv": input_custom_cpv,
                "cpv_memo": cpv_memo,
                "sku_spec_ids": sku_spec_ids,
                "sku_delivery_times": sku_delivery_times,
                "sku_hd_length": sku_hd_length,
                "sku_hd_height": sku_hd_height,
                "sku_hd_lamp_quantity": sku_hd_lamp_quantity,
                "location.state": location_state,
                "location.city": location_city,
                "food_security.prd_license_no": food_security_prd_license_no,
                "food_security.design_code": food_security_design_code,
                "food_security.factory": food_security_factory,
                "food_security.factory_site": food_security_factory_site,
                "food_security.contact": food_security_contact,
                "food_security.mix": food_security_mix,
                "food_security.plan_storage": food_security_plan_storage,
                "food_security.period": food_security_period,
                "food_security.food_additive": food_security_food_additive,
                "food_security.supplier": food_security_supplier,
                "food_security.product_date_start": food_security_product_date_start,
                "food_security.product_date_end": food_security_product_date_end,
                "food_security.stock_date_start": food_security_stock_date_start,
                "food_security.stock_date_end": food_security_stock_date_end,
                "food_security.health_product_no": food_security_health_product_no,
                "locality_life.obs": locality_life_obs,
                "locality_life.version": locality_life_version,
                "locality_life.packageid": locality_life_packageid,
                "ms_payment.price": ms_payment_price,
                "ms_payment.voucher_price": ms_payment_voucher_price,
                "ms_payment.reference_price": ms_payment_reference_price,
                "delivery_time.delivery_time": delivery_time_delivery_time,
                "delivery_time.delivery_time_type": delivery_time_delivery_time_type,
                "delivery_time.need_delivery_time": delivery_time_need_delivery_time,
                "cid": cid,
                "props": props,
                "num": num,
                "price": price,
                "title": title,
                "desc": desc,
                "post_fee": post_fee,
                "express_fee": express_fee,
                "ems_fee": ems_fee,
                "list_time": list_time,
                "increment": increment,
                "image": image,
                "stuff_status": stuff_status,
                "auction_point": auction_point,
                "property_alias": property_alias,
                "seller_cids": seller_cids,
                "postage_id": postage_id,
                "outer_id": outer_id,
                "product_id": product_id,
                "pic_path": pic_path,
                "auto_fill": auto_fill,
                "is_taobao": is_taobao,
                "is_ex": is_ex,
                "is_3D": is_3_d,
                "is_replace_sku": is_replace_sku,
                "lang": lang,
                "has_discount": has_discount,
                "has_showcase": has_showcase,
                "approve_status": approve_status,
                "freight_payer": freight_payer,
                "valid_thru": valid_thru,
                "has_invoice": has_invoice,
                "has_warranty": has_warranty,
                "sell_promise": sell_promise,
                "cod_postage_id": cod_postage_id,
                "is_lightning_consignment": is_lightning_consignment,
                "weight": weight,
                "is_xinpin": is_xinpin,
                "sub_stock": sub_stock,
                "item_size": item_size,
                "item_weight": item_weight,
                "sell_point": sell_point,
                "barcode": barcode,
                "newprepay": newprepay,
                "qualification": qualification,
                "o2o_bind_service": o2o_bind_service,
                "ignorewarning": ignorewarning,
                "features": features,
                "after_sale_id": after_sale_id,
                "change_prop": change_prop,
                "desc_modules": desc_modules,
                "is_offline": is_offline,
                "wireless_desc": wireless_desc,
                "spu_confirm": spu_confirm,
                "input_pids": input_pids,
                "sku_quantities": sku_quantities,
                "sku_prices": sku_prices,
                "sku_properties": sku_properties,
                "sku_outer_ids": sku_outer_ids,
                "input_str": input_str,
                "sku_barcode": sku_barcode,
                "video_id": video_id,
                "interactive_id": interactive_id,
                "lease_extends_info": lease_extends_info,
                "locality_life.expirydate": locality_life_expirydate,
                "locality_life.network_id": locality_life_network_id,
                "locality_life.merchant": locality_life_merchant,
                "locality_life.verification": locality_life_verification,
                "locality_life.refund_ratio": locality_life_refund_ratio,
                "locality_life.choose_logis": locality_life_choose_logis,
                "locality_life.onsale_auto_refund_ratio": locality_life_onsale_auto_refund_ratio,
                "locality_life.refundmafee": locality_life_refundmafee,
                "locality_life.eticket": locality_life_eticket,
                "empty_fields": empty_fields,
                "scenic_ticket_pay_way": scenic_ticket_pay_way,
                "scenic_ticket_book_cost": scenic_ticket_book_cost,
                "paimai_info.mode": paimai_info_mode,
                "paimai_info.deposit": paimai_info_deposit,
                "paimai_info.interval": paimai_info_interval,
                "paimai_info.reserve": paimai_info_reserve,
                "paimai_info.valid_hour": paimai_info_valid_hour,
                "paimai_info.valid_minute": paimai_info_valid_minute,
                "global_stock_type": global_stock_type,
                "global_stock_country": global_stock_country,
                "global_stock_delivery_place": global_stock_delivery_place,
                "global_stock_tax_free_promise": global_stock_tax_free_promise
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_add(
            self,
            location_state,
            location_city,
            num,
            type,
            stuff_status,
            title,
            desc,
            cid,
            sku_spec_ids='',
            sku_delivery_times='',
            sku_hd_length='',
            sku_hd_height='',
            sku_hd_lamp_quantity='',
            input_str='',
            input_pids='',
            sku_properties='',
            sku_quantities='',
            sku_prices='',
            sku_outer_ids='',
            sku_barcode='',
            delivery_time_need_delivery_time='',
            delivery_time_delivery_time_type='',
            delivery_time_delivery_time='',
            ms_payment_reference_price='',
            ms_payment_voucher_price='',
            ms_payment_price='',
            locality_life_obs='',
            locality_life_version='',
            locality_life_packageid='',
            food_security_prd_license_no='',
            food_security_design_code='',
            food_security_factory='',
            food_security_factory_site='',
            food_security_contact='',
            food_security_mix='',
            food_security_plan_storage='',
            food_security_period='',
            food_security_food_additive='',
            food_security_supplier='',
            food_security_product_date_start='',
            food_security_product_date_end='',
            food_security_stock_date_start='',
            food_security_stock_date_end='',
            food_security_health_product_no='',
            price='',
            approve_status='onsale',
            props='',
            freight_payer='seller',
            valid_thru='14',
            has_invoice='false',
            has_warranty='false',
            has_showcase='false',
            seller_cids='',
            has_discount='false',
            post_fee='',
            express_fee='',
            ems_fee='',
            list_time='',
            increment='',
            image='',
            postage_id='',
            auction_point='',
            property_alias='',
            lang='zh_CN',
            outer_id='',
            product_id='',
            pic_path='',
            auto_fill='',
            is_taobao='',
            is_ex='',
            is_3_d='',
            sell_promise='',
            cod_postage_id='',
            is_lightning_consignment='',
            weight='',
            is_xinpin='false',
            sub_stock='0',
            scenic_ticket_pay_way='',
            scenic_ticket_book_cost='',
            item_size='',
            item_weight='',
            sell_point='',
            barcode='',
            newprepay='',
            qualification='',
            o2o_bind_service='',
            features='',
            ignorewarning='',
            after_sale_id='',
            change_prop='',
            desc_modules='',
            is_offline='',
            wireless_desc='',
            spu_confirm='',
            video_id='',
            interactive_id='',
            lease_extends_info='',
            brokerage='',
            biz_code='',
            image_urls='',
            locality_life_choose_logis='',
            locality_life_expirydate='',
            locality_life_network_id='',
            locality_life_merchant='',
            locality_life_verification='',
            locality_life_refund_ratio='',
            locality_life_onsale_auto_refund_ratio='',
            locality_life_refundmafee='',
            locality_life_eticket='',
            paimai_info_mode='',
            paimai_info_deposit='',
            paimai_info_interval='',
            paimai_info_reserve='',
            paimai_info_valid_hour='',
            paimai_info_valid_minute='',
            global_stock_type='',
            global_stock_country='',
            support_custom_made='',
            custom_made_type_id='',
            global_stock_delivery_place='',
            global_stock_tax_free_promise='',
            input_custom_cpv='',
            cpv_memo=''
    ):
        """
        添加一个商品
        此接口用于新增一个商品 <br/>商品所属的卖家是当前会话的用户 <br/>商品的属性和sku的属性有包含的关系，商品的价格要位于sku的价格区间之中（例如，sku价格有5元、10元两种，那么商品的价格就需要大于等于5元，小于等于10元，否则新增商品会失败） <br/>商品的类目和商品的价格、sku的价格都有一定的相关性（具体的关系要通过类目属性查询接口获得） <br/>商品的运费承担方式和邮费设置有相关性，卖家承担运费不用设置邮费，买家承担运费需要设置邮费 <br/>当关键属性值选择了“其他”的时候，需要输入input_pids和input_str商品才能添加成功。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22

        :param location_state: 所在地省份。如浙江
        :param location_city: 所在地城市。如杭州 。
        :param num: 商品数量。取值范围:0-900000000的整数。且需要等于Sku所有数量的和。拍卖商品中增加拍只能为1，荷兰拍要在[2,500)范围内。
        :param type: 发布类型。可选值:fixed(一口价),auction(拍卖)。B商家不能发布拍卖商品，而且拍卖商品是没有SKU的。拍卖商品发布时需要附加拍卖商品信息：拍卖类型(paimai_info.mode，拍卖类型包括三种：增价拍[1]，荷兰拍[2]以及降价拍[3])，商品数量(num)，起拍价(price)，价格幅度(increament)，保证金(paimai_info.deposit)。另外拍卖商品支持自定义销售周期，通过paimai_info.valid_hour和paimai_info.valid_minute来指定。对于降价拍来说需要设置降价周期(paimai_info.interval)和拍卖保留价(paimai_info.reserve)。注意：通过taobao.item.get接口获取拍卖信息时，会返回除了valid_hour和valid_minute之外的所有拍卖信息。
        :param stuff_status: 新旧程度。可选值：new(新)，second(二手)。B商家不能发布二手商品。如果是二手商品，特定类目下属性里面必填新旧成色属性
        :param title: 宝贝标题。不能超过30字符，受违禁词控制。天猫图书管控类目最大允许120字符；
        :param desc: 宝贝描述。字数要大于5个字符，小于25000个字符，受违禁词控制
        :param cid: 叶子类目id
        :param sku_spec_ids: 此参数暂时不起作用
        :param sku_delivery_times: 此参数暂时不起作用
        :param sku_hd_length: 家装建材类目，商品SKU的长度，正整数，单位为cm，部分类目必选。 数据和SKU一一对应，用,分隔，如：20,30,30
        :param sku_hd_height: 家装建材类目，商品SKU的高度，单位为cm，部分类目必选。 天猫和淘宝格式不同。天猫：可选值为：'0-15', '15-25', '25-50', '50-60', '60-80', '80-120', '120-160', '160-200'。 数据和SKU一一对应，用,分隔，格式如：15-25,25-50,25-50。 淘宝：正整数，单位为cm,格式如：20,30,30
        :param sku_hd_lamp_quantity: 家装建材类目，商品SKU的灯头数量，正整数，大于等于3，部分类目必选。天猫商家专用。 数据和SKU一一对应，用,分隔，如：3,5,7
        :param input_str: 用户自行输入的子属性名和属性值，结构:'父属性值;一级子属性名;一级子属性值;二级子属性名;自定义输入值,....',如：“耐克;耐克系列;科比系列;科比系列;2K5,Nike乔丹鞋;乔丹系列;乔丹鞋系列;乔丹鞋系列;json5”，多个自定义属性用','分割，input_str需要与input_pids一一对应，注：通常一个类目下用户可输入的关键属性不超过1个。所有属性别名加起来不能超过3999字节。此处不可以使用“其他”、“其它”和“其她”这三个词。
        :param input_pids: 用户自行输入的类目属性ID串，结构：'pid1,pid2,pid3'，如：'20000'（表示品牌） 注：通常一个类目下用户可输入的关键属性不超过1个。
        :param sku_properties: 更新的sku的属性串，调用taobao.itemprops.get获取。格式:pid1:vid;pid2:vid,多个sku属性之间用逗号分隔。该字段内的属性需要在props字段同时包含。如果新增商品包含了sku，则此字段一定要传入,字段长度要控制在512个字节以内。
        :param sku_quantities: Sku的数量串，结构如：num1,num2,num3 如：2,3
        :param sku_prices: Sku的价格串，结构如：10.00,5.00,… 精确到2位小数;单位:元。如:200.07，表示:200元7分
        :param sku_outer_ids: Sku的外部id串，结构如：1234,1342,… sku_properties, sku_quantities, sku_prices, sku_outer_ids在输入数据时要一一对应，如果没有sku_outer_ids也要写上这个参数，入参是','(这个是两个sku的示列，逗号数应该是sku个数减1)；该参数最大长度是512个字节
        :param sku_barcode: sku层面的条形码，多个SKU情况，与SKU价格库存格式类似，用逗号分隔
        :param delivery_time_need_delivery_time: 设置是否使用发货时间，商品级别，sku级别
        :param delivery_time_delivery_time_type: 发货时间类型：绝对发货时间或者相对发货时间
        :param delivery_time_delivery_time: 商品级别设置的发货时间。设置了商品级别的发货时间，相对发货时间，则填写相对发货时间的天数（大于3）；绝对发货时间，则填写yyyy-mm-dd格式，如2013-11-11
        :param ms_payment_reference_price: 参考价。该商品订单首次支付价格为 订金 价格，用户可根据 参考价 估算全款。详见说明：http://bangpai.taobao.com/group/thread/15031186-303287205.htm
        :param ms_payment_voucher_price: 尾款可抵扣金额。详见说明：http://bangpai.taobao.com/group/thread/15031186-303287205.htm
        :param ms_payment_price: 订金。在“线上付订金线下付尾款”模式中，有订金、尾款可抵扣金额和参考价，三者需要同时填写。该商品订单首次支付价格为 订金 价格，用户可根据 参考价 估算全款。该模式有别于“一口价”付款方式，针对一个商品，只能选择两种付款方式中的一种，其适用于家装、二手车等场景。详见说明：http://bangpai.taobao.com/group/thread/15031186-303287205.htm
        :param locality_life_obs: 预约门店是否支持门店自提,1:是
        :param locality_life_version: 新版电子凭证字段
        :param locality_life_packageid: 新版电子凭证包 id
        :param food_security_prd_license_no: 生产许可证号
        :param food_security_design_code: 产品标准号
        :param food_security_factory: 厂名
        :param food_security_factory_site: 厂址
        :param food_security_contact: 厂家联系方式
        :param food_security_mix: 配料表
        :param food_security_plan_storage: 储藏方法
        :param food_security_period: 保质期，默认有单位，传入数字
        :param food_security_food_additive: 食品添加剂
        :param food_security_supplier: 供货商
        :param food_security_product_date_start: 生产开始日期，格式必须为yyyy-MM-dd
        :param food_security_product_date_end: 生产结束日期,格式必须为yyyy-MM-dd
        :param food_security_stock_date_start: 进货开始日期，要在生产日期之后，格式必须为yyyy-MM-dd
        :param food_security_stock_date_end: 进货结束日期，要在生产日期之后，格式必须为yyyy-MM-dd
        :param food_security_health_product_no: 健字号，保健品/膳食营养补充剂 这个类目下特有的信息，此类目下无需填写生产许可证编号（QS），如果填写了生产许可证编号（QS）将被忽略不保存；保存宝贝时，标题前会自动加上健字号产品名称一起作为宝贝标题；
        :param price: 商品价格。取值范围:0-100000000;精确到2位小数;单位:元。如:200.07，表示:200元7分。需要在正确的价格区间内。拍卖商品对应的起拍价。
        :param approve_status: 商品上传后的状态。可选值:onsale(出售中),instock(仓库中);默认值:onsale
        :param props: 商品属性列表。格式:pid:vid;pid:vid。属性的pid调用taobao.itemprops.get取得，属性值的vid用taobao.itempropvalues.get取得vid。 如果该类目下面没有属性，可以不用填写。如果有属性，必选属性必填，其他非必选属性可以选择不填写.属性不能超过35对。所有属性加起来包括分割符不能超过549字节，单个属性没有限制。 如果有属性是可输入的话，则用字段input_str填入属性的值
        :param freight_payer: 运费承担方式。可选值:seller（卖家承担）,buyer(买家承担);默认值:seller。卖家承担不用设置邮费和postage_id.买家承担的时候，必填邮费和postage_id 如果用户设置了运费模板会优先使用运费模板，否则要同步设置邮费（post_fee,express_fee,ems_fee）
        :param valid_thru: 有效期。可选值:7,14;单位:天;默认值:14
        :param has_invoice: 是否有发票。可选值:true,false (商城卖家此字段必须为true);默认值:false(无发票)
        :param has_warranty: 是否有保修。可选值:true,false;默认值:false(不保修)
        :param has_showcase: 橱窗推荐。可选值:true,false;默认值:false(不推荐)
        :param seller_cids: 商品所属的店铺类目列表。按逗号分隔。结构:',cid1,cid2,...,'，如果店铺类目存在二级类目，必须传入子类目cids。
        :param has_discount: 支持会员打折。可选值:true,false;默认值:false(不打折)
        :param post_fee: 平邮费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:5.07，表示:5元7分. 注:post_fee,express_fee,ems_fee需要一起填写
        :param express_fee: 快递费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:15.07，表示:15元7分
        :param ems_fee: ems费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:25.07，表示:25元7分
        :param list_time: 定时上架时间。(时间格式：yyyy-MM-dd HH:mm:ss)
        :param increment: 加价(降价)幅度。如果为0，代表系统代理幅度。对于增价拍和荷兰拍来说是加价幅度，对于降价拍来说是降价幅度。
        :param image: 商品主图片。类型:JPG,GIF;最大长度:3M。（推荐使用pic_path字段，先把图片上传到卖家图片空间）
        :param postage_id: 宝贝所属的运费模板ID。取值范围：整数且必须是该卖家的运费模板的ID（可通过taobao.delivery.template.get获得当前会话用户的所有邮费模板）
        :param auction_point: 商品的积分返点比例。如:5,表示:返点比例0.5%. 注意：返点比例必须是>0的整数，而且最大是90,即为9%.B商家在发布非虚拟商品时，返点必须是 5的倍数，即0.5%的倍数。其它是1的倍数，即0.1%的倍数。无名良品商家发布商品时，复用该字段记录积分宝返点比例，返点必须是对应类目的返点步长的整数倍，默认是5，即0.5%。注意此时该字段值依旧必须是>0的整数，最高值不超过500，即50%
        :param property_alias: 属性值别名。如pid:vid:别名;pid1:vid1:别名1 ，其中：pid是属性id vid是属性值id。总长度不超过800个字符，如'123:333:你好'，引号内的是10个字符。
        :param lang: 商品文字的字符集。繁体传入'zh_HK'，简体传入'zh_CN'，不传默认为简体
        :param outer_id: 商品外部编码，该字段的最大长度是64个字节
        :param product_id: 商品所属的产品ID(B商家发布商品需要用)
        :param pic_path: （推荐）商品主图需要关联的图片空间的相对url。这个url所对应的图片必须要属于当前用户。pic_path和image只需要传入一个,如果两个都传，默认选择pic_path
        :param auto_fill: 代充商品类型。在代充商品的类目下，不传表示不标记商品类型（交易搜索中就不能通过标记搜到相关的交易了）。可选类型： no_mark(不做类型标记) time_card(点卡软件代充) fee_card(话费软件代充)
        :param is_taobao: 是否在淘宝上显示（如果传FALSE，则在淘宝主站无法显示该商品）
        :param is_ex: 是否在外店显示
        :param is_3_d: 是否是3D
        :param sell_promise: 是否承诺退换货服务!虚拟商品无须设置此项!
        :param cod_postage_id: 此为货到付款运费模板的ID，对应的JAVA类型是long,如果COD卖家应用了货到付款运费模板，此值要进行设置。该字段已经废弃
        :param is_lightning_consignment: 实物闪电发货
        :param weight: 商品的重量(商超卖家专用字段)
        :param is_xinpin: 商品是否为新品。只有在当前类目开通新品,并且当前用户拥有该类目下发布新品权限时才能设置is_xinpin为true，否则设置true后会返回错误码:isv.invalid-permission:add-xinpin。同时只有一口价全新的宝贝才能设置为新品，否则会返回错误码：isv.invalid-parameter:xinpin。不设置该参数值或设置为false效果一致。
        :param sub_stock: 商品是否支持拍下减库存:1支持;2取消支持(付款减库存);0(默认)不更改集市卖家默认拍下减库存;商城卖家默认付款减库存
        :param scenic_ticket_pay_way: 景区门票类宝贝发布时候，当卖家签订了支付宝代扣协议时候，需要选择支付方式：全额支付和订金支付。当scenic_ticket_pay_way为1时表示全额支付，为2时表示订金支付
        :param scenic_ticket_book_cost: 景区门票在选择订金支付时候，需要交的预订费。传入的值是1到20之间的数值，小数点后最多可以保留两位（多余的部分将做四舍五入的处理）。这个数值表示的是预订费的比例，最终的预订费为 scenic_ticket_book_cost乘一口价除以100
        :param item_size: 表示商品的体积，如果需要使用按体积计费的运费模板，一定要设置这个值。该值的单位为立方米（m3），如果是其他单位，请转换成成立方米。该值支持两种格式的设置：格式1：bulk:3,单位为立方米(m3),表示直接设置为商品的体积。格式2：length:10;breadth:10;height:10，单位为米（m）。体积和长宽高都支持小数类型。在传入体积或长宽高时候，不能带单位。体积的单位默认为立方米（m3），长宽高的单位默认为米(m)该值支持两种格式的设置：格式1：bulk:3,单位为立方米(m3),表示直接设置为商品的体积。格式2：length:10;breadth:10;height:10，单位为米（m）
        :param item_weight: 商品的重量，用于按重量计费的运费模板。注意：单位为kg。只能传入数值类型（包含小数），不能带单位，单位默认为kg。
        :param sell_point: 商品卖点信息，最长150个字符。天猫商家和集市卖家都可用。
        :param barcode: 商品条形码
        :param newprepay: 该宝贝是否支持【7天无理由退货】，卖家选择的值只是一个因素，最终以类目和选择的属性条件来确定是否支持7天。填入字符0，表示不支持；未填写或填人字符1，表示支持7天无理由退货；
        :param qualification: 商品资质信息
        :param o2o_bind_service: 汽车O2O绑定线下服务标记，如不为空，表示关联服务，否则，不关联服务。
        :param features: 宝贝特征值，格式为：【key1:value1;key2:value2;key3:value3;】，key和value用【:】分隔，key&value之间用【;】分隔，只有在Top支持的特征值才能保存到宝贝上，目前支持的Key列表为：mysize_tp
        :param ignorewarning: 忽略警告提示.
        :param after_sale_id: 售后说明模板id
        :param change_prop: 基础色数据，淘宝不使用
        :param desc_modules: 已废弃
        :param is_offline: 是否是线下商品。1：线上商品（默认值）；2：线上或线下商品；3：线下商品。
        :param wireless_desc: 无线的宝贝描述
        :param spu_confirm: 手机类目spu 优化，信息确认字段
        :param video_id: 主图视频id
        :param interactive_id: 主图视频互动信息id，必须填写主图视频id才能有互动信息id
        :param lease_extends_info: 租赁扩展信息
        :param brokerage: 仅淘小铺卖家需要。佣金比例(15.3对应的佣金比例为15.3%).只支持小数点后1位。多余的位数四舍五入(15.32会保存为15.3%
        :param biz_code: 业务身份编码。淘小铺编码为'taobao-taoxiaopu'
        :param image_urls: 商品主图，多张主图用该字段。使用该字段时。pic_path,image必须为空.
        :param locality_life_choose_logis: 发布电子凭证宝贝时候表示是否使用邮寄 0: 代表不使用邮寄； 1：代表使用邮寄；如果不设置这个值，代表不使用邮寄
        :param locality_life_expirydate: 本地生活电子交易凭证业务，目前此字段只涉及到的信息为有效期;如果有效期为起止日期类型，此值为2012-08-06,2012-08-16如果有效期为【购买成功日 至】类型则格式为2012-08-16如果有效期为天数类型则格式为15
        :param locality_life_network_id: 网点ID
        :param locality_life_merchant: 码商信息，格式为 码商id:nick
        :param locality_life_verification: 核销打款 1代表核销打款 0代表非核销打款
        :param locality_life_refund_ratio: 退款比例，百分比%前的数字,1-100的正整数值
        :param locality_life_onsale_auto_refund_ratio: 电子凭证售中自动退款比例，百分比%前的数字，介于1-100之间的整数
        :param locality_life_refundmafee: 退款码费承担方。发布电子凭证宝贝的时候会增加“退款码费承担方”配置项，可选填：(1)s（卖家承担） (2)b(买家承担)
        :param locality_life_eticket: 电子凭证业务属性，数据字典是: 1、is_card:1 (暂时不用) 2、consume_way:4 （1 串码 ，4 身份证）3、consume_midmnick ：(核销放行账号:用户id-用户名，支持多个，用逗号分隔,例如 1234-测试账号35,1345-测试账号56）4、market:eticket (电子凭证商品标记) 5、has_pos:1 (1 表示商品配置线下门店，在detail上进行展示 ，没有或者其他值只不展示)格式是: k1:v2;k2:v2;........ 如：has_pos:1;market:eticket;consume_midmnick:901409638-OPPO;consume_way:4
        :param paimai_info_mode: 拍卖商品选择的拍卖类型，拍卖类型包括三种：增价拍(1)，荷兰拍(2)和降价拍(3)。
        :param paimai_info_deposit: 拍卖宝贝的保证金。对于增价拍和荷兰拍来说保证金有两种模式：淘宝默认模式（首次出价金额的10%），自定义固定保证金（固定冻结金额只能输入不超过30万的正整数），并且保证金只冻结1次。对于降价拍来说保证金只有淘宝默认的（竞拍价格的10% * 竞拍数量），并且每次出价都需要冻结保证金。对于拍卖宝贝来说，保证金是必须的，但是默认使用淘宝默认保证金模式，只有用户需要使用自定义固定保证金的时候才需要使用到这个参数，如果该参数不传或传入0则代表使用默认。
        :param paimai_info_interval: 降价拍宝贝的降价周期(分钟)。降价拍宝贝的价格每隔paimai_info.interval时间会下降一次increment。
        :param paimai_info_reserve: 降价拍宝贝的保留价。对于降价拍来说，paimai_info.reserve必须大于0，且小于price-increment，而且（price-paimai_info.reserve）/increment的计算结果必须为整数
        :param paimai_info_valid_hour: 自定义销售周期的小时数。拍卖宝贝可以自定义销售周期，这里指定销售周期的小时数。注意，该参数只作为输入参数，不能通过taobao.item.get接口获取。
        :param paimai_info_valid_minute: 自定义销售周期的分钟数。拍卖宝贝可以自定义销售周期，这里是指定销售周期的分钟数。自定义销售周期的小时数。拍卖宝贝可以自定义销售周期，这里指定销售周期的小时数。注意，该参数只作为输入参数，不能通过taobao.item.get接口获取。
        :param global_stock_type: 全球购商品采购地（库存类型），有两种库存类型：现货和代购参数值为1时代表现货，值为2时代表代购。注意：使用时请与 全球购商品采购地（地区/国家）配合使用
        :param global_stock_country: 全球购商品采购地（地区/国家）,默认值只在全球购商品采购地（库存类型选择情况生效），地区国家值请填写法定的国家名称，类如（美国, 香港, 日本, 英国, 新西兰, 德国, 韩国, 荷兰, 法国, 意大利, 台湾, 澳门, 加拿大, 瑞士, 西班牙, 泰国, 新加坡, 马来西亚, 菲律宾），不要使用其他
        :param support_custom_made: 是否支持定制市场 true代表支持，false代表支持,如果为空代表与之前保持不变不会修改
        :param custom_made_type_id: 定制工具Id如果支持定制市场，这个值不填写，就用之前的定制工具Id，之前的定制工具Id没有值就默认为-1
        :param global_stock_delivery_place: 全球购商品发货地，发货地现在有两种类型：“国内”和“海外及港澳台”，参数值为1时代表“国内”，值为2时代表“海外及港澳台”，默认为国内。注意：卖家必须已经签署并启用“海外直邮”合约，才能选择发货地为“海外及港澳台”
        :param global_stock_tax_free_promise: 全球购商品卖家包税承诺，当值为true时，代表卖家承诺包税。注意：请与“全球购商品发货地”配合使用，包税承诺必须满足：1、发货地为海外及港澳台 2、卖家已经签署并启用“包税合约”合约
        :param input_custom_cpv: 针对当前商品的自定义属性值，目前是针对销售属性值自定义的，所以调用方需要把自定义属性值对应的虚拟属性值ID（负整数，例如例子中的 -1和-2）像标准属性值值的id一样设置到SKU上，如果自定义属性值有属性值图片，也要设置到属性图片上
        :param cpv_memo: 针对当前商品的标准属性值的补充说明，让买家更加了解商品信息减少交易纠纷
        """
        return self._top_request(
            "taobao.item.add",
            {
                "location.state": location_state,
                "location.city": location_city,
                "num": num,
                "type": type,
                "stuff_status": stuff_status,
                "title": title,
                "desc": desc,
                "cid": cid,
                "sku_spec_ids": sku_spec_ids,
                "sku_delivery_times": sku_delivery_times,
                "sku_hd_length": sku_hd_length,
                "sku_hd_height": sku_hd_height,
                "sku_hd_lamp_quantity": sku_hd_lamp_quantity,
                "input_str": input_str,
                "input_pids": input_pids,
                "sku_properties": sku_properties,
                "sku_quantities": sku_quantities,
                "sku_prices": sku_prices,
                "sku_outer_ids": sku_outer_ids,
                "sku_barcode": sku_barcode,
                "delivery_time.need_delivery_time": delivery_time_need_delivery_time,
                "delivery_time.delivery_time_type": delivery_time_delivery_time_type,
                "delivery_time.delivery_time": delivery_time_delivery_time,
                "ms_payment.reference_price": ms_payment_reference_price,
                "ms_payment.voucher_price": ms_payment_voucher_price,
                "ms_payment.price": ms_payment_price,
                "locality_life.obs": locality_life_obs,
                "locality_life.version": locality_life_version,
                "locality_life.packageid": locality_life_packageid,
                "food_security.prd_license_no": food_security_prd_license_no,
                "food_security.design_code": food_security_design_code,
                "food_security.factory": food_security_factory,
                "food_security.factory_site": food_security_factory_site,
                "food_security.contact": food_security_contact,
                "food_security.mix": food_security_mix,
                "food_security.plan_storage": food_security_plan_storage,
                "food_security.period": food_security_period,
                "food_security.food_additive": food_security_food_additive,
                "food_security.supplier": food_security_supplier,
                "food_security.product_date_start": food_security_product_date_start,
                "food_security.product_date_end": food_security_product_date_end,
                "food_security.stock_date_start": food_security_stock_date_start,
                "food_security.stock_date_end": food_security_stock_date_end,
                "food_security.health_product_no": food_security_health_product_no,
                "price": price,
                "approve_status": approve_status,
                "props": props,
                "freight_payer": freight_payer,
                "valid_thru": valid_thru,
                "has_invoice": has_invoice,
                "has_warranty": has_warranty,
                "has_showcase": has_showcase,
                "seller_cids": seller_cids,
                "has_discount": has_discount,
                "post_fee": post_fee,
                "express_fee": express_fee,
                "ems_fee": ems_fee,
                "list_time": list_time,
                "increment": increment,
                "image": image,
                "postage_id": postage_id,
                "auction_point": auction_point,
                "property_alias": property_alias,
                "lang": lang,
                "outer_id": outer_id,
                "product_id": product_id,
                "pic_path": pic_path,
                "auto_fill": auto_fill,
                "is_taobao": is_taobao,
                "is_ex": is_ex,
                "is_3D": is_3_d,
                "sell_promise": sell_promise,
                "cod_postage_id": cod_postage_id,
                "is_lightning_consignment": is_lightning_consignment,
                "weight": weight,
                "is_xinpin": is_xinpin,
                "sub_stock": sub_stock,
                "scenic_ticket_pay_way": scenic_ticket_pay_way,
                "scenic_ticket_book_cost": scenic_ticket_book_cost,
                "item_size": item_size,
                "item_weight": item_weight,
                "sell_point": sell_point,
                "barcode": barcode,
                "newprepay": newprepay,
                "qualification": qualification,
                "o2o_bind_service": o2o_bind_service,
                "features": features,
                "ignorewarning": ignorewarning,
                "after_sale_id": after_sale_id,
                "change_prop": change_prop,
                "desc_modules": desc_modules,
                "is_offline": is_offline,
                "wireless_desc": wireless_desc,
                "spu_confirm": spu_confirm,
                "video_id": video_id,
                "interactive_id": interactive_id,
                "lease_extends_info": lease_extends_info,
                "brokerage": brokerage,
                "biz_code": biz_code,
                "image_urls": image_urls,
                "locality_life.choose_logis": locality_life_choose_logis,
                "locality_life.expirydate": locality_life_expirydate,
                "locality_life.network_id": locality_life_network_id,
                "locality_life.merchant": locality_life_merchant,
                "locality_life.verification": locality_life_verification,
                "locality_life.refund_ratio": locality_life_refund_ratio,
                "locality_life.onsale_auto_refund_ratio": locality_life_onsale_auto_refund_ratio,
                "locality_life.refundmafee": locality_life_refundmafee,
                "locality_life.eticket": locality_life_eticket,
                "paimai_info.mode": paimai_info_mode,
                "paimai_info.deposit": paimai_info_deposit,
                "paimai_info.interval": paimai_info_interval,
                "paimai_info.reserve": paimai_info_reserve,
                "paimai_info.valid_hour": paimai_info_valid_hour,
                "paimai_info.valid_minute": paimai_info_valid_minute,
                "global_stock_type": global_stock_type,
                "global_stock_country": global_stock_country,
                "support_custom_made": support_custom_made,
                "custom_made_type_id": custom_made_type_id,
                "global_stock_delivery_place": global_stock_delivery_place,
                "global_stock_tax_free_promise": global_stock_tax_free_promise,
                "input_custom_cpv": input_custom_cpv,
                "cpv_memo": cpv_memo
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_img_upload(
            self,
            num_iid,
            id='',
            position='',
            image='',
            is_major='false',
            is_rectangle='false'
    ):
        """
        添加商品图片
        添加一张商品图片到num_iid指定的商品中
        传入的num_iid所对应的商品必须属于当前会话的用户
        如果更新图片需要设置itemimg_id，且该itemimg_id的图片记录需要属于传入的num_iid对应的商品。如果新增图片则不用设置 。
        使用taobao.item.seller.get中返回的item_imgs字段获取图片id。
        商品图片有数量和大小上的限制，根据卖家享有的服务（如：卖家订购了多图服务等），商品图片数量限制不同。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23

        :param num_iid: 商品数字ID，该参数必须
        :param id: 商品图片id(如果是更新图片，则需要传该参数)
        :param position: 图片序号
        :param image: 商品图片内容类型:JPG,GIF;最大:3M 。支持的文件类型：gif,jpg,jpeg,png
        :param is_major: 是否将该图片设为主图,可选值:true,false;默认值:false(非主图)
        :param is_rectangle: 是否3:4长方形图片，绑定3:4主图视频时用于上传3:4商品主图
        """
        return self._top_request(
            "taobao.item.img.upload",
            {
                "num_iid": num_iid,
                "id": id,
                "position": position,
                "image": image,
                "is_major": is_major,
                "is_rectangle": is_rectangle
            },
            result_processor=lambda x: x["item_img"]
        )

    def taobao_item_img_delete(
            self,
            num_iid,
            id,
            is_sixth_pic='false'
    ):
        """
        删除商品图片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24

        :param num_iid: 商品数字ID
        :param id: 商品图片ID；如果是竖图，请将id的值设置为1
        :param is_sixth_pic: 标记是否要删除第6张图，因为第6张图与普通商品图片不是存储在同一个位置的无图片ID，所以要通过一个标记来判断是否为第6张图，目前第6张图业务主要用在女装业务下
        """
        return self._top_request(
            "taobao.item.img.delete",
            {
                "num_iid": num_iid,
                "id": id,
                "is_sixth_pic": is_sixth_pic
            },
            result_processor=lambda x: x["item_img"]
        )

    def taobao_item_propimg_delete(
            self,
            id,
            num_iid
    ):
        """
        删除属性图片
        删除propimg_id 所指定的商品属性图片 <br/>传入的num_iid所对应的商品必须属于当前会话的用户 <br/>propimg_id对应的属性图片需要属于num_iid对应的商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25

        :param id: 商品属性图片ID
        :param num_iid: 商品数字ID，必选
        """
        return self._top_request(
            "taobao.item.propimg.delete",
            {
                "id": id,
                "num_iid": num_iid
            },
            result_processor=lambda x: x["prop_img"]
        )

    def taobao_item_propimg_upload(
            self,
            num_iid,
            properties,
            image='',
            id='',
            position='0'
    ):
        """
        添加或修改属性图片
        添加一张商品属性图片到num_iid指定的商品中 <br/>传入的num_iid所对应的商品必须属于当前会话的用户 <br/>图片的属性必须要是颜色的属性，这个在前台显示的时候需要和sku进行关联的 <br/>商品属性图片只有享有服务的卖家（如：淘宝大卖家、订购了淘宝多图服务的卖家）才能上传 <br/>商品属性图片有数量和大小上的限制，最多不能超过24张（每个颜色属性都有一张）。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26

        :param num_iid: 商品数字ID，必选
        :param properties: 属性列表。调用taobao.itemprops.get获取类目属性，属性必须是颜色属性，再用taobao.itempropvalues.get取得vid。格式:pid:vid。
        :param image: 属性图片内容。类型:JPG,GIF;图片大小不超过:3M
        :param id: 属性图片ID。如果是新增不需要填写
        :param position: 图片位置
        """
        return self._top_request(
            "taobao.item.propimg.upload",
            {
                "num_iid": num_iid,
                "properties": properties,
                "image": image,
                "id": id,
                "position": position
            },
            result_processor=lambda x: x["prop_img"]
        )

    def taobao_item_sku_add(
            self,
            num_iid,
            properties,
            quantity,
            price,
            outer_id='',
            item_price='',
            lang='zh_CN',
            ignorewarning=''
    ):
        """
        添加SKU
        新增一个sku到num_iid指定的商品中 <br/>传入的iid所对应的商品必须属于当前会话的用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27

        :param num_iid: Sku所属商品数字id，可通过 taobao.item.get 获取。必选
        :param properties: Sku属性串。格式:pid:vid;pid:vid,如:1627207:3232483;1630696:3284570,表示:机身颜色:军绿色;手机套餐:一电一充。
        :param quantity: Sku的库存数量。sku的总数量应该小于等于商品总数量(Item的NUM)。取值范围:大于零的整数
        :param price: Sku的销售价格。商品的价格要在商品所有的sku的价格之间。精确到2位小数;单位:元。如:200.07，表示:200元7分
        :param outer_id: Sku的商家外部id
        :param item_price: sku所属商品的价格。当用户新增sku，使商品价格不属于sku价格之间的时候，用于修改商品的价格，使sku能够添加成功
        :param lang: Sku文字的版本。可选值:zh_HK(繁体),zh_CN(简体);默认值:zh_CN
        :param ignorewarning: 忽略警告提示.
        """
        return self._top_request(
            "taobao.item.sku.add",
            {
                "num_iid": num_iid,
                "properties": properties,
                "quantity": quantity,
                "price": price,
                "outer_id": outer_id,
                "item_price": item_price,
                "lang": lang,
                "ignorewarning": ignorewarning
            },
            result_processor=lambda x: x["sku"]
        )

    def taobao_item_sku_get(
            self,
            fields,
            sku_id,
            num_iid=''
    ):
        """
        获取SKU
        获取sku_id所对应的sku数据 <br/>sku_id对应的sku要属于传入的nick对应的卖家
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28

        :param fields: 需返回的字段列表。可选值：Sku结构体中的所有字段；字段之间用“,”分隔。
        :param sku_id: Sku的id。可以通过taobao.item.get得到
        :param num_iid: 商品的数字IID（num_iid和nick必传一个，推荐用num_iid），传商品的数字id返回的结果里包含cspu（SKu上的产品规格信息）。
        """
        return self._top_request(
            "taobao.item.sku.get",
            {
                "fields": fields,
                "sku_id": sku_id,
                "num_iid": num_iid
            },
            result_processor=lambda x: x["sku"]
        )

    def taobao_item_sku_update(
            self,
            num_iid,
            properties,
            quantity='',
            price='',
            outer_id='',
            item_price='',
            lang='zh_CN',
            ignorewarning=''
    ):
        """
        更新SKU信息
        *更新一个sku的数据 <br/>*需要更新的sku通过属性properties进行匹配查找 <br/>*商品的数量和价格必须大于等于0 <br/>*sku记录会更新到指定的num_iid对应的商品中 <br/>*num_iid对应的商品必须属于当前的会话用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29

        :param num_iid: Sku所属商品数字id，可通过 taobao.item.get 获取
        :param properties: Sku属性串。格式:pid:vid;pid:vid,如: 1627207:3232483;1630696:3284570,表示机身颜色:军绿色;手机套餐:一电一充。<br/>如果包含自定义属性，则格式为pid:vid;pid2:vid2;$pText:vText , 其中$pText:vText为自定义属性。限制：其中$pText的’$’前缀不能少，且pText和vText文本中不可以存在 冒号:和分号;以及逗号，
        :param quantity: Sku的库存数量。sku的总数量应该小于等于商品总数量(Item的NUM)，sku数量变化后item的总数量也会随着变化。取值范围:大于等于零的整数
        :param price: Sku的销售价格。精确到2位小数;单位:元。如:200.07，表示:200元7分。修改后的sku价格要保证商品的价格在所有sku价格所形成的价格区间内（例如：商品价格为6元，sku价格有5元、10元两种，如果要修改5元sku的价格，那么修改的范围只能是0-6元之间；如果要修改10元的sku，那么修改的范围只能是6到无穷大的区间中）
        :param outer_id: Sku的商家外部id
        :param item_price: sku所属商品的价格。当用户更新sku，使商品价格不属于sku价格之间的时候，用于修改商品的价格，使sku能够更新成功
        :param lang: Sku文字的版本。可选值:zh_HK(繁体),zh_CN(简体);默认值:zh_CN
        :param ignorewarning: 忽略警告提示.
        """
        return self._top_request(
            "taobao.item.sku.update",
            {
                "num_iid": num_iid,
                "properties": properties,
                "quantity": quantity,
                "price": price,
                "outer_id": outer_id,
                "item_price": item_price,
                "lang": lang,
                "ignorewarning": ignorewarning
            },
            result_processor=lambda x: x["sku"]
        )

    def taobao_item_skus_get(
            self,
            fields,
            num_iids
    ):
        """
        根据商品ID列表获取SKU信息
        * 获取多个商品下的所有sku
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30

        :param fields: 需返回的字段列表。可选值：Sku结构体中的所有字段；字段之间用“,”分隔。
        :param num_iids: sku所属商品数字id，必选。num_iid个数不能超过40个
        """
        return self._top_request(
            "taobao.item.skus.get",
            {
                "fields": fields,
                "num_iids": num_iids
            },
            result_processor=lambda x: x["skus"]
        )

    def taobao_item_update_delisting(
            self,
            num_iid
    ):
        """
        商品下架
        * 单个商品下架<br/>    * 输入的num_iid必须属于当前会话用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31

        :param num_iid: 商品数字ID，该参数必须
        """
        return self._top_request(
            "taobao.item.update.delisting",
            {
                "num_iid": num_iid
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_update_listing(
            self,
            num_iid,
            num
    ):
        """
        一口价商品上架
        * 单个商品上架<br/>* 输入的num_iid必须属于当前会话用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32

        :param num_iid: 商品数字ID，该参数必须
        :param num: 需要上架的商品的数量。取值范围:大于零的整数。如果商品有sku，则上架数量默认为所有sku数量总和，不可修改。否则商品数量根据设置数量调整为num
        """
        return self._top_request(
            "taobao.item.update.listing",
            {
                "num_iid": num_iid,
                "num": num
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_delete(
            self,
            num_iid
    ):
        """
        删除单条商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=112

        :param num_iid: 商品数字ID，该参数必须
        """
        return self._top_request(
            "taobao.item.delete",
            {
                "num_iid": num_iid
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_joint_img(
            self,
            num_iid,
            pic_path,
            id='0',
            is_major='false',
            position='0',
            is_rectangle='false'
    ):
        """
        商品关联子图
        * 关联一张商品图片到num_iid指定的商品中<br/>    * 传入的num_iid所对应的商品必须属于当前会话的用户<br/>    * 商品图片关联在卖家身份和图片来源上的限制，卖家要是B卖家或订购了多图服务才能关联图片，并且图片要来自于卖家自己的图片空间才行<br/>    * 商品图片数量有限制。不管是上传的图片还是关联的图片，他们的总数不能超过一定限额
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=147

        :param num_iid: 商品数字ID，必选
        :param pic_path: 图片URL,图片空间图片的相对地址
        :param id: 商品图片id(如果是更新图片，则需要传该参数)
        :param is_major: 上传的图片是否关联为商品主图
        :param position: 图片序号
        :param is_rectangle: 是否3:4长方形图片，绑定3:4主图视频时用于上传3:4商品主图
        """
        return self._top_request(
            "taobao.item.joint.img",
            {
                "num_iid": num_iid,
                "pic_path": pic_path,
                "id": id,
                "is_major": is_major,
                "position": position,
                "is_rectangle": is_rectangle
            },
            result_processor=lambda x: x["item_img"]
        )

    def taobao_item_joint_propimg(
            self,
            properties,
            pic_path,
            num_iid,
            id='0',
            position='0'
    ):
        """
        商品关联属性图
        * 关联一张商品属性图片到num_iid指定的商品中<br/>    * 传入的num_iid所对应的商品必须属于当前会话的用户<br/>    * 图片的属性必须要是颜色的属性，这个在前台显示的时候需要和sku进行关联的<br/>    * 商品图片关联在卖家身份和图片来源上的限制，卖家要是B卖家或订购了多图服务才能关联图片，并且图片要来自于卖家自己的图片空间才行<br/>    * 商品图片数量有限制。不管是上传的图片还是关联的图片，他们的总数不能超过一定限额，最多不能超过24张（每个颜色属性都有一张）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=148

        :param properties: 属性列表。调用taobao.itemprops.get获取，属性必须是颜色属性，格式:pid:vid。
        :param pic_path: 图片地址(传入图片相对地址即可,即不需包含 http://img02.taobao.net/bao/uploaded )
        :param num_iid: 商品数字ID，必选
        :param id: 属性图片ID。如果是新增不需要填写
        :param position: 图片序号
        """
        return self._top_request(
            "taobao.item.joint.propimg",
            {
                "properties": properties,
                "pic_path": pic_path,
                "num_iid": num_iid,
                "id": id,
                "position": position
            },
            result_processor=lambda x: x["prop_img"]
        )

    def taobao_items_inventory_get(
            self,
            fields,
            q='',
            banner='',
            cid='',
            seller_cids='',
            page_no='1',
            page_size='40',
            has_discount='',
            order_by='list_time:desc',
            is_taobao='',
            is_ex='',
            start_modified='',
            end_modified='',
            auction_type=''
    ):
        """
        得到当前会话用户库存中的商品列表
        获取当前用户作为卖家的仓库中的商品列表，并能根据传入的搜索条件对仓库中的商品列表进行过滤
        只能获得商品的部分信息，商品的详细信息请通过taobao.item.seller.get获取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=162

        :param fields: 需返回的字段列表。可选值为 Item 商品结构体中的以下字段：
                       approve_status,num_iid,title,nick,type,cid,pic_url,num,props,valid_thru, list_time,price,has_discount,has_invoice,has_warranty,has_showcase, modified,delist_time,postage_id,seller_cids,outer_id；字段之间用“,”分隔。<br>
                       不支持其他字段，如果需要获取其他字段数据，调用taobao.item.seller.get。
        :param q: 搜索字段。搜索商品的title。
        :param banner: 分类字段。可选值:<br>
                       regular_shelved(定时上架)<br>
                       never_on_shelf(从未上架)<br>
                       off_shelf(我下架的)<br>
                       <font color='red'>for_shelved(等待所有上架)<br>
                       sold_out(全部卖完)<br>
                       violation_off_shelf(违规下架的)<br>
                       默认查询for_shelved(等待所有上架)这个状态的商品<br></font>
                       注：for_shelved(等待所有上架)=regular_shelved(定时上架)+never_on_shelf(从未上架)+off_shelf(我下架的)
        :param cid: 商品类目ID。ItemCat中的cid字段。可以通过taobao.itemcats.get取到
        :param seller_cids: 卖家店铺内自定义类目ID。多个之间用“,”分隔。可以根据taobao.sellercats.list.get获得.(<font color='red'>注：目前最多支持32个ID号传入</font>)
        :param page_no: 页码。取值范围:大于零小于等于101的整数;默认值为1，即返回第一页数据。当页码超过101页时系统就会报错，故请大家在用此接口获取数据时尽可能的细化自己的搜索条件，例如根据修改时间分段获取商品。
        :param page_size: 每页条数。取值范围:大于零的整数;最大值：200；默认值：40。
        :param has_discount: 是否参与会员折扣。可选值：true，false。默认不过滤该条件
        :param order_by: 排序方式。格式为column:asc/desc ，column可选值:list_time(上架时间),delist_time(下架时间),num(商品数量)，modified(最近修改时间);默认上架时间降序(即最新上架排在前面)。如按照上架时间降序排序方式为list_time:desc
        :param is_taobao: 商品是否在淘宝显示
        :param is_ex: 商品是否在外部网店显示
        :param start_modified: 商品起始修改时间
        :param end_modified: 商品结束修改时间
        :param auction_type: 商品类型：a-拍卖,b-一口价
        """
        return self._top_request(
            "taobao.items.inventory.get",
            {
                "fields": fields,
                "q": q,
                "banner": banner,
                "cid": cid,
                "seller_cids": seller_cids,
                "page_no": page_no,
                "page_size": page_size,
                "has_discount": has_discount,
                "order_by": order_by,
                "is_taobao": is_taobao,
                "is_ex": is_ex,
                "start_modified": start_modified,
                "end_modified": end_modified,
                "auction_type": auction_type
            }
        )

    def taobao_items_custom_get(
            self,
            outer_id,
            fields
    ):
        """
        根据外部ID取商品
        跟据卖家设定的商品外部id获取商品 <br/>这个商品对应卖家从传入的session中获取，需要session绑定
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=163

        :param outer_id: 商品的外部商品ID，支持批量，最多不超过40个。
        :param fields: 需返回的字段列表，参考：Item商品结构体说明，其中barcode、sku.barcode等条形码字段暂不支持；多个字段之间用“,”分隔。
        """
        return self._top_request(
            "taobao.items.custom.get",
            {
                "outer_id": outer_id,
                "fields": fields
            },
            result_processor=lambda x: x["items"]
        )

    def taobao_skus_custom_get(
            self,
            outer_id,
            fields
    ):
        """
        根据外部ID取商品SKU
        跟据卖家设定的Sku的外部id获取商品，如果一个outer_id对应多个Sku会返回所有符合条件的sku <br/>这个Sku所属卖家从传入的session中获取，需要session绑定(注：iid标签里是num_iid的值，可以用作num_iid使用)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=164

        :param outer_id: Sku的外部商家ID
        :param fields: 需返回的字段列表。可选值：Sku结构体中的所有字段；字段之间用“,”隔开
        """
        return self._top_request(
            "taobao.skus.custom.get",
            {
                "outer_id": outer_id,
                "fields": fields
            },
            result_processor=lambda x: x["skus"]
        )

    def taobao_item_sku_delete(
            self,
            num_iid,
            properties,
            item_price='',
            item_num='',
            lang='zh_CN',
            ignorewarning=''
    ):
        """
        删除SKU
        删除一个sku的数据<br/>需要删除的sku通过属性properties进行匹配查找
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=314

        :param num_iid: Sku所属商品数字id，可通过 taobao.item.get 获取。必选
        :param properties: Sku属性串。格式:pid:vid;pid:vid,如: 1627207:3232483;1630696:3284570,表示机身颜色:军绿色;手机套餐:一电一充
        :param item_price: sku所属商品的价格。当用户删除sku，使商品价格不属于sku价格之间的时候，用于修改商品的价格，使sku能够删除成功
        :param item_num: sku所属商品的数量,大于0的整数。当用户删除sku，使商品数量不等于sku数量之和时候，用于修改商品的数量，使sku能够删除成功。特别是删除最后一个sku的时候，一定要设置商品数量到正常的值，否则删除失败
        :param lang: Sku文字的版本。可选值:zh_HK(繁体),zh_CN(简体);默认值:zh_CN
        :param ignorewarning: 忽略警告提示.
        """
        return self._top_request(
            "taobao.item.sku.delete",
            {
                "num_iid": num_iid,
                "properties": properties,
                "item_price": item_price,
                "item_num": item_num,
                "lang": lang,
                "ignorewarning": ignorewarning
            },
            result_processor=lambda x: x["sku"]
        )

    def taobao_aftersale_get(
            self
    ):
        """
        查询用户售后服务模板
        查询用户设置的售后服务模板，仅返回标题和id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10448

        """
        return self._top_request(
            "taobao.aftersale.get",
            result_processor=lambda x: x["after_sales"]
        )

    def taobao_item_quantity_update(
            self,
            num_iid,
            quantity,
            sku_id='',
            outer_id='',
            type='1'
    ):
        """
        宝贝/SKU库存修改
        提供按照全量或增量形式修改宝贝/SKU库存的功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10591

        :param num_iid: 商品数字ID，必填参数
        :param quantity: 库存修改值，必选。当全量更新库存时，quantity必须为大于等于0的正整数；当增量更新库存时，quantity为整数，可小于等于0。若增量更新时传入的库存为负数，则负数与实际库存之和不能小于0。比如当前实际库存为1，传入增量更新quantity=-1，库存改为0
        :param sku_id: 要操作的SKU的数字ID，可选。如果不填默认修改宝贝的库存，如果填上则修改该SKU的库存
        :param outer_id: SKU的商家编码，可选参数。如果不填则默认修改宝贝的库存，如果填了则按照商家编码搜索出对应的SKU并修改库存。当sku_id和本字段都填写时以sku_id为准搜索对应SKU
        :param type: 库存更新方式，可选。1为全量更新，2为增量更新。如果不填，默认为全量更新
        """
        return self._top_request(
            "taobao.item.quantity.update",
            {
                "num_iid": num_iid,
                "quantity": quantity,
                "sku_id": sku_id,
                "outer_id": outer_id,
                "type": type
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_templates_get(
            self
    ):
        """
        获取用户宝贝详情页模板名称
        查询当前登录用户的店铺的宝贝详情页的模板名称
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10704

        """
        return self._top_request(
            "taobao.item.templates.get",
            result_processor=lambda x: x["item_template_list"]
        )

    def taobao_item_price_update(
            self,
            num_iid,
            location_state='',
            location_city='',
            cid='',
            props='',
            num='',
            price='',
            title='',
            desc='',
            post_fee='',
            express_fee='',
            ems_fee='',
            list_time='',
            increment='',
            image='',
            stuff_status='',
            auction_point='',
            property_alias='',
            seller_cids='',
            postage_id='',
            outer_id='',
            product_id='',
            pic_path='',
            auto_fill='',
            is_taobao='',
            is_ex='',
            is_3_d='',
            is_replace_sku='',
            lang='',
            has_discount='',
            has_showcase='',
            approve_status='',
            freight_payer='',
            valid_thru='',
            has_invoice='',
            has_warranty='',
            sell_promise='',
            cod_postage_id='',
            is_lightning_consignment='',
            weight='',
            is_xinpin='',
            sub_stock='0',
            ignorewarning='',
            input_pids='',
            sku_quantities='',
            sku_prices='',
            sku_properties='',
            sku_outer_ids='',
            input_str=''
    ):
        """
        更新商品价格
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10927

        :param num_iid: 商品数字ID，该参数必须
        :param location_state: 所在地省份。如浙江 具体可以下载http://dl.open.taobao.com/sdk/商品城市列表.rar 取到
        :param location_city: 所在地城市。如杭州 具体可以下载http://dl.open.taobao.com/sdk/商品城市列表.rar 取到
        :param cid: 叶子类目id
        :param props: 商品属性列表。格式:pid:vid;pid:vid。属性的pid调用taobao.itemprops.get取得，属性值的vid用taobao.itempropvalues.get取得vid。 如果该类目下面没有属性，可以不用填写。如果有属性，必选属性必填，其他非必选属性可以选择不填写.属性不能超过35对。所有属性加起来包括分割符不能超过549字节，单个属性没有限制。 如果有属性是可输入的话，则用字段input_str填入属性的值。
        :param num: 商品数量，取值范围:0-999999的整数。且需要等于Sku所有数量的和
        :param price: 商品价格。取值范围:0-100000000;精确到2位小数;单位:元。如:200.07，表示:200元7分。需要在正确的价格区间内。
        :param title: 宝贝标题. 不能超过60字符,受违禁词控制
        :param desc: 商品描述. 字数要大于5个字符，小于25000个字符 ，受违禁词控制
        :param post_fee: 平邮费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:5.07，表示:5元7分, 注:post_fee,express_fee,ems_fee需一起填写
        :param express_fee: 快递费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:15.07，表示:15元7分
        :param ems_fee: ems费用。取值范围:0.01-999.00;精确到2位小数;单位:元。如:25.07，表示:25元7分
        :param list_time: 上架时间。不论是更新架下的商品还是出售中的商品，如果这个字段小于当前时间则直接上架商品，并且上架的时间为更新商品的时间，此时item.is_timing为false，如果大于当前时间则宝贝会下架进入定时上架的宝贝中。
        :param increment: 加价幅度 如果为0，代表系统代理幅度
        :param image: 商品图片。类型:JPG,GIF;最大长度:500k
        :param stuff_status: 商品新旧程度。可选值:new（全新）,unused（闲置）,second（二手）。
        :param auction_point: 商品的积分返点比例。如：5 表示返点比例0.5%. 注意：返点比例必须是>0的整数，而且最大是90,即为9%.B商家在发布非虚拟商品时，返点必须是 5的倍数，即0.5%的倍数。其它是1的倍数，即0.1%的倍数。无名良品商家发布商品时，复用该字段记录积分宝返点比例，返点必须是对应类目的返点步长的整数倍，默认是5，即0.5%。注意此时该字段值依旧必须是>0的整数，注意此时该字段值依旧必须是>0的整数，最高值不超过500，即50%
        :param property_alias: 属性值别名。如pid:vid:别名;pid1:vid1:别名1， pid:属性id vid:值id。总长度不超过512字节
        :param seller_cids: 重新关联商品与店铺类目，结构:',cid1,cid2,...,'，如果店铺类目存在二级类目，必须传入子类目cids。
        :param postage_id: 宝贝所属的运费模板ID。取值范围：整数且必须是该卖家的运费模板的ID（可通过taobao.postages.get获得当前会话用户的所有邮费模板）
        :param outer_id: 商家编码
        :param product_id: 商品所属的产品ID(B商家发布商品需要用)
        :param pic_path: 商品主图需要关联的图片空间的相对url。这个url所对应的图片必须要属于当前用户。pic_path和image只需要传入一个,如果两个都传，默认选择pic_path
        :param auto_fill: 代充商品类型。只有少数类目下的商品可以标记上此字段，具体哪些类目可以上传可以通过taobao.itemcat.features.get获得。在代充商品的类目下，不传表示不标记商品类型（交易搜索中就不能通过标记搜到相关的交易了）。可选类型： <br/>no_mark(不做类型标记) <br/>time_card(点卡软件代充) <br/>fee_card(话费软件代充)
        :param is_taobao: 是否在淘宝上显示
        :param is_ex: 是否在外店显示
        :param is_3_d: 是否是3D
        :param is_replace_sku: 是否替换sku
        :param lang: 商品文字的版本，繁体传入”zh_HK”，简体传入”zh_CN”
        :param has_discount: 支持会员打折。可选值:true,false;
        :param has_showcase: 橱窗推荐。可选值:true,false;
        :param approve_status: 商品上传后的状态。可选值:onsale（出售中）,instock（库中），如果同时更新商品状态为出售中及list_time为将来的时间，则商品还是处于定时上架的状态, 此时item.is_timing为true
        :param freight_payer: 运费承担方式。运费承担方式。可选值:seller（卖家承担）,buyer(买家承担);
        :param valid_thru: 有效期。可选值:7,14;单位:天;
        :param has_invoice: 是否有发票。可选值:true,false (商城卖家此字段必须为true)
        :param has_warranty: 是否有保修。可选值:true,false;
        :param sell_promise: 是否承诺退换货服务!虚拟商品无须设置此项!
        :param cod_postage_id: 货到付款运费模板ID
        :param is_lightning_consignment: 实物闪电发货。注意：在售的闪电发货产品不允许取消闪电发货，需要先下架商品才能取消闪电发货标记
        :param weight: 商品的重量(商超卖家专用字段)
        :param is_xinpin: 商品是否为新品。只有在当前类目开通新品,并且当前用户拥有该类目下发布新品权限时才能设置is_xinpin为true，否则设置true后会返回错误码:isv.invalid-permission:xinpin。同时只有一口价全新的宝贝才能设置为新品，否则会返回错误码：isv.invalid-parameter:xinpin。不设置参数就保持原有值。
        :param sub_stock: 商品是否支持拍下减库存:1支持;2取消支持(付款减库存);0(默认)不更改 集市卖家默认拍下减库存; 商城卖家默认付款减库存
        :param ignorewarning: 忽略警告提示.
        :param input_pids: 用户自行输入的类目属性ID串，结构：'pid1,pid2,pid3'，如：'20000'（表示品牌） 注：通常一个类目下用户可输入的关键属性不超过1个。
        :param sku_quantities: 更新的Sku的数量串，结构如：num1,num2,num3 如:2,3,4
        :param sku_prices: 更新的Sku的价格串，结构如：10.00,5.00,… 精确到2位小数;单位:元。如:200.07，表示:200元7分
        :param sku_properties: 更新的Sku的属性串，调用taobao.itemprops.get获取类目属性，如果属性是销售属性，再用taobao.itempropvalues.get取得vid。格式:pid:vid;pid:vid。该字段内的销售属性也需要在props字段填写 。如果更新时有对Sku进行操作，则Sku的properties一定要传入。
        :param sku_outer_ids: Sku的外部id串，结构如：1234,1342,… sku_properties, sku_quantities, sku_prices, sku_outer_ids在输入数据时要一一对应，如果没有sku_outer_ids也要写上这个参数，入参是','(这个是两个sku的示列，逗号数应该是sku个数减1)；该参数最大长度是512个字节
        :param input_str: 用户自行输入的子属性名和属性值，结构:'父属性值;一级子属性名;一级子属性值;二级子属性名;自定义输入值,....',如：“耐克;耐克系列;科比系列;科比系列;2K5,Nike乔丹鞋;乔丹系列;乔丹鞋系列;乔丹鞋系列;json5”，多个自定义属性用','分割，input_str需要与input_pids一一对应，注：通常一个类目下用户可输入的关键属性不超过1个。所有属性别名加起来不能超过3999字节。此处不可以使用“其他”、“其它”和“其她”这三个词。
        """
        return self._top_request(
            "taobao.item.price.update",
            {
                "num_iid": num_iid,
                "location.state": location_state,
                "location.city": location_city,
                "cid": cid,
                "props": props,
                "num": num,
                "price": price,
                "title": title,
                "desc": desc,
                "post_fee": post_fee,
                "express_fee": express_fee,
                "ems_fee": ems_fee,
                "list_time": list_time,
                "increment": increment,
                "image": image,
                "stuff_status": stuff_status,
                "auction_point": auction_point,
                "property_alias": property_alias,
                "seller_cids": seller_cids,
                "postage_id": postage_id,
                "outer_id": outer_id,
                "product_id": product_id,
                "pic_path": pic_path,
                "auto_fill": auto_fill,
                "is_taobao": is_taobao,
                "is_ex": is_ex,
                "is_3D": is_3_d,
                "is_replace_sku": is_replace_sku,
                "lang": lang,
                "has_discount": has_discount,
                "has_showcase": has_showcase,
                "approve_status": approve_status,
                "freight_payer": freight_payer,
                "valid_thru": valid_thru,
                "has_invoice": has_invoice,
                "has_warranty": has_warranty,
                "sell_promise": sell_promise,
                "cod_postage_id": cod_postage_id,
                "is_lightning_consignment": is_lightning_consignment,
                "weight": weight,
                "is_xinpin": is_xinpin,
                "sub_stock": sub_stock,
                "ignorewarning": ignorewarning,
                "input_pids": input_pids,
                "sku_quantities": sku_quantities,
                "sku_prices": sku_prices,
                "sku_properties": sku_properties,
                "sku_outer_ids": sku_outer_ids,
                "input_str": input_str
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_sku_price_update(
            self,
            num_iid,
            properties,
            quantity='',
            price='',
            outer_id='',
            item_price='',
            lang='zh_CN',
            ignorewarning=''
    ):
        """
        更新商品SKU的价格
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10928

        :param num_iid: Sku所属商品数字id，可通过 taobao.item.get 获取
        :param properties: Sku属性串。格式:pid:vid;pid:vid,如: 1627207:3232483;1630696:3284570,表示机身颜色:军绿色;手机套餐:一电一充
        :param quantity: Sku的库存数量。sku的总数量应该小于等于商品总数量(Item的NUM)，sku数量变化后item的总数量也会随着变化。取值范围:大于等于零的整数
        :param price: Sku的销售价格。精确到2位小数;单位:元。如:200.07，表示:200元7分。修改后的sku价格要保证商品的价格在所有sku价格所形成的价格区间内（例如：商品价格为6元，sku价格有5元、10元两种，如果要修改5元sku的价格，那么修改的范围只能是0-6元之间；如果要修改10元的sku，那么修改的范围只能是6到无穷大的区间中）
        :param outer_id: Sku的商家外部id
        :param item_price: sku所属商品的价格。当用户更新sku，使商品价格不属于sku价格之间的时候，用于修改商品的价格，使sku能够更新成功
        :param lang: Sku文字的版本。可选值:zh_HK(繁体),zh_CN(简体);默认值:zh_CN
        :param ignorewarning: 忽略警告提示.
        """
        return self._top_request(
            "taobao.item.sku.price.update",
            {
                "num_iid": num_iid,
                "properties": properties,
                "quantity": quantity,
                "price": price,
                "outer_id": outer_id,
                "item_price": item_price,
                "lang": lang,
                "ignorewarning": ignorewarning
            },
            result_processor=lambda x: x["sku"]
        )

    def taobao_ump_promotion_get(
            self,
            item_id
    ):
        """
        商品优惠详情查询
        商品优惠详情查询，可查询商品设置的详细优惠。包括限时折扣，满就送等官方优惠以及第三方优惠。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11039

        :param item_id: 商品id
        """
        return self._top_request(
            "taobao.ump.promotion.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["promotions"]
        )

    def taobao_skus_quantity_update(
            self,
            num_iid,
            type='1',
            skuid_quantities='',
            outerid_quantities=''
    ):
        """
        SKU库存修改
        提供按照全量/增量的方式批量修改SKU库存的功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21169

        :param num_iid: 商品数字ID，必填参数
        :param type: 库存更新方式，可选。1为全量更新，2为增量更新。如果不填，默认为全量更新。当选择全量更新时，如果库存更新值传入的是负数，会出错并返回错误码；当选择增量更新时，如果库存更新值为负数且绝对值大于当前库存，则sku库存会设置为0.
        :param skuid_quantities: sku库存批量修改入参，用于指定一批sku和每个sku的库存修改值，特殊可填。格式为skuId:库存修改值;skuId:库存修改值。最多支持20个SKU同时修改。
        :param outerid_quantities: 特殊可选，skuIdQuantities为空的时候用该字段通过outerId来指定sku和其库存修改值。格式为outerId:库存修改值;outerId:库存修改值。当skuIdQuantities不为空的时候该字段失效。当一个outerId对应多个sku时，所有匹配到的sku都会被修改库存。最多支持20个SKU同时修改。
        """
        return self._top_request(
            "taobao.skus.quantity.update",
            {
                "num_iid": num_iid,
                "type": type,
                "skuid_quantities": skuid_quantities,
                "outerid_quantities": outerid_quantities
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_item_anchor_get(
            self,
            type,
            cat_id
    ):
        """
        获取可用宝贝描述规范化模块
        根据类目id和宝贝描述规范化打标类型获取该类目可用的宝贝描述模块中的锚点
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21433

        :param type: 宝贝模板类型是人工打标还是自动打标：人工打标为1，自动打标为0.人工和自动打标为-1.(最小值：-1，最大值：1)
        :param cat_id: 对应类目编号
        """
        return self._top_request(
            "taobao.item.anchor.get",
            {
                "type": type,
                "cat_id": cat_id
            }
        )

    def tmall_item_desc_modules_get(
            self,
            cat_id,
            usr_id
    ):
        """
        商品描述模块信息获取
        商品描述模块信息获取，包括运营设定的类目级别的模块信息以及用户自定义模块数量约束。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22171

        :param cat_id: 淘宝后台发布商品的叶子类目id，可通过taobao.itemcats.get查到。api 访问地址http://api.taobao.com/apidoc/api.htm?spm=0.0.0.0.CFhhk4&path=cid:3-apiId:122
        :param usr_id: 商家主帐号id
        """
        return self._top_request(
            "tmall.item.desc.modules.get",
            {
                "cat_id": cat_id,
                "usr_id": usr_id
            },
            result_processor=lambda x: x["modular_desc_info"]
        )

    def taobao_item_add_rules_get(
            self,
            category_id
    ):
        """
        商品发布规则信息获取接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        在新的发布模式下，isv需要先获取一份发布规则，然后根据规则传入数据。该接口提供规则查询功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22651

        :param category_id: 发布宝贝的叶子类目id
        """
        return self._top_request(
            "taobao.item.add.rules.get",
            {
                "category_id": category_id
            },
            result_processor=lambda x: x["add_rules"]
        )

    def taobao_item_barcode_update(
            self,
            item_id,
            item_barcode='',
            sku_ids='',
            sku_barcodes='',
            isforce='false',
            src=''
    ):
        """
        更新商品条形码信息
        通过该接口，将商品以及SKU上得条形码信息补全
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22894

        :param item_id: 被更新商品的ID
        :param item_barcode: 商品条形码，如果不用更新，可选择不填
        :param sku_ids: 被更新SKU的ID列表，中间以英文逗号进行分隔。如果没有SKU或者不需要更新SKU的条形码，不需要设置
        :param sku_barcodes: SKU维度的条形码，和sku_ids字段一一对应，中间以英文逗号分隔
        :param isforce: 是否强制保存商品条码。true：强制保存false ：需要执行条码库校验
        :param src: 访问来源，这字段提供给千牛扫码枪用，其他调用方，不需要填写
        """
        return self._top_request(
            "taobao.item.barcode.update",
            {
                "item_id": item_id,
                "item_barcode": item_barcode,
                "sku_ids": sku_ids,
                "sku_barcodes": sku_barcodes,
                "isforce": isforce,
                "src": src
            },
            result_processor=lambda x: x["item"]
        )

    def tmall_item_schema_add(
            self,
            category_id,
            product_id,
            xml_data
    ):
        """
        天猫根据规则发布商品
        天猫TopSchema发布商品。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23255

        :param category_id: 商品发布的目标类目，必须是叶子类目
        :param product_id: 发布商品的productId，如果tmall.product.match.schema.get获取到得字段为空，这个参数传入0，否则需要通过tmall.product.schema.match查询到得可用productId
        :param xml_data: 根据tmall.item.add.schema.get生成的商品发布规则入参数据
        """
        return self._top_request(
            "tmall.item.schema.add",
            {
                "category_id": category_id,
                "product_id": product_id,
                "xml_data": xml_data
            }
        )

    def tmall_item_add_schema_get(
            self,
            category_id,
            product_id,
            type='b',
            isv_init=''
    ):
        """
        天猫发布商品规则获取
        通过类目以及productId获取商品发布规则；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23256

        :param category_id: 商品发布的目标类目，必须是叶子类目
        :param product_id: 商品发布的目标product_id
        :param type: 发布商品类型，一口价填“b”，拍卖填'a'
        :param isv_init: 正常接口调用时，请忽略这个参数或者填FALSE。这个参数提供给ISV对接Schema时，如果想先获取了解所有字段和规则，可以将此字段设置为true，product_id也就不需要提供了，设置为0即可
        """
        return self._top_request(
            "tmall.item.add.schema.get",
            {
                "category_id": category_id,
                "product_id": product_id,
                "type": type,
                "isv_init": isv_init
            },
            result_processor=lambda x: x["add_item_result"]
        )

    def tmall_product_add_schema_get(
            self,
            category_id,
            brand_id=''
    ):
        """
        产品发布规则获取接口
        获取用户发布产品的规则
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23257

        :param category_id: 商品发布的目标类目，必须是叶子类目
        :param brand_id: 品牌ID
        """
        return self._top_request(
            "tmall.product.add.schema.get",
            {
                "category_id": category_id,
                "brand_id": brand_id
            },
            result_processor=lambda x: x["add_product_rule"]
        )

    def tmall_product_match_schema_get(
            self,
            category_id
    ):
        """
        获取匹配产品规则
        ISV发布商品前，需要先查找到产品ID，这个接口返回查找产品规则入参规则
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23258

        :param category_id: 商品发布的目标类目，必须是叶子类目
        """
        return self._top_request(
            "tmall.product.match.schema.get",
            {
                "category_id": category_id
            },
            result_processor=lambda x: x["match_result"]
        )

    def tmall_product_schema_match(
            self,
            category_id,
            propvalues
    ):
        """
        product匹配接口
        根据tmall.product.match.schema.get获取到的规则，填充相应地的字段值以及类目，匹配符合条件的产品，返回匹配product结果，注意，有可能返回多个产品ID，以逗号分隔（尤其是图书类目）；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23259

        :param category_id: 商品发布的目标类目，必须是叶子类目
        :param propvalues: 根据tmall.product.match.schema.get获取到的模板，ISV将需要的字段填充好相应的值结果XML。
        """
        return self._top_request(
            "tmall.product.schema.match",
            {
                "category_id": category_id,
                "propvalues": propvalues
            },
            result_processor=lambda x: x["match_result"]
        )

    def tmall_product_schema_add(
            self,
            category_id,
            xml_data,
            brand_id=''
    ):
        """
        使用Schema文件发布一个产品
        Schema体系发布一个产品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23260

        :param category_id: 商品发布的目标类目，必须是叶子类目
        :param xml_data: 根据tmall.product.add.schema.get生成的产品发布规则入参数据
        :param brand_id: 品牌ID
        """
        return self._top_request(
            "tmall.product.schema.add",
            {
                "category_id": category_id,
                "xml_data": xml_data,
                "brand_id": brand_id
            },
            result_processor=lambda x: x["add_product_result"]
        )

    def taobao_item_update_schema_get(
            self,
            item_id,
            category_id='0'
    ):
        """
        商品编辑规则信息获取接口
        在新的编辑模式下，isv需要先获取一份编辑商品的规则和数据，然后根据规则传入数据。该接口提供编辑规则查询功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23265

        :param item_id: 商品id
        :param category_id: 商品类目id
        """
        return self._top_request(
            "taobao.item.update.schema.get",
            {
                "item_id": item_id,
                "category_id": category_id
            },
            result_processor=lambda x: x["update_rules"]
        )

    def taobao_item_add_schema_get(
            self,
            category_id
    ):
        """
        商品发布规则信息获取接口
        在新的发布模式下，isv需要先获取一份发布规则，然后根据规则传入数据。该接口提供规则查询功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23266

        :param category_id: 发布宝贝的叶子类目id
        """
        return self._top_request(
            "taobao.item.add.schema.get",
            {
                "category_id": category_id
            },
            result_processor=lambda x: x["add_rules"]
        )

    def taobao_item_schema_add(
            self,
            category_id,
            xml_data
    ):
        """
        基于xml格式的商品发布api
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        isv将宝贝信息，组装成特定格式的xml数据作为参数，进行发布商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23267

        :param category_id: 当前发布的叶子类目
        :param xml_data: 将需要发布的商品数据组装成的xml格式数据
        """
        return self._top_request(
            "taobao.item.schema.add",
            {
                "category_id": category_id,
                "xml_data": xml_data
            },
            result_processor=lambda x: x["add_result"]
        )

    def taobao_item_schema_update(
            self,
            item_id,
            xml_data,
            category_id='0',
            incremental='true'
    ):
        """
        新模式下的商品编辑接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        新模式下的商品编辑接口，在调用该接口的时候，需要先调用taobao.item.update.rules.get接口获取编辑规则和数据集。然后按照约定的参数传入规则，调用该接口进行编辑商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23268

        :param item_id: 编辑商品的商品id
        :param xml_data: 编辑商品时候的修改内容
        :param category_id: 如果重新选择类目后，传入重新选择的叶子类目id
        :param incremental: 是否增量更新。为true只改xml_data里传入的值。为false时，将根据xml_data的数据对原始宝贝数据进行全量覆盖,这个参数暂时不支持
        """
        return self._top_request(
            "taobao.item.schema.update",
            {
                "item_id": item_id,
                "xml_data": xml_data,
                "category_id": category_id,
                "incremental": incremental
            },
            result_processor=lambda x: x["update_result"]
        )

    def tmall_product_update_schema_get(
            self,
            product_id
    ):
        """
        产品更新规则获取接口
        获取用户更新产品的规则
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23432

        :param product_id: 产品编号
        """
        return self._top_request(
            "tmall.product.update.schema.get",
            {
                "product_id": product_id
            },
            result_processor=lambda x: x["update_product_schema"]
        )

    def tmall_product_schema_update(
            self,
            xml_data,
            product_id
    ):
        """
        产品更新接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23433

        :param xml_data: 根据tmall.product.update.schema.get生成的产品更新规则入参数据
        :param product_id: 产品编号
        """
        return self._top_request(
            "tmall.product.schema.update",
            {
                "xml_data": xml_data,
                "product_id": product_id
            },
            result_processor=lambda x: x["update_product_result"]
        )

    def tmall_item_schema_update(
            self,
            item_id,
            xml_data,
            category_id='',
            product_id=''
    ):
        """
        天猫根据规则编辑商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23434

        :param item_id: 需要编辑的商品ID
        :param xml_data: 根据tmall.item.update.schema.get生成的商品编辑规则入参数据
        :param category_id: 商品发布的目标类目，必须是叶子类目。如果没有切换类目需求不需要填写
        :param product_id: 商品发布的目标product_id。如果没有切换类目或者切换产品的需求，参数不用填写
        """
        return self._top_request(
            "tmall.item.schema.update",
            {
                "item_id": item_id,
                "xml_data": xml_data,
                "category_id": category_id,
                "product_id": product_id
            }
        )

    def tmall_item_update_schema_get(
            self,
            item_id,
            category_id='',
            product_id=''
    ):
        """
        天猫编辑商品规则获取
        Schema方式编辑天猫商品时，编辑商品规则获取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23435

        :param item_id: 需要编辑的商品ID
        :param category_id: 商品发布的目标类目，必须是叶子类目。如果没有切换类目需求，不需要填写。
        :param product_id: 商品发布的目标product_id。如果没有切换产品的需求，参数可以不填写。
        """
        return self._top_request(
            "tmall.item.update.schema.get",
            {
                "item_id": item_id,
                "category_id": category_id,
                "product_id": product_id
            },
            result_processor=lambda x: x["update_item_result"]
        )

    def tmall_product_schema_get(
            self,
            product_id
    ):
        """
        产品信息获取schema获取
        产品信息获取接口schema形式返回
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23565

        :param product_id: 产品编号
        """
        return self._top_request(
            "tmall.product.schema.get",
            {
                "product_id": product_id
            },
            result_processor=lambda x: x["get_product_result"]
        )

    def tmall_item_increment_update_schema_get(
            self,
            item_id,
            xml_data=''
    ):
        """
        天猫增量更新商品规则获取
        增量方式修改天猫商品的规则获取的API。<br/>1.接口返回支持增量修改的字段以及相应字段的规则。<br/>2.如果入参xml_data指定了更新的字段，则只返回指定字段的规则（ISV如果功能性很强，如明确更新Title，请拼装好次字段以提升API整体性能）；<br/>3.ISV初次接入，开发阶段，此字段不填可以看到所有支持增量的字段；但是如果上线功能明确，请尽量遵守第2条<br/>4.如果ISV对字段规则非常清晰，可以直接组装入参数据提交到tmall.item.schema.increment.update进行数据更新。但是最好不要写死，比如每天还是有对此接口功能的一次比对。<br/>---（感谢爱慕旗舰店提供API命名）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23781

        :param item_id: 需要编辑的商品ID
        :param xml_data: 如果入参xml_data指定了更新的字段，则只返回指定字段的规则（ISV如果功能性很强，如明确更新Title，请拼装好此字段以提升API整体性能）
        """
        return self._top_request(
            "tmall.item.increment.update.schema.get",
            {
                "item_id": item_id,
                "xml_data": xml_data
            },
            result_processor=lambda x: x["update_item_result"]
        )

    def tmall_item_schema_increment_update(
            self,
            item_id,
            xml_data
    ):
        """
        天猫根据规则增量更新商品
        增量方式修改天猫商品的API。只要是此接口支持增量修改的字段，可以同时更新。（感谢爱慕旗舰店提供API命名）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23782

        :param item_id: 需要编辑的商品ID
        :param xml_data: 根据tmall.item.increment.update.schema.get生成的商品增量编辑规则入参数据。需要更新的字段，一定要在入参的XML重点update_fields字段中明确指明
        """
        return self._top_request(
            "tmall.item.schema.increment.update",
            {
                "item_id": item_id,
                "xml_data": xml_data
            }
        )

    def tmall_item_price_update(
            self,
            item_id,
            item_price='',
            sku_prices=None,
            options=None
    ):
        """
        天猫商品/SKU价格更新接口
        天猫商品/SKU价格更新接口，支持商品、SKU价格同时更新，支持同一商品下的SKU批量更新。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24249

        :param item_id: 商品ID
        :param item_price: 被更新商品价格
        :param sku_prices: 更新SKU价格时候的SKU价格对象；如果没有SKU或者不更新SKU价格，可以不填;查找SKU目前支持ID，属性串和商家编码三种模式，建议选用一种最合适的，切勿滥用，一次调用中如果混合使用，更新结果不可预期！
        :param options: 商品价格更新时候的可选参数
        """
        return self._top_request(
            "tmall.item.price.update",
            {
                "item_id": item_id,
                "item_price": item_price,
                "sku_prices": sku_prices,
                "options": options
            },
            result_processor=lambda x: x["price_update_result"]
        )

    def taobao_item_schema_increment_update(
            self,
            item_id,
            parameters,
            category_id='0'
    ):
        """
        集市schema增量编辑
        根据schema规则增量修改宝贝信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24269

        :param item_id: 商品id
        :param parameters: 修改字段
        :param category_id: 商品类目id
        """
        return self._top_request(
            "taobao.item.schema.increment.update",
            {
                "item_id": item_id,
                "parameters": parameters,
                "category_id": category_id
            },
            result_processor=lambda x: x["item_id"]
        )

    def taobao_item_increment_update_schema_get(
            self,
            item_id,
            category_id='0',
            update_fields='all'
    ):
        """
        获取增量更新规则
        获取增量更新规则，目前开发的字段有title, subTitle, description, wl_description
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24302

        :param item_id: 宝贝id
        :param category_id: 宝贝类目id
        :param update_fields: 修改字段
        """
        return self._top_request(
            "taobao.item.increment.update.schema.get",
            {
                "item_id": item_id,
                "category_id": category_id,
                "update_fields": update_fields
            },
            result_processor=lambda x: x["update_rules"]
        )

    def tmall_item_sizemapping_templates_list(
            self
    ):
        """
        获取天猫商品尺码表模板列表
        获取所有尺码表模板列表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24327

        """
        return self._top_request(
            "tmall.item.sizemapping.templates.list",
            result_processor=lambda x: x["size_mapping_templates"]
        )

    def tmall_item_sizemapping_template_get(
            self,
            template_id
    ):
        """
        获取天猫商品尺码表模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24328

        :param template_id: 尺码表模板ID
        """
        return self._top_request(
            "tmall.item.sizemapping.template.get",
            {
                "template_id": template_id
            },
            result_processor=lambda x: x["size_mapping_template"]
        )

    def tmall_item_sizemapping_template_delete(
            self,
            template_id
    ):
        """
        删除天猫商品尺码表模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24329

        :param template_id: 尺码表模板ID
        """
        return self._top_request(
            "tmall.item.sizemapping.template.delete",
            {
                "template_id": template_id
            },
            result_processor=lambda x: x["template_id"]
        )

    def tmall_item_sizemapping_template_update(
            self,
            template_id,
            template_name,
            template_content
    ):
        """
        更新天猫商品尺码表模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24330

        :param template_id: 尺码表模板ID
        :param template_name: 尺码表模板名称
        :param template_content: 尺码表模板内容，格式为'尺码值:维度名称:数值,尺码值:维度名称:数值'。其中，数值的单位，长度单位为厘米（cm），体重单位为公斤（kg）。尺码值，维度数据不能包含数字，特殊字符。数值为0-999.9的数字，且最多一位小数。
        """
        return self._top_request(
            "tmall.item.sizemapping.template.update",
            {
                "template_id": template_id,
                "template_name": template_name,
                "template_content": template_content
            },
            result_processor=lambda x: x["size_mapping_template"]
        )

    def tmall_item_sizemapping_template_create(
            self,
            template_name,
            template_content
    ):
        """
        新增天猫商品尺码表模板
        新增天猫商品尺码表模板<br/><br/>男鞋、女鞋、运动鞋、户外鞋类目，尺码表维度为：<br/>脚长（cm） 必选<br/><br/>内衣-文胸类目，尺码表维度为：<br/>上胸围（cm） 必选<br/>下胸围（cm） 必选<br/>上下胸围差（cm） 必选<br/>身高（cm）<br/>体重（公斤）<br/><br/>内衣-内裤类目，尺码表维度为：<br/>腰围（cm） 必选<br/>臀围（cm） 必选<br/>身高（cm）<br/>体重（公斤）<br/>裤长（cm）<br/>裆部（cm）<br/>脚口（cm）<br/>腿围（cm）<br/><br/>内衣-睡衣上衣/保暖上衣/睡袍类目，尺码表维度为：<br/>身高（cm） 必选<br/>胸围（cm） 必选<br/>体重（公斤）<br/>腰围（cm）<br/>肩宽（cm）<br/>袖长（cm）<br/>衣长（cm）<br/>背宽（cm）<br/>前长（cm）<br/>下摆围（cm）<br/>袖口（cm）<br/>袖肥（cm）<br/>领口（cm）<br/><br/>内衣-睡裤/保暖裤类目，尺码维度为：<br/>身高（cm） 必选<br/>腰围（cm） 必选<br/>体重（公斤）<br/>臀围（cm）<br/>裆部（cm）<br/>裤长（cm）<br/>脚口（cm）<br/>腿围（cm）<br/>裤侧长（cm）<br/><br/>内衣-睡裙类目，尺码维度为：<br/>身高（cm） 必选<br/>胸围（cm） 必选<br/>体重（公斤）<br/>裙长（cm）<br/>腰围（cm）<br/>袖长（cm）<br/>肩宽（cm）<br/>背宽（cm）<br/>腿围（cm）<br/>臀围（cm）<br/>底摆（cm）<br/><br/>男装、女装、运动服、户外服等上装类目，尺码维度为（至少两项必选）：<br/>身高（cm）<br/>体重（公斤）<br/>肩宽（cm）<br/>胸围（cm）<br/>腰围（cm）<br/>袖长（cm）<br/>衣长（cm）<br/>背宽（cm）<br/>前长（cm）<br/>摆围（cm）<br/>下摆围（cm）<br/>袖口（cm）<br/>袖肥（cm）<br/>中腰（cm）<br/>领深（cm）<br/>领高（cm）<br/>领宽（cm）<br/>领围（cm）<br/>圆摆后中长（cm）<br/>平摆衣长（cm）<br/>圆摆衣长（cm）<br/><br/>男装、女装、运动服、户外服等下装类目，尺码维度为（至少两项必选）：<br/>身高（cm）<br/>体重（公斤）<br/>腰围（cm）<br/>臀围（cm）<br/>裤长（cm）<br/>裙长（cm）<br/>裙摆长（cm）<br/>腿围（cm）<br/>膝围（cm）<br/>小脚围（cm）<br/>拉伸腰围（cm）<br/>坐围（cm）<br/>拉伸坐围（cm）<br/>脚口（cm）<br/>前浪（cm）<br/>后浪（cm）<br/>横档（cm）<br/><br/>如果上述维度满足，可以自定义最多5个维度。<br/><br/>模板格式为：<br/>尺码值:维度名称:数值<br/>如：M:身高（cm）:160,L:身高（cm）:170
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24331

        :param template_name: 尺码表模板名称
        :param template_content: 尺码表模板内容，格式为'尺码值:维度名称:数值,尺码值:维度名称:数值'。其中，数值的单位，长度单位为厘米（cm），体重单位为公斤（kg）。尺码值，维度数据不能包含数字，特殊字符。数值为0-999.9的数字，且最多一位小数。
        """
        return self._top_request(
            "tmall.item.sizemapping.template.create",
            {
                "template_name": template_name,
                "template_content": template_content
            },
            result_processor=lambda x: x["size_mapping_template"]
        )

    def alibaba_wholesale_goods_get(
            self,
            country_code='',
            id=''
    ):
        """
        查询阿里巴巴批发市场商品详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24358

        :param country_code: country_code
        :param id: id
        """
        return self._top_request(
            "alibaba.wholesale.goods.get",
            {
                "country_code": country_code,
                "id": id
            },
            result_processor=lambda x: x["wholesale_goods_result"]
        )

    def alibaba_wholesale_goods_search(
            self,
            param_search_goods_option=None
    ):
        """
        批发市场产品搜索
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24366

        :param param_search_goods_option: SearchGoodsOption
        """
        return self._top_request(
            "alibaba.wholesale.goods.search",
            {
                "param_search_goods_option": param_search_goods_option
            },
            result_processor=lambda x: x["wholesale_goods_search_result"]
        )

    def taobao_item_seller_get(
            self,
            fields,
            num_iid
    ):
        """
        获取单个商品详细信息
        获取单个商品的全部信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24625

        :param fields: 需要返回的商品字段列表。可选值：Item商品结构体中所有字段均可返回，多个字段用“,”分隔。
        :param num_iid: 商品数字ID
        """
        return self._top_request(
            "taobao.item.seller.get",
            {
                "fields": fields,
                "num_iid": num_iid
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_items_seller_list_get(
            self,
            fields,
            num_iids
    ):
        """
        批量获取商品详细信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24626

        :param fields: 需要返回的商品字段列表。可选值：点击返回结果中的Item结构体中能展示出来的所有字段，多个字段用“,”分隔。注：返回所有sku信息的字段名称是sku而不是skus。
        :param num_iids: 商品ID列表，多个ID用半角逗号隔开，一次最多不超过20个。注：获取不存在的商品ID或获取别人的商品都不会报错，但没有商品数据返回。
        """
        return self._top_request(
            "taobao.items.seller.list.get",
            {
                "fields": fields,
                "num_iids": num_iids
            },
            result_processor=lambda x: x["items"]
        )

    def tmall_item_outerid_update(
            self,
            item_id,
            outer_id='',
            sku_outers=None
    ):
        """
        天猫商品/SKU商家编码更新接口
        天猫商品/SKU商家编码更新接口；支持商品、SKU的商家编码同时更新；支持同一商品下的SKU批量更新。（感谢sample小雨提供接口命名）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25076

        :param item_id: 商品ID
        :param outer_id: 商品维度商家编码，如果不修改可以不传；清空请设置空串
        :param sku_outers: 商品SKU更新OuterId时候用的数据
        """
        return self._top_request(
            "tmall.item.outerid.update",
            {
                "item_id": item_id,
                "outer_id": outer_id,
                "sku_outers": sku_outers
            },
            result_processor=lambda x: x["outerid_update_result"]
        )

    def tmall_item_shiptime_update(
            self,
            item_id,
            option,
            ship_time='',
            sku_ship_times=None
    ):
        """
        更新商品发货时间
        增加更新删除商品/SKU发货时间(支持同一商品下的SKU同时批量更新)
        1.
        {
        'shipTimeType': 2,  ----相对发货时间（值为1则为绝对发货时间）
        'updateType': 0 ---更新SKU
        },
        按照指定SKU更新指定SKU的发货时间，如果原本是商品级发货时间，商品级发货时间也清空
        2.
        {
        'shipTimeType': 0, -- 删除发货时间
        'updateType': 0 --更新SKU
        },
        按照指定SKU删除指定SKU的发货时间
        3.
        {
        'shipTimeType': 2,  ----相对发货时间（值为1则为绝对发货时间）
        'updateType': 1 ---更新商品
        },
        更新商品级发货时间，如果原本是SKU级发货时间，清空所有SKU上的发货时间
        4.
        {
        'shipTimeType': 0, -- 删除发货时间
        'updateType': 1 --更新商品
        },
        删除商品级的发货时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25393

        :param item_id: 商品ID
        :param option: 批量更新商品/SKU发货时间的备选项
        :param ship_time: 被更新发货时间（商品级）；格式和具体设置的发货时间格式相关。绝对发货时间填写yyyy-MM-dd;相对发货时间填写数字。发货时间必须在当前时间后三天。如果设置的绝对时间小于当前时间的三天后，会清除该商品的发货时间设置。如果是相对时间小于3，则会提示出错。如果shiptimeType为0，要清除商品上的发货时间，该字段可以填任意字符,也可以不填。
        :param sku_ship_times: 被更新SKU的发货时间，后台会根据三个子属性去查找匹配的sku，如果找到就默认对sku进行更新，当无匹配sku且更新类型针对sku，会报错。
        """
        return self._top_request(
            "tmall.item.shiptime.update",
            {
                "item_id": item_id,
                "option": option,
                "ship_time": ship_time,
                "sku_ship_times": sku_ship_times
            }
        )

    def alibaba_wholesale_shippingline_template_init(
            self,
            initial_template=None
    ):
        """
        创建初始模板
        创建默认的几种运费模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25397

        :param initial_template: 创建初始运费模板参数
        """
        return self._top_request(
            "alibaba.wholesale.shippingline.template.init",
            {
                "initial_template": initial_template
            },
            result_processor=lambda x: x["init_template_response"]
        )

    def tmall_item_simpleschema_add(
            self,
            schema_xml_fields
    ):
        """
        天猫简化发布商品
        天猫简化版schema发布商品。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25455

        :param schema_xml_fields: 根据tmall.item.add.simpleschema.get生成的商品发布规则入参数据
        """
        return self._top_request(
            "tmall.item.simpleschema.add",
            {
                "schema_xml_fields": schema_xml_fields
            }
        )

    def tmall_item_add_simpleschema_get(
            self
    ):
        """
        天猫发布商品规则获取
        通过商家信息获取商品发布字段和规则。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25456

        """
        return self._top_request(
            "tmall.item.add.simpleschema.get"
        )

    def taobao_item_qualification_display_get(
            self,
            param='',
            item_id='',
            category_id=''
    ):
        """
        资质采集配置异步获取接口
        根据类目，商品，属性等参与动态获得资质采集配置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25726

        :param param: 参数列表，为key和value都是string的map的转化的json格式
        :param item_id: 商品id
        :param category_id: 类目id
        """
        return self._top_request(
            "taobao.item.qualification.display.get",
            {
                "param": param,
                "item_id": item_id,
                "category_id": category_id
            },
            result_processor=lambda x: x["display_conf"]
        )

    def tmall_item_simpleschema_update(
            self,
            item_id,
            schema_xml_fields
    ):
        """
        天猫简化编辑商品
        国外大商家天猫简化编辑商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26072

        :param item_id: 商品id
        :param schema_xml_fields: 编辑商品时提交的xml信息
        """
        return self._top_request(
            "tmall.item.simpleschema.update",
            {
                "item_id": item_id,
                "schema_xml_fields": schema_xml_fields
            }
        )

    def tmall_item_quantity_update(
            self,
            item_id,
            sku_quantities=None,
            options=None,
            item_quantity=''
    ):
        """
        天猫商品/SKU库存更新接口
        天猫商品/SKU库存更新接口；支持商品库存更新；支持同一商品下的SKU批量更新。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26178

        :param item_id: 商品id
        :param sku_quantities: 更新SKU库存时候的SKU库存对象；如果没有SKU或者不更新SKU库存，可以不填;查找SKU目前支持ID，属性串和商家编码三种模式，建议选用一种最合适的，切勿滥用，一次调用中如果混合使用，更新结果不可预期！
        :param options: 商品库存更新时候的可选参数
        :param item_quantity: 商品库存数；增量编辑方式支持正数、负数
        """
        return self._top_request(
            "tmall.item.quantity.update",
            {
                "item_id": item_id,
                "sku_quantities": sku_quantities,
                "options": options,
                "item_quantity": item_quantity
            },
            result_processor=lambda x: x["quantity_update_result"]
        )

    def tmall_item_update_simpleschema_get(
            self,
            item_id='0'
    ):
        """
        官网同购编辑商品的get接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26232

        :param item_id: 商品id
        """
        return self._top_request(
            "tmall.item.update.simpleschema.get",
            {
                "item_id": item_id
            }
        )

    def tmall_item_calculate_hscode_get(
            self,
            item_id=''
    ):
        """
        算法获取hscode
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26569

        :param item_id: 商品id
        """
        return self._top_request(
            "tmall.item.calculate.hscode.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["results"]
        )

    def tmall_item_combine_get(
            self,
            item_id=''
    ):
        """
        组合商品获取接口
        查询组合商品的SKU信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27227

        :param item_id: 组合商品ID
        """
        return self._top_request(
            "tmall.item.combine.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["results"]
        )

    def tmall_item_zhizu_shoes_size_upload(
            self,
            item_id,
            type,
            size_mapping
    ):
        """
        知足鞋品数据上传接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫尺码拍照导购需要提供给北京知足公司上传测鞋数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27340

        :param item_id: 天猫商品ID
        :param type: 知足鞋类型
        :param size_mapping: 天猫尺码和知足码映射
        """
        return self._top_request(
            "tmall.item.zhizu.shoes.size.upload",
            {
                "item_id": item_id,
                "type": type,
                "size_mapping": size_mapping
            }
        )

    def tmall_item_dapei_template_query(
            self,
            title='',
            page_index='1',
            page_size='10000'
    ):
        """
        搭配查询接口
        根据条件获取搭配内容
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27341

        :param title: 搭配标题
        :param page_index: 页码
        :param page_size: 分页大小
        """
        return self._top_request(
            "tmall.item.dapei.template.query",
            {
                "title": title,
                "page_index": page_index,
                "page_size": page_size
            }
        )

    def taobao_baike_import_zhubao_picture(
            self,
            picture
    ):
        """
        百科图片数据导入
        用于接入外部--图片--录入到商品百科中
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30112

        :param picture: 图片二进制数据
        """
        return self._top_request(
            "taobao.baike.import.zhubao.picture",
            {
                "picture": picture
            }
        )

    def taobao_baike_import_zhubao_data(
            self,
            data_json_str
    ):
        """
        导入数据到商品百科服务
        用于接入外部数据录入到商品百科中
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30113

        :param data_json_str: 约定的Json数据
        """
        return self._top_request(
            "taobao.baike.import.zhubao.data",
            {
                "data_json_str": data_json_str
            }
        )

    def tmall_item_hscode_detail_get(
            self,
            hscode
    ):
        """
        通过hscode获取计量单位
        通过hscode获取计量单位和销售单位
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30672

        :param hscode: hscode
        """
        return self._top_request(
            "tmall.item.hscode.detail.get",
            {
                "hscode": hscode
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_baike_permission_zhubao_disable(
            self,
            user_app_secret='',
            topic_name=''
    ):
        """
        百科开发平台录入权限控制
        用于外部数据删除数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30777

        :param user_app_secret: 权限密码
        :param topic_name: 词条主键名称
        """
        return self._top_request(
            "taobao.baike.permission.zhubao.disable",
            {
                "user_app_secret": user_app_secret,
                "topic_name": topic_name
            }
        )

    def taobao_baike_permission_zhubao_enable(
            self,
            user_app_secret='',
            user_topic_name=''
    ):
        """
        珠宝权限删除恢复接口
        阿里巴巴商品百科开放数据删除恢复服务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30778

        :param user_app_secret: 权限密码
        :param user_topic_name: 词条主键名称
        """
        return self._top_request(
            "taobao.baike.permission.zhubao.enable",
            {
                "user_app_secret": user_app_secret,
                "user_topic_name": user_topic_name
            }
        )

    def tmall_item_edu_update_outline(
            self,
            item_id,
            outline_info
    ):
        """
        教育联盟商家上传课程大纲api
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31082

        :param item_id: 宝贝Id
        :param outline_info: 课程大纲json
        """
        return self._top_request(
            "tmall.item.edu.update.outline",
            {
                "item_id": item_id,
                "outline_info": outline_info
            }
        )

    def alibaba_interact_item_query(
            self,
            store_code,
            item_id,
            include_lbs='false',
            include_sku='false',
            include_service='false'
    ):
        """
        查询单个门店商品
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31480

        :param store_code: 门店编码
        :param item_id: 商品Id
        :param include_lbs: 是否包含位置信息
        :param include_sku: 是否包含Sku信息
        :param include_service: 是否包含服务信息
        """
        return self._top_request(
            "alibaba.interact.item.query",
            {
                "store_code": store_code,
                "item_id": item_id,
                "include_lbs": include_lbs,
                "include_sku": include_sku,
                "include_service": include_service
            }
        )

    def alibaba_interact_item_search(
            self,
            param0,
            param1=None,
            include_lbs='false',
            include_sku='false',
            include_service='false'
    ):
        """
        门店商品搜索接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31498

        :param param0: 门店编码
        :param param1: 查询参数
        :param include_lbs: 是否包含lbs
        :param include_sku: 是否包含sku信息
        :param include_service: 是否包含服务信息
        """
        return self._top_request(
            "alibaba.interact.item.search",
            {
                "param0": param0,
                "param1": param1,
                "include_lbs": include_lbs,
                "include_sku": include_sku,
                "include_service": include_service
            }
        )

    def alibaba_gpu_add_schema_get(
            self,
            leaf_cat_id,
            provider_id,
            brand_id=''
    ):
        """
        获取产品发布规则接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38593

        :param leaf_cat_id: 叶子类目ID
        :param provider_id: 当前用户所在渠道如0代表天猫，8代表淘宝
        :param brand_id: 品牌ID
        """
        return self._top_request(
            "alibaba.gpu.add.schema.get",
            {
                "leaf_cat_id": leaf_cat_id,
                "provider_id": provider_id,
                "brand_id": brand_id
            },
            result_processor=lambda x: x["add_product_rule"]
        )

    def alibaba_gpu_schema_add(
            self,
            leaf_cat_id,
            schema_xml_fields,
            provider_id,
            brand_id=''
    ):
        """
        使用schema文件发布产品
        使用Schema文件发布一个产品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38595

        :param leaf_cat_id: 叶子类目ID
        :param schema_xml_fields: 根据alibaba.gpu.add.schema.get获取的规则提交上来的schema
        :param provider_id: 当前用户所在渠道如0代表天猫，8代表淘宝
        :param brand_id: 品牌ID
        """
        return self._top_request(
            "alibaba.gpu.schema.add",
            {
                "leaf_cat_id": leaf_cat_id,
                "schema_xml_fields": schema_xml_fields,
                "provider_id": provider_id,
                "brand_id": brand_id
            },
            result_processor=lambda x: x["add_product_result"]
        )

    def alibaba_gpu_update_schema_get(
            self,
            product_id,
            provider_id
    ):
        """
        获取产品编辑schema规则的接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38596

        :param product_id: 产品ID
        :param provider_id: 当前用户所在渠道如0代表天猫，8代表淘宝
        """
        return self._top_request(
            "alibaba.gpu.update.schema.get",
            {
                "product_id": product_id,
                "provider_id": provider_id
            },
            result_processor=lambda x: x["update_product_rule"]
        )

    def alibaba_gpu_schema_update(
            self,
            product_id,
            schema_xml_fields,
            provider_id
    ):
        """
        产品更新接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38597

        :param product_id: 产品ID
        :param schema_xml_fields: 更新产品提交的schema数据
        :param provider_id: 当前用户所在渠道如0代表天猫，8代表淘宝
        """
        return self._top_request(
            "alibaba.gpu.schema.update",
            {
                "product_id": product_id,
                "schema_xml_fields": schema_xml_fields,
                "provider_id": provider_id
            },
            result_processor=lambda x: x["update_product_result"]
        )

    def alibaba_gpu_schema_catsearch(
            self,
            leaf_cat_id,
            current_page,
            page_size,
            provider_id
    ):
        """
        按类目查询spu接口
        按类目查询spu的schema接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38598

        :param leaf_cat_id: 叶子类目ID
        :param current_page: 当前页
        :param page_size: 每页大小
        :param provider_id: 渠道Id，如0代表天猫，8代表淘宝
        """
        return self._top_request(
            "alibaba.gpu.schema.catsearch",
            {
                "leaf_cat_id": leaf_cat_id,
                "current_page": current_page,
                "page_size": page_size,
                "provider_id": provider_id
            }
        )

    def taobao_isp_pos_vendor_feedback(
            self,
            source,
            action,
            device_code,
            op_time,
            province,
            province_code,
            city,
            city_code,
            district,
            district_code,
            latitude,
            longitude,
            user_nick,
            address='',
            item_id='',
            item_name='',
            category='',
            properties='',
            item_quantity='',
            coupon_id='',
            div_pay_amt='',
            total_pay_amt='',
            trade_no='',
            trade_type='',
            user_mobile='',
            out_user='',
            out_biz_id=''
    ):
        """
        新零售线下售货机POS数据回流
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        新零售行业线下售货机商品数据回流，提供给相关线下售卖机终端ISV，将商品的POS信息回流，包括但不限于事件、用户昵称、商品信息、购买量、消费金额等。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39455

        :param source: 行业数据来源，需要与指定业务对接人授权注册。天猫无人售货机:TMALL_VENDOR;天猫游戏桌面:TMALL_BOX
        :param action: 'ACTION枚举值： ITEM_CLICK（商品点击时必须设置ITEM_ID） BUY_CLICK（点击购买） VENDING_MACHINE_SHIPMENT（售货机出货时TRADE_TYPE、TRADE_NO要设置） RECEIVE_COUPONS（领取优惠券时必须设置COUPON_ID） SHARE_CLICK（内容分享）'
        :param device_code: 设备硬件CODE
        :param op_time: 用户操作时间
        :param province: 省名称
        :param province_code: 省代码
        :param city: 设备所在城市名称
        :param city_code: 城市代码
        :param district: 行政区名称
        :param district_code: 行政区代码
        :param latitude: 设备地理位置纬度（采用火星坐标系）
        :param longitude: 设备地理位置经度（采用火星坐标系）
        :param user_nick: 用户淘宝昵称
        :param address: 设备的详细地址信息
        :param item_id: 商品id
        :param item_name: 商品名称
        :param category: 商品所属类目
        :param properties: 商品属性信息，(格式：属性名:属性值),多个属性间用分号分隔
        :param item_quantity: 商品购买数量
        :param coupon_id: '例如官方领取优惠券链接里的activityId： https://taoquan.taobao.com/coupon/unify_apply.htm?sellerId=2649119619&activityId=9d390579777e41a981b54aa4d6154f5e'
        :param div_pay_amt: 商品购买金额
        :param total_pay_amt: 订单支付总额（一个订单内的多个商品总消费金额）
        :param trade_no: 订单编号
        :param trade_type: 'trade_type枚举值： alipay_trade（支付宝订单类型，对应的trade_no设置支付宝订单号） tmall_trade（天猫/淘宝订单类型，对应的trade_no设置天猫/淘宝订单号）'
        :param user_mobile: 用户手机号码
        :param out_user: 外部设备识别用户标识
        :param out_biz_id: 数据外部编码，保证数据唯一性
        """
        return self._top_request(
            "taobao.isp.pos.vendor.feedback",
            {
                "source": source,
                "action": action,
                "device_code": device_code,
                "op_time": op_time,
                "province": province,
                "province_code": province_code,
                "city": city,
                "city_code": city_code,
                "district": district,
                "district_code": district_code,
                "latitude": latitude,
                "longitude": longitude,
                "user_nick": user_nick,
                "address": address,
                "item_id": item_id,
                "item_name": item_name,
                "category": category,
                "properties": properties,
                "item_quantity": item_quantity,
                "coupon_id": coupon_id,
                "div_pay_amt": div_pay_amt,
                "total_pay_amt": total_pay_amt,
                "trade_no": trade_no,
                "trade_type": trade_type,
                "user_mobile": user_mobile,
                "out_user": out_user,
                "out_biz_id": out_biz_id
            }
        )

    def taobao_item_carturl_get(
            self,
            item_ids,
            callback_url='',
            user_nick='',
            user_id='',
            ext_params=''
    ):
        """
        加购URL获取
        获取加购URL，支持添加商品到购物车
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40701

        :param item_ids: 商品信息，格式为 商品ID_SKU ID_数量，多条记录以逗号(,)分割
        :param callback_url: 回调地址，需要是EWS域名地址。可不填，默认到购物车页面
        :param user_nick: 商家Nick，优先使用user_id
        :param user_id: 商家ID
        :param ext_params: 扩展属性，关注店铺的时候会传递下去，格式为K:V|K:V格式
        """
        return self._top_request(
            "taobao.item.carturl.get",
            {
                "item_ids": item_ids,
                "callback_url": callback_url,
                "user_nick": user_nick,
                "user_id": user_id,
                "ext_params": ext_params
            }
        )


class TbJiaoYi(DingTalkBaseAPI):
    """
    交易API
    """

    def taobao_trades_sold_get(
            self,
            fields,
            start_created='',
            end_created='',
            status='',
            buyer_nick='',
            type='',
            ext_type='',
            rate_status='',
            tag='',
            page_no='1',
            page_size='40',
            use_has_next='false',
            buyer_open_id=''
    ):
        """
        查询卖家已卖出的交易数据（根据创建时间）
        搜索当前会话用户作为卖家已卖出的交易数据（只能获取到三个月以内的交易信息）<br/><br/>1. 返回的数据结果是以订单的创建时间倒序排列的。<br/><br/>2. 返回的数据结果只包含了订单的部分数据，可通过taobao.trade.fullinfo.get获取订单详情。<br/><br/> <span style='color:red'>注意：type字段的说明，如果该字段不传，接口默认只查4种类型订单，非默认查询的订单是不返回。遇到订单查不到的情况的，通常都是这个原因造成。解决办法就是type加上订单类型就可正常返回了。用taobao.trade.fullinfo.get 查订单fields返回type 很容易的能知道订单的类型（type）</span>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=46

        :param fields: 需要返回的字段列表，多个字段用半角逗号分隔，可选值为返回示例中能看到的所有字段。rx_audit_status=0,处方药未审核
        :param start_created: 查询三个月内交易创建时间开始。格式:yyyy-MM-dd HH:mm:ss
        :param end_created: 查询交易创建时间结束。格式:yyyy-MM-dd HH:mm:ss
        :param status: 交易状态（<a href='http://open.taobao.com/doc/detail.htm?id=102856' target='_blank'>查看可选值</a>），默认查询所有交易状态的数据，除了默认值外每次只能查询一种状态。
        :param buyer_nick: 买家昵称
        :param type: 交易类型列表，同时查询多种交易类型可用逗号分隔。<span style='color:red;font-weight: bold;'>默认同时查询guarantee_trade,auto_delivery,ec,cod,step 这5 种的交易类型的数据；查询所有交易类型的数据，需要设置下面全部可选值。</span>可选值：fixed(一口价)auction(拍卖)guarantee_trade(一口价、拍卖)step(分阶段付款，万人团，阶梯团订单）independent_simple_trade(旺店入门版交易)independent_shop_trade(旺店标准版交易)auto_delivery(自动发货)ec(直冲)cod(货到付款)game_equipment(游戏装备)shopex_trade(ShopEX交易)netcn_trade(万网交易)external_trade(统一外部交易)instant_trade (即时到账)b2c_cod(大商家货到付款)hotel_trade(酒店类型交易)super_market_trade(商超交易)super_market_cod_trade(商超货到付款交易)taohua(淘花网交易类型）waimai(外卖交易类型）o2o_offlinetrade（O2O交易）nopaid（即时到帐/趣味猜交易类型）step (万人团) eticket(电子凭证) tmall_i18n（天猫国际）;nopaid （无付款交易）insurance_plus（保险）finance（基金）注：guarantee_trade是一个组合查询条件，并不是一种交易类型，获取批量或单个订单中不会返回此种类型的订单。pre_auth_type(预授权0元购) lazada（获取lazada订单类型）
        :param ext_type: 可选值有ershou(二手市场的订单）,service（商城服务子订单）mark（双十一大促活动异常订单）作为扩展类型筛选只能做单个ext_type查询，不能全部查询所有的ext_type类型
        :param rate_status: 评价状态，默认查询所有评价状态的数据，除了默认值外每次只能查询一种状态。<br>可选值：RATE_UNBUYER(买家未评)RATE_UNSELLER(卖家未评)RATE_BUYER_UNSELLER(买家已评，卖家未评)RATE_UNBUYER_SELLER(买家未评，卖家已评)RATE_BUYER_SELLER(买家已评,卖家已评)
        :param tag: 卖家对交易的自定义分组标签，目前可选值为：time_card（点卡软件代充），fee_card（话费软件代充）
        :param page_no: 页码。取值范围:大于零的整数; 默认值:1
        :param page_size: 每页条数。取值范围:大于零的整数; 默认值:40;最大值:100
        :param use_has_next: 是否启用has_next的分页方式，如果指定true,则返回的结果中不包含总记录数，但是会新增一个是否存在下一页的的字段，通过此种方式获取增量交易，接口调用成功率在原有的基础上有所提升。
        :param buyer_open_id: 买家的openId
        """
        return self._top_request(
            "taobao.trades.sold.get",
            {
                "fields": fields,
                "start_created": start_created,
                "end_created": end_created,
                "status": status,
                "buyer_nick": buyer_nick,
                "type": type,
                "ext_type": ext_type,
                "rate_status": rate_status,
                "tag": tag,
                "page_no": page_no,
                "page_size": page_size,
                "use_has_next": use_has_next,
                "buyer_open_id": buyer_open_id
            }
        )

    def taobao_trade_get(
            self,
            fields,
            tid
    ):
        """
        获取单笔交易的部分信息(性能高)
        获取单笔交易的部分信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=47

        :param fields: 需要返回的字段列表，多个字段用半角逗号分隔，可选值为返回示例中能看到的所有字段。
        :param tid: 交易编号
        """
        return self._top_request(
            "taobao.trade.get",
            {
                "fields": fields,
                "tid": tid
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trade_memo_add(
            self,
            tid,
            memo,
            flag='0'
    ):
        """
        对一笔交易添加备注
        根据登录用户的身份（买家或卖家），自动添加相应的交易备注,不能重复调用些接口添加备注，需要更新备注请用taobao.trade.memo.update
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=48

        :param tid: 交易编号
        :param memo: 交易备注。最大长度: 1000个字节
        :param flag: 交易备注旗帜，可选值为：0(灰色), 1(红色), 2(黄色), 3(绿色), 4(蓝色), 5(粉红色)，默认值为0
        """
        return self._top_request(
            "taobao.trade.memo.add",
            {
                "tid": tid,
                "memo": memo,
                "flag": flag
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trade_memo_update(
            self,
            tid,
            memo='',
            flag='',
            reset='false'
    ):
        """
        修改交易备注
        需要商家或以上权限才可调用此接口，可重复调用本接口更新交易备注，本接口同时具有添加备注的功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=49

        :param tid: 交易编号
        :param memo: 卖家交易备注。最大长度: 1000个字节
        :param flag: 卖家交易备注旗帜，可选值为：0(灰色), 1(红色), 2(黄色), 3(绿色), 4(蓝色), 5(粉红色)，默认值为0
        :param reset: 是否对memo的值置空若为true，则不管传入的memo字段的值是否为空，都将会对已有的memo值清空，慎用；若用false，则会根据memo是否为空来修改memo的值：若memo为空则忽略对已有memo字段的修改，若memo非空，则使用新传入的memo覆盖已有的memo的值
        """
        return self._top_request(
            "taobao.trade.memo.update",
            {
                "tid": tid,
                "memo": memo,
                "flag": flag,
                "reset": reset
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_refunds_receive_get(
            self,
            fields,
            status='',
            buyer_nick='',
            type='',
            start_modified='',
            end_modified='',
            page_no='1',
            page_size='40',
            use_has_next='false'
    ):
        """
        查询卖家收到的退款列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=52

        :param fields: 需要返回的字段。目前支持有：refund_id, tid, title, buyer_nick, seller_nick, total_fee, status, created, refund_fee, oid, good_status, company_name, sid, payment, reason, desc, has_good_return, modified, order_status,refund_phase
        :param status: 退款状态，默认查询所有退款状态的数据，除了默认值外每次只能查询一种状态。WAIT_SELLER_AGREE(买家已经申请退款，等待卖家同意) WAIT_BUYER_RETURN_GOODS(卖家已经同意退款，等待买家退货) WAIT_SELLER_CONFIRM_GOODS(买家已经退货，等待卖家确认收货) SELLER_REFUSE_BUYER(卖家拒绝退款) CLOSED(退款关闭) SUCCESS(退款成功)
        :param buyer_nick: 买家昵称
        :param type: 交易类型列表，一次查询多种类型可用半角逗号分隔，默认同时查询guarantee_trade, auto_delivery这两种类型的数据，<a href='http://open.taobao.com/doc/detail.htm?id=102855' target='_blank'>查看可选值</a>
        :param start_modified: 查询修改时间开始。格式: yyyy-MM-dd HH:mm:ss
        :param end_modified: 查询修改时间结束。格式: yyyy-MM-dd HH:mm:ss
        :param page_no: 页码。取值范围:大于零的整数; 默认值:1
        :param page_size: 每页条数。取值范围:大于零的整数; 默认值:40;最大值:100
        :param use_has_next: 是否启用has_next的分页方式，如果指定true,则返回的结果中不包含总记录数，但是会新增一个是否存在下一页的的字段，通过此种方式获取增量退款，接口调用成功率在原有的基础上有所提升。
        """
        return self._top_request(
            "taobao.refunds.receive.get",
            {
                "fields": fields,
                "status": status,
                "buyer_nick": buyer_nick,
                "type": type,
                "start_modified": start_modified,
                "end_modified": end_modified,
                "page_no": page_no,
                "page_size": page_size,
                "use_has_next": use_has_next
            }
        )

    def taobao_trade_fullinfo_get(
            self,
            fields,
            tid
    ):
        """
        获取单笔交易的详细信息
        <br/>1. 只有在交易成功的状态下才能取到交易佣金，其它状态下取到的都是零或空值
        <br/>2. 只有单笔订单的情况下Trade数据结构中才包含商品相关的信息
        <br/>3. 获取到的Order中的payment字段在单笔子订单时包含物流费用，多笔子订单时不包含物流费用
        <br/>4. 获取红包优惠金额可以使用字段 coupon_fee
        <br/>5. 请按需获取字段，减少TOP系统的压力
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=54

        :param fields: 需要返回的字段列表，多个字段用半角逗号分隔，可选值为返回示例中能看到的所有字段。
        :param tid: 交易编号
        """
        return self._top_request(
            "taobao.trade.fullinfo.get",
            {
                "fields": fields,
                "tid": tid
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trade_close(
            self,
            tid,
            close_reason
    ):
        """
        卖家关闭一笔交易
        关闭一笔订单，可以是主订单或子订单。当订单从创建到关闭时间小于10s的时候，会报“CLOSE_TRADE_TOO_FAST”错误。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=83

        :param tid: 主订单或子订单编号。
        :param close_reason: 交易关闭原因。可以选择的理由有：1.未及时付款2、买家不想买了3、买家信息填写错误，重新拍4、恶意买家/同行捣乱5、缺货6、买家拍错了7、同城见面交易
        """
        return self._top_request(
            "taobao.trade.close",
            {
                "tid": tid,
                "close_reason": close_reason
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trades_sold_increment_get(
            self,
            fields,
            start_modified,
            end_modified,
            status='',
            type='',
            buyer_nick='',
            ext_type='',
            tag='',
            page_no='1',
            page_size='40',
            use_has_next='false'
    ):
        """
        查询卖家已卖出的增量交易数据（根据修改时间）
        搜索当前会话用户作为卖家已卖出的增量交易数据（只能获取到三个月以内的交易信息）<br/><br/>1. 一次请求只能查询时间跨度为一天的增量交易记录，即end_modified - start_modified <= 1天。<br/><br/>2. 返回的数据结果是以订单的修改时间倒序排列的，通过从后往前翻页的方式可以避免漏单问题。<br/><br/>3. 返回的数据结果只包含了订单的部分数据，可通过taobao.trade.fullinfo.get获取订单详情。<br/><br/>4. <span style='color:red'>使用<a href='http://open.taobao.com/doc/detail.htm?spm=0.0.0.0.F9TTxy&id=101744'>消息服务</a>监听订单变更事件，可以实时获取订单更新数据。</span><br/><br/>注意：type字段的说明，如果该字段不传，接口默认只查4种类型订单，非默认查询的订单是不返回。遇到订单查不到的情况的，通常都是这个原因造成。解决办法就是type加上订单类型就可正常返回了。用taobao.trade.fullinfo.get 查订单fields返回type 很容易的能知道订单的类型（type）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=128

        :param fields: 需要返回的字段列表，多个字段用半角逗号分隔，可选值为返回示例中能看到的所有字段。rx_audit_status=0为处方药未审核状态
        :param start_modified: 查询修改开始时间(修改时间跨度不能大于一天)。格式:yyyy-MM-dd HH:mm:ss
        :param end_modified: 查询修改结束时间，必须大于修改开始时间(修改时间跨度不能大于一天)，格式:yyyy-MM-dd HH:mm:ss。<span style='color:red;font-weight: bold;'>建议使用30分钟以内的时间跨度，能大大提高响应速度和成功率</span>。
        :param status: 交易状态（<a href='http://open.taobao.com/doc/detail.htm?id=102856' target='_blank'>查看可选值</a>），默认查询所有交易状态的数据，除了默认值外每次只能查询一种状态。
        :param type: 交易类型列表（<a href='http://open.taobao.com/doc/detail.htm?id=102855' target='_blank'>查看可选值</a>），一次查询多种类型可用半角逗号分隔，默认同时查询guarantee_trade,auto_delivery,ec,cod,step这5种类型的数据。
        :param buyer_nick: 买家昵称
        :param ext_type: 可选值有ershou(二手市场的订单）,service（商城服务子订单）mark（双十一大促活动异常订单）作为扩展类型筛选只能做单个ext_type查询，不能全部查询所有的ext_type类型
        :param tag: 卖家对交易的自定义分组标签，目前可选值为：time_card（点卡软件代充），fee_card（话费软件代充）
        :param page_no: 页码。取值范围:大于零的整数;默认值:1。<span style='color:red;font-weight: bold;'>注：必须采用倒序的分页方式（从最后一页往回取）才能避免漏单问题。</span>
        :param page_size: 每页条数。取值范围：1~100，默认值：40。<span style='color:red;font-weight: bold;'>建议使用40~50，可以提高成功率，减少超时数量</span>。
        :param use_has_next: 是否启用has_next的分页方式，如果指定true,则返回的结果中不包含总记录数，但是会新增一个是否存在下一页的的字段，<span style='color:red;font-weight: bold;'>通过此种方式获取增量交易，效率在原有的基础上有80%的提升</span>。
        """
        return self._top_request(
            "taobao.trades.sold.increment.get",
            {
                "fields": fields,
                "start_modified": start_modified,
                "end_modified": end_modified,
                "status": status,
                "type": type,
                "buyer_nick": buyer_nick,
                "ext_type": ext_type,
                "tag": tag,
                "page_no": page_no,
                "page_size": page_size,
                "use_has_next": use_has_next
            }
        )

    def taobao_trade_confirmfee_get(
            self,
            tid
    ):
        """
        获取交易确认收货费用
        获取交易确认收货费用，可以获取主订单或子订单的确认收货费用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=146

        :param tid: 交易主订单或子订单ID
        """
        return self._top_request(
            "taobao.trade.confirmfee.get",
            {
                "tid": tid
            },
            result_processor=lambda x: x["trade_confirm_fee"]
        )

    def taobao_trade_ordersku_update(
            self,
            oid,
            sku_id='',
            sku_props=''
    ):
        """
        更新交易的销售属性
        只能更新发货前子订单的销售属性 <br/>只能更新价格相同的销售属性。对于拍下减库存的交易会同步更新销售属性的库存量。对于旺店的交易，要使用商品扩展信息中的SKU价格来比较。 <br/>必须使用sku_id或sku_props中的一个参数来更新，如果两个都传的话，sku_id优先
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=240

        :param oid: 子订单编号（对于单笔订单的交易可以传交易编号）。
        :param sku_id: 销售属性编号，可以通过taobao.item.skus.get获取订单对应的商品的所有销售属性。
        :param sku_props: 销售属性组合串，格式：p1:v1;p2:v2，如：1627207:28329;20509:28314。可以通过taobao.item.skus.get获取订单对应的商品的所有销售属性。
        """
        return self._top_request(
            "taobao.trade.ordersku.update",
            {
                "oid": oid,
                "sku_id": sku_id,
                "sku_props": sku_props
            },
            result_processor=lambda x: x["order"]
        )

    def taobao_trade_shippingaddress_update(
            self,
            tid,
            receiver_name='',
            receiver_phone='',
            receiver_mobile='',
            receiver_state='',
            receiver_city='',
            receiver_district='',
            receiver_address='',
            receiver_zip=''
    ):
        """
        更改交易的收货地址
        只能更新一笔交易里面的买家收货地址 <br/>只能更新发货前（即买家已付款，等待卖家发货状态）的交易的买家收货地址 <br/>更新后的发货地址可以通过taobao.trade.fullinfo.get查到 <br/>参数中所说的字节为GBK编码的（英文和数字占1字节，中文占2字节）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=241

        :param tid: 交易编号。
        :param receiver_name: 收货人全名。最大长度为50个字节。
        :param receiver_phone: 固定电话。最大长度为30个字节。
        :param receiver_mobile: 移动电话。最大长度为11个字节。
        :param receiver_state: 省份。最大长度为32个字节。如：浙江
        :param receiver_city: 城市。最大长度为32个字节。如：杭州
        :param receiver_district: 区/县。最大长度为32个字节。如：西湖区
        :param receiver_address: 收货地址。最大长度为228个字节。
        :param receiver_zip: 邮政编码。必须由6个数字组成。
        """
        return self._top_request(
            "taobao.trade.shippingaddress.update",
            {
                "tid": tid,
                "receiver_name": receiver_name,
                "receiver_phone": receiver_phone,
                "receiver_mobile": receiver_mobile,
                "receiver_state": receiver_state,
                "receiver_city": receiver_city,
                "receiver_district": receiver_district,
                "receiver_address": receiver_address,
                "receiver_zip": receiver_zip
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trade_amount_get(
            self,
            tid,
            fields
    ):
        """
        交易帐务查询
        卖家查询该笔交易的资金帐务相关的数据；
        1. 只供卖家使用，买家不可使用
        2. 可查询所有的状态的交易，但不同状态时交易的相关数据可能会有不同
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10481

        :param tid: 交易编号
        :param fields: 订单帐务详情需要返回的字段信息，可选值如下：1. TradeAmount中可指定的fields：tid,alipay_no,created,pay_time,end_time,total_fee,payment,post_fee,cod_fee,commission_fee,buyer_obtain_point_fee2. OrderAmount中可指定的fields：order_amounts.oid,order_amounts.title,order_amounts.num_iid,order_amounts.sku_properties_name,order_amounts.sku_id,order_amounts.num,order_amounts.price,order_amounts.discount_fee,order_amounts.adjust_fee,order_amounts.payment,order_amounts.promotion_name3. order_amounts(返回OrderAmount的所有内容)4. promotion_details(指定该值会返回主订单的promotion_details中除id之外的所有字段)
        """
        return self._top_request(
            "taobao.trade.amount.get",
            {
                "tid": tid,
                "fields": fields
            },
            result_processor=lambda x: x["trade_amount"]
        )

    def taobao_trade_receivetime_delay(
            self,
            tid,
            days
    ):
        """
        延长交易收货时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10560

        :param tid: 主订单号
        :param days: 延长收货的天数，可选值为：3, 5, 7, 10。
        """
        return self._top_request(
            "taobao.trade.receivetime.delay",
            {
                "tid": tid,
                "days": days
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trade_postage_update(
            self,
            tid,
            post_fee
    ):
        """
        修改交易邮费价格
        修改订单邮费接口，通过传入订单编号和邮费价格，修改订单的邮费，返回修改时间modified,邮费post_fee,总费用total_fee。
        <br/> <span style='color:red'> API取消加邮费功能通知：http://open.taobao.com/support/announcement_detail.htm?tid=24750</span>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10897

        :param tid: 主订单编号
        :param post_fee: 邮费价格(邮费单位是元）
        """
        return self._top_request(
            "taobao.trade.postage.update",
            {
                "tid": tid,
                "post_fee": post_fee
            },
            result_processor=lambda x: x["trade"]
        )

    def taobao_trades_sold_incrementv_get(
            self,
            fields,
            start_create,
            end_create,
            status='',
            type='',
            ext_type='',
            tag='',
            page_no='1',
            page_size='40',
            use_has_next='false'
    ):
        """
        查询卖家已卖出的增量交易数据（根据入库时间）
        搜索当前会话用户作为卖家已卖出的增量交易数据（只能获取到三个月以内的交易信息） <br/><br/>1. 一次请求只能查询时间跨度为一天的增量交易记录，即end_create - start_create <= 1天。 <br/><br/>2. 返回的数据结果是以订单入库时间的倒序排列的(该时间和订单修改时间不同)，通过从后往前翻页的方式可以避免漏单问题。 <br/><br/>3. 返回的数据结果只包含了订单的部分数据，可通过taobao.trade.fullinfo.get获取订单详情。 <br/><br/>4. 使用主动通知监听订单变更事件，可以实时获取订单更新数据。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21470

        :param fields: 需要返回的字段列表，多个字段用半角逗号分隔，可选值为返回示例中能看到的所有字段。
        :param start_create: 查询入库开始时间(修改时间跨度不能大于一天)。格式:yyyy-MM-dd HH:mm:ss
        :param end_create: 查询入库结束时间，必须大于入库开始时间(修改时间跨度不能大于一天)，格式:yyyy-MM-dd HH:mm:ss。<span style='color:red;font-weight: bold;'>建议使用30分钟以内的时间跨度，能大大提高响应速度和成功率</span>。
        :param status: 交易状态（<a href='http://open.taobao.com/doc/detail.htm?id=102856' target='_blank'>查看可选值</a>），默认查询所有交易状态的数据，除了默认值外每次只能查询一种状态。
        :param type: 交易类型列表（<a href='http://open.taobao.com/doc/detail.htm?id=102855' target='_blank'>查看可选值</a>），一次查询多种类型可用半角逗号分隔，默认同时查询guarantee_trade,auto_delivery,ec,cod,step这5种类型的数据。
        :param ext_type: 可选值有ershou(二手市场的订单）,service（商城服务子订单）mark（双十一大促活动异常订单）作为扩展类型筛选只能做单个ext_type查询，不能全部查询所有的ext_type类型
        :param tag: 卖家对交易的自定义分组标签，目前可选值为：time_card（点卡软件代充），fee_card（话费软件代充）
        :param page_no: 页码。取值范围:大于零的整数;默认值:1。<span style='color:red;font-weight: bold;'>注：必须采用倒序的分页方式（从最后一页往回取）才能避免漏单问题。</span>
        :param page_size: 每页条数。取值范围：1~100，默认值：40。<span style='color:red;font-weight: bold;'>建议使用40~50，可以提高成功率，减少超时数量</span>。
        :param use_has_next: 是否启用has_next的分页方式，如果指定true,则返回的结果中不包含总记录数，但是会新增一个是否存在下一页的的字段，<span style='color:red;font-weight: bold;'>通过此种方式获取增量交易，效率在原有的基础上有80%的提升</span>。
        """
        return self._top_request(
            "taobao.trades.sold.incrementv.get",
            {
                "fields": fields,
                "start_create": start_create,
                "end_create": end_create,
                "status": status,
                "type": type,
                "ext_type": ext_type,
                "tag": tag,
                "page_no": page_no,
                "page_size": page_size,
                "use_has_next": use_has_next
            }
        )

    def taobao_trade_acquire_online_precreate(
            self,
            cart_info,
            guide_id,
            guide_name,
            shop_id,
            shop_name,
            delivery_type,
            total_fee='',
            ext_info=''
    ):
        """
        导购宝预下单
        提供导购宝预下单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23219

        :param cart_info: 购物车信息，格式为：商品id:skuId:购买数量。一次购买多个商品，用分号隔开。
        :param guide_id: 导购员id
        :param guide_name: 导购员姓名
        :param shop_id: 店铺id
        :param shop_name: 店铺名称
        :param delivery_type: 是店内提货还是线上发货，店内提货传入:inshop；线上发货传入：online
        :param total_fee: 修改订单总额，单位为分
        :param ext_info: 传入扩展参数，目前包含外部订单号设置
        """
        return self._top_request(
            "taobao.trade.acquire.online.precreate",
            {
                "cart_info": cart_info,
                "guide_id": guide_id,
                "guide_name": guide_name,
                "shop_id": shop_id,
                "shop_name": shop_name,
                "delivery_type": delivery_type,
                "total_fee": total_fee,
                "ext_info": ext_info
            },
            result_processor=lambda x: x["online_precreate_order_result"]
        )

    def alibaba_trade_aliance_create(
            self,
            param_isv_create_order_param=None
    ):
        """
        推客平台订单回流
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29515

        :param param_isv_create_order_param: 下单请求
        """
        return self._top_request(
            "alibaba.trade.aliance.create",
            {
                "param_isv_create_order_param": param_isv_create_order_param
            }
        )

    def alibaba_wdk_trade_order_query(
            self,
            query=None
    ):
        """
        查询外部交易订单接口
        通过该接口可以在盒马查询交易订单，并处理相关业务流程。主要用于和外部商户的订单进行同步和融合业务流程处理
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33907

        :param query: 订单查询
        """
        return self._top_request(
            "alibaba.wdk.trade.order.query",
            {
                "query": query
            }
        )

    def alibaba_wdk_trade_order_create(
            self,
            trade=None
    ):
        """
        外部交易订单创单接口
        通过该接口可以再盒马创建交易订单，并处理相关业务流程。主要用于和外部商户的订单进行同步和融合业务流程处理
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33908

        :param trade: 待创建的订单
        """
        return self._top_request(
            "alibaba.wdk.trade.order.create",
            {
                "trade": trade
            }
        )

    def alibaba_wdk_trade_order_cancel(
            self,
            trade=None
    ):
        """
        外部交易订单取消接口
        通过该接口可以再盒马取消交易订单，并处理相关业务流程。主要用于和外部商户的订单进行同步和融合业务流程处理
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33912

        :param trade: 待取消的订单
        """
        return self._top_request(
            "alibaba.wdk.trade.order.cancel",
            {
                "trade": trade
            }
        )

    def taobao_trade_fullinfo_get_customization(
            self,
            fields,
            tid
    ):
        """
        获取单笔交易业务定制的详细信息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        获取单笔交易的详细信息
        <br/>1. 只有在交易成功的状态下才能取到交易佣金，其它状态下取到的都是零或空值
        <br/>2. 只有单笔订单的情况下Trade数据结构中才包含商品相关的信息
        <br/>3. 获取到的Order中的payment字段在单笔子订单时包含物流费用，多笔子订单时不包含物流费用
        <br/>4. 获取红包优惠金额可以使用字段 coupon_fee
        <br/>5. 请按需获取字段，减少TOP系统的压力
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35072

        :param fields: 需要返回的字段列表，多个字段用半角逗号分隔，可选值为返回示例中能看到的所有字段。
        :param tid: 交易编号
        """
        return self._top_request(
            "taobao.trade.fullinfo.get.customization",
            {
                "fields": fields,
                "tid": tid
            },
            result_processor=lambda x: x["trade"]
        )

    def alibaba_wdk_trade_refund_create(
            self,
            refund_goods_create_request=None
    ):
        """
        外部渠道逆向订单创建
        该接口是创建退货订单的服务。当外部渠道发起退款后，调用此接口可以完成五道口底层交易、履约、配送等一系列流程进行退货。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36172

        :param refund_goods_create_request: 退货请求
        """
        return self._top_request(
            "alibaba.wdk.trade.refund.create",
            {
                "refund_goods_create_request": refund_goods_create_request
            }
        )

    def alibaba_wdk_trade_refund_query(
            self,
            refund_goods_query_request=None
    ):
        """
        外部渠道查询退货订单详情接口
        该接口提供给外部渠道商家，比如欧尚外卖等查询退货订单详情，里面包含退货进度等信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36173

        :param refund_goods_query_request: 查询请求
        """
        return self._top_request(
            "alibaba.wdk.trade.refund.query",
            {
                "refund_goods_query_request": refund_goods_query_request
            },
            result_processor=lambda x: x["refund_goods_query_result"]
        )

    def alibaba_wdk_trade_refund_inform(
            self,
            inform_refund_success_request=None
    ):
        """
        外部渠道通知淘鲜达退款成功接口
        该接口用于外部渠道退款成功后，通知淘鲜达底层履约完成退款流程。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36213

        :param inform_refund_success_request: 通知退款成功请求
        """
        return self._top_request(
            "alibaba.wdk.trade.refund.inform",
            {
                "inform_refund_success_request": inform_refund_success_request
            }
        )

    def alibaba_omni_saas_order_create(
            self,
            goods_details,
            store_id,
            buyer_id_type,
            pay_channel,
            store_id_type,
            request_no,
            buyer_id='',
            device='',
            device_no='',
            operator_id='',
            coupon_infos=None
    ):
        """
        订单创建接口
        服务商利用现有的saas系统和阿里完成交易系统的对接
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37282

        :param goods_details: 商品列表
        :param store_id: 门店ID
        :param buyer_id_type: ALIPAY：支付宝用户；TAOBAO：淘宝会员码；MOBILE：手机号
        :param pay_channel: ALIPAY：支付宝付款；BANK_CARD：刷卡
        :param store_id_type: PLACE：淘宝商户中心门店ID；CUSTOM：商户自有门店编码，需要维护到淘宝商户中心
        :param request_no: 请求号，用于标识一次请求
        :param buyer_id: 买家标识，淘系用户或用户手机号。当支付渠道为支付宝时，此字段为淘宝会员码或支付宝付款码。(当前仅支持淘系用户，手机号下单稍后开放)
        :param device: 收银设备类型
        :param device_no: 收银设备号
        :param operator_id: 收银员标识
        :param coupon_infos: 商家自有优惠
        """
        return self._top_request(
            "alibaba.omni.saas.order.create",
            {
                "goods_details": goods_details,
                "store_id": store_id,
                "buyer_id_type": buyer_id_type,
                "pay_channel": pay_channel,
                "store_id_type": store_id_type,
                "request_no": request_no,
                "buyer_id": buyer_id,
                "device": device,
                "device_no": device_no,
                "operator_id": operator_id,
                "coupon_infos": coupon_infos
            }
        )

    def tmall_ascp_orders_sale_create(
            self,
            channel_order_request=None
    ):
        """
        ASCP渠道中心销售单创建接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38211

        :param channel_order_request: 请求对象
        """
        return self._top_request(
            "tmall.ascp.orders.sale.create",
            {
                "channel_order_request": channel_order_request
            }
        )

    def taobao_rdc_aligenius_ordermsg_update(
            self,
            oid,
            status,
            tid
    ):
        """
        订单消息状态回传
        用于订单消息处理状态回传
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39964

        :param oid: 子订单（消息中传的子订单）
        :param status: 处理状态，1=成功，2=处理失败
        :param tid: 主订单（消息中传的主订单）
        """
        return self._top_request(
            "taobao.rdc.aligenius.ordermsg.update",
            {
                "oid": oid,
                "status": status,
                "tid": tid
            }
        )

    def taobao_rdc_aligenius_order_event_update(
            self,
            param0=None
    ):
        """
        erp订单相关状态变更信息回传
        供erp回传订单变更状态事件
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41264

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.order.event.update",
            {
                "param0": param0
            }
        )

    def tmall_cloudstore_trade_pos_update(
            self,
            serial_no='',
            request_time='',
            pos_no='',
            order_id=''
    ):
        """
        pos单号回传
        云店交易订单pos单号回传
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41970

        :param serial_no: 流水号
        :param request_time: 请求时间
        :param pos_no: pos单号
        :param order_id: 主订单号
        """
        return self._top_request(
            "tmall.cloudstore.trade.pos.update",
            {
                "serial_no": serial_no,
                "request_time": request_time,
                "pos_no": pos_no,
                "order_id": order_id
            }
        )

    def taobao_trade_voucher_upload(
            self,
            file_name,
            file_data,
            seller_nick='',
            buyer_nick=''
    ):
        """
        淘宝交易凭证上传
        定制化交易流程中，涉及到 买家自定义 图片、声音、视频 等多富媒体文件，且该商品或服务的附属sku，通过此接口上传作为交易凭证。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26451

        :param file_name: 上传文件的名称
        :param file_data: 文件
        :param seller_nick: 该笔订单的卖家Nick
        :param buyer_nick: 该笔订单的买家Nick（混淆nick）
        """
        return self._top_request(
            "taobao.trade.voucher.upload",
            {
                "file_name": file_name,
                "file_data": file_data,
                "seller_nick": seller_nick,
                "buyer_nick": buyer_nick
            },
            result_processor=lambda x: x["file"]
        )

    def cainiao_refund_refundactions_display(
            self,
            param0=None
    ):
        """
        退货退款操作的展示信息(展现给买家)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43911

        :param param0: 请求入参
        """
        return self._top_request(
            "cainiao.refund.refundactions.display",
            {
                "param0": param0
            }
        )

    def cainiao_refund_refundactions_get(
            self,
            order_id
    ):
        """
        判断该订单能执行的逆向操作集合列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43912

        :param order_id: 子订单ID
        """
        return self._top_request(
            "cainiao.refund.refundactions.get",
            {
                "order_id": order_id
            }
        )

    def cainiao_refund_refundactions_judgement(
            self,
            param0=None
    ):
        """
        判断当前用户是否能对订单执行一些逆向操作，比如退货操作
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43913

        :param param0: 操作请求
        """
        return self._top_request(
            "cainiao.refund.refundactions.judgement",
            {
                "param0": param0
            }
        )


class TbPingJia(DingTalkBaseAPI):
    """
    评价API
    """

    def taobao_traderates_get(
            self,
            fields,
            rate_type,
            role,
            result='',
            page_no='',
            page_size='40',
            start_date='',
            end_date='',
            tid='',
            use_has_next='false',
            num_iid=''
    ):
        """
        搜索评价信息
        搜索评价信息，只能获取距今180天内的评价记录(只支持查询卖家给出或得到的评价)。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=55

        :param fields: 需返回的字段列表。可选值：TradeRate 结构中的所有字段，多个字段之间用“,”分隔
        :param rate_type: 评价类型。可选值:get(得到),give(给出)
        :param role: 评价者角色即评价的发起方。可选值:seller(卖家),buyer(买家)。 当 give buyer 以买家身份给卖家的评价； 当 get seller 以买家身份得到卖家给的评价； 当 give seller 以卖家身份给买家的评价； 当 get buyer 以卖家身份得到买家给的评价。
        :param result: 评价结果。可选值:good(好评),neutral(中评),bad(差评)
        :param page_no: 页码。取值范围:大于零的整数最大限制为200; 默认值:1
        :param page_size: 每页获取条数。默认值40，最小值1，最大值150。
        :param start_date: 评价开始时。如果只输入开始时间，那么能返回开始时间之后的评价数据。
        :param end_date: 评价结束时间。如果只输入结束时间，那么全部返回所有评价数据。
        :param tid: 交易订单id，可以是父订单id号，也可以是子订单id号
        :param use_has_next: 是否启用has_next的分页方式，如果指定true,则返回的结果中不包含总记录数，但是会新增一个是否存在下一页的的字段，通过此种方式获取评价信息，效率在原有的基础上有80%的提升。
        :param num_iid: 商品的数字ID
        """
        return self._top_request(
            "taobao.traderates.get",
            {
                "fields": fields,
                "rate_type": rate_type,
                "role": role,
                "result": result,
                "page_no": page_no,
                "page_size": page_size,
                "start_date": start_date,
                "end_date": end_date,
                "tid": tid,
                "use_has_next": use_has_next,
                "num_iid": num_iid
            }
        )

    def taobao_traderate_add(
            self,
            tid,
            result,
            role,
            oid='',
            content='',
            anony=''
    ):
        """
        新增单个评价
        新增单个评价(<font color='red'>注：在评价之前需要对订单成功的时间进行判定（end_time）,如果超过15天，不能再通过该接口进行评价</font>)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=56

        :param tid: 交易ID
        :param result: 评价结果,可选值:good(好评),neutral(中评),bad(差评)
        :param role: 评价者角色,可选值:seller(卖家),buyer(买家)
        :param oid: 子订单ID
        :param content: 评价内容,最大长度: 500个汉字 .注意：当评价结果为good时就不用输入评价内容.评价内容为neutral/bad的时候需要输入评价内容
        :param anony: 是否匿名,卖家评不能匿名。可选值:true(匿名),false(非匿名)。注意：如果交易订单匿名，则评价也匿名
        """
        return self._top_request(
            "taobao.traderate.add",
            {
                "tid": tid,
                "result": result,
                "role": role,
                "oid": oid,
                "content": content,
                "anony": anony
            },
            result_processor=lambda x: x["trade_rate"]
        )

    def taobao_traderate_list_add(
            self,
            tid,
            result,
            role,
            content='',
            anony=''
    ):
        """
        针对父子订单新增批量评价
        针对父子订单新增批量评价(<font color='red'>注：在评价之前需要对订单成功的时间进行判定（end_time）,如果超过15天，不用再通过该接口进行评价</font>)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=57

        :param tid: 交易ID
        :param result: 评价结果。可选值:good(好评),neutral(中评),bad(差评)
        :param role: 评价者角色。可选值:seller(卖家),buyer(买家)
        :param content: 评价内容,最大长度: 500个汉字 .注意：当评价结果为good时就不用输入评价内容.评价内容为neutral/bad的时候需要输入评价内容
        :param anony: 是否匿名，卖家评不能匿名。可选值:true(匿名),false(非匿名)。 注意：如果买家匿名购买，那么买家的评价默认匿名
        """
        return self._top_request(
            "taobao.traderate.list.add",
            {
                "tid": tid,
                "result": result,
                "role": role,
                "content": content,
                "anony": anony
            },
            result_processor=lambda x: x["trade_rate"]
        )

    def taobao_traderate_explain_add(
            self,
            oid,
            reply
    ):
        """
        商城评价解释接口
        商城卖家给评价做出解释（买家追加评论后、评价超过30天的，都不能再做评价解释）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21158

        :param oid: 子订单ID
        :param reply: 评价解释内容,最大长度: 500个汉字
        """
        return self._top_request(
            "taobao.traderate.explain.add",
            {
                "oid": oid,
                "reply": reply
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_traderate_impr_imprword_byfeedid_get(
            self,
            child_trade_id
    ):
        """
        根据交易订单id和卖家nick查询到该条评价的印象标签
        根据卖家nick和交易id（如果是子订单，输入子订单id），查询到该条评价的大家印象标签
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21897

        :param child_trade_id: 交易订单id号（如果包含子订单，请使用子订单id号）
        """
        return self._top_request(
            "taobao.traderate.impr.imprword.byfeedid.get",
            {
                "child_trade_id": child_trade_id
            },
            result_processor=lambda x: x["impr_feed"]
        )

    def taobao_traderate_impr_imprwords_get(
            self,
            cat_root_id,
            cat_leaf_id=''
    ):
        """
        评价大家印象印象短语接口
        根据淘宝后台类目的一级类目和叶子类目
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21899

        :param cat_root_id: 淘宝一级类目id
        :param cat_leaf_id: 淘宝叶子类目id
        """
        return self._top_request(
            "taobao.traderate.impr.imprwords.get",
            {
                "cat_root_id": cat_root_id,
                "cat_leaf_id": cat_leaf_id
            },
            result_processor=lambda x: x["impr_words"]
        )

    def tmall_traderate_feeds_get(
            self,
            child_trade_id
    ):
        """
        查询子订单对应的评价、追评以及语义标签
        通过子订单ID获取天猫订单对应的评价，追评，以及对应的语义标签
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22532

        :param child_trade_id: 交易子订单ID
        """
        return self._top_request(
            "tmall.traderate.feeds.get",
            {
                "child_trade_id": child_trade_id
            },
            result_processor=lambda x: x["tmall_rate_info"]
        )

    def tmall_traderate_itemtags_get(
            self,
            item_id
    ):
        """
        通过商品ID获取标签列表
        通过商品ID获取标签详细信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22533

        :param item_id: 商品ID
        """
        return self._top_request(
            "tmall.traderate.itemtags.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["tags"]
        )


class TbWuLiu(DingTalkBaseAPI):
    """
    物流API
    """

    def taobao_areas_get(
            self,
            fields
    ):
        """
        查询地址区域
        查询标准地址区域代码信息。可以直接参考最新的行政区域代码：
        <a href='http://www.stats.gov.cn/tjsj/tjbz/tjyqhdmhcxhfdm/2016/index.html'>http://www.stats.gov.cn/tjsj/tjbz/tjyqhdmhcxhfdm/2016/index.html</a>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=59

        :param fields: 需返回的字段列表.可选值:Area 结构中的所有字段;多个字段之间用','分隔.如:id,type,name,parent_id,zip.
        """
        return self._top_request(
            "taobao.areas.get",
            {
                "fields": fields
            },
            result_processor=lambda x: x["areas"]
        )

    def taobao_logistics_companies_get(
            self,
            fields,
            is_recommended='',
            order_mode=''
    ):
        """
        查询物流公司信息
        查询淘宝网合作的物流公司信息，用于发货接口。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=233

        :param fields: 需返回的字段列表。可选值:LogisticCompany 结构中的所有字段;多个字段间用','逗号隔开.<br/>如:id,code,name,reg_mail_no<br/><br><font color='red'>说明：</font><br/><br>id：物流公司ID<br/><br>code：物流公司code<br/><br>name：物流公司名称<br/><br>reg_mail_no：物流公司对应的运单规则
        :param is_recommended: 是否查询推荐物流公司.可选值:true,false.如果不提供此参数,将会返回所有支持电话联系的物流公司.
        :param order_mode: 推荐物流公司的下单方式.可选值:offline(电话联系/自己联系),online(在线下单),all(即电话联系又在线下单). 此参数仅仅用于is_recommended 为ture时。就是说对于推荐物流公司才可用.如果不选择此参数将会返回推荐物流中支持电话联系的物流公司.
        """
        return self._top_request(
            "taobao.logistics.companies.get",
            {
                "fields": fields,
                "is_recommended": is_recommended,
                "order_mode": order_mode
            },
            result_processor=lambda x: x["logistics_companies"]
        )

    def taobao_logistics_orders_detail_get(
            self,
            fields,
            tid='',
            buyer_nick='',
            status='',
            seller_confirm='',
            receiver_name='',
            start_created='',
            end_created='',
            freight_payer='',
            type='',
            page_no='1',
            page_size='40'
    ):
        """
        批量查询物流订单,返回详细信息
        查询物流订单的详细信息，涉及用户隐私字段。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=234

        :param fields: 需返回的字段列表.可选值:Shipping 物流数据结构中所有字段.fileds中可以指定返回以上任意一个或者多个字段,以','分隔.
        :param tid: 交易ID.如果加入tid参数的话,不用传其他的参数,但是仅会返回一条物流订单信息.
        :param buyer_nick: 买家昵称
        :param status: 物流状态.可查看数据结构 Shipping 中的status字段.
        :param seller_confirm: 卖家是否发货.可选值:yes(是),no(否).如:yes.
        :param receiver_name: 收货人姓名
        :param start_created: 创建时间开始.格式:yyyy-MM-dd HH:mm:ss
        :param end_created: 创建时间结束.格式:yyyy-MM-dd HH:mm:ss
        :param freight_payer: 谁承担运费.可选值:buyer(买家),seller(卖家).如:buyer
        :param type: 物流方式.可选值:post(平邮),express(快递),ems(EMS).如:post
        :param page_no: 页码.该字段没传 或 值<1 ,则默认page_no为1
        :param page_size: 每页条数.该字段没传 或 值<1 ，则默认page_size为40
        """
        return self._top_request(
            "taobao.logistics.orders.detail.get",
            {
                "fields": fields,
                "tid": tid,
                "buyer_nick": buyer_nick,
                "status": status,
                "seller_confirm": seller_confirm,
                "receiver_name": receiver_name,
                "start_created": start_created,
                "end_created": end_created,
                "freight_payer": freight_payer,
                "type": type,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_logistics_orders_get(
            self,
            fields,
            tid='',
            buyer_nick='',
            status='',
            seller_confirm='',
            receiver_name='',
            start_created='',
            end_created='',
            freight_payer='',
            type='',
            page_no='1',
            page_size='40'
    ):
        """
        批量查询物流订单
        批量查询物流订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=235

        :param fields: 需返回的字段列表.可选值:Shipping 物流数据结构中的以下字段: <br><br/>tid,order_code,seller_nick,buyer_nick,delivery_start, delivery_end,out_sid,item_title,receiver_name, created,modified,status,type,freight_payer,seller_confirm,company_name,sub_tids,is_spilt；<br>多个字段之间用','分隔。如tid,seller_nick,buyer_nick,delivery_start。
        :param tid: 交易ID.如果加入tid参数的话,不用传其他的参数,若传入tid：非拆单场景，仅会返回一条物流订单信息；拆单场景，会返回多条物流订单信息
        :param buyer_nick: 买家昵称
        :param status: 物流状态.查看数据结构 Shipping 中的status字段.
        :param seller_confirm: 卖家是否发货.可选值:yes(是),no(否).如:yes
        :param receiver_name: 收货人姓名
        :param start_created: 创建时间开始
        :param end_created: 创建时间结束
        :param freight_payer: 谁承担运费.可选值:buyer(买家),seller(卖家).如:buyer
        :param type: 物流方式.可选值:post(平邮),express(快递),ems(EMS).如:post
        :param page_no: 页码.该字段没传 或 值<1 ,则默认page_no为1
        :param page_size: 每页条数.该字段没传 或 值<1 ,则默认page_size为40
        """
        return self._top_request(
            "taobao.logistics.orders.get",
            {
                "fields": fields,
                "tid": tid,
                "buyer_nick": buyer_nick,
                "status": status,
                "seller_confirm": seller_confirm,
                "receiver_name": receiver_name,
                "start_created": start_created,
                "end_created": end_created,
                "freight_payer": freight_payer,
                "type": type,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_logistics_partners_get(
            self,
            source_id='0',
            target_id='0',
            service_type='',
            goods_value='0',
            is_need_carriage='false'
    ):
        """
        查询支持起始地到目的地范围的物流公司
        查询物流公司信息（可以查询目的地可不可达情况）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=271

        :param source_id: 物流公司揽货地地区码（必须是区、县一级的）.参考:http://www.stats.gov.cn/tjsj/tjbz/xzqhdm/201401/t20140116_501070.html或者调用 taobao.areas.get 获取
        :param target_id: 物流公司派送地地区码（必须是区、县一级的）.参考:http://www.stats.gov.cn/tjsj/tjbz/xzqhdm/201401/t20140116_501070.html或者调用 taobao.areas.get 获取
        :param service_type: 服务类型，根据此参数可查出提供相应服务类型的物流公司信息(物流公司状态正常)，可选值：cod(货到付款)、online(在线下单)、 offline(自己联系)、limit(限时物流)。然后再根据source_id,target_id,goods_value这三个条件来过滤物流公司. 目前输入自己联系服务类型将会返回空，因为自己联系并没有具体的服务信息，所以不会有记录。
        :param goods_value: 货物价格.只有当选择货到付款此参数才会有效
        :param is_need_carriage: 是否需要揽收资费信息，默认false。在此值为false时，返回内容中将无carriage。在未设置source_id或target_id的情况下，无法查询揽收资费信息。自己联系无揽收资费记录。
        """
        return self._top_request(
            "taobao.logistics.partners.get",
            {
                "source_id": source_id,
                "target_id": target_id,
                "service_type": service_type,
                "goods_value": goods_value,
                "is_need_carriage": is_need_carriage
            },
            result_processor=lambda x: x["logistics_partners"]
        )

    def taobao_logistics_trace_search(
            self,
            tid,
            is_split='0',
            sub_tid=''
    ):
        """
        物流流转信息查询
        用户根据淘宝交易号查询物流流转信息，如2010-8-10 15：23：00到达杭州集散地。<br/>此接口的返回信息都由物流公司提供。（备注：使用线下发货（offline.send）的运单，不支持运单状态的实时跟踪，只要一发货，状态就会变为<status>对方已签收</status>，该字段仅对线上发货（online.send）的运单有效。）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10463

        :param tid: 淘宝交易号，请勿传非淘宝交易号
        :param is_split: 表明是否是拆单，默认值0，1表示拆单
        :param sub_tid: 拆单子订单列表，当is_split=1时，需要传人；对应的数据是：子订单号的列表。
        """
        return self._top_request(
            "taobao.logistics.trace.search",
            {
                "tid": tid,
                "is_split": is_split,
                "sub_tid": sub_tid
            }
        )

    def taobao_logistics_address_search(
            self,
            rdef=''
    ):
        """
        查询卖家地址库
        通过此接口查询卖家地址库，
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10683

        :param rdef: 可选，参数列表如下：<br><font color='red'>no_def:查询非默认地址<br>get_def:查询默认取货地址，也即对应卖家后台地址库中发货地址（send_def暂不起作用）<br>cancel_def:查询默认退货地址<br>缺省此参数时，查询所有当前用户的地址库</font>
        """
        return self._top_request(
            "taobao.logistics.address.search",
            {
                "rdef": rdef
            },
            result_processor=lambda x: x["addresses"]
        )

    def taobao_logistics_address_remove(
            self,
            contact_id
    ):
        """
        删除卖家地址库
        用此接口删除卖家地址库
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10684

        :param contact_id: 地址库ID
        """
        return self._top_request(
            "taobao.logistics.address.remove",
            {
                "contact_id": contact_id
            },
            result_processor=lambda x: x["address_result"]
        )

    def taobao_logistics_address_modify(
            self,
            contact_id,
            contact_name,
            province,
            city,
            addr,
            country='',
            zip_code='',
            phone='',
            mobile_phone='',
            seller_company='',
            memo='',
            get_def='',
            cancel_def=''
    ):
        """
        卖家地址库修改
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10685

        :param contact_id: 地址库ID
        :param contact_name: 联系人姓名<br/><font color='red'>长度不可超过20个字节</font>
        :param province: 所在省
        :param city: 所在市
        :param addr: 详细街道地址，不需要重复填写省/市/区
        :param country: 区、县<br/><br><font color='red'>如果所在地区是海外的可以为空，否则为必参</font>
        :param zip_code: 地区邮政编码<br/><br><font color='red'>如果所在地区是海外的可以为空，否则为必参</font>
        :param phone: 电话号码,手机与电话必需有一个
        :param mobile_phone: 手机号码，手机与电话必需有一个 <br><font color='red'>手机号码不能超过20位</font>
        :param seller_company: 公司名称,<br/><br><font color='red'>公司名称长能不能超过20字节</font>
        :param memo: 备注,<br><font color='red'>备注不能超过256字节</font>
        :param get_def: 默认取货地址。<br><br/><font color='red'>选择此项(true)，将当前地址设为默认取货地址，撤消原默认取货地址</font>
        :param cancel_def: 默认退货地址。<br><br/><font color='red'>选择此项(true)，将当前地址设为默认退货地址，撤消原默认退货地址</font>
        """
        return self._top_request(
            "taobao.logistics.address.modify",
            {
                "contact_id": contact_id,
                "contact_name": contact_name,
                "province": province,
                "city": city,
                "addr": addr,
                "country": country,
                "zip_code": zip_code,
                "phone": phone,
                "mobile_phone": mobile_phone,
                "seller_company": seller_company,
                "memo": memo,
                "get_def": get_def,
                "cancel_def": cancel_def
            },
            result_processor=lambda x: x["address_result"]
        )

    def taobao_logistics_address_add(
            self,
            contact_name,
            province,
            city,
            addr,
            country='',
            zip_code='',
            phone='',
            mobile_phone='',
            seller_company='',
            memo='',
            get_def='',
            cancel_def=''
    ):
        """
        卖家地址库新增接口
        通过此接口新增卖家地址库,卖家最多可添加5条地址库,新增第一条卖家地址，将会自动设为默认地址库
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10686

        :param contact_name: 联系人姓名 <font color='red'>长度不可超过20个字节</font>
        :param province: 所在省
        :param city: 所在市
        :param addr: 详细街道地址，不需要重复填写省/市/区
        :param country: 区、县<br/><br><font color='red'>如果所在地区是海外的可以为空，否则为必参</font>
        :param zip_code: 地区邮政编码<br/><br><font color='red'>如果所在地区是海外的可以为空，否则为必参</font>
        :param phone: 电话号码,手机与电话必需有一个
        :param mobile_phone: 手机号码，手机与电话必需有一个<br/><br><font color='red'>手机号码不能超过20位</font>
        :param seller_company: 公司名称,<br><font color='red'>公司名称长能不能超过20字节</font>
        :param memo: 备注,<br><font color='red'>备注不能超过256字节</font>
        :param get_def: 默认取货地址。<br><br/><font color='red'>选择此项(true)，将当前地址设为默认取货地址，撤消原默认取货地址</font>
        :param cancel_def: 默认退货地址。<br><br/><font color='red'>选择此项(true)，将当前地址设为默认退货地址，撤消原默认退货地址</font>
        """
        return self._top_request(
            "taobao.logistics.address.add",
            {
                "contact_name": contact_name,
                "province": province,
                "city": city,
                "addr": addr,
                "country": country,
                "zip_code": zip_code,
                "phone": phone,
                "mobile_phone": mobile_phone,
                "seller_company": seller_company,
                "memo": memo,
                "get_def": get_def,
                "cancel_def": cancel_def
            },
            result_processor=lambda x: x["address_result"]
        )

    def taobao_logistics_online_send(
            self,
            tid,
            company_code,
            sub_tid='',
            is_split='',
            out_sid='',
            sender_id='',
            cancel_id='',
            feature='',
            seller_ip=''
    ):
        """
        在线订单发货处理（支持货到付款）
        用户调用该接口可实现在线订单发货（支持货到付款）<br/>调用该接口实现在线下单发货，有两种情况：<br><br/><font color='red'>如果不输入运单号的情况：交易状态不会改变，需要调用taobao.logistics.online.confirm确认发货后交易状态才会变成卖家已发货。<br><br/>如果输入运单号的情况发货：交易订单状态会直接变成卖家已发货 。</font>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10687

        :param tid: 淘宝交易ID
        :param company_code: 物流公司代码.如'POST'就代表中国邮政,'ZJS'就代表宅急送.调用 taobao.logistics.companies.get 获取。
        :param sub_tid: 需要拆单发货的子订单集合，针对的是一笔交易下有多个子订单需要分开发货的场景；1次可传人多个子订单号，子订单间用逗号隔开；为空表示不做拆单发货。
        :param is_split: 表明是否是拆单，默认值0，1表示拆单
        :param out_sid: 运单号.具体一个物流公司的真实运单号码。淘宝官方物流会校验，请谨慎传入；
        :param sender_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。如果为空，取的卖家的默认取货地址
        :param cancel_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。 如果为空，取的卖家的默认退货地址
        :param feature: feature参数格式 范例: identCode=tid1:识别码1,识别码2|tid2:识别码3;machineCode=tid3:3C机器号A,3C机器号B identCode为识别码的KEY,machineCode为3C的KEY,多个key之间用”;”分隔 “tid1:识别码1,识别码2|tid2:识别码3”为identCode对应的value。 '|'不同商品间的分隔符。 例1商品和2商品，之间就用'|'分开。 TID就是商品代表的子订单号，对应taobao.trade.fullinfo.get 接口获得的oid字段。(通过OID可以唯一定位到当前商品上) ':'TID和具体传入参数间的分隔符。冒号前表示TID,之后代表该商品的参数属性。 ',' 属性间分隔符。（对应商品数量，当存在一个商品的数量超过1个时，用逗号分开）。 具体:当订单中A商品的数量为2个，其中手机串号分别为'12345','67890'。 参数格式：identCode=TIDA:12345,67890。 TIDA对应了A宝贝，冒号后用逗号分隔的'12345','67890'.说明本订单A宝贝的数量为2，值分别为'12345','67890'。 当存在'|'时，就说明订单中存在多个商品，商品间用'|'分隔了开来。|'之后的内容含义同上。retailStoreId=12345，发货门店ID或仓信息。retailStoreType=STORE: 发货门店类别，STORE表示门店，WAREHOUSE表示电商仓。对于全渠道订单回传的商家，retailStoreId和retailStoreType字段为必填字段。
        :param seller_ip: 商家的IP地址
        """
        return self._top_request(
            "taobao.logistics.online.send",
            {
                "tid": tid,
                "company_code": company_code,
                "sub_tid": sub_tid,
                "is_split": is_split,
                "out_sid": out_sid,
                "sender_id": sender_id,
                "cancel_id": cancel_id,
                "feature": feature,
                "seller_ip": seller_ip
            },
            result_processor=lambda x: x["shipping"]
        )

    def taobao_logistics_online_cancel(
            self,
            tid
    ):
        """
        取消物流订单接口
        调此接口取消发货的订单，重新选择物流公司发货。前提是物流公司未揽收货物。对未发货和已经被物流公司揽收的物流订单，是不能取消的。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10688

        :param tid: 淘宝交易ID
        """
        return self._top_request(
            "taobao.logistics.online.cancel",
            {
                "tid": tid
            }
        )

    def taobao_logistics_online_confirm(
            self,
            tid,
            out_sid,
            sub_tid='',
            is_split='',
            seller_ip=''
    ):
        """
        确认发货通知接口
        <br><font color='red'>仅使用taobao.logistics.online.send 发货时，未输入运单号的情况下，需要使用该接口确认发货。发货接口主要有taobao.logistics.offline.send 和taobao.logistics.online.send <br><br/>确认发货的目的是让交易流程继承走下去，确认发货后交易状态会由【买家已付款】变为【卖家已发货】，然后买家才可以确认收货，货款打入卖家账号。货到付款的订单除外
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10689

        :param tid: 淘宝交易ID
        :param out_sid: 运单号.具体一个物流公司的真实运单号码。淘宝官方物流会校验，请谨慎传入；
        :param sub_tid: 拆单子订单列表，对应的数据是：子订单号的列表。可以不传，但是如果传了则必须符合传递的规则。子订单必须是操作的物流订单的子订单的真子集
        :param is_split: 表明是否是拆单，默认值0，1表示拆单
        :param seller_ip: 商家的IP地址
        """
        return self._top_request(
            "taobao.logistics.online.confirm",
            {
                "tid": tid,
                "out_sid": out_sid,
                "sub_tid": sub_tid,
                "is_split": is_split,
                "seller_ip": seller_ip
            },
            result_processor=lambda x: x["shipping"]
        )

    def taobao_logistics_offline_send(
            self,
            tid,
            out_sid,
            company_code,
            sub_tid='',
            is_split='',
            sender_id='',
            cancel_id='',
            feature='',
            seller_ip=''
    ):
        """
        自己联系物流（线下物流）发货
        用户调用该接口可实现自己联系发货（线下物流），使用该接口发货，交易订单状态会直接变成卖家已发货。不支持货到付款、在线下单类型的订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10690

        :param tid: 淘宝交易ID
        :param out_sid: 运单号.具体一个物流公司的真实运单号码。淘宝官方物流会校验，请谨慎传入；
        :param company_code: 物流公司代码.如'POST'就代表中国邮政,'ZJS'就代表宅急送.调用 taobao.logistics.companies.get 获取。
        :param sub_tid: 需要拆单发货的子订单集合，针对的是一笔交易下有多个子订单需要分开发货的场景；1次可传人多个子订单号，子订单间用逗号隔开；为空表示不做拆单发货。
        :param is_split: 表明是否是拆单，默认值0，1表示拆单
        :param sender_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。如果为空，取的卖家的默认取货地址
        :param cancel_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。 如果为空，取的卖家的默认退货地址
        :param feature: feature参数格式 范例: identCode=tid1:识别码1,识别码2|tid2:识别码3;machineCode=tid3:3C机器号A,3C机器号B identCode为识别码的KEY,machineCode为3C的KEY,多个key之间用”;”分隔 “tid1:识别码1,识别码2|tid2:识别码3”为identCode对应的value。 '|'不同商品间的分隔符。 例1商品和2商品，之间就用'|'分开。 TID就是商品代表的子订单号，对应taobao.trade.fullinfo.get 接口获得的oid字段。(通过OID可以唯一定位到当前商品上) ':'TID和具体传入参数间的分隔符。冒号前表示TID,之后代表该商品的参数属性。 ',' 属性间分隔符。（对应商品数量，当存在一个商品的数量超过1个时，用逗号分开）。 具体:当订单中A商品的数量为2个，其中手机串号分别为'12345','67890'。 参数格式：identCode=TIDA:12345,67890。 TIDA对应了A宝贝，冒号后用逗号分隔的'12345','67890'.说明本订单A宝贝的数量为2，值分别为'12345','67890'。 当存在'|'时，就说明订单中存在多个商品，商品间用'|'分隔了开来。|'之后的内容含义同上。retailStoreId=12345，发货门店ID或仓信息。retailStoreType=STORE: 发货门店类别，STORE表示门店，WAREHOUSE表示电商仓。对于全渠道订单回传的商家，retailStoreId和retailStoreType字段为必填字段。
        :param seller_ip: 商家的IP地址
        """
        return self._top_request(
            "taobao.logistics.offline.send",
            {
                "tid": tid,
                "out_sid": out_sid,
                "company_code": company_code,
                "sub_tid": sub_tid,
                "is_split": is_split,
                "sender_id": sender_id,
                "cancel_id": cancel_id,
                "feature": feature,
                "seller_ip": seller_ip
            },
            result_processor=lambda x: x["shipping"]
        )

    def taobao_logistics_dummy_send(
            self,
            tid,
            feature='',
            seller_ip=''
    ):
        """
        无需物流（虚拟）发货处理
        用户调用该接口可实现无需物流（虚拟）发货,使用该接口发货，交易订单状态会直接变成卖家已发货
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10691

        :param tid: 淘宝交易ID
        :param feature: feature参数格式<br>范例: identCode=tid1:识别码1,识别码2|tid2:识别码3;machineCode=tid3:3C机器号A,3C机器号B<br>identCode为识别码的KEY,machineCode为3C的KEY,多个key之间用”;”分隔<br>“tid1:识别码1,识别码2|tid2:识别码3”为identCode对应的value。'|'不同商品间的分隔符。<br>例1商品和2商品，之间就用'|'分开。<br>TID就是商品代表的子订单号，对应taobao.trade.fullinfo.get 接口获得的oid字段。(通过OID可以唯一定位到当前商品上)<br>':'TID和具体传入参数间的分隔符。冒号前表示TID,之后代表该商品的参数属性。<br>',' 属性间分隔符。（对应商品数量，当存在一个商品的数量超过1个时，用逗号分开）。<br>具体:当订单中A商品的数量为2个，其中手机串号分别为'12345','67890'。<br>参数格式：identCode=TIDA:12345,67890。TIDA对应了A宝贝，冒号后用逗号分隔的'12345','67890'.说明本订单A宝贝的数量为2，值分别为'12345','67890'。<br>当存在'|'时，就说明订单中存在多个商品，商品间用'|'分隔了开来。|'之后的内容含义同上。
        :param seller_ip: 商家的IP地址
        """
        return self._top_request(
            "taobao.logistics.dummy.send",
            {
                "tid": tid,
                "feature": feature,
                "seller_ip": seller_ip
            },
            result_processor=lambda x: x["shipping"]
        )

    def taobao_delivery_template_get(
            self,
            template_ids,
            fields,
            user_nick=''
    ):
        """
        获取用户指定运费模板信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10915

        :param template_ids: 需要查询的运费模板ID列表
        :param fields: 需返回的字段列表。 <br/> <br/><B><br/>可选值:示例中的值;各字段之间用','隔开<br/></B><br/><br/><br/> <br/><font color=blue><br/>template_id：查询模板ID <br/> <br/>template_name:查询模板名称 <br/> <br/>assumer：查询服务承担放 <br/> <br/>valuation:查询计价规则 <br/> <br/>supports:查询增值服务列表 <br/> <br/>created:查询模板创建时间 <br/> <br/>modified:查询修改时间<br/> <br/>consign_area_id:运费模板上设置的卖家发货地址最小级别ID<br/> <br/>address:卖家地址信息<br/></font><br/><br/><br/> <br/><font color=bule><br/>query_cod:查询货到付款运费信息<br/> <br/>query_post:查询平邮运费信息 <br/> <br/>query_express:查询快递公司运费信息 <br/> <br/>query_ems:查询EMS运费信息<br/> <br/>query_bzsd:查询普通保障速递运费信息<br/> <br/>query_wlb:查询物流宝保障速递运费信息<br/> <br/>query_furniture:查询家装物流运费信息<br/><font color=blue><br/><br/><br/><br/><font color=red>不能有空格</font>
        :param user_nick: 在没有登录的情况下根据淘宝用户昵称查询指定的模板
        """
        return self._top_request(
            "taobao.delivery.template.get",
            {
                "template_ids": template_ids,
                "fields": fields,
                "user_nick": user_nick
            }
        )

    def taobao_delivery_templates_get(
            self,
            fields
    ):
        """
        获取用户下所有模板
        根据用户ID获取用户下所有模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10916

        :param fields: 需返回的字段列表。 <br/> <br/><B><br/>可选值:示例中的值;各字段之间用','隔开<br/></B><br/><br/><br/> <br/><font color=blue><br/>template_id：查询模板ID <br/> <br/>template_name:查询模板名称 <br/> <br/>assumer：查询服务承担放 <br/> <br/>valuation:查询计价规则 <br/> <br/>supports:查询增值服务列表 <br/> <br/>created:查询模板创建时间 <br/> <br/>modified:查询修改时间<br/> <br/>consign_area_id:运费模板上设置的卖家发货地址最小级别ID<br/> <br/>address:卖家地址信息<br/></font><br/><br/><br/> <br/><font color=bule><br/>query_cod:查询货到付款运费信息<br/> <br/>query_post:查询平邮运费信息 <br/> <br/>query_express:查询快递公司运费信息 <br/> <br/>query_ems:查询EMS运费信息<br/> <br/>query_bzsd:查询普通保障速递运费信息<br/> <br/>query_wlb:查询物流宝保障速递运费信息<br/> <br/>query_furniture:查询家装物流运费信息<br/><font color=blue><br/><br/><br/><br/><font color=red>不能有空格</font>
        """
        return self._top_request(
            "taobao.delivery.templates.get",
            {
                "fields": fields
            }
        )

    def taobao_delivery_template_delete(
            self,
            template_id
    ):
        """
        删除运费模板
        根据用户指定的模板ID删除指定的模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10917

        :param template_id: 运费模板ID
        """
        return self._top_request(
            "taobao.delivery.template.delete",
            {
                "template_id": template_id
            },
            result_processor=lambda x: x["complete"]
        )

    def taobao_delivery_template_add(
            self,
            name,
            assumer,
            valuation,
            template_types,
            template_dests,
            template_start_standards,
            template_start_fees,
            template_add_standards,
            template_add_fees,
            consign_area_id=''
    ):
        """
        新增运费模板
        增加运费模板的外部接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10918

        :param name: 运费模板的名称，长度不能超过50个字节
        :param assumer: 可选值：0、1 ，说明如下<br>0:表示买家承担服务费;<br>1:表示卖家承担服务费
        :param valuation: 可选值：0、1、3，说明如下。<br>0:表示按宝贝件数计算运费 <br>1:表示按宝贝重量计算运费
                          <br>3:表示按宝贝体积计算运费
        :param template_types: 运费方式:平邮 (post),快递公司(express),EMS (ems),货到付款(cod),物流宝保障速递(wlb),家装物流(furniture)结构:value1;value2;value3;value4
                               如: post;express;ems;cod
                               <br/><font color=red>
                               注意:在添加多个运费方式时,字符串中使用 ';' 分号区分
                               。template_dests(指定地区)
                               template_start_standards(首费标准)、template_start_fees(首费)、template_add_standards(增费标准)、template_add_fees(增费)必须与template_types的分号数量相同. </font>
                               <br>
                               <font color=blue>
                               注意：<br/>
                               1、post,ems,express三种运费方式必须填写一个<br/>
                               2、只有订购了货到付款用户可以填cod;只有家装物流用户可以填写furniture
                               只有订购了保障速递的用户可以填写bzsd,只有物流宝用户可以填写wlb<br/>
                               3、如果是货到付款用户当没有填写cod运送方式时，运费模板会默认继承express的费用为cod的费用<br/>
                               4、如果是保障速递户当没有填写bzsd运送方式时，运费模板会默认继承express的费用为bzsd的费用<br/>
                               5、由于3和4的原因所以当是货到付款用户或保障速递用户时如果没填写对应的运送方式express是必须填写的
                               </font>
        :param template_dests: 邮费子项涉及的地区.结构: value1;value2;value3,value4
                               <br>如:1,110000;1,110000;1,310000;1,320000,330000。 aredId解释(1=全国,110000=北京,310000=上海,320000=江苏,330000=浙江)
                               如果template_types设置为post;ems;exmpress;cod则表示为平邮(post)指定默认地区(全国)和北京地区的运费;其他的类似以分号区分一一对应
                               <br/>可以用taobao.areas.get接口获取.或者参考：http://www.stats.gov.cn/tjbz/xzqhdm/t20080215_402462675.htm<br/>
                               <br/><font color=red>每个运费方式设置涉及的地区中必须包含全国地区（areaId=1）表示默认运费,可以只设置默认运费</font>
                               <br><font color=blue>注意:为多个地区指定指定不同（首费标准、首费、增费标准、增费一项不一样就算不同）的运费以逗号','区分，
                               template_start_standards(首费标准)、template_start_fees(首费)、
                               template_add_standards(增费标准)、
                               template_add_fees(增费)必须与template_types分号数量相同。如果为需要为多个地区指定相同运费则地区之间用“|”隔开即可。</font>
                               <font color=red>如果卖家没有传入发货地址则默认地址库的发货地址</font>
        :param template_start_standards: 首费标准：当valuation(记价方式)为0时输入1-9999范围内的整数<br/><font color=red>当valuation(记价方式)为1时输入0.1-9999.9范围内的小数只能包含以为小数（单位为千克）<br/><font color=blue>当valuation(记价方式)为3时输入0.1-999.9范围内的数值，数值只能包含一位小数（单位为 立方米）
                                         <br/>
                                         <font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param template_start_fees: 首费：输入0.00-999.99（最多包含两位小数）
                                    <br/><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param template_add_standards: 增费标准：当valuation(记价方式)为0时输入1-9999范围内的整数<br/><font color=red>当valuation(记价方式)为1时输入0.1-9999.9范围内的小数只能包含以为小数（单位为千克）<br/><font color=blue>当valuation(记价方式)为3时输入0.1-999.9范围内的数值，数值只能包含一位小数（单位为 立方米）
                                       <br/>
                                       <br><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param template_add_fees: 增费：输入0.00-999.99（最多包含两位小数）<br/><br/><br/><br/><font color=blue>增费必须小于等于首费，但是当首费为0时增费可以大于首费</font><br/><br/><br/><br/><br/><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param consign_area_id: 卖家发货地址区域ID<br/><br/><br/><font color=blue>可以不填，如果没有填写取卖家默认发货地址的区域ID，如果需要输入必须用taobao.areas.get接口获取.或者参考：http://www.stats.gov.cn/tjbz/xzqhdm/t20080215_402462675.htm <br/></font><br/><br/><br/><br/><font color=red>注意：填入该值时必须取您的发货地址最小区域级别ID，比如您的发货地址是：某省XX市xx区（县）时需要填入区(县)的ID，当然有些地方没有区或县可以直接填市级别的ID</font>
        """
        return self._top_request(
            "taobao.delivery.template.add",
            {
                "name": name,
                "assumer": assumer,
                "valuation": valuation,
                "template_types": template_types,
                "template_dests": template_dests,
                "template_start_standards": template_start_standards,
                "template_start_fees": template_start_fees,
                "template_add_standards": template_add_standards,
                "template_add_fees": template_add_fees,
                "consign_area_id": consign_area_id
            },
            result_processor=lambda x: x["delivery_template"]
        )

    def taobao_delivery_template_update(
            self,
            template_id,
            template_types,
            template_dests,
            template_start_standards,
            template_start_fees,
            template_add_standards,
            template_add_fees,
            name='',
            assumer=''
    ):
        """
        修改运费模板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10919

        :param template_id: 需要修改的模板对应的模板ID
        :param template_types: 运费方式:平邮 (post),快递公司(express),EMS (ems),货到付款(cod)结构:value1;value2;value3;value4 <br/>如: post;express;ems;cod <br/><br/><font color=red><br/>注意:在添加多个运费方式时,字符串中使用 ';' 分号区分。template_dests(指定地区) template_start_standards(首费标准)、template_start_fees(首费)、template_add_standards(增费标准)、template_add_fees(增费)必须与template_types的分号数量相同. <br/> </font><br/><br/><br/><font color=blue><br/>普通用户：post,ems,express三种运费方式必须填写一个，不能填写cod。<br/>货到付款用户：如果填写了cod运费方式，则post,ems,express三种运费方式也必须填写一个，如果没有填写cod则填写的运费方式中必须存在express</font>
        :param template_dests: 邮费子项涉及的地区.结构: value1;value2;value3,value4
                               <br>如:1,110000;1,110000;1,310000;1,320000,330000。 aredId解释(1=全国,110000=北京,310000=上海,320000=江苏,330000=浙江)
                               如果template_types设置为post;ems;exmpress;cod则表示为平邮(post)指定默认地区(全国)和北京地区的运费;其他的类似以分号区分一一对应
                               <br/>可以用taobao.areas.get接口获取.或者参考：http://www.stats.gov.cn/tjbz/xzqhdm/t20080215_402462675.htm
                               <br/><font color=red>每个运费方式设置的设涉及地区中必须包含全国地区（areaId=1）表示默认运费,可以只设置默认运费</font>
                               <br><font color=blue>注意:为多个地区指定指定不同（首费标准、首费、增费标准、增费一项不一样就算不同）的运费以逗号','区分，
                               template_start_standards(首费标准)、template_start_fees(首费)、
                               template_add_standards(增费标准)、
                               template_add_fees(增费)必须与template_types分号数量相同。如果为需要为多个地区指定相同运费则地区之间用“|”隔开即可。</font>
        :param template_start_standards: 首费标准：当valuation(记价方式)为0时输入1-9999范围内的整数<br><font color=blue>首费标准目前只能为1</br>
                                         <br><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param template_start_fees: 首费：输入0.01-999.99（最多包含两位小数）<br/><br/><font color=blue> 首费不能为0</font><br><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param template_add_standards: 增费标准：当valuation(记价方式)为0时输入1-9999范围内的整数<br><font color=blue>增费标准目前只能为1</font>
                                       <br><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param template_add_fees: 增费：输入0.00-999.99（最多包含两位小数）<br/><font color=blue>增费可以为0</font><br/><font color=red>输入的格式分号个数和template_types数量一致，逗号个数必须与template_dests以分号隔开之后一一对应的数量一致</font>
        :param name: 模板名称，长度不能大于50个字节
        :param assumer: 可选值：0,1 <br>  说明<br>0:表示买家承担服务费;<br>1:表示卖家承担服务费
        """
        return self._top_request(
            "taobao.delivery.template.update",
            {
                "template_id": template_id,
                "template_types": template_types,
                "template_dests": template_dests,
                "template_start_standards": template_start_standards,
                "template_start_fees": template_start_fees,
                "template_add_standards": template_add_standards,
                "template_add_fees": template_add_fees,
                "name": name,
                "assumer": assumer
            },
            result_processor=lambda x: x["complete"]
        )

    def taobao_logistics_order_create(
            self,
            goods_names,
            goods_quantities,
            item_values,
            seller_wangwang_id,
            logis_type='OFFLINE',
            logis_company_code='',
            mail_no='',
            trade_id='',
            is_consign='true',
            shipping='1'
    ):
        """
        创建物流订单
        用户调用该接口可以创建物流订单。目前仅支持手工订单的创建，创建完毕后默认自动使用“自己联系”的方式发货并且初始状态为”已发货“。也可以通过可选参数选择是否发货以及何种方式进行发货。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10997

        :param goods_names: 运送的货物名称列表，用|号隔开
        :param goods_quantities: 运送货物的数量列表，用|号隔开
        :param item_values: 运送货物的单价列表(注意：单位为分），用|号隔开
        :param seller_wangwang_id: 卖家旺旺号
        :param logis_type: 发货方式,默认为自己联系发货。可选值:ONLINE(在线下单)、OFFLINE(自己联系)。
        :param logis_company_code: 发货的物流公司代码，如申通=STO，圆通=YTO。is_consign=true时，此项必填。
        :param mail_no: 发货的物流公司运单号。在logis_type=OFFLINE且is_consign=true时，此项必填。
        :param trade_id: 订单的交易号码
        :param is_consign: 创建订单同时是否进行发货，默认发货。
        :param shipping: 运费承担方式。1为买家承担运费，2为卖家承担运费，其他值为错误参数。
        """
        return self._top_request(
            "taobao.logistics.order.create",
            {
                "goods_names": goods_names,
                "goods_quantities": goods_quantities,
                "item_values": item_values,
                "seller_wangwang_id": seller_wangwang_id,
                "logis_type": logis_type,
                "logis_company_code": logis_company_code,
                "mail_no": mail_no,
                "trade_id": trade_id,
                "is_consign": is_consign,
                "shipping": shipping
            },
            result_processor=lambda x: x["trade_id"]
        )

    def taobao_logistics_consign_resend(
            self,
            tid,
            out_sid,
            company_code,
            sub_tid='',
            is_split='0',
            feature='',
            seller_ip=''
    ):
        """
        修改物流公司和运单号
        支持卖家发货后修改运单号;支持在线下单和自己联系两种发货方式;使用条件：<br><br/>1、必须是已发货订单，自己联系发货的必须24小时内才可修改；在线下单的，必须下单后物流公司未揽收成功前才可修改；<br/>2、自己联系只能切换为自己联系的公司，在线下单也只能切换为在线下单的物流公司。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21492

        :param tid: 淘宝交易ID
        :param out_sid: 运单号.具体一个物流公司的真实运单号码。淘宝官方物流会校验，请谨慎传入；
        :param company_code: 物流公司代码.如'POST'代表中国邮政,'ZJS'代表宅急送。调用 taobao.logistics.companies.get 获取。<br><font color='red'>如果是货到付款订单，选择的物流公司必须支持货到付款发货方式</font>
        :param sub_tid: 拆单子订单列表，对应的数据是：子订单号列表。可以不传，但是如果传了则必须符合传递的规则。子订单必须是操作的物流订单的子订单的真子集！
        :param is_split: 表明是否是拆单，默认值0，1表示拆单
        :param feature: feature参数格式<br>范例: identCode=tid1:识别码1,识别码2|tid2:识别码3;machineCode=tid3:3C机器号A,3C机器号B<br>identCode为识别码的KEY,machineCode为3C的KEY,多个key之间用”;”分隔<br>“tid1:识别码1,识别码2|tid2:识别码3”为identCode对应的value。'|'不同商品间的分隔符。<br>例1商品和2商品，之间就用'|'分开。<br>TID就是商品代表的子订单号，对应taobao.trade.fullinfo.get 接口获得的oid字段。(通过OID可以唯一定位到当前商品上)<br>':'TID和具体传入参数间的分隔符。冒号前表示TID,之后代表该商品的参数属性。<br>',' 属性间分隔符。（对应商品数量，当存在一个商品的数量超过1个时，用逗号分开）。<br>具体:当订单中A商品的数量为2个，其中手机串号分别为'12345','67890'。<br>参数格式：identCode=TIDA:12345,67890。TIDA对应了A宝贝，冒号后用逗号分隔的'12345','67890'.说明本订单A宝贝的数量为2，值分别为'12345','67890'。<br>当存在'|'时，就说明订单中存在多个商品，商品间用'|'分隔了开来。|'之后的内容含义同上。
        :param seller_ip: 商家的IP地址
        """
        return self._top_request(
            "taobao.logistics.consign.resend",
            {
                "tid": tid,
                "out_sid": out_sid,
                "company_code": company_code,
                "sub_tid": sub_tid,
                "is_split": is_split,
                "feature": feature,
                "seller_ip": seller_ip
            },
            result_processor=lambda x: x["shipping"]
        )

    def taobao_logistics_consign_order_createandsend(
            self,
            user_id,
            order_source,
            order_type,
            logis_type,
            company_id,
            trade_id,
            s_name,
            s_area_id,
            s_address,
            s_zip_code,
            s_prov_name,
            s_city_name,
            r_name,
            r_area_id,
            r_address,
            r_zip_code,
            r_prov_name,
            r_city_name,
            item_json_string,
            mail_no='',
            shipping='',
            s_mobile_phone='',
            s_telephone='',
            s_dist_name='',
            r_mobile_phone='',
            r_telephone='',
            r_dist_name=''
    ):
        """
        创建订单并发货
        创建物流订单，并发货。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22172

        :param user_id: 用户ID
        :param order_source: 订单来源，值选择：30
        :param order_type: 订单类型，值固定选择：30
        :param logis_type: 物流订单物流类型，值固定选择：2
        :param company_id: 物流公司ID
        :param trade_id: 交易流水号，淘外订单号或者商家内部交易流水号
        :param s_name: 发件人名称
        :param s_area_id: 发件人区域ID
        :param s_address: 发件人街道地址
        :param s_zip_code: 发件人出编
        :param s_prov_name: 省
        :param s_city_name: 市
        :param r_name: 收件人名称
        :param r_area_id: 收件人区域ID
        :param r_address: 收件人街道地址
        :param r_zip_code: 收件人邮编
        :param r_prov_name: 省
        :param r_city_name: 市
        :param item_json_string: 物品的json数据。
        :param mail_no: 运单号
        :param shipping: 费用承担方式 1买家承担运费 2卖家承担运费
        :param s_mobile_phone: 手机号码
        :param s_telephone: 电话号码
        :param s_dist_name: 区
        :param r_mobile_phone: 手机号码
        :param r_telephone: 电话号码
        :param r_dist_name: 区
        """
        return self._top_request(
            "taobao.logistics.consign.order.createandsend",
            {
                "user_id": user_id,
                "order_source": order_source,
                "order_type": order_type,
                "logis_type": logis_type,
                "company_id": company_id,
                "trade_id": trade_id,
                "s_name": s_name,
                "s_area_id": s_area_id,
                "s_address": s_address,
                "s_zip_code": s_zip_code,
                "s_prov_name": s_prov_name,
                "s_city_name": s_city_name,
                "r_name": r_name,
                "r_area_id": r_area_id,
                "r_address": r_address,
                "r_zip_code": r_zip_code,
                "r_prov_name": r_prov_name,
                "r_city_name": r_city_name,
                "item_json_string": item_json_string,
                "mail_no": mail_no,
                "shipping": shipping,
                "s_mobile_phone": s_mobile_phone,
                "s_telephone": s_telephone,
                "s_dist_name": s_dist_name,
                "r_mobile_phone": r_mobile_phone,
                "r_telephone": r_telephone,
                "r_dist_name": r_dist_name
            }
        )

    def taobao_logistics_address_reachable(
            self,
            partner_ids,
            service_type,
            area_code='',
            address='',
            source_area_code=''
    ):
        """
        判定服务是否可达
        根据输入的目标地址，判断服务是否可达。
        现已支持筛单的快递公司共15家：中国邮政、EMS、国通、汇通、快捷、全峰、优速、圆通、宅急送、中通、顺丰、天天、韵达、德邦快递、申通
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22688

        :param partner_ids: 物流公司编码ID，可以从这个接口获取所有物流公司的标准编码taobao.logistics.companies.get，可以传入多个值，以英文逗号分隔，如“1000000952,1000000953”
        :param service_type: 服务对应的数字编码，如揽派范围对应88
        :param area_code: 标准区域编码(三级行政区编码或是四级行政区)，可以通过taobao.areas.get获取，如北京市朝阳区为110105
        :param address: 详细地址
        :param source_area_code: 发货地，标准区域编码(四级行政)，可以通过taobao.areas.get获取，如浙江省杭州市余杭区闲林街道为 330110011
        """
        return self._top_request(
            "taobao.logistics.address.reachable",
            {
                "partner_ids": partner_ids,
                "service_type": service_type,
                "area_code": area_code,
                "address": address,
                "source_area_code": source_area_code
            },
            result_processor=lambda x: x["reachable_result_list"]
        )

    def taobao_logistics_order_shengxian_confirm(
            self,
            out_sid,
            delivery_type,
            tid='',
            sender_id='',
            cancel_id='',
            seller_ip='',
            logis_id='',
            service_code=''
    ):
        """
        物流宝生鲜冷链的发货
        优鲜送，生鲜业务使用该接口发货，交易订单状态会直接变成卖家已发货。不支持货到付款、在线下单类型的订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23039

        :param out_sid: 运单号.具体一个物流公司的真实运单号码。支持多个运单号，多个运单号之间用英文分号（;）隔开，不能重复。淘宝官方物流会校验，请谨慎传入；
        :param delivery_type: 1：冷链。0：常温
        :param tid: 淘宝交易ID
        :param sender_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。<font color='red'>如果为空，取的卖家的默认取货地址</font>如果service_code不为空，默认使用service_code如果service_code为空，sender_id不为空，使用sender_id对应的地址发货如果service_code与sender_id都为空，使用默认地址发货
        :param cancel_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。<br><font color='red'>如果为空，取的卖家的默认退货地址</font><br>
        :param seller_ip: 商家的IP地址
        :param logis_id: 物流订单ID 。同淘宝交易订单互斥，淘宝交易号存在，，以淘宝交易号为准
        :param service_code: 仓库的服务代码标示，代码一个仓库的实体。(可以通过taobao.wlb.stores.baseinfo.get接口查询)如果service_code为空，默认通过sender_id来发货
        """
        return self._top_request(
            "taobao.logistics.order.shengxian.confirm",
            {
                "out_sid": out_sid,
                "delivery_type": delivery_type,
                "tid": tid,
                "sender_id": sender_id,
                "cancel_id": cancel_id,
                "seller_ip": seller_ip,
                "logis_id": logis_id,
                "service_code": service_code
            }
        )

    def taobao_wlb_stores_baseinfo_get(
            self,
            type='0'
    ):
        """
        商家按照仓的类型获取仓库接口
        通过USERID和仓库类型，获取商家自有仓库或菜鸟仓库或全部仓库
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23040

        :param type: 0.商家仓库.1.菜鸟仓库.2全部被划分的仓库
        """
        return self._top_request(
            "taobao.wlb.stores.baseinfo.get",
            {
                "type": type
            }
        )

    def taobao_logistics_address_reachablebatch_get(
            self,
            address_list=None
    ):
        """
        批量判定服务是否可达
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23566

        :param address_list: 筛单用户输入地址结构
        """
        return self._top_request(
            "taobao.logistics.address.reachablebatch.get",
            {
                "address_list": address_list
            },
            result_processor=lambda x: x["reachable_results"]
        )

    def taobao_logistics_consign_tc_confirm(
            self,
            provider_id,
            app_name,
            out_trade_no,
            seller_id,
            trade_id,
            extend_fields='',
            goods_list=None
    ):
        """
        通知交易确认发货接口
        下述业务场景可以使用此接口通知相关的交易订单发货：
        1、发货过程分为多段操作，在确定发货之前，不需要通知交易，当货确认已发出之后，才通知交易发货。
        2、发货过程涉及到多个订单，其中一个订单是跟实操的发货操作同步的，剩下的订单，需要在实操的订单发货之后，一并通知交易发货。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25147

        :param provider_id: 货主id
        :param app_name: ERP的名称
        :param out_trade_no: 已发货的外部订单号
        :param seller_id: 卖家id
        :param trade_id: 待发货的淘宝主交易号
        :param extend_fields: 扩展字段 K:V;
        :param goods_list: 发货的包裹
        """
        return self._top_request(
            "taobao.logistics.consign.tc.confirm",
            {
                "provider_id": provider_id,
                "app_name": app_name,
                "out_trade_no": out_trade_no,
                "seller_id": seller_id,
                "trade_id": trade_id,
                "extend_fields": extend_fields,
                "goods_list": goods_list
            }
        )

    def taobao_wms_order_warehouse_route_get(
            self,
            order_code
    ):
        """
        获取订单仓库路由信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25412

        :param order_code: 订单编号
        """
        return self._top_request(
            "taobao.wms.order.warehouse.route.get",
            {
                "order_code": order_code
            }
        )

    def taobao_wlb_order_jz_query(
            self,
            tid='',
            jz_receiver_to=None,
            sender_id='',
            ins_jz_receiver_t_o=None
    ):
        """
        家装业务查询物流公司api
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26007

        :param tid: 交易id
        :param jz_receiver_to: 家装收货人信息
        :param sender_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。如果为空，取的卖家的默认取货地址
        :param ins_jz_receiver_t_o: 家装安装服务收货人信息
        """
        return self._top_request(
            "taobao.wlb.order.jz.query",
            {
                "tid": tid,
                "jz_receiver_to": jz_receiver_to,
                "sender_id": sender_id,
                "ins_jz_receiver_t_o": ins_jz_receiver_t_o
            }
        )

    def taobao_wlb_order_jz_consign(
            self,
            tid,
            lg_tp_dto,
            sender_id='',
            jz_receiver_to=None,
            jz_top_args=None,
            ins_tp_dto=None,
            ins_receiver_to=None
    ):
        """
        家装发货接口
        家装类订单使用该接口发货
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26009

        :param tid: 交易号
        :param lg_tp_dto: 物流公司信息
        :param sender_id: 卖家联系人地址库ID，可以通过taobao.logistics.address.search接口查询到地址库ID。如果为空，取的卖家的默认取货地址
        :param jz_receiver_to: 家装收货人信息,如果为空,则取默认收货信息
        :param jz_top_args: 发货参数
        :param ins_tp_dto: 安装公司信息,需要安装时,才填写
        :param ins_receiver_to: 安装收货人信息,如果为空,则取默认收货人信息
        """
        return self._top_request(
            "taobao.wlb.order.jz.consign",
            {
                "tid": tid,
                "lg_tp_dto": lg_tp_dto,
                "sender_id": sender_id,
                "jz_receiver_to": jz_receiver_to,
                "jz_top_args": jz_top_args,
                "ins_tp_dto": ins_tp_dto,
                "ins_receiver_to": ins_receiver_to
            }
        )

    def cainiao_cboss_workplatform_workorder_task_notify(
            self,
            content
    ):
        """
        TOP-SPI工单任务下发接口
        TOP-SPI工单任务下发接口（菜鸟--->商家ISV）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27022

        :param content: content
        """
        return self._top_request(
            "cainiao.cboss.workplatform.workorder.task.notify",
            {
                "content": content
            },
            result_processor=lambda x: x["response"]
        )

    def cainiao_cboss_workplatform_workorder_process_notify(
            self,
            content
    ):
        """
        菜鸟工单系统的工单进度下发
        菜鸟工单系统的工单进度下发（SPI）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27025

        :param content: 服务入参
        """
        return self._top_request(
            "cainiao.cboss.workplatform.workorder.process.notify",
            {
                "content": content
            },
            result_processor=lambda x: x["response"]
        )

    def cainiao_cboss_workplatform_workorder_create(
            self,
            work_order_type='null',
            biz_type='null',
            memo='null',
            member_id='null',
            member_role='null',
            creator_id='null',
            creator_role='null',
            biz_entity_value='null',
            shop_user_id='null',
            trade_id='null',
            source='null',
            source_sign='null',
            mail_no='null',
            features='null',
            attach_path_list='null'
    ):
        """
        菜鸟工单创建接口
        菜鸟工单创建接口，目前调用者ISV
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27052

        :param work_order_type: 工单类型
        :param biz_type: 业务类型
        :param memo: 工单创建备注
        :param member_id: 货主商家用户id
        :param member_role: 货主用户角色
        :param creator_id: 创建者淘宝id（区分子账号）
        :param creator_role: 创建者角色
        :param biz_entity_value: 外部业务系统主键
        :param shop_user_id: 店铺用户id
        :param trade_id: 交易订单id
        :param source: 工单来源
        :param source_sign: 来源签名，用于唯一区分不同的来源方
        :param mail_no: 运单号
        :param features: 扩展字段
        :param attach_path_list: 凭证地址列表
        """
        return self._top_request(
            "cainiao.cboss.workplatform.workorder.create",
            {
                "work_order_type": work_order_type,
                "biz_type": biz_type,
                "memo": memo,
                "member_id": member_id,
                "member_role": member_role,
                "creator_id": creator_id,
                "creator_role": creator_role,
                "biz_entity_value": biz_entity_value,
                "shop_user_id": shop_user_id,
                "trade_id": trade_id,
                "source": source,
                "source_sign": source_sign,
                "mail_no": mail_no,
                "features": features,
                "attach_path_list": attach_path_list
            }
        )

    def cainiao_cboss_workplatform_operation_reply(
            self,
            work_order_id='null',
            task_id='null',
            action_time='2016-05-23 19:00:00',
            action_type='0',
            dealer_user_id='null',
            dealer_contact='null',
            memo='null',
            attach_path='null',
            features='null'
    ):
        """
        菜鸟工单操作回传
        菜鸟工单进度下发接口，目前调用者ISV
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27053

        :param work_order_id: 工单id
        :param task_id: 工单任务id
        :param action_time: 任务操作时间
        :param action_type: 任务操作类型
        :param dealer_user_id: 操作者userId
        :param dealer_contact: 操作者联系方式
        :param memo: 商家工单操作回传备注
        :param attach_path: 凭证照片地址拼接
        :param features: 扩展字段
        """
        return self._top_request(
            "cainiao.cboss.workplatform.operation.reply",
            {
                "work_order_id": work_order_id,
                "task_id": task_id,
                "action_time": action_time,
                "action_type": action_type,
                "dealer_user_id": dealer_user_id,
                "dealer_contact": dealer_contact,
                "memo": memo,
                "attach_path": attach_path,
                "features": features
            }
        )

    def cainiao_cboss_workplatform_biztype_queryall(
            self,
            trade_id,
            level=''
    ):
        """
        菜鸟工单平台根据交易订单查询某条业务线上的所有业务类型
        菜鸟工单平台根据交易订单查询某条业务线上的所有业务类型。 目前调用者ISV
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27054

        :param trade_id: tradeId
        :param level: level
        """
        return self._top_request(
            "cainiao.cboss.workplatform.biztype.queryall",
            {
                "trade_id": trade_id,
                "level": level
            }
        )

    def cainiao_cboss_workplatform_biztype_querybyid(
            self,
            biz_type_id='null'
    ):
        """
        菜鸟工单平台根据业务类型id查询业务类型详细信息
        菜鸟工单平台根据业务类型id查询业务类型详细信息。 目前调用者ISV
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27055

        :param biz_type_id: 业务类型id
        """
        return self._top_request(
            "cainiao.cboss.workplatform.biztype.querybyid",
            {
                "biz_type_id": biz_type_id
            }
        )

    def cainiao_cboss_workplatform_logistics_iscainiaoorder(
            self,
            trade_id='null'
    ):
        """
        根据交易单号判断是否为菜鸟发货订单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27056

        :param trade_id: 交易单号
        """
        return self._top_request(
            "cainiao.cboss.workplatform.logistics.iscainiaoorder",
            {
                "trade_id": trade_id
            }
        )

    def cainiao_member_courier_cpresign(
            self,
            account_id
    ):
        """
        cp清理离职用户信息
        CP清理内部离职的用户信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28075

        :param account_id: 菜鸟用户id
        """
        return self._top_request(
            "cainiao.member.courier.cpresign",
            {
                "account_id": account_id
            }
        )

    def taobao_nextone_logistics_sign_update(
            self,
            refund_id,
            sign_status
    ):
        """
        AG物流签收状态写接口
        商家上传退货的签收状态给AG
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28151

        :param refund_id: 退款编号
        :param sign_status: 货物签收状态
        """
        return self._top_request(
            "taobao.nextone.logistics.sign.update",
            {
                "refund_id": refund_id,
                "sign_status": sign_status
            }
        )

    def taobao_nextone_logistics_warehouse_update(
            self,
            refund_id,
            warehouse_status
    ):
        """
        AG退货入仓状态写接口
        商家上传退货入仓状态给ag
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28206

        :param refund_id: 退款编号
        :param warehouse_status: 退货入仓状态 1.已入仓
        """
        return self._top_request(
            "taobao.nextone.logistics.warehouse.update",
            {
                "refund_id": refund_id,
                "warehouse_status": warehouse_status
            }
        )

    def alibaba_alink_message_history_action(
            self,
            action,
            index=''
    ):
        """
        操作历史消息
        阿里智能操作历史消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28329

        :param action: 删除：delete,已读：read
        :param index: 消息id
        """
        return self._top_request(
            "alibaba.alink.message.history.action",
            {
                "action": action,
                "index": index
            }
        )

    def taobao_wlb_import_threepl_resource_get(
            self,
            type,
            from_id,
            to_address
    ):
        """
        3PL直邮获取资源列表
        获取3pl直邮的发货可用资源
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28876

        :param type: ONLINE或者OFFLINE
        :param from_id: 发货地区域id
        :param to_address: 收件人地址
        """
        return self._top_request(
            "taobao.wlb.import.threepl.resource.get",
            {
                "type": type,
                "from_id": from_id,
                "to_address": to_address
            }
        )

    def taobao_wlb_import_threepl_offline_consign(
            self,
            trade_id='',
            res_id='',
            res_code='',
            waybill_no='',
            from_id=''
    ):
        """
        3PL直邮线下发货
        菜鸟认证直邮线下发货
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28877

        :param trade_id: 交易单号
        :param res_id: 资源id
        :param res_code: 资源code
        :param waybill_no: 运单号
        :param from_id: 发件人地址库id
        """
        return self._top_request(
            "taobao.wlb.import.threepl.offline.consign",
            {
                "trade_id": trade_id,
                "res_id": res_id,
                "res_code": res_code,
                "waybill_no": waybill_no,
                "from_id": from_id
            }
        )

    def cainiao_logistics_package_overview_get(
            self,
            from_date,
            to_date
    ):
        """
        获取物流管家的包裹基础数据
        获取物流管家的包裹基础数据，包含：发货包裹数、揽收包裹数、派送包裹数、已发货未揽收包裹数、已揽收未中转包裹数、已中转未派件包裹数、已派件未签收包裹数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34711

        :param from_date: 开始日期，最小为T-30。返回的结果包含开始日期当日的数据。
        :param to_date: 结束日期，最大为T日。返回的结果包含结束日期当日的数据。
        """
        return self._top_request(
            "cainiao.logistics.package.overview.get",
            {
                "from_date": from_date,
                "to_date": to_date
            },
            result_processor=lambda x: x["data"]
        )

    def cainiao_logistics_abnormal_get(
            self,
            from_date,
            to_date
    ):
        """
        获取物流管家的异常包裹数据
        获取物流管家的异常包裹数据，包括异常包裹总数、揽收超时包裹、中转超时包裹、派签超时包裹、虚假签收包裹、快递反馈异常包裹。时间可在T-15至T日范围内自定义。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34712

        :param from_date: 开始日期，最小为T-15日。返回结果包含开始日期当天的数据。
        :param to_date: 结束日期，最大为T日。返回结果中的揽收超时包裹数、中转超时包裹数、派签超时包裹数、包含结束日期当天的数据，虚假签收包裹数和快递反馈异常包裹数不包含结束日期当天的数据。
        """
        return self._top_request(
            "cainiao.logistics.abnormal.get",
            {
                "from_date": from_date,
                "to_date": to_date
            },
            result_processor=lambda x: x["data"]
        )

    def cainiao_logistics_performance_get(
            self,
            date_num
    ):
        """
        获取物流管家的物流绩效数据
        获取物流管家的物流绩效数据，包含平均支付-发货时长、平均发货-揽收时长、平均发货-签收时长、平均揽收-签收时长、平均支付-签收时长。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34713

        :param date_num: date_num只能取7、15或者30。比如：7表示获取近7天的物流绩效数据，不包含今天的数据。
        """
        return self._top_request(
            "cainiao.logistics.performance.get",
            {
                "date_num": date_num
            },
            result_processor=lambda x: x["performance"]
        )

    def taobao_logistics_express_modify_appoint(
            self,
            express_modify_appoint_top_request
    ):
        """
        快递改约api
        商家通过此api操作修改物流单，交易单的收货人地址、收货人联系方式、预约配送日期
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35777

        :param express_modify_appoint_top_request: 改约请求对象
        """
        return self._top_request(
            "taobao.logistics.express.modify.appoint",
            {
                "express_modify_appoint_top_request": express_modify_appoint_top_request
            }
        )

    def taobao_rdc_aligenius_warehouse_reverse_update(
            self,
            param0
    ):
        """
        销退单状态回传
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37381

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.warehouse.reverse.update",
            {
                "param0": param0
            }
        )

    def taobao_rdc_aligenius_distribution_logistics_cancel(
            self,
            param0=None
    ):
        """
        截配状态回传接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37391

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.distribution.logistics.cancel",
            {
                "param0": param0
            }
        )

    def cainiao_mdec_nooneshelf_order_save(
            self,
            nos_order
    ):
        """
        无人货架-订单同步
        无人货架-实时订单信息同步
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37830

        :param nos_order: 订单对象
        """
        return self._top_request(
            "cainiao.mdec.nooneshelf.order.save",
            {
                "nos_order": nos_order
            }
        )

    def cainiao_mdec_nooneshelf_baseinfo_save(
            self,
            nos_baseinfo
    ):
        """
        无人货架-货架基础信息同步
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37831

        :param nos_baseinfo: 货架基础信息
        """
        return self._top_request(
            "cainiao.mdec.nooneshelf.baseinfo.save",
            {
                "nos_baseinfo": nos_baseinfo
            }
        )

    def cainiao_mdec_nooneshelf_inventory_save(
            self,
            nos_inventory
    ):
        """
        无人货架-货架库存信息同步
        无人货架-库存信息同步
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37832

        :param nos_inventory: 货架库存对象
        """
        return self._top_request(
            "cainiao.mdec.nooneshelf.inventory.save",
            {
                "nos_inventory": nos_inventory
            }
        )

    def taobao_rdc_aligenius_warehouse_resend_logistics_msg_post(
            self,
            param0
    ):
        """
        补发单物流信息回传
        补发单erp物流信息回传平台
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38741

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.warehouse.resend.logistics.msg.post",
            {
                "param0": param0
            }
        )

    def taobao_rdc_aligenius_warehouse_resend_update(
            self,
            param0
    ):
        """
        补发单状态回传
        补发单状态回传接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38742

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.warehouse.resend.update",
            {
                "param0": param0
            }
        )

    def alibaba_ele_fengniao_trade_update(
            self,
            param=None
    ):
        """
        更新蜂鸟扣费状态
        汇金扣费成功后，回调该接口更新扣费状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39569

        :param param: param 参数
        """
        return self._top_request(
            "alibaba.ele.fengniao.trade.update",
            {
                "param": param
            }
        )

    def alibaba_ele_fengniao_cancel_merchant(
            self,
            param
    ):
        """
        商户取消
        商户取消配送
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39570

        :param param: 参数param
        """
        return self._top_request(
            "alibaba.ele.fengniao.cancel.merchant",
            {
                "param": param
            },
            result_processor=lambda x: x["message"]
        )

    def alibaba_ele_fengniao_order_push(
            self,
            param
    ):
        """
        推送订单
        推送淘宝订单至蜂鸟开放平台配送
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39571

        :param param: 参数param
        """
        return self._top_request(
            "alibaba.ele.fengniao.order.push",
            {
                "param": param
            },
            result_processor=lambda x: x["message"]
        )

    def alibaba_ele_fengniao_user_time_query(
            self,
            param
    ):
        """
        蜂鸟询用户T
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39573

        :param param: 询T入参
        """
        return self._top_request(
            "alibaba.ele.fengniao.user.time.query",
            {
                "param": param
            }
        )

    def alibaba_ele_fengniao_order_query(
            self,
            param=None
    ):
        """
        查询订单基本信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39586

        :param param: 参数
        """
        return self._top_request(
            "alibaba.ele.fengniao.order.query",
            {
                "param": param
            }
        )

    def alibaba_ele_fengniao_shippingorder_event(
            self,
            app_id='',
            partner_order_code=''
    ):
        """
        查询运单事件信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39587

        :param app_id: appid
        :param partner_order_code: 外部订单号
        """
        return self._top_request(
            "alibaba.ele.fengniao.shippingorder.event",
            {
                "app_id": app_id,
                "partner_order_code": partner_order_code
            }
        )

    def alibaba_ele_fengniao_carrierdriver_location(
            self,
            app_id='',
            partner_order_code=''
    ):
        """
        查询骑手当前位置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=39599

        :param app_id: appid
        :param partner_order_code: 外部订单号
        """
        return self._top_request(
            "alibaba.ele.fengniao.carrierdriver.location",
            {
                "app_id": app_id,
                "partner_order_code": partner_order_code
            }
        )

    def alibaba_ele_fengniao_service_package_query(
            self,
            param
    ):
        """
        预采购服务包查询接口
        查询门店所在经纬度可用服务包的接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40494

        :param param: 入参
        """
        return self._top_request(
            "alibaba.ele.fengniao.service.package.query",
            {
                "param": param
            },
            result_processor=lambda x: x["service_packages"]
        )

    def alibaba_ele_fengniao_chainstore_ranges(
            self,
            merchant_code='',
            app_id='',
            chainstore_code=''
    ):
        """
        蜂鸟查询门店配送范围接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40495

        :param merchant_code: 商户code
        :param app_id: appId
        :param chainstore_code: 门店code
        """
        return self._top_request(
            "alibaba.ele.fengniao.chainstore.ranges",
            {
                "merchant_code": merchant_code,
                "app_id": app_id,
                "chainstore_code": chainstore_code
            },
            result_processor=lambda x: x["range_list"]
        )

    def alibaba_ele_fengniao_chainstore_contract_cancel(
            self,
            param=None
    ):
        """
        门店解约接口
        调用成功后，门店和蜂鸟解除物流合同，不能再使用此门店推单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41491

        :param param: 系统自动生成
        """
        return self._top_request(
            "alibaba.ele.fengniao.chainstore.contract.cancel",
            {
                "param": param
            },
            result_processor=lambda x: x["message"]
        )

    def alibaba_ele_fengniao_chainstore_contract_change(
            self,
            param
    ):
        """
        门店改签合同接口
        通过调用接口，门店改签物流服务包
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41492

        :param param: 系统自动生成
        """
        return self._top_request(
            "alibaba.ele.fengniao.chainstore.contract.change",
            {
                "param": param
            },
            result_processor=lambda x: x["message"]
        )

    def alibaba_ele_fengniao_chainstore_update(
            self,
            param=None
    ):
        """
        修改门店信息接口
        修改门店的经纬度，文本地址，电话，门店名
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41493

        :param param: 入参
        """
        return self._top_request(
            "alibaba.ele.fengniao.chainstore.update",
            {
                "param": param
            },
            result_processor=lambda x: x["message"]
        )

    def alibaba_ele_fengniao_merchant_contract_cancel(
            self,
            param=None
    ):
        """
        蜂鸟商户解约接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        通过调用此接口，商家及商家下的所有门店解除蜂鸟物流服务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41494

        :param param: 系统自动生成
        """
        return self._top_request(
            "alibaba.ele.fengniao.merchant.contract.cancel",
            {
                "param": param
            },
            result_processor=lambda x: x["message"]
        )

    def taobao_rdc_aligenius_warehouse_reverse_event_update(
            self,
            param0
    ):
        """
        销退单事件回传接口
        用于erp回传销退单相关信息到平台
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42139

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.warehouse.reverse.event.update",
            {
                "param0": param0
            }
        )

    def taobao_rdc_aligenius_warehouse_reverse_uploading(
            self,
            param0=None
    ):
        """
        销退单上传
        主要用于商家上传仓库销退单信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42161

        :param param0: 参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.warehouse.reverse.uploading",
            {
                "param0": param0
            }
        )


class TbDianPu(DingTalkBaseAPI):
    """
    店铺API
    """

    def taobao_shopcats_list_get(
            self,
            fields='cid,parent_cid,name,is_parent'
    ):
        """
        获取前台展示的店铺类目
        获取淘宝面向买家的浏览导航类目（跟后台卖家商品管理的类目有差异）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=64

        :param fields: 需要返回的字段列表，见ShopCat，默认返回：cid,parent_cid,name,is_parent
        """
        return self._top_request(
            "taobao.shopcats.list.get",
            {
                "fields": fields
            },
            result_processor=lambda x: x["shop_cats"]
        )

    def taobao_sellercats_list_get(
            self
    ):
        """
        获取前台展示的店铺内卖家自定义商品类目
        此API获取当前卖家店铺在淘宝前端被展示的浏览导航类目（面向买家）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=65

        """
        return self._top_request(
            "taobao.sellercats.list.get",
            result_processor=lambda x: x["seller_cats"]
        )

    def taobao_sellercats_list_add(
            self,
            name,
            pict_url='',
            parent_cid='',
            sort_order=''
    ):
        """
        添加卖家自定义类目
        此API添加卖家店铺内自定义类目 <br/>父类目parent_cid值等于0：表示此类目为店铺下的一级类目，值不等于0：表示此类目有父类目 <br/>注：因为缓存的关系,添加的新类目需8个小时后才可以在淘宝页面上正常显示，但是不影响在该类目下商品发布
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=66

        :param name: 卖家自定义类目名称。不超过20个字符
        :param pict_url: 链接图片URL地址。(绝对地址，格式：http://host/image_path)
        :param parent_cid: 父类目编号，如果类目为店铺下的一级类目：值等于0，如果类目为子类目，调用获取taobao.sellercats.list.get父类目编号
        :param sort_order: 该类目在页面上的排序位置,取值范围:大于零的整数
        """
        return self._top_request(
            "taobao.sellercats.list.add",
            {
                "name": name,
                "pict_url": pict_url,
                "parent_cid": parent_cid,
                "sort_order": sort_order
            },
            result_processor=lambda x: x["seller_cat"]
        )

    def taobao_sellercats_list_update(
            self,
            cid,
            name='',
            pict_url='',
            sort_order=''
    ):
        """
        更新卖家自定义类目
        此API更新卖家店铺内自定义类目 <br/>注：因为缓存的关系，添加的新类目需8个小时后才可以在淘宝页面上正常显示，但是不影响在该类目下商品发布
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=67

        :param cid: 卖家自定义类目编号
        :param name: 卖家自定义类目名称。不超过20个字符
        :param pict_url: 链接图片URL地址
        :param sort_order: 该类目在页面上的排序位置,取值范围:大于零的整数
        """
        return self._top_request(
            "taobao.sellercats.list.update",
            {
                "cid": cid,
                "name": name,
                "pict_url": pict_url,
                "sort_order": sort_order
            },
            result_processor=lambda x: x["seller_cat"]
        )

    def taobao_shop_get(
            self,
            fields,
            nick
    ):
        """
        获取卖家店铺的基本信息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=68

        :param fields: 需返回的字段列表。可选值：Shop 结构中的所有字段；多个字段之间用逗号(,)分隔
        :param nick: 卖家昵称
        """
        return self._top_request(
            "taobao.shop.get",
            {
                "fields": fields,
                "nick": nick
            },
            result_processor=lambda x: x["shop"]
        )

    def taobao_shop_update(
            self,
            title='',
            bulletin='',
            desc=''
    ):
        """
        更新店铺基本信息
        目前只支持标题、公告和描述的更新
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=69

        :param title: 店铺标题。不超过30个字符；过滤敏感词，如淘咖啡、阿里巴巴等。title, bulletin和desc至少必须传一个
        :param bulletin: 店铺公告。不超过1024个字符
        :param desc: 店铺描述。10～2000个字符以内
        """
        return self._top_request(
            "taobao.shop.update",
            {
                "title": title,
                "bulletin": bulletin,
                "desc": desc
            },
            result_processor=lambda x: x["shop"]
        )

    def taobao_shop_getbytitle(
            self,
            title,
            fields='sid,cid,title,nick,desc,bulletin,pic_path,created,modified,shop_score'
    ):
        """
        根据店铺名称获取店铺信息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24852

        :param title: 店铺名称，必须严格匹配（不支持模糊查询）
        :param fields: 代表需要获取的店铺信息：sid,cid,title,nick,desc,bulletin,pic_path,created,modified,shop_score
        """
        return self._top_request(
            "taobao.shop.getbytitle",
            {
                "title": title,
                "fields": fields
            }
        )

    def alibaba_interact_sensor_clipbroad(
            self,
            un_named=''
    ):
        """
        Weex页面设置或读取剪切板
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30887

        :param un_named: 客户端鉴权使用，实际不会发送或接收数据
        """
        return self._top_request(
            "alibaba.interact.sensor.clipbroad",
            {
                "un_named": un_named
            }
        )

    def alibaba_interact_windvane_call(
            self,
            un_named=''
    ):
        """
        Weex页面调用windvane
        客户端鉴权使用，实际不会发送或接收数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30888

        :param un_named: 客户端鉴权使用，实际不会发送或接收数据
        """
        return self._top_request(
            "alibaba.interact.windvane.call",
            {
                "un_named": un_named
            },
            result_processor=lambda x: x["unnamed"]
        )

    def alibaba_data_item_get(
            self,
            un_named=''
    ):
        """
        获取商品
        获取商品信息，作为客户端Weex鉴权的虚拟api
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30897

        :param un_named: 获取商品信息，作为客户端Weex鉴权的虚拟api
        """
        return self._top_request(
            "alibaba.data.item.get",
            {
                "un_named": un_named
            },
            result_processor=lambda x: x["unnamed"]
        )

    def alibaba_data_coupon_get(
            self,
            un_named=''
    ):
        """
        获取优惠券信息
        获取优惠券信息，仅作客户端鉴权虚拟api使用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30898

        :param un_named: 客户端鉴权虚拟api使用
        """
        return self._top_request(
            "alibaba.data.coupon.get",
            {
                "un_named": un_named
            },
            result_processor=lambda x: x["unnamed"]
        )

    def alibaba_data_recommond_get(
            self,
            un_named=''
    ):
        """
        获取推荐信息
        获取优惠券信息，仅作客户端鉴权虚拟api使用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30899

        :param un_named: 客户端鉴权虚拟api使用
        """
        return self._top_request(
            "alibaba.data.recommond.get",
            {
                "un_named": un_named
            },
            result_processor=lambda x: x["unnamed"]
        )

    def alibaba_taobao_shop_cat_neo_get(
            self,
            un_named=''
    ):
        """
        店铺mtop接口鉴权虚拟api
        获取优惠券信息，仅作客户端鉴权虚拟api使用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30900

        :param un_named: 客户端鉴权虚拟api
        """
        return self._top_request(
            "alibaba.taobao.shop.cat.neo.get",
            {
                "un_named": un_named
            },
            result_processor=lambda x: x["unnamed"]
        )

    def taobao_store_followurl_get(
            self,
            callback_url='',
            user_nick='',
            user_id=''
    ):
        """
        获取店铺关注URL
        获取关注店铺的URL
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40699

        :param callback_url: 关注完成后的回调地址,需要是EWS地址。如果不设置，会跳转到店铺首页
        :param user_nick: 商家nick
        :param user_id: 商家ID
        """
        return self._top_request(
            "taobao.store.followurl.get",
            {
                "callback_url": callback_url,
                "user_nick": user_nick,
                "user_id": user_id
            },
            result_processor=lambda x: x["url"]
        )

    def taobao_shop_seller_get(
            self,
            fields
    ):
        """
        卖家店铺基础信息查询
        获取卖家店铺的基本信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42908

        :param fields: 需返回的字段列表。可选值：Shop 结构中的所有字段；多个字段之间用逗号(,)分隔
        """
        return self._top_request(
            "taobao.shop.seller.get",
            {
                "fields": fields
            },
            result_processor=lambda x: x["shop"]
        )


class TbFenXiao(DingTalkBaseAPI):
    """
    分销API
    """

    def taobao_fenxiao_orders_get(
            self,
            status='',
            start_created='',
            end_created='',
            time_type='',
            page_no='',
            page_size='',
            purchase_order_id='',
            fields='',
            tc_order_id=''
    ):
        """
        查询采购单信息
        分销商或供应商均可用此接口查询采购单信息（代销）； (发货请调用物流API中的发货接口taobao.logistics.offline.send 进行发货，需要注意的是这里是供应商发货，因此调发货接口时需要传人供应商账号对应的sessionkey，tid 需传入供销平台的采购单（即fenxiao_id  分销流水号）)。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=180

        :param status: 交易状态，不传默认查询所有采购单根据身份选择自身状态可选值: *供应商： WAIT_SELLER_SEND_GOODS(等待发货) WAIT_SELLER_CONFIRM_PAY(待确认收款) WAIT_BUYER_PAY(等待付款) WAIT_BUYER_CONFIRM_GOODS(已发货) TRADE_REFUNDING(退款中) TRADE_FINISHED(采购成功) TRADE_CLOSED(已关闭) *分销商： WAIT_BUYER_PAY(等待付款) WAIT_BUYER_CONFIRM_GOODS(待收货确认) TRADE_FOR_PAY(已付款) TRADE_REFUNDING(退款中) TRADE_FINISHED(采购成功) TRADE_CLOSED(已关闭)
        :param start_created: 起始时间 格式 yyyy-MM-dd HH:mm:ss.支持到秒的查询。若不传时分秒，默认为0时0分0秒。若purchase_order_id没传，则此参数必传。
        :param end_created: 结束时间 格式 yyyy-MM-dd HH:mm:ss.支持到秒的查询。若不传时分秒，默认为0时0分0秒。若purchase_order_id没传，则此参数必传。
        :param time_type: 可选值：trade_time_type(采购单按照成交时间范围查询),update_time_type(采购单按照更新时间范围查询)
        :param page_no: 页码。（大于0的整数。默认为1）
        :param page_size: 每页条数。（每页条数不超过50条）
        :param purchase_order_id: 采购单编号或分销流水号，若其它参数没传，则此参数必传。
        :param fields: 多个字段用','分隔。<br/><br/>fields<br/>如果为空：返回所有采购单对象(purchase_orders)字段。<br/>如果不为空：返回指定采购单对象(purchase_orders)字段。<br/><br/>例1：<br/>sub_purchase_orders.tc_order_id 表示只返回tc_order_id <br/>例2：<br/>sub_purchase_orders表示只返回子采购单列表
        :param tc_order_id: 采购单下游买家订单id
        """
        return self._top_request(
            "taobao.fenxiao.orders.get",
            {
                "status": status,
                "start_created": start_created,
                "end_created": end_created,
                "time_type": time_type,
                "page_no": page_no,
                "page_size": page_size,
                "purchase_order_id": purchase_order_id,
                "fields": fields,
                "tc_order_id": tc_order_id
            }
        )

    def taobao_fenxiao_product_add(
            self,
            name,
            productcat_id,
            standard_price,
            retail_price_low,
            retail_price_high,
            category_id,
            quantity,
            desc,
            prov,
            city,
            postage_type='',
            have_invoice='',
            have_quarantee='',
            trade_type='',
            cost_price='',
            outer_id='',
            postage_id='',
            postage_ordinary='',
            postage_fast='',
            postage_ems='',
            discount_id='',
            is_authz='yes',
            pic_path='',
            image='',
            properties='',
            property_alias='',
            input_properties='',
            sku_standard_prices='',
            sku_cost_prices='',
            sku_outer_ids='',
            sku_quantitys='',
            sku_properties='',
            dealer_cost_price='',
            sku_dealer_cost_prices='',
            item_id='',
            standard_retail_price='',
            spu_id=''
    ):
        """
        添加产品
        添加分销平台产品数据。业务逻辑与分销系统前台页面一致。<br/><br/>    * 产品图片默认为空<br/>    * 产品发布后默认为下架状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=325

        :param name: 产品名称，长度不超过60个字节。
        :param productcat_id: 产品线ID
        :param standard_price: 采购基准价格，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param retail_price_low: 最低零售价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param retail_price_high: 最高零售价，单位：元。例：“10.56”。必须在0.01元到10000000元之间，最高零售价必须大于最低零售价。
        :param category_id: 所属类目id，参考Taobao.itemcats.get，不支持成人等类目，输入成人类目id保存提示类目属性错误。
        :param quantity: 产品库存必须是1到999999。
        :param desc: 产品描述，长度为5到25000字符。
        :param prov: 所在地：省，例：“浙江”
        :param city: 所在地：市，例：“杭州”
        :param postage_type: 运费类型，可选值：seller（供应商承担运费）、buyer（分销商承担运费）,默认seller。
        :param have_invoice: 是否有发票，可选值：false（否）、true（是），默认false。
        :param have_quarantee: 是否有保修，可选值：false（否）、true（是），默认false。
        :param trade_type: 分销方式：AGENT（只做代销，默认值）、DEALER（只做经销）、ALL（代销和经销都做）
        :param cost_price: 代销采购价格，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param outer_id: 商家编码，长度不能超过60个字节。
        :param postage_id: 运费模板ID，参考taobao.postages.get。
        :param postage_ordinary: 平邮费用，单位：元。例：“10.56”。 大小为0.01元到999999元之间。
        :param postage_fast: 快递费用，单位：元。例：“10.56”。 大小为0.01元到999999元之间。
        :param postage_ems: ems费用，单位：元。例：“10.56”。 大小为0.00元到999999元之间。
        :param discount_id: 折扣ID
        :param is_authz: 添加产品时，添加入参isAuthz:yes|no <br/>yes:需要授权 <br/>no:不需要授权 <br/>默认是需要授权
        :param pic_path: 产品主图图片空间相对路径或绝对路径
        :param image: 产品主图，大小不超过500k，格式为gif,jpg,jpeg,png,bmp等图片
        :param properties: 产品属性，格式为pid:vid;pid:vid
        :param property_alias: 属性别名，格式为：pid:vid:alias;pid:vid:alias（alias为别名）
        :param input_properties: 自定义属性。格式为pid:value;pid:value
        :param sku_standard_prices: sku的采购基准价。如果多个，用逗号分隔，并与其他sku信息保持相同顺序
        :param sku_cost_prices: sku的采购价。如果多个，用逗号分隔，并与其他sku信息保持相同顺序
        :param sku_outer_ids: sku的商家编码。如果多个，用逗号分隔，并与其他sku信息保持相同顺序
        :param sku_quantitys: sku的库存。如果多个，用逗号分隔，并与其他sku信息保持相同顺序
        :param sku_properties: sku的属性。如果多个，用逗号分隔，并与其他sku信息保持相同顺序
        :param dealer_cost_price: 经销采购价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param sku_dealer_cost_prices: sku的经销采购价。如果多个，用逗号分隔，并与其他sku信息保持相同顺序。其中每个值的单位：元。例：“10.56,12.3”。必须在0.01元到10000000元之间。
        :param item_id: 导入的商品ID
        :param standard_retail_price: 零售基准价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param spu_id: 产品spuID，达尔文产品必须要传spuID，否则不能发布。其他非达尔文产品，看情况传
        """
        return self._top_request(
            "taobao.fenxiao.product.add",
            {
                "name": name,
                "productcat_id": productcat_id,
                "standard_price": standard_price,
                "retail_price_low": retail_price_low,
                "retail_price_high": retail_price_high,
                "category_id": category_id,
                "quantity": quantity,
                "desc": desc,
                "prov": prov,
                "city": city,
                "postage_type": postage_type,
                "have_invoice": have_invoice,
                "have_quarantee": have_quarantee,
                "trade_type": trade_type,
                "cost_price": cost_price,
                "outer_id": outer_id,
                "postage_id": postage_id,
                "postage_ordinary": postage_ordinary,
                "postage_fast": postage_fast,
                "postage_ems": postage_ems,
                "discount_id": discount_id,
                "is_authz": is_authz,
                "pic_path": pic_path,
                "image": image,
                "properties": properties,
                "property_alias": property_alias,
                "input_properties": input_properties,
                "sku_standard_prices": sku_standard_prices,
                "sku_cost_prices": sku_cost_prices,
                "sku_outer_ids": sku_outer_ids,
                "sku_quantitys": sku_quantitys,
                "sku_properties": sku_properties,
                "dealer_cost_price": dealer_cost_price,
                "sku_dealer_cost_prices": sku_dealer_cost_prices,
                "item_id": item_id,
                "standard_retail_price": standard_retail_price,
                "spu_id": spu_id
            }
        )

    def taobao_fenxiao_product_update(
            self,
            pid,
            postage_type='',
            have_invoice='',
            have_quarantee='',
            status='',
            name='',
            standard_price='',
            cost_price='',
            retail_price_low='',
            retail_price_high='',
            outer_id='',
            quantity='',
            desc='',
            prov='',
            city='',
            postage_id='',
            postage_ordinary='',
            postage_fast='',
            postage_ems='',
            sku_ids='',
            sku_cost_prices='',
            sku_quantitys='',
            sku_outer_ids='',
            discount_id='',
            sku_standard_prices='',
            sku_properties='',
            sku_properties_del='',
            is_authz='',
            pic_path='',
            image='',
            properties='',
            property_alias='',
            input_properties='',
            dealer_cost_price='',
            sku_dealer_cost_prices='',
            category_id='',
            standard_retail_price=''
    ):
        """
        更新产品
        更新分销平台产品数据，不传更新数据返回失败<br><br/>1. 对sku进行增、删操作时，原有的sku_ids字段会被忽略，请使用sku_properties和sku_properties_del。<br>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=326

        :param pid: 产品ID
        :param postage_type: 运费类型，可选值：seller（供应商承担运费）、buyer（分销商承担运费）。
        :param have_invoice: 是否有发票，可选值：false（否）、true（是），默认false。
        :param have_quarantee: 是否有保修，可选值：false（否）、true（是），默认false。
        :param status: 发布状态，可选值：up（上架）、down（下架）、delete（删除），输入非法字符则忽略。
        :param name: 产品名称，长度不超过60个字节。
        :param standard_price: 采购基准价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param cost_price: 代销采购价格，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param retail_price_low: 最低零售价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param retail_price_high: 最高零售价，单位：元。例：“10.56”。必须在0.01元到10000000元之间，最高零售价必须大于最低零售价。
        :param outer_id: 商家编码，长度不能超过60个字节。
        :param quantity: 产品库存必须是1到999999。
        :param desc: 产品描述，长度为5到25000字符。
        :param prov: 所在地：省，例：“浙江”
        :param city: 所在地：市，例：“杭州”
        :param postage_id: 运费模板ID，参考taobao.postages.get。更新时必须指定运费类型为 buyer，否则不更新。
        :param postage_ordinary: 平邮费用，单位：元。例：“10.56”。大小为0.01元到999999元之间。更新时必须指定运费类型为buyer，否则不更新。
        :param postage_fast: 快递费用，单位：元。例：“10.56”。大小为0.01元到999999元之间。更新时必须指定运费类型为buyer，否则不更新。
        :param postage_ems: ems费用，单位：元。例：“10.56”。大小为0.01元到999999元之间。更新时必须指定运费类型为buyer，否则不更新。
        :param sku_ids: sku id列表，例：1001,1002,1003。如果传入sku_properties将忽略此参数。
        :param sku_cost_prices: sku采购价格，单位元，例：'10.50,11.00,20.50'，字段必须和上面的sku_ids或sku_properties保持一致。
        :param sku_quantitys: sku库存，单位元，例：'10,20,30'，字段必须和sku_ids或sku_properties保持一致。
        :param sku_outer_ids: sku商家编码 ，单位元，例：'S1000,S1002,S1003'，字段必须和上面的id或sku_properties保持一致，如果没有可以写成',,'
        :param discount_id: 折扣ID
        :param sku_standard_prices: sku采购基准价，单位元，例：'10.50,11.00,20.50'，字段必须和上面的sku_ids或sku_properties保持一致。
        :param sku_properties: sku属性。格式:pid:vid;pid:vid,表示一组属性如:1627207:3232483;1630696:3284570,表示一组:机身颜色:军绿色;手机套餐:一电一充。多组之间用逗号“,”区分。(属性的pid调用taobao.itemprops.get取得，属性值的vid用taobao.itempropvalues.get取得vid)<br/>通过此字段可新增和更新sku。若传入此值将忽略sku_ids字段。sku其他字段与此值保持一致。
        :param sku_properties_del: 根据sku属性删除sku信息。需要按组删除属性。
        :param is_authz: 产品是否需要授权isAuthz:yes|no <br/>yes:需要授权 <br/>no:不需要授权
        :param pic_path: 产品主图图片空间相对路径或绝对路径
        :param image: 主图图片，如果pic_path参数不空，则优先使用pic_path，忽略该参数
        :param properties: 产品属性
        :param property_alias: 属性别名
        :param input_properties: 自定义属性。格式为pid:value;pid:value
        :param dealer_cost_price: 经销采购价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        :param sku_dealer_cost_prices: sku的经销采购价。如果多个，用逗号分隔，并与其他sku信息保持相同顺序。其中每个值的单位：元。例：“10.56,12.3”。必须在0.01元到10000000元之间。
        :param category_id: 所属类目id，参考Taobao.itemcats.get，不支持成人等类目，输入成人类目id保存提示类目属性错误。
        :param standard_retail_price: 零售基准价，单位：元。例：“10.56”。必须在0.01元到10000000元之间。
        """
        return self._top_request(
            "taobao.fenxiao.product.update",
            {
                "pid": pid,
                "postage_type": postage_type,
                "have_invoice": have_invoice,
                "have_quarantee": have_quarantee,
                "status": status,
                "name": name,
                "standard_price": standard_price,
                "cost_price": cost_price,
                "retail_price_low": retail_price_low,
                "retail_price_high": retail_price_high,
                "outer_id": outer_id,
                "quantity": quantity,
                "desc": desc,
                "prov": prov,
                "city": city,
                "postage_id": postage_id,
                "postage_ordinary": postage_ordinary,
                "postage_fast": postage_fast,
                "postage_ems": postage_ems,
                "sku_ids": sku_ids,
                "sku_cost_prices": sku_cost_prices,
                "sku_quantitys": sku_quantitys,
                "sku_outer_ids": sku_outer_ids,
                "discount_id": discount_id,
                "sku_standard_prices": sku_standard_prices,
                "sku_properties": sku_properties,
                "sku_properties_del": sku_properties_del,
                "is_authz": is_authz,
                "pic_path": pic_path,
                "image": image,
                "properties": properties,
                "property_alias": property_alias,
                "input_properties": input_properties,
                "dealer_cost_price": dealer_cost_price,
                "sku_dealer_cost_prices": sku_dealer_cost_prices,
                "category_id": category_id,
                "standard_retail_price": standard_retail_price
            }
        )

    def taobao_fenxiao_productcats_get(
            self,
            fields=''
    ):
        """
        查询产品线列表
        查询供应商的所有产品线数据。根据登陆用户来查询，不需要其他入参
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=327

        :param fields: 返回字段列表
        """
        return self._top_request(
            "taobao.fenxiao.productcats.get",
            {
                "fields": fields
            }
        )

    def taobao_fenxiao_products_get(
            self,
            outer_id='',
            productcat_id='',
            pids='',
            fields='',
            start_modified='',
            end_modified='',
            page_no='',
            page_size='',
            sku_number='',
            is_authz='',
            item_ids=''
    ):
        """
        查询产品列表
        查询供应商的产品数据。<br/><br/>    * 入参传入pids将优先查询，即只按这个条件查询。<br/>    *入参传入sku_number将优先查询(没有传入pids)，即只按这个条件查询(最多显示50条)<br/>    * 入参fields传skus将查询sku的数据，不传该参数默认不查询，返回产品的其它信息。<br/>    * 入参fields传入images将查询多图数据，不传只返回主图数据。<br/>    * 入参fields仅对传入pids生效（只有按ID查询时，才能查询额外的数据）<br/>    * 查询结果按照产品发布时间倒序，即时间近的数据在前。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=328

        :param outer_id: 商家编码
        :param productcat_id: 产品线ID
        :param pids: 产品ID列表（最大限制30），用逗号分割，例如：“1001,1002,1003,1004,1005”
        :param fields: 指定查询额外的信息，可选值：skus（sku数据）、images（多图），多个可选值用逗号分割。
        :param start_modified: 开始修改时间
        :param end_modified: 结束修改时间
        :param page_no: 页码（大于0的整数，默认1）
        :param page_size: 每页记录数（默认20，最大50）
        :param sku_number: sku商家编码
        :param is_authz: 查询产品列表时，查询入参“是否需要授权”<br/>yes:需要授权 <br/>no:不需要授权
        :param item_ids: 可根据导入的商品ID列表查询，优先级次于产品ID、sku_numbers，高于其他分页查询条件。最大限制20，用逗号分割，例如：“1001,1002,1003,1004,1005”
        """
        return self._top_request(
            "taobao.fenxiao.products.get",
            {
                "outer_id": outer_id,
                "productcat_id": productcat_id,
                "pids": pids,
                "fields": fields,
                "start_modified": start_modified,
                "end_modified": end_modified,
                "page_no": page_no,
                "page_size": page_size,
                "sku_number": sku_number,
                "is_authz": is_authz,
                "item_ids": item_ids
            }
        )

    def taobao_fenxiao_distributors_get(
            self,
            nicks
    ):
        """
        获取分销商信息
        查询和当前登录供应商有合作关系的分销商的信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10379

        :param nicks: 分销商用户名列表。多个之间以“,”分隔;最多支持50个分销商用户名。
        """
        return self._top_request(
            "taobao.fenxiao.distributors.get",
            {
                "nicks": nicks
            },
            result_processor=lambda x: x["distributors"]
        )

    def taobao_fenxiao_cooperation_update(
            self,
            distributor_id,
            grade_id,
            trade_type=''
    ):
        """
        更新合作关系等级
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        供应商更新合作的分销商等级
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10568

        :param distributor_id: 分销商ID
        :param grade_id: 等级ID(0代表取消)
        :param trade_type: 分销方式(新增)： AGENT(代销)、DEALER(经销)(默认为代销)
        """
        return self._top_request(
            "taobao.fenxiao.cooperation.update",
            {
                "distributor_id": distributor_id,
                "grade_id": grade_id,
                "trade_type": trade_type
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_grades_get(
            self
    ):
        """
        分销商等级查询
        根据供应商ID，查询他的分销商等级信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10569

        """
        return self._top_request(
            "taobao.fenxiao.grades.get",
            result_processor=lambda x: x["fenxiao_grades"]
        )

    def taobao_fenxiao_discounts_get(
            self,
            discount_id='',
            ext_fields=''
    ):
        """
        获取折扣信息
        查询折扣信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10571

        :param discount_id: 折扣ID
        :param ext_fields: 指定查询额外的信息，可选值：DETAIL（查询折扣详情），多个可选值用逗号分割。（只允许指定折扣ID情况下使用）
        """
        return self._top_request(
            "taobao.fenxiao.discounts.get",
            {
                "discount_id": discount_id,
                "ext_fields": ext_fields
            }
        )

    def taobao_fenxiao_order_confirm_paid(
            self,
            purchase_order_id,
            confirm_remark=''
    ):
        """
        确认收款
        供应商确认收款（非支付宝交易）。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10692

        :param purchase_order_id: 采购单编号。
        :param confirm_remark: 确认支付信息（字数小于100）
        """
        return self._top_request(
            "taobao.fenxiao.order.confirm.paid",
            {
                "purchase_order_id": purchase_order_id,
                "confirm_remark": confirm_remark
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_distributor_items_get(
            self,
            distributor_id='',
            start_modified='',
            end_modified='',
            page_no='',
            page_size='',
            product_id=''
    ):
        """
        查询商品下载记录
        供应商查询分销商商品下载记录。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10693

        :param distributor_id: 分销商ID 。
        :param start_modified: 设置开始时间。空为不设置。
        :param end_modified: 设置结束时间,空为不设置。
        :param page_no: 页码（大于0的整数，默认1）
        :param page_size: 每页记录数（默认20，最大50）
        :param product_id: 产品ID
        """
        return self._top_request(
            "taobao.fenxiao.distributor.items.get",
            {
                "distributor_id": distributor_id,
                "start_modified": start_modified,
                "end_modified": end_modified,
                "page_no": page_no,
                "page_size": page_size,
                "product_id": product_id
            }
        )

    def taobao_fenxiao_cooperation_get(
            self,
            status='',
            start_date='',
            end_date='',
            trade_type='',
            page_no='',
            page_size=''
    ):
        """
        供应商或分销商获取合作关系信息
        获取供应商的合作关系信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10694

        :param status: 合作状态： NORMAL(合作中)、 ENDING(终止中) 、END (终止)
        :param start_date: 合作开始时间yyyy-MM-dd HH:mm:ss
        :param end_date: 合作结束时间yyyy-MM-dd HH:mm:ss
        :param trade_type: 分销方式：AGENT(代销) 、DEALER（经销）
        :param page_no: 页码（大于0的整数，默认1）
        :param page_size: 每页记录数（默认20，最大50）
        """
        return self._top_request(
            "taobao.fenxiao.cooperation.get",
            {
                "status": status,
                "start_date": start_date,
                "end_date": end_date,
                "trade_type": trade_type,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_fenxiao_login_user_get(
            self
    ):
        """
        获取分销用户登录信息
        获取用户登录信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10781

        """
        return self._top_request(
            "taobao.fenxiao.login.user.get",
            result_processor=lambda x: x["login_user"]
        )

    def taobao_fenxiao_product_sku_delete(
            self,
            product_id,
            properties
    ):
        """
        产品SKU删除接口
        根据sku properties删除sku数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10944

        :param product_id: 产品id
        :param properties: sku属性
        """
        return self._top_request(
            "taobao.fenxiao.product.sku.delete",
            {
                "product_id": product_id,
                "properties": properties
            }
        )

    def taobao_fenxiao_product_sku_add(
            self,
            product_id,
            standard_price,
            properties,
            quantity='',
            agent_cost_price='',
            sku_number='',
            dealer_cost_price=''
    ):
        """
        产品sku添加接口
        添加产品SKU信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10945

        :param product_id: 产品ID
        :param standard_price: 采购基准价，最大值1000000000
        :param properties: sku属性
        :param quantity: sku产品库存，在0到1000000之间，如果不传，则库存为0
        :param agent_cost_price: 代销采购价
        :param sku_number: 商家编码
        :param dealer_cost_price: 经销采购价
        """
        return self._top_request(
            "taobao.fenxiao.product.sku.add",
            {
                "product_id": product_id,
                "standard_price": standard_price,
                "properties": properties,
                "quantity": quantity,
                "agent_cost_price": agent_cost_price,
                "sku_number": sku_number,
                "dealer_cost_price": dealer_cost_price
            }
        )

    def taobao_fenxiao_product_sku_update(
            self,
            product_id,
            properties,
            quantity='',
            standard_price='',
            agent_cost_price='',
            sku_number='',
            dealer_cost_price=''
    ):
        """
        产品sku编辑接口
        产品SKU信息更新
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10946

        :param product_id: 产品ID
        :param properties: sku属性
        :param quantity: 产品SKU库存
        :param standard_price: 采购基准价
        :param agent_cost_price: 代销采购价
        :param sku_number: 商家编码
        :param dealer_cost_price: 经销采购价
        """
        return self._top_request(
            "taobao.fenxiao.product.sku.update",
            {
                "product_id": product_id,
                "properties": properties,
                "quantity": quantity,
                "standard_price": standard_price,
                "agent_cost_price": agent_cost_price,
                "sku_number": sku_number,
                "dealer_cost_price": dealer_cost_price
            }
        )

    def taobao_fenxiao_product_skus_get(
            self,
            product_id
    ):
        """
        SKU查询接口
        产品sku查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10947

        :param product_id: 产品ID
        """
        return self._top_request(
            "taobao.fenxiao.product.skus.get",
            {
                "product_id": product_id
            }
        )

    def taobao_fenxiao_product_image_upload(
            self,
            product_id,
            position,
            pic_path='',
            image='',
            properties=''
    ):
        """
        产品图片上传
        产品主图图片空间相对路径或绝对路径添加或更新，或者是图片上传。如果指定位置的图片已存在，则覆盖原有信息。如果位置为1,自动设为主图；如果位置为0，表示属性图片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10948

        :param product_id: 产品ID
        :param position: 图片位置，0-14之间。0：操作sku属性图片，1：主图，2-5：细节图，6-14：额外主图
        :param pic_path: 产品主图图片空间相对路径或绝对路径
        :param image: 产品图片
        :param properties: properties表示sku图片的属性。key:value形式，key是pid，value是vid。如果position是0的话，则properties需要是必传项
        """
        return self._top_request(
            "taobao.fenxiao.product.image.upload",
            {
                "product_id": product_id,
                "position": position,
                "pic_path": pic_path,
                "image": image,
                "properties": properties
            }
        )

    def taobao_fenxiao_product_image_delete(
            self,
            product_id,
            position,
            properties=''
    ):
        """
        产品图片删除
        产品图片删除，只删除图片信息，不真正删除图片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10949

        :param product_id: 产品ID
        :param position: 图片位置
        :param properties: properties表示sku图片的属性。key:value形式，key是pid，value是vid。如果position是0的话，则properties需要是必传项
        """
        return self._top_request(
            "taobao.fenxiao.product.image.delete",
            {
                "product_id": product_id,
                "position": position,
                "properties": properties
            }
        )

    def taobao_fenxiao_order_remark_update(
            self,
            purchase_order_id,
            supplier_memo,
            supplier_memo_flag=''
    ):
        """
        修改采购单备注
        供应商修改采购单备注
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11106

        :param purchase_order_id: 采购单编号
        :param supplier_memo: 备注内容(供应商操作)
        :param supplier_memo_flag: 旗子的标记，1-5之间的数字。非1-5之间，都采用1作为默认。<br/>1:红色<br/>2:黄色<br/>3:绿色<br/>4:蓝色<br/>5:粉红色
        """
        return self._top_request(
            "taobao.fenxiao.order.remark.update",
            {
                "purchase_order_id": purchase_order_id,
                "supplier_memo": supplier_memo,
                "supplier_memo_flag": supplier_memo_flag
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_trademonitor_get(
            self,
            start_created='',
            end_created='',
            page_no='1',
            page_size='50',
            product_id='',
            distributor_nick='',
            fields=''
    ):
        """
        查询经销商品监控
        仅限供应商调用此接口查询经销商品监控信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21391

        :param start_created: 起始时间 格式 yyyy-MM-dd HH:mm:ss.支持到秒的查询。若不传时分秒，默认为0时0分0秒。
        :param end_created: 结束时间 格式 yyyy-MM-dd HH:mm:ss.支持到秒的查询。若不传时分秒，默认为0时0分0秒。
        :param page_no: 页码。（大于0的整数。小于1按1计）
        :param page_size: 每页条数。（每页条数不超过50条，超过50或小于0均按50计）
        :param product_id: 产品id
        :param distributor_nick: 经销商的淘宝账号
        :param fields: 多个字段用','分隔。 fields 如果为空：返回所有采购单对象(purchase_orders)字段。 如果不为空：返回指定采购单对象(purchase_orders)字段。例如：trade_monitors.item_title表示只返回item_title
        """
        return self._top_request(
            "taobao.fenxiao.trademonitor.get",
            {
                "start_created": start_created,
                "end_created": end_created,
                "page_no": page_no,
                "page_size": page_size,
                "product_id": product_id,
                "distributor_nick": distributor_nick,
                "fields": fields
            }
        )

    def taobao_fenxiao_product_gradeprice_update(
            self,
            product_id,
            target_type,
            ids,
            prices,
            trade_type='',
            sku_id=''
    ):
        """
        根据sku设置折扣价
        供应商可以针对产品不同的sku，指定对应交易类型（代销or经销）方式下，设定折扣方式（按等级or指定分销商）以及对应优惠后的采购价格
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21420

        :param product_id: 产品Id
        :param target_type: 选择折扣方式：GRADE（按等级进行设置）;DISCITUTOR(按分销商进行设置）。例如'GRADE,DISTRIBUTOR'
        :param ids: 会员等级的id或者分销商id，例如：”1001,2001,1002”
        :param prices: 优惠价格,大小为0到100000000之间的整数或两位小数，例：优惠价格为：100元2角5分，传入的参数应写成：100.25
        :param trade_type: 交易类型： AGENT(代销)、DEALER(经销)，ALL(代销和经销)
        :param sku_id: skuId，如果产品有skuId,必须要输入skuId;没有skuId的时候不必选
        """
        return self._top_request(
            "taobao.fenxiao.product.gradeprice.update",
            {
                "product_id": product_id,
                "target_type": target_type,
                "ids": ids,
                "prices": prices,
                "trade_type": trade_type,
                "sku_id": sku_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_product_map_add(
            self,
            product_id,
            sc_item_id='',
            sku_ids='',
            sc_item_ids='',
            not_check_outer_code='true'
    ):
        """
        创建分销和后端商品映射关系
        创建分销和供应链商品映射关系。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21589

        :param product_id: 分销产品id。
        :param sc_item_id: 后端商品id（如果当前分销产品没有sku和后端商品时需要指定）。
        :param sku_ids: 分销产品的sku id。逗号分隔，顺序需要保证与sc_item_ids一致（没有sku就不传）。
        :param sc_item_ids: 在有sku的情况下，与各个sku对应的后端商品id列表。逗号分隔，顺序需要保证与sku_ids一致。
        :param not_check_outer_code: 是否需要校验商家编码，true不校验，false校验。
        """
        return self._top_request(
            "taobao.fenxiao.product.map.add",
            {
                "product_id": product_id,
                "sc_item_id": sc_item_id,
                "sku_ids": sku_ids,
                "sc_item_ids": sc_item_ids,
                "not_check_outer_code": not_check_outer_code
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_scitem_outercode_get(
            self,
            outer_code
    ):
        """
        根据outerCode查询商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21593

        :param outer_code: 商品编码
        """
        return self._top_request(
            "taobao.scitem.outercode.get",
            {
                "outer_code": outer_code
            },
            result_processor=lambda x: x["sc_item"]
        )

    def taobao_scitem_get(
            self,
            item_id
    ):
        """
        根据id查询商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21594

        :param item_id: 商品id
        """
        return self._top_request(
            "taobao.scitem.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["sc_item"]
        )

    def taobao_scitem_query(
            self,
            item_name='',
            outer_code='',
            bar_code='',
            item_type='',
            wms_code='',
            page_index='',
            page_size=''
    ):
        """
        查询后端商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21595

        :param item_name: 商品名称
        :param outer_code: 商家给商品的一个编码
        :param bar_code: 条形码
        :param item_type: ITEM类型(只允许输入以下英文或空) NORMAL 0:普通商品; COMBINE 1:是否是组合商品 DISTRIBUTION
        :param wms_code: 仓库编码
        :param page_index: 当前页码数
        :param page_size: 分页记录个数，如果用户输入的记录数大于50，则一页显示50条记录
        """
        return self._top_request(
            "taobao.scitem.query",
            {
                "item_name": item_name,
                "outer_code": outer_code,
                "bar_code": bar_code,
                "item_type": item_type,
                "wms_code": wms_code,
                "page_index": page_index,
                "page_size": page_size
            }
        )

    def taobao_scitem_update(
            self,
            item_id='',
            outer_code='',
            item_name='',
            item_type='0',
            update_properties='',
            bar_code='',
            wms_code='',
            is_friable='',
            is_dangerous='',
            is_costly='',
            is_warranty='',
            weight='',
            length='',
            width='',
            height='',
            volume='',
            price='',
            remark='',
            matter_status='',
            brand_id='',
            brand_name='',
            spu_id='',
            remove_properties='',
            is_area_sale=''
    ):
        """
        根据商品ID或商家编码修改后端商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21596

        :param item_id: 后端商品ID，跟outer_code必须指定一个
        :param outer_code: 商家编码，跟item_id必须指定一个
        :param item_name: 商品名称
        :param item_type: 0.普通供应链商品 1.供应链组合主商品
        :param update_properties: 需要更新的商品属性格式是  p1:v1,p2:v2,p3:v3
        :param bar_code: 条形码
        :param wms_code: 仓储商编码
        :param is_friable: 是否易碎 0：不是  1：是
        :param is_dangerous: 是否危险 0：不是  0：是
        :param is_costly: 是否是贵重品 0:不是 1：是
        :param is_warranty: 是否保质期：0:不是 1：是
        :param weight: weight
        :param length: 长度 单位：mm
        :param width: 宽 单位：mm
        :param height: 高 单位：mm
        :param volume: 体积：立方厘米
        :param price: price
        :param remark: remark
        :param matter_status: 0:液体，1：粉体，2：固体
        :param brand_id: 品牌id
        :param brand_name: brand_Name
        :param spu_id: 淘宝SKU产品级编码CSPU ID
        :param remove_properties: 移除商品属性P列表,P由系统分配：p1；p2
        :param is_area_sale: 1表示区域销售，0或是空是非区域销售
        """
        return self._top_request(
            "taobao.scitem.update",
            {
                "item_id": item_id,
                "outer_code": outer_code,
                "item_name": item_name,
                "item_type": item_type,
                "update_properties": update_properties,
                "bar_code": bar_code,
                "wms_code": wms_code,
                "is_friable": is_friable,
                "is_dangerous": is_dangerous,
                "is_costly": is_costly,
                "is_warranty": is_warranty,
                "weight": weight,
                "length": length,
                "width": width,
                "height": height,
                "volume": volume,
                "price": price,
                "remark": remark,
                "matter_status": matter_status,
                "brand_id": brand_id,
                "brand_name": brand_name,
                "spu_id": spu_id,
                "remove_properties": remove_properties,
                "is_area_sale": is_area_sale
            },
            result_processor=lambda x: x["update_rows"]
        )

    def taobao_scitem_add(
            self,
            item_name,
            outer_code,
            item_type='0',
            properties='',
            bar_code='',
            wms_code='',
            is_friable='',
            is_dangerous='',
            is_costly='',
            is_warranty='',
            weight='',
            length='',
            width='',
            height='',
            volume='',
            price='',
            remark='',
            matter_status='',
            brand_id='',
            brand_name='',
            spu_id='',
            is_area_sale='0'
    ):
        """
        发布后端商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21597

        :param item_name: 商品名称
        :param outer_code: 商家编码
        :param item_type: 0.普通供应链商品 1.供应链组合主商品
        :param properties: 商品属性格式是  p1:v1,p2:v2,p3:v3
        :param bar_code: 条形码
        :param wms_code: 仓储商编码
        :param is_friable: 是否易碎 0：不是  1：是
        :param is_dangerous: 是否危险 0：不是  1：是
        :param is_costly: 是否是贵重品 0:不是 1：是
        :param is_warranty: 是否保质期：0:不是 1：是
        :param weight: 重量 单位：g
        :param length: 长度 单位：mm
        :param width: 宽 单位：mm
        :param height: 高 单位：mm
        :param volume: 体积：立方厘米
        :param price: 价格 单位：分
        :param remark: remark
        :param matter_status: 0:液体，1：粉体，2：固体
        :param brand_id: 品牌id
        :param brand_name: brand_Name
        :param spu_id: spuId或是cspuid
        :param is_area_sale: 1表示区域销售，0或是空是非区域销售
        """
        return self._top_request(
            "taobao.scitem.add",
            {
                "item_name": item_name,
                "outer_code": outer_code,
                "item_type": item_type,
                "properties": properties,
                "bar_code": bar_code,
                "wms_code": wms_code,
                "is_friable": is_friable,
                "is_dangerous": is_dangerous,
                "is_costly": is_costly,
                "is_warranty": is_warranty,
                "weight": weight,
                "length": length,
                "width": width,
                "height": height,
                "volume": volume,
                "price": price,
                "remark": remark,
                "matter_status": matter_status,
                "brand_id": brand_id,
                "brand_name": brand_name,
                "spu_id": spu_id,
                "is_area_sale": is_area_sale
            },
            result_processor=lambda x: x["sc_item"]
        )

    def taobao_scitem_map_delete(
            self,
            sc_item_id,
            user_nick=''
    ):
        """
        失效指定用户的商品与后端商品的映射关系
        根据后端商品Id，失效指定用户的商品与后端商品的映射关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21598

        :param sc_item_id: 后台商品ID
        :param user_nick: 店铺用户nick。 如果该参数为空则删除后端商品与当前调用人的商品映射关系;如果不为空则删除指定用户与后端商品的映射关系
        """
        return self._top_request(
            "taobao.scitem.map.delete",
            {
                "sc_item_id": sc_item_id,
                "user_nick": user_nick
            },
            result_processor=lambda x: x["module"]
        )

    def taobao_scitem_map_query(
            self,
            item_id,
            sku_id=''
    ):
        """
        查找IC商品或分销商品与后端商品的关联信息
        查找IC商品或分销商品与后端商品的关联信息。skuId如果不传就查找该itemId下所有的sku
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21599

        :param item_id: map_type为1：前台ic商品id<br/>map_type为2：分销productid
        :param sku_id: map_type为1：前台ic商品skuid <br/>map_type为2：分销商品的skuid
        """
        return self._top_request(
            "taobao.scitem.map.query",
            {
                "item_id": item_id,
                "sku_id": sku_id
            },
            result_processor=lambda x: x["sc_item_maps"]
        )

    def taobao_scitem_map_add(
            self,
            item_id,
            sku_id='',
            sc_item_id='',
            outer_code='',
            need_check='false'
    ):
        """
        创建IC商品与后端商品的映射关系
        创建IC商品或分销商品与后端商品的映射关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21600

        :param item_id: 前台ic商品id
        :param sku_id: 前台ic商品skuid
        :param sc_item_id: sc_item_id和outer_code 其中一个不能为空
        :param outer_code: sc_item_id和outer_code 其中一个不能为空
        :param need_check: 默认值为false<br/>true:进行高级校验,前端商品或SKU的商家编码必须与后端商品的商家编码一致，否则会拒绝关联<br/>false:不进行高级校验
        """
        return self._top_request(
            "taobao.scitem.map.add",
            {
                "item_id": item_id,
                "sku_id": sku_id,
                "sc_item_id": sc_item_id,
                "outer_code": outer_code,
                "need_check": need_check
            },
            result_processor=lambda x: x["outer_code"]
        )

    def taobao_inventory_store_query(
            self,
            store_code=''
    ):
        """
        查询仓库信息
        查询商家仓信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21601

        :param store_code: 商家的仓库编码
        """
        return self._top_request(
            "taobao.inventory.store.query",
            {
                "store_code": store_code
            },
            result_processor=lambda x: x["store_list"]
        )

    def taobao_inventory_adjust_trade(
            self,
            tb_order_type,
            operate_time,
            biz_unique_code,
            items
    ):
        """
        交易库存调整单
        商家交易调整库存，淘宝交易、B2B经销等
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21607

        :param tb_order_type: 订单类型：B2C、B2B
        :param operate_time: 业务操作时间
        :param biz_unique_code: 商家外部定单号
        :param items: 商品初始库存信息： [{ 'TBOrderCode”:”淘宝交易号”,'TBSubOrderCode ':'淘宝子交易单号,赠品可以不填','”isGift”:”TRUE或者FALSE,是否赠品”,storeCode':'商家仓库编码',' scItemId ':'商品后端ID','scItemCode':'商品商家编码',' originScItemId ':'原商品ID','inventoryType':'','quantity':'111','isComplete':'TRUE或者FALSE，是否全部确认出库'}]
        """
        return self._top_request(
            "taobao.inventory.adjust.trade",
            {
                "tb_order_type": tb_order_type,
                "operate_time": operate_time,
                "biz_unique_code": biz_unique_code,
                "items": items
            }
        )

    def taobao_inventory_adjust_external(
            self,
            biz_unique_code,
            items,
            store_code,
            reduce_type='',
            biz_type='',
            operate_type='',
            occupy_operate_code='',
            operate_time=''
    ):
        """
        非交易库存调整单
        商家非交易调整库存，调拨出库、盘点等时调用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21608

        :param biz_unique_code: 商家外部定单号
        :param items: 商品初始库存信息： [{'scItemId':'商品后端ID，如果有传scItemCode,参数可以为0','scItemCode':'商品商家编码','inventoryType':'库存类型  1：正常,”direction”: 1: 盘盈 -1: 盘亏,参数可选,'quantity':'数量(正数)'}]
        :param store_code: 商家仓库编码
        :param reduce_type: test
        :param biz_type: 外部订单类型, BALANCE:盘点、NON_TAOBAO_TRADE:非淘宝交易、ALLOCATE:调拨、OTHERS:其他
        :param operate_type: test
        :param occupy_operate_code: 库存占用返回的操作码.operate_type 为OUTBOUND时，如果是确认事先进行过的库存占用，需要传入当时返回的操作码，并且明细必须与申请时保持一致
        :param operate_time: 业务操作时间
        """
        return self._top_request(
            "taobao.inventory.adjust.external",
            {
                "biz_unique_code": biz_unique_code,
                "items": items,
                "store_code": store_code,
                "reduce_type": reduce_type,
                "biz_type": biz_type,
                "operate_type": operate_type,
                "occupy_operate_code": occupy_operate_code,
                "operate_time": operate_time
            }
        )

    def taobao_inventory_store_manage(
            self,
            operate_type,
            store_code,
            store_name,
            store_type,
            address,
            address_area_name,
            alias_name='',
            contact='',
            phone='',
            postcode=''
    ):
        """
        创建或更新仓库
        创建商家仓或者更新商家仓信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21611

        :param operate_type: 参数定义，ADD：新建; UPDATE：更新
        :param store_code: 商家的仓库编码，不允许重复，不允许更新
        :param store_name: 商家的仓库名称，可更新
        :param store_type: 仓库类型，可更新。目前只支持自有仓，TYPE_OWN：自有物理仓
        :param address: 仓库的物理地址，可更新
        :param address_area_name: 仓库区域名，可更新
        :param alias_name: 仓库简称，可更新
        :param contact: 联系人，可更新
        :param phone: 联系电话，可更新
        :param postcode: 邮编，可更新
        """
        return self._top_request(
            "taobao.inventory.store.manage",
            {
                "operate_type": operate_type,
                "store_code": store_code,
                "store_name": store_name,
                "store_type": store_type,
                "address": address,
                "address_area_name": address_area_name,
                "alias_name": alias_name,
                "contact": contact,
                "phone": phone,
                "postcode": postcode
            },
            result_processor=lambda x: x["store_list"]
        )

    def taobao_inventory_initial(
            self,
            store_code,
            items
    ):
        """
        库存初始化
        商家仓库存初始化接口，直接按照商家指定的商品库存数进行填充，没有单据核对，不参与库存对账
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21612

        :param store_code: 商家仓库编码
        :param items: 商品初始库存信息： [{'scItemId':'商品后端ID，如果有传scItemCode,参数可以为0','scItemCode':'商品商家编码','inventoryType':'库存类型  1：正常,2：损坏,3：冻结,10：质押,11-20:用户自定义','quantity':'数量'}]
        """
        return self._top_request(
            "taobao.inventory.initial",
            {
                "store_code": store_code,
                "items": items
            },
            result_processor=lambda x: x["tip_infos"]
        )

    def taobao_inventory_query(
            self,
            sc_item_ids,
            sc_item_codes='',
            seller_nick='',
            store_codes=''
    ):
        """
        查询商品库存信息
        商家查询商品总体库存信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21613

        :param sc_item_ids: 后端商品ID 列表，控制到50个
        :param sc_item_codes: 后端商品的商家编码列表，控制到50个
        :param seller_nick: 卖家昵称
        :param store_codes: 仓库列表:GLY001^GLY002
        """
        return self._top_request(
            "taobao.inventory.query",
            {
                "sc_item_ids": sc_item_ids,
                "sc_item_codes": sc_item_codes,
                "seller_nick": seller_nick,
                "store_codes": store_codes
            }
        )

    def taobao_fenxiao_requisitions_get(
            self,
            status='',
            apply_start='',
            apply_end='',
            page_no='',
            page_size=''
    ):
        """
        合作申请查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21617

        :param status: 申请状态（1-申请中、2-成功、3-被退回、4-已撤消、5-过期）
        :param apply_start: 申请开始时间yyyy-MM-dd
        :param apply_end: 申请结束时间yyyy-MM-dd
        :param page_no: 页码（大于0的整数，默认1）
        :param page_size: 每页记录数（默认20，最大50）
        """
        return self._top_request(
            "taobao.fenxiao.requisitions.get",
            {
                "status": status,
                "apply_start": apply_start,
                "apply_end": apply_end,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_fenxiao_cooperation_productcat_add(
            self,
            cooperate_id,
            product_line_list,
            grade_id=''
    ):
        """
        追加授权产品线
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21618

        :param cooperate_id: 合作关系id
        :param product_line_list: 产品线id列表，若有多个，以逗号分隔
        :param grade_id: 等级ID（为空则不修改）
        """
        return self._top_request(
            "taobao.fenxiao.cooperation.productcat.add",
            {
                "cooperate_id": cooperate_id,
                "product_line_list": product_line_list,
                "grade_id": grade_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_cooperation_audit(
            self,
            requisition_id,
            audit_result,
            remark,
            product_line_list_agent='',
            product_line_list_dealer=''
    ):
        """
        合作授权审批
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21619

        :param requisition_id: 合作申请Id
        :param audit_result: 1:审批通过，审批通过后要加入授权产品线列表；<br/>2:审批拒绝
        :param remark: 备注
        :param product_line_list_agent: 当审批通过时需要指定授权产品线id列表(代销)，如果没传则表示不开通，且经销和代销的授权产品线列表至少传入一种，同时传入则表示都开通。
        :param product_line_list_dealer: 当审批通过时需要指定授权产品线id列表(经销)，如果没传则表示不开通，且经销和代销的授权产品线列表至少传入一种，同时传入则表示都开通。
        """
        return self._top_request(
            "taobao.fenxiao.cooperation.audit",
            {
                "requisition_id": requisition_id,
                "audit_result": audit_result,
                "remark": remark,
                "product_line_list_agent": product_line_list_agent,
                "product_line_list_dealer": product_line_list_dealer
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_cooperation_terminate(
            self,
            cooperate_id,
            end_remain_days,
            end_remark
    ):
        """
        终止合作
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        终止与分销商的合作关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21620

        :param cooperate_id: 合作编号
        :param end_remain_days: 终止天数，可选1,2,3,5,7,15，在多少天数内终止
        :param end_remark: 终止说明（5-2000字）
        """
        return self._top_request(
            "taobao.fenxiao.cooperation.terminate",
            {
                "cooperate_id": cooperate_id,
                "end_remain_days": end_remain_days,
                "end_remark": end_remark
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_product_gradeprice_get(
            self,
            product_id,
            sku_id='',
            trade_type=''
    ):
        """
        等级折扣查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21621

        :param product_id: 产品id
        :param sku_id: skuId
        :param trade_type: 经、代销模式（1：代销、2：经销）
        """
        return self._top_request(
            "taobao.fenxiao.product.gradeprice.get",
            {
                "product_id": product_id,
                "sku_id": sku_id,
                "trade_type": trade_type
            }
        )

    def taobao_fenxiao_grade_add(
            self,
            name
    ):
        """
        新建等级
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21624

        :param name: 等级名称，等级名称不可重复
        """
        return self._top_request(
            "taobao.fenxiao.grade.add",
            {
                "name": name
            }
        )

    def taobao_fenxiao_productcat_delete(
            self,
            product_line_id
    ):
        """
        删除产品线
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21625

        :param product_line_id: 产品线ID
        """
        return self._top_request(
            "taobao.fenxiao.productcat.delete",
            {
                "product_line_id": product_line_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_productcat_update(
            self,
            product_line_id,
            name='',
            retail_low_percent='',
            retail_high_percent='',
            agent_cost_percent='',
            dealer_cost_percent=''
    ):
        """
        修改产品线
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21626

        :param product_line_id: 产品线ID
        :param name: 产品线名称
        :param retail_low_percent: 最低零售价比例，注意：100.00%，则输入为10000
        :param retail_high_percent: 最高零售价比例，注意：100.00%，则输入为10000
        :param agent_cost_percent: 代销默认采购价比例，注意：100.00%，则输入为10000
        :param dealer_cost_percent: 经销默认采购价比例，注意：100.00%，则输入为10000
        """
        return self._top_request(
            "taobao.fenxiao.productcat.update",
            {
                "product_line_id": product_line_id,
                "name": name,
                "retail_low_percent": retail_low_percent,
                "retail_high_percent": retail_high_percent,
                "agent_cost_percent": agent_cost_percent,
                "dealer_cost_percent": dealer_cost_percent
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_productcat_add(
            self,
            name,
            retail_low_percent,
            retail_high_percent,
            agent_cost_percent,
            dealer_cost_percent
    ):
        """
        新增产品线
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21627

        :param name: 产品线名称
        :param retail_low_percent: 最低零售价比例，注意：100.00%，则输入为10000
        :param retail_high_percent: 最高零售价比例，注意：100.00%，则输入为10000
        :param agent_cost_percent: 代销默认采购价比例，注意：100.00%，则输入为10000
        :param dealer_cost_percent: 经销默认采购价比例，注意：100.00%，则输入为10000
        """
        return self._top_request(
            "taobao.fenxiao.productcat.add",
            {
                "name": name,
                "retail_low_percent": retail_low_percent,
                "retail_high_percent": retail_high_percent,
                "agent_cost_percent": agent_cost_percent,
                "dealer_cost_percent": dealer_cost_percent
            }
        )

    def taobao_inventory_initial_item(
            self,
            sc_item_id,
            store_inventorys
    ):
        """
        商品库存初始化
        商家仓商品初始化在各个仓中库存
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21728

        :param sc_item_id: 后端商品id
        :param store_inventorys: 商品初始库存信息： [{'storeCode':'必选,商家仓库编号','inventoryType':'可选，库存类型 1：正常,2：损坏,3：冻结,10：质押,11-20:用户自定义,默认为1','quantity':'必选,数量'}]
        """
        return self._top_request(
            "taobao.inventory.initial.item",
            {
                "sc_item_id": sc_item_id,
                "store_inventorys": store_inventorys
            },
            result_processor=lambda x: x["tip_infos"]
        )

    def taobao_fenxiao_refund_get(
            self,
            sub_order_id,
            query_seller_refund=''
    ):
        """
        查询采购单退款信息
        分销商或供应商可以查询某子单的退款信息，以及下游订单的退款信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21873

        :param sub_order_id: 要查询的退款子单的id
        :param query_seller_refund: 是否查询下游买家的退款信息
        """
        return self._top_request(
            "taobao.fenxiao.refund.get",
            {
                "sub_order_id": sub_order_id,
                "query_seller_refund": query_seller_refund
            },
            result_processor=lambda x: x["refund_detail"]
        )

    def taobao_fenxiao_distributor_products_get(
            self,
            order_by='',
            time_type='',
            download_status='',
            trade_type='',
            end_time='',
            fields='',
            item_ids='',
            productcat_id='',
            pids='',
            start_time='',
            page_no='',
            page_size='',
            supplier_nick=''
    ):
        """
        分销商查询产品信息
        分销商查询供应商产品信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21875

        :param order_by: order_by
        :param time_type: time_type
        :param download_status: download_status
        :param trade_type: trade_type
        :param end_time: 结束时间
        :param fields: 指定查询额外的信息，可选值：skus（sku数据）、images（多图），多个可选值用逗号分割。
        :param item_ids: 根据商品ID列表查询，优先级次于产品ID列表，高于其他分页查询条件。如果商品不是分销商品，自动过滤。最大限制20，用逗号分割，例如：“1001,1002,1003,1004,1005”
        :param productcat_id: 产品线ID
        :param pids: 产品ID列表（最大限制30），用逗号分割，例如：“1001,1002,1003,1004,1005”
        :param start_time: 开始修改时间
        :param page_no: 页码（大于0的整数，默认1）
        :param page_size: 每页记录数（默认20，最大50）
        :param supplier_nick: 供应商nick，分页查询时，必传
        """
        return self._top_request(
            "taobao.fenxiao.distributor.products.get",
            {
                "order_by": order_by,
                "time_type": time_type,
                "download_status": download_status,
                "trade_type": trade_type,
                "end_time": end_time,
                "fields": fields,
                "item_ids": item_ids,
                "productcat_id": productcat_id,
                "pids": pids,
                "start_time": start_time,
                "page_no": page_no,
                "page_size": page_size,
                "supplier_nick": supplier_nick
            }
        )

    def taobao_fenxiao_dealer_requisitionorder_close(
            self,
            dealer_order_id,
            reason,
            reason_detail
    ):
        """
        供应商/分销商关闭采购申请/经销采购单
        供应商或分销商关闭采购申请/经销采购单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22028

        :param dealer_order_id: 经销采购单编号
        :param reason: 关闭原因：<br/>1：长时间无法联系到分销商，取消交易。<br/>2：分销商错误提交申请，取消交易。<br/>3：缺货，近段时间都无法发货。<br/>4：分销商恶意提交申请单。<br/>5：其他原因。
        :param reason_detail: 关闭详细原因，字数5-200字
        """
        return self._top_request(
            "taobao.fenxiao.dealer.requisitionorder.close",
            {
                "dealer_order_id": dealer_order_id,
                "reason": reason,
                "reason_detail": reason_detail
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_dealer_requisitionorder_agree(
            self,
            dealer_order_id
    ):
        """
        供应商/分销商通过采购申请/经销采购单申请
        供应商或分销商通过采购申请/经销采购单审核
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22031

        :param dealer_order_id: 采购申请/经销采购单编号
        """
        return self._top_request(
            "taobao.fenxiao.dealer.requisitionorder.agree",
            {
                "dealer_order_id": dealer_order_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_dealer_requisitionorder_get(
            self,
            start_date,
            end_date,
            identity,
            page_no='1',
            page_size='50',
            order_status='0',
            fields=''
    ):
        """
        批量查询采购申请/经销采购单
        批量查询采购申请/经销采购单，目前支持供应商和分销商查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22032

        :param start_date: 采购申请/经销采购单最早修改时间
        :param end_date: 采购申请/经销采购单最迟修改时间。与start_date字段的最大时间间隔不能超过30天
        :param identity: 查询者自己在所要查询的采购申请/经销采购单中的身份。<br/>1：供应商。<br/>2：分销商。<br/>注：填写其他值当做错误处理。
        :param page_no: 页码（大于0的整数。无值或小于1的值按默认值1计）
        :param page_size: 每页条数（大于0但小于等于50的整数。无值或大于50或小于1的值按默认值50计）
        :param order_status: 采购申请/经销采购单状态。<br/>0：全部状态。<br/>1：分销商提交申请，待供应商审核。<br/>2：供应商驳回申请，待分销商确认。<br/>3：供应商修改后，待分销商确认。<br/>4：分销商拒绝修改，待供应商再审核。<br/>5：审核通过下单成功，待分销商付款。<br/>7：付款成功，待供应商发货。<br/>8：供应商发货，待分销商收货。<br/>9：分销商收货，交易成功。<br/>10：采购申请/经销采购单关闭。<br/><br/>注：无值按默认值0计，超出状态范围返回错误信息。
        :param fields: 多个字段用','分隔。 fields 如果为空：返回所有采购申请/经销采购单对象(dealer_orders)字段。 如果不为空：返回指定采购单对象(dealer_orders)字段。 例1： dealer_order_details.product_id 表示只返回product_id 例2： dealer_order_details 表示只返回明细列表
        """
        return self._top_request(
            "taobao.fenxiao.dealer.requisitionorder.get",
            {
                "start_date": start_date,
                "end_date": end_date,
                "identity": identity,
                "page_no": page_no,
                "page_size": page_size,
                "order_status": order_status,
                "fields": fields
            }
        )

    def taobao_fenxiao_dealer_requisitionorder_query(
            self,
            dealer_order_ids,
            fields=''
    ):
        """
        按编号查询采购申请/经销采购单
        按编号查询采购申请/经销采购单，目前支持供应商和分销商查询。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22033

        :param dealer_order_ids: 经销采购单编号。<br/>多个编号用英文符号的逗号隔开。最多支持50个经销采购单编号的查询。
        :param fields: 多个字段用','分隔。 fields 如果为空：返回所有经销采购单对象(dealer_orders)字段。 如果不为空：返回指定采购单对象(dealer_orders)字段。 例1： dealer_order_details.product_id 表示只返回product_id 例2： dealer_order_details 表示只返回明细列表
        """
        return self._top_request(
            "taobao.fenxiao.dealer.requisitionorder.query",
            {
                "dealer_order_ids": dealer_order_ids,
                "fields": fields
            },
            result_processor=lambda x: x["dealer_orders"]
        )

    def taobao_fenxiao_dealer_requisitionorder_remark_update(
            self,
            dealer_order_id,
            supplier_memo_flag,
            supplier_memo=''
    ):
        """
        修改经销采购单备注
        供应商修改经销采购单备注
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22522

        :param dealer_order_id: 经销采购单ID
        :param supplier_memo_flag: 旗子的标记，必选。<br/>1-5之间的数字。<br/>非1-5之间，都采用1作为默认。<br/>1:红色<br/>2:黄色<br/>3:绿色<br/>4:蓝色<br/>5:粉红色
        :param supplier_memo: 备注留言，可为空
        """
        return self._top_request(
            "taobao.fenxiao.dealer.requisitionorder.remark.update",
            {
                "dealer_order_id": dealer_order_id,
                "supplier_memo_flag": supplier_memo_flag,
                "supplier_memo": supplier_memo
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_fenxiao_dealer_requisitionorder_create(
            self,
            order_detail,
            province,
            address,
            buyer_name,
            logistics_type='',
            city='',
            district='',
            post_code='',
            phone='',
            mobile='',
            id_card_number=''
    ):
        """
        创建经销采购申请
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22852

        :param order_detail: 采购清单，存放多个采购明细，每个采购明细内部以‘:’隔开，多个采购明细之间以‘,’隔开. 例(分销产品id:skuid:购买数量:申请单价,分销产品id:skuid:购买数量:申请单价)，申请单价的单位为分。不存在sku请留空skuid，如（分销产品id::购买数量:申请单价）
        :param province: 收货人所在省份
        :param address: 收货人所在街道地址
        :param buyer_name: 买家姓名（自提方式填写提货人姓名）
        :param logistics_type: 配送方式。SELF_PICKUP：自提；LOGISTICS：仓库发货
        :param city: 收货人所在市
        :param district: 收货人所在区
        :param post_code: 收货人所在地区邮政编码
        :param phone: 买家联系电话（此字段和mobile字段至少填写一个）
        :param mobile: 买家的手机号码（1、此字段与phone字段至少填写一个。2、自提方式下此字段必填，保存提货人联系电话）
        :param id_card_number: 身份证号（自提方式必填，填写提货人身份证号码，提货时用于确认提货人身份）
        """
        return self._top_request(
            "taobao.fenxiao.dealer.requisitionorder.create",
            {
                "order_detail": order_detail,
                "province": province,
                "address": address,
                "buyer_name": buyer_name,
                "logistics_type": logistics_type,
                "city": city,
                "district": district,
                "post_code": post_code,
                "phone": phone,
                "mobile": mobile,
                "id_card_number": id_card_number
            },
            result_processor=lambda x: x["dealer_order_id"]
        )

    def taobao_fenxiao_refund_query(
            self,
            start_date,
            end_date,
            page_no='1',
            page_size='50',
            query_seller_refund=''
    ):
        """
        批量查询采购退款
        供应商按查询条件批量查询代销采购退款
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23000

        :param start_date: 代销采购退款单最早修改时间
        :param end_date: 代销采购退款最迟修改时间。与start_date的最大时间间隔不能超过30天
        :param page_no: 页码（大于0的整数。无值或小于1的值按默认值1计）
        :param page_size: 每页条数（大于0但小于等于50的整数。无值或大于50或小于1的值按默认值50计）
        :param query_seller_refund: 是否查询下游买家的退款信息
        """
        return self._top_request(
            "taobao.fenxiao.refund.query",
            {
                "start_date": start_date,
                "end_date": end_date,
                "page_no": page_no,
                "page_size": page_size,
                "query_seller_refund": query_seller_refund
            }
        )

    def taobao_fenxiao_product_quantity_update(
            self,
            product_id,
            quantity,
            type='',
            properties=''
    ):
        """
        产品库存修改
        修改产品库存信息，支持全量修改以及增量修改两种方式
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23616

        :param product_id: 产品ID
        :param quantity: 库存修改值。产品有sku时，与sku属性顺序对应，用,分隔。产品无sku时，只写库存值。
                         当全量更新库存时，quantity必须为大于等于0的正整数；当增量更新库存时，quantity为整数，可小于等于0。若增量更新时传入的库存为负数，则负数与实际库存之和不能小于0。比如当前实际库存为1，传入增量更新quantity=-1，库存改为0
        :param type: 库存更新方式，可选。1为全量更新，2为增量更新。如果不填，默认为全量更新。当选择全量更新时，如果库存更新值传入的是负数，会出错并返回错误码；当选择增量更新时，如果库存更新值为负数且绝对值大于当前库存，则sku库存会设置为0
        :param properties: sku属性值，产品有sku时填写，多个sku用,分隔。为空时默认该产品无sku，则只修改产品的库存。
        """
        return self._top_request(
            "taobao.fenxiao.product.quantity.update",
            {
                "product_id": product_id,
                "quantity": quantity,
                "type": type,
                "properties": properties
            }
        )

    def tmall_inventory_query_forstore(
            self,
            param_list
    ):
        """
        查询后端商品仓库库存
        商家查询后端商品仓库库存
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26481

        :param param_list: 查询列表
        """
        return self._top_request(
            "tmall.inventory.query.forstore",
            {
                "param_list": param_list
            }
        )

    def taobao_inventory_warehouse_query(
            self,
            page_no,
            page_size
    ):
        """
        分页查询商家仓信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27165

        :param page_no: 页码
        :param page_size: 页大小
        """
        return self._top_request(
            "taobao.inventory.warehouse.query",
            {
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_inventory_warehouse_get(
            self,
            store_code
    ):
        """
        获取商家仓信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27166

        :param store_code: 仓库编码
        """
        return self._top_request(
            "taobao.inventory.warehouse.get",
            {
                "store_code": store_code
            }
        )

    def taobao_inventory_warehouse_manage(
            self,
            ware_house_dto
    ):
        """
        创建商家仓或者更新商家仓信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27167

        :param ware_house_dto: 仓库信息
        """
        return self._top_request(
            "taobao.inventory.warehouse.manage",
            {
                "ware_house_dto": ware_house_dto
            }
        )

    def taobao_region_warehouse_manage(
            self,
            store_code,
            regions
    ):
        """
        编辑仓库覆盖范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27170

        :param store_code: 仓库编码
        :param regions: 可映射三级地址,例: 广东省
        """
        return self._top_request(
            "taobao.region.warehouse.manage",
            {
                "store_code": store_code,
                "regions": regions
            }
        )

    def taobao_region_sale_query(
            self,
            item_id,
            sku_id,
            sale_region_level
    ):
        """
        查询商品销售区域
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27171

        :param item_id: 商品id
        :param sku_id: 无sku传0
        :param sale_region_level: 1:国家;2:省;3: 市;4:区县
        """
        return self._top_request(
            "taobao.region.sale.query",
            {
                "item_id": item_id,
                "sku_id": sku_id,
                "sale_region_level": sale_region_level
            }
        )

    def taobao_region_warehouse_query(
            self,
            store_code
    ):
        """
        查询仓库覆盖范围
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27173

        :param store_code: 仓库编码
        """
        return self._top_request(
            "taobao.region.warehouse.query",
            {
                "store_code": store_code
            }
        )

    def taobao_region_price_manage(
            self,
            item_id,
            regional_price_dtos,
            sku_id='0',
            is_full='true'
    ):
        """
        编辑区域价格
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27253

        :param item_id: 商品id
        :param regional_price_dtos: 列表
        :param sku_id: 无sku传0
        :param is_full: true:全量, false:增量
        """
        return self._top_request(
            "taobao.region.price.manage",
            {
                "item_id": item_id,
                "regional_price_dtos": regional_price_dtos,
                "sku_id": sku_id,
                "is_full": is_full
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_region_price_query(
            self,
            item_id,
            sku_id='0',
            regional_price_dtos=None
    ):
        """
        区域价格查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27254

        :param item_id: 商品id
        :param sku_id: 无sku可传0
        :param regional_price_dtos: 不传则返回所有设置的区域价格
        """
        return self._top_request(
            "taobao.region.price.query",
            {
                "item_id": item_id,
                "sku_id": sku_id,
                "regional_price_dtos": regional_price_dtos
            }
        )

    def taobao_region_price_cancle(
            self,
            item_id,
            sku_id
    ):
        """
        取消区域价格
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27286

        :param item_id: 商品
        :param sku_id: 无sku传0
        """
        return self._top_request(
            "taobao.region.price.cancle",
            {
                "item_id": item_id,
                "sku_id": sku_id
            }
        )

    def taobao_fenxiao_product_import_from_auction(
            self,
            trade_type,
            auction_id,
            product_line_id
    ):
        """
        导入商品生成产品
        供应商选择关联店铺的前台宝贝，导入生成产品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32941

        :param trade_type: 导入产品需要支持的交易类型：[1 代销][ 2 经销 ][3 代销和经销]
        :param auction_id: 店铺宝贝id
        :param product_line_id: 产品线id
        """
        return self._top_request(
            "taobao.fenxiao.product.import.from.auction",
            {
                "trade_type": trade_type,
                "auction_id": auction_id,
                "product_line_id": product_line_id
            }
        )

    def taobao_fenxiao_product_to_channel_import(
            self,
            channel,
            product_id
    ):
        """
        产品导入到渠道
        支持供应商将已有产品导入到某个渠道销售
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33028

        :param channel: 要导入的渠道[21 零售PLUS]目前仅支持此渠道
        :param product_id: 要导入的产品id
        """
        return self._top_request(
            "taobao.fenxiao.product.to.channel.import",
            {
                "channel": channel,
                "product_id": product_id
            }
        )

    def tmall_channel_products_get(
            self,
            top_query_product_d_o=None
    ):
        """
        查询供应商的产品数据
        查询供应商的产品数据。
        * 入参传入pids将优先查询，即只按这个条件查询。
        *入参传入sku_number将优先查询(没有传入pids)，即只按这个条件查询(最多显示50条)
        * 入参fields传skus将查询sku的数据，不传该参数默认不查询，返回产品的其它信息。
        * 入参fields传入images将查询多图数据，不传只返回主图数据。
        * 入参fields仅对传入pids生效（只有按ID查询时，才能查询额外的数据）
        * 查询结果按照产品发布时间倒序，即时间近的数据在前。
        * 传入channel 渠道，会只返回相应渠道的产品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33036

        :param top_query_product_d_o: top_query_product_d_o
        """
        return self._top_request(
            "tmall.channel.products.get",
            {
                "top_query_product_d_o": top_query_product_d_o
            }
        )

    def taobao_fenxiao_trade_prepay_offline_add(
            self,
            offline_add_prepay_param
    ):
        """
        线下预存款流水增加
        渠道分销供应商上传线下流水预存款（增加）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33474

        :param offline_add_prepay_param: 增加流水
        """
        return self._top_request(
            "taobao.fenxiao.trade.prepay.offline.add",
            {
                "offline_add_prepay_param": offline_add_prepay_param
            },
            result_processor=lambda x: x["success"]
        )

    def taobao_fenxiao_trade_prepay_offline_reduce(
            self,
            offline_reduce_prepay_param
    ):
        """
        渠道分销供应商上传线下流水预存款（减少）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33475

        :param offline_reduce_prepay_param: 减少流水
        """
        return self._top_request(
            "taobao.fenxiao.trade.prepay.offline.reduce",
            {
                "offline_reduce_prepay_param": offline_reduce_prepay_param
            },
            result_processor=lambda x: x["success"]
        )


class TbWangWang(DingTalkBaseAPI):
    """
    旺旺API
    """

    def taobao_wangwang_eservice_chatpeers_get(
            self
    ):
        """
        获取聊天对象列表
        获取聊天对象列表，只能获取最近1个月内的数据且查询时间段<=7天,只支持xml返回 。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=330

        """
        return self._top_request(
            "taobao.wangwang.eservice.chatpeers.get"
        )

    def taobao_wangwang_abstract_logquery(
            self,
            to_id,
            from_id,
            start_date,
            end_date,
            count='1000',
            next_key='',
            charset=''
    ):
        """
        聊天记录查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21804

        :param to_id: 买家id，有cntaobao前缀
        :param from_id: 卖家id，有cntaobao前缀
        :param start_date: utc
        :param end_date: utc
        :param count: 获取记录条数，默认值是1000
        :param next_key: 设置了这个值，那么聊天记录就从这个点开始查询
        :param charset: 传入参数的字符集
        """
        return self._top_request(
            "taobao.wangwang.abstract.logquery",
            {
                "to_id": to_id,
                "from_id": from_id,
                "start_date": start_date,
                "end_date": end_date,
                "count": count,
                "next_key": next_key,
                "charset": charset
            }
        )

    def taobao_wangwang_abstract_getwordlist(
            self,
            charset=''
    ):
        """
        获取关键词列表
        获取关键词列表，只支持json返回
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21805

        :param charset: 传入参数的字符集
        """
        return self._top_request(
            "taobao.wangwang.abstract.getwordlist",
            {
                "charset": charset
            }
        )

    def taobao_wangwang_abstract_deleteword(
            self,
            word,
            charset=''
    ):
        """
        删除关键词
        删除关键词，只支持json返回
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21806

        :param word: 关键词，长度大于0
        :param charset: 传入参数的字符集
        """
        return self._top_request(
            "taobao.wangwang.abstract.deleteword",
            {
                "word": word,
                "charset": charset
            }
        )

    def taobao_wangwang_abstract_addword(
            self,
            word,
            charset=''
    ):
        """
        增加关键词
        增加关键词，只支持json返回
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21807

        :param word: 关键词，长度大于0
        :param charset: 传入参数的字符集
        """
        return self._top_request(
            "taobao.wangwang.abstract.addword",
            {
                "word": word,
                "charset": charset
            }
        )

    def taobao_wangwang_abstract_initialize(
            self,
            charset=''
    ):
        """
        模糊查询服务初始化
        模糊查询服务初始化，只支持json返回
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21808

        :param charset: 传入参数的字符集
        """
        return self._top_request(
            "taobao.wangwang.abstract.initialize",
            {
                "charset": charset
            }
        )

    def taobao_wangwang_clientid_bind(
            self,
            app_name,
            client_id
    ):
        """
        nick和客户端id绑定
        绑定nick和客户端id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24613

        :param app_name: 应用名
        :param client_id: 客户端Id
        """
        return self._top_request(
            "taobao.wangwang.clientid.bind",
            {
                "app_name": app_name,
                "client_id": client_id
            }
        )

    def taobao_wangwang_clientid_unbind(
            self,
            app_name,
            client_id
    ):
        """
        解除账号和客户端Id的绑定
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24614

        :param app_name: 应用名
        :param client_id: 客户端Id
        """
        return self._top_request(
            "taobao.wangwang.clientid.unbind",
            {
                "app_name": app_name,
                "client_id": client_id
            }
        )

    def taobao_qianniu_kefueval_get(
            self,
            query_ids,
            btime,
            etime
    ):
        """
        客服评价详情接口
        获取买家对客服的服务评价
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27589

        :param query_ids: 客服的nick，多个用逗号分隔，不要超过10个，带cntaobao的长nick
        :param btime: 开始时间，格式yyyyMMddHHmmss
        :param etime: 结束时间,格式yyyyMMddHHmmss
        """
        return self._top_request(
            "taobao.qianniu.kefueval.get",
            {
                "query_ids": query_ids,
                "btime": btime,
                "etime": etime
            }
        )

    def taobao_wangwang_eservice_chatrelation_get(
            self,
            chat_relation_request
    ):
        """
        聊天关系获取接口
        获取指定时间区间内的聊天关系（查询账号，和谁，在哪天说过话）。如：
        A 和 B 在2016-09-01 和 2016-09-02 都说过话。以A为查询账号，则该接口将返回：
        2016-09-01， B
        2016-09-02， B
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27954

        :param chat_relation_request: 请求参数
        """
        return self._top_request(
            "taobao.wangwang.eservice.chatrelation.get",
            {
                "chat_relation_request": chat_relation_request
            }
        )

    def taobao_wangwang_eservice_abs_word_get(
            self
    ):
        """
        获取关键词
        获取关键词列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28015

        """
        return self._top_request(
            "taobao.wangwang.eservice.abs.word.get",
            result_processor=lambda x: x["words"]
        )

    def taobao_wangwang_eservice_abs_word_del(
            self,
            word
    ):
        """
        删除关键词接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28016

        :param word: 要删除的关键字
        """
        return self._top_request(
            "taobao.wangwang.eservice.abs.word.del",
            {
                "word": word
            }
        )

    def taobao_wangwang_eservice_abs_word_add(
            self,
            word
    ):
        """
        添加关键词
        为聊天记录查询接口添加关键词。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28017

        :param word: 关键词
        """
        return self._top_request(
            "taobao.wangwang.eservice.abs.word.add",
            {
                "word": word
            }
        )

    def taobao_wangwang_eservice_abs_chatlog_get(
            self,
            query_info
    ):
        """
        商家聊天记录查询接口
        查询授权商家的聊天记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28019

        :param query_info: 查询参数
        """
        return self._top_request(
            "taobao.wangwang.eservice.abs.chatlog.get",
            {
                "query_info": query_info
            }
        )


class TbTaoBaoKe(DingTalkBaseAPI):
    """
    淘宝客API
    """

    def taobao_tbk_item_get(
            self,
            fields,
            q='',
            cat='',
            itemloc='',
            sort='',
            is_tmall='',
            is_overseas='',
            start_price='',
            end_price='',
            start_tk_rate='',
            end_tk_rate='',
            platform='',
            page_no='0',
            page_size='0'
    ):
        """
        淘宝客商品查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24515

        :param fields: 需返回的字段列表
        :param q: 查询词
        :param cat: 后台类目ID，用,分割，最大10个，该ID可以通过taobao.itemcats.get接口获取到
        :param itemloc: 所在地
        :param sort: 排序_des（降序），排序_asc（升序），销量（total_sales），淘客佣金比率（tk_rate）， 累计推广量（tk_total_sales），总支出佣金（tk_total_commi）
        :param is_tmall: 是否商城商品，设置为true表示该商品是属于淘宝商城商品，设置为false或不设置表示不判断这个属性
        :param is_overseas: 是否海外商品，设置为true表示该商品是属于海外商品，设置为false或不设置表示不判断这个属性
        :param start_price: 折扣价范围下限，单位：元
        :param end_price: 折扣价范围上限，单位：元
        :param start_tk_rate: 淘客佣金比率上限，如：1234表示12.34%
        :param end_tk_rate: 淘客佣金比率下限，如：1234表示12.34%
        :param platform: 链接形式：1：PC，2：无线，默认：１
        :param page_no: 第几页，默认：１
        :param page_size: 页大小，默认20，1~100
        """
        return self._top_request(
            "taobao.tbk.item.get",
            {
                "fields": fields,
                "q": q,
                "cat": cat,
                "itemloc": itemloc,
                "sort": sort,
                "is_tmall": is_tmall,
                "is_overseas": is_overseas,
                "start_price": start_price,
                "end_price": end_price,
                "start_tk_rate": start_tk_rate,
                "end_tk_rate": end_tk_rate,
                "platform": platform,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_tbk_item_recommend_get(
            self,
            fields,
            num_iid,
            count='20',
            platform='1'
    ):
        """
        淘宝客商品关联推荐查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24517

        :param fields: 需返回的字段列表
        :param num_iid: 商品Id
        :param count: 返回数量，默认20，最大值40
        :param platform: 链接形式：1：PC，2：无线，默认：１
        """
        return self._top_request(
            "taobao.tbk.item.recommend.get",
            {
                "fields": fields,
                "num_iid": num_iid,
                "count": count,
                "platform": platform
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_item_info_get(
            self,
            num_iids,
            platform='1',
            ip=''
    ):
        """
        淘宝客商品详情（简版）
        淘宝客商品详情查询（简版）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24518

        :param num_iids: 商品ID串，用,分割，最大40个
        :param platform: 链接形式：1：PC，2：无线，默认：１
        :param ip: ip地址，影响邮费获取，如果不传或者传入不准确，邮费无法精准提供
        """
        return self._top_request(
            "taobao.tbk.item.info.get",
            {
                "num_iids": num_iids,
                "platform": platform,
                "ip": ip
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_shop_get(
            self,
            fields,
            q,
            sort='',
            is_tmall='',
            start_credit='',
            end_credit='',
            start_commission_rate='',
            end_commission_rate='',
            start_total_action='',
            end_total_action='',
            start_auction_count='',
            end_auction_count='',
            platform='1',
            page_no='1',
            page_size='20'
    ):
        """
        淘宝客店铺查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24521

        :param fields: 需返回的字段列表
        :param q: 查询词
        :param sort: 排序_des（降序），排序_asc（升序），佣金比率（commission_rate）， 商品数量（auction_count），销售总数量（total_auction）
        :param is_tmall: 是否商城的店铺，设置为true表示该是属于淘宝商城的店铺，设置为false或不设置表示不判断这个属性
        :param start_credit: 信用等级下限，1~20
        :param end_credit: 信用等级上限，1~20
        :param start_commission_rate: 淘客佣金比率下限，1~10000
        :param end_commission_rate: 淘客佣金比率上限，1~10000
        :param start_total_action: 店铺商品总数下限
        :param end_total_action: 店铺商品总数上限
        :param start_auction_count: 累计推广商品下限
        :param end_auction_count: 累计推广商品上限
        :param platform: 链接形式：1：PC，2：无线，默认：１
        :param page_no: 第几页，默认1，1~100
        :param page_size: 页大小，默认20，1~100
        """
        return self._top_request(
            "taobao.tbk.shop.get",
            {
                "fields": fields,
                "q": q,
                "sort": sort,
                "is_tmall": is_tmall,
                "start_credit": start_credit,
                "end_credit": end_credit,
                "start_commission_rate": start_commission_rate,
                "end_commission_rate": end_commission_rate,
                "start_total_action": start_total_action,
                "end_total_action": end_total_action,
                "start_auction_count": start_auction_count,
                "end_auction_count": end_auction_count,
                "platform": platform,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_tbk_shop_recommend_get(
            self,
            fields,
            user_id,
            count='',
            platform=''
    ):
        """
        淘宝客店铺关联推荐查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24522

        :param fields: 需返回的字段列表
        :param user_id: 卖家Id
        :param count: 返回数量，默认20，最大值40
        :param platform: 链接形式：1：PC，2：无线，默认：１
        """
        return self._top_request(
            "taobao.tbk.shop.recommend.get",
            {
                "fields": fields,
                "user_id": user_id,
                "count": count,
                "platform": platform
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_uatm_favorites_item_get(
            self,
            adzone_id,
            favorites_id,
            fields,
            platform='1',
            page_size='20',
            unid='',
            page_no='1'
    ):
        """
        获取淘宝联盟选品库的宝贝信息
        指定选品库id，获取该选品库的宝贝信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26619

        :param adzone_id: 推广位id，需要在淘宝联盟后台创建；且属于appkey备案的媒体id（siteid），如何获取adzoneid，请参考，http://club.alimama.com/read-htm-tid-6333967.html?spm=0.0.0.0.msZnx5
        :param favorites_id: 选品库的id
        :param fields: 需要输出则字段列表，逗号分隔
        :param platform: 链接形式：1：PC，2：无线，默认：１
        :param page_size: 页大小，默认20，1~100
        :param unid: 自定义输入串，英文和数字组成，长度不能大于12个字符，区分不同的推广渠道
        :param page_no: 第几页，默认：1，从1开始计数
        """
        return self._top_request(
            "taobao.tbk.uatm.favorites.item.get",
            {
                "adzone_id": adzone_id,
                "favorites_id": favorites_id,
                "fields": fields,
                "platform": platform,
                "page_size": page_size,
                "unid": unid,
                "page_no": page_no
            }
        )

    def taobao_tbk_uatm_favorites_get(
            self,
            fields,
            page_no='1',
            page_size='20',
            type='-1'
    ):
        """
        获取淘宝联盟选品库列表
        枚举出淘宝客在淘宝联盟超级搜索，特色频道中，创建的选品库列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26620

        :param fields: 需要返回的字段列表，不能为空，字段名之间使用逗号分隔
        :param page_no: 第几页，从1开始计数
        :param page_size: 默认20，页大小，即每一页的活动个数
        :param type: 默认值-1；选品库类型，1：普通选品组，2：高佣选品组，-1，同时输出所有类型的选品组
        """
        return self._top_request(
            "taobao.tbk.uatm.favorites.get",
            {
                "fields": fields,
                "page_no": page_no,
                "page_size": page_size,
                "type": type
            }
        )

    def taobao_tbk_ju_tqg_get(
            self,
            adzone_id,
            fields,
            start_time,
            end_time,
            page_no='1',
            page_size='40'
    ):
        """
        淘抢购api
        获取淘抢购的数据，淘客商品转淘客链接，非淘客商品输出普通链接
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27543

        :param adzone_id: 推广位id（推广位申请方式：http://club.alimama.com/read.php?spm=0.0.0.0.npQdST&tid=6306396&ds=1&page=1&toread=1）
        :param fields: 需返回的字段列表
        :param start_time: 最早开团时间
        :param end_time: 最晚开团时间
        :param page_no: 第几页，默认1，1~100
        :param page_size: 页大小，默认40，1~40
        """
        return self._top_request(
            "taobao.tbk.ju.tqg.get",
            {
                "adzone_id": adzone_id,
                "fields": fields,
                "start_time": start_time,
                "end_time": end_time,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_tbk_item_click_extract(
            self,
            click_url
    ):
        """
        链接解析api
        从长链接或短链接中解析出open_iid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28156

        :param click_url: 长链接或短链接
        """
        return self._top_request(
            "taobao.tbk.item.click.extract",
            {
                "click_url": click_url
            }
        )

    def taobao_tbk_item_guess_like(
            self,
            adzone_id,
            os,
            ip,
            ua,
            net,
            user_nick='',
            user_id='',
            idfa='',
            imei='',
            imei_md5='',
            apnm='',
            mn='',
            page_no='1',
            page_size='20'
    ):
        """
        淘宝客商品猜你喜欢
        淘宝客猜你喜欢商品API
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29528

        :param adzone_id: 广告位ID
        :param os: 系统类型，ios, android, other
        :param ip: 客户端ip
        :param ua: userAgent
        :param net: 联网方式，wifi, cell, unknown
        :param user_nick: 用户昵称，from cookie : _nk_或者tracknick ; from百川sdk : nick
        :param user_id: 用户数字ID，from cookie : unb
        :param idfa: ios广告跟踪id
        :param imei: android设备imei
        :param imei_md5: android设备imeiMD5值，32位小写
        :param apnm: 应用包名
        :param mn: 设备型号
        :param page_no: 第几页
        :param page_size: 页大小
        """
        return self._top_request(
            "taobao.tbk.item.guess.like",
            {
                "adzone_id": adzone_id,
                "os": os,
                "ip": ip,
                "ua": ua,
                "net": net,
                "user_nick": user_nick,
                "user_id": user_id,
                "idfa": idfa,
                "imei": imei,
                "imei_md5": imei_md5,
                "apnm": apnm,
                "mn": mn,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_dg_item_coupon_get(
            self,
            adzone_id,
            platform='1',
            cat='',
            page_size='20',
            q='',
            page_no='1'
    ):
        """
        好券清单API【导购】
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29821

        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param platform: 1：PC，2：无线，默认：1
        :param cat: 后台类目ID，用,分割，最大10个，该ID可以通过taobao.itemcats.get接口获取到
        :param page_size: 页大小，默认20，1~100
        :param q: 查询词
        :param page_no: 第几页，默认：1（当后台类目和查询词均不指定的时候，最多出10000个结果，即page_no*page_size不能超过10000；当指定类目或关键词的时候，则最多出100个结果）
        """
        return self._top_request(
            "taobao.tbk.dg.item.coupon.get",
            {
                "adzone_id": adzone_id,
                "platform": platform,
                "cat": cat,
                "page_size": page_size,
                "q": q,
                "page_no": page_no
            }
        )

    def taobao_tbk_coupon_get(
            self,
            me='',
            item_id='',
            activity_id=''
    ):
        """
        阿里妈妈推广券信息查询
        阿里妈妈推广券信息查询。传入商品ID+券ID，或者传入me参数，均可查询券信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31106

        :param me: 带券ID与商品ID的加密串
        :param item_id: 商品ID
        :param activity_id: 券ID
        """
        return self._top_request(
            "taobao.tbk.coupon.get",
            {
                "me": me,
                "item_id": item_id,
                "activity_id": activity_id
            },
            result_processor=lambda x: x["data"]
        )

    def taobao_tbk_tpwd_create(
            self,
            text,
            url,
            user_id='',
            logo='',
            ext=''
    ):
        """
        淘宝客淘口令
        提供淘客生成淘口令接口，淘客提交口令内容、logo、url等参数，生成淘口令关键key如：￥SADadW￥，后续进行文案包装组装用于传播
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31127

        :param text: 口令弹框内容
        :param url: 口令跳转目标页
        :param user_id: 生成口令的淘宝用户ID
        :param logo: 口令弹框logoURL
        :param ext: 扩展字段JSON格式
        """
        return self._top_request(
            "taobao.tbk.tpwd.create",
            {
                "text": text,
                "url": url,
                "user_id": user_id,
                "logo": logo,
                "ext": ext
            },
            result_processor=lambda x: x["data"]
        )

    def taobao_tbk_dg_newuser_order_get(
            self,
            activity_id,
            page_size='20',
            adzone_id='',
            page_no='1',
            start_time='',
            end_time=''
    ):
        """
        淘宝客新用户订单API--导购
        拉新API
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33892

        :param activity_id: 活动id， 活动名称与活动ID列表，请参见https://tbk.bbs.taobao.com/detail.html?appId=45301&postId=8599277
        :param page_size: 页大小，默认20，1~100
        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param page_no: 页码，默认1
        :param start_time: 开始时间，当活动为淘宝活动，表示最早注册时间；当活动为支付宝活动，表示最早绑定时间；当活动为天猫活动，表示最早领取红包时间
        :param end_time: 结束时间，当活动为淘宝活动，表示最晚结束时间；当活动为支付宝活动，表示最晚绑定时间；当活动为天猫活动，表示最晚领取红包的时间
        """
        return self._top_request(
            "taobao.tbk.dg.newuser.order.get",
            {
                "activity_id": activity_id,
                "page_size": page_size,
                "adzone_id": adzone_id,
                "page_no": page_no,
                "start_time": start_time,
                "end_time": end_time
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_sc_newuser_order_get(
            self,
            activity_id,
            page_size='20',
            adzone_id='',
            page_no='1',
            site_id='',
            end_time='',
            start_time=''
    ):
        """
        淘宝客新用户订单API--社交
        拉新API
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33897

        :param activity_id: 活动id， 现有活动id包括： 2月手淘拉新：119013_2 3月手淘拉新：119013_3 4月手淘拉新：119013_4 拉手机支付宝新用户_赚赏金：200000_5
        :param page_size: 页大小，默认20，1~100
        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param page_no: 页码，默认1
        :param site_id: mm_xxx_xxx_xxx的第二位
        :param end_time: 结束时间，当活动为淘宝活动，表示最晚结束时间；当活动为支付宝活动，表示最晚绑定时间；当活动为天猫活动，表示最晚领取红包的时间
        :param start_time: 开始时间，当活动为淘宝活动，表示最早注册时间；当活动为支付宝活动，表示最早绑定时间；当活动为天猫活动，表示最早领取红包时间
        """
        return self._top_request(
            "taobao.tbk.sc.newuser.order.get",
            {
                "activity_id": activity_id,
                "page_size": page_size,
                "adzone_id": adzone_id,
                "page_no": page_no,
                "site_id": site_id,
                "end_time": end_time,
                "start_time": start_time
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_dg_optimus_material(
            self,
            adzone_id,
            page_size='20',
            page_no='1',
            material_id='0',
            device_value='',
            device_encrypt='',
            device_type='',
            content_id='',
            content_source='',
            item_id=''
    ):
        """
        淘宝客物料下行-导购
        通用物料推荐，传入官方公布的物料id，可获取指定物料
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33947

        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param page_size: 页大小，默认20，1~100
        :param page_no: 第几页，默认：1
        :param material_id: 官方的物料Id(详细物料id见：https://tbk.bbs.taobao.com/detail.html?appId=45301&postId=8576096)
        :param device_value: 智能匹配-设备号加密后的值（MD5加密需32位小写）
        :param device_encrypt: 智能匹配-设备号加密类型：MD5
        :param device_type: 智能匹配-设备号类型：IMEI，或者IDFA，或者UTDID（UTDID不支持MD5加密）
        :param content_id: 内容专用-内容详情ID
        :param content_source: 内容专用-内容渠道信息
        :param item_id: 商品ID，用于相似商品推荐
        """
        return self._top_request(
            "taobao.tbk.dg.optimus.material",
            {
                "adzone_id": adzone_id,
                "page_size": page_size,
                "page_no": page_no,
                "material_id": material_id,
                "device_value": device_value,
                "device_encrypt": device_encrypt,
                "device_type": device_type,
                "content_id": content_id,
                "content_source": content_source,
                "item_id": item_id
            }
        )

    def taobao_tbk_dg_material_optional(
            self,
            adzone_id,
            start_dsr='',
            page_size='20',
            page_no='1',
            platform='1',
            end_tk_rate='',
            start_tk_rate='',
            end_price='',
            start_price='',
            is_overseas='',
            is_tmall='',
            sort='',
            itemloc='',
            cat='',
            q='',
            material_id='2836',
            has_coupon='',
            ip='',
            need_free_shipment='',
            need_prepay='',
            include_pay_rate_30='',
            include_good_rate='',
            include_rfd_rate='',
            npx_level='1',
            end_ka_tk_rate='',
            start_ka_tk_rate='',
            device_encrypt='',
            device_value='',
            device_type=''
    ):
        """
        通用物料搜索API（导购）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=35896

        :param adzone_id: mm_xxx_xxx_12345678三段式的最后一段数字
        :param start_dsr: 商品筛选(特定媒体支持)-店铺dsr评分。筛选大于等于当前设置的店铺dsr评分的商品0-50000之间
        :param page_size: 页大小，默认20，1~100
        :param page_no: 第几页，默认：１
        :param platform: 链接形式：1：PC，2：无线，默认：１
        :param end_tk_rate: 商品筛选-淘客佣金比率上限。如：1234表示12.34%
        :param start_tk_rate: 商品筛选-淘客佣金比率下限。如：1234表示12.34%
        :param end_price: 商品筛选-折扣价范围上限。单位：元
        :param start_price: 商品筛选-折扣价范围下限。单位：元
        :param is_overseas: 商品筛选-是否海外商品。true表示属于海外商品，false或不设置表示不限
        :param is_tmall: 商品筛选-是否天猫商品。true表示属于天猫商品，false或不设置表示不限
        :param sort: 排序_des（降序），排序_asc（升序），销量（total_sales），淘客佣金比率（tk_rate）， 累计推广量（tk_total_sales），总支出佣金（tk_total_commi），价格（price）
        :param itemloc: 商品筛选-所在地
        :param cat: 商品筛选-后台类目ID。用,分割，最大10个，该ID可以通过taobao.itemcats.get接口获取到
        :param q: 商品筛选-查询词
        :param material_id: 官方的物料Id(详细物料id见：https://tbk.bbs.taobao.com/detail.html?appId=45301&postId=8576096)，不传时默认为2836
        :param has_coupon: 优惠券筛选-是否有优惠券。true表示该商品有优惠券，false或不设置表示不限
        :param ip: ip参数影响邮费获取，如果不传或者传入不准确，邮费无法精准提供
        :param need_free_shipment: 商品筛选-是否包邮。true表示包邮，false或不设置表示不限
        :param need_prepay: 商品筛选-是否加入消费者保障。true表示加入，false或不设置表示不限
        :param include_pay_rate_30: 商品筛选(特定媒体支持)-成交转化是否高于行业均值。True表示大于等于，false或不设置表示不限
        :param include_good_rate: 商品筛选-好评率是否高于行业均值。True表示大于等于，false或不设置表示不限
        :param include_rfd_rate: 商品筛选(特定媒体支持)-退款率是否低于行业均值。True表示大于等于，false或不设置表示不限
        :param npx_level: 商品筛选-牛皮癣程度。取值：1不限，2无，3轻微
        :param end_ka_tk_rate: 商品筛选-KA媒体淘客佣金比率上限。如：1234表示12.34%
        :param start_ka_tk_rate: 商品筛选-KA媒体淘客佣金比率下限。如：1234表示12.34%
        :param device_encrypt: 智能匹配-设备号加密类型：MD5
        :param device_value: 智能匹配-设备号加密后的值（MD5加密需32位小写）
        :param device_type: 智能匹配-设备号类型：IMEI，或者IDFA，或者UTDID（UTDID不支持MD5加密）
        """
        return self._top_request(
            "taobao.tbk.dg.material.optional",
            {
                "adzone_id": adzone_id,
                "start_dsr": start_dsr,
                "page_size": page_size,
                "page_no": page_no,
                "platform": platform,
                "end_tk_rate": end_tk_rate,
                "start_tk_rate": start_tk_rate,
                "end_price": end_price,
                "start_price": start_price,
                "is_overseas": is_overseas,
                "is_tmall": is_tmall,
                "sort": sort,
                "itemloc": itemloc,
                "cat": cat,
                "q": q,
                "material_id": material_id,
                "has_coupon": has_coupon,
                "ip": ip,
                "need_free_shipment": need_free_shipment,
                "need_prepay": need_prepay,
                "include_pay_rate_30": include_pay_rate_30,
                "include_good_rate": include_good_rate,
                "include_rfd_rate": include_rfd_rate,
                "npx_level": npx_level,
                "end_ka_tk_rate": end_ka_tk_rate,
                "start_ka_tk_rate": start_ka_tk_rate,
                "device_encrypt": device_encrypt,
                "device_value": device_value,
                "device_type": device_type
            }
        )

    def taobao_tbk_dg_newuser_order_sum(
            self,
            page_size,
            page_no,
            activity_id,
            adzone_id='',
            site_id='',
            settle_month=''
    ):
        """
        拉新活动汇总API--导购
        拉新活动汇总API
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36836

        :param page_size: 页大小，默认20，1~100
        :param page_no: 页码，默认1
        :param activity_id: 活动id， 活动名称与活动ID列表，请参见https://tbk.bbs.taobao.com/detail.html?appId=45301&postId=8599277
        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param site_id: mm_xxx_xxx_xxx的第二位
        :param settle_month: 结算月份
        """
        return self._top_request(
            "taobao.tbk.dg.newuser.order.sum",
            {
                "page_size": page_size,
                "page_no": page_no,
                "activity_id": activity_id,
                "adzone_id": adzone_id,
                "site_id": site_id,
                "settle_month": settle_month
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_sc_newuser_order_sum(
            self,
            page_size,
            page_no,
            activity_id,
            adzone_id='',
            site_id='',
            settle_month=''
    ):
        """
        拉新活动汇总API--社交
        拉新活动汇总API
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36837

        :param page_size: 页大小，默认20，1~100
        :param page_no: 页码，默认1
        :param activity_id: 活动ID，现有活动id包括： 2月手淘拉新：119013_2；3月手淘拉新：119013_3；
        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param site_id: mm_xxx_xxx_xxx的第二位
        :param settle_month: 结算月份，需按照活动的结算月份传入具体的值：201807
        """
        return self._top_request(
            "taobao.tbk.sc.newuser.order.sum",
            {
                "page_size": page_size,
                "page_no": page_no,
                "activity_id": activity_id,
                "adzone_id": adzone_id,
                "site_id": site_id,
                "settle_month": settle_month
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_tbk_sc_optimus_material(
            self,
            adzone_id,
            site_id,
            page_size='20',
            page_no='1',
            material_id='0',
            device_type='',
            device_encrypt='',
            device_value='',
            content_id='',
            content_source='',
            item_id=''
    ):
        """
        淘宝客擎天柱通用物料API - 社交
        通用物料推荐，传入官方公布的物料id，可获取指定物料
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37884

        :param adzone_id: mm_xxx_xxx_xxx的第三位
        :param site_id: mm_xxx_xxx_xxx的第二位
        :param page_size: 页大小，默认20，1~100
        :param page_no: 第几页，默认：1
        :param material_id: 官方的物料Id(详细物料id见：https://tbk.bbs.taobao.com/detail.html?appId=45301&postId=8576096)
        :param device_type: 智能匹配-设备号类型：IMEI，或者IDFA，或者UTDID（UTDID不支持MD5加密）
        :param device_encrypt: 智能匹配-设备号加密类型：MD5
        :param device_value: 智能匹配-设备号加密后的值（MD5加密需32位小写）
        :param content_id: 内容专用-内容详情ID
        :param content_source: 内容专用-内容渠道信息
        :param item_id: 商品ID，用于相似商品推荐
        """
        return self._top_request(
            "taobao.tbk.sc.optimus.material",
            {
                "adzone_id": adzone_id,
                "site_id": site_id,
                "page_size": page_size,
                "page_no": page_no,
                "material_id": material_id,
                "device_type": device_type,
                "device_encrypt": device_encrypt,
                "device_value": device_value,
                "content_id": content_id,
                "content_source": content_source,
                "item_id": item_id
            }
        )

    def taobao_tbk_activitylink_get(
            self,
            adzone_id,
            promotion_scene_id,
            platform='1',
            union_id='',
            sub_pid='',
            relation_id=''
    ):
        """
        淘宝联盟官方活动推广API-媒体
        淘宝联盟官方活动推广API-媒体，从官方活动列表页取出“官方活动ID”，支持二方、三方
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41918

        :param adzone_id: 推广位id，mm_xx_xx_xx pid三段式中的第三段。adzone_id需属于appKey拥有者
        :param promotion_scene_id: 官方活动ID，从官方活动页获取
        :param platform: 1：PC，2：无线，默认：１
        :param union_id: 自定义输入串，英文和数字组成，长度不能大于12个字符，区分不同的推广渠道
        :param sub_pid: 媒体平台下达人的淘客pid
        :param relation_id: 渠道关系ID，仅适用于渠道推广场景
        """
        return self._top_request(
            "taobao.tbk.activitylink.get",
            {
                "adzone_id": adzone_id,
                "promotion_scene_id": promotion_scene_id,
                "platform": platform,
                "union_id": union_id,
                "sub_pid": sub_pid,
                "relation_id": relation_id
            }
        )

    def taobao_tbk_sc_activitylink_toolget(
            self,
            adzone_id,
            site_id,
            promotion_scene_id,
            platform='1',
            union_id='',
            relation_id=''
    ):
        """
        淘宝联盟官方活动推广API-工具
        淘宝联盟官方活动推广API-工具，从官方活动列表页取出“官方活动ID”进行转链，支持二方、三方
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41921

        :param adzone_id: 推广位id，mm_xx_xx_xx pid三段式中的第三段
        :param site_id: 推广位id，mm_xx_xx_xx pid三段式中的第二段
        :param promotion_scene_id: 官方活动ID，从官方活动页获取
        :param platform: 1：PC，2：无线，默认：１
        :param union_id: 自定义输入串，英文和数字组成，长度不能大于12个字符，区分不同的推广渠道
        :param relation_id: 渠道关系ID，仅适用于渠道推广场景
        """
        return self._top_request(
            "taobao.tbk.sc.activitylink.toolget",
            {
                "adzone_id": adzone_id,
                "site_id": site_id,
                "promotion_scene_id": promotion_scene_id,
                "platform": platform,
                "union_id": union_id,
                "relation_id": relation_id
            }
        )

    def taobao_tbk_dg_punish_order_get(
            self,
            af_order_option=None
    ):
        """
        处罚订单查询 -导购-私域用户管理专用
        新增处罚订单查询API，提供媒体调用查询能力。这个是给媒体自己用的
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42050

        :param af_order_option: 入参的对象
        """
        return self._top_request(
            "taobao.tbk.dg.punish.order.get",
            {
                "af_order_option": af_order_option
            }
        )

    def taobao_tbk_content_get(
            self,
            adzone_id,
            type='',
            before_timestamp='',
            count='10',
            cid='',
            image_width='300',
            image_height='300',
            content_set=''
    ):
        """
        淘客媒体内容输出
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31137

        :param adzone_id: 推广位
        :param type: 内容类型，1:图文、2: 图集、3: 短视频
        :param before_timestamp: 表示取这个时间点以前的数据，以毫秒为单位（出参中的last_timestamp是指本次返回的内容中最早一条的数据，下一次作为before_timestamp传过来，即可实现翻页）
        :param count: 表示期望获取条数
        :param cid: 类目
        :param image_width: 图片宽度
        :param image_height: 图片高度
        :param content_set: 默认可不传,内容库类型:1 优质内容
        """
        return self._top_request(
            "taobao.tbk.content.get",
            {
                "adzone_id": adzone_id,
                "type": type,
                "before_timestamp": before_timestamp,
                "count": count,
                "cid": cid,
                "image_width": image_width,
                "image_height": image_height,
                "content_set": content_set
            },
            result_processor=lambda x: x["data"]
        )

    def taobao_tbk_content_effect_get(
            self,
            option=None
    ):
        """
        淘客媒体内容效果输出
        淘客媒体内容维度的效果输出。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37130

        :param option: 入参
        """
        return self._top_request(
            "taobao.tbk.content.effect.get",
            {
                "option": option
            }
        )

    def taobao_tbk_dg_vegas_tlj_create(
            self,
            adzone_id,
            item_id,
            total_num,
            name,
            user_total_win_num_limit,
            security_switch,
            per_face,
            send_start_time,
            campaign_type='MKT',
            send_end_time='',
            use_end_time='',
            use_end_time_mode='',
            use_start_time=''
    ):
        """
        淘礼金创建
        创建淘礼金
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40173

        :param adzone_id: 妈妈广告位Id
        :param item_id: 宝贝id
        :param total_num: 淘礼金总个数
        :param name: 淘礼金名称，最大10个字符
        :param user_total_win_num_limit: 单用户累计中奖次数上限
        :param security_switch: 安全开关
        :param per_face: 单个淘礼金面额，支持两位小数，单位元
        :param send_start_time: 发放开始时间
        :param campaign_type: CPS佣金计划类型
        :param send_end_time: 发放截止时间
        :param use_end_time: 使用结束日期。如果是结束时间模式为相对时间，时间格式为1-7直接的整数, 例如，1（相对领取时间1天）； 如果是绝对时间，格式为yyyy-MM-dd，例如，2019-01-29，表示到2019-01-29 23:59:59结束
        :param use_end_time_mode: 结束日期的模式,1:相对时间，2:绝对时间
        :param use_start_time: 使用开始日期。相对时间，无需填写，以用户领取时间作为使用开始时间。绝对时间，格式 yyyy-MM-dd，例如，2019-01-29，表示从2019-01-29 00:00:00 开始
        """
        return self._top_request(
            "taobao.tbk.dg.vegas.tlj.create",
            {
                "adzone_id": adzone_id,
                "item_id": item_id,
                "total_num": total_num,
                "name": name,
                "user_total_win_num_limit": user_total_win_num_limit,
                "security_switch": security_switch,
                "per_face": per_face,
                "send_start_time": send_start_time,
                "campaign_type": campaign_type,
                "send_end_time": send_end_time,
                "use_end_time": use_end_time,
                "use_end_time_mode": use_end_time_mode,
                "use_start_time": use_start_time
            }
        )

    def taobao_tbk_dg_vegas_tlj_instance_report(
            self,
            rights_id
    ):
        """
        导购淘礼金实例维度报表
        淘礼金实例维度相关报表数据查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43317

        :param rights_id: 实例ID
        """
        return self._top_request(
            "taobao.tbk.dg.vegas.tlj.instance.report",
            {
                "rights_id": rights_id
            }
        )


class TbGongJu(DingTalkBaseAPI):
    """
    工具API
    """

    def taobao_time_get(
            self
    ):
        """
        获取淘宝系统当前时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=120

        """
        return self._top_request(
            "taobao.time.get",
            result_processor=lambda x: x["time"]
        )

    def taobao_kfc_keyword_search(
            self,
            content,
            nick='',
            apply=''
    ):
        """
        关键词过滤匹配
        对输入的文本信息进行禁忌关键词匹配，返回匹配的结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10385

        :param content: 需要过滤的文本信息
        :param nick: 发布信息的淘宝会员名，可以不传
        :param apply: 应用点，分为一级应用点、二级应用点。其中一级应用点通常是指某一个系统或产品，比如淘宝的商品应用（taobao_auction）；二级应用点，是指一级应用点下的具体的分类，比如商品标题(title)、商品描述(content)。不同的二级应用可以设置不同关键词。<br/><br/>这里的apply参数是由一级应用点与二级应用点合起来的字符（一级应用点+'.'+二级应用点），如taobao_auction.title。<br/><br/><br/>通常apply参数是不需要传递的。如有特殊需求（比如特殊的过滤需求，需要自己维护一套自己词库），需传递此参数。
        """
        return self._top_request(
            "taobao.kfc.keyword.search",
            {
                "content": content,
                "nick": nick,
                "apply": apply
            },
            result_processor=lambda x: x["kfc_search_result"]
        )

    def taobao_topats_result_get(
            self,
            task_id
    ):
        """
        获取异步任务结果
        获取异步任务结果。<br/><br/>异步API使用方法，请查看：<a href='http://open.taobao.com/doc/detail.htm?id=30'>异步API使用说明</a><br/><br/>1. 只能获取AppKey自身创建的异步任务结果<br/><br/>2. 如果任务状态为done，则返回任务结果（如taobao.topats.trades.fullinfo.get）或下载地址（如taobao.topats.trades.sold.get）<br/><br/>3. 任务结果下载地址只能使用一次，需要重复下载可重新调用此接口获取下载地址
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10415

        :param task_id: 任务ID，创建任务时返回的task_id
        """
        return self._top_request(
            "taobao.topats.result.get",
            {
                "task_id": task_id
            },
            result_processor=lambda x: x["task"]
        )

    def taobao_topats_task_delete(
            self,
            task_id
    ):
        """
        取消已经创建的异步任务
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        可用于取消已经创建的异步任务。</br><br/>条件限制：</br><br/>1)一次只可以取消一个任务</br><br/>2）只能取消自己创建的任务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11069

        :param task_id: 需要取消的任务ID
        """
        return self._top_request(
            "taobao.topats.task.delete",
            {
                "task_id": task_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_appip_get(
            self
    ):
        """
        获取ISV发起请求服务器IP
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21784

        """
        return self._top_request(
            "taobao.appip.get",
            result_processor=lambda x: x["ip"]
        )

    def taobao_openuid_change(
            self,
            open_uid,
            target_app_key
    ):
        """
        淘宝openUid 转换
        将当前应用所属的openUId 转换为对应targetAppkey的openUid
        规则：
        1.如果两个appkey是应用前后台关系，可以直接转换；
        2.如果appkey和targetAppkey都属于同一个开发者，不允许互相转换。
        3.如果appkey和targetAppkey不属于同一个开发者，不允许互相转换。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23831

        :param open_uid: openUid
        :param target_app_key: 转换到的appkey
        """
        return self._top_request(
            "taobao.openuid.change",
            {
                "open_uid": open_uid,
                "target_app_key": target_app_key
            },
            result_processor=lambda x: x["new_open_uid"]
        )

    def taobao_top_auth_token_refresh(
            self,
            refresh_token
    ):
        """
        刷新Access Token
        根据refresh_token重新生成token
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25387

        :param refresh_token: grantType==refresh_token 时需要
        """
        return self._top_request(
            "taobao.top.auth.token.refresh",
            {
                "refresh_token": refresh_token
            },
            result_processor=lambda x: x["token_result"]
        )

    def taobao_top_auth_token_create(
            self,
            code,
            uuid=''
    ):
        """
        获取Access Token
        用户通过code换获取access_token，https only
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25388

        :param code: 授权code，grantType==authorization_code 时需要
        :param uuid: 与生成code的uuid配对
        """
        return self._top_request(
            "taobao.top.auth.token.create",
            {
                "code": code,
                "uuid": uuid
            },
            result_processor=lambda x: x["token_result"]
        )

    def taobao_httpdns_get(
            self
    ):
        """
        TOPDNS配置
        获取TOP DNS配置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25414

        """
        return self._top_request(
            "taobao.httpdns.get"
        )

    def taobao_top_ipout_get(
            self
    ):
        """
        获取开放平台出口IP段
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25441

        """
        return self._top_request(
            "taobao.top.ipout.get",
            result_processor=lambda x: x["ip_list"]
        )

    def taobao_top_secret_get(
            self,
            random_num,
            secret_version='',
            customer_user_id=''
    ):
        """
        获取TOP通道解密秘钥
        top sdk通过api获取对应解密秘钥
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26567

        :param random_num: 伪随机数
        :param secret_version: 秘钥版本号
        :param customer_user_id: 自定义用户id
        """
        return self._top_request(
            "taobao.top.secret.get",
            {
                "random_num": random_num,
                "secret_version": secret_version,
                "customer_user_id": customer_user_id
            }
        )

    def taobao_top_secret_register(
            self,
            user_id=''
    ):
        """
        注册加密账号
        提供给isv注册非淘系账号秘钥，isv依赖sdk自主加、解密
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27385

        :param user_id: 用户id，保证唯一
        """
        return self._top_request(
            "taobao.top.secret.register",
            {
                "user_id": user_id
            }
        )

    def taobao_top_sdk_feedback_upload(
            self,
            type,
            content=''
    ):
        """
        sdk信息回调
        sdk回调客户端基本信息到开放平台，用于做监控之类，有助于帮助isv监控系统稳定性
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27512

        :param type: 1、回传加密信息
        :param content: 具体内容，json形式
        """
        return self._top_request(
            "taobao.top.sdk.feedback.upload",
            {
                "type": type,
                "content": content
            },
            result_processor=lambda x: x["upload_interval"]
        )

    def taobao_streetest_session_get(
            self
    ):
        """
        根据获取压测用户的sessionKey
        根据正常用户sessionKey获取对应压测用户的sessionKey，该sessionKey只能用户服务商全链路压测
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28185

        """
        return self._top_request(
            "taobao.streetest.session.get",
            result_processor=lambda x: x["stree_test_session_key"]
        )

    def taobao_qimen_trade_users_get(
            self,
            page_index,
            page_size
    ):
        """
        获取奇门用户列表
        获取已开通奇门订单服务的用户列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28674

        :param page_index: 每页的数量
        :param page_size: 页数
        """
        return self._top_request(
            "taobao.qimen.trade.users.get",
            {
                "page_index": page_index,
                "page_size": page_size
            }
        )

    def taobao_qimen_trade_user_delete(
            self
    ):
        """
        删除奇门订单链路用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28675

        """
        return self._top_request(
            "taobao.qimen.trade.user.delete",
            result_processor=lambda x: x["modal"]
        )

    def taobao_qimen_trade_user_add(
            self,
            memo=''
    ):
        """
        添加奇门订单链路用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28676

        :param memo: 商家备注
        """
        return self._top_request(
            "taobao.qimen.trade.user.add",
            {
                "memo": memo
            }
        )

    def taobao_qimen_events_produce(
            self,
            messages
    ):
        """
        批量发送奇门事件
        批量发送消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28677

        :param messages: 奇门事件列表, 最多50条
        """
        return self._top_request(
            "taobao.qimen.events.produce",
            {
                "messages": messages
            }
        )

    def taobao_qimen_event_produce(
            self,
            status,
            tid,
            ext='',
            platform='',
            create='',
            nick=''
    ):
        """
        发出奇门事件
        当订单被处理时，用于通知奇门系统。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28678

        :param status: 事件状态，如QIMEN_ERP_TRANSFER，QIMEN_ERP_CHECK
        :param tid: 淘宝订单号
        :param ext: JSON格式扩展字段
        :param platform: 商家平台编码.MAIN:官方渠道,JD:京东,DD:当当,PP:拍拍,YX:易讯,EBAY:ebay,AMAZON:亚马逊,SN:苏宁,GM:国美,WPH:唯品会,JM:聚美,MGJ:蘑菇街,YT:银泰,YHD:1号店,1688:1688,POS:POS门店,OTHER:其他
        :param create: 订单创建时间,数字
        :param nick: 外部商家名称。必须同时填写platform
        """
        return self._top_request(
            "taobao.qimen.event.produce",
            {
                "status": status,
                "tid": tid,
                "ext": ext,
                "platform": platform,
                "create": create,
                "nick": nick
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_tmc_auth_get(
            self,
            group=''
    ):
        """
        TMC授权token
        TMC连接授权Token
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29033

        :param group: tmc组名
        """
        return self._top_request(
            "taobao.tmc.auth.get",
            {
                "group": group
            }
        )

    def taobao_rdc_aligenius_refunds_check(
            self,
            param
    ):
        """
        退款信息审核
        根据退款信息，对退款单进行审核
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29161

        :param param: 入参
        """
        return self._top_request(
            "taobao.rdc.aligenius.refunds.check",
            {
                "param": param
            }
        )

    def taobao_rdc_aligenius_autorefunds_query(
            self,
            start_time,
            end_time,
            page_no,
            page_size
    ):
        """
        退款信息查询
        （此接口后期将不再维护，请勿使用）供第三方商家查询授权给自己的所有退款款订单的退款信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29167

        :param start_time: 查询数据时间段开始时间
        :param end_time: 查询数据时间段结束时间
        :param page_no: 页数
        :param page_size: 每页数据数
        """
        return self._top_request(
            "taobao.rdc.aligenius.autorefunds.query",
            {
                "start_time": start_time,
                "end_time": end_time,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def alibaba_interact_sensor_ui(
            self,
            un_named=''
    ):
        """
        基本ui操作
        Weex 基本UI操作
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30850

        :param un_named: 仅作客户端鉴权使用，不会发送接收请求
        """
        return self._top_request(
            "alibaba.interact.sensor.ui",
            {
                "un_named": un_named
            },
            result_processor=lambda x: x["unnamed"]
        )

    def tmall_marketing_face_skindetect(
            self,
            image,
            source,
            front_camera='0',
            mixnick='mixnick'
    ):
        """
        肌肤检测
        提供人脸肌肤属性报告
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31053

        :param image: 图片的base64（必须以base64,开头）
        :param source: isv标识
        :param front_camera: 前置摄像头1，后置摄像头0
        :param mixnick: 混淆nick
        """
        return self._top_request(
            "tmall.marketing.face.skindetect",
            {
                "image": image,
                "source": source,
                "front_camera": front_camera,
                "mixnick": mixnick
            },
            result_processor=lambda x: x["detect_result"]
        )

    def alibaba_retail_shorturl_get(
            self,
            source_url,
            options=None
    ):
        """
        短链接获取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31753

        :param source_url: 源url
        :param options: 系统自动生成
        """
        return self._top_request(
            "alibaba.retail.shorturl.get",
            {
                "source_url": source_url,
                "options": options
            }
        )

    def alibaba_retail_device_trade_ship(
            self,
            device_type,
            device_id,
            trade_no,
            ship_detail_list,
            order_update_option=None
    ):
        """
        贩卖机掉货成功通知
        贩卖机发货
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31756

        :param device_type: 设备类型
        :param device_id: 设备ID
        :param trade_no: 内部订单编号
        :param ship_detail_list: 详情
        :param order_update_option: 选项
        """
        return self._top_request(
            "alibaba.retail.device.trade.ship",
            {
                "device_type": device_type,
                "device_id": device_id,
                "trade_no": trade_no,
                "ship_detail_list": ship_detail_list,
                "order_update_option": order_update_option
            }
        )

    def taobao_files_get(
            self,
            start_date,
            end_date,
            status=''
    ):
        """
        业务文件获取
        获取业务方暂存给ISV的文件列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32298

        :param start_date: 搜索开始时间
        :param end_date: 搜索结束时间
        :param status: 下载链接状态。1:未下载。2:已下载
        """
        return self._top_request(
            "taobao.files.get",
            {
                "start_date": start_date,
                "end_date": end_date,
                "status": status
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_openlink_session_get(
            self,
            code
    ):
        """
        获取授权session信息
        帮助第三方isv生成三方session
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32315

        :param code: 授权码
        """
        return self._top_request(
            "taobao.openlink.session.get",
            {
                "code": code
            }
        )

    def taobao_cloudbridge_caseinvest_execute(
            self,
            api_name='',
            data=''
    ):
        """
        红盾云桥案件协查服务
        通过API接口直接提供政府部门录入及查询函件服务
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=32353

        :param api_name: 方法名称
        :param data: 方法参数
        """
        return self._top_request(
            "taobao.cloudbridge.caseinvest.execute",
            {
                "api_name": api_name,
                "data": data
            }
        )

    def taobao_openuid_get(
            self
    ):
        """
        获取授权账号对应的OpenUid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33220

        """
        return self._top_request(
            "taobao.openuid.get",
            result_processor=lambda x: x["open_uid"]
        )

    def taobao_openuid_get_bytrade(
            self,
            tid
    ):
        """
        通过订单获取对应买家的openUID
        通过订单获取对应买家的openUID,需要卖家授权
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33221

        :param tid: 订单ID
        """
        return self._top_request(
            "taobao.openuid.get.bytrade",
            {
                "tid": tid
            },
            result_processor=lambda x: x["open_uid"]
        )

    def taobao_openuid_get_bymixnick(
            self,
            mix_nick
    ):
        """
        通过mixnick转换openuid
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33223

        :param mix_nick: 无线类应用获取到的混淆的nick
        """
        return self._top_request(
            "taobao.openuid.get.bymixnick",
            {
                "mix_nick": mix_nick
            },
            result_processor=lambda x: x["open_uid"]
        )

    def tmall_marketing_face_faceinference(
            self,
            image,
            source,
            user_id='mixnick'
    ):
        """
        人脸特征推断
        提供人脸特征检测功能，如脸型，发型等信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33934

        :param image: 图片url
        :param source: isv标识
        :param user_id: 混淆昵称
        """
        return self._top_request(
            "tmall.marketing.face.faceinference",
            {
                "image": image,
                "source": source,
                "user_id": user_id
            },
            result_processor=lambda x: x["inference_result"]
        )

    def alibaba_cuntao_interact_requisition_update(
            self,
            status='"makeEnd"',
            uuid_list=''
    ):
        """
        更新物料制作状态
        村淘物料下沉，更新物料制作状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40393

        :param status: 物料制作状态
        :param uuid_list: 申请单id列表
        """
        return self._top_request(
            "alibaba.cuntao.interact.requisition.update",
            {
                "status": status,
                "uuid_list": uuid_list
            }
        )

    def alibaba_cuntao_interact_requisition_get(
            self,
            page_size='20',
            gmt_create_end='',
            gmt_create_start='',
            page_index='0'
    ):
        """
        供应商获取物料申请单列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40394

        :param page_size: 页大小，默认20
        :param gmt_create_end: 截止时间戳，开区间
        :param gmt_create_start: 开始时间戳，闭区间
        :param page_index: 页码，从0开始
        """
        return self._top_request(
            "alibaba.cuntao.interact.requisition.get",
            {
                "page_size": page_size,
                "gmt_create_end": gmt_create_end,
                "gmt_create_start": gmt_create_start,
                "page_index": page_index
            }
        )

    def alibaba_interact_supplier_award_resource_get_cuntao(
            self,
            user_nick,
            activity_key,
            lng='',
            lat=''
    ):
        """
        权益池信息查询
        农村淘宝营销互动权益池信息查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42702

        :param user_nick: 用户昵称
        :param activity_key: 活动code
        :param lng: 经度
        :param lat: 纬度
        """
        return self._top_request(
            "alibaba.interact.supplier.award.resource.get.cuntao",
            {
                "user_nick": user_nick,
                "activity_key": activity_key,
                "lng": lng,
                "lat": lat
            }
        )


class TbWuLiuBao(DingTalkBaseAPI):
    """
    物流宝API
    """

    def taobao_wlb_order_consign(
            self,
            wlb_order_code
    ):
        """
        物流宝订单已发货通知接口
        如果erp导入淘宝交易订单到物流宝，当物流宝订单已发货的时候，erp需要调用该接口来通知物流订单和淘宝交易订单已发货
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10597

        :param wlb_order_code: 物流宝订单编号
        """
        return self._top_request(
            "taobao.wlb.order.consign",
            {
                "wlb_order_code": wlb_order_code
            },
            result_processor=lambda x: x["modify_time"]
        )

    def taobao_wlb_notify_message_confirm(
            self,
            message_id
    ):
        """
        物流宝通知消息确认
        确认物流宝可执行消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10603

        :param message_id: 物流宝通知消息的id，通过taobao.wlb.notify.message.page.get接口得到的WlbMessage数据结构中的id字段
        """
        return self._top_request(
            "taobao.wlb.notify.message.confirm",
            {
                "message_id": message_id
            },
            result_processor=lambda x: x["gmt_modified"]
        )

    def taobao_wlb_notify_message_page_get(
            self,
            msg_code='',
            page_no='',
            page_size='',
            start_date='',
            end_date='',
            status=''
    ):
        """
        物流宝通知消息查询接口
        物流宝提供的消息通知查询接口，消息内容包括;出入库单不一致消息，取消订单成功消息，盘点单消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10604

        :param msg_code: 通知消息编码： STOCK_IN_NOT_CONSISTENT---入库单不一致 CANCEL_ORDER_SUCCESS---取消订单成功 INVENTORY_CHECK---盘点 CANCEL_ORDER_FAILURE---取消订单失败 ORDER_REJECT--wms拒单 ORDER_CONFIRMED--订单处理成功
        :param page_no: 分页查询页数
        :param page_size: 分页查询的每页页数
        :param start_date: 记录开始时间
        :param end_date: 记录截至时间
        :param status: 消息状态： 不需要确认：NO_NEED_CONFIRM 已确认：CONFIRMED 待确认：TO_BE_CONFIRM
        """
        return self._top_request(
            "taobao.wlb.notify.message.page.get",
            {
                "msg_code": msg_code,
                "page_no": page_no,
                "page_size": page_size,
                "start_date": start_date,
                "end_date": end_date,
                "status": status
            }
        )

    def taobao_wlb_tmsorder_query(
            self,
            order_code,
            page_no='1',
            page_size='20'
    ):
        """
        通过物流订单编号查询物流信息
        通过物流订单编号分页查询物流信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10609

        :param order_code: 物流订单编号
        :param page_no: 当前页
        :param page_size: 分页记录个数，如果用户输入的记录数大于50，则一页显示50条记录
        """
        return self._top_request(
            "taobao.wlb.tmsorder.query",
            {
                "order_code": order_code,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_wlb_item_map_get(
            self,
            item_id
    ):
        """
        根据物流宝商品ID查询商品映射关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10614

        :param item_id: 要查询映射关系的物流宝商品id
        """
        return self._top_request(
            "taobao.wlb.item.map.get",
            {
                "item_id": item_id
            }
        )

    def taobao_wlb_item_combination_get(
            self,
            item_id
    ):
        """
        根据商品id查询商品组合关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10615

        :param item_id: 要查询的组合商品id
        """
        return self._top_request(
            "taobao.wlb.item.combination.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["item_id_list"]
        )

    def taobao_wlb_inventorylog_query(
            self,
            item_id='',
            store_code='',
            order_code='',
            gmt_start='',
            gmt_end='',
            page_no='1',
            page_size='20',
            op_user_id='',
            op_type=''
    ):
        """
        根据商品ID查询所有库存变更记录
        通过商品ID等几个条件来分页查询库存变更记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10617

        :param item_id: 商品ID
        :param store_code: 仓库编码
        :param order_code: 单号
        :param gmt_start: 起始修改时间,大于等于该时间
        :param gmt_end: 结束修改时间,小于等于该时间
        :param page_no: 当前页
        :param page_size: 分页记录个数
        :param op_user_id: 可指定授权的用户来查询
        :param op_type: 库存操作作类型(可以为空) CHU_KU 1-出库 RU_KU 2-入库 FREEZE 3-冻结 THAW 4-解冻 CHECK_FREEZE 5-冻结确认 CHANGE_KU 6-库存类型变更 若值不在范围内，则按CHU_KU处理
        """
        return self._top_request(
            "taobao.wlb.inventorylog.query",
            {
                "item_id": item_id,
                "store_code": store_code,
                "order_code": order_code,
                "gmt_start": gmt_start,
                "gmt_end": gmt_end,
                "page_no": page_no,
                "page_size": page_size,
                "op_user_id": op_user_id,
                "op_type": op_type
            }
        )

    def taobao_wlb_item_update(
            self,
            id,
            update_property_key_list='',
            delete_property_key_list='',
            update_property_value_list='',
            name='',
            title='',
            remark='',
            is_friable='',
            is_dangerous='',
            color='',
            weight='',
            length='',
            width='',
            height='',
            volume='',
            goods_cat='',
            pricing_cat='',
            package_material=''
    ):
        """
        物流宝商品修改
        修改物流宝商品信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10619

        :param id: 要修改的商品id
        :param update_property_key_list: 需要修改的商品属性值的列表，如果属性不存在，则新增属性
        :param delete_property_key_list: 需要删除的商品属性key列表
        :param update_property_value_list: 需要修改的属性值的列表
        :param name: 要修改的商品名称
        :param title: 要修改的商品标题
        :param remark: 要修改的商品备注
        :param is_friable: 是否易碎品
        :param is_dangerous: 是否危险品
        :param color: 商品颜色
        :param weight: 商品重量，单位G
        :param length: 商品长度，单位厘米
        :param width: 商品宽度，单位厘米
        :param height: 商品高度，单位厘米
        :param volume: 商品体积，单位立方厘米
        :param goods_cat: 商品货类
        :param pricing_cat: 商品计价货类
        :param package_material: 商品包装材料类型
        """
        return self._top_request(
            "taobao.wlb.item.update",
            {
                "id": id,
                "update_property_key_list": update_property_key_list,
                "delete_property_key_list": delete_property_key_list,
                "update_property_value_list": update_property_value_list,
                "name": name,
                "title": title,
                "remark": remark,
                "is_friable": is_friable,
                "is_dangerous": is_dangerous,
                "color": color,
                "weight": weight,
                "length": length,
                "width": width,
                "height": height,
                "volume": volume,
                "goods_cat": goods_cat,
                "pricing_cat": pricing_cat,
                "package_material": package_material
            },
            result_processor=lambda x: x["gmt_modified"]
        )

    def taobao_wlb_subscription_query(
            self,
            status='',
            page_no='1',
            page_size='20'
    ):
        """
        查询商家定购的所有服务
        查询商家定购的所有服务,可通过入参状态来筛选
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10620

        :param status: 状态 <br/>AUDITING 1-待审核; <br/>CANCEL 2-撤销 ;<br/>CHECKED 3-审核通过 ;<br/>FAILED 4-审核未通过 ;<br/>SYNCHRONIZING 5-同步中;<br/>只允许输入上面指定的值，且可以为空，为空时查询所有状态。若输错了，则按AUDITING处理。
        :param page_no: 当前页
        :param page_size: 分页记录个数，如果用户输入的记录数大于50，则一页显示50条记录
        """
        return self._top_request(
            "taobao.wlb.subscription.query",
            {
                "status": status,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_wlb_order_page_get(
            self,
            page_size='20',
            page_no='1',
            order_status='',
            order_code='',
            order_type='',
            order_sub_type='',
            end_time='',
            start_time=''
    ):
        """
        分页查询物流宝订单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10622

        :param page_size: 每页多少条
        :param page_no: 分页的第几页
        :param order_status: TMS拒签：-100 接收方拒签：-200
        :param order_code: 物流订单编号
        :param order_type: 订单类型: （1）NORMAL_OUT ：正常出库 （2）NORMAL_IN：正常入库 （3）RETURN_IN：退货入库 （4）EXCHANGE_OUT：换货出库
        :param order_sub_type: 订单子类型： （1）OTHER： 其他 （2）TAOBAO_TRADE： 淘宝交易 （3）OTHER_TRADE：其他交易 （4）ALLCOATE： 调拨 （5）CHECK: 盘点单 （6）PURCHASE: 采购单
        :param end_time: 查询截止时间
        :param start_time: 查询开始时间
        """
        return self._top_request(
            "taobao.wlb.order.page.get",
            {
                "page_size": page_size,
                "page_no": page_no,
                "order_status": order_status,
                "order_code": order_code,
                "order_type": order_type,
                "order_sub_type": order_sub_type,
                "end_time": end_time,
                "start_time": start_time
            }
        )

    def taobao_wlb_orderstatus_get(
            self,
            order_code
    ):
        """
        物流宝订单流转状态查询
        根据物流宝订单号查询物流宝订单至目前为止的流转状态列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10625

        :param order_code: 物流宝订单编码
        """
        return self._top_request(
            "taobao.wlb.orderstatus.get",
            {
                "order_code": order_code
            },
            result_processor=lambda x: x["wlb_order_status"]
        )

    def taobao_wlb_order_cancel(
            self,
            wlb_order_code
    ):
        """
        取消物流宝订单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10626

        :param wlb_order_code: 物流宝订单编号
        """
        return self._top_request(
            "taobao.wlb.order.cancel",
            {
                "wlb_order_code": wlb_order_code
            }
        )

    def taobao_wlb_item_get(
            self,
            item_id
    ):
        """
        根据商品ID获取商品信息
        根据商品ID获取商品信息,除了获取商品信息外还可获取商品属性信息和库存信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10627

        :param item_id: 商品ID
        """
        return self._top_request(
            "taobao.wlb.item.get",
            {
                "item_id": item_id
            },
            result_processor=lambda x: x["item"]
        )

    def taobao_wlb_tradeorder_get(
            self,
            trade_id,
            trade_type,
            sub_trade_id=''
    ):
        """
        根据交易号获取物流宝订单
        根据交易类型和交易id查询物流宝订单详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10628

        :param trade_id: 指定交易类型的交易号
        :param trade_type: 交易类型: TAOBAO--淘宝交易 OTHER_TRADE--其它交易
        :param sub_trade_id: 子交易号
        """
        return self._top_request(
            "taobao.wlb.tradeorder.get",
            {
                "trade_id": trade_id,
                "trade_type": trade_type,
                "sub_trade_id": sub_trade_id
            },
            result_processor=lambda x: x["wlb_order_list"]
        )

    def taobao_wlb_inventory_detail_get(
            self,
            item_id,
            inventory_type_list='',
            store_code=''
    ):
        """
        查询库存详情
        查询库存详情，通过商品ID获取发送请求的卖家的库存详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10629

        :param item_id: 商品ID
        :param inventory_type_list: 库存类型列表，值包括：<br/>VENDIBLE--可销售库存<br/>FREEZE--冻结库存<br/>ONWAY--在途库存<br/>DEFECT--残次品库存<br/>ENGINE_DAMAGE--机损<br/>BOX_DAMAGE--箱损<br/>EXPIRATION--过保
        :param store_code: 仓库编码
        """
        return self._top_request(
            "taobao.wlb.inventory.detail.get",
            {
                "item_id": item_id,
                "inventory_type_list": inventory_type_list,
                "store_code": store_code
            }
        )

    def taobao_wlb_order_create(
            self,
            order_sub_type,
            order_type,
            out_biz_code,
            store_code,
            is_finished,
            order_item_list,
            schedule_type='',
            order_code='',
            prev_order_code='',
            tms_service_code='',
            tms_order_code='',
            schedule_start='',
            schedule_end='',
            order_flag='',
            alipay_no='',
            total_amount='',
            payable_amount='',
            service_fee='',
            buyer_nick='',
            receiver_info='',
            sender_info='',
            expect_start_time='',
            expect_end_time='',
            package_count='',
            tms_info='',
            invoince_info='',
            attributes='',
            remark=''
    ):
        """
        创建物流宝订单
        创建物流宝订单，由外部ISV或者ERP，Elink，淘宝交易产生
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10630

        :param order_sub_type: 订单子类型：  （1）OTHER： 其他  （2）TAOBAO_TRADE： 淘宝交易  （3）OTHER_TRADE：其他交易  （4）ALLCOATE： 调拨  （5）PURCHASE:采购
        :param order_type: 订单类型:  （1）NORMAL_OUT ：正常出库  （2）NORMAL_IN：正常入库  （3）RETURN_IN：退货入库  （4）EXCHANGE_OUT：换货出库
        :param out_biz_code: 外部订单业务ID，该编号在isv中是唯一编号， 用来控制并发，去重用
        :param store_code: 仓库编码
        :param is_finished: 该物流宝订单是否已完成，如果完成则设置为true，如果为false，则需要等待继续创建订单商品信息。
        :param order_item_list: 订单商品列表： {'order_item_list':[{'trade_code':'可选,淘宝交易订单，并且不是赠品，必须要传订单来源编号',' sub_trade_code ':'可选,淘宝子交易号','item_id':'必须,商品Id','item_code':'必须,商家编码','item_name':'可选,物流宝商品名称','item_quantity':'必选,计划数量','item_price':'必选,物品价格,单位为分','owner_user_nick
                                ':'可选,货主nick 代销模式下会存在','flag':'判断是否为赠品0 不是1是','remarks':'可选,备注','batch_remark':'可选，批次描述信息会把这个信息带给WMS，但不会跟物流宝库存相关联'，'inventory_type':'库存类型1 可销售库存 101 类型用来定义残次品 201 冻结类型库存 301 在途库存','picture_url':'图片Url','distributor_user_nick': '分销商NICK',必选'ext_order_item_code':'可选，外部商品的商家编码']} ======================================== 如果订单中的商品条目数大于50条的时候，我们会校验，不能创建成功，需要你按照50个一批的数量传，需要分批调用该接口，第二次传的时候，需要带上wlb_order_code和is_finished和order_item_list三个字段是必传的，is_finished为true表示传输完毕，为false表示还没完全传完。
        :param schedule_type: 投递时延要求:  （1）INSTANT_ARRIVED： 当日达  （2）TOMMORROY_MORNING_ARRIVED：次晨达  （3）TOMMORROY_ARRIVED：次日达  （4）工作日：WORK_DAY  （5）节假日：WEEKED_DAY
        :param order_code: 物流宝订单编号，该接口约定每次最多只能传50条order_item_list，如果一个物流宝订单超过50条商品的时候，需要批量来调用该接口，第一次调用的时候，wlb_order_code为空，如果第一次创建成功，该接口返回wlb_order_code，其后继续再该订单上添加商品条目，需要带上wlb_order_code，out_biz_code，order_item_list,is_finished四个字段。
        :param prev_order_code: 源订单编号
        :param tms_service_code: 物流公司编码
        :param tms_order_code: 运单编号，退货单时可能会使用
        :param schedule_start: 投递时间范围要求,格式'13:20'用分号隔开
        :param schedule_end: 投递时间范围要求,格式'15:20'用分号隔开
        :param order_flag: 用字符串格式来表示订单标记列表：比如COD^PRESELL^SPLIT^LIMIT 等，中间用“^”来隔开 ---------------------------------------- 订单标记list（所有字母全部大写）： 1: COD –货到付款 2: LIMIT-限时配送 3: PRESELL-预售 5:COMPLAIN-已投诉 7:SPLIT-拆单， 8:EXCHANGE-换货， 9:VISIT-上门 ， 10: MODIFYTRANSPORT-是否可改配送方式，<br/>: 是否可改配送方式  默认可更改<br/>11 CONSIGN 物流宝代理发货,自动更改发货状态<br/>12: SELLER_AFFORD 是否卖家承担运费 默认是，即没 13: SYNC_RETURN_BILL，同时退回发票
        :param alipay_no: 支付宝交易号
        :param total_amount: 总金额
        :param payable_amount: 应收金额，cod订单必选
        :param service_fee: cod服务费，只有cod订单的时候，才需要这个字段
        :param buyer_nick: 买家呢称
        :param receiver_info: 收货方信息，必须传， 手机和电话必选其一。
                              收货方信息：
                              邮编^^^省^^^市^^^区^^^具体地址^^^收件方名称^^^手机^^^电话
                              如果某一个字段的数据为空时，必须传NA
        :param sender_info: 发货方信息，发货方信息必须传， 手机和电话必选其一。 发货方信息：
                            邮编^^^省^^^市^^^区^^^具体地址^^^收件方名称^^^手机^^^电话
                            如果某一个字段的数据为空时，必须传NA
        :param expect_start_time: 计划开始送达时间  在入库单中可能会使用
        :param expect_end_time: 期望结束时间，在入库单会使用到
        :param package_count: 包裹件数，入库单和出库单中会用到
        :param tms_info: 出库单中可能会用到<br/>运输公司名称^^^运输公司联系人^^^运输公司运单号^^^运输公司电话^^^运输公司联系人身份证号<br/><br/>========================================<br/>如果某一个字段的数据为空时，必须传NA
        :param invoince_info: {'invoince_info': [{'bill_type':'发票类型，必选', 'bill_title':'发票抬头，必选', 'bill_amount':'发票金额(单位是分)，必选','bill_content':'发票内容，可选'}]}
        :param attributes: 该字段暂时保留
        :param remark: 备注
        """
        return self._top_request(
            "taobao.wlb.order.create",
            {
                "order_sub_type": order_sub_type,
                "order_type": order_type,
                "out_biz_code": out_biz_code,
                "store_code": store_code,
                "is_finished": is_finished,
                "order_item_list": order_item_list,
                "schedule_type": schedule_type,
                "order_code": order_code,
                "prev_order_code": prev_order_code,
                "tms_service_code": tms_service_code,
                "tms_order_code": tms_order_code,
                "schedule_start": schedule_start,
                "schedule_end": schedule_end,
                "order_flag": order_flag,
                "alipay_no": alipay_no,
                "total_amount": total_amount,
                "payable_amount": payable_amount,
                "service_fee": service_fee,
                "buyer_nick": buyer_nick,
                "receiver_info": receiver_info,
                "sender_info": sender_info,
                "expect_start_time": expect_start_time,
                "expect_end_time": expect_end_time,
                "package_count": package_count,
                "tms_info": tms_info,
                "invoince_info": invoince_info,
                "attributes": attributes,
                "remark": remark
            }
        )

    def taobao_wlb_item_query(
            self,
            is_sku='',
            status='',
            item_type='',
            name='',
            title='',
            item_code='',
            parent_id='',
            page_no='1',
            page_size='20'
    ):
        """
        分页查询商品
        根据状态、卖家、SKU等信息查询商品列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10631

        :param is_sku: 是否是最小库存单元，只有最小库存单元的商品才可以有库存,值只能给'true','false'来表示;  若值不在范围内，则按true处理
        :param status: 只能输入以下值或空：  ITEM_STATUS_VALID -- 1 表示 有效；  ITEM_STATUS_LOCK -- 2 表示锁住。  若值不在范围内，按ITEM_STATUS_VALID处理
        :param item_type: ITEM类型(只允许输入以下英文或空)  NORMAL 0:普通商品;  COMBINE 1:是否是组合商品  DISTRIBUTION 2:是否是分销商品(货主是别人)  若值不在范围内，则按NORMAL处理
        :param name: 商品名称
        :param title: 商品前台销售名字
        :param item_code: 商家编码
        :param parent_id: 父ID,只有is_sku=1时才能有父ID，商品也可以没有付商品
        :param page_no: 当前页
        :param page_size: 分页记录个数，如果用户输入的记录数大于50，则一页显示50条记录
        """
        return self._top_request(
            "taobao.wlb.item.query",
            {
                "is_sku": is_sku,
                "status": status,
                "item_type": item_type,
                "name": name,
                "title": title,
                "item_code": item_code,
                "parent_id": parent_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_wlb_wlborder_get(
            self,
            wlb_order_code
    ):
        """
        根据物流宝订单编号查询物流宝订单概要信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10632

        :param wlb_order_code: 物流宝订单编码
        """
        return self._top_request(
            "taobao.wlb.wlborder.get",
            {
                "wlb_order_code": wlb_order_code
            },
            result_processor=lambda x: x["wlb_order"]
        )

    def taobao_wlb_item_add(
            self,
            name,
            item_code,
            is_sku,
            title='',
            remark='',
            type='NORMAL',
            pro_name_list='',
            pro_value_list='',
            is_friable='',
            is_dangerous='',
            color='',
            weight='',
            length='',
            width='',
            height='',
            volume='',
            goods_cat='',
            pricing_cat='',
            package_material='',
            price='',
            support_batch=''
    ):
        """
        添加单个物流宝商品
        添加物流宝商品，支持物流宝子商品和属性添加
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10633

        :param name: 商品名称
        :param item_code: 商品编码
        :param is_sku: 是否sku
        :param title: 商品标题
        :param remark: 商品备注
        :param type: NORMAL--普通商品<br/>COMBINE--组合商品<br/>DISTRIBUTION--分销
        :param pro_name_list: 属性名列表,目前支持的属性：<br/>毛重:GWeight	<br/>净重:Nweight<br/>皮重: Tweight<br/>自定义属性：<br/>paramkey1<br/>paramkey2<br/>paramkey3<br/>paramkey4
        :param pro_value_list: 属性值列表：<br/>10,8
        :param is_friable: 是否易碎品
        :param is_dangerous: 是否危险品
        :param color: 商品颜色
        :param weight: 商品重量，单位G
        :param length: 商品长度，单位毫米
        :param width: 商品宽度，单位毫米
        :param height: 商品高度，单位毫米
        :param volume: 商品体积，单位立方厘米
        :param goods_cat: 货类
        :param pricing_cat: 计价货类
        :param package_material: 商品包装材料类型
        :param price: 商品价格，单位：分
        :param support_batch: 是否支持批次
        """
        return self._top_request(
            "taobao.wlb.item.add",
            {
                "name": name,
                "item_code": item_code,
                "is_sku": is_sku,
                "title": title,
                "remark": remark,
                "type": type,
                "pro_name_list": pro_name_list,
                "pro_value_list": pro_value_list,
                "is_friable": is_friable,
                "is_dangerous": is_dangerous,
                "color": color,
                "weight": weight,
                "length": length,
                "width": width,
                "height": height,
                "volume": volume,
                "goods_cat": goods_cat,
                "pricing_cat": pricing_cat,
                "package_material": package_material,
                "price": price,
                "support_batch": support_batch
            },
            result_processor=lambda x: x["item_id"]
        )

    def taobao_wlb_out_inventory_change_notify(
            self,
            type,
            op_type,
            source,
            item_id,
            change_count,
            result_count,
            out_biz_code,
            order_source_code='',
            store_code=''
    ):
        """
        外部库存变化通知（企业物流用户使用）
        拥有自有仓的企业物流用户通过该接口把自有仓的库存通知到物流宝，由物流宝维护该库存，控制前台显示库存的准确性。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10896

        :param type: WLB_ITEM--物流宝商品 IC_ITEM--淘宝商品 IC_SKU--淘宝sku
        :param op_type: OUT--出库 IN--入库
        :param source: （1）OTHER： 其他 （2）TAOBAO_TRADE： 淘宝交易 （3）OTHER_TRADE：其他交易 （4）ALLCOATE： 调拨 （5）CHECK:盘点 （6）PURCHASE:采购
        :param item_id: 物流宝商品id或前台宝贝id（由type类型决定）
        :param change_count: 库存变化数量
        :param result_count: 本次库存变化后库存余额
        :param out_biz_code: 库存变化唯一标识，用于去重，一个外部唯一编码唯一标识一次库存变化。
        :param order_source_code: 订单号，如果source为TAOBAO_TRADE,order_source_code必须为淘宝交易号
        :param store_code: 目前非必须，系统会选择默认值
        """
        return self._top_request(
            "taobao.wlb.out.inventory.change.notify",
            {
                "type": type,
                "op_type": op_type,
                "source": source,
                "item_id": item_id,
                "change_count": change_count,
                "result_count": result_count,
                "out_biz_code": out_biz_code,
                "order_source_code": order_source_code,
                "store_code": store_code
            },
            result_processor=lambda x: x["gmt_modified"]
        )

    def taobao_wlb_item_batch_query(
            self,
            item_ids,
            store_code='',
            page_no='1',
            page_size='20'
    ):
        """
        批次库存查询接口
        根据用户id，item id list和store code来查询商品库存信息和批次信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10970

        :param item_ids: 需要查询的商品ID列表，以字符串表示，ID间以;隔开
        :param store_code: 仓库编号
        :param page_no: 分页查询参数，指定查询页数，默认为1
        :param page_size: 分页查询参数，每页查询数量，默认20，最大值50,大于50时按照50条查询
        """
        return self._top_request(
            "taobao.wlb.item.batch.query",
            {
                "item_ids": item_ids,
                "store_code": store_code,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_inventory_ipc_inventorydetail_get(
            self,
            sc_item_id,
            page_index,
            page_size,
            status_query,
            biz_order_id='',
            biz_sub_order_id=''
    ):
        """
        库存明细查询
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        查询库存明细
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22175

        :param sc_item_id: 仓储商品id
        :param page_index: 当前页数
        :param page_size: 一页显示多少条
        :param status_query: 1:查询预扣  4：查询占用
        :param biz_order_id: 主订单号，可选
        :param biz_sub_order_id: 子订单号，可选
        """
        return self._top_request(
            "taobao.inventory.ipc.inventorydetail.get",
            {
                "sc_item_id": sc_item_id,
                "page_index": page_index,
                "page_size": page_size,
                "status_query": status_query,
                "biz_order_id": biz_order_id,
                "biz_sub_order_id": biz_sub_order_id
            },
            result_processor=lambda x: x["inventory_details"]
        )

    def taobao_wlb_orderdetail_date_get(
            self,
            start_time,
            end_time,
            page_size='20',
            page_no='1'
    ):
        """
        按照日期范围查询物流订单详情
        外部ERP可通过该接口查询一段时间内的物流宝订单，以及订单详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22288

        :param start_time: 创建时间起始
        :param end_time: 创建时间结束
        :param page_size: 分页大小
        :param page_no: 分页下标
        """
        return self._top_request(
            "taobao.wlb.orderdetail.date.get",
            {
                "start_time": start_time,
                "end_time": end_time,
                "page_size": page_size,
                "page_no": page_no
            }
        )

    def taobao_wlb_waybill_shengxian_get(
            self,
            biz_code,
            delivery_type,
            order_channels_type,
            trade_id,
            sender_address_id='',
            service_code='',
            feature=''
    ):
        """
        商家获取生鲜电子面单号
        商家通过交易订单号获取电子面单接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23617

        :param biz_code: 物流服务方代码，生鲜配送：YXSR
        :param delivery_type: 物流服务类型。冷链：602，常温：502
        :param order_channels_type: 订单渠道： 淘宝平台订单：'TB'; 天猫平台订单：'TM'; 京东：'JD'; 拍拍：'PP'; 易迅平台订单：'YX'; 一号店平台订单：'YHD'; 当当网平台订单：'DD'; EBAY：'EBAY'; QQ网购：'QQ'; 苏宁：'SN'; 国美：'GM'; 唯品会：'WPH'; 聚美：'Jm'; 乐峰：'LF'; 蘑菇街：'MGJ'; 聚尚：'JS'; 银泰：'YT'; VANCL：'VANCL'; 邮乐：'YL'; 优购：'YG'; 拍鞋：'PX'; 1688平台：'1688';
        :param trade_id: 交易号，传入交易号不能存在拆单场景。
        :param sender_address_id: 商家淘宝地址信息ID
        :param service_code: 仓库的服务代码标示，代码一个仓库的实体。(可以通过taobao.wlb.stores.baseinfo.get接口查询)
        :param feature: 预留扩展字段
        """
        return self._top_request(
            "taobao.wlb.waybill.shengxian.get",
            {
                "biz_code": biz_code,
                "delivery_type": delivery_type,
                "order_channels_type": order_channels_type,
                "trade_id": trade_id,
                "sender_address_id": sender_address_id,
                "service_code": service_code,
                "feature": feature
            }
        )

    def taobao_wlb_order_jzpartner_query(
            self,
            taobao_trade_id='',
            service_type=''
    ):
        """
        查询家装服务商列表
        为支持家装类目的商家，对绑定家装物流服务的订单可以在商家的ERP中发货、批量发货，因此开发根据服务类型查询所有的服务商列表的接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24051

        :param taobao_trade_id: 淘宝交易订单号，如果不填写Tid则必须填写serviceType。如果填写Tid，则表明只需要查询对应订单的服务商。
        :param service_type: serviceType表示查询所有的支持服务类型的服务商。 家装干线服务     11 家装干支服务     12 家装干支装服务   13 卫浴大件干线     14 卫浴大件干支     15 卫浴大件安装     16 地板干线         17 地板干支         18 地板安装         19 灯具安装         20 卫浴小件安装     21 （注：同一个服务商针对不同类型的serviceType是具有不同的tpCode的）
        """
        return self._top_request(
            "taobao.wlb.order.jzpartner.query",
            {
                "taobao_trade_id": taobao_trade_id,
                "service_type": service_type
            }
        )

    def taobao_wlb_order_jzwithins_consign(
            self,
            tid,
            tms_partner,
            jz_consign_args,
            ins_partner=None
    ):
        """
        家装发货接口
        为支持家装类目的商家，对绑定家装物流服务的订单可以在商家的ERP中发货、批量发货，因此开发带安装服务商的发货接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24052

        :param tid: 淘宝交易订单号
        :param tms_partner: 物流服务商信息
        :param jz_consign_args: 家装物流发货参数
        :param ins_partner: 物流服务商信息
        """
        return self._top_request(
            "taobao.wlb.order.jzwithins.consign",
            {
                "tid": tid,
                "tms_partner": tms_partner,
                "jz_consign_args": jz_consign_args,
                "ins_partner": ins_partner
            }
        )

    def taobao_wlb_wms_inventory_lack_upload(
            self,
            content=None
    ):
        """
        缺货通知
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24617

        :param content: 缺货通知信息
        """
        return self._top_request(
            "taobao.wlb.wms.inventory.lack.upload",
            {
                "content": content
            }
        )

    def taobao_wlb_wms_inventory_count(
            self,
            content=None
    ):
        """
        损益单回传
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24621

        :param content: 损益单回传信息
        """
        return self._top_request(
            "taobao.wlb.wms.inventory.count",
            {
                "content": content
            }
        )

    def cainiao_merchant_inventory_adjust(
            self,
            adjust_request,
            app_name,
            operation=''
    ):
        """
        商家库存调整
        商家仓库存调整接口，目前仅支持全量更新
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34736

        :param adjust_request: 商家仓编辑库存
        :param app_name: 调用方应用名
        :param operation: 操作
        """
        return self._top_request(
            "cainiao.merchant.inventory.adjust",
            {
                "adjust_request": adjust_request,
                "app_name": app_name,
                "operation": operation
            }
        )

    def taobao_uop_tob_order_create(
            self,
            delivery_order=None
    ):
        """
        ToB仓储发货
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36089

        :param delivery_order: ERP出库对象
        """
        return self._top_request(
            "taobao.uop.tob.order.create",
            {
                "delivery_order": delivery_order
            }
        )

    def cainiao_bms_order_consign_confirm(
            self,
            content=None
    ):
        """
        BMS出库通知
        BMS出库后，通知ISV
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28394

        :param content: 通知消息主体
        """
        return self._top_request(
            "cainiao.bms.order.consign.confirm",
            {
                "content": content
            }
        )


class TbZhiTongChe(DingTalkBaseAPI):
    """
    直通车API
    """

    def taobao_simba_keywords_recommend_get(
            self,
            adgroup_id,
            page_size,
            page_no,
            nick='',
            search='',
            pertinence='',
            order_by='non'
    ):
        """
        取得一个推广组的推荐关键词列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10527

        :param adgroup_id: 推广组ID
        :param page_size: 返回的每页数据量大小,最大200
        :param page_no: 返回的第几页数据，默认为1
        :param nick: 主人昵称
        :param search: 搜索量,设置此值后返回的就是大于此搜索量的词列表
        :param pertinence: 相关度
        :param order_by: 排序方式: 搜索量 search_volume 市场平均价格 average_price 相关度 relevance 不排序 non 默认为 non
        """
        return self._top_request(
            "taobao.simba.keywords.recommend.get",
            {
                "adgroup_id": adgroup_id,
                "page_size": page_size,
                "page_no": page_no,
                "nick": nick,
                "search": search,
                "pertinence": pertinence,
                "order_by": order_by
            },
            result_processor=lambda x: x["recommend_words"]
        )

    def taobao_simba_customers_authorized_get(
            self
    ):
        """
        取得当前登录用户的授权账户列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10528

        """
        return self._top_request(
            "taobao.simba.customers.authorized.get",
            result_processor=lambda x: x["nicks"]
        )

    def taobao_simba_tools_items_top_get(
            self,
            keyword,
            ip,
            nick=''
    ):
        """
        取得一个关键词的推广组排名列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10529

        :param keyword: 关键词
        :param ip: 输入的必须是一个符合ipv4或者ipv6格式的IP地址
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.tools.items.top.get",
            {
                "keyword": keyword,
                "ip": ip,
                "nick": nick
            },
            result_processor=lambda x: x["rankeditems"]
        )

    def taobao_simba_creatives_get(
            self,
            nick='',
            creative_ids='',
            adgroup_id=''
    ):
        """
        批量获得创意
        取得一个推广组的所有创意或者根据一个创意Id列表取得一组创意；<br/>如果同时提供了推广组Id和创意id列表，则优先使用推广组Id；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10530

        :param nick: 主人昵称
        :param creative_ids: 创意Id数组，最多200个
        :param adgroup_id: 推广组Id
        """
        return self._top_request(
            "taobao.simba.creatives.get",
            {
                "nick": nick,
                "creative_ids": creative_ids,
                "adgroup_id": adgroup_id
            },
            result_processor=lambda x: x["creatives"]
        )

    def taobao_simba_creatives_record_get(
            self,
            creative_ids,
            nick=''
    ):
        """
        批量得到创意修改记录
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        根据一个创意Id列表取得创意对应的修改记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10531

        :param creative_ids: 创意Id数组，最多200个
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.creatives.record.get",
            {
                "creative_ids": creative_ids,
                "nick": nick
            },
            result_processor=lambda x: x["creativerecords"]
        )

    def taobao_simba_creative_update(
            self,
            adgroup_id,
            creative_id,
            title,
            img_url,
            nick='',
            picture_id=''
    ):
        """
        修改创意与
        更新一个创意的信息，可以设置创意标题、创意图片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10532

        :param adgroup_id: 推广组Id
        :param creative_id: 创意Id
        :param title: 创意标题，最多20个汉字
        :param img_url: 创意图片地址，必须是推广组对应商品的图片之一
        :param nick: 主人昵称
        :param picture_id: 如果用户开通了创意本地上传图片功能的，可以使用该用户图片空间的图片来修改创意，pictureId为图片空间中图片的pictureId，img_url为图片空间中图片链接地址，如果是使用的主图或副图修改创意，则pictureId必须为空
        """
        return self._top_request(
            "taobao.simba.creative.update",
            {
                "adgroup_id": adgroup_id,
                "creative_id": creative_id,
                "title": title,
                "img_url": img_url,
                "nick": nick,
                "picture_id": picture_id
            },
            result_processor=lambda x: x["creativerecord"]
        )

    def taobao_simba_creative_delete(
            self,
            creative_id,
            nick=''
    ):
        """
        删除创意
        删除一个创意
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10533

        :param creative_id: 创意Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.creative.delete",
            {
                "creative_id": creative_id,
                "nick": nick
            },
            result_processor=lambda x: x["creative"]
        )

    def taobao_simba_creative_add(
            self,
            adgroup_id,
            title,
            img_url,
            nick=''
    ):
        """
        增加创意
        创建一个创意
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10534

        :param adgroup_id: 推广组Id
        :param title: 创意标题，最多20个汉字
        :param img_url: 创意图片地址，必须是推广组对应商品的图片之一
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.creative.add",
            {
                "adgroup_id": adgroup_id,
                "title": title,
                "img_url": img_url,
                "nick": nick
            },
            result_processor=lambda x: x["creative"]
        )

    def taobao_simba_adgroups_item_exist(
            self,
            campaign_id,
            item_id,
            nick='',
            product_id='101001005'
    ):
        """
        商品是否推广
        判断在一个推广计划中是否已经推广了一个商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10535

        :param campaign_id: 推广计划Id
        :param item_id: 商品Id
        :param nick: 主人昵称
        :param product_id: 产品类型 101001005 代表普通推广，101001014代表销量明星
        """
        return self._top_request(
            "taobao.simba.adgroups.item.exist",
            {
                "campaign_id": campaign_id,
                "item_id": item_id,
                "nick": nick,
                "product_id": product_id
            },
            result_processor=lambda x: x["exist"]
        )

    def taobao_simba_adgroup_update(
            self,
            adgroup_id,
            use_nonsearch_default_price='',
            online_status='',
            default_price='',
            nonsearch_max_price='',
            nick=''
    ):
        """
        更新一个推广组的信息
        更新一个推广组的信息，可以设置默认出价、是否上线、非搜索出价、非搜索是否使用默认出价
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10539

        :param adgroup_id: 推广组Id
        :param use_nonsearch_default_price: 非搜索是否使用默认出价，false-不用；true-使用；默认为true;
        :param online_status: 用户设置的上下线状态 offline-下线(暂停竞价)； online-上线；默认为online
        :param default_price: 默认出价，单位是分，不能小于5
        :param nonsearch_max_price: 非搜索出价，单位是分，不能小于5，如果use_nonseatch_default_price为使用默认出价，则此nonsearch_max_price字段传入的数据不起作用，商品将使用默认非搜索出价
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.adgroup.update",
            {
                "adgroup_id": adgroup_id,
                "use_nonsearch_default_price": use_nonsearch_default_price,
                "online_status": online_status,
                "default_price": default_price,
                "nonsearch_max_price": nonsearch_max_price,
                "nick": nick
            },
            result_processor=lambda x: x["adgroup"]
        )

    def taobao_simba_adgroup_delete(
            self,
            adgroup_id,
            nick=''
    ):
        """
        删除一个推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10540

        :param adgroup_id: 推广组Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.adgroup.delete",
            {
                "adgroup_id": adgroup_id,
                "nick": nick
            },
            result_processor=lambda x: x["adgroup"]
        )

    def taobao_simba_adgroup_add(
            self,
            campaign_id,
            item_id,
            default_price,
            title,
            img_url,
            nick=''
    ):
        """
        创建一个推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10541

        :param campaign_id: 推广计划Id
        :param item_id: 商品Id
        :param default_price: 推广组默认出价，单位为分，不能小于5 不能大于日最高限额
        :param title: 创意标题，最多20个汉字
        :param img_url: 创意图片地址，必须是商品的图片之一
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.adgroup.add",
            {
                "campaign_id": campaign_id,
                "item_id": item_id,
                "default_price": default_price,
                "title": title,
                "img_url": img_url,
                "nick": nick
            },
            result_processor=lambda x: x["adgroup"]
        )

    def taobao_simba_campaign_platform_update(
            self,
            campaign_id,
            search_channels,
            nonsearch_channels='',
            outside_discount='100',
            mobile_discount='100',
            nick=''
    ):
        """
        更新一个推广计划的平台设置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10542

        :param campaign_id: 推广计划Id
        :param search_channels: 搜索投放频道代码数组，频道代码必须是直通车搜索类频道列表中的值。1：淘宝站内搜索，8、无线站内搜索；16:无线站外搜索
        :param nonsearch_channels: 非搜索投放频道代码数组，频道代码必须是直通车非搜索类频道列表中的值。1、淘宝站内定向；2、站外定向；8、无线站内定向；16、无线站外定向
        :param outside_discount: 溢价的百分比，必须是大于等于 1小于等于200的整数
        :param mobile_discount: 站内无线频道折扣。百分比，数值必须是大于等于 1小于等于200的整数
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.platform.update",
            {
                "campaign_id": campaign_id,
                "search_channels": search_channels,
                "nonsearch_channels": nonsearch_channels,
                "outside_discount": outside_discount,
                "mobile_discount": mobile_discount,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_platform"]
        )

    def taobao_simba_campaign_platform_get(
            self,
            campaign_id,
            nick=''
    ):
        """
        取得一个推广计划的投放平台设置
        获得一个推广计划的投放平台设置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10543

        :param campaign_id: 推广计划Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.platform.get",
            {
                "campaign_id": campaign_id,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_platform"]
        )

    def taobao_simba_campaign_channeloptions_get(
            self
    ):
        """
        取得推广计划的可设置投放频道列表
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        取得推广计划的可设置投放频道列表--接口已经废弃
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10544

        """
        return self._top_request(
            "taobao.simba.campaign.channeloptions.get",
            result_processor=lambda x: x["channel_options"]
        )

    def taobao_simba_campaign_areaoptions_get(
            self
    ):
        """
        取得推广计划的可设置投放地域列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10545

        """
        return self._top_request(
            "taobao.simba.campaign.areaoptions.get",
            result_processor=lambda x: x["area_options"]
        )

    def taobao_simba_campaign_schedule_update(
            self,
            campaign_id,
            schedule,
            nick=''
    ):
        """
        更新一个推广计划的分时折扣设置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10546

        :param campaign_id: 推广计划Id
        :param schedule: 值为：“all”；或者用“;”分割的每天的设置字符串，该字符串为用“,”分割的时段折扣字符串，格式为：起始时间-结束时间:折扣，其中时间是24小时格式记录，折扣是1-150整数，表示折扣百分比；
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.schedule.update",
            {
                "campaign_id": campaign_id,
                "schedule": schedule,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_schedule"]
        )

    def taobao_simba_campaign_schedule_get(
            self,
            campaign_id,
            nick=''
    ):
        """
        取得一个推广计划的分时折扣设置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10547

        :param campaign_id: 推广计划Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.schedule.get",
            {
                "campaign_id": campaign_id,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_schedule"]
        )

    def taobao_simba_campaign_budget_update(
            self,
            use_smooth,
            campaign_id,
            budget='',
            nick=''
    ):
        """
        更新一个推广计划的日限额
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10548

        :param use_smooth: 是否平滑消耗：false-否，true-是
        :param campaign_id: 推广计划Id
        :param budget: 如果为空则取消限额；否则必须为整数，单位是元，不得小于30；
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.budget.update",
            {
                "use_smooth": use_smooth,
                "campaign_id": campaign_id,
                "budget": budget,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_budget"]
        )

    def taobao_simba_campaign_budget_get(
            self,
            campaign_id,
            nick=''
    ):
        """
        取得一个推广计划的日限额
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10549

        :param campaign_id: 推广计划Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.budget.get",
            {
                "campaign_id": campaign_id,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_budget"]
        )

    def taobao_simba_campaign_area_update(
            self,
            campaign_id,
            area,
            nick=''
    ):
        """
        更新一个推广计划的投放地域
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10550

        :param campaign_id: 推广计划Id
        :param area: 值为：“all”；或者用“,”分割的数字，数字必须是直通车全国省市列表的AreaID；
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.area.update",
            {
                "campaign_id": campaign_id,
                "area": area,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_area"]
        )

    def taobao_simba_account_balance_get(
            self,
            nick=''
    ):
        """
        获取实时余额，”元”为单位
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10551

        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.account.balance.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["balance"]
        )

    def taobao_simba_keywords_delete(
            self,
            campaign_id,
            keyword_ids,
            nick=''
    ):
        """
        删除一批关键词
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10554

        :param campaign_id: 推广计划Id
        :param keyword_ids: 关键词Id数组，最多100个
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.keywords.delete",
            {
                "campaign_id": campaign_id,
                "keyword_ids": keyword_ids,
                "nick": nick
            },
            result_processor=lambda x: x["keywords"]
        )

    def taobao_simba_campaign_area_get(
            self,
            campaign_id,
            nick=''
    ):
        """
        取得一个推广计划的投放地域设置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10555

        :param campaign_id: 推广计划Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.area.get",
            {
                "campaign_id": campaign_id,
                "nick": nick
            },
            result_processor=lambda x: x["campaign_area"]
        )

    def taobao_simba_campaign_update(
            self,
            online_status,
            campaign_id,
            title,
            nick=''
    ):
        """
        更新一个推广计划
        更新一个推广计划，可以设置推广计划名字，修改推广计划上下线状态。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10556

        :param online_status: 用户设置的上下限状态；offline-下线；online-上线；
        :param campaign_id: 推广计划Id
        :param title: 推广计划名称，不能多余40个字符，不能和客户其他推广计划同名。
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.campaign.update",
            {
                "online_status": online_status,
                "campaign_id": campaign_id,
                "title": title,
                "nick": nick
            },
            result_processor=lambda x: x["campaign"]
        )

    def taobao_simba_campaigns_get(
            self,
            nick='',
            type='0'
    ):
        """
        取得一组推广计划
        取得一个客户的推广计划；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10557

        :param nick: 主人昵称
        :param type: 计划类型0位标准计划，16位销量明星计划
        """
        return self._top_request(
            "taobao.simba.campaigns.get",
            {
                "nick": nick,
                "type": type
            },
            result_processor=lambda x: x["campaigns"]
        )

    def taobao_simba_campaign_add(
            self,
            title,
            nick='',
            type='0'
    ):
        """
        创建一个推广计划
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10558

        :param title: 推广计划名称，不能多余20个汉字，不能和客户其他推广计划同名。
        :param nick: 主人昵称
        :param type: 计划类型，当前仅支持两种标准推广0，销量明星16，默认为0
        """
        return self._top_request(
            "taobao.simba.campaign.add",
            {
                "title": title,
                "nick": nick,
                "type": type
            },
            result_processor=lambda x: x["campaign"]
        )

    def taobao_simba_keywordids_deleted_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        获取删除的词ID
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10797

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.keywordids.deleted.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["deleted_keyword_ids"]
        )

    def taobao_simba_keywordids_changed_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        获取修改的词ID
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10798

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.keywordids.changed.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["changed_keyword_ids"]
        )

    def taobao_simba_adgroupids_deleted_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        获取删除的推广组ID
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10799

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.adgroupids.deleted.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["deleted_adgroup_ids"]
        )

    def taobao_simba_adgroupids_changed_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        获取修改的推广组ID
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10800

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.adgroupids.changed.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["changed_adgroupids"]
        )

    def taobao_simba_creativeids_deleted_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        获取删除的创意ID
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10801

        :param start_time: 得到这个时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.creativeids.deleted.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["deleted_creative_ids"]
        )

    def taobao_simba_creativeids_changed_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        获取修改的创意ID
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10802

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.creativeids.changed.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["changed_creative_ids"]
        )

    def taobao_simba_login_authsign_get(
            self,
            nick=''
    ):
        """
        获取登陆权限签名
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10950

        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.login.authsign.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["subway_token"]
        )

    def taobao_simba_rpt_custbase_get(
            self,
            subway_token,
            start_time,
            end_time,
            source,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        客户账户报表基础数据对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10951

        :param subway_token: 权限验证信息
        :param start_time: 开始日期，格式yyyy-mm-dd
        :param end_time: 结束日期，格式yyyy-mm-dd
        :param source: 数据来源（站内：1，站外：2 ，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.custbase.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "source": source,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_cust_base_list"]
        )

    def taobao_simba_rpt_adgroupkeywordeffect_get(
            self,
            campaign_id,
            adgroup_id,
            start_time,
            end_time,
            source,
            subway_token,
            search_type,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广组下的词效果报表数据查询(明细数据不分类型查询)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10952

        :param campaign_id: 推广计划ID
        :param adgroup_id: 推广组ID
        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd；开始时间结束时间间隔不能超过30天
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param subway_token: 权限校验参数
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH）可多选例如：SEARCH,CAT
        :param nick: 主人昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.adgroupkeywordeffect.get",
            {
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "start_time": start_time,
                "end_time": end_time,
                "source": source,
                "subway_token": subway_token,
                "search_type": search_type,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_adgroupkeyword_effect_list"]
        )

    def taobao_simba_rpt_adgroupkeywordbase_get(
            self,
            campaign_id,
            adgroup_id,
            start_time,
            end_time,
            source,
            subway_token,
            search_type,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广组下的词基础报表数据查询(明细数据不分类型查询)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10953

        :param campaign_id: 推广计划ID
        :param adgroup_id: 推广组ID
        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param subway_token: 权限校验参数
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH）可多选例如：SEARCH,CAT
        :param nick: 主人昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.adgroupkeywordbase.get",
            {
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "start_time": start_time,
                "end_time": end_time,
                "source": source,
                "subway_token": subway_token,
                "search_type": search_type,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_adgroupkeyword_base_list"]
        )

    def taobao_simba_rpt_custeffect_get(
            self,
            start_time,
            end_time,
            subway_token,
            source,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        用户账户报表效果数据查询（只有汇总数据，无分类数据）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10954

        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd
        :param subway_token: 权限校验参数
        :param source: 数据来源（站内：1，站外：2 ，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param nick: 主人昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.custeffect.get",
            {
                "start_time": start_time,
                "end_time": end_time,
                "subway_token": subway_token,
                "source": source,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_cust_effect_list"]
        )

    def taobao_simba_rpt_campadgroupeffect_get(
            self,
            subway_token,
            start_time,
            end_time,
            campaign_id,
            source,
            search_type,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广计划下的推广组报表效果数据查询(只有汇总数据，无分类类型)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10955

        :param subway_token: 权限验证信息
        :param start_time: 开始日期，格式yyyy-mm-dd
        :param end_time: 结束日期，格式yyyy-mm-dd
        :param campaign_id: 查询推广计划id
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如：SEARCH,CAT
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.campadgroupeffect.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "campaign_id": campaign_id,
                "source": source,
                "search_type": search_type,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_campadgroup_effect_list"]
        )

    def taobao_simba_rpt_adgroupeffect_get(
            self,
            subway_token,
            campaign_id,
            adgroup_id,
            start_time,
            end_time,
            search_type,
            source,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广组效果报表数据对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10956

        :param subway_token: 权限校验参数
        :param campaign_id: 推广计划id
        :param adgroup_id: 推广组id
        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT,<br/>定向投放：NOSEARCH ）可以一次取多个例如：SEARCH,CAT
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5）可多选，以逗号分隔
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.adgroupeffect.get",
            {
                "subway_token": subway_token,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "start_time": start_time,
                "end_time": end_time,
                "search_type": search_type,
                "source": source,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_adgroup_effect_list"]
        )

    def taobao_simba_rpt_campadgroupbase_get(
            self,
            subway_token,
            start_time,
            end_time,
            campaign_id,
            source,
            search_type,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广计划下的推广组报表基础数据查询(只有汇总数据，无分类类型)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10957

        :param subway_token: 权限验证信息
        :param start_time: 开始日期,格式yyyy-mm-dd
        :param end_time: 结束日期,格式yyyy-mm-dd
        :param campaign_id: 查询推广计划id
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5, 汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如：SEARCH,CAT
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.campadgroupbase.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "campaign_id": campaign_id,
                "source": source,
                "search_type": search_type,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_campadgroup_base_list"]
        )

    def taobao_simba_rpt_adgroupbase_get(
            self,
            subway_token,
            campaign_id,
            adgroup_id,
            start_time,
            end_time,
            search_type,
            source,
            nick='',
            page_no='1',
            page_size=''
    ):
        """
        推广组基础报表数据对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10958

        :param subway_token: 权限校验参数
        :param campaign_id: 推广计划id
        :param adgroup_id: 推广组id
        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT,<br/>定向投放：NOSEARCH）可以一次取多个例如：SEARCH,CAT
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5）可多选，以逗号分隔
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.adgroupbase.get",
            {
                "subway_token": subway_token,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "start_time": start_time,
                "end_time": end_time,
                "search_type": search_type,
                "source": source,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_adgroup_base_list"]
        )

    def taobao_simba_rpt_campaigneffect_get(
            self,
            subway_token,
            start_time,
            end_time,
            campaign_id,
            search_type,
            source,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广计划效果报表数据对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10959

        :param subway_token: 权限校验参数
        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd
        :param campaign_id: 推广计划id
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT,<br/>定向投放：NOSEARCH 全部：ALL）可以一次取多个例如：SEARCH,CAT
        :param source: 数据来源（站内：1，站外：2）可多选以逗号分隔，默认值为：1,2
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.campaigneffect.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "campaign_id": campaign_id,
                "search_type": search_type,
                "source": source,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_campaign_effect_list"]
        )

    def taobao_simba_rpt_adgroupcreativeeffect_get(
            self,
            subway_token,
            start_time,
            end_time,
            campaign_id,
            adgroup_id,
            source,
            search_type,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广组下的创意报表效果数据查询(汇总数据，不分类型)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10960

        :param subway_token: 权限验证信息
        :param start_time: 开始日期，格式yyyy-mm-dd
        :param end_time: 结束日期，格式yyyy-mm-dd
        :param campaign_id: 查询推广计划id
        :param adgroup_id: 推广组id
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如：SEARCH,CAT
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.adgroupcreativeeffect.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "source": source,
                "search_type": search_type,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_adgroupcreative_effect_list"]
        )

    def taobao_simba_rpt_campaignbase_get(
            self,
            subway_token,
            start_time,
            end_time,
            search_type,
            source,
            campaign_id,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广计划报表基础数据对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10961

        :param subway_token: 权限校验参数
        :param start_time: 开始时间，格式yyyy-mm-dd
        :param end_time: 结束时间，格式yyyy-mm-dd
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH 全部：ALL）可以一次取多个例如：SEARCH,CAT
        :param source: 数据来源（站内：1，站外：2）可多选以逗号分隔，默认值为：1,2
        :param campaign_id: 推广计划id
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.campaignbase.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "search_type": search_type,
                "source": source,
                "campaign_id": campaign_id,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_campaign_base_list"]
        )

    def taobao_simba_rpt_adgroupcreativebase_get(
            self,
            subway_token,
            start_time,
            end_time,
            campaign_id,
            adgroup_id,
            source,
            search_type,
            nick='',
            page_no='1',
            page_size='500'
    ):
        """
        推广组下创意报表基础数据查询(汇总数据，不分类型)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10962

        :param subway_token: 权限验证信息
        :param start_time: 开始日期，格式yyyy-mm-dd
        :param end_time: 结束日期，格式yyyy-mm-dd
        :param campaign_id: 查询推广计划id
        :param adgroup_id: 推广组id
        :param source: 数据来源（PC站内：1，PC站外：2，无线站内：4，无线站外 : 5，汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如1,2
        :param search_type: 报表类型（搜索：SEARCH,类目出价：CAT, 定向投放：NOSEARCH汇总：SUMMARY）SUMMARY必须单选，其他值可多选例如：SEARCH,CAT
        :param nick: 昵称
        :param page_no: 页码
        :param page_size: 每页大小
        """
        return self._top_request(
            "taobao.simba.rpt.adgroupcreativebase.get",
            {
                "subway_token": subway_token,
                "start_time": start_time,
                "end_time": end_time,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "source": source,
                "search_type": search_type,
                "nick": nick,
                "page_no": page_no,
                "page_size": page_size
            },
            result_processor=lambda x: x["rpt_adgroupcreative_base_list"]
        )

    def taobao_topats_simba_campkeywordeffect_get(
            self,
            campaign_id,
            time_slot,
            search_type,
            source
    ):
        """
        推广计划下的词报表效果数据查询
        推广计划下的词报表效果数据查询<br/>
        异步API使用方法，请查看：<a href='http://open.taobao.com/doc/detail.htm?id=30'>异步API使用说明</a><br/>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11058

        :param campaign_id: 查询推广计划ID
        :param time_slot: 时间参数（昨天：DAY、 前一周：7DAY 、前15天：15DAY 、前30天：30DAY 、前90天：90DAY）单选
        :param search_type: 报表类型。可选值：SEARCH（搜索）、CAT（类目出价）、 NOSEARCH（定向投放），可多选，如：SEARCH,CAT
        :param source: 数据来源。可选值：1（站内）、2（站外）、SUMMARY（汇总），其中SUMMARY必须单选，其它值可多选，如：1,2
        """
        return self._top_request(
            "taobao.topats.simba.campkeywordeffect.get",
            {
                "campaign_id": campaign_id,
                "time_slot": time_slot,
                "search_type": search_type,
                "source": source
            },
            result_processor=lambda x: x["task"]
        )

    def taobao_topats_simba_campkeywordbase_get(
            self,
            campaign_id,
            time_slot,
            search_type,
            source
    ):
        """
        推广计划下的词报表效果数据查询
        推广计划下的词报表效果数据查询<br/>
        异步API使用方法，请查看：<a href='http://open.taobao.com/doc/detail.htm?id=30'>异步API使用说明</a><br/>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11059

        :param campaign_id: 查询推广计划ID
        :param time_slot: 时间参数（昨天：DAY、 前一周：7DAY 、前15天：15DAY 、前30天：30DAY 、前90天：90DAY）单选
        :param search_type: 报表类型。可选值：SEARCH（搜索）、CAT（类目出价）、 NOSEARCH（定向投放），可多选，如：SEARCH,CAT
        :param source: 数据来源。可选值：1（站内）、2（站外）、SUMMARY（汇总），其中SUMMARY必须单选，其它值可多选，如：1,2
        """
        return self._top_request(
            "taobao.topats.simba.campkeywordbase.get",
            {
                "campaign_id": campaign_id,
                "time_slot": time_slot,
                "search_type": search_type,
                "source": source
            },
            result_processor=lambda x: x["task"]
        )

    def taobao_simba_keywords_qscore_get(
            self,
            adgroup_id,
            nick=''
    ):
        """
        取得一个推广组的所有关键词的质量得分或者根据关键词Id列表取得一组关键词的质量得分
        取得一个推广组的所有关键词的质量得分列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11091

        :param adgroup_id: 推广组Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.keywords.qscore.get",
            {
                "adgroup_id": adgroup_id,
                "nick": nick
            },
            result_processor=lambda x: x["keyword_qscore_list"]
        )

    def taobao_simba_adgroups_changed_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        分页获取修改的推广组ID和修改时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21249

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.adgroups.changed.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["adgroups"]
        )

    def taobao_simba_creatives_changed_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        分页获取修改过的广告创意ID和修改时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21255

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.creatives.changed.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["creatives"]
        )

    def taobao_simba_keywords_changed_get(
            self,
            start_time,
            nick='',
            page_size='200',
            page_no='1'
    ):
        """
        分页获取修改过的关键词ID、宝贝id、修改时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21256

        :param start_time: 得到此时间点之后的数据，不能大于一个月
        :param nick: 主人昵称
        :param page_size: 返回的每页数据量大小,默认200最大1000
        :param page_no: 返回的第几页数据，默认为1
        """
        return self._top_request(
            "taobao.simba.keywords.changed.get",
            {
                "start_time": start_time,
                "nick": nick,
                "page_size": page_size,
                "page_no": page_no
            },
            result_processor=lambda x: x["keywords"]
        )

    def taobao_simba_adgroupsbycampaignid_get(
            self,
            page_size,
            page_no,
            nick='',
            campaign_id=''
    ):
        """
        批量得到推广计划下的推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21676

        :param page_size: 页尺寸，最大200，如果入参adgroup_ids有传入值，则page_size和page_no值不起作用。如果adgrpup_ids为空而campaign_id有值，此时page_size和page_no值才是返回的页数据大小和页码
        :param page_no: 页码，从1开始
        :param nick: 主人昵称
        :param campaign_id: 推广计划Id
        """
        return self._top_request(
            "taobao.simba.adgroupsbycampaignid.get",
            {
                "page_size": page_size,
                "page_no": page_no,
                "nick": nick,
                "campaign_id": campaign_id
            },
            result_processor=lambda x: x["adgroups"]
        )

    def taobao_simba_adgroupsbyadgroupids_get(
            self,
            page_size,
            page_no,
            nick='',
            adgroup_ids=''
    ):
        """
        批量得到推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21679

        :param page_size: 页尺寸，最大200，如果入参adgroup_ids有传入值，则page_size和page_no值不起作用。如果adgrpup_ids为空而campaign_id有值，此时page_size和page_no值才是返回的页数据大小和页码
        :param page_no: 页码，从1开始
        :param nick: 主人昵称
        :param adgroup_ids: 推广组Id列表
        """
        return self._top_request(
            "taobao.simba.adgroupsbyadgroupids.get",
            {
                "page_size": page_size,
                "page_no": page_no,
                "nick": nick,
                "adgroup_ids": adgroup_ids
            },
            result_processor=lambda x: x["adgroups"]
        )

    def taobao_simba_adgroup_onlineitemsvon_get(
            self,
            nick='',
            order_field='',
            order_by='',
            page_size='',
            page_no='',
            product_id='101001005'
    ):
        """
        获取用户上架在线销售的全部宝贝
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21680

        :param nick: 主人昵称
        :param order_field: 排序字段，starts：按开始时间排序bidCount:按销量排序
        :param order_by: 排序，true:降序， false:升序
        :param page_size: 页尺寸，最大200
        :param page_no: 页码，从1开始,最大50。最大只能获取1W个宝贝
        :param product_id: 推广单元类型 101001005代表标准推广，101001014代表销量明星推广
        """
        return self._top_request(
            "taobao.simba.adgroup.onlineitemsvon.get",
            {
                "nick": nick,
                "order_field": order_field,
                "order_by": order_by,
                "page_size": page_size,
                "page_no": page_no,
                "product_id": product_id
            },
            result_processor=lambda x: x["page_item"]
        )

    def taobao_simba_keywordsvon_add(
            self,
            adgroup_id,
            keyword_prices,
            nick=''
    ):
        """
        创建一批关键词
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21681

        :param adgroup_id: 推广组id
        :param keyword_prices: 关键词，出价和匹配方式json字符串，word:词，不能有一些特殊字符。maxPrice：价格，是整数，以“分”为单位，不能小于5，不能大于日限额,当使用默认出价时必须将这个值设置为0。; isDefaultPrice：是否使用默认出价，只能是0，1(0代表不使用，1代表使用)。matchscope只能是1,2,4（1代表精确匹配，2代表子串匹配，4代表广泛匹配）。
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.keywordsvon.add",
            {
                "adgroup_id": adgroup_id,
                "keyword_prices": keyword_prices,
                "nick": nick
            },
            result_processor=lambda x: x["keywords"]
        )

    def taobao_simba_keywordsbyadgroupid_get(
            self,
            nick='',
            adgroup_id=''
    ):
        """
        取得一个推广组的所有关键词
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21682

        :param nick: 主人昵称
        :param adgroup_id: 推广组Id
        """
        return self._top_request(
            "taobao.simba.keywordsbyadgroupid.get",
            {
                "nick": nick,
                "adgroup_id": adgroup_id
            },
            result_processor=lambda x: x["keywords"]
        )

    def taobao_simba_keywordsbykeywordids_get(
            self,
            nick='',
            keyword_ids=''
    ):
        """
        根据一个关键词Id列表取得一组关键词
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21683

        :param nick: 主人昵称
        :param keyword_ids: 关键词Id数组，最多200个；
        """
        return self._top_request(
            "taobao.simba.keywordsbykeywordids.get",
            {
                "nick": nick,
                "keyword_ids": keyword_ids
            },
            result_processor=lambda x: x["keywords"]
        )

    def taobao_simba_keywordscat_qscore_get(
            self,
            adgroup_id,
            nick=''
    ):
        """
        取得一个推广组的所有关键词和类目出价的质量得分
        取得一个推广组的所有关键词和类目出价的质量得分列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21684

        :param adgroup_id: 推广组Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.keywordscat.qscore.get",
            {
                "adgroup_id": adgroup_id,
                "nick": nick
            },
            result_processor=lambda x: x["qscore"]
        )

    def taobao_simba_keywords_pricevon_set(
            self,
            keywordid_prices,
            nick=''
    ):
        """
        设置一批关键词的出价
        设置一批关键词的出价，包含无线独立出价
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21685

        :param keywordid_prices: 关键词ID，出价和匹配方式json字符串，keywordId:词ID，整数。maxPrice：价格，是整数，以“分”为单位，不能小于5，不能大于日限额,也不能大于99.99元,当使用默认出价时必须将这个值设置为0。; isDefaultPrice：是否使用默认出价，只能是0。matchscope只能是1,2,4（1代表精确匹配，4代表广泛匹配），maxMobilePrice：代表无线出价（如果是0，则代表无线出价=PC*无线溢价），mobileIsDefaultPrice代表无线出价是否采用pc*无线溢价，只能传0
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.keywords.pricevon.set",
            {
                "keywordid_prices": keywordid_prices,
                "nick": nick
            },
            result_processor=lambda x: x["keywords"]
        )

    def taobao_simba_insight_catsforecastnew_get(
            self,
            bidword_list
    ):
        """
        获取词的相关类目预测数据
        根据给定的词，预测这些词的相关类目
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23571

        :param bidword_list: 需要查询的词列表
        """
        return self._top_request(
            "taobao.simba.insight.catsforecastnew.get",
            {
                "bidword_list": bidword_list
            },
            result_processor=lambda x: x["category_forecast_list"]
        )

    def taobao_simba_insight_relatedwords_get(
            self,
            bidword_list,
            number
    ):
        """
        获取词的相关词
        获取给定词的若干相关词，返回结果中越相关的权重越大，排在越前面，根据number参数对返回结果进行截断。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23573

        :param bidword_list: 要查询的词列表
        :param number: 表示返回数据的条数
        """
        return self._top_request(
            "taobao.simba.insight.relatedwords.get",
            {
                "bidword_list": bidword_list,
                "number": number
            },
            result_processor=lambda x: x["related_words_result_list"]
        )

    def taobao_simba_insight_catstopwordnew_get(
            self,
            cat_id,
            start_date,
            end_date,
            dimension,
            page_size
    ):
        """
        获取类目下最热门的词
        按照某个维度，查询某个类目下最热门的词，维度有点击，展现，花费，点击率等，具体可以按哪些字段进行排序，参考参数说明，比如选择了impression，则返回该类目下展现量最高那几个词。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23574

        :param cat_id: 类目id
        :param start_date: 查询开始时间，格式必须为：yyyy-MM-dd
        :param end_date: 查询截止时间，格式只能是：yyyy-MM-dd
        :param dimension: 表示查询的维度，比如选择click，则查询该类目下点击量最大的词，可供选择的值有：impression, click, cost, ctr, cpc, coverage, transactiontotal, transactionshippingtotal, favtotal, roi
        :param page_size: 返回前多少条数据
        """
        return self._top_request(
            "taobao.simba.insight.catstopwordnew.get",
            {
                "cat_id": cat_id,
                "start_date": start_date,
                "end_date": end_date,
                "dimension": dimension,
                "page_size": page_size
            },
            result_processor=lambda x: x["topword_data_list"]
        )

    def taobao_simba_insight_catsworddata_get(
            self,
            cat_id,
            bidword_list,
            start_date,
            end_date
    ):
        """
        获取类目下关键词的数据
        获取给定词在给定类目下的详细数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23575

        :param cat_id: 类目id
        :param bidword_list: 需要查询的关键词列表，最大长度100。
        :param start_date: 开始时间，格式只能为：yyyy-MM-dd
        :param end_date: 结束时间，格式只能为：yyyy-MM-dd
        """
        return self._top_request(
            "taobao.simba.insight.catsworddata.get",
            {
                "cat_id": cat_id,
                "bidword_list": bidword_list,
                "start_date": start_date,
                "end_date": end_date
            },
            result_processor=lambda x: x["catword_data_list"]
        )

    def taobao_simba_insight_wordsdata_get(
            self,
            bidword_list,
            start_date,
            end_date
    ):
        """
        获取关键词的大盘数据
        获取关键词的详细数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23576

        :param bidword_list: 关键词列表，最多可传100个。
        :param start_date: 开始时间
        :param end_date: 结束时间
        """
        return self._top_request(
            "taobao.simba.insight.wordsdata.get",
            {
                "bidword_list": bidword_list,
                "start_date": start_date,
                "end_date": end_date
            },
            result_processor=lambda x: x["word_data_list"]
        )

    def taobao_simba_insight_wordssubdata_get(
            self,
            bidword_list,
            start_date,
            end_date
    ):
        """
        获取关键词按流量细分的数据
        获取关键词按流量进行细分的数据，返回结果中network表示流量的来源，意义如下：1->PC站内，2->PC站外,4->无线站内 5->无线站外
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23577

        :param bidword_list: 关键词列表
        :param start_date: 开始时间
        :param end_date: 结束时间
        """
        return self._top_request(
            "taobao.simba.insight.wordssubdata.get",
            {
                "bidword_list": bidword_list,
                "start_date": start_date,
                "end_date": end_date
            },
            result_processor=lambda x: x["word_subdata_list"]
        )

    def taobao_simba_insight_wordsareadata_get(
            self,
            bidword,
            start_date,
            end_date
    ):
        """
        获取关键词按地域进行细分的数据
        获取关键词按地域细分的详细数据，目前地域只能细化到省级别，返回的结果中包括市，是为了方便以后扩展，目前结果中市的值等于省。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23578

        :param bidword: 关键词
        :param start_date: 开始时间，格式：yyyy-MM-dd
        :param end_date: 结束时间，格式：yyyy-MM-dd
        """
        return self._top_request(
            "taobao.simba.insight.wordsareadata.get",
            {
                "bidword": bidword,
                "start_date": start_date,
                "end_date": end_date
            },
            result_processor=lambda x: x["word_areadata_list"]
        )

    def taobao_simba_insight_wordspricedata_get(
            self,
            bidword,
            start_date,
            end_date
    ):
        """
        关键词按竞价区间的分布数据
        获取关键词按竞价区间进行细分的数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23579

        :param bidword: 关键词
        :param start_date: 开始时间，格式：yyyy-MM-dd
        :param end_date: 结束时间，格式：yyyy-MM-dd
        """
        return self._top_request(
            "taobao.simba.insight.wordspricedata.get",
            {
                "bidword": bidword,
                "start_date": start_date,
                "end_date": end_date
            },
            result_processor=lambda x: x["word_pricedata_list"]
        )

    def taobao_simba_insight_catsinfo_get(
            self,
            type,
            category_id_list=''
    ):
        """
        类目信息获取
        获取类目信息，此接口既提供所有顶级类目的查询，又提供给定类目id自身信息和子类目信息的查询，所以可以根据此接口逐层获取所有的类目信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23580

        :param type: 表示请求的类型：0表示请求所有顶级类目的信息，这时可以忽略第二个参数，1表示获取给定的类目id的详细信息，2表示获取给定类目id的所有子类目的详细信息
        :param category_id_list: 需要查询的类目id
        """
        return self._top_request(
            "taobao.simba.insight.catsinfo.get",
            {
                "type": type,
                "category_id_list": category_id_list
            },
            result_processor=lambda x: x["category_info_list"]
        )

    def taobao_simba_insight_catsdata_get(
            self,
            category_id_list,
            start_date,
            end_date
    ):
        """
        获取类目的大盘数据
        根据类目id获取类目的大盘数据，包括展现指数，点击指数，点击率，本次提供的insight相关的其它接口的都是这种情况。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23581

        :param category_id_list: 表示要查询的类目id
        :param start_date: 开始时间，格式：yyyy-MM-dd
        :param end_date: 查询截止时间，格式：yyyy-MM-dd
        """
        return self._top_request(
            "taobao.simba.insight.catsdata.get",
            {
                "category_id_list": category_id_list,
                "start_date": start_date,
                "end_date": end_date
            },
            result_processor=lambda x: x["cat_data_list"]
        )

    def taobao_simba_rpt_targetingtageffect_get(
            self,
            start_time,
            end_time,
            nick='',
            campaign_id='',
            adgroup_id='',
            page_size='',
            page_number=''
    ):
        """
        获取定向效果报表数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24977

        :param start_time: 起始时间
        :param end_time: 终止时间 ,必须小于今天
        :param nick: 被操作者昵称
        :param campaign_id: 计划id
        :param adgroup_id: 推广组id
        :param page_size: 页面大小
        :param page_number: 页码
        """
        return self._top_request(
            "taobao.simba.rpt.targetingtageffect.get",
            {
                "start_time": start_time,
                "end_time": end_time,
                "nick": nick,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "page_size": page_size,
                "page_number": page_number
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_rpt_targetingtagbase_get(
            self,
            start_time,
            end_time,
            nick='',
            campaign_id='',
            adgroup_id='',
            page_size='',
            page_number=''
    ):
        """
        定向基础报表
        获取定向基础报表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24993

        :param start_time: 起始时间
        :param end_time: 结束时间
        :param nick: 被操作者昵称
        :param campaign_id: 计划id
        :param adgroup_id: 推广组id
        :param page_size: 分页大小
        :param page_number: 页码
        """
        return self._top_request(
            "taobao.simba.rpt.targetingtagbase.get",
            {
                "start_time": start_time,
                "end_time": end_time,
                "nick": nick,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "page_size": page_size,
                "page_number": page_number
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_rtrpt_creative_get(
            self,
            campaign_id,
            adgroup_id,
            the_date,
            nick=''
    ):
        """
        获取创意实时报表数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25050

        :param campaign_id: 推广计划id
        :param adgroup_id: 推广组id
        :param the_date: 日期，格式yyyy-mm-dd
        :param nick: 用户名
        """
        return self._top_request(
            "taobao.simba.rtrpt.creative.get",
            {
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "the_date": the_date,
                "nick": nick
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_rtrpt_campaign_get(
            self,
            the_date,
            nick=''
    ):
        """
        获取推广计划实时报表数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25051

        :param the_date: 日期，格式yyyy-mm-dd
        :param nick: 用户名
        """
        return self._top_request(
            "taobao.simba.rtrpt.campaign.get",
            {
                "the_date": the_date,
                "nick": nick
            },
            result_processor=lambda x: x["resultss"]
        )

    def taobao_simba_rtrpt_bidword_get(
            self,
            campaign_id,
            adgroup_id,
            the_date,
            nick=''
    ):
        """
        获取推广词实时报表数据
        获取推广词报表数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25052

        :param campaign_id: 推广计划id
        :param adgroup_id: 推广组id
        :param the_date: 日期，格式yyyy-mm-dd
        :param nick: 用户名
        """
        return self._top_request(
            "taobao.simba.rtrpt.bidword.get",
            {
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "the_date": the_date,
                "nick": nick
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_rtrpt_adgroup_get(
            self,
            campaign_id,
            the_date,
            nick='',
            page_size='',
            page_number=''
    ):
        """
        获取推广组实时报表数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25053

        :param campaign_id: 推广计划id
        :param the_date: 日期，格式yyyy-mm-dd
        :param nick: 用户名
        :param page_size: 每页大小
        :param page_number: 页码
        """
        return self._top_request(
            "taobao.simba.rtrpt.adgroup.get",
            {
                "campaign_id": campaign_id,
                "the_date": the_date,
                "nick": nick,
                "page_size": page_size,
                "page_number": page_number
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_rtrpt_cust_get(
            self,
            the_date,
            nick=''
    ):
        """
        获取账户实时报表数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25054

        :param the_date: 日期，格式yyyy-mm-dd
        :param nick: 昵称
        """
        return self._top_request(
            "taobao.simba.rtrpt.cust.get",
            {
                "the_date": the_date,
                "nick": nick
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_keywords_qscore_split_get(
            self,
            ad_group_id,
            bidword_ids,
            nick=''
    ):
        """
        新质量分服务
        获取关键词新的质量分
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25653

        :param ad_group_id: 推广组id
        :param bidword_ids: 词id数组（最多批量获取20个）
        :param nick: 账号昵称
        """
        return self._top_request(
            "taobao.simba.keywords.qscore.split.get",
            {
                "ad_group_id": ad_group_id,
                "bidword_ids": bidword_ids,
                "nick": nick
            }
        )

    def taobao_simba_adgroup_mobilediscount_delete(
            self,
            adgroup_ids,
            nick=''
    ):
        """
        批量删除adgroup的移动溢价
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25655

        :param adgroup_ids: adgroup主键数组（批量最多支持200个）
        :param nick: 昵称
        """
        return self._top_request(
            "taobao.simba.adgroup.mobilediscount.delete",
            {
                "adgroup_ids": adgroup_ids,
                "nick": nick
            }
        )

    def taobao_simba_adgroup_mobilediscount_update(
            self,
            adgroup_ids,
            mobile_discount,
            nick=''
    ):
        """
        对推广组进行单独移动溢价
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25656

        :param adgroup_ids: 推广组id数组(推广组id集合元素个数在1-200个之间，推广组id需要在同一个推广计划中)
        :param mobile_discount: 折扣（折扣值在1-400之间）
        :param nick: 昵称
        """
        return self._top_request(
            "taobao.simba.adgroup.mobilediscount.update",
            {
                "adgroup_ids": adgroup_ids,
                "mobile_discount": mobile_discount,
                "nick": nick
            }
        )

    def taobao_simba_keywords_realtime_ranking_batch_get(
            self,
            nick,
            ad_group_id,
            bidword_ids
    ):
        """
        获取关键词的新版实时排名
        根据关键词ID获取关键词的新版实时排名
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26824

        :param nick: 旺旺名称
        :param ad_group_id: adgroupId
        :param bidword_ids: 关键词列表集合,id用半角逗号分割，一次最多20个
        """
        return self._top_request(
            "taobao.simba.keywords.realtime.ranking.batch.get",
            {
                "nick": nick,
                "ad_group_id": ad_group_id,
                "bidword_ids": bidword_ids
            }
        )

    def taobao_simba_serchcrowd_get(
            self,
            nick,
            adgroup_id
    ):
        """
        根据推广单元id获取搜索溢价人群
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27294

        :param nick: 被操作者的淘宝昵称
        :param adgroup_id: 推广单元id
        """
        return self._top_request(
            "taobao.simba.serchcrowd.get",
            {
                "nick": nick,
                "adgroup_id": adgroup_id
            },
            result_processor=lambda x: x["adgrouptargetingtags"]
        )

    def taobao_simba_serchcrowd_batch_delete(
            self,
            nick,
            adgroup_crowd_ids,
            sub_nick=''
    ):
        """
        单品搜索人群批量取消溢价
        删除单品搜索人群溢价功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27295

        :param nick: 被操作者的淘宝昵称
        :param adgroup_crowd_ids: 需要删除的人群id
        :param sub_nick: 子帐号nick
        """
        return self._top_request(
            "taobao.simba.serchcrowd.batch.delete",
            {
                "nick": nick,
                "adgroup_crowd_ids": adgroup_crowd_ids,
                "sub_nick": sub_nick
            }
        )

    def taobao_simba_searchtagtemplate_get(
            self,
            nick='starshop001',
            sub_nick=''
    ):
        """
        获取搜索人群TOP用户可添加人群信息
        获取搜索人群用户可添加人群信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27296

        :param nick: 被操作者的淘宝昵称
        :param sub_nick: 子帐号nick
        """
        return self._top_request(
            "taobao.simba.searchtagtemplate.get",
            {
                "nick": nick,
                "sub_nick": sub_nick
            },
            result_processor=lambda x: x["template_list"]
        )

    def taobao_simba_serchcrowd_state_batch_update(
            self,
            nick,
            adgroup_crowd_ids,
            adgroup_id,
            state=''
    ):
        """
        单品搜索人群修改状态
        暂停或启用单品推广搜索人群溢价
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27297

        :param nick: 被操作者的淘宝昵称
        :param adgroup_crowd_ids: 需要修改出价的人群包id,批量传入时用,分割
        :param adgroup_id: 推广单元id
        :param state: 人群状态,0:暂停;1:启用
        """
        return self._top_request(
            "taobao.simba.serchcrowd.state.batch.update",
            {
                "nick": nick,
                "adgroup_crowd_ids": adgroup_crowd_ids,
                "adgroup_id": adgroup_id,
                "state": state
            }
        )

    def taobao_simba_searchcrowd_batch_add(
            self,
            nick,
            adgroup_id,
            adgroup_targeting_tags
    ):
        """
        推广单元增加搜索人群
        推广单元新增搜索人群
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27298

        :param nick: 被操作者的淘宝昵称
        :param adgroup_id: 推广单元id
        :param adgroup_targeting_tags: 新增人群信息,批量接口,入参为list,溢价(discount)范围为[105,400]
        """
        return self._top_request(
            "taobao.simba.searchcrowd.batch.add",
            {
                "nick": nick,
                "adgroup_id": adgroup_id,
                "adgroup_targeting_tags": adgroup_targeting_tags
            },
            result_processor=lambda x: x["adgrouptargetingtags"]
        )

    def taobao_simba_serchcrowd_price_batch_update(
            self,
            nick,
            adgroup_crowd_ids,
            adgroup_id,
            sub_nick='',
            discount=''
    ):
        """
        单品推广搜索人群修改溢价
        单品推广搜索人群修改溢价, 不支持跨推广单元修改
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27299

        :param nick: 被操作者的淘宝昵称
        :param adgroup_crowd_ids: 需要修改出价的人群包id,批量传入的时候用,分割
        :param adgroup_id: 推广单元id
        :param sub_nick: 子帐号nick
        :param discount: 人群溢价比例，溢价范围[5,300]
        """
        return self._top_request(
            "taobao.simba.serchcrowd.price.batch.update",
            {
                "nick": nick,
                "adgroup_crowd_ids": adgroup_crowd_ids,
                "adgroup_id": adgroup_id,
                "sub_nick": sub_nick,
                "discount": discount
            },
            result_processor=lambda x: x["adgrouptargetingtags"]
        )

    def taobao_simba_rtrpt_targetingtag_get(
            self,
            nick,
            campaign_id,
            adgroup_id,
            the_date,
            traffic_type
    ):
        """
        搜索人群实时报表
        获取搜搜人群实时报表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27346

        :param nick: 旺旺名称
        :param campaign_id: 推广计划id
        :param adgroup_id: 推广单元id
        :param the_date: 日期，格式yyyy-mm-dd
        :param traffic_type: 流量类型 1: PC站内, 2: PC站外 , 4: 无线站内, 5: 无线站外,支持多种一起查询,如1,2,4,5
        """
        return self._top_request(
            "taobao.simba.rtrpt.targetingtag.get",
            {
                "nick": nick,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "the_date": the_date,
                "traffic_type": traffic_type
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_rpt_targetingtag_get(
            self,
            nick='',
            campaign_id='',
            adgroup_id='',
            start_time='',
            end_time='',
            traffic_type=''
    ):
        """
        搜索人群离线报表
        获取搜搜人群实时报表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27371

        :param nick: 用户旺旺名称
        :param campaign_id: 推广计划id
        :param adgroup_id: 推广单元id
        :param start_time: 开始时间
        :param end_time: 结束时间
        :param traffic_type: 流量类型 1: PC站内, 2: PC站外 , 4: 无线站内, 5: 无线站外,支持多种一起查询,如1,2,4,5
        """
        return self._top_request(
            "taobao.simba.rpt.targetingtag.get",
            {
                "nick": nick,
                "campaign_id": campaign_id,
                "adgroup_id": adgroup_id,
                "start_time": start_time,
                "end_time": end_time,
                "traffic_type": traffic_type
            },
            result_processor=lambda x: x["results"]
        )

    def taobao_simba_customers_sid_get(
            self
    ):
        """
        查看功能权限
        查询用户是否拥有某个功能权限
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29157

        """
        return self._top_request(
            "taobao.simba.customers.sid.get"
        )

    def taobao_simba_keyword_findbyids(
            self,
            bidword_ids=''
    ):
        """
        （新）根据一堆关键词ids获取关键词
        根据一个关键词Id列表取得一组关键词
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43330

        :param bidword_ids: 关键词ids
        """
        return self._top_request(
            "taobao.simba.keyword.findbyids",
            {
                "bidword_ids": bidword_ids
            }
        )

    def taobao_simba_keyword_add(
            self,
            bidwords,
            adgroup_id=''
    ):
        """
        （新）关键词新增接口
        （新）关键词更新相关接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43342

        :param bidwords: 关键词相关信息
        :param adgroup_id: 推广单元id
        """
        return self._top_request(
            "taobao.simba.keyword.add",
            {
                "bidwords": bidwords,
                "adgroup_id": adgroup_id
            }
        )

    def taobao_simba_keyword_update(
            self,
            bidwords
    ):
        """
        （新）关键词更新相关接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43343

        :param bidwords: 关键词相关信息
        """
        return self._top_request(
            "taobao.simba.keyword.update",
            {
                "bidwords": bidwords
            }
        )

    def taobao_simba_keyword_findbyadgroupid(
            self,
            adgroup_id=''
    ):
        """
        根据推广单元id获取关键词
        根据一个关键词Id列表取得一组关键词
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43344

        :param adgroup_id: 推广单元id
        """
        return self._top_request(
            "taobao.simba.keyword.findbyadgroupid",
            {
                "adgroup_id": adgroup_id
            }
        )

    def taobao_simba_salestar_campaign_budget_update(
            self,
            campaign_id,
            budget=''
    ):
        """
        销量明星跟新预算相关接口
        更新一个推广计划的日限额
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43351

        :param campaign_id: 推广计划Id
        :param budget: 如果为空则取消限额；否则必须为整数，单位是元，不得小于30；
        """
        return self._top_request(
            "taobao.simba.salestar.campaign.budget.update",
            {
                "campaign_id": campaign_id,
                "budget": budget
            },
            result_processor=lambda x: x["campaign_budget"]
        )

    def taobao_simba_salestar_adgroup_update(
            self,
            adgroup_id,
            online_status=''
    ):
        """
        销量明星更新一个推广组的信息
        更新一个推广组的信息，可以设置 是否上线
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43354

        :param adgroup_id: 推广组Id
        :param online_status: 用户设置的上下线状态 offline-下线(暂停竞价)； online-上线；默认为online
        """
        return self._top_request(
            "taobao.simba.salestar.adgroup.update",
            {
                "adgroup_id": adgroup_id,
                "online_status": online_status
            },
            result_processor=lambda x: x["adgroup"]
        )

    def taobao_simba_salestar_keywords_qscore_split_get(
            self,
            ad_group_id,
            bidword_ids,
            nick=''
    ):
        """
        (新)销量明星质量分相关接口
        获取关键词新的质量分
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43356

        :param ad_group_id: 推广组id
        :param bidword_ids: 词id数组（最多批量获取20个）
        :param nick: 账号昵称
        """
        return self._top_request(
            "taobao.simba.salestar.keywords.qscore.split.get",
            {
                "ad_group_id": ad_group_id,
                "bidword_ids": bidword_ids,
                "nick": nick
            }
        )

    def taobao_simba_salestar_adgroup_delete(
            self,
            adgroup_id,
            nick=''
    ):
        """
        (新)销量明星删除推广单元接口
        删除一个推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43357

        :param adgroup_id: 推广组Id
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.salestar.adgroup.delete",
            {
                "adgroup_id": adgroup_id,
                "nick": nick
            }
        )

    def taobao_simba_salestar_adgroup_findbycampid(
            self,
            page_size,
            page_no,
            campaign_id=''
    ):
        """
        (销量明星)批量获取推广计划下的推广组信息
        批量得到推广计划下的推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43358

        :param page_size: 页尺寸，最大200，如果入参adgroup_ids有传入值，则page_size和page_no值不起作用。如果adgrpup_ids为空而campaign_id有值，此时page_size和page_no值才是返回的页数据大小和页码
        :param page_no: 页码，从1开始
        :param campaign_id: 推广计划Id
        """
        return self._top_request(
            "taobao.simba.salestar.adgroup.findbycampid",
            {
                "page_size": page_size,
                "page_no": page_no,
                "campaign_id": campaign_id
            },
            result_processor=lambda x: x["adgroups"]
        )

    def taobao_simba_salestar_creatives_get(
            self,
            nick='',
            creative_ids='',
            adgroup_id=''
    ):
        """
        （新）批量获取创意
        取得一个推广组的所有创意或者根据一个创意Id列表取得一组创意；<br/>如果同时提供了推广组Id和创意id列表，则优先使用推广组Id；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43359

        :param nick: 主人昵称
        :param creative_ids: 创意Id数组，最多200个
        :param adgroup_id: 推广组Id
        """
        return self._top_request(
            "taobao.simba.salestar.creatives.get",
            {
                "nick": nick,
                "creative_ids": creative_ids,
                "adgroup_id": adgroup_id
            },
            result_processor=lambda x: x["creatives"]
        )

    def taobao_simba_salestar_creative_add(
            self,
            adgroup_id,
            title,
            img_url,
            nick=''
    ):
        """
        （新）新建创意
        创建一个创意
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43360

        :param adgroup_id: 推广组Id
        :param title: 创意标题，最多20个汉字
        :param img_url: 创意图片地址，必须是推广组对应商品的图片之一
        :param nick: 主人昵称
        """
        return self._top_request(
            "taobao.simba.salestar.creative.add",
            {
                "adgroup_id": adgroup_id,
                "title": title,
                "img_url": img_url,
                "nick": nick
            },
            result_processor=lambda x: x["creative"]
        )

    def taobao_simba_salestar_creative_update(
            self,
            adgroup_id,
            creative_id,
            title,
            img_url,
            picture_id=''
    ):
        """
        销量明星更新创意相关接口
        更新一个创意的信息，可以设置创意标题、创意图片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43361

        :param adgroup_id: 推广组Id
        :param creative_id: 创意Id
        :param title: 创意标题，最多20个汉字
        :param img_url: 创意图片地址，必须是推广组对应商品的图片之一
        :param picture_id: 如果用户开通了创意本地上传图片功能的，可以使用该用户图片空间的图片来修改创意，pictureId为图片空间中图片的pictureId，img_url为图片空间中图片链接地址，如果是使用的主图或副图修改创意，则pictureId必须为空
        """
        return self._top_request(
            "taobao.simba.salestar.creative.update",
            {
                "adgroup_id": adgroup_id,
                "creative_id": creative_id,
                "title": title,
                "img_url": img_url,
                "picture_id": picture_id
            },
            result_processor=lambda x: x["creativerecord"]
        )

    def taobao_simba_salestar_creative_delete(
            self,
            creative_id
    ):
        """
        (新)销量明星删除创意相关接口
        删除一个创意
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43363

        :param creative_id: 创意Id
        """
        return self._top_request(
            "taobao.simba.salestar.creative.delete",
            {
                "creative_id": creative_id
            },
            result_processor=lambda x: x["creative"]
        )

    def taobao_simba_salestar_adgroup_add(
            self,
            campaign_id,
            item_id,
            title,
            img_url
    ):
        """
        (新)创建一个推广组
        创建一个推广组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43366

        :param campaign_id: 推广计划Id
        :param item_id: 商品Id
        :param title: 创意标题，最多20个汉字
        :param img_url: 创意图片地址，必须是商品的图片之一
        """
        return self._top_request(
            "taobao.simba.salestar.adgroup.add",
            {
                "campaign_id": campaign_id,
                "item_id": item_id,
                "title": title,
                "img_url": img_url
            },
            result_processor=lambda x: x["adgroup"]
        )

    def taobao_simba_salestar_keywords_delete(
            self,
            bidword_ids=''
    ):
        """
        销量明星关键词删除
        （新）关键词删除相关接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43394

        :param bidword_ids: 关键词ids
        """
        return self._top_request(
            "taobao.simba.salestar.keywords.delete",
            {
                "bidword_ids": bidword_ids
            }
        )

    def taobao_simba_salestar_keywords_recommend_get(
            self,
            adgroup_id,
            product_id='101001005'
    ):
        """
        销量明星api相关接口
        取得一个推广组的推荐关键词列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43400

        :param adgroup_id: 推广组ID
        :param product_id: 产品类型101001005代表标准推广，101001014代表销量明星
        """
        return self._top_request(
            "taobao.simba.salestar.keywords.recommend.get",
            {
                "adgroup_id": adgroup_id,
                "product_id": product_id
            },
            result_processor=lambda x: x["recommend_words"]
        )


class TbJiPiao(DingTalkBaseAPI):
    """
    机票API
    """

    def taobao_alitrip_flightchange_add(
            self,
            flight_change_data_do
    ):
        """
        航变信息录入接口
        代理商调用航变录入接口,
        简要描述:完成航变信息的自动录入后飞猪机票平台会发生的动作是匹配所有该代理人的订单,如果接口参数指定了飞猪机票订单号，发生的动作是匹配该代理人的指定订单；
        找到与航变航班相关的订单给旅客下发航变短信并出发IVR自动外呼旅客。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27819

        :param flight_change_data_do: 录入参数类
        """
        return self._top_request(
            "taobao.alitrip.flightchange.add",
            {
                "flight_change_data_do": flight_change_data_do
            }
        )

    def taobao_alitrip_flightchange_get(
            self,
            search_option=None
    ):
        """
        获取航变信息
        查询航变是为了两个目的，阿里旅行抓取到未确认航变的航变信息源时可以由代理确认航变，同时对于确认航变的航变信息也共享给代理人做本体业务使用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27822

        :param search_option: 查询信息封装
        """
        return self._top_request(
            "taobao.alitrip.flightchange.get",
            {
                "search_option": search_option
            },
            result_processor=lambda x: x["result_d_o"]
        )

    def taobao_alitrip_ie_agent_shopping_push(
            self,
            param0=None
    ):
        """
        国际机票大卖家Shopping推送
        用于国际机票大卖家主动推送Shopping结果更新缓存报价。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29496

        :param param0: 政策推送结构体
        """
        return self._top_request(
            "taobao.alitrip.ie.agent.shopping.push",
            {
                "param0": param0
            }
        )

    def alitrip_tripvp_agent_order_issue(
            self,
            agent_id,
            issue_pro_order_vo
    ):
        """
        廉航辅营正向订单出货接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30161

        :param agent_id: 代理商ID
        :param issue_pro_order_vo: 出货通知信息
        """
        return self._top_request(
            "alitrip.tripvp.agent.order.issue",
            {
                "agent_id": agent_id,
                "issue_pro_order_vo": issue_pro_order_vo
            },
            result_processor=lambda x: x["is_success"]
        )

    def alitrip_tripvp_agent_order_get(
            self,
            agent_id,
            trade_order_id
    ):
        """
        廉航辅营正向订单查询详情接口
        【国际机票】查询辅营订单详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30242

        :param agent_id: 代理商ID
        :param trade_order_id: 辅营的订单号
        """
        return self._top_request(
            "alitrip.tripvp.agent.order.get",
            {
                "agent_id": agent_id,
                "trade_order_id": trade_order_id
            }
        )

    def taobao_alitrip_totoro_auxproduct_delete(
            self,
            del_aux_product_rq
    ):
        """
        廉航辅营产品删除
        廉航辅营产品删除接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31192

        :param del_aux_product_rq: 廉航辅营产品删除请求
        """
        return self._top_request(
            "taobao.alitrip.totoro.auxproduct.delete",
            {
                "del_aux_product_rq": del_aux_product_rq
            }
        )

    def taobao_alitrip_totoro_auxproduct_push(
            self,
            push_aux_products_rq
    ):
        """
        廉航辅营产品投放
        廉航辅营产品投放接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31193

        :param push_aux_products_rq: 廉航辅营产品推送请求
        """
        return self._top_request(
            "taobao.alitrip.totoro.auxproduct.push",
            {
                "push_aux_products_rq": push_aux_products_rq
            }
        )

    def taobao_alitrip_ie_agent_refund_new_fillconfirmfee(
            self,
            param_refund_order_fill_confirm_fee_rq=None
    ):
        """
        新模型-回填申请单费用
        1. 回填退票费用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37897

        :param param_refund_order_fill_confirm_fee_rq: 请求
        """
        return self._top_request(
            "taobao.alitrip.ie.agent.refund.new.fillconfirmfee",
            {
                "param_refund_order_fill_confirm_fee_rq": param_refund_order_fill_confirm_fee_rq
            }
        )


class TbONSXiaoXiFuWu(DingTalkBaseAPI):
    """
    ONS消息服务
    """

    def taobao_jushita_jms_user_get(
            self,
            user_nick
    ):
        """
        查询某个用户是否同步消息
        查询某个用户是否同步消息，只支持单个查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25343

        :param user_nick: 需要查询的用户名
        """
        return self._top_request(
            "taobao.jushita.jms.user.get",
            {
                "user_nick": user_nick
            },
            result_processor=lambda x: x["ons_user"]
        )

    def taobao_jushita_jms_user_add(
            self,
            topic_names=''
    ):
        """
        添加ONS消息同步用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25344

        :param topic_names: topic列表,不填则继承appkey所订阅的topic
        """
        return self._top_request(
            "taobao.jushita.jms.user.add",
            {
                "topic_names": topic_names
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_jushita_jms_user_delete(
            self,
            user_nick
    ):
        """
        删除ONS消息同步用户
        删除ONS消息同步用户，删除后用户的消息将不会推送到聚石塔的ONS中
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25345

        :param user_nick: 需要停止同步消息的用户nick
        """
        return self._top_request(
            "taobao.jushita.jms.user.delete",
            {
                "user_nick": user_nick
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_jushita_jms_group_get(
            self,
            page_no='1',
            page_size='40',
            group_names=''
    ):
        """
        查询ONS分组
        查询当前appkey在ONS中已有的分组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25634

        :param page_no: 页码
        :param page_size: 每页返回多少个分组
        :param group_names: 要查询分组的名称，多个分组用半角逗号分隔，不传代表查询所有分组信息，但不会返回组下面的用户信息。如果应用没有设置分组则返回空。组名不能以default开头，default开头是系统默认的组。
        """
        return self._top_request(
            "taobao.jushita.jms.group.get",
            {
                "page_no": page_no,
                "page_size": page_size,
                "group_names": group_names
            }
        )

    def taobao_jushita_jms_group_delete(
            self,
            group_name,
            nicks='',
            user_platform='tbUIC'
    ):
        """
        删除ONS分组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25635

        :param group_name: 分组名称，分组删除后，用户的消息将会存储于默认分组中。警告：由于分组已经删除，用户之前未消费的消息将无法再获取。不能以default开头，default开头为系统默认组。
        :param nicks: 用户列表，不传表示删除整个分组，如果用户全部删除后，也会自动删除整个分组
        :param user_platform: 用户所属于的平台类型，tbUIC:淘宝用户; icbu: icbu用户
        """
        return self._top_request(
            "taobao.jushita.jms.group.delete",
            {
                "group_name": group_name,
                "nicks": nicks,
                "user_platform": user_platform
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_jushita_jms_topics_get(
            self,
            nick
    ):
        """
        根据用户nick获取开通的消息列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26278

        :param nick: 卖家nick
        """
        return self._top_request(
            "taobao.jushita.jms.topics.get",
            {
                "nick": nick
            }
        )


class TbYingXiao(DingTalkBaseAPI):
    """
    营销API
    """

    def taobao_promotion_coupon_send(
            self,
            coupon_id,
            buyer_nick
    ):
        """
        店铺优惠券发放接口
        通过接口批量发放店铺优惠券（每次只能发送100张，只能发给当前授权卖家店铺的会员），发送成功则返回为空，发送失败则返回失败的买家列表和发送成功的买家和优惠券的number。注：如果所有买家都发放失败的话，is_success也为true，建议调用者根据返回的集合判断是否送入的买家都发放成功了
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10581

        :param coupon_id: 优惠券的id
        :param buyer_nick: 买家昵称用半角','号分割
        """
        return self._top_request(
            "taobao.promotion.coupon.send",
            {
                "coupon_id": coupon_id,
                "buyer_nick": buyer_nick
            }
        )

    def taobao_promotion_activity_get(
            self,
            activity_id=''
    ):
        """
        查询某个卖家的店铺优惠券领取活动
        查询某个卖家的店铺优惠券领取活动<br/>返回，优惠券领取活动ID，优惠券ID，总领用量，每人限领量，已领取数量<br/>领取活动状态，优惠券领取链接<br/>最多50个优惠券
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10582

        :param activity_id: 活动的id
        """
        return self._top_request(
            "taobao.promotion.activity.get",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["activitys"]
        )

    def taobao_promotion_coupon_add(
            self,
            denominations,
            end_time,
            condition='',
            start_time=''
    ):
        """
        创建店铺优惠券接口
        创建店铺优惠券。有效期内的店铺优惠券总数量不超过50张
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10584

        :param denominations: 优惠券的面额，必须是3，5，10，20，50，100
        :param end_time: 优惠券的截止日期
        :param condition: 订单满多少元才能用这个优惠券，500就是满500元才能使用
        :param start_time: 优惠券的生效时间
        """
        return self._top_request(
            "taobao.promotion.coupon.add",
            {
                "denominations": denominations,
                "end_time": end_time,
                "condition": condition,
                "start_time": start_time
            },
            result_processor=lambda x: x["coupon_id"]
        )

    def taobao_promotion_coupons_get(
            self,
            coupon_id='',
            end_time='',
            denominations='',
            page_no='1',
            page_size='20'
    ):
        """
        查询卖家优惠券
        查询卖家已经创建的优惠券，接口返回信息：优惠券ID，面值，创建时间，有效期，使用条件，使用渠道，创建渠道，优惠券总数量
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10585

        :param coupon_id: 优惠券的id，唯一标识这个优惠券
        :param end_time: 优惠券的截止日期
        :param denominations: 优惠券的面额，必须是3，5，10，20，50,100
        :param page_no: 查询的页号，结果集是分页返回的，每页20条
        :param page_size: 每页条数
        """
        return self._top_request(
            "taobao.promotion.coupons.get",
            {
                "coupon_id": coupon_id,
                "end_time": end_time,
                "denominations": denominations,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_promotion_meal_get(
            self,
            meal_id='',
            status=''
    ):
        """
        搭配套餐查询
        搭配套餐查询。每个卖家最多创建50个搭配套餐，所以查询不会分页，会将所有的满足状态的搭配套餐全部查出。该接口不会校验商品的下架或库存为0，查询结果的状态表明搭配套餐在数据库中的状态，商品的状态请isv自己验证。在卖家后台页面点击查看会触发数据库状态的修改。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10594

        :param meal_id: 搭配套餐id
        :param status: 套餐状态。有效：VALID;失效：INVALID(有效套餐为可使用的套餐,无效套餐为套餐中有商品下架或库存为0时)。默认时两种情况都会查询。
        """
        return self._top_request(
            "taobao.promotion.meal.get",
            {
                "meal_id": meal_id,
                "status": status
            },
            result_processor=lambda x: x["meal_list"]
        )

    def taobao_ump_mbb_getbyid(
            self,
            id
    ):
        """
        获取营销积木块
        根据积木块id获取营销积木块。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10709

        :param id: 积木块的id
        """
        return self._top_request(
            "taobao.ump.mbb.getbyid",
            {
                "id": id
            },
            result_processor=lambda x: x["mbb"]
        )

    def taobao_ump_mbbs_get(
            self,
            type=''
    ):
        """
        获取营销积木块列表
        获取营销积木块列表，可以根据类型获取，也可以将该字段设为空，获取所有的
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10710

        :param type: 积木块类型。如果该字段为空表示查出所有类型的<br/>现在有且仅有如下几种：resource,condition,action,target
        """
        return self._top_request(
            "taobao.ump.mbbs.get",
            {
                "type": type
            },
            result_processor=lambda x: x["mbbs"]
        )

    def taobao_ump_tool_get(
            self,
            tool_id
    ):
        """
        查询工具
        根据工具id获取一个工具对象
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10711

        :param tool_id: 工具的id
        """
        return self._top_request(
            "taobao.ump.tool.get",
            {
                "tool_id": tool_id
            },
            result_processor=lambda x: x["content"]
        )

    def taobao_ump_tools_get(
            self,
            tool_code=''
    ):
        """
        查询工具列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10712

        :param tool_code: 工具编码
        """
        return self._top_request(
            "taobao.ump.tools.get",
            {
                "tool_code": tool_code
            },
            result_processor=lambda x: x["tools"]
        )

    def taobao_ump_tool_add(
            self,
            content
    ):
        """
        新增优惠工具
        新增优惠工具。通过ump sdk来完成将积木块拼装成为工具的操作，再使用本接口上传优惠工具。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10713

        :param content: 工具内容，由sdk里面的MarketingTool生成的json字符串
        """
        return self._top_request(
            "taobao.ump.tool.add",
            {
                "content": content
            },
            result_processor=lambda x: x["tool_id"]
        )

    def taobao_ump_tool_delete(
            self,
            tool_id
    ):
        """
        删除工具
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        删除营销工具。当工具正在被使用的时候，是不能被删除的。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10715

        :param tool_id: 营销工具id
        """
        return self._top_request(
            "taobao.ump.tool.delete",
            {
                "tool_id": tool_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_activity_add(
            self,
            tool_id,
            content
    ):
        """
        新增优惠活动
        新增优惠活动。设置优惠活动的基本信息，比如活动时间，活动针对的对象（可以是满足某些条件的买家）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10716

        :param tool_id: 工具id
        :param content: 活动内容，通过ump sdk里面的MarkeitngTool来生成，name必须属于“营销标签词库”——https://huodong.m.taobao.com/api/data/v2/5fe5e737d3314fa2973297f86f7bff3a.js?file=5fe5e737d3314fa2973297f86f7bff3a.js中的word值中的一种。
        """
        return self._top_request(
            "taobao.ump.activity.add",
            {
                "tool_id": tool_id,
                "content": content
            },
            result_processor=lambda x: x["act_id"]
        )

    def taobao_ump_activity_update(
            self,
            act_id,
            content
    ):
        """
        修改活动信息
        修改营销活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10717

        :param act_id: 活动id
        :param content: 营销活动内容，json格式，通过ump sdk 的marketingTool来生成
        """
        return self._top_request(
            "taobao.ump.activity.update",
            {
                "act_id": act_id,
                "content": content
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_activity_delete(
            self,
            act_id
    ):
        """
        删除营销活动
        删除营销活动。对应的活动详情等将会被全部删除。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10718

        :param act_id: 活动id
        """
        return self._top_request(
            "taobao.ump.activity.delete",
            {
                "act_id": act_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_activity_get(
            self,
            act_id
    ):
        """
        查询营销活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10719

        :param act_id: 活动id
        """
        return self._top_request(
            "taobao.ump.activity.get",
            {
                "act_id": act_id
            },
            result_processor=lambda x: x["content"]
        )

    def taobao_ump_activities_get(
            self,
            tool_id,
            page_no,
            page_size
    ):
        """
        查询活动列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10720

        :param tool_id: 工具id
        :param page_no: 分页的页码
        :param page_size: 每页的最大条数
        """
        return self._top_request(
            "taobao.ump.activities.get",
            {
                "tool_id": tool_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_ump_detail_get(
            self,
            detail_id
    ):
        """
        查询活动详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10721

        :param detail_id: 活动详情的id
        """
        return self._top_request(
            "taobao.ump.detail.get",
            {
                "detail_id": detail_id
            },
            result_processor=lambda x: x["content"]
        )

    def taobao_ump_details_get(
            self,
            act_id,
            page_no,
            page_size
    ):
        """
        查询活动详情列表
        分页查询优惠详情列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10722

        :param act_id: 营销活动id
        :param page_no: 分页的页码
        :param page_size: 每页的最大条数
        """
        return self._top_request(
            "taobao.ump.details.get",
            {
                "act_id": act_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_ump_detail_add(
            self,
            act_id,
            content
    ):
        """
        新增活动详情
        增加活动详情。活动详情里面包括活动的范围（店铺，商品），活动的参数（比如具体的折扣），参与类型（全部，部分，部分不参加）等信息。当参与类型为部分或部分不参加的时候需要和taobao.ump.range.add来配合使用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10723

        :param act_id: 增加工具详情
        :param content: 活动详情内容，json格式，可以通过ump sdk中的MarketingTool来进行处理
        """
        return self._top_request(
            "taobao.ump.detail.add",
            {
                "act_id": act_id,
                "content": content
            },
            result_processor=lambda x: x["detail_id"]
        )

    def taobao_ump_detail_update(
            self,
            detail_id,
            content
    ):
        """
        修改活动详情
        更新活动详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10724

        :param detail_id: 活动详情id
        :param content: 活动详情内容，可以通过ump sdk中的MarketingTool来生成这个内容
        """
        return self._top_request(
            "taobao.ump.detail.update",
            {
                "detail_id": detail_id,
                "content": content
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_detail_delete(
            self,
            detail_id
    ):
        """
        删除活动详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10725

        :param detail_id: 活动详情id
        """
        return self._top_request(
            "taobao.ump.detail.delete",
            {
                "detail_id": detail_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_range_add(
            self,
            act_id,
            type,
            ids
    ):
        """
        添加活动范围
        指定某项活动中，某个商家的某些类型物品（指定商品或者别的）参加或者不参加活动。当活动详情的参与类型为部分或者部分不参加的时候，需要指定具体哪部分参加或者不参加，使用本接口完成操作。比如部分商品满就送，这里的range用来指定具体哪些商品参加满就送活动。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10726

        :param act_id: 活动id
        :param type: 范围的类型，比如是全店，商品，见：MarketingConstants.PARTICIPATE_TYPE_*
        :param ids: id列表，当范围类型为商品时，该id为商品id.多个id用逗号隔开，一次不超过50个
        """
        return self._top_request(
            "taobao.ump.range.add",
            {
                "act_id": act_id,
                "type": type,
                "ids": ids
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_range_delete(
            self,
            act_id,
            type,
            ids
    ):
        """
        删除活动范围
        去指先前指定在某项活动中，某些类型的物品参加或者不参加活动的设置
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10727

        :param act_id: 活动id
        :param type: 范围的类型，比如是全店，商品，类目见：MarketingConstants.PARTICIPATE_TYPE_*
        :param ids: id列表，当范围类型为商品时，该id为商品id；当范围类型为类目时，该id为类目id
        """
        return self._top_request(
            "taobao.ump.range.delete",
            {
                "act_id": act_id,
                "type": type,
                "ids": ids
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_range_get(
            self,
            act_id
    ):
        """
        查询活动范围
        查询某个卖家所有参加或者不参加某项活动的物品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10728

        :param act_id: 活动id
        """
        return self._top_request(
            "taobao.ump.range.get",
            {
                "act_id": act_id
            },
            result_processor=lambda x: x["ranges"]
        )

    def taobao_ump_mbb_getbycode(
            self,
            code
    ):
        """
        根据营销积木块代码获取积木块
        根据营销积木块代码获取积木块。接口返回该代码最新版本的积木块。如果要查询某个非最新版本的积木块，可以使用积木块id查询接口。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10729

        :param code: 营销积木块code
        """
        return self._top_request(
            "taobao.ump.mbb.getbycode",
            {
                "code": code
            },
            result_processor=lambda x: x["mbb"]
        )

    def taobao_promotion_limitdiscount_detail_get(
            self,
            limit_discount_id
    ):
        """
        限时打折详情查询
        限时打折详情查询。查询出指定限时打折的对应商品记录信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10770

        :param limit_discount_id: 限时打折ID。这个针对查询唯一限时打折情况。
        """
        return self._top_request(
            "taobao.promotion.limitdiscount.detail.get",
            {
                "limit_discount_id": limit_discount_id
            },
            result_processor=lambda x: x["item_discount_detail_list"]
        )

    def taobao_promotion_limitdiscount_get(
            self,
            limit_discount_id='',
            status='',
            start_time='',
            end_time='',
            page_number=''
    ):
        """
        限时打折查询
        分页查询某个卖家的限时打折信息。每页20条数据，按照结束时间降序排列。也可指定某一个限时打折id查询唯一的限时打折信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10771

        :param limit_discount_id: 限时打折ID。这个针对查询唯一限时打折情况。若此字段不为空，则说明操作为单条限时打折记录查询，其他字段忽略。若想分页按条件查询，这个字段置为空。
        :param status: 限时打折活动状态。ALL:全部状态;OVER:已结束;DOING:进行中;PROPARE:未开始(只支持大写)。当limit_discount_id为空时，为空时，默认为全部的状态。
        :param start_time: 限时打折开始时间。输入的时间会被截取，年月日有效，时分秒忽略。
        :param end_time: 限时打折结束时间。输入的时间会被截取，年月日有效，时分秒忽略。
        :param page_number: 分页页号。默认1。当页数大于最大页数时，结果为最大页数的数据。
        """
        return self._top_request(
            "taobao.promotion.limitdiscount.get",
            {
                "limit_discount_id": limit_discount_id,
                "status": status,
                "start_time": start_time,
                "end_time": end_time,
                "page_number": page_number
            }
        )

    def taobao_marketing_promotion_kfc(
            self,
            promotion_title,
            promotion_desc
    ):
        """
        定向优惠活动名称与描述违禁词检查
        活动名称与描述违禁词检查
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10792

        :param promotion_title: 活动名称
        :param promotion_desc: 活动描述
        """
        return self._top_request(
            "taobao.marketing.promotion.kfc",
            {
                "promotion_title": promotion_title,
                "promotion_desc": promotion_desc
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_mbbs_list_get(
            self,
            ids
    ):
        """
        通过ids列表获取营销积木块列表
        通过营销积木id列表来获取营销积木块列表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10845

        :param ids: 营销积木块id组成的字符串。
        """
        return self._top_request(
            "taobao.ump.mbbs.list.get",
            {
                "ids": ids
            },
            result_processor=lambda x: x["mbbs"]
        )

    def taobao_ump_activities_list_get(
            self,
            ids
    ):
        """
        营销活动列表查询
        按照营销活动id的列表ids，查询对应的营销活动列表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10846

        :param ids: 营销活动id列表。
        """
        return self._top_request(
            "taobao.ump.activities.list.get",
            {
                "ids": ids
            },
            result_processor=lambda x: x["activities"]
        )

    def taobao_ump_detail_list_add(
            self,
            act_id,
            details
    ):
        """
        营销详情添加
        批量添加营销活动。替代单条添加营销详情的的API。此接口适用针对某个营销活动的多档设置，会按顺序插入detail。若在整个事务过程中出现断点，会将已插入完成的detail_id返回，注意记录这些id，并将其删除，会对交易过程中的优惠产生影响。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10854

        :param act_id: 营销活动id。
        :param details: 营销详情的列表。此列表由detail的json字符串组成。最多插入为10个。
        """
        return self._top_request(
            "taobao.ump.detail.list.add",
            {
                "act_id": act_id,
                "details": details
            },
            result_processor=lambda x: x["detail_id_list"]
        )

    def taobao_promotion_coupon_buyer_search(
            self,
            seller_nick='',
            status='',
            page_size='20',
            current_page='1',
            end_time=''
    ):
        """
        查询买家在相关app领取的优惠券信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10855

        :param seller_nick: 卖家昵称
        :param status: 券状态  '正常',1 '已删除',-1 '已使用',-2 '冻结',0
        :param page_size: 每页数据 建议20左右
        :param current_page: 当前第几页  从第一页开始
        :param end_time: 结束时间
        """
        return self._top_request(
            "taobao.promotion.coupon.buyer.search",
            {
                "seller_nick": seller_nick,
                "status": status,
                "page_size": page_size,
                "current_page": current_page,
                "end_time": end_time
            }
        )

    def taobao_promotionmisc_tool_check(
            self,
            tool_id,
            meta_allow=''
    ):
        """
        UMP工具检测
        UMP工具检测。ISV通过该接口检测（通过taobao.ump.tool.add）创建的UMP工具（tool）是否符合规范，如果不符合，则返回错误信息和对应的解决方案的；工具检测通过后才可以提交工具审核邮件，提交工具审核时，需提供该接口的返回值。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21644

        :param tool_id: 工具ID, taobao.ump.tool.add成功后返回的id。
        :param meta_allow: 可使用的元数据。PRD审核后，会告诉isv可使用的元数据。
        """
        return self._top_request(
            "taobao.promotionmisc.tool.check",
            {
                "tool_id": tool_id,
                "meta_allow": meta_allow
            }
        )

    def tmall_promotag_tag_apply(
            self,
            tag_name,
            tag_desc,
            start_time,
            end_time
    ):
        """
        优惠标签申请
        创建优惠标签
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21771

        :param tag_name: 标签名称。
                         注意在UMP中使用新人群标签top变成大写的“NEW_” 如：老标签是top1234，新标签是NEW_1234 。
        :param tag_desc: 标签用途描述
        :param start_time: 标签开始时间
        :param end_time: 标签结束时间
        """
        return self._top_request(
            "tmall.promotag.tag.apply",
            {
                "tag_name": tag_name,
                "tag_desc": tag_desc,
                "start_time": start_time,
                "end_time": end_time
            }
        )

    def tmall_promotag_tag_find(
            self,
            page_no,
            page_size,
            tag_name='',
            tag_id=''
    ):
        """
        查询标签接口
        查询用户创建的所有标签
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21772

        :param page_no: 当前页码
        :param page_size: 每页显示个数
        :param tag_name: 标签名称，查询时可选项
        :param tag_id: 标签ID
        """
        return self._top_request(
            "tmall.promotag.tag.find",
            {
                "page_no": page_no,
                "page_size": page_size,
                "tag_name": tag_name,
                "tag_id": tag_id
            },
            result_processor=lambda x: x["query_result"]
        )

    def tmall_promotag_taguser_judge(
            self,
            tag_id,
            nick
    ):
        """
        用户标签判断接口
        查询用户是否有标签
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21773

        :param tag_id: 标签ID
        :param nick: 买家昵称
        """
        return self._top_request(
            "tmall.promotag.taguser.judge",
            {
                "tag_id": tag_id,
                "nick": nick
            }
        )

    def tmall_promotag_taguser_remove(
            self,
            tag_id,
            nick
    ):
        """
        给用户移除优惠标签
        给用户载体去标
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21774

        :param tag_id: 标签ID
        :param nick: 买家昵称
        """
        return self._top_request(
            "tmall.promotag.taguser.remove",
            {
                "tag_id": tag_id,
                "nick": nick
            },
            result_processor=lambda x: x["is_success"]
        )

    def tmall_promotag_taguser_save(
            self,
            tag_id,
            nick
    ):
        """
        给用户打上优惠标签
        给用户载体打标
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21775

        :param tag_id: 标签ID
        :param nick: 买家昵称
        """
        return self._top_request(
            "tmall.promotag.taguser.save",
            {
                "tag_id": tag_id,
                "nick": nick
            },
            result_processor=lambda x: x["is_success"]
        )

    def tmall_promotag_tag_removetag(
            self,
            tag_id
    ):
        """
        删除标签定义
        用于删除标签定义，但是要确保目前该标签没有人群在使用。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22224

        :param tag_id: 需要删除的标签id
        """
        return self._top_request(
            "tmall.promotag.tag.removetag",
            {
                "tag_id": tag_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_item_activity_update(
            self,
            activity_id,
            name,
            participate_range,
            start_time,
            end_time,
            is_user_tag='false',
            user_tag='',
            is_decrease_money='false',
            decrease_amount='',
            is_discount='false',
            discount_rate=''
    ):
        """
        修改无条件单品优惠活动
        修改无条件单品优惠活动。<br/>1、该接口只修改活动基本信息和打折信息，如需要增加、删除参与该活动的商品，请调用taobao.promotionmisc.activity.range.add和taobao.promotionmisc.activity.range.remove接口。 <br/>2、使用该接口时需要同时把未做修改的字段值也传入。 <br/><br/>3、该接口受店铺最低折扣限制，如优惠不生效，请让卖家检查该优惠是否低于店铺的最低折扣设置。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22248

        :param activity_id: 活动id。
        :param name: 活动名称。
        :param participate_range: 活动范围：0表示全部参与； 1表示部分商品参与。
        :param start_time: 活动开始时间。
        :param end_time: 活动结束时间。
        :param is_user_tag: 是否指定用户标签。
        :param user_tag: 用户标签。当is_user_tag为true时，该值才有意义。
        :param is_decrease_money: 是否有减钱行为。
        :param decrease_amount: 减多少钱。当is_decrease_money为true时，该值才有意义。注意：该值单位为分，即100表示1元。
        :param is_discount: 是否有打折行为。
        :param discount_rate: 折扣值。当is_discount为true时，该值才有意义。注意：800表示8折。
        """
        return self._top_request(
            "taobao.promotionmisc.item.activity.update",
            {
                "activity_id": activity_id,
                "name": name,
                "participate_range": participate_range,
                "start_time": start_time,
                "end_time": end_time,
                "is_user_tag": is_user_tag,
                "user_tag": user_tag,
                "is_decrease_money": is_decrease_money,
                "decrease_amount": decrease_amount,
                "is_discount": is_discount,
                "discount_rate": discount_rate
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_mjs_activity_add(
            self,
            name,
            type,
            participate_range,
            start_time,
            end_time,
            is_amount_over='false',
            total_price='',
            is_amount_multiple='false',
            is_item_count_over='false',
            item_count='',
            is_item_multiple='false',
            is_shop_member='false',
            shop_member_level='',
            is_user_tag='false',
            user_tag='',
            is_decrease_money='false',
            decrease_amount='',
            is_discount='false',
            discount_rate='',
            is_send_gift='false',
            gift_name='',
            gift_id='',
            gift_url='',
            is_free_post='false',
            exclude_area=''
    ):
        """
        创建满就送活动
        创建满就送活动。<br/>1、可以选择是全店参加或者部分商品参加：participate_range：0表示全部参与； 1表示部分商品参与。 2、如果是部分商品参加，则需要通过taobao.promotionmisc.activity.range.add接口来指定需要参加的商品。 3、该接口创建的优惠受店铺最低折扣限制，如优惠不生效，请让卖家检查该优惠是否低于店铺的最低折扣设置。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22249

        :param name: 活动名称。
        :param type: 活动类型： 1表示商品级别的活动；2表示店铺级别的活动
        :param participate_range: 活动范围：0表示全部参与； 1表示部分商品参与。
        :param start_time: 活动开始时间。
        :param end_time: 活动结束时间。
        :param is_amount_over: 是否有满元条件。
        :param total_price: 满多少元。当is_amount_over为true时，该才字段有意义。注意：单位是分，即10000表示100元。
        :param is_amount_multiple: 满元是否上不封顶。当is_amount_over为true时，该值才有意义。当该值为true时，表示满元上不封顶，例如满100元减10元，当满200时，则减20元。。。默认为false。
        :param is_item_count_over: 是否有满件条件。
        :param item_count: 满多少件。当is_item_count_over为true时，该值才有意义。
        :param is_item_multiple: 满件是否上不封顶。当is_amount_multiple为true时，该值才有意义。当该值为true时，表示满件上不封顶，例如满10件减2元，当满20件时，则减4元。。。 默认为false。
        :param is_shop_member: 是否有店铺会员等级条件。
        :param shop_member_level: 店铺会员等级，当is_shop_member为true时，该值才有意义。0：店铺客户；1：普通客户；2：高级会员；3：VIP会员； 4：至尊VIP会员。
        :param is_user_tag: 是否指定用户标签。
        :param user_tag: 用户标签。当is_user_tag为true时，该值才有意义。
        :param is_decrease_money: 是否有减钱行为。
        :param decrease_amount: 减多少钱。当is_decrease_money为true时，该值才有意义。注意：该值单位为分，即100表示1元。
        :param is_discount: 是否有打折行为。
        :param discount_rate: 折扣值。当is_discount为true时，该值才有意义。注意：800表示8折。
        :param is_send_gift: 是否有送礼品行为。
        :param gift_name: 礼品名称。当is_send_gift为true时，该值才有意义。
        :param gift_id: 礼品id，当is_send_gift为true时，该值才有意义。 1）只有填写真实的淘宝商品id时，才能生成物流单，并且在确定订单的页面上可以点击该商品名称跳转到商品详情页面。2）当礼物为实物商品时(有宝贝id),礼物必须为上架商品,不能为虚拟商品,不能为拍卖商品,不能有sku,不符合条件的,不做为礼物。
        :param gift_url: 商品详情的url，当is_send_gift为true时，该值才有效。
        :param is_free_post: 是否有免邮行为。
        :param exclude_area: 免邮的排除地区，即，除指定地区外，其他地区包邮。当is_free_post为true时，该值才有意义。代码使用*链接，代码为行政区划代码。
        """
        return self._top_request(
            "taobao.promotionmisc.mjs.activity.add",
            {
                "name": name,
                "type": type,
                "participate_range": participate_range,
                "start_time": start_time,
                "end_time": end_time,
                "is_amount_over": is_amount_over,
                "total_price": total_price,
                "is_amount_multiple": is_amount_multiple,
                "is_item_count_over": is_item_count_over,
                "item_count": item_count,
                "is_item_multiple": is_item_multiple,
                "is_shop_member": is_shop_member,
                "shop_member_level": shop_member_level,
                "is_user_tag": is_user_tag,
                "user_tag": user_tag,
                "is_decrease_money": is_decrease_money,
                "decrease_amount": decrease_amount,
                "is_discount": is_discount,
                "discount_rate": discount_rate,
                "is_send_gift": is_send_gift,
                "gift_name": gift_name,
                "gift_id": gift_id,
                "gift_url": gift_url,
                "is_free_post": is_free_post,
                "exclude_area": exclude_area
            }
        )

    def taobao_promotionmisc_mjs_activity_get(
            self,
            activity_id
    ):
        """
        查询满就送活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22250

        :param activity_id: 活动id。
        """
        return self._top_request(
            "taobao.promotionmisc.mjs.activity.get",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["mjs_promotion"]
        )

    def taobao_promotionmisc_activity_range_add(
            self,
            activity_id,
            ids
    ):
        """
        增加活动参与的商品
        增加活动参与的商品，部分商品参与的活动，最大支持指定150个商品。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22251

        :param activity_id: 活动id。
        :param ids: 商品id,多个id用逗号隔开，一次不超过50个。
        """
        return self._top_request(
            "taobao.promotionmisc.activity.range.add",
            {
                "activity_id": activity_id,
                "ids": ids
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_activity_range_list_get(
            self,
            activity_id
    ):
        """
        查询活动参与的商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22252

        :param activity_id: 活动id
        """
        return self._top_request(
            "taobao.promotionmisc.activity.range.list.get",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["promotion_range_list"]
        )

    def taobao_promotionmisc_activity_range_remove(
            self,
            activity_id,
            ids
    ):
        """
        去除活动参与的商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22253

        :param activity_id: 活动id。
        :param ids: 商品id,多个id用逗号隔开。
        """
        return self._top_request(
            "taobao.promotionmisc.activity.range.remove",
            {
                "activity_id": activity_id,
                "ids": ids
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_activity_range_all_remove(
            self,
            activity_id
    ):
        """
        清空活动参与的商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22254

        :param activity_id: 活动id。
        """
        return self._top_request(
            "taobao.promotionmisc.activity.range.all.remove",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_mjs_activity_list_get(
            self,
            activity_type,
            page_no,
            page_size
    ):
        """
        查询满就送活动列表
        查询满就送活动列表。注意，该接口的返回值中，只包含活动的主要信息，如activity_id，name，description，start_time，end_time，type，participate_range。优惠的详细信息，请通过taobao.promotionmisc.mjs.activity.get获取。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22255

        :param activity_type: 活动类型： 1表示商品级别的活动；2表示店铺级别的活动。
        :param page_no: 页码。
        :param page_size: 每页记录数，最大支持50 。
        """
        return self._top_request(
            "taobao.promotionmisc.mjs.activity.list.get",
            {
                "activity_type": activity_type,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_promotionmisc_mjs_activity_update(
            self,
            activity_id,
            name,
            participate_range,
            start_time,
            end_time,
            is_amount_over='false',
            total_price='',
            is_amount_multiple='false',
            is_item_count_over='false',
            item_count='',
            is_item_multiple='false',
            is_shop_member='false',
            shop_member_level='',
            is_user_tag='false',
            user_tag='',
            is_decrease_money='false',
            decrease_amount='',
            is_discount='false',
            discount_rate='',
            is_send_gift='false',
            gift_name='',
            gift_id='',
            gift_url='',
            is_free_post='false',
            exclude_area=''
    ):
        """
        修改满就送活动
        修改满就送活动。 <br/>1、该接口只修改活动基本信息和打折信息，如需要增加、删除参与该活动的商品，请调用taobao.promotionmisc.activity.range.add和taobao.promotionmisc.activity.range.remove接口。 <br/>2、使用该接口时需要同时把未做修改的字段值也传入。 <br/>3、该接口创建的优惠受店铺最低折扣限制，如优惠不生效，请让卖家检查该优惠是否低于店铺的最低折扣设置。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22256

        :param activity_id: 活动id。
        :param name: 活动名称。
        :param participate_range: 活动范围：0表示全部参与； 1表示部分商品参与。
        :param start_time: 活动开始时间。
        :param end_time: 活动结束时间。
        :param is_amount_over: 是否有满元条件。
        :param total_price: 满多少元。当is_amount_over为true时，该才字段有意义。注意：单位是分，即10000表示100元。
        :param is_amount_multiple: 满元是否上不封顶。当is_amount_over为true时，该值才有意义。当该值为true时，表示满元上不封顶，例如满100元减10元，当满200时，则减20元。。。默认为false。
        :param is_item_count_over: 是否有满件条件。
        :param item_count: 满多少件。当is_item_count_over为true时，该值才有意义。
        :param is_item_multiple: 满件是否上不封顶。当is_amount_multiple为true时，该值才有意义。当该值为true时，表示满件上不封顶，例如满10件减2元，当满20件时，则减4元。。。 默认为false。
        :param is_shop_member: 是否有店铺会员等级条件。
        :param shop_member_level: 店铺会员等级，当is_shop_member为true时，该值才有意义。0：店铺客户；1：普通客户；2：高级会员；3：VIP会员； 4：至尊VIP会员。
        :param is_user_tag: 是否指定用户标签。
        :param user_tag: 用户标签。当is_user_tag为true时，该值才有意义。
        :param is_decrease_money: 是否有减钱行为。
        :param decrease_amount: 减多少钱。当is_decrease_money为true时，该值才有意义。注意：该值单位为分，即100表示1元。
        :param is_discount: 是否有打折行为。
        :param discount_rate: 折扣值。当is_discount为true时，该值才有意义。注意：800表示8折。
        :param is_send_gift: 是否有送礼品行为。
        :param gift_name: 礼品名称。当is_send_gift为true时，该值才有意义。
        :param gift_id: 礼品id，当is_send_gift为true时，该值才有意义。 1）只有填写真实的淘宝商品id时，才能生成物流单，并且在确定订单的页面上可以点击该商品名称跳转到商品详情页面。2）当礼物为实物商品时(有宝贝id),礼物必须为上架商品,不能为虚拟商品,不能为拍卖商品,不能有sku,不符合条件的,不做为礼物。
        :param gift_url: 商品详情的url，当is_send_gift为true时，该值才有效。
        :param is_free_post: 是否有免邮行为。
        :param exclude_area: 免邮的排除地区，即，除指定地区外，其他地区包邮。当is_free_post为true时，该值才有意义。代码使用*链接，代码为行政区划代码。
        """
        return self._top_request(
            "taobao.promotionmisc.mjs.activity.update",
            {
                "activity_id": activity_id,
                "name": name,
                "participate_range": participate_range,
                "start_time": start_time,
                "end_time": end_time,
                "is_amount_over": is_amount_over,
                "total_price": total_price,
                "is_amount_multiple": is_amount_multiple,
                "is_item_count_over": is_item_count_over,
                "item_count": item_count,
                "is_item_multiple": is_item_multiple,
                "is_shop_member": is_shop_member,
                "shop_member_level": shop_member_level,
                "is_user_tag": is_user_tag,
                "user_tag": user_tag,
                "is_decrease_money": is_decrease_money,
                "decrease_amount": decrease_amount,
                "is_discount": is_discount,
                "discount_rate": discount_rate,
                "is_send_gift": is_send_gift,
                "gift_name": gift_name,
                "gift_id": gift_id,
                "gift_url": gift_url,
                "is_free_post": is_free_post,
                "exclude_area": exclude_area
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_mjs_activity_delete(
            self,
            activity_id
    ):
        """
        删除满就送活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22257

        :param activity_id: 活动id。
        """
        return self._top_request(
            "taobao.promotionmisc.mjs.activity.delete",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_item_activity_add(
            self,
            name,
            participate_range,
            start_time,
            end_time,
            is_user_tag='false',
            user_tag='',
            is_decrease_money='false',
            decrease_amount='',
            is_discount='false',
            discount_rate=''
    ):
        """
        创建无条件单品优惠活动
        创建无条件单品优惠活动。1、可以选择是全店参加或者部分商品参加：participate_range：0表示全部参与； 1表示部分商品参与。<br/>2、如果是部分商品参加，则需要通过taobao.promotionmisc.activity.range.add接口来指定需要参加的商品。<br/>3、该接口创建的优惠受店铺最低折扣限制，如优惠不生效，请让卖家检查该优惠是否低于店铺的最低折扣设置。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22258

        :param name: 活动名称，超过5个汉字时，商品详情中显示的优惠名称为：卖家优惠。
        :param participate_range: 活动范围：0表示全部参与； 1表示部分商品参与。
        :param start_time: 活动开始时间。
        :param end_time: 活动结束时间。
        :param is_user_tag: 是否指定用户标签。
        :param user_tag: 用户标签。当is_user_tag为true时，该值才有意义。
        :param is_decrease_money: 是否有减钱行为。
        :param decrease_amount: 减多少钱。当is_decrease_money为true时，该值才有意义。注意：该值单位为分，即100表示1元。
        :param is_discount: 是否有打折行为。
        :param discount_rate: 折扣值。当is_discount为true时，该值才有意义。注意：800表示8折。
        """
        return self._top_request(
            "taobao.promotionmisc.item.activity.add",
            {
                "name": name,
                "participate_range": participate_range,
                "start_time": start_time,
                "end_time": end_time,
                "is_user_tag": is_user_tag,
                "user_tag": user_tag,
                "is_decrease_money": is_decrease_money,
                "decrease_amount": decrease_amount,
                "is_discount": is_discount,
                "discount_rate": discount_rate
            }
        )

    def taobao_promotionmisc_item_activity_get(
            self,
            activity_id
    ):
        """
        查询无条件单品优惠活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22259

        :param activity_id: 活动id。
        """
        return self._top_request(
            "taobao.promotionmisc.item.activity.get",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["item_promotion"]
        )

    def taobao_promotionmisc_item_activity_list_get(
            self,
            page_no,
            page_size
    ):
        """
        查询无条件单品优惠活动列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22260

        :param page_no: 页码。
        :param page_size: 每页记录数，最大支持50 。
        """
        return self._top_request(
            "taobao.promotionmisc.item.activity.list.get",
            {
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_promotionmisc_item_activity_delete(
            self,
            activity_id
    ):
        """
        删除无条件单品优惠活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22261

        :param activity_id: 活动id。
        """
        return self._top_request(
            "taobao.promotionmisc.item.activity.delete",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_ump_shoutaotag_add(
            self,
            item_id,
            buyer_id,
            channel_key
    ):
        """
        手淘定向优惠打标接口
        手淘定向优惠的优惠标签打标接口
        给特定的手淘买家打上优惠标记，标记承载在自己的业务标签库中，标签有效期为7天。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24886

        :param item_id: 商品ID
        :param buyer_id: 买家ID
        :param channel_key: 渠道KEY
        """
        return self._top_request(
            "taobao.ump.shoutaotag.add",
            {
                "item_id": item_id,
                "buyer_id": buyer_id,
                "channel_key": channel_key
            },
            result_processor=lambda x: x["add_result"]
        )

    def tmall_promotion_tip_campaign_remove(
            self,
            campaign_id
    ):
        """
        天猫特价宝删除活动接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫特价宝删除活动接口。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25073

        :param campaign_id: 要删除的活动ID
        """
        return self._top_request(
            "tmall.promotion.tip.campaign.remove",
            {
                "campaign_id": campaign_id
            },
            result_processor=lambda x: x["remove_rst"]
        )

    def tmall_promotion_tip_item_remove(
            self,
            item_id,
            campaign_id
    ):
        """
        天猫特价宝删除活动商品
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫特价宝删除活动商品。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25074

        :param item_id: 商品id
        :param campaign_id: 活动id
        """
        return self._top_request(
            "tmall.promotion.tip.item.remove",
            {
                "item_id": item_id,
                "campaign_id": campaign_id
            },
            result_processor=lambda x: x["remove_rst"]
        )

    def tmall_promotion_tip_item_adds(
            self,
            item_proms
    ):
        """
        天猫特价宝批量添加商品
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25075

        :param item_proms: 商品优惠参数
        """
        return self._top_request(
            "tmall.promotion.tip.item.adds",
            {
                "item_proms": item_proms
            },
            result_processor=lambda x: x["adds_rst"]
        )

    def tmall_promotion_tip_item_modify(
            self,
            item_prom
    ):
        """
        天猫特价宝修改商品优惠
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫特价宝修改商品优惠。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25078

        :param item_prom: 商品优惠参数
        """
        return self._top_request(
            "tmall.promotion.tip.item.modify",
            {
                "item_prom": item_prom
            },
            result_processor=lambda x: x["modify_rst"]
        )

    def tmall_promotion_tip_campaign_modify(
            self,
            campaign
    ):
        """
        天猫营销修改活动
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫营销活动修改接口。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25079

        :param campaign: 天猫单品活动对象
        """
        return self._top_request(
            "tmall.promotion.tip.campaign.modify",
            {
                "campaign": campaign
            },
            result_processor=lambda x: x["modify_rst"]
        )

    def tmall_promotion_tip_campaign_create(
            self,
            campaign
    ):
        """
        天猫营销创建活动接口
        天猫营销创建活动接口。开放给KA商家进行系统对接，创建优惠促销活动。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25080

        :param campaign: 天猫单品活动对象
        """
        return self._top_request(
            "tmall.promotion.tip.campaign.create",
            {
                "campaign": campaign
            },
            result_processor=lambda x: x["campaign_id"]
        )

    def tmall_promotion_tip_item_modifys(
            self,
            item_proms
    ):
        """
        天猫特价宝批量修改商品优惠
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫特价宝批量修改商品优惠。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25081

        :param item_proms: 商品优惠参数
        """
        return self._top_request(
            "tmall.promotion.tip.item.modifys",
            {
                "item_proms": item_proms
            },
            result_processor=lambda x: x["modifys_rst"]
        )

    def tmall_promotion_tip_item_add(
            self,
            item_prom
    ):
        """
        天猫特价宝添加活动商品
        天猫特价宝添加活动商品优惠。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25082

        :param item_prom: 商品优惠参数
        """
        return self._top_request(
            "tmall.promotion.tip.item.add",
            {
                "item_prom": item_prom
            },
            result_processor=lambda x: x["add_rst"]
        )

    def tmall_promotion_item_promotion_get(
            self,
            page,
            campaign_id,
            pagesize='0',
            brand_id='0'
    ):
        """
        天猫特价宝查询某个活动下面的单品优惠信息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫特价宝，根据活动ID，查询某个活动下面的单品优惠信息，需要进行分页查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25093

        :param page: 查询页码
        :param campaign_id: 活动id
        :param pagesize: 每页大小，[0,100]
        :param brand_id: 品牌ID
        """
        return self._top_request(
            "tmall.promotion.item.promotion.get",
            {
                "page": page,
                "campaign_id": campaign_id,
                "pagesize": pagesize,
                "brand_id": brand_id
            },
            result_processor=lambda x: x["tjb_item_prom_rst"]
        )

    def tmall_promotion_tjb_campaign_get(
            self,
            query_start,
            query_end,
            campaign_id='0',
            page_num='1'
    ):
        """
        天猫特价宝活动获取接口
        美的提的需求，获取美的品牌的特价宝优惠
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25094

        :param query_start: 优惠时段：开始时间
        :param query_end: 优惠时段：结束时间
        :param campaign_id: 活动ID
        :param page_num: 查询页码
        """
        return self._top_request(
            "tmall.promotion.tjb.campaign.get",
            {
                "query_start": query_start,
                "query_end": query_end,
                "campaign_id": campaign_id,
                "page_num": page_num
            },
            result_processor=lambda x: x["tjb_campaign_rst"]
        )

    def taobao_promotion_benefit_activity_delete(
            self,
            relation_id
    ):
        """
        删除关联的活动权益
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25171

        :param relation_id: ISV活动关联权益后获得的关联ID
        """
        return self._top_request(
            "taobao.promotion.benefit.activity.delete",
            {
                "relation_id": relation_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotion_benefit_activity_send(
            self,
            send_request,
            receiver_id='',
            nick='',
            plat_nick='',
            mix_receiver_id=''
    ):
        """
        活动权益发放接口
        活动权益发放接口，用于卖家针对活动进行权益发放
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25193

        :param send_request: 单个权益发放请求
        :param receiver_id: 非混淆的接收者id
        :param nick: 混淆的接收者nick
        :param plat_nick: 非混淆的接收者nick
        :param mix_receiver_id: 混淆的接收者id
        """
        return self._top_request(
            "taobao.promotion.benefit.activity.send",
            {
                "send_request": send_request,
                "receiver_id": receiver_id,
                "nick": nick,
                "plat_nick": plat_nick,
                "mix_receiver_id": mix_receiver_id
            }
        )

    def taobao_mobile_promotion_benefit_activity_send(
            self,
            single_benefit_request
    ):
        """
        手淘专用单用户发放接口
        卖家活动中需要通过该API来发放对应的权益。手淘专用单用户发放接口。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25194

        :param single_benefit_request: 单用户权益发放请求
        """
        return self._top_request(
            "taobao.mobile.promotion.benefit.activity.send",
            {
                "single_benefit_request": single_benefit_request
            },
            result_processor=lambda x: x["send_result"]
        )

    def taobao_promotion_benefit_selector(
            self,
            query
    ):
        """
        权益选择器接口
        权益选择器，查询用户已有权益，提供用户进行已拥有权益的选择操作，权益发放的前置操作
        1、目前top的接口只开了1，4，13，14 四种权益, 支付宝红包--1；流量钱包--4；优酷会员--13；彩票-- 14<br/>
        2、目前只有'支付宝红包'有'benefit_type': 'ALIPAY_COUPON',其它三个没有benefit_type   <br/>
        3、接口文档中写的 优酷会员卡--2 写错了，正确的是13（已接口返回为准）<br/>
        4、step=2用config_id查，即1，4，13，14  <br/>
        5、step=3权益id指具体采购的权益id，可以认为是采购的主键（权益id 可以通过step=2 获得 ）  <br/>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25195

        :param query: 权益选择器请求
        """
        return self._top_request(
            "taobao.promotion.benefit.selector",
            {
                "query": query
            }
        )

    def taobao_promotion_benefit_activity_update(
            self,
            update_request
    ):
        """
        修改关联的活动权益
        修改卖家活动中关联的对应的权益。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25203

        :param update_request: 修改关联的权益的活动请求
        """
        return self._top_request(
            "taobao.promotion.benefit.activity.update",
            {
                "update_request": update_request
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotion_benefit_activity_relation(
            self,
            relation_request
    ):
        """
        关联活动权益
        卖家活动中需要通过该API来关联的对应的权益。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25204

        :param relation_request: 活动关联权益请求参数
        """
        return self._top_request(
            "taobao.promotion.benefit.activity.relation",
            {
                "relation_request": relation_request
            }
        )

    def taobao_promotion_benefit_activity_time_update(
            self,
            relation_id,
            start_time,
            end_time
    ):
        """
        更新关联活动有效时间
        更新关联权益的活动有效时间
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25205

        :param relation_id: ISV活动关联权益后获得的关联ID
        :param start_time: 活动的开始时间
        :param end_time: 活动的i结束时间
        """
        return self._top_request(
            "taobao.promotion.benefit.activity.time.update",
            {
                "relation_id": relation_id,
                "start_time": start_time,
                "end_time": end_time
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotion_benefit_activity_detail_get(
            self,
            query_request
    ):
        """
        活动关联的权益详情获取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25206

        :param query_request: 查询活动关联权益详情的请求
        """
        return self._top_request(
            "taobao.promotion.benefit.activity.detail.get",
            {
                "query_request": query_request
            }
        )

    def taobao_promotion_coupon_seller_search(
            self,
            seller_nick='',
            current_page='1',
            page_size='20',
            spread_ids=''
    ):
        """
        查询绑定卖家优惠券相关信息
        查询绑定卖家相关优惠券信息  如isv  百川 等外部业务方
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25239

        :param seller_nick: 卖家昵称
        :param current_page: 当前第几页  从第一页开始
        :param page_size: 每页数据 最大20左右
        :param spread_ids: 券id集合
        """
        return self._top_request(
            "taobao.promotion.coupon.seller.search",
            {
                "seller_nick": seller_nick,
                "current_page": current_page,
                "page_size": page_size,
                "spread_ids": spread_ids
            }
        )

    def taobao_promotion_coupon_apply(
            self,
            seller_id,
            spread_id
    ):
        """
        优惠券领取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25253

        :param seller_id: 卖家id
        :param spread_id: 传播id
        """
        return self._top_request(
            "taobao.promotion.coupon.apply",
            {
                "seller_id": seller_id,
                "spread_id": spread_id
            }
        )

    def taobao_mobile_promotion_benefit_activity_send_share(
            self,
            benefit_type,
            biz_id,
            detail_id,
            feed_id,
            relation_id,
            send_count,
            trace_id,
            unique_id,
            share_key='',
            share_users=''
    ):
        """
        手淘专用单用户发放接口
        卖家活动中需要通过该API来发放对应的权益。手淘专用、验证分享链路。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25732

        :param benefit_type: 权益类型    其中ALIPAY_COUPON 对应的type值是1
        :param biz_id: 权益关联的活动ID
        :param detail_id: 活动详情id
        :param feed_id: 广播ID
        :param relation_id: 关联活动id
        :param send_count: 权益发放数量
        :param trace_id: 调试线索
        :param unique_id: 事务id
        :param share_key: 和bizId一起使用，标记分享链路的key。
        :param share_users: 分享链路上的用户及用户是否发奖，u1_true,u2_true
        """
        return self._top_request(
            "taobao.mobile.promotion.benefit.activity.send.share",
            {
                "benefit_type": benefit_type,
                "biz_id": biz_id,
                "detail_id": detail_id,
                "feed_id": feed_id,
                "relation_id": relation_id,
                "send_count": send_count,
                "trace_id": trace_id,
                "unique_id": unique_id,
                "share_key": share_key,
                "share_users": share_users
            },
            result_processor=lambda x: x["send_result"]
        )

    def taobao_promotionmisc_common_item_detail_list_get(
            self,
            activity_id,
            page_no,
            page_size
    ):
        """
        查询通用单品优惠详情列表
        查询通用单品优惠详情列表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25847

        :param activity_id: 优惠活动ID
        :param page_no: 分页页码，页码从1开始
        :param page_size: 分页大小，不能超过50
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.detail.list.get",
            {
                "activity_id": activity_id,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_promotionmisc_common_item_detail_get(
            self,
            activity_id,
            detail_id
    ):
        """
        查询通用单品优惠详情
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        查询通用单品优惠详情。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25848

        :param activity_id: 优惠活动ID
        :param detail_id: 优惠详情ID
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.detail.get",
            {
                "activity_id": activity_id,
                "detail_id": detail_id
            }
        )

    def taobao_promotionmisc_common_item_activity_add(
            self,
            name,
            description,
            start_time,
            end_time,
            is_user_tag='false',
            user_tag=''
    ):
        """
        创建通用单品优惠活动
        创建通用单品优惠活动。
        1、该接口只创建活动的基本信息，如需要增加、删除参与该活动的商品及优惠，请调用taobao.promotionmisc.common.item.detail.add和taobao.promotionmisc.common.item.detail.delete接口
        2、同一卖家下的活动数量限制为30个，超过限制需先调用taobao.promotionmisc.common.item.activity.delete接口删除无用的活动后才可再创建新的活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25849

        :param name: 活动名称，不能超过32字符
        :param description: 活动描述，不能超过100字符
        :param start_time: 活动开始时间
        :param end_time: 活动结束时间
        :param is_user_tag: 是否指定人群标签
        :param user_tag: 用户标签。当is_user_tag为true时，该值才有意义。
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.activity.add",
            {
                "name": name,
                "description": description,
                "start_time": start_time,
                "end_time": end_time,
                "is_user_tag": is_user_tag,
                "user_tag": user_tag
            }
        )

    def taobao_promotionmisc_common_item_detail_delete(
            self,
            activity_id,
            detail_id
    ):
        """
        删除通用单品优惠详情
        删除通用单品优惠详情。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25850

        :param activity_id: 优惠活动ID
        :param detail_id: 优惠详情ID
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.detail.delete",
            {
                "activity_id": activity_id,
                "detail_id": detail_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_common_item_detail_update(
            self,
            activity_id,
            detail_id,
            item_id,
            promotion_type,
            promotion_value
    ):
        """
        修改通用单品优惠详情
        修改通用单品优惠详情。
        1、该接口只修改活动下参与的商品的优惠信息，如需要增加、删除活动，请调用taobao.promotionmisc.common.item.activity.add和taobao.promotionmisc.common.item.activity.delete接口；
        2、使用该接口时需要把未做修改的字段值也传入；
        3、此接口受卖家最低折扣限制，如果优惠力度大于卖家设置的最低折扣则不能修改
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25851

        :param activity_id: 优惠活动ID
        :param detail_id: 优惠详情ID
        :param item_id: 商品ID
        :param promotion_type: 优惠类型，只有两种可选值：0-减钱；1-打折
        :param promotion_value: 优惠力度，其值的解释方式由promotion_type定义：当为减钱时解释成减钱数量，如：900表示减9元；当为打折时解释成打折折扣，如：900表示打9折
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.detail.update",
            {
                "activity_id": activity_id,
                "detail_id": detail_id,
                "item_id": item_id,
                "promotion_type": promotion_type,
                "promotion_value": promotion_value
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_common_item_detail_add(
            self,
            activity_id,
            item_id,
            promotion_type,
            promotion_value
    ):
        """
        创建通用单品优惠详情
        创建通用单品优惠详情。
        1、使用此接口在指定的优惠活动下创建参与的商品的优惠信息，如还未创建活动，需要先使用接口taobao.promotionmisc.common.item.activity.add创建优惠活动；
        2、同一卖家同一活动下的优惠详情数量限制为150个，超过限制需先调用taobao.promotionmisc.common.item.detail.delete接口删除无用的详情后才可再创建新的优惠详情；
        3、此接口受卖家最低折扣限制，如果优惠力度大于卖家设置的最低折扣则不能创建
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25852

        :param activity_id: 优惠活动ID
        :param item_id: 商品ID
        :param promotion_type: 优惠类型，只有两种可选值：0-减钱；1-打折
        :param promotion_value: 优惠力度，其值的解释方式由promotion_type定义：当为减钱时解释成减钱数量，如：900表示减9元；当为打折时解释成打折折扣，如：900表示打9折
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.detail.add",
            {
                "activity_id": activity_id,
                "item_id": item_id,
                "promotion_type": promotion_type,
                "promotion_value": promotion_value
            }
        )

    def taobao_promotionmisc_common_item_activity_list_get(
            self,
            page_no,
            page_size
    ):
        """
        查询通用单品优惠活动列表
        查询通用单品优惠活动列表。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25857

        :param page_no: 分页页码，页码从1开始
        :param page_size: 分页大小，不能超过50
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.activity.list.get",
            {
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_promotionmisc_common_item_activity_get(
            self,
            activity_id
    ):
        """
        查询通用单品优惠活动
        查询通用单品优惠活动。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25858

        :param activity_id: 优惠活动ID
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.activity.get",
            {
                "activity_id": activity_id
            }
        )

    def taobao_promotionmisc_common_item_activity_delete(
            self,
            activity_id
    ):
        """
        删除通用单品优惠活动
        删除通用单品优惠活动。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25860

        :param activity_id: 优惠活动ID
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.activity.delete",
            {
                "activity_id": activity_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_promotionmisc_common_item_activity_update(
            self,
            activity_id,
            name,
            description,
            start_time,
            end_time,
            is_user_tag='false',
            user_tag=''
    ):
        """
        修改通用单品优惠活动
        修改通用单品优惠活动。
        1、该接口只修改活动基本信息，如需要增加、删除参与该活动的商品及优惠，请调用taobao.promotionmisc.common.item.detail.add和taobao.promotionmisc.common.item.detail.delete接口
        2、使用该接口时需要把未做修改的字段值也传入
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25861

        :param activity_id: 优惠活动ID
        :param name: 活动名称，不能超过32字符
        :param description: 活动描述，不能超过100字符
        :param start_time: 活动开始时间
        :param end_time: 活动结束时间
        :param is_user_tag: 是否指定人群标签
        :param user_tag: 用户标签。当is_user_tag为true时，该值才有意义。
        """
        return self._top_request(
            "taobao.promotionmisc.common.item.activity.update",
            {
                "activity_id": activity_id,
                "name": name,
                "description": description,
                "start_time": start_time,
                "end_time": end_time,
                "is_user_tag": is_user_tag,
                "user_tag": user_tag
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_mobile_promotion_coupon_seller_search(
            self,
            trace_id='',
            spread_ids='',
            page_size='',
            current_page=''
    ):
        """
        查询绑定卖家优惠券相关信息(手淘专用)
        查询绑定卖家相关优惠券信息 如isv 百川 等外部业务方
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25902

        :param trace_id: 请求id 排查线索 需保证单次调用唯一
        :param spread_ids: 券id集合
        :param page_size: 每页数据 最大20左右
        :param current_page: 当前第几页 从第一页开始
        """
        return self._top_request(
            "taobao.mobile.promotion.coupon.seller.search",
            {
                "trace_id": trace_id,
                "spread_ids": spread_ids,
                "page_size": page_size,
                "current_page": current_page
            },
            result_processor=lambda x: x["coupon_search_result"]
        )

    def taobao_mobile_promotion_coupon_apply(
            self,
            trace_id,
            spread_id,
            feed_id,
            biz_id
    ):
        """
        优惠券领取(手淘专用)
        优惠券领取
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25903

        :param trace_id: 请求唯一id，问题排查
        :param spread_id: 传播id
        :param feed_id: 广播id
        :param biz_id: 三方活动id
        """
        return self._top_request(
            "taobao.mobile.promotion.coupon.apply",
            {
                "trace_id": trace_id,
                "spread_id": spread_id,
                "feed_id": feed_id,
                "biz_id": biz_id
            },
            result_processor=lambda x: x["coupon_apply_result"]
        )

    def alibaba_interact_lotterydraw_dodraw(
            self,
            lottery_draw_query
    ):
        """
        无线端抽奖接口
        商家抽奖平台无线端抽奖接口开放
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28513

        :param lottery_draw_query: 抽奖请求对象
        """
        return self._top_request(
            "alibaba.interact.lotterydraw.dodraw",
            {
                "lottery_draw_query": lottery_draw_query
            }
        )

    def alibaba_marketing_lottery_activity_query(
            self,
            lottery_activity_query
    ):
        """
        抽奖平台奖池查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28525

        :param lottery_activity_query: 查询抽奖活动请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.activity.query",
            {
                "lottery_activity_query": lottery_activity_query
            }
        )

    def alibaba_marketing_lottery_activity_bind(
            self,
            lottery_activity_rel
    ):
        """
        抽奖平台奖池绑定接口
        抽奖平台奖池关联接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28526

        :param lottery_activity_rel: 关联抽奖活动请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.activity.bind",
            {
                "lottery_activity_rel": lottery_activity_rel
            }
        )

    def alibaba_marketing_lottery_draw_dodraw(
            self,
            lottery_draw_query
    ):
        """
        抽奖平台抽奖接口
        抽奖平台PC端抽奖接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28527

        :param lottery_draw_query: 抽奖请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.draw.dodraw",
            {
                "lottery_draw_query": lottery_draw_query
            }
        )

    def alibaba_marketing_lottery_activity_create(
            self,
            lottery_activity_create
    ):
        """
        抽奖平台奖池创建接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28528

        :param lottery_activity_create: 抽奖活动创建请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.activity.create",
            {
                "lottery_activity_create": lottery_activity_create
            }
        )

    def alibaba_marketing_lottery_activity_unbind(
            self,
            lottery_activity_rel
    ):
        """
        抽奖平台奖池解绑接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28529

        :param lottery_activity_rel: 解绑抽奖活动请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.activity.unbind",
            {
                "lottery_activity_rel": lottery_activity_rel
            }
        )

    def alibaba_marketing_lottery_activity_delete(
            self,
            lottery_activity_delete
    ):
        """
        抽奖平台活动删除接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28530

        :param lottery_activity_delete: 抽奖活动删除对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.activity.delete",
            {
                "lottery_activity_delete": lottery_activity_delete
            }
        )

    def alibaba_marketing_lottery_award_query(
            self,
            lottery_award_inst_query
    ):
        """
        抽奖平台查询可用奖品接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28534

        :param lottery_award_inst_query: 查询奖品请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.award.query",
            {
                "lottery_award_inst_query": lottery_award_inst_query
            }
        )

    def alibaba_marketing_lottery_award_append(
            self,
            lottery_award_append
    ):
        """
        抽奖平台奖品添加接口
        抽奖平台奖品添加接口，目前仅用于奖池众筹项目
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28990

        :param lottery_award_append: 奖品添加请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.award.append",
            {
                "lottery_award_append": lottery_award_append
            }
        )

    def alibaba_marketing_lottery_schema_create(
            self,
            lottery_schema_create
    ):
        """
        抽奖平台抽奖方案创建接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29025

        :param lottery_schema_create: 创建抽奖方案请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.schema.create",
            {
                "lottery_schema_create": lottery_schema_create
            }
        )

    def alibaba_marketing_lottery_rule_save(
            self,
            lottery_rule_create
    ):
        """
        抽奖平台抽奖规则保存接口
        抽奖平台抽奖规则保存接口，对于同一主体，保存新规则会失效老的规则
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29026

        :param lottery_rule_create: 抽奖规则保存请求对象
        """
        return self._top_request(
            "alibaba.marketing.lottery.rule.save",
            {
                "lottery_rule_create": lottery_rule_create
            }
        )

    def alibaba_wdk_marketing_coupon_sendma(
            self,
            param0
    ):
        """
        发放匿名码
        根据优惠券活动id打印单个匿名码
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36399

        :param param0: 发放匿名码入参
        """
        return self._top_request(
            "alibaba.wdk.marketing.coupon.sendma",
            {
                "param0": param0
            }
        )

    def tmall_promotion_coupon_use(
            self,
            extra='',
            biz_type='',
            buyer_id='',
            buyer_nick='',
            seller_id='',
            coupon_id=''
    ):
        """
        券核销接口
        核销用户的一张优惠券，返回核销结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38483

        :param extra: 扩展字段
        :param biz_type: 业务类型
        :param buyer_id: buyer_id、buyer_nick至少填一个， 都填写以id为准
        :param buyer_nick: buyer_id、buyer_nick至少填一个， 都填写以id为准
        :param seller_id: 商家id
        :param coupon_id: 优惠券id
        """
        return self._top_request(
            "tmall.promotion.coupon.use",
            {
                "extra": extra,
                "biz_type": biz_type,
                "buyer_id": buyer_id,
                "buyer_nick": buyer_nick,
                "seller_id": seller_id,
                "coupon_id": coupon_id
            }
        )

    def tmall_promotion_coupon_query(
            self,
            biz_type,
            buyer_id='',
            buyer_nick=''
    ):
        """
        查询可用优惠券列表
        查询用户的可用优惠券列表，仅包含优惠券基本信息和用户nick
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38496

        :param biz_type: 业务类型
        :param buyer_id: buyer_id、buyer_nick至少填一个， 都填写以id为准
        :param buyer_nick: buyer_id、buyer_nick至少填一个， 都填写以id为准
        """
        return self._top_request(
            "tmall.promotion.coupon.query",
            {
                "biz_type": biz_type,
                "buyer_id": buyer_id,
                "buyer_nick": buyer_nick
            }
        )

    def tmall_promotion_coupon_user(
            self,
            biz_type='',
            pay_code='',
            extra=''
    ):
        """
        用户信息查询接口
        开发给外部合作商（例如：苏宁），通过会员付款码获得会员nick
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38606

        :param biz_type: 例如：suning
        :param pay_code: 会员付款码
        :param extra: 扩展字段
        """
        return self._top_request(
            "tmall.promotion.coupon.user",
            {
                "biz_type": biz_type,
                "pay_code": pay_code,
                "extra": extra
            }
        )

    def taobao_ump_promotion_global_discount_get(
            self
    ):
        """
        获取卖家最低折扣
        提供卖家最低折扣查询功能
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38626

        """
        return self._top_request(
            "taobao.ump.promotion.global.discount.get"
        )

    def tmall_seiya_unline_launch(
            self,
            params
    ):
        """
        seiya线下货品投放接口
        seiya线下第三方合作终端货品投放接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40652

        :param params: 投放请求参数
        """
        return self._top_request(
            "tmall.seiya.unline.launch",
            {
                "params": params
            }
        )

    def tmall_seiya_launch_monitor(
            self,
            task_id,
            asset_code,
            biz_id,
            model_ids,
            time,
            extend=''
    ):
        """
        天猫品牌营销线下广告监播上报接口
        天猫品牌营销线下投放广告监播上报接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=41906

        :param task_id: 投放任务id，每个业务方分配一个
        :param asset_code: 终端编号
        :param biz_id: 业务id
        :param model_ids: 投放物料id集合多个逗号分隔
        :param time: 终端拉取投放物料时间戳,毫秒
        :param extend: 扩展信息
        """
        return self._top_request(
            "tmall.seiya.launch.monitor",
            {
                "task_id": task_id,
                "asset_code": asset_code,
                "biz_id": biz_id,
                "model_ids": model_ids,
                "time": time,
                "extend": extend
            }
        )

    def tmall_seiya_unline_failover_launch(
            self,
            show_id,
            terminal_id,
            longitude,
            latitude,
            labels=''
    ):
        """
        seiya线下投放容灾接口
        seiya线下第三方合作终端货品投放接口(带容灾能力)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42598

        :param show_id: 投放展示id
        :param terminal_id: 下线终端id
        :param longitude: 经度
        :param latitude: 纬度
        :param labels: 标签
        """
        return self._top_request(
            "tmall.seiya.unline.failover.launch",
            {
                "show_id": show_id,
                "terminal_id": terminal_id,
                "longitude": longitude,
                "latitude": latitude,
                "labels": labels
            },
            result_processor=lambda x: x["result_list"]
        )

    def tmall_seiya_underline_upload(
            self,
            asset_code,
            exposur_count,
            pick_count,
            date
    ):
        """
        seiya线下三方合作商上报数据接口
        seiya线下三方合作商上报数据接口，比如曝光数、取件数等等，只写接口，无读取数据。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42984

        :param asset_code: 机柜编码
        :param exposur_count: 曝光次数
        :param pick_count: 取件次数
        :param date: 日期（YYYYMMDD）
        """
        return self._top_request(
            "tmall.seiya.underline.upload",
            {
                "asset_code": asset_code,
                "exposur_count": exposur_count,
                "pick_count": pick_count,
                "date": date
            }
        )

    def tmall_seiya_click_report(
            self,
            task_id,
            asset_code,
            screen_count,
            model_id,
            model_type,
            click_time
    ):
        """
        天猫来店业务三方广告公司点击上报接口
        天猫来店业务三方广告公司线下点位用户互动点击上报接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43340

        :param task_id: 广告所属投放任务id(广告体taskId属性)
        :param asset_code: 机柜id
        :param screen_count: 第几屏点击
        :param model_id: 广告id（itemid）
        :param model_type: 广告类型
        :param click_time: 终端发生点击时间戳(单位毫秒)
        """
        return self._top_request(
            "tmall.seiya.click.report",
            {
                "task_id": task_id,
                "asset_code": asset_code,
                "screen_count": screen_count,
                "model_id": model_id,
                "model_type": model_type,
                "click_time": click_time
            }
        )


class TbShuJu(DingTalkBaseAPI):
    """
    数据API
    """

    def yunos_datatech_opendata_get(
            self,
            context=None,
            rdam_request=None
    ):
        """
        YunOS服务平台数据获取通用接口
        可以通过此接口拿到yunos服务平台对外提供的聚合数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27508

        :param context: 上下文参数
        :param rdam_request: 系统自动生成
        """
        return self._top_request(
            "yunos.datatech.opendata.get",
            {
                "context": context,
                "rdam_request": rdam_request
            }
        )

    def alibaba_dt_tmllcar_leadsinfo(
            self,
            shop_code,
            app_name,
            name,
            password
    ):
        """
        天猫汽车线索产品潜客数据
        1.	线索分发是天猫汽车行业流量端最中要的产品，经过前两年的业务和数据端的积累已经对整体业务流程和方案有了清晰的思路；目前数据段已经产沉淀2000W汽车潜客数据，通过运营尝试得到了较好的效果，今年将通过与商家端合作（大搜车-卖车管家）完成潜客分发-商家报价-潜客触达-线索分发-线下核销等一整个汽车人群运营闭环；这个接口反馈大搜车线下门店周围潜客规模及热门车型数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38780

        :param shop_code: shopcode
        :param app_name: app_name
        :param name: name
        :param password: pssword
        """
        return self._top_request(
            "alibaba.dt.tmllcar.leadsinfo",
            {
                "shop_code": shop_code,
                "app_name": app_name,
                "name": name,
                "password": password
            }
        )

    def alibaba_dt_tmllcar_pricevalidate(
            self,
            app_name,
            name,
            password,
            price,
            city_name,
            model_name
    ):
        """
        线索报价价格校验
        根据选定的车型和城市，校验汽车价格是否通过
        入参：车型ID，城市名称，价格
        输出：N 校验失败，校验成功不返回值
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=40216

        :param app_name: tt
        :param name: tt
        :param password: tt
        :param price: tt
        :param city_name: tt
        :param model_name: tt
        """
        return self._top_request(
            "alibaba.dt.tmllcar.pricevalidate",
            {
                "app_name": app_name,
                "name": name,
                "password": password,
                "price": price,
                "city_name": city_name,
                "model_name": model_name
            },
            result_processor=lambda x: x["results"]
        )


class TbJiuDian(DingTalkBaseAPI):
    """
    酒店API
    """

    def taobao_xhotel_rooms_ids_get(
            self,
            start_date='',
            end_date='',
            page_no='1'
    ):
        """
        已废弃，勿用
        请使用商家后台导出即可
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23281

        :param start_date: 创建宝贝起始时间
        :param end_date: 创建宝贝截止时间
        :param page_no: 页数
        """
        return self._top_request(
            "taobao.xhotel.rooms.ids.get",
            {
                "start_date": start_date,
                "end_date": end_date,
                "page_no": page_no
            }
        )

    def taobao_xhotel_order_official_qualification_get(
            self,
            notify_url,
            return_url,
            guest_name,
            out_member_account,
            id_number,
            check_out,
            check_in,
            out_u_u_i_d,
            total_fee,
            hotel_code,
            out_oid,
            extend_attrs='',
            mobile_no='',
            vendor='',
            id_type='',
            encrypt_type='',
            alipay_account='',
            daily_price_info='',
            room_num=''
    ):
        """
        官网信用住用户资质校验
        官网信用住在下单前对用户进行资质校验，资质校验通过才能进行信用支付
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26041

        :param notify_url: 卖家接收阿里旅行订单状态变更的服务地址（需要实现阿里旅行提供的服务通知规范）
        :param return_url: 阿里旅行支付（下单）结束后跳转卖家的页面地址（必须）
        :param guest_name: 入住人姓名（必选）
        :param out_member_account: 外部会员账号（必选）
        :param id_number: 身份证号，必选
        :param check_out: 客人离店日期, 最多支持9间夜
        :param check_in: 客人入住日期
        :param out_u_u_i_d: 外部请求序列表号\流水号，单次请求的唯一标识(必须)
        :param total_fee: 总的收费金额，单位为分(必须)
        :param hotel_code: 酒店外部编码
        :param out_oid: 外部订单号（必选），阿里旅行会根据此值进行幂等性校验
        :param extend_attrs: 扩展字段，json串，用户后续的营销、统计、定制等需求，目前已有key列表：      is_new_user：是否是卖家新用户，1-是，0或者key为null，表示不是      is_first_stay：是否是卖家首住，1-是，0或者key为null，表示不是     （已有列表必须传递）
        :param mobile_no: 用户手机号(可选)
        :param vendor: 商家在淘宝给分配的渠道名（建议填充较好）
        :param id_type: 证件类型, 默认0:身份证; 1: 护照; 2:警官证; 3:士兵证; 4: 回乡证。目前只支持身份证
        :param encrypt_type: 加密方式, 默认0: 不加密, 信息会通过淘宝开放平台传输, 阿里旅行可以获取到具体信息;      * 目前只支持不加密
        :param alipay_account: 用户支付宝唯一识别码(可选)
        :param daily_price_info: 每日房价,json格式 ，如果是多间房，则是每日多间房总房价(可选)      * eg:{'day':'2015-08-12','price':48800},      {'day':'2015-08-13','price':48800}
        :param room_num: 房间数
        """
        return self._top_request(
            "taobao.xhotel.order.official.qualification.get",
            {
                "notify_url": notify_url,
                "return_url": return_url,
                "guest_name": guest_name,
                "out_member_account": out_member_account,
                "id_number": id_number,
                "check_out": check_out,
                "check_in": check_in,
                "out_u_u_i_d": out_u_u_i_d,
                "total_fee": total_fee,
                "hotel_code": hotel_code,
                "out_oid": out_oid,
                "extend_attrs": extend_attrs,
                "mobile_no": mobile_no,
                "vendor": vendor,
                "id_type": id_type,
                "encrypt_type": encrypt_type,
                "alipay_account": alipay_account,
                "daily_price_info": daily_price_info,
                "room_num": room_num
            }
        )

    def taobao_xhotel_order_offline_settle_cancel(
            self,
            tid='',
            reason='',
            out_id='',
            notify_url='',
            out_uuid=''
    ):
        """
        线下信用住取消结账专用接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26105

        :param tid: 阿里旅行订单号，淘宝订单号或外部订单号二选一必填
        :param reason: 取消结账的原因
        :param out_id: 外部订单号，和tid二选一必填（建议都写入）
        :param notify_url: 暂时无意义，无需传入
        :param out_uuid: 请求流水号
        """
        return self._top_request(
            "taobao.xhotel.order.offline.settle.cancel",
            {
                "tid": tid,
                "reason": reason,
                "out_id": out_id,
                "notify_url": notify_url,
                "out_uuid": out_uuid
            }
        )

    def taobao_xhotel_order_hotelsign_query(
            self,
            out_uuid,
            hotel_code='',
            vendor='taobao',
            type='1',
            page_no='1'
    ):
        """
        获取直连酒店（客栈）签约上线进度信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27152

        :param out_uuid: 请求流水
        :param hotel_code: 商家酒店编码
        :param vendor: 商家vendor
        :param type: 1
        :param page_no: 页码
        """
        return self._top_request(
            "taobao.xhotel.order.hotelsign.query",
            {
                "out_uuid": out_uuid,
                "hotel_code": hotel_code,
                "vendor": vendor,
                "type": type,
                "page_no": page_no
            }
        )

    def taobao_xhotel_data_service_seller_serviceindex(
            self,
            vendor='',
            start_row='',
            page_size='',
            report_end_date='',
            report_start_date=''
    ):
        """
        卖家服务指数查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38661

        :param vendor: 渠道商名称
        :param start_row: 分页参数
        :param page_size: 分页参数
        :param report_end_date: 查询截止日期
        :param report_start_date: 查询开始日期
        """
        return self._top_request(
            "taobao.xhotel.data.service.seller.serviceindex",
            {
                "vendor": vendor,
                "start_row": start_row,
                "page_size": page_size,
                "report_end_date": report_end_date,
                "report_start_date": report_start_date
            }
        )

    def taobao_xhotel_data_service_hotel_serviceindex(
            self,
            hid='',
            vendor='',
            start_row='',
            page_size='',
            report_end_date='',
            report_start_date='',
            supplier=''
    ):
        """
        酒店服务指数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38848

        :param hid: 酒店id
        :param vendor: 渠道商名称
        :param start_row: 1
        :param page_size: 10
        :param report_end_date: 查询时间段结束
        :param report_start_date: 查询时间段开始
        :param supplier: 供应商名称
        """
        return self._top_request(
            "taobao.xhotel.data.service.hotel.serviceindex",
            {
                "hid": hid,
                "vendor": vendor,
                "start_row": start_row,
                "page_size": page_size,
                "report_end_date": report_end_date,
                "report_start_date": report_start_date,
                "supplier": supplier
            }
        )

    def taobao_xhotel_data_service_order_detail(
            self,
            start_date='',
            is_call_no_order='',
            hid='',
            is_spec_time_order='',
            vendor='',
            page_size='',
            end_date='',
            is_no_room_compen='',
            start_row='',
            is_seller_deny='',
            is_seller_refund='',
            supplier=''
    ):
        """
        服务订单详情
        服务订单详情top接口构建
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38872

        :param start_date: 查询开始日期
        :param is_call_no_order: 是否查无订单 1:是 0:否
        :param hid: 酒店id
        :param is_spec_time_order: 是否特殊时段订单 1:是 0:否
        :param vendor: 渠道商名称
        :param page_size: 分页大小
        :param end_date: 查询结束时间
        :param is_no_room_compen: 是否到店无房 1:是
        :param start_row: 分页参数
        :param is_seller_deny: 是否拒单 1:是 0:否
        :param is_seller_refund: 是否卖家原因退款 1:是 0:否
        :param supplier: 供应商名称
        """
        return self._top_request(
            "taobao.xhotel.data.service.order.detail",
            {
                "start_date": start_date,
                "is_call_no_order": is_call_no_order,
                "hid": hid,
                "is_spec_time_order": is_spec_time_order,
                "vendor": vendor,
                "page_size": page_size,
                "end_date": end_date,
                "is_no_room_compen": is_no_room_compen,
                "start_row": start_row,
                "is_seller_deny": is_seller_deny,
                "is_seller_refund": is_seller_refund,
                "supplier": supplier
            }
        )


class TbJuHuaSuan(DingTalkBaseAPI):
    """
    聚划算API
    """

    def taobao_ju_items_search(
            self,
            param_top_item_query=None
    ):
        """
        聚划算商品搜索接口
        搜索聚划算商品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28762

        :param param_top_item_query: query
        """
        return self._top_request(
            "taobao.ju.items.search",
            {
                "param_top_item_query": param_top_item_query
            }
        )


class TbDianPuHuiYuanGuanLi(DingTalkBaseAPI):
    """
    店铺会员管理API
    """

    def taobao_crm_grade_get(
            self
    ):
        """
        卖家查询等级规则
        卖家查询等级规则，包括店铺客户、普通会员、高级会员、VIP会员、至尊VIP会员四个等级的信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10824

        """
        return self._top_request(
            "taobao.crm.grade.get",
            result_processor=lambda x: x["grade_promotions"]
        )

    def taobao_crm_shopvip_cancel(
            self
    ):
        """
        卖家取消店铺vip的优惠
        此接口用于取消VIP优惠
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10825

        """
        return self._top_request(
            "taobao.crm.shopvip.cancel",
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_members_increment_get(
            self,
            current_page,
            grade='-1',
            start_modify='',
            end_modify='',
            page_size=''
    ):
        """
        增量获取卖家会员
        增量获取会员列表，接口返回符合查询条件的所有会员。任何状态更改都会返回,最大允许100
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10826

        :param current_page: 显示第几页的会员，如果输入的页码大于总共的页码数，例如总共10页，但是current_page的值为11，则返回空白页，最小页数为1
        :param grade: 会员等级
        :param start_modify: 卖家修改会员信息的时间起点.
        :param end_modify: 卖家修改会员信息的时间终点.如果不填写此字段,默认为当前时间.
        :param page_size: 每页显示的会员数，page_size的值不能超过100，最小值要大于1
        """
        return self._top_request(
            "taobao.crm.members.increment.get",
            {
                "current_page": current_page,
                "grade": grade,
                "start_modify": start_modify,
                "end_modify": end_modify,
                "page_size": page_size
            }
        )

    def taobao_crm_members_group_batchadd(
            self,
            group_ids,
            buyer_nicks
    ):
        """
        给一批会员添加一个分组
        为一批会员添加分组，接口返回添加是否成功,如至少有一个会员的分组添加成功，接口就返回成功，否则返回失败，如果当前会员已经拥有当前分组，则直接跳过
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10829

        :param group_ids: 分组id
        :param buyer_nicks: 买家昵称列表
        """
        return self._top_request(
            "taobao.crm.members.group.batchadd",
            {
                "group_ids": group_ids,
                "buyer_nicks": buyer_nicks
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_grade_set(
            self,
            grade,
            discount,
            hierarchy,
            amount='',
            count=''
    ):
        """
        卖家设置等级规则
        设置等级信息，可以设置层级等级，也可以单独设置一个等级。出于安全原因，折扣现最低只能设置到700即7折。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10831

        :param grade: 会员等级，用逗号分隔。买家会员级别0：店铺客户 1：普通会员 2 ：高级会员 3：VIP会员 4：至尊VIP
        :param discount: 会员级别折扣率。会员等级越高，折扣必须越低。<br/>950即9.5折，888折即8.88折。出于安全原因，折扣现最低只能设置到700即7折。
        :param hierarchy: 是否设置达到某一会员等级的交易量和交易额，必填。4个级别都需要设置，如入参为true,true,true,false时，表示设置达到高级会员、VIP会员的交易量或者交易额，不设置达到至尊会员的交易量和交易额
        :param amount: 只对设置的层级等级有效，必须要在amount和count参数中选择一个<br><br/>amount参数的填写规范：升级到下一个级别的需要的交易额，单位为分,必须全部填写.例如10000,20000,30000，其中10000表示非会员升级到普通的所需的交易额，20000表示普通升级到高级所需的交易额，层级等级中最高等级的下一个等级默认为0。会员等级越高，所需交易额必须越高。
        :param count: 只对设置的层级等级有效，必须要在amount和count参数中选择一个<br><br/>count参数的填写规范：<br/>升级到下一个级别的需要的交易量,必须全部填写. 以逗号分隔,例如100,200,300，其中100表示非会员升级到普通会员交易量。层级等级中最高等级的下一个等级的交易量默认为0。会员等级越高，交易量必须越高。
        """
        return self._top_request(
            "taobao.crm.grade.set",
            {
                "grade": grade,
                "discount": discount,
                "hierarchy": hierarchy,
                "amount": amount,
                "count": count
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_members_groups_batchdelete(
            self,
            buyer_nicks,
            group_ids
    ):
        """
        批量删除分组
        批量删除多个会员的公共分组，接口返回删除是否成功，该接口只删除多个会员的公共分组，不是公共分组的，不进行删除。如果入参只输入一个会员，则表示删除该会员的某些分组。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10832

        :param buyer_nicks: 买家昵称列表
        :param group_ids: 会员需要删除的分组
        """
        return self._top_request(
            "taobao.crm.members.groups.batchdelete",
            {
                "buyer_nicks": buyer_nicks,
                "group_ids": group_ids
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_groups_get(
            self,
            current_page,
            page_size=''
    ):
        """
        查询卖家的分组
        查询卖家的分组，返回查询到的分组列表，分页返回分组
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10833

        :param current_page: 显示第几页的分组，如果输入的页码大于总共的页码数，例如总共10页，但是current_page的值为11，则返回空白页，最小页码为1
        :param page_size: 每页显示的记录数，其最大值不能超过100条，最小值为1，默认20条
        """
        return self._top_request(
            "taobao.crm.groups.get",
            {
                "current_page": current_page,
                "page_size": page_size
            }
        )

    def taobao_crm_group_update(
            self,
            group_id,
            new_group_name
    ):
        """
        修改一个已经存在的分组
        修改一个已经存在的分组，接口返回分组的修改是否成功
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10834

        :param group_id: 分组的id
        :param new_group_name: 新的分组名，分组名称不能包含|或者：
        """
        return self._top_request(
            "taobao.crm.group.update",
            {
                "group_id": group_id,
                "new_group_name": new_group_name
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_group_add(
            self,
            group_name
    ):
        """
        卖家创建一个分组
        卖家创建一个新的分组，接口返回一个创建成功的分组的id
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10835

        :param group_name: 分组名称，每个卖家最多可以拥有100个分组
        """
        return self._top_request(
            "taobao.crm.group.add",
            {
                "group_name": group_name
            }
        )

    def taobao_crm_members_search(
            self,
            current_page,
            buyer_nick='',
            grade='-1',
            min_trade_amount='',
            max_trade_amount='',
            min_trade_count='',
            max_trade_count='',
            min_last_trade_time='',
            max_last_trade_time='',
            relation_source='',
            min_item_num='',
            province='',
            group_id='',
            page_size='',
            max_item_num=''
    ):
        """
        获取卖家会员（高级查询）
        会员列表的高级查询，接口返回符合条件的会员列表.<br><br/>注：建议获取09年以后的数据，09年之前的数据不是很完整
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10836

        :param current_page: 显示第几页的会员，如果输入的页码大于总共的页码数，例如总共10页，但是current_page的值为11，则返回空白页，最小页数为1.最大1000页
        :param buyer_nick: 买家昵称
        :param grade: 会员等级
        :param min_trade_amount: 最小交易额，单位为元
        :param max_trade_amount: 最大交易额，单位为元
        :param min_trade_count: 最小交易量
        :param max_trade_count: 最大交易量
        :param min_last_trade_time: 最早上次交易时间（订单创建时间）
        :param max_last_trade_time: 最迟上次交易时间
        :param relation_source: 关系来源，1交易成功，2未成交，3卖家手动吸纳
        :param min_item_num: 最小交易宝贝件数
        :param province: 北京=1,天津=2,河北省=3,山西省=4,内蒙古自治区=5,辽宁省=6,吉林省=7,黑龙江省=8,上海=9,江苏省=10,浙江省=11,安徽省=12,福建省=13,江西省=14,山东省=15,河南省=16,湖北省=17,湖南省=18, 广东省=19,广西壮族自治区=20,海南省=21,重庆=22,四川省=23,贵州省=24,云南省=25,西藏自治区26,陕西省=27,甘肃省=28,青海省=29,宁夏回族自治区=30,新疆维吾尔自治区=31,台湾省=32,香港特别行政区=33,澳门特别行政区=34,海外=35.注:请注意:从2014.4.23之后,省市将采用地区标准码,请通过物流API taobao.areas.get接口获取,参考:http://api.taobao.com/apidoc/api.htm?spm=0.0.0.0.nOOF9g&path=cid:7-apiId:59.API对于老的省市代码兼容会逐步下线.
        :param group_id: 分组id
        :param page_size: 每页显示的会员数量，page_size的最大值不能超过100，最小值不能小于1
        :param max_item_num: 最大交易宝贝件数
        """
        return self._top_request(
            "taobao.crm.members.search",
            {
                "current_page": current_page,
                "buyer_nick": buyer_nick,
                "grade": grade,
                "min_trade_amount": min_trade_amount,
                "max_trade_amount": max_trade_amount,
                "min_trade_count": min_trade_count,
                "max_trade_count": max_trade_count,
                "min_last_trade_time": min_last_trade_time,
                "max_last_trade_time": max_last_trade_time,
                "relation_source": relation_source,
                "min_item_num": min_item_num,
                "province": province,
                "group_id": group_id,
                "page_size": page_size,
                "max_item_num": max_item_num
            }
        )

    def taobao_crm_memberinfo_update(
            self,
            buyer_nick,
            status,
            grade='',
            province='',
            city='',
            trade_count='',
            trade_amount='',
            close_trade_count='',
            close_trade_amount='',
            group_ids='',
            item_num=''
    ):
        """
        编辑会员资料
        编辑会员的基本资料，接口返回会员信息修改是否成功
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10837

        :param buyer_nick: 买家昵称
        :param status: 用于描述会员的状态，normal表示正常，blacklist表示黑名单(不享受会员折扣).
        :param grade: 会员等级，1：普通客户，2：高级会员，3：高级会员 ，4：至尊vip只有正常会员才给予升级，对于status blacklist的会员升级无效
        :param province: 北京=1,天津=2,河北省=3,山西省=4,内蒙古自治区=5,辽宁省=6,吉林省=7,黑龙江省=8,上海=9,江苏省=10,浙江省=11,安徽省=12,福建省=13,江西省=14,山东省=15,河南省=16,湖北省=17,湖南省=18, 广东省=19,广西壮族自治区=20,海南省=21,重庆=22,四川省=23,贵州省=24,云南省=25,西藏自治区=26,陕西省=27,甘肃省=28,青海省=29,宁夏回族自治区=30,新疆维吾尔自治区=31,台湾省=32,香港特别行政区=33,澳门特别行政区=34,海外=35，约定36为清除Province设置.请注意:从2014.4.15之后,省市将采用地区标准码,请通过物流API taobao.areas.get接口获取,参考:http://api.taobao.com/apidoc/api.htm?spm=0.0.0.0.nOOF9g&path=cid:7-apiId:59.API对于老的省市代码兼容会逐步下线.
        :param city: 城市.请注意:从2014.4.15之后,省市将采用地区标准码,请通过物流API taobao.areas.get接口获取,参考:http://api.taobao.com/apidoc/api.htm?spm=0.0.0.0.nOOF9g&path=cid:7-apiId:59.API对于老的省市代码兼容会逐步下线
        :param trade_count: 交易笔数
        :param trade_amount: 交易金额，单位：分
        :param close_trade_count: 交易关闭次数
        :param close_trade_amount: 交易关闭金额，单位：分
        :param group_ids: 分组的id集合字符串
        :param item_num: 宝贝件数
        """
        return self._top_request(
            "taobao.crm.memberinfo.update",
            {
                "buyer_nick": buyer_nick,
                "status": status,
                "grade": grade,
                "province": province,
                "city": city,
                "trade_count": trade_count,
                "trade_amount": trade_amount,
                "close_trade_count": close_trade_count,
                "close_trade_amount": close_trade_amount,
                "group_ids": group_ids,
                "item_num": item_num
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_members_get(
            self,
            current_page,
            buyer_nick='',
            grade='-1',
            min_trade_amount='',
            max_trade_amount='',
            min_trade_count='',
            max_trade_count='',
            min_last_trade_time='',
            max_last_trade_time='',
            page_size=''
    ):
        """
        获取卖家的会员（基本查询）
        查询卖家的会员，进行基本的查询，返回符合条件的会员列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10838

        :param current_page: 显示第几页的会员，如果输入的页码大于总共的页码数，例如总共10页，但是current_page的值为11，则返回空白页，最小页数为1，最大页数为1000
        :param buyer_nick: 买家的昵称
        :param grade: 会员等级,如果不传入值则默认为全部等级。
        :param min_trade_amount: 最小交易额,单位为元
        :param max_trade_amount: 最大交易额，单位为元
        :param min_trade_count: 最小交易量
        :param max_trade_count: 最大交易量
        :param min_last_trade_time: 最早上次交易时间
        :param max_last_trade_time: 最迟上次交易时间
        :param page_size: 表示每页显示的会员数量,page_size的最大值不能超过100条,最小值不能低于1，
        """
        return self._top_request(
            "taobao.crm.members.get",
            {
                "current_page": current_page,
                "buyer_nick": buyer_nick,
                "grade": grade,
                "min_trade_amount": min_trade_amount,
                "max_trade_amount": max_trade_amount,
                "min_trade_count": min_trade_count,
                "max_trade_count": max_trade_count,
                "min_last_trade_time": min_last_trade_time,
                "max_last_trade_time": max_last_trade_time,
                "page_size": page_size
            }
        )

    def taobao_crm_group_delete(
            self,
            group_id
    ):
        """
        删除分组
        将该分组下的所有会员移除出该组，同时删除该分组。注：删除分组为异步任务，必须先调用taobao.crm.grouptask.check 确保涉及属性上没有任务。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10841

        :param group_id: 要删除的分组id
        """
        return self._top_request(
            "taobao.crm.group.delete",
            {
                "group_id": group_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_group_append(
            self,
            from_group_id,
            to_group_id
    ):
        """
        将一个分组添加到另外一个分组
        将某分组下的所有会员添加到另一个分组,注：1.该操作为异步任务，建议先调用taobao.crm.grouptask.check 确保涉及分组上没有任务；2.若分组下某会员分组数超最大限额，则该会员不会被添加到新分组，同时不影响其余会员添加分组，接口调用依然返回成功。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10842

        :param from_group_id: 添加的来源分组
        :param to_group_id: 添加的目标分组
        """
        return self._top_request(
            "taobao.crm.group.append",
            {
                "from_group_id": from_group_id,
                "to_group_id": to_group_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_group_move(
            self,
            from_group_id,
            to_group_id
    ):
        """
        分组移动
        将一个分组下的所有会员移动到另一个分组，会员从原分组中删除<br/>注：移动属性为异步任务建议先调用taobao.crm.grouptask.check 确保涉及属性上没有任务。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10843

        :param from_group_id: 需要移动的分组
        :param to_group_id: 目的分组
        """
        return self._top_request(
            "taobao.crm.group.move",
            {
                "from_group_id": from_group_id,
                "to_group_id": to_group_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_crm_grouptask_check(
            self,
            group_id
    ):
        """
        查询分组任务是否完成
        检查一个分组上是否有异步任务,异步任务包括1.将一个分组下的所有用户添加到另外一个分组2.将一个分组下的所有用户移动到另外一个分组3.删除某个分组<br/>若分组上有任务则该属性不能被操作。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10844

        :param group_id: 分组id
        """
        return self._top_request(
            "taobao.crm.grouptask.check",
            {
                "group_id": group_id
            },
            result_processor=lambda x: x["is_finished"]
        )

    def taobao_crm_grademkt_member_add(
            self,
            parameter,
            feather,
            buyer_nick
    ):
        """
        会员等级营销-会员吸纳
        商家通过该接口吸纳线上店铺会员。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21528

        :param parameter: 会员属性-json format生成方法见http://open.taobao.com/doc/detail.htm?id=101281
        :param feather: 系统属性，json格式
        :param buyer_nick: 会员nick
        """
        return self._top_request(
            "taobao.crm.grademkt.member.add",
            {
                "parameter": parameter,
                "feather": feather,
                "buyer_nick": buyer_nick
            },
            result_processor=lambda x: x["model"]
        )

    def taobao_crm_grademkt_member_detail_query(
            self,
            feather,
            parameter
    ):
        """
        会员等级营销-等级营销活动查询
        商家通过该接口查询等级营销活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21531

        :param feather: 扩展字段
        :param parameter: 创建营销详情，生成方法见http://open.taobao.com/doc/detail.htm?id=101281
        """
        return self._top_request(
            "taobao.crm.grademkt.member.detail.query",
            {
                "feather": feather,
                "parameter": parameter
            },
            result_processor=lambda x: x["model"]
        )

    def taobao_crm_grademkt_member_detail_delete(
            self,
            feather,
            parameter
    ):
        """
        会员等级营销-删除商品等级营销明细
        删除商品等级营销明细
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21533

        :param feather: 扩展字段
        :param parameter: 创建营销详情，生成方法见http://open.taobao.com/doc/detail.htm?id=101281
        """
        return self._top_request(
            "taobao.crm.grademkt.member.detail.delete",
            {
                "feather": feather,
                "parameter": parameter
            },
            result_processor=lambda x: x["module"]
        )

    def taobao_crm_grademkt_member_detail_create(
            self,
            feather,
            parameter
    ):
        """
        会员等级营销-创建商品等级营销明细
        创建商品等级营销明细
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21534

        :param feather: 扩展字段
        :param parameter: 创建营销详情，生成方法见http://open.taobao.com/doc/detail.htm?id=101281
        """
        return self._top_request(
            "taobao.crm.grademkt.member.detail.create",
            {
                "feather": feather,
                "parameter": parameter
            },
            result_processor=lambda x: x["module"]
        )

    def taobao_crm_grademkt_member_query(
            self,
            parameter,
            feather,
            buyer_nick
    ):
        """
        会员等级营销-会员关系查询
        商家通过该接口查询线上店铺会员。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21536

        :param parameter: 会员属性-json format生成方法见http://open.taobao.com/doc/detail.htm?id=101281
        :param feather: 系统属性，json格式
        :param buyer_nick: 会员nick
        """
        return self._top_request(
            "taobao.crm.grademkt.member.query",
            {
                "parameter": parameter,
                "feather": feather,
                "buyer_nick": buyer_nick
            },
            result_processor=lambda x: x["module"]
        )

    def taobao_crm_member_group_get(
            self,
            buyer_nick
    ):
        """
        获取买家身上的标签
        获取买家身上的标签，不返回标签的总人数
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22293

        :param buyer_nick: 会员Nick
        """
        return self._top_request(
            "taobao.crm.member.group.get",
            {
                "buyer_nick": buyer_nick
            },
            result_processor=lambda x: x["groups"]
        )

    def taobao_crm_service_channel_shortlink_create(
            self,
            short_link_name,
            link_type,
            short_link_data=''
    ):
        """
        ECRM创建淘短链服务
        可生成店铺宝贝、店铺首页、活动链接、订单链接等4种可呼起手机淘宝APP至对应页面的淘短链。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25034

        :param short_link_name: 淘短链名称（最多只能16个中文字符，类型为订单链接时传入订单ID）。
        :param link_type: 淘短链类型：LT_ITEM（商品淘短链）,LT_SHOP（店铺首页淘短链）,LT_ACTIVITY（活动页淘短链）,LT_TRADE（订单详情页淘短链）。
        :param short_link_data: 类型为LT_ITEM时必须传入商品ID，类型为LT_SHOP时必须传入空值，类型为LT_ACTIVITY时传入活动页URL（URL必须是taobao.com,tmall.com,jaeapp.com这三个域名下的URL），类型为LT_TRADE时传入订单ID。
        """
        return self._top_request(
            "taobao.crm.service.channel.shortlink.create",
            {
                "short_link_name": short_link_name,
                "link_type": link_type,
                "short_link_data": short_link_data
            },
            result_processor=lambda x: x["short_link"]
        )

    def taobao_crm_card_update(
            self,
            card_name,
            card_type,
            card_url,
            attribute=''
    ):
        """
        添加外部卡信息
        会员卡开放后，业务方通过调用此接口添加外部卡信息，需要对每个卖家都调用一次，去更新此卖家相应外部卡的地址、名称、图片等信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25283

        :param card_name: 卡名称，如校园卡、天猫美妆卡
        :param card_type: 卡类型
        :param card_url: 卖家此卡的URL
        :param attribute: 属性，卡的其它信息
        """
        return self._top_request(
            "taobao.crm.card.update",
            {
                "card_name": card_name,
                "card_type": card_type,
                "card_url": card_url,
                "attribute": attribute
            }
        )

    def taobao_crm_exchange_crowdinstance_add(
            self,
            reason,
            crowd_instance_id,
            buyer_nick
    ):
        """
        向活动人群实例中增加买家
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25567

        :param reason: 操作原因
        :param crowd_instance_id: 人群实例ID
        :param buyer_nick: 买家昵称
        """
        return self._top_request(
            "taobao.crm.exchange.crowdinstance.add",
            {
                "reason": reason,
                "crowd_instance_id": crowd_instance_id,
                "buyer_nick": buyer_nick
            },
            result_processor=lambda x: x["sub_success"]
        )

    def taobao_crm_exchange_crowdinstance_delete(
            self,
            reason,
            crowd_instance_id,
            buyer_nick
    ):
        """
        删除人群实例中的指定买家
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25583

        :param reason: 操作原因
        :param crowd_instance_id: 人群实例ID
        :param buyer_nick: 买家昵称
        """
        return self._top_request(
            "taobao.crm.exchange.crowdinstance.delete",
            {
                "reason": reason,
                "crowd_instance_id": crowd_instance_id,
                "buyer_nick": buyer_nick
            },
            result_processor=lambda x: x["sub_success"]
        )

    def taobao_crm_exchange_activity_create(
            self,
            exchange_activity_create_dto
    ):
        """
        创建积分兑换活动
        创建针对积分兑换类型的活动
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25584

        :param exchange_activity_create_dto: 创建积分兑换活动
        """
        return self._top_request(
            "taobao.crm.exchange.activity.create",
            {
                "exchange_activity_create_dto": exchange_activity_create_dto
            }
        )


class TbTaoDianDianWaiMai(DingTalkBaseAPI):
    """
    淘点点外卖
    """

    def taobao_waimai_delivery_location_update(
            self,
            deliverer_name,
            lng,
            deliverer_phone,
            lat
    ):
        """
        点点送配送员位置POI回传
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24874

        :param deliverer_name: 配送员姓名
        :param lng: 地图坐标：经度
        :param deliverer_phone: 配送员手机号码
        :param lat: 地图坐标：维度
        """
        return self._top_request(
            "taobao.waimai.delivery.location.update",
            {
                "deliverer_name": deliverer_name,
                "lng": lng,
                "deliverer_phone": deliverer_phone,
                "lat": lat
            },
            result_processor=lambda x: x["delivery_location_result"]
        )

    def taobao_waimai_delivery_confirm(
            self,
            delivery_order_no,
            lng,
            lat
    ):
        """
        点点送配送员妥投
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24876

        :param delivery_order_no: 物流订单号
        :param lng: 地图坐标：经度
        :param lat: 地图坐标：维度
        """
        return self._top_request(
            "taobao.waimai.delivery.confirm",
            {
                "delivery_order_no": delivery_order_no,
                "lng": lng,
                "lat": lat
            },
            result_processor=lambda x: x["delivery_order_result"]
        )

    def taobao_waimai_delivery_pickup(
            self,
            delivery_order_no,
            lng,
            lat
    ):
        """
        点点送配送员取件
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        点点送配送员取件接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=24877

        :param delivery_order_no: 物流订单号
        :param lng: 地图坐标：经度
        :param lat: 地图坐标：维度
        """
        return self._top_request(
            "taobao.waimai.delivery.pickup",
            {
                "delivery_order_no": delivery_order_no,
                "lng": lng,
                "lat": lat
            },
            result_processor=lambda x: x["delivery_order_result"]
        )

    def taobao_waimai_order_accept(
            self,
            delivery_order_no,
            deliverer_id='',
            deliverer_name='',
            deliverer_phone='',
            card_no='',
            lng='',
            lat=''
    ):
        """
        淘点点订单确认接单接口
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        淘点点订单通知后，配送商确认接单接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25277

        :param delivery_order_no: 物流单号
        :param deliverer_id: 配送员id
        :param deliverer_name: 配送员姓名
        :param deliverer_phone: 配送员手机号
        :param card_no: 配送员身份证号
        :param lng: 配送员位置：经度
        :param lat: 配送员位置：维度
        """
        return self._top_request(
            "taobao.waimai.order.accept",
            {
                "delivery_order_no": delivery_order_no,
                "deliverer_id": deliverer_id,
                "deliverer_name": deliverer_name,
                "deliverer_phone": deliverer_phone,
                "card_no": card_no,
                "lng": lng,
                "lat": lat
            },
            result_processor=lambda x: x["accept_order_result"]
        )

    def taobao_waimai_order_ack(
            self,
            delivery_order_no
    ):
        """
        淘点点订单消息通知回执
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        配送商收到点点送订单通知消息后，调用该回执接口，通知点点送表示已收到消息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25321

        :param delivery_order_no: 物流单号
        """
        return self._top_request(
            "taobao.waimai.order.ack",
            {
                "delivery_order_no": delivery_order_no
            },
            result_processor=lambda x: x["ack_order_result"]
        )

    def taobao_waimai_delivery_update(
            self,
            delivery_order_no,
            deliverer_name,
            deliverer_phone,
            lng,
            lat,
            deliverer_id='',
            card_no=''
    ):
        """
        更新配送员信息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        配送商系统向点点送更新配送员信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25407

        :param delivery_order_no: 物流单号
        :param deliverer_name: 配送员姓名
        :param deliverer_phone: 配送员手机号
        :param lng: 配送员位置：经度
        :param lat: 配送员位置：维度
        :param deliverer_id: 配送员id
        :param card_no: 配送员身份证号
        """
        return self._top_request(
            "taobao.waimai.delivery.update",
            {
                "delivery_order_no": delivery_order_no,
                "deliverer_name": deliverer_name,
                "deliverer_phone": deliverer_phone,
                "lng": lng,
                "lat": lat,
                "deliverer_id": deliverer_id,
                "card_no": card_no
            },
            result_processor=lambda x: x["delivery_update_result"]
        )

    def taobao_waimai_order_autoaccept(
            self,
            order_id
    ):
        """
        点点卖家自动接单并选择配送商送
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        淘点点卖家自动接单并选择配送商送，卖家通过该接口只能操作自己店铺的订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25432

        :param order_id: 订单号
        """
        return self._top_request(
            "taobao.waimai.order.autoaccept",
            {
                "order_id": order_id
            },
            result_processor=lambda x: x["autoaccept_order_result"]
        )

    def taobao_waimai_delivery_failedorder(
            self,
            failed_reason='',
            delivery_order_no='0'
    ):
        """
        点点送运单失败
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        点点送外卖运单失败接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27532

        :param failed_reason: 运单失败原因
        :param delivery_order_no: 运单Id
        """
        return self._top_request(
            "taobao.waimai.delivery.failedorder",
            {
                "failed_reason": failed_reason,
                "delivery_order_no": delivery_order_no
            },
            result_processor=lambda x: x["delivery_order_result"]
        )


class TbDuoMeiTiPingTai(DingTalkBaseAPI):
    """
    多媒体平台API
    """

    def taobao_picture_category_get(
            self,
            type='',
            picture_category_id='',
            picture_category_name='',
            parent_id='-1',
            modified_time=''
    ):
        """
        获取图片分类信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=137

        :param type: 1
        :param picture_category_id: 图片分类ID
        :param picture_category_name: 图片分类名，不支持模糊查询
        :param parent_id: 取二级分类时设置为对应父分类id<br/>取一级分类时父分类id设为0<br/>取全部分类的时候不设或设为-1
        :param modified_time: 图片分类的修改时间点，格式:yyyy-MM-dd HH:mm:ss。查询此修改时间点之后到目前的图片分类。
        """
        return self._top_request(
            "taobao.picture.category.get",
            {
                "type": type,
                "picture_category_id": picture_category_id,
                "picture_category_name": picture_category_name,
                "parent_id": parent_id,
                "modified_time": modified_time
            },
            result_processor=lambda x: x["picture_categories"]
        )

    def taobao_picture_get(
            self,
            picture_id='',
            picture_category_id='',
            deleted='',
            title='',
            order_by='time:desc',
            start_date='',
            end_date='',
            page_no='1',
            page_size='40',
            modified_time='',
            client_type='client:all',
            urls='',
            is_https='false'
    ):
        """
        获取图片信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=138

        :param picture_id: 图片ID
        :param picture_category_id: 图片分类ID
        :param deleted: 是否删除,unfroze代表没有删除
        :param title: 图片标题,最大长度50字符,中英文都算一字符
        :param order_by: 图片查询结果排序,time:desc按上传时间从晚到早(默认), time:asc按上传时间从早到晚,sizes:desc按图片从大到小，sizes:asc按图片从小到大,默认time:desc
        :param start_date: 查询上传开始时间点,格式:yyyy-MM-dd HH:mm:ss
        :param end_date: 查询上传结束时间点,格式:yyyy-MM-dd HH:mm:ss
        :param page_no: 页码.传入值为1代表第一页,传入值为2代表第二页,依此类推,默认值为1
        :param page_size: 每页条数.每页返回最多返回100条,默认值40
        :param modified_time: 图片被修改的时间点，格式:yyyy-MM-dd HH:mm:ss。查询此修改时间点之后到目前的图片。
        :param client_type: 图片使用，如果是pc宝贝detail使用，设置为client:computer，查询出来的图片是符合pc的宝贝detail显示的如果是手机宝贝detail使用，设置为client:phone，查询出来的图片是符合手机的宝贝detail显示的,默认值是全部
        :param urls: 图片url查询接口
        :param is_https: 是否获取https的链接
        """
        return self._top_request(
            "taobao.picture.get",
            {
                "picture_id": picture_id,
                "picture_category_id": picture_category_id,
                "deleted": deleted,
                "title": title,
                "order_by": order_by,
                "start_date": start_date,
                "end_date": end_date,
                "page_no": page_no,
                "page_size": page_size,
                "modified_time": modified_time,
                "client_type": client_type,
                "urls": urls,
                "is_https": is_https
            },
            result_processor=lambda x: x["pictures"]
        )

    def taobao_picture_delete(
            self,
            picture_ids
    ):
        """
        删除图片空间图片
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=139

        :param picture_ids: 图片ID字符串,可以一个也可以一组,用英文逗号间隔,如450,120,155.限制数量是100
        """
        return self._top_request(
            "taobao.picture.delete",
            {
                "picture_ids": picture_ids
            },
            result_processor=lambda x: x["success"]
        )

    def taobao_picture_upload(
            self,
            picture_category_id,
            img,
            image_input_title,
            title='',
            client_type='client:computer',
            is_https='false'
    ):
        """
        上传单张图片
        图片空间上传接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=140

        :param picture_category_id: 图片分类ID，设置具体某个分类ID或设置0上传到默认分类，只能传入一个分类
        :param img: 图片二进制文件流,不能为空,允许png、jpg、gif图片格式,3M以内。
        :param image_input_title: 包括后缀名的图片标题,不能为空，如Bule.jpg,有些卖家希望图片上传后取图片文件的默认名
        :param title: 图片标题,如果为空,传的图片标题就取去掉后缀名的image_input_title,超过50字符长度会截取50字符,重名会在标题末尾加'(1)';标题末尾已经有'(数字)'了，则数字加1
        :param client_type: 图片上传的来源，有电脑版本宝贝发布，手机版本宝贝发布client:computer电脑版本宝贝使用，client:phone手机版本宝贝使用。注意：当client:phone时，图片限制为宽度在480-620之间，长度不能超过960，否则会报错。
        :param is_https: 是否获取https连接
        """
        return self._top_request(
            "taobao.picture.upload",
            {
                "picture_category_id": picture_category_id,
                "img": img,
                "image_input_title": image_input_title,
                "title": title,
                "client_type": client_type,
                "is_https": is_https
            },
            result_processor=lambda x: x["picture"]
        )

    def taobao_media_category_update(
            self,
            media_category_id,
            media_category_name
    ):
        """
        更新媒体文件分类
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=143

        :param media_category_id: 文件分类ID,不能为空
        :param media_category_name: 文件分类名，最大长度20字符，中英文都算一字符,不能为空
        """
        return self._top_request(
            "taobao.media.category.update",
            {
                "media_category_id": media_category_id,
                "media_category_name": media_category_name
            },
            result_processor=lambda x: x["success"]
        )

    def taobao_picture_category_add(
            self,
            picture_category_name,
            parent_id='0'
    ):
        """
        新增图片分类信息
        同一卖家最多添加500个图片分类，图片分类名称长度最大为20个字符
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=227

        :param picture_category_name: 图片分类名称，最大长度20字符，中文字符算2个字符，不能为空
        :param parent_id: 图片分类的父分类,一级分类的parent_id为0,二级分类的则为其父分类的picture_category_id
        """
        return self._top_request(
            "taobao.picture.category.add",
            {
                "picture_category_name": picture_category_name,
                "parent_id": parent_id
            },
            result_processor=lambda x: x["picture_category"]
        )

    def taobao_picture_category_update(
            self,
            category_id,
            category_name='',
            parent_id=''
    ):
        """
        更新图片分类
        更新图片分类的名字，或者更新图片分类的父分类（即分类移动）。只能移动2级分类到非2级分类，默认分类和1级分类不可移动。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10909

        :param category_id: 要更新的图片分类的id
        :param category_name: 图片分类的新名字，最大长度20字符，不能为空
        :param parent_id: 图片分类的新父分类id
        """
        return self._top_request(
            "taobao.picture.category.update",
            {
                "category_id": category_id,
                "category_name": category_name,
                "parent_id": parent_id
            },
            result_processor=lambda x: x["done"]
        )

    def taobao_picture_replace(
            self,
            picture_id,
            image_data
    ):
        """
        替换图片
        替换一张图片，只替换图片数据，图片名称，图片分类等保持不变。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10910

        :param picture_id: 要替换的图片的id，必须大于0
        :param image_data: 图片二进制文件流,不能为空,允许png、jpg、gif图片格式
        """
        return self._top_request(
            "taobao.picture.replace",
            {
                "picture_id": picture_id,
                "image_data": image_data
            },
            result_processor=lambda x: x["done"]
        )

    def taobao_picture_update(
            self,
            picture_id,
            new_name
    ):
        """
        修改图片名字
        修改指定图片的图片名
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10912

        :param picture_id: 要更改名字的图片的id
        :param new_name: 新的图片名，最大长度50字符，不能为空
        """
        return self._top_request(
            "taobao.picture.update",
            {
                "picture_id": picture_id,
                "new_name": new_name
            },
            result_processor=lambda x: x["done"]
        )

    def taobao_picture_userinfo_get(
            self
    ):
        """
        查询图片空间用户的信息
        查询用户的图片空间使用信息，包括：订购量，已使用容量，免费容量，总的可使用容量，订购有效期，剩余容量
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10914

        """
        return self._top_request(
            "taobao.picture.userinfo.get",
            result_processor=lambda x: x["user_info"]
        )

    def taobao_picture_isreferenced_get(
            self,
            picture_id
    ):
        """
        图片是否被引用
        查询图片是否被引用，被引用返回true，未被引用返回false
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10921

        :param picture_id: 图片id
        """
        return self._top_request(
            "taobao.picture.isreferenced.get",
            {
                "picture_id": picture_id
            },
            result_processor=lambda x: x["is_referenced"]
        )

    def taobao_media_file_add(
            self,
            dir_id,
            name,
            file_data,
            biz_code,
            ext=''
    ):
        """
        多媒体平台文件添加
        用户通过top上传文件到多媒体平台
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11028

        :param dir_id: 文件属于的那个目录的目录编号
        :param name: 上传文件的名称
        :param file_data: 文件
        :param biz_code: 接入多媒体平台的业务code
                         每个应用有一个特有的业务code
        :param ext: 额外信息
        """
        return self._top_request(
            "taobao.media.file.add",
            {
                "dir_id": dir_id,
                "name": name,
                "file_data": file_data,
                "biz_code": biz_code,
                "ext": ext
            },
            result_processor=lambda x: x["file"]
        )

    def taobao_picture_pictures_get(
            self,
            start_date='',
            picture_category_id='',
            order_by='time:desc',
            title='',
            page_size='40',
            end_date='',
            current_page='1',
            start_modified_date='',
            deleted='',
            picture_id='',
            client_type='client:all',
            urls='',
            is_https='false'
    ):
        """
        图片获取
        图片空间对外的图片获取接口，该接口只针对分页获取，获取某一页的图片，该接口不支持总数的查询asd
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25591

        :param start_date: 查询上传结束时间点,格式:yyyy-MM-dd HH:mm:ss
        :param picture_category_id: 图片分类
        :param order_by: 图片查询结果排序,time:desc按上传时间从晚到早(默认), time:asc按上传时间从早到晚,sizes:desc按图片从大到小，sizes:asc按图片从小到大,默认time:desc
        :param title: 图片标题,最大长度50字符,中英文都算一字符
        :param page_size: 每页条数.每页返回最多返回100条,默认值40
        :param end_date: 结束时间
        :param current_page: 页码.传入值为1代表第一页,传入值为2代表第二页,依此类推,默认值为1
        :param start_modified_date: 图片被修改的时间点，格式:yyyy-MM-dd HH:mm:ss。查询此修改时间点之后到目前的图片。
        :param deleted: 是否删除,deleted代表没有删除
        :param picture_id: 图片ID
        :param client_type: 图片使用，如果是pc宝贝detail使用，设置为client:computer，查询出来的图片是符合pc的宝贝detail显示的如果是手机宝贝detail使用，设置为client:phone，查询出来的图片是符合手机的宝贝detail显示的,默认值是全部
        :param urls: 图片url查询接口
        :param is_https: 是否获取https的链接
        """
        return self._top_request(
            "taobao.picture.pictures.get",
            {
                "start_date": start_date,
                "picture_category_id": picture_category_id,
                "order_by": order_by,
                "title": title,
                "page_size": page_size,
                "end_date": end_date,
                "current_page": current_page,
                "start_modified_date": start_modified_date,
                "deleted": deleted,
                "picture_id": picture_id,
                "client_type": client_type,
                "urls": urls,
                "is_https": is_https
            },
            result_processor=lambda x: x["pictures"]
        )

    def taobao_picture_pictures_count(
            self,
            start_date='',
            picture_category_id='',
            title='',
            end_date='',
            start_modified_date='',
            deleted='',
            picture_id='',
            client_type='client:all'
    ):
        """
        图片总数查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25592

        :param start_date: 查询上传开始时间点,格式:yyyy-MM-dd HH:mm:ss
        :param picture_category_id: 图片分类
        :param title: 文件名
        :param end_date: 查询上传结束时间点,格式:yyyy-MM-dd HH:mm:ss
        :param start_modified_date: 图片被修改的时间点，格式:yyyy-MM-dd HH:mm:ss。查询此修改时间点之后到目前的图片。
        :param deleted: 是否删除,undeleted代表没有删除,deleted表示删除
        :param picture_id: 图片ID
        :param client_type: 图片使用，如果是pc宝贝detail使用，设置为client:computer，查询出来的图片是符合pc的宝贝detail显示的如果是手机宝贝detail使用，设置为client:phone，查询出来的图片是符合手机的宝贝detail显示的,默认值是全部
        """
        return self._top_request(
            "taobao.picture.pictures.count",
            {
                "start_date": start_date,
                "picture_category_id": picture_category_id,
                "title": title,
                "end_date": end_date,
                "start_modified_date": start_modified_date,
                "deleted": deleted,
                "picture_id": picture_id,
                "client_type": client_type
            },
            result_processor=lambda x: x["totals"]
        )

    def taobao_interactive_list_getbyuser(
            self,
            current_page='1',
            page_size='10'
    ):
        """
        用户获取视频互动列表
        根据用户来获取用户编辑的互动列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27943

        :param current_page: 当前页
        :param page_size: 每页多少
        """
        return self._top_request(
            "taobao.interactive.list.getbyuser",
            {
                "current_page": current_page,
                "page_size": page_size
            }
        )

    def taobao_media_video_list(
            self,
            search_condition
    ):
        """
        获取商家视频列表
        用于获取授权商家的视频列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27990

        :param search_condition: 搜索条件
        """
        return self._top_request(
            "taobao.media.video.list",
            {
                "search_condition": search_condition
            }
        )


class TbZiZhangHaoGuanLi(DingTalkBaseAPI):
    """
    子账号管理API
    """

    def taobao_sellercenter_subusers_get(
            self,
            nick
    ):
        """
        查询指定账户的子账号列表
        根据主账号nick查询该账号下所有的子账号列表，只能查询属于自己的账号信息 (主账号以及所属子账号)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10762

        :param nick: 表示卖家昵称
        """
        return self._top_request(
            "taobao.sellercenter.subusers.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["subusers"]
        )

    def taobao_sellercenter_user_permissions_get(
            self,
            nick
    ):
        """
        获取指定用户的权限集合
        获取指定用户的权限集合，并不组装成树。如果是主账号，返回所有的权限列表；如果是子账号，返回所有已授权的权限。只能查询属于自己的账号信息 (如果是主账号，则是主账号以及所属子账号，如果是子账号则是对应主账号以及所属子账号)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10763

        :param nick: 用户标识，次入参必须为子账号比如zhangsan:cool。如果只输入主账号zhangsan，将报错。
        """
        return self._top_request(
            "taobao.sellercenter.user.permissions.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["permissions"]
        )

    def taobao_sellercenter_subuser_permissions_roles_get(
            self,
            nick
    ):
        """
        查询指定的子账号的权限和角色信息
        查询指定的子账号的被直接赋予的权限信息和角色信息。<br/>返回对象中包括直接赋予子账号的权限点信息、被赋予的角色以及角色的对应权限点信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10814

        :param nick: 子账号昵称(子账号标识)
        """
        return self._top_request(
            "taobao.sellercenter.subuser.permissions.roles.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["subuser_permission"]
        )

    def taobao_sellercenter_rolemembers_get(
            self,
            role_id
    ):
        """
        获取指定卖家和指定角色的下属员工列表
        获取指定卖家的角色下属员工列表，只能获取属于登陆者自己的信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10815

        :param role_id: 角色id
        """
        return self._top_request(
            "taobao.sellercenter.rolemembers.get",
            {
                "role_id": role_id
            },
            result_processor=lambda x: x["subusers"]
        )

    def taobao_sellercenter_roles_get(
            self,
            nick
    ):
        """
        获取指定卖家的角色列表
        获取指定卖家的角色列表，只能获取属于登陆者自己的信息。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10816

        :param nick: 卖家昵称(只允许查询自己的信息：当前登陆者)
        """
        return self._top_request(
            "taobao.sellercenter.roles.get",
            {
                "nick": nick
            },
            result_processor=lambda x: x["roles"]
        )

    def taobao_sellercenter_role_info_get(
            self,
            role_id
    ):
        """
        获取指定角色信息
        获取指定角色的信息。只能查询属于自己的角色信息 (主账号或者某个主账号的子账号登陆，只能查询属于该主账号的角色信息)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10817

        :param role_id: 角色id
        """
        return self._top_request(
            "taobao.sellercenter.role.info.get",
            {
                "role_id": role_id
            },
            result_processor=lambda x: x["role"]
        )

    def taobao_sellercenter_role_add(
            self,
            name,
            nick,
            description='',
            permission_codes=''
    ):
        """
        子账号角色的新增（指定卖家）
        给指定的卖家创建新的子账号角色<br/><br/>如果需要授权的权限点有下级权限点或上级权限点，把该权限点的父权限点和该权限点的所有子权限都一并做赋权操作，并递归处理<br/>例如：权限点列表如下<br/><br/>code=sell 宝贝管理<br/><br/>---------|code=sm 店铺管理<br/><br/>---------|---------|code=sm-design 如店铺装修<br/><br/>---------|---------|---------|code=sm-tbd-visit内店装修入口<br/><br/>---------|---------|---------|code=sm-tbd-publish内店装修发布<br/><br/>---------|---------|code=phone 手机淘宝店铺<br/><br/>调用改接口给code=sm-design店铺装修赋权时，同时会将下列权限点都赋予默认角色<br/><br/>code=sell 宝贝管理<br/><br/>---------|code=sm 店铺管理<br/><br/>---------|---------|code=sm-design 如店铺装修<br/><br/>---------|---------|---------|code=sm-tbd-visit内店装修入口<br/><br/>---------|---------|---------|code=sm-tbd-publish内店装修发布<br/>
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10820

        :param name: 角色名
        :param nick: 表示卖家昵称
        :param description: 角色描述
        :param permission_codes: 需要授权的权限点permission_code列表,以','分割.其code值可以通过调用taobao.sellercenter.user.permissions.get返回，其中permission.is_authorize=1的权限点可以通过本接口授权给对应角色。
        """
        return self._top_request(
            "taobao.sellercenter.role.add",
            {
                "name": name,
                "nick": nick,
                "description": description,
                "permission_codes": permission_codes
            },
            result_processor=lambda x: x["role"]
        )

    def taobao_subusers_get(
            self,
            user_nick
    ):
        """
        获取指定账户的子账号简易信息列表
        获取主账号下的子账号简易账号信息集合。（只能通过主账号登陆并且查询该属于主账号的子账号信息）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21686

        :param user_nick: 主账号用户名
        """
        return self._top_request(
            "taobao.subusers.get",
            {
                "user_nick": user_nick
            },
            result_processor=lambda x: x["subaccounts"]
        )

    def taobao_subuser_fullinfo_get(
            self,
            sub_id='',
            fields='',
            sub_nick=''
    ):
        """
        获取指定账户子账号的详细信息
        获取指定账户子账号的详细信息，其中包括子账号的账号信息以及员工、部门、职务信息（只能通过主账号登陆并查询属于该主账号下的某个子账号详细信息）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21687

        :param sub_id: 子账号ID（传参中sub_id和sub_nick至少需要其中一个，若sub_id与sub_nick同时传入并且合法，那么sub_nick优先，以sub_nick查询子账号）
        :param fields: 传入所需要的参数信息（若不需要获取子账号或主账号的企业邮箱地址，则无需传入该参数；若需要获取子账号或主账号的企业邮箱地址，则需要传入fields；可选参数值为subuser_email和user_email，传入其他参数值均无效；两个参数都需要则以逗号隔开传入即可，例如：subuser_email,user_email）
        :param sub_nick: 子账号用户名（传参中sub_id和sub_nick至少需要其中一个，若sub_id与sub_nick同时传入并且合法，那么sub_nick优先，以sub_nick查询子账号）
        """
        return self._top_request(
            "taobao.subuser.fullinfo.get",
            {
                "sub_id": sub_id,
                "fields": fields,
                "sub_nick": sub_nick
            },
            result_processor=lambda x: x["sub_fullinfo"]
        )

    def taobao_subuser_departments_get(
            self,
            user_nick
    ):
        """
        获取指定账户的所有部门列表
        获取指定账户的所有部门列表，其实包括有每个部门的ID、父部门ID、部门名称（通过主账号登陆只能查询属于该主账号下的所有部门信息）。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21689

        :param user_nick: 主账号用户名
        """
        return self._top_request(
            "taobao.subuser.departments.get",
            {
                "user_nick": user_nick
            },
            result_processor=lambda x: x["departments"]
        )

    def taobao_subuser_dutys_get(
            self,
            user_nick
    ):
        """
        获取指定账户的所有职务信息列表
        通过主账号Nick获取该账户下的所有职务信息，职务信息中包括职务ID、职务名称以及职务等级（通过主账号登陆只能获取属于该主账号下的职务信息）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21696

        :param user_nick: 主账号用户名
        """
        return self._top_request(
            "taobao.subuser.dutys.get",
            {
                "user_nick": user_nick
            },
            result_processor=lambda x: x["dutys"]
        )

    def taobao_subuser_employee_add(
            self,
            sub_id,
            department_id,
            employee_name,
            sex,
            duty_id='',
            entry_date='',
            employee_num='',
            employee_nickname='',
            work_location='',
            office_phone='',
            personal_mobile='',
            personal_email='',
            id_card_num='',
            leader_id=''
    ):
        """
        新增子账号的员工信息
        给指定子账号新增一个员工信息（通过主账号登陆只能新建属于该主账号的员工信息）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21697

        :param sub_id: 子账号ID
        :param department_id: 当前员工所属部门ID
        :param employee_name: 员工姓名
        :param sex: 员工性别 1：男; 2:女
        :param duty_id: 当前员工担任职务ID
        :param entry_date: 员工入职时间
        :param employee_num: 员工工号(可以用大小写英文字母和数字)
        :param employee_nickname: 员工花名
        :param work_location: 工作地点
        :param office_phone: 办公电话
        :param personal_mobile: 员工手机号码
        :param personal_email: 员工私人邮箱
        :param id_card_num: 员工身份证号码
        :param leader_id: 直接上级的员工ID
        """
        return self._top_request(
            "taobao.subuser.employee.add",
            {
                "sub_id": sub_id,
                "department_id": department_id,
                "employee_name": employee_name,
                "sex": sex,
                "duty_id": duty_id,
                "entry_date": entry_date,
                "employee_num": employee_num,
                "employee_nickname": employee_nickname,
                "work_location": work_location,
                "office_phone": office_phone,
                "personal_mobile": personal_mobile,
                "personal_email": personal_email,
                "id_card_num": id_card_num,
                "leader_id": leader_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_subuser_info_update(
            self,
            sub_id,
            is_disable_subaccount='',
            is_dispatch=''
    ):
        """
        修改指定账户子账号的基本信息
        修改指定账户子账号的基本信息（通过主账号登陆只能修改属于该主账号的子账号基本信息）
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21698

        :param sub_id: 子账号ID
        :param is_disable_subaccount: 是否停用子账号 true:表示停用该子账号false:表示开启该子账号
        :param is_dispatch: 子账号是否参与分流 true:参与分流 false:不参与分流
        """
        return self._top_request(
            "taobao.subuser.info.update",
            {
                "sub_id": sub_id,
                "is_disable_subaccount": is_disable_subaccount,
                "is_dispatch": is_dispatch
            },
            result_processor=lambda x: x["is_success"]
        )


class TbFuWuPingTai(DingTalkBaseAPI):
    """
    服务平台API
    """

    def taobao_vas_subscribe_get(
            self,
            article_code,
            nick
    ):
        """
        订购关系查询
        用于ISV根据登录进来的淘宝会员名查询该为该会员开通哪些收费项目，ISV只能查询自己名下的应用及收费项目的订购情况
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10586

        :param article_code: 商品编码，从合作伙伴后台（my.open.taobao.com）-收费管理-收费项目列表 能够获得该应用的商品代码
        :param nick: 淘宝会员名
        """
        return self._top_request(
            "taobao.vas.subscribe.get",
            {
                "article_code": article_code,
                "nick": nick
            },
            result_processor=lambda x: x["article_user_subscribes"]
        )

    def taobao_vas_order_search(
            self,
            article_code,
            item_code='',
            nick='',
            start_created='',
            end_created='',
            biz_type='0',
            biz_order_id='0',
            order_id='0',
            page_size='',
            page_no=''
    ):
        """
        订单记录导出
        用于ISV查询自己名下的应用及收费项目的订单记录（已付款订单）。<br/>建议用于查询前一日的历史记录，不适合用作实时数据查询。<br/>现在只能查询90天以内的数据<br/>该接口限制每分钟所有appkey调用总和只能有800次。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10587

        :param article_code: 应用收费代码，从合作伙伴后台（my.open.taobao.com）-收费管理-收费项目列表 能够获得该应用的收费代码
        :param item_code: 收费项目代码，从合作伙伴后台（my.open.taobao.com）-收费管理-收费项目列表 能够获得收费项目代码
        :param nick: 淘宝会员名
        :param start_created: 订单创建时间（订购时间）起始值（当start_created和end_created都不填写时，默认返回最近90天的数据）
        :param end_created: 订单创建时间（订购时间）结束值
        :param biz_type: 订单类型，1=新订 2=续订 3=升级 4=后台赠送 5=后台自动续订 6=订单审核后生成订购关系（暂时用不到） 空=全部
        :param biz_order_id: 订单号
        :param order_id: 子订单号
        :param page_size: 一页包含的记录数
        :param page_no: 页码
        """
        return self._top_request(
            "taobao.vas.order.search",
            {
                "article_code": article_code,
                "item_code": item_code,
                "nick": nick,
                "start_created": start_created,
                "end_created": end_created,
                "biz_type": biz_type,
                "biz_order_id": biz_order_id,
                "order_id": order_id,
                "page_size": page_size,
                "page_no": page_no
            }
        )

    def taobao_vas_subsc_search(
            self,
            article_code,
            page_size='',
            page_no='',
            item_code='',
            start_deadline='',
            end_deadline='',
            status='',
            autosub='',
            expire_notice='',
            nick=''
    ):
        """
        订购记录导出
        用于ISV查询自己名下的应用及收费项目的订购记录
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10588

        :param article_code: 应用收费代码，从合作伙伴后台（my.open.taobao.com）-收费管理-收费项目列表 能够获得该应用的收费代码
        :param page_size: 一页包含的记录数
        :param page_no: 页码
        :param item_code: 收费项目代码，从合作伙伴后台（my.open.taobao.com）-收费管理-收费项目列表 能够获得收费项目代码
        :param start_deadline: 到期时间起始值（当start_deadline和end_deadline都不填写时，默认返回最近90天的数据）
        :param end_deadline: 到期时间结束值
        :param status: 订购记录状态，1=有效 2=过期 空=全部
        :param autosub: 是否自动续费，true=自动续费 false=非自动续费 空=全部
        :param expire_notice: 是否到期提醒，true=到期提醒 false=非到期提醒 空=全部
        :param nick: 淘宝会员名
        """
        return self._top_request(
            "taobao.vas.subsc.search",
            {
                "article_code": article_code,
                "page_size": page_size,
                "page_no": page_no,
                "item_code": item_code,
                "start_deadline": start_deadline,
                "end_deadline": end_deadline,
                "status": status,
                "autosub": autosub,
                "expire_notice": expire_notice,
                "nick": nick
            }
        )

    def taobao_fuwu_sale_link_gen(
            self,
            param_str,
            nick=''
    ):
        """
        服务平台营销链接生成接口
        服务商通过使用该接口来产生营销链接，通过把这种链接发送给商家来做自定义人群的服务营销<br><br/>注：session是param_str这个参数串创建者生成的session，这个创建者与入参中的nick是不一致的。例如：A开发者创建了一个param_str的字符串，要为B商家生成一个营销链接，session必须是A开发者创建的session。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21642

        :param param_str: 从服务商后台，营销链接功能中生成的参数串直接复制使用。不要修改，否则抛错。
        :param nick: 用户需要营销的目标人群中的用户nick
        """
        return self._top_request(
            "taobao.fuwu.sale.link.gen",
            {
                "param_str": param_str,
                "nick": nick
            },
            result_processor=lambda x: x["url"]
        )

    def taobao_fuwu_scores_get(
            self,
            current_page,
            date,
            page_size='40'
    ):
        """
        服务平台评价查询接口
        根据日期、查询appkey对应服务评价，每次调用只能查询某一天服务评价信息，可设置分页查询，页大小最大为100，非实时接口，延迟时间为30分钟
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22111

        :param current_page: 当前页
        :param date: 评价日期，查询某一天的评价
        :param page_size: 每页获取条数。默认值40，最小值1，最大值100。
        """
        return self._top_request(
            "taobao.fuwu.scores.get",
            {
                "current_page": current_page,
                "date": date,
                "page_size": page_size
            },
            result_processor=lambda x: x["score_result"]
        )

    def taobao_weike_subscinfo_get(
            self,
            seller_name='',
            start_time='',
            end_time='',
            page_num=''
    ):
        """
        需求订单查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23058

        :param seller_name: 商家旺旺名称
        :param start_time: 时间范围开始时间
        :param end_time: 时间范围结束时间
        :param page_num: 页码
        """
        return self._top_request(
            "taobao.weike.subscinfo.get",
            {
                "seller_name": seller_name,
                "start_time": start_time,
                "end_time": end_time,
                "page_num": page_num
            }
        )

    def taobao_weike_performance_put(
            self,
            id='',
            per_info_wrapper=None
    ):
        """
        提交客服绩效接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23059

        :param id: 订单id
        :param per_info_wrapper: 绩效数据封装类
        """
        return self._top_request(
            "taobao.weike.performance.put",
            {
                "id": id,
                "per_info_wrapper": per_info_wrapper
            }
        )

    def taobao_fuwu_sku_get(
            self,
            article_code,
            nick
    ):
        """
        获取内购服务及SKU详情
        通过服务code和用户nick，获取该服务对应的收费项目的sku信息，包括价格、可购买周期、用户能否购买等信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25685

        :param article_code: 服务code
        :param nick: 用户的淘宝nick
        """
        return self._top_request(
            "taobao.fuwu.sku.get",
            {
                "article_code": article_code,
                "nick": nick
            }
        )

    def taobao_fuwu_sp_confirm_apply(
            self,
            param_income_confirm_d_t_o
    ):
        """
        内购服务确认单申请接口
        isv能通过该接口发起确认申请单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25686

        :param param_income_confirm_d_t_o: 确认单申请
        """
        return self._top_request(
            "taobao.fuwu.sp.confirm.apply",
            {
                "param_income_confirm_d_t_o": param_income_confirm_d_t_o
            },
            result_processor=lambda x: x["apply_result"]
        )

    def taobao_fuwu_purchase_order_confirm(
            self,
            param_order_confirm_query_d_t_o
    ):
        """
        服务市场内购服务下单接口
        通过传入服务市场商品的itemcode等信息，返回给服务商内购服务的下单链接
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25687

        :param param_order_confirm_query_d_t_o: 内购服务下单接口参数
        """
        return self._top_request(
            "taobao.fuwu.purchase.order.confirm",
            {
                "param_order_confirm_query_d_t_o": param_order_confirm_query_d_t_o
            },
            result_processor=lambda x: x["url"]
        )

    def taobao_fuwu_sp_billreord_add(
            self,
            param_bill_record_d_t_o
    ):
        """
        内购服务确认单明细上传接口
        isv能通过该接口上传确认单明细数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25688

        :param param_bill_record_d_t_o: 确认单的账单明细
        """
        return self._top_request(
            "taobao.fuwu.sp.billreord.add",
            {
                "param_bill_record_d_t_o": param_bill_record_d_t_o
            },
            result_processor=lambda x: x["add_result"]
        )

    def taobao_fuwu_purchase_order_pay(
            self,
            appkey,
            order_id='',
            out_order_id='',
            device_type=''
    ):
        """
        内购服务订单付款页获取接口
        通过接口获取某一订单的付款页面链接
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25689

        :param appkey: APPKEY，必填
        :param order_id: 订单号，与外部订单号二选一
        :param out_order_id: 外部订单号，使用该参数完成查询订单等操作，与外部订单号二选一
        :param device_type: 设备类型，目前只支持PC，可选
        """
        return self._top_request(
            "taobao.fuwu.purchase.order.pay",
            {
                "appkey": appkey,
                "order_id": order_id,
                "out_order_id": out_order_id,
                "device_type": device_type
            },
            result_processor=lambda x: x["url"]
        )

    def tmall_service_settleadjustment_search(
            self,
            end_time,
            start_time
    ):
        """
        服务商15分钟获取数据
        天猫服务平台，按修改时间，时间间隔在15中内（包含15分钟），获取调整单数据
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27437

        :param end_time: 结束时间
        :param start_time: 开始时间
        """
        return self._top_request(
            "tmall.service.settleadjustment.search",
            {
                "end_time": end_time,
                "start_time": start_time
            }
        )

    def taobao_weike_eservice_schedule_get(
            self,
            start_date,
            end_date,
            order_id='',
            seller_nick='',
            sp_nick=''
    ):
        """
        客服排班信息查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27489

        :param start_date: 起始日期，起始日期和结束日期跨度不能超过31天
        :param end_date: 结束日期，起始日期和结束日期跨度不能超过31天
        :param order_id: 订单ID，orderId、sellerNick、spNick三者不能同时为Null
        :param seller_nick: 商家子账号昵称，orderId、sellerNick、spNick三者不能同时为Null
        :param sp_nick: 服务商子账号昵称，orderId、sellerNick、spNick三者不能同时为Null
        """
        return self._top_request(
            "taobao.weike.eservice.schedule.get",
            {
                "start_date": start_date,
                "end_date": end_date,
                "order_id": order_id,
                "seller_nick": seller_nick,
                "sp_nick": sp_nick
            }
        )

    def taobao_weike_eservice_subusers_get(
            self,
            order_id
    ):
        """
        客服外包订单分配的商家子账号列表
        获取客服外包订单分配的商家子账号列表，以及授权状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27490

        :param order_id: 订单ID
        """
        return self._top_request(
            "taobao.weike.eservice.subusers.get",
            {
                "order_id": order_id
            }
        )

    def taobao_weike_eservice_schedule_put(
            self,
            order_id,
            cs_schedulings
    ):
        """
        提交客服排班信息
        添加、更新、删除排班信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27505

        :param order_id: 订单ID
        :param cs_schedulings: 按天排班信息
        """
        return self._top_request(
            "taobao.weike.eservice.schedule.put",
            {
                "order_id": order_id,
                "cs_schedulings": cs_schedulings
            }
        )

    def taobao_weike_eservice_order_get(
            self,
            start_date='',
            scheduling_state='',
            seller_nick='',
            page_size='',
            end_date='',
            order_id_list='',
            page_num=''
    ):
        """
        客服外包订单查询
        用于客服外包中服务商查询订单列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27610

        :param start_date: 订单服务开始日期
        :param scheduling_state: 订单是否可以排班
        :param seller_nick: 商家昵称
        :param page_size: 每页记录数（默认20，最大不超过20）
        :param end_date: 订单服务结束日期
        :param order_id_list: 订单ID列表，最大不超过20个（这个参数指定后，其它过滤条件失效）
        :param page_num: 页码（默认为1）
        """
        return self._top_request(
            "taobao.weike.eservice.order.get",
            {
                "start_date": start_date,
                "scheduling_state": scheduling_state,
                "seller_nick": seller_nick,
                "page_size": page_size,
                "end_date": end_date,
                "order_id_list": order_id_list,
                "page_num": page_num
            }
        )

    def tmall_car_lease_synchronizeplans(
            self,
            item_id,
            plans
    ):
        """
        同步租赁方案
        租赁公司同步还款计划
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28658

        :param item_id: 商品id
        :param plans: 租赁计划
        """
        return self._top_request(
            "tmall.car.lease.synchronizeplans",
            {
                "item_id": item_id,
                "plans": plans
            }
        )

    def tmall_car_lease_reserve(
            self,
            order_id,
            text,
            vin,
            flag,
            buyer_nick,
            buyer_id='0'
    ):
        """
        整车租车回传预约信息
        租赁公司回传预约到店信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28659

        :param order_id: 订单id
        :param text: 文案
        :param vin: 车架号
        :param flag: 1 代表 车辆到店，已预约用户到店提车   ; 2 车辆到店，未能联系到用户
        :param buyer_nick: 买家昵称
        :param buyer_id: 买家id
        """
        return self._top_request(
            "tmall.car.lease.reserve",
            {
                "order_id": order_id,
                "text": text,
                "vin": vin,
                "flag": flag,
                "buyer_nick": buyer_nick,
                "buyer_id": buyer_id
            }
        )

    def tmall_car_lease_riskcallback(
            self,
            credit_info=None
    ):
        """
        整车租赁风控模型回调
        租赁公司回调风控结果
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28660

        :param credit_info: 授信结果
        """
        return self._top_request(
            "tmall.car.lease.riskcallback",
            {
                "credit_info": credit_info
            }
        )

    def tmall_car_lease_consume(
            self,
            cosume_code_req_d_t_o=None
    ):
        """
        汽车租赁核销
        租赁公司回传信息，核销
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28661

        :param cosume_code_req_d_t_o: 核销请求
        """
        return self._top_request(
            "tmall.car.lease.consume",
            {
                "cosume_code_req_d_t_o": cosume_code_req_d_t_o
            }
        )

    def tmall_car_lease_tailpaymentback(
            self,
            tail_payment_d_t_o=None
    ):
        """
        尾款处置方案回传
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28683

        :param tail_payment_d_t_o: 尾款方案
        """
        return self._top_request(
            "tmall.car.lease.tailpaymentback",
            {
                "tail_payment_d_t_o": tail_payment_d_t_o
            }
        )

    def tmall_car_lease_itemcarinfo(
            self,
            item_id='0'
    ):
        """
        整车租赁商品四级车型信息
        整车租赁项目发布宝贝需要4级车型库，4级车型库信息需要回传
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29001

        :param item_id: 商品id
        """
        return self._top_request(
            "tmall.car.lease.itemcarinfo",
            {
                "item_id": item_id
            }
        )

    def tmall_car_lease_fourlevelcar(
            self,
            value_id='0',
            property_id='0'
    ):
        """
        获取四级车型库
        整车租赁和商家对接之后，商家需要获取车型信息，调用接口同步方案
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29733

        :param value_id: 属性值id
        :param property_id: 属性id
        """
        return self._top_request(
            "tmall.car.lease.fourlevelcar",
            {
                "value_id": value_id,
                "property_id": property_id
            }
        )

    def tmall_car_trydrive_updatereserveinfo(
            self,
            app_name='tmall',
            car_no='浙A12345',
            id_card='1321232131',
            mobile='12323344',
            name='张三',
            uuid='1111',
            status='1'
    ):
        """
        更新试驾预约单信息
        ！！！该接口已在官方文档下线，请谨慎使用！！！

        天猫汽车嘉年华预约试驾，需要和大搜车进行对接，大搜车和支付宝对接之后更新预约
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31461

        :param app_name: 外部系统参数
        :param car_no: 车牌号
        :param id_card: 身份证
        :param mobile: 手机号
        :param name: 姓名
        :param uuid: 唯一申请号，必填，风控的时候已经传入
        :param status: 6:授权通过,7:授权失败,1:风控拒绝,2:风控通过,12:还车成功，13:退款成功
        """
        return self._top_request(
            "tmall.car.trydrive.updatereserveinfo",
            {
                "app_name": app_name,
                "car_no": car_no,
                "id_card": id_card,
                "mobile": mobile,
                "name": name,
                "uuid": uuid,
                "status": status
            }
        )

    def tmall_car_leaseorder_get(
            self,
            order_id='0'
    ):
        """
        获取租赁订单信息
        卖家通过供销平台获取代销商的订单信息，但是部分情况下网商银行订单号获取不到，需要提供接口或者工具给卖家
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34313

        :param order_id: 订单号
        """
        return self._top_request(
            "tmall.car.leaseorder.get",
            {
                "order_id": order_id
            }
        )

    def tmall_car_contract_download(
            self,
            order_id='0',
            html='false'
    ):
        """
        合同下载
        目前天猫开新车会在线上签署一份合同，协议，需要一个个在已卖出打开，另存为pdf，人工一个个下载比较麻烦，期望通过接口直接读取pdf；
        因为比较耗时，建议一个个下载，假设并发下载，很可能限流，每天的调用量有限；
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=34314

        :param order_id: 天猫订单号
        :param html: 是否下载html，true是html，false是pdf， html速度会快一点
        """
        return self._top_request(
            "tmall.car.contract.download",
            {
                "order_id": order_id,
                "html": html
            }
        )

    def tmall_servicecenter_tp_funds_recover_query(
            self,
            query
    ):
        """
        服务商资金权益逆向扣回的查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37065

        :param query: query入参
        """
        return self._top_request(
            "tmall.servicecenter.tp.funds.recover.query",
            {
                "query": query
            }
        )

    def tmall_servicecenter_tp_funds_send_query(
            self,
            query
    ):
        """
        服务商资金权益发放的查询接口
        服务商资金权益发放结果的查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37069

        :param query: 入参对象
        """
        return self._top_request(
            "tmall.servicecenter.tp.funds.send.query",
            {
                "query": query
            }
        )

    def tmall_msf_verify(
            self,
            shop_id='',
            biz_type='',
            code=''
    ):
        """
        喵师傅核销接口
        msf服务核销的top接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37992

        :param shop_id: 111
        :param biz_type: 111
        :param code: 111
        """
        return self._top_request(
            "tmall.msf.verify",
            {
                "shop_id": shop_id,
                "biz_type": biz_type,
                "code": code
            }
        )

    def tmall_msf_receive(
            self,
            shop_id='',
            biz_type='',
            code=''
    ):
        """
        签收接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=37993

        :param shop_id: 1
        :param biz_type: 1
        :param code: 1
        """
        return self._top_request(
            "tmall.msf.receive",
            {
                "shop_id": shop_id,
                "biz_type": biz_type,
                "code": code
            }
        )

    def tmall_car_lease_freedownpayment_put(
            self,
            pre_end_time,
            pre_start_time,
            item_id,
            time_range_list,
            ref_activity_id=''
    ):
        """
        同步直租车免首付商品活动信息
        汽车行业直租车免首付需求中，用与对商品打标，活动范围设置，在消费者端商品详情页、订单等环节透出，表示该商品为直租免首付商品。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43845

        :param pre_end_time: 活动预热结束时间:格式：yyyy.MM.dd HH:mm:ss
        :param pre_start_time: 活动预热开始时间:格式：yyyy.MM.dd HH:mm:ss
        :param item_id: 商品ID
        :param time_range_list: 活动时间范围节点(json格式字符串)：<br/> 开始时间(startTime),格式：yyyy.MM.dd HH:mm:ss <br/>  结束时间(endTime),格式：yyyy.MM.dd HH:mm:ss <br/>  名额(amount)
        :param ref_activity_id: 外部活动ID
        """
        return self._top_request(
            "tmall.car.lease.freedownpayment.put",
            {
                "pre_end_time": pre_end_time,
                "pre_start_time": pre_start_time,
                "item_id": item_id,
                "time_range_list": time_range_list,
                "ref_activity_id": ref_activity_id
            }
        )

    def tmall_car_lease_item_activity_get(
            self
    ):
        """
        查询汽车租赁活动信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43876

        """
        return self._top_request(
            "tmall.car.lease.item.activity.get"
        )


class TbTuiKuan(DingTalkBaseAPI):
    """
    退款API
    """

    def taobao_refunds_apply_get(
            self,
            fields,
            status='',
            seller_nick='',
            type='',
            page_no='1',
            page_size='40'
    ):
        """
        查询买家申请的退款列表
        查询买家申请的退款列表，且查询外店的退款列表时需要指定交易类型
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=51

        :param fields: 需要返回的字段。目前支持有：refund_id, tid, title, buyer_nick, seller_nick, total_fee, status, created, refund_fee
        :param status: 退款状态，默认查询所有退款状态的数据，除了默认值外每次只能查询一种状态。<br/>WAIT_SELLER_AGREE(买家已经申请退款，等待卖家同意) <br/>WAIT_BUYER_RETURN_GOODS(卖家已经同意退款，等待买家退货) <br/>WAIT_SELLER_CONFIRM_GOODS(买家已经退货，等待卖家确认收货) <br/>SELLER_REFUSE_BUYER(卖家拒绝退款) <br/>CLOSED(退款关闭) <br/>SUCCESS(退款成功)
        :param seller_nick: 卖家昵称
        :param type: 交易类型列表，一次查询多种类型可用半角逗号分隔，默认同时查询guarantee_trade, auto_delivery的2种类型的数据。<br/>fixed(一口价) <br/>auction(拍卖) <br/>guarantee_trade(一口价、拍卖) <br/>independent_simple_trade(旺店入门版交易) <br/>independent_shop_trade(旺店标准版交易) <br/>auto_delivery(自动发货) <br/>ec(直冲) <br/>cod(货到付款) <br/>fenxiao(分销) <br/>game_equipment(游戏装备) <br/>shopex_trade(ShopEX交易) <br/>netcn_trade(万网交易) <br/>external_trade(统一外部交易)
        :param page_no: 页码。传入值为 1 代表第一页，传入值为 2 代表第二页，依此类推。默认返回的数据是从第一页开始
        :param page_size: 每页条数。取值范围:大于零的整数; 默认值:40;最大值:100
        """
        return self._top_request(
            "taobao.refunds.apply.get",
            {
                "fields": fields,
                "status": status,
                "seller_nick": seller_nick,
                "type": type,
                "page_no": page_no,
                "page_size": page_size
            }
        )

    def taobao_refund_get(
            self,
            fields,
            refund_id
    ):
        """
        获取单笔退款详情
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=53

        :param fields: 需要返回的字段。目前支持有：refund_id, alipay_no, tid, oid, buyer_nick, seller_nick, total_fee, status, created, refund_fee, good_status, has_good_return, payment, reason, desc, num_iid, title, price, num, good_return_time, company_name, sid, address, shipping_type, refund_remind_timeout, refund_phase, refund_version, operation_contraint, attribute, outer_id, sku
        :param refund_id: 退款单号
        """
        return self._top_request(
            "taobao.refund.get",
            {
                "fields": fields,
                "refund_id": refund_id
            },
            result_processor=lambda x: x["refund"]
        )

    def taobao_refund_messages_get(
            self,
            fields,
            refund_id,
            page_no='1',
            page_size='40',
            refund_phase=''
    ):
        """
        查询退款留言/凭证列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=124

        :param fields: 需返回的字段列表。可选值：RefundMessage结构体中的所有字段，以半角逗号(,)分隔。
        :param refund_id: 退款单号
        :param page_no: 页码
        :param page_size: 每页条数
        :param refund_phase: 退款阶段，可选值：onsale（售中），aftersale（售后），天猫退款为必传。
        """
        return self._top_request(
            "taobao.refund.messages.get",
            {
                "fields": fields,
                "refund_id": refund_id,
                "page_no": page_no,
                "page_size": page_size,
                "refund_phase": refund_phase
            }
        )

    def taobao_refund_message_add(
            self,
            refund_id,
            content,
            image
    ):
        """
        创建退款留言/凭证
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=125

        :param refund_id: 退款编号。
        :param content: 留言内容。最大长度: 400个字节
        :param image: 图片（凭证）。类型: JPG,GIF,PNG;最大为: 500K
        """
        return self._top_request(
            "taobao.refund.message.add",
            {
                "refund_id": refund_id,
                "content": content,
                "image": image
            },
            result_processor=lambda x: x["refund_message"]
        )

    def taobao_refund_refuse(
            self,
            refund_id,
            refuse_message,
            refuse_proof='',
            refund_phase='',
            refund_version='',
            refuse_reason_id=''
    ):
        """
        卖家拒绝退款
        卖家拒绝单笔退款（包含退款和退款退货）交易，要求如下：<br/>1. 传入的refund_id和相应的tid, oid必须匹配<br/>2. 如果一笔订单只有一笔子订单，则tid必须与oid相同<br/>3. 只有卖家才能执行拒绝退款操作<br/>4. 以下三种情况不能退款：卖家未发货；7天无理由退换货；网游订单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10480

        :param refund_id: 退款单号
        :param refuse_message: 拒绝退款时的说明信息，长度2-200
        :param refuse_proof: 拒绝退款时的退款凭证，一般是卖家拒绝退款时使用的发货凭证，最大长度130000字节，支持的图片格式：GIF, JPG, PNG。天猫退款为必填项。
        :param refund_phase: 可选值为：售中：onsale，售后：aftersale，天猫退款为必填项。
        :param refund_version: 退款版本号，天猫退款为必填项。
        :param refuse_reason_id: 拒绝原因编号，会提供用户拒绝原因列表供选择
        """
        return self._top_request(
            "taobao.refund.refuse",
            {
                "refund_id": refund_id,
                "refuse_message": refuse_message,
                "refuse_proof": refuse_proof,
                "refund_phase": refund_phase,
                "refund_version": refund_version,
                "refuse_reason_id": refuse_reason_id
            }
        )

    def taobao_rp_refunds_agree(
            self,
            refund_infos,
            code=''
    ):
        """
        同意退款
        卖家同意退款，支持批量退款，只允许子账号操作。淘宝退款一次最多能退20笔，总金额不超过6000元；天猫退款一次最多能退30笔，总金额不超过10000元。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22465

        :param refund_infos: 退款信息，格式：refund_id|amount|version|phase，其中refund_id为退款编号，amount为退款金额（以分为单位），version为退款最后更新时间（时间戳格式），phase为退款阶段（可选值为：onsale, aftersale，天猫退款必值，淘宝退款不需要传），多个退款以半角逗号分隔。
        :param code: 短信验证码，如果退款金额达到一定的数量，后端会返回调用失败，并同时往卖家的手机发送一条短信验证码。接下来用收到的短信验证码再次发起API调用即可完成退款操作。
        """
        return self._top_request(
            "taobao.rp.refunds.agree",
            {
                "refund_infos": refund_infos,
                "code": code
            }
        )

    def taobao_rp_returngoods_agree(
            self,
            refund_id,
            name='',
            address='',
            post='',
            tel='',
            mobile='',
            remark='',
            refund_phase='',
            refund_version='',
            seller_address_id='',
            post_fee_bear_role=''
    ):
        """
        卖家同意退货
        卖家同意退货，支持淘宝和天猫的订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22466

        :param refund_id: 退款编号
        :param name: 卖家姓名，淘宝退款为必填项。
        :param address: 卖家提供的退货地址，淘宝退款为必填项。
        :param post: 卖家提供的退货地址的邮编，淘宝退款为必填项。
        :param tel: 卖家座机，淘宝退款为必填项。
        :param mobile: 卖家手机，淘宝退款为必填项。
        :param remark: 卖家退货留言，天猫退款为必填项。
        :param refund_phase: 售中：onsale，售后：aftersale，天猫退款为必填项。
        :param refund_version: 退款版本号，天猫退款为必填项。
        :param seller_address_id: 卖家收货地址编号，天猫淘宝退款都为必填项。
        :param post_fee_bear_role: 邮费承担方，买家承担值为1，卖家承担值为0
        """
        return self._top_request(
            "taobao.rp.returngoods.agree",
            {
                "refund_id": refund_id,
                "name": name,
                "address": address,
                "post": post,
                "tel": tel,
                "mobile": mobile,
                "remark": remark,
                "refund_phase": refund_phase,
                "refund_version": refund_version,
                "seller_address_id": seller_address_id,
                "post_fee_bear_role": post_fee_bear_role
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_rp_refund_review(
            self,
            refund_id,
            operator,
            refund_phase,
            refund_version,
            result,
            message
    ):
        """
        审核退款单
        审核退款单，标志是否可用于批量退款，目前仅支持天猫订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23875

        :param refund_id: 退款单编号
        :param operator: 审核人姓名
        :param refund_phase: 退款阶段，可选值：售中：onsale，售后：aftersale
        :param refund_version: 退款最后更新时间，以时间戳的方式表示
        :param result: 审核是否可用于批量退款，可选值：true（审核通过），false（审核不通过或反审核）
        :param message: 审核留言
        """
        return self._top_request(
            "taobao.rp.refund.review",
            {
                "refund_id": refund_id,
                "operator": operator,
                "refund_phase": refund_phase,
                "refund_version": refund_version,
                "result": result,
                "message": message
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_rp_returngoods_refill(
            self,
            refund_id,
            refund_phase,
            logistics_waybill_no,
            logistics_company_code
    ):
        """
        卖家回填物流信息
        卖家收到货物回填物流信息，如果买家已经回填物流信息，则接口报错，目前仅支持天猫订单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23876

        :param refund_id: 退款单编号
        :param refund_phase: 退款阶段，可选值：售中：onsale，售后：aftersale
        :param logistics_waybill_no: 物流公司运单号
        :param logistics_company_code: 物流公司编号
        """
        return self._top_request(
            "taobao.rp.returngoods.refill",
            {
                "refund_id": refund_id,
                "refund_phase": refund_phase,
                "logistics_waybill_no": logistics_waybill_no,
                "logistics_company_code": logistics_company_code
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_rp_returngoods_refuse(
            self,
            refund_id,
            refund_phase,
            refund_version,
            refuse_proof,
            refuse_reason_id=''
    ):
        """
        卖家拒绝退货
        卖家拒绝退货，目前仅支持天猫退货。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=23877

        :param refund_id: 退款编号
        :param refund_phase: 退款服务状态，售后或者售中
        :param refund_version: 退款版本号
        :param refuse_proof: 拒绝退货凭证图片，必须图片格式，大小不能超过5M
        :param refuse_reason_id: 拒绝原因编号，会提供拒绝原因列表供选择
        """
        return self._top_request(
            "taobao.rp.returngoods.refuse",
            {
                "refund_id": refund_id,
                "refund_phase": refund_phase,
                "refund_version": refund_version,
                "refuse_proof": refuse_proof,
                "refuse_reason_id": refuse_reason_id
            }
        )

    def taobao_refund_refusereason_get(
            self,
            refund_id,
            fields,
            refund_phase
    ):
        """
        获取拒绝原因列表
        获取商家拒绝原因列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26204

        :param refund_id: 退款编号
        :param fields: 返回参数
        :param refund_phase: 售中或售后
        """
        return self._top_request(
            "taobao.refund.refusereason.get",
            {
                "refund_id": refund_id,
                "fields": fields,
                "refund_phase": refund_phase
            }
        )

    def taobao_rdc_aligenius_sendgoods_cancel(
            self,
            param
    ):
        """
        取消发货
        提供商家在仅退款中发送取消发货状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29304

        :param param: 请求参数
        """
        return self._top_request(
            "taobao.rdc.aligenius.sendgoods.cancel",
            {
                "param": param
            }
        )

    def taobao_rdc_aligenius_order_returngoods_notify(
            self,
            parent_order_id='',
            refund_return_notes=None
    ):
        """
        退货单创建结果反馈
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31063

        :param parent_order_id: 主订单号
        :param refund_return_notes: 退货单创建结果列表
        """
        return self._top_request(
            "taobao.rdc.aligenius.order.returngoods.notify",
            {
                "parent_order_id": parent_order_id,
                "refund_return_notes": refund_return_notes
            }
        )

    def tmall_dispute_receive_get(
            self,
            fields,
            status='',
            page_size='20',
            use_has_next='',
            type='fixed',
            refund_id='',
            page_no='1',
            buyer_nick='',
            start_modified='',
            end_modified=''
    ):
        """
        天猫逆向纠纷查询
        展示商家所有退款信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33878

        :param fields: 需要返回的字段。目前支持有：refund_id, alipay_no, tid, buyer_nick, seller_nick, status, created, modified, order_status, refund_fee, good_status, show_return_logistic(展现买家退货的物流信息), show_exchange_logistic(展现换货的物流信息), time_out, oid, refund_version, title, num, dispute_request, reason, desc
        :param status: 退款状态，默认查询所有退款状态的数据，除了默认值外每次只能查询一种状态。WAIT_SELLER_AGREE(买家已经申请退款，等待卖家同意);WAIT_BUYER_RETURN_GOODS(卖家已经同意退款，等待买家退货);WAIT_SELLER_CONFIRM_GOODS(买家已经退货，等待卖家确认收货);CLOSED(退款关闭); SUCCESS(退款成功);SELLER_REFUSE_BUYER(卖家拒绝退款);WAIT_BUYER_CONFIRM_REDO_SEND_GOODS(等待买家确认重新邮寄的货物);WAIT_SELLER_CONFIRM_RETURN_ADDRESS(等待卖家确认退货地址);WAIT_SELLER_CONSIGN_GOOGDS(卖家确认收货,等待卖家发货);EXCHANGE_TRANSFORM_TO_REFUND(换货关闭,转退货退款);EXCHANGE_WAIT_BUYER_CONFIRM_GOODS(卖家已发货,等待买家确认收货);POST_FEE_DISPUTE_WAIT_ACTIVATE(邮费单已创建,待激活)
        :param page_size: 每页条数。取值范围:大于零的整数; 默认值:20;最大值:100
        :param use_has_next: 是否启用has_next的分页方式，如果指定true,则返回的结果中不包含总记录数，但是会新增一个是否存在下一页的的字段，通过此种方式获取增量退款，接口调用成功率在原有的基础上有所提升。
        :param type: 交易类型列表，一次查询多种类型可用半角逗号分隔，默认同时查询guarantee_trade, auto_delivery这两种类型的数据，查看可选值
        :param refund_id: 逆向纠纷单号id
        :param page_no: 页码。取值范围:大于零的整数; 默认值:1
        :param buyer_nick: 买家昵称
        :param start_modified: 查询修改时间开始。格式: yyyy-MM-dd HH:mm:ss
        :param end_modified: 查询修改时间结束。格式: yyyy-MM-dd HH:mm:ss
        """
        return self._top_request(
            "tmall.dispute.receive.get",
            {
                "fields": fields,
                "status": status,
                "page_size": page_size,
                "use_has_next": use_has_next,
                "type": type,
                "refund_id": refund_id,
                "page_no": page_no,
                "buyer_nick": buyer_nick,
                "start_modified": start_modified,
                "end_modified": end_modified
            }
        )


class TbZhiJianPinKong(DingTalkBaseAPI):
    """
    质检品控API
    """

    def taobao_qt_report_get(
            self,
            qt_code
    ):
        """
        查询质检报告
        质检报告查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10898

        :param qt_code: 质检编号
        """
        return self._top_request(
            "taobao.qt.report.get",
            {
                "qt_code": qt_code
            },
            result_processor=lambda x: x["qt_report"]
        )

    def taobao_ts_property_get(
            self,
            service_item_code
    ):
        """
        淘宝服务属性查询
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10899

        :param service_item_code: 服务收费项code
        """
        return self._top_request(
            "taobao.ts.property.get",
            {
                "service_item_code": service_item_code
            },
            result_processor=lambda x: x["service_item_property"]
        )

    def taobao_ts_subscribe_get(
            self,
            servcie_item_code,
            nick
    ):
        """
        淘宝服务订购关系查询
        ts订购关系状态查询. 暂只支持1口价服务.
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10900

        :param servcie_item_code: 服务收费项code
        :param nick: 订购用户昵称
        """
        return self._top_request(
            "taobao.ts.subscribe.get",
            {
                "servcie_item_code": servcie_item_code,
                "nick": nick
            },
            result_processor=lambda x: x["service_subscribe"]
        )

    def taobao_qt_report_update(
            self,
            servcie_item_code,
            sp_name,
            nick,
            qt_code,
            qt_name,
            item_url,
            item_desc,
            qt_type,
            status,
            qt_standard='',
            report_url='',
            is_passed='',
            message='',
            ext_attr='',
            gmt_submit='',
            gmt_report='',
            gmt_expiry='',
            num_iid=''
    ):
        """
        更新质检报告
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10901

        :param servcie_item_code: 收费项code
        :param sp_name: 质检服务商名称
        :param nick: 送检者昵称
        :param qt_code: 一个质检服务唯一标识质量检验单的编号
        :param qt_name: 质检名称
        :param item_url: 宝贝样品url
        :param item_desc: 样品宝贝描述
        :param qt_type: (1L, '聚划算'),<br/>(2L, '消保'),<br/>(3L, '分销'),<br/>(4L, '抽检'),<br/>(5L, '良无限线下数据'),<br/>(6L, '入驻/续签商城'),<br/>(7L, '买家质检维权'),<br/>(8L, '实地验证'),<br/>(9L, '淘宝买家订单商品鉴定'),<br/>(10L,'假一赔三');
        :param status: 0:已提交申请<br/>1:已收到样品<br/>2:已出检测结果<br/>3.已出具报告
        :param qt_standard: 质检标准<br/>status=3 时必须非空
        :param report_url: 质检报告源文件url<br/>status状态为3时必须非空
        :param is_passed: 只有status=3时赋值, <br/>0:未通过1:通过 空代表未判定
        :param message: 检测结果消息描述
        :param ext_attr: 自定义属性字段;分号分隔
        :param gmt_submit: 送检日期
        :param gmt_report: 提交报告结果时间
        :param gmt_expiry: 质检有效到期时间，一般为一年有效期<br/>status状态为3时必须非空
        :param num_iid: 外部ID，和QT_TYPE 一起表示某种平台的实体ID。
                        QT_TYPE=9的时候，num_iid为淘宝订单号
        """
        return self._top_request(
            "taobao.qt.report.update",
            {
                "servcie_item_code": servcie_item_code,
                "sp_name": sp_name,
                "nick": nick,
                "qt_code": qt_code,
                "qt_name": qt_name,
                "item_url": item_url,
                "item_desc": item_desc,
                "qt_type": qt_type,
                "status": status,
                "qt_standard": qt_standard,
                "report_url": report_url,
                "is_passed": is_passed,
                "message": message,
                "ext_attr": ext_attr,
                "gmt_submit": gmt_submit,
                "gmt_report": gmt_report,
                "gmt_expiry": gmt_expiry,
                "num_iid": num_iid
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_qt_report_add(
            self,
            servcie_item_code,
            sp_name,
            nick,
            qt_code,
            qt_name,
            item_url,
            item_desc,
            qt_type,
            status,
            gmt_submit,
            qt_standard='',
            report_url='',
            is_passed='',
            message='',
            ext_attr='',
            gmt_report='',
            gmt_expiry='',
            num_iid=''
    ):
        """
        上传质检报告
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=10902

        :param servcie_item_code: 收费项code
        :param sp_name: 质检服务商名称
        :param nick: 送检者昵称
        :param qt_code: 一个质检服务唯一标识质量检验单的编号
        :param qt_name: 质检名称
        :param item_url: 样品链接.<br/>QT_TYPE=9的时候，请填写N\A
        :param item_desc: 样品信息描述
        :param qt_type: (1L, '聚划算'),<br/>(2L, '消保'),<br/>(3L, '分销'),<br/>(4L, '抽检'),<br/>(5L, '良无限线下数据'),<br/>(6L, '入驻/续签商城'),<br/>(7L, '买家质检维权'),<br/>(8L, '实地验证'),<br/>(9L, '淘宝买家订单商品鉴定'),<br/>(10L,'假一赔三');
        :param status: 0:已提交申请<br/>1:已收到样品<br/>2:已出检测结果<br/>3.已出具报告
        :param gmt_submit: 送检日期
        :param qt_standard: 质检标准<br/>status=3 时 必须非空
        :param report_url: 质检报告源文件url<br/>status状态为3时必须非空
        :param is_passed: 只有status=3时赋值, <br/>true 质检结果合格,false质检结果不合格.<br/>留空表示成分鉴定,不做判定
        :param message: 检测结果消息描述
        :param ext_attr: 自定义属性字段;分号分隔
        :param gmt_report: 提交报告结果时间
        :param gmt_expiry: 质检有效到期时间，一般为一年有效期<br/>status状态为3时必须非空
        :param num_iid: 当前接口只有淘宝订单真假鉴定（QT_TYPE=9）的报告在该字段传入订单号，其他类型报告都不需要传输该值
        """
        return self._top_request(
            "taobao.qt.report.add",
            {
                "servcie_item_code": servcie_item_code,
                "sp_name": sp_name,
                "nick": nick,
                "qt_code": qt_code,
                "qt_name": qt_name,
                "item_url": item_url,
                "item_desc": item_desc,
                "qt_type": qt_type,
                "status": status,
                "gmt_submit": gmt_submit,
                "qt_standard": qt_standard,
                "report_url": report_url,
                "is_passed": is_passed,
                "message": message,
                "ext_attr": ext_attr,
                "gmt_report": gmt_report,
                "gmt_expiry": gmt_expiry,
                "num_iid": num_iid
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_qt_report_delete(
            self,
            qt_code
    ):
        """
        质检报告删除接口
        删除质检报告
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21226

        :param qt_code: 一个质检服务唯一标识质量检验单的编号
        """
        return self._top_request(
            "taobao.qt.report.delete",
            {
                "qt_code": qt_code
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_qt_reports_get(
            self,
            sp_name,
            qt_type,
            servcie_item_code,
            nick,
            start_time='',
            end_time=''
    ):
        """
        批量查询质检报告
        批量查询质检报告，目前只支持查询qtType=11（天猫真假鉴定）类型的报告
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22479

        :param sp_name: 质检服务商名
        :param qt_type: 质检类型，目前只支持查询qt_type=11的类型
        :param servcie_item_code: 收费项code
        :param nick: 送检者昵称
        :param start_time: 查询时间段的开始时间
        :param end_time: 查询时间段的结束时间
        """
        return self._top_request(
            "taobao.qt.reports.get",
            {
                "sp_name": sp_name,
                "qt_type": qt_type,
                "servcie_item_code": servcie_item_code,
                "nick": nick,
                "start_time": start_time,
                "end_time": end_time
            },
            result_processor=lambda x: x["reports"]
        )


class TbGuanLianTuiJian(DingTalkBaseAPI):
    """
    关联推荐API
    """

    def alibaba_orp_recommend(
            self,
            appid,
            call_source,
            userid='0',
            params=''
    ):
        """
        推荐场景开放接口
        推荐开放接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26006

        :param appid: 推荐场景ID
        :param call_source: 调用来源,格式TOP_
        :param userid: 买家数字ID（如果需要）
        :param params: 推荐使用的参数
        """
        return self._top_request(
            "alibaba.orp.recommend",
            {
                "appid": appid,
                "call_source": call_source,
                "userid": userid,
                "params": params
            },
            result_processor=lambda x: x["recommend"]
        )


class TbTianMaoFuWuShangPin(DingTalkBaseAPI):
    """
    天猫服务商品API
    """

    def tmall_servicecenter_contracts_search(
            self,
            start,
            end
    ):
        """
        获取合同类的服务工单信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11121

        :param start: 开始时间:  开始时间和结束时间不能超过15分钟
        :param end: 结束时间:  开始时间和结束时间不能超过15分钟
        """
        return self._top_request(
            "tmall.servicecenter.contracts.search",
            {
                "start": start,
                "end": end
            },
            result_processor=lambda x: x["service_contract_packet"]
        )

    def tmall_servicecenter_tasks_search(
            self,
            start,
            end
    ):
        """
        查询任务类工单信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11122

        :param start: 开始时间:  开始时间和结束时间不能超过15分钟
        :param end: 结束时间:  开始时间和结束时间不能超过15分钟
        """
        return self._top_request(
            "tmall.servicecenter.tasks.search",
            {
                "start": start,
                "end": end
            },
            result_processor=lambda x: x["service_task_packet"]
        )

    def tmall_servicecenter_workcard_status_update(
            self,
            workcard_id,
            type,
            status,
            updater,
            update_date,
            buyer_id='',
            effect_date='',
            expire_date='',
            comments='',
            address='',
            contact_name='',
            contact_phone='',
            service_date='',
            complete_date='',
            service_voucher_pics='',
            attribute='',
            service_center_code='',
            service_center_name='',
            service_fee='',
            is_visit='',
            before_service_memo='',
            after_service_memo='',
            phone_imei='',
            sub_status='',
            service_center_manager_phone='',
            service_center_manager_name='',
            service_center_address='',
            work_card_install_detail_list=None,
            service_fee_detail=''
    ):
        """
        服务商反馈服务的执行情况
        1 如果服务商受理了此服务，修改合同状态为：已受理=3<br/>2 如果服务商没有受理此服务，修改合同状态为：已拒绝=10<br/>3 如果服务商执行了此服务，修改合同状态为：已执行=4<br/>4 如果服务商执行服务成功，修改合同状态为：已完成=5<br/>5 如果此工单为合同类型的工单，当服务商受理了此服务后，会进行分账
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11123

        :param workcard_id: 工单id
        :param type: 工单类型： 2（合同） 或者 1(任务）
        :param status: 目前仅支持5种状态的反馈：3=授理， 10=拒绝 ，4=执行 ，5=成功，11=失败。（所有状态列表： -1： 初始化 0： 生成 1： 生效 2： 申请 3： 受理 4： 执行 5： 成功 9： 结算 10： 拒绝 11： 失败 12 ： 撤销 13： 暂停 19： 终止）
        :param updater: api调用者
        :param update_date: 更新时间
        :param buyer_id: 买家id
        :param effect_date: 服务生效时间 ：工单类型为合同工单时，必选！
        :param expire_date: 服务失效时间 ：工单类型为合同工单时，必选！
        :param comments: 备注,256个字符以内
        :param address: 任务类工单，预约或者上门地址
        :param contact_name: 任务执行，预约联系人
        :param contact_phone: 任务执行，预约联系人电话
        :param service_date: 服务预约时间
        :param complete_date: 服务完成时间
        :param service_voucher_pics: 服务凭证上传的图片URL链接，多个以;隔开
        :param attribute: 属性定义。例如无忧退货服务，K-V对定义，每对KV用“;”分割，“:”号左边是key右边是value，value如果有多个则以“,”分割。 reasons   :  原因，可能有多个 succeedCount     :    取件成功个数 failedCount    :    取件失败个数 cancelCount      :     取件取消个数 totalCount       :      总取件个数，totalCount= succeedCount + failedCount + cancelCount
        :param service_center_code: 服务商网点内部编码
        :param service_center_name: 服务商网点名字
        :param service_fee: 单元是分
        :param is_visit: 是否上门
        :param before_service_memo: 说明
        :param after_service_memo: 说明
        :param phone_imei: 手机号码
        :param sub_status: 服务子状态：30 表示“服务已申请（上门）” 31表示“服务改约（上门）” 400表示“服务结果（待件上门）” 410表示“服务结果（拖机维修）” 411表示“服务结果（换机）” 420表示“服务结果（上门不可维修）”
        :param service_center_manager_phone: 网点负责人联系电话
        :param service_center_manager_name: 网点负责人
        :param service_center_address: 网点地址
        :param work_card_install_detail_list: 一个工单可能包含多件商品，比如空调可能有两台，录入每天机器的安装情况
        :param service_fee_detail: json string。费用单位为分
        """
        return self._top_request(
            "tmall.servicecenter.workcard.status.update",
            {
                "workcard_id": workcard_id,
                "type": type,
                "status": status,
                "updater": updater,
                "update_date": update_date,
                "buyer_id": buyer_id,
                "effect_date": effect_date,
                "expire_date": expire_date,
                "comments": comments,
                "address": address,
                "contact_name": contact_name,
                "contact_phone": contact_phone,
                "service_date": service_date,
                "complete_date": complete_date,
                "service_voucher_pics": service_voucher_pics,
                "attribute": attribute,
                "service_center_code": service_center_code,
                "service_center_name": service_center_name,
                "service_fee": service_fee,
                "is_visit": is_visit,
                "before_service_memo": before_service_memo,
                "after_service_memo": after_service_memo,
                "phone_imei": phone_imei,
                "sub_status": sub_status,
                "service_center_manager_phone": service_center_manager_phone,
                "service_center_manager_name": service_center_manager_name,
                "service_center_address": service_center_address,
                "work_card_install_detail_list": work_card_install_detail_list,
                "service_fee_detail": service_fee_detail
            }
        )

    def tmall_servicecenter_task_get(
            self,
            parent_biz_order_id
    ):
        """
        服务工单拉取
        接口供服务供应商通过交易主订单查询其未拉取的任务类工单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21669

        :param parent_biz_order_id: Taobao主交易订单ID
        """
        return self._top_request(
            "tmall.servicecenter.task.get",
            {
                "parent_biz_order_id": parent_biz_order_id
            },
            result_processor=lambda x: x["service_task_packet"]
        )

    def tmall_servicecenter_workcard_push(
            self,
            biz_order_id,
            status,
            attributes='',
            desc='',
            service_reserve_time='',
            service_reserve_address=''
    ):
        """
        推送服务工单信息
        服务商家推送工单信息到天猫。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=25775

        :param biz_order_id: 淘宝交易订单号
        :param status: 0=初始化, 3=授理， 10=拒绝 ，4=执行 ，5=成功，11=失败
        :param attributes: 属性列表。使用半角分号隔开,字符串前后都需要有半角分号
        :param desc: 描述
        :param service_reserve_time: 服务预约安装时间
        :param service_reserve_address: 服务预约安装地址。四级地址与街道地址用空格隔开
        """
        return self._top_request(
            "tmall.servicecenter.workcard.push",
            {
                "biz_order_id": biz_order_id,
                "status": status,
                "attributes": attributes,
                "desc": desc,
                "service_reserve_time": service_reserve_time,
                "service_reserve_address": service_reserve_address
            },
            result_processor=lambda x: x["workcard_result"]
        )

    def tmall_msf_identify_status_query(
            self,
            order_id,
            service_type
    ):
        """
        喵师傅定案核销状态查询
        喵师傅定案核销状态查询，供服务商erp系统调用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26096

        :param order_id: 天猫订单号
        :param service_type: 服务类型，0 家装的送货上门并安装 1 单向安装 2 建材的送货上门 3 建材的安装
        """
        return self._top_request(
            "tmall.msf.identify.status.query",
            {
                "order_id": order_id,
                "service_type": service_type
            }
        )

    def tmall_msf_reservation(
            self,
            reserv_info=None
    ):
        """
        喵师傅服务预约API
        喵师傅预约api
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=26099

        :param reserv_info: 预约内容
        """
        return self._top_request(
            "tmall.msf.reservation",
            {
                "reserv_info": reserv_info
            }
        )

    def tmall_service_code_consume(
            self,
            consume_code,
            operator_nick,
            shop_id
    ):
        """
        天猫服务平台服务核销
        天猫服务平台－服务核销
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27070

        :param consume_code: 核销码
        :param operator_nick: 核销帐号
        :param shop_id: 门店id
        """
        return self._top_request(
            "tmall.service.code.consume",
            {
                "consume_code": consume_code,
                "operator_nick": operator_nick,
                "shop_id": shop_id
            }
        )

    def tmall_servicecenter_anomalyrecourse_querybyid(
            self,
            anomaly_recourse_id='4234'
    ):
        """
        根据一键求助id查询指定服务商的一键求助单
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27414

        :param anomaly_recourse_id: 一键求助的id
        """
        return self._top_request(
            "tmall.servicecenter.anomalyrecourse.querybyid",
            {
                "anomaly_recourse_id": anomaly_recourse_id
            }
        )

    def tmall_service_settleadjustment_request(
            self,
            param_settle_adjustment_request
    ):
        """
        创建结算调整单
        提供给服务商在对结算有异议时，发起结算调整单。
        通过说明工单ID，调整费用值，调整原因进行新建结算调整单。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27435

        :param param_settle_adjustment_request: 父节点
        """
        return self._top_request(
            "tmall.service.settleadjustment.request",
            {
                "param_settle_adjustment_request": param_settle_adjustment_request
            }
        )

    def tmall_service_settleadjustment_get(
            self,
            id
    ):
        """
        查询结算调整单单条记录
        提供给服务商通过结算调整单id获取结算调整单信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27438

        :param id: 结算调整单ID
        """
        return self._top_request(
            "tmall.service.settleadjustment.get",
            {
                "id": id
            }
        )

    def tmall_service_settleadjustment_cancel(
            self,
            id,
            comments
    ):
        """
        取消结算调整单
        提供给服务商在对取消已经发起的结算调整单。
        通过说明调整单ID进行结算调整单取消。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27440

        :param id: 结算调整单ID
        :param comments: 取消原因说明
        """
        return self._top_request(
            "tmall.service.settleadjustment.cancel",
            {
                "id": id,
                "comments": comments
            }
        )

    def tmall_servicecenter_task_feedbacknoneedservice(
            self,
            param
    ):
        """
        服务商反馈无需安装工单接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=27441

        :param param: 入参对象
        """
        return self._top_request(
            "tmall.servicecenter.task.feedbacknoneedservice",
            {
                "param": param
            }
        )

    def tmall_servicecenter_task_queryrefund(
            self,
            workcard_list
    ):
        """
        查询任务类工单是否退款
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=28216

        :param workcard_list: 工单id列表
        """
        return self._top_request(
            "tmall.servicecenter.task.queryrefund",
            {
                "workcard_list": workcard_list
            }
        )

    def tmall_servicecenter_picture_upload(
            self,
            img,
            picture_name,
            is_https='true'
    ):
        """
        上传图片
        给服务商ERP系统使用，用于上传图片保存在天猫，一般用于工单信息回传时候保存服务商的服务证明信息相关的图片。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=29583

        :param img: 图片文件二进制流
        :param picture_name: 图片全称包括扩展名。目前支持 jpg jpeg png
        :param is_https: true返回Https地址
        """
        return self._top_request(
            "tmall.servicecenter.picture.upload",
            {
                "img": img,
                "picture_name": picture_name,
                "is_https": is_https
            }
        )

    def tmall_servicecenter_worker_create(
            self,
            worker_dto=None
    ):
        """
        服务商工人信息创建
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30001

        :param worker_dto: 11
        """
        return self._top_request(
            "tmall.servicecenter.worker.create",
            {
                "worker_dto": worker_dto
            }
        )

    def tmall_servicecenter_worker_query(
            self,
            identity_id=''
    ):
        """
        工人信息查询
        查询服务商对应的工人信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30506

        :param identity_id: 身份证号
        """
        return self._top_request(
            "tmall.servicecenter.worker.query",
            {
                "identity_id": identity_id
            }
        )

    def tmall_servicecenter_service_type_queryall(
            self
    ):
        """
        服务供应链服务类型
        查询天猫服务类型列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=30507

        """
        return self._top_request(
            "tmall.servicecenter.service.type.queryall"
        )

    def tmall_servicecenter_worker_querycapacitytask(
            self,
            query=None
    ):
        """
        查询需求容量
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31305

        :param query: 查询对象
        """
        return self._top_request(
            "tmall.servicecenter.worker.querycapacitytask",
            {
                "query": query
            },
            result_processor=lambda x: x["result_base"]
        )

    def tmall_servicecenter_worker_querypage(
            self,
            page_index='0'
    ):
        """
        查询工人列表
        服务商查询工人列表
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=31306

        :param page_index: 页码
        """
        return self._top_request(
            "tmall.servicecenter.worker.querypage",
            {
                "page_index": page_index
            }
        )

    def tmall_servicecenter_servicestore_updateservicestore(
            self,
            service_store
    ):
        """
        修改网点信息
        修改网点信息。对于同一个服务商，通过 service_store_code 保证网点唯一性。需要保证网点存在才能修改。
        错误码
        1, 服务商昵称无效
        2, 缺少省份
        3, 缺少城市
        4, 缺少区域
        5, 缺少详细地址
        6, 传入地址不在标准地址库中，无法解析
        7, 缺少网点编码
        8, 缺少网点名称
        9, 缺少网点电话
        10, 网点已存在
        11, 网点不存在
        12, 系统错误
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33688

        :param service_store: 网点
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.updateservicestore",
            {
                "service_store": service_store
            }
        )

    def tmall_servicecenter_servicestore_deleteservicestore(
            self,
            service_store_code
    ):
        """
        删除网点
        删除网点信息。对于同一个服务商，通过 service_store_code 删除网点。
        错误码
        1, 服务商昵称无效
        2, 缺少省份
        3, 缺少城市
        4, 缺少区域
        5, 缺少详细地址
        6, 传入地址不在标准地址库中，无法解析
        7, 缺少网点编码
        8, 缺少网点名称
        9, 缺少网点电话
        10, 网点已存在
        11, 网点不存在
        12, 系统错误
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33689

        :param service_store_code: 网点名称
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.deleteservicestore",
            {
                "service_store_code": service_store_code
            }
        )

    def tmall_servicecenter_servicestore_deleteservicestorecoverservice(
            self,
            service_store_code,
            biz_type
    ):
        """
        删除网点覆盖的服务
        天猫服务平台删除网点覆盖的服务，
        必选字段：serviceStoreCode、bizType
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33690

        :param service_store_code: 网点编码
        :param biz_type: 业务类型
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.deleteservicestorecoverservice",
            {
                "service_store_code": service_store_code,
                "biz_type": biz_type
            }
        )

    def tmall_servicecenter_servicestore_createservicestore(
            self,
            service_store
    ):
        """
        服务网点创建
        创建网点信息。对于同一个服务商，通过 service_store_code 保证网点唯一性。需要保证网点不存在才能创建。地址信息：中文和编码二选一，都填则以编码address_code为准。
        错误码
        1, 服务商昵称无效
        2, 缺少省份
        3, 缺少城市
        4, 缺少区域
        5, 缺少详细地址
        6, 传入地址不在标准地址库中，无法解析
        7, 缺少网点编码
        8, 缺少网点名称
        9, 缺少网点电话
        10, 网点已存在
        11, 网点不存在
        12, 系统错误
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33691

        :param service_store: 网点
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.createservicestore",
            {
                "service_store": service_store
            }
        )

    def tmall_servicecenter_worker_delete(
            self,
            name='',
            phone='',
            biz_type=''
    ):
        """
        删除工人
        删除工人信息。该接口为多个业务公用，部分字段可忽略。对于电器预约安装业务，同一个服务商，通过工人姓名+手机号+biz_type 保证唯一性。工人已存在才可以删除。
        错误码如下
        100000, 系统错误
        100001, 工人信息校验失败
        100002, 用户校验失败
        100003, 操作失败
        10004,工人信息为空
        10005,服务商id为空或者服务商名称为空
        10006, 工人不存在
        10007, 工人已存在
        10008, 缺少工人姓名
        10009, 缺少工人电话
        10010, 网点不存在
        11000, category_id 无效
        11001, biz_type 无效
        20001,已查询到最后一页
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33696

        :param name: 工人姓名
        :param phone: 工人手机号
        :param biz_type: 业务类型,电器预约安装业务填appliance_install
        """
        return self._top_request(
            "tmall.servicecenter.worker.delete",
            {
                "name": name,
                "phone": phone,
                "biz_type": biz_type
            }
        )

    def tmall_servicecenter_worker_update(
            self,
            worker=None
    ):
        """
        修改工人信息
        修改工人信息。该接口为多个业务公用，部分字段可忽略。对于电器预约安装业务，同一个服务商，通过工人姓名+手机号+biz_type 保证唯一性。工人已存在才可以修改。
        错误码如下
        100000, 系统错误
        100001, 工人信息校验失败
        100002, 用户校验失败
        100003, 操作失败
        10004,工人信息为空
        10005,服务商id为空或者服务商名称为空
        10006, 工人不存在
        10007, 工人已存在
        10008, 缺少工人姓名
        10009, 缺少工人电话
        10010, 网点不存在
        11000, category_id 无效
        11001, biz_type 无效
        20001,已查询到最后一页
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33699

        :param worker: 工人信息
        """
        return self._top_request(
            "tmall.servicecenter.worker.update",
            {
                "worker": worker
            }
        )

    def tmall_servicecenter_servicemonitormessage_update(
            self,
            service_monitor_message_id,
            status,
            memo=''
    ):
        """
        服务商更新预警消息状态
        服务商收到预警后，需要进行回复已读状态，并可填写备注
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33931

        :param service_monitor_message_id: 预警消息id
        :param status: 可更新状态：3、已读
        :param memo: 预警处理备注
        """
        return self._top_request(
            "tmall.servicecenter.servicemonitormessage.update",
            {
                "service_monitor_message_id": service_monitor_message_id,
                "status": status,
                "memo": memo
            }
        )

    def tmall_servicecenter_servicemonitormessage_search(
            self,
            start,
            end
    ):
        """
        根据时间段查询服务商的服务预警消息列表(15分钟内)
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=33932

        :param start: 开始时间long值
        :param end: 结束时间long值，与start相差15分钟
        """
        return self._top_request(
            "tmall.servicecenter.servicemonitormessage.search",
            {
                "start": start,
                "end": end
            }
        )

    def tmall_fuwu_serviceitem_list(
            self,
            seller_id='',
            itemids=''
    ):
        """
        获取服务商品扩展信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=36961

        :param seller_id: 商品所属卖家账号id
        :param itemids: 商品id列表，有数量限制
        """
        return self._top_request(
            "tmall.fuwu.serviceitem.list",
            {
                "seller_id": seller_id,
                "itemids": itemids
            }
        )

    def tmall_servicecenter_msf_tasks_update(
            self,
            worker_mobile,
            service_code,
            source,
            biz_order_ids,
            seqs
    ):
        """
        喵师傅工人任务批量完成接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=38133

        :param worker_mobile: 工人手机号
        :param service_code: 服务编码
        :param source: 调用来源
        :param biz_order_ids: 子订单号列表。最多100个
        :param seqs: 周期序号。必须与子订单列表对应
        """
        return self._top_request(
            "tmall.servicecenter.msf.tasks.update",
            {
                "worker_mobile": worker_mobile,
                "service_code": service_code,
                "source": source,
                "biz_order_ids": biz_order_ids,
                "seqs": seqs
            }
        )

    def tmall_servicecenter_workcard_reassign(
            self,
            reassign_store_request=None
    ):
        """
        工单改派门店
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42625

        :param reassign_store_request: 请求入参
        """
        return self._top_request(
            "tmall.servicecenter.workcard.reassign",
            {
                "reassign_store_request": reassign_store_request
            }
        )

    def tmall_servicecenter_workcard_verifycode_resend(
            self,
            workcard_id='',
            service_store_id=''
    ):
        """
        重发核销码
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42626

        :param workcard_id: 工单id
        :param service_store_id: 门店/网点id
        """
        return self._top_request(
            "tmall.servicecenter.workcard.verifycode.resend",
            {
                "workcard_id": workcard_id,
                "service_store_id": service_store_id
            }
        )

    def tmall_servicecenter_servicestore_update(
            self,
            param_service_store_d_t_o=None
    ):
        """
        修改门店信息
        用于修改门店/网点信息。多个业务共用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42627

        :param param_service_store_d_t_o: 网点/门店
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.update",
            {
                "param_service_store_d_t_o": param_service_store_d_t_o
            }
        )

    def tmall_servicecenter_workcard_query(
            self,
            service_store_id='',
            identify_code='',
            id='',
            gmt_create_start='',
            gmt_create_end='',
            biz_order_id='',
            current_page='1',
            page_size='50'
    ):
        """
        工单查询接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42632

        :param service_store_id: 门店/网点id
        :param identify_code: 核销码
        :param id: 工单id
        :param gmt_create_start: 工单创建开始时间
        :param gmt_create_end: 工单创建结束时间，必须与工单创建开始时间一起传入，且间隔不超过15分钟
        :param biz_order_id: 淘宝交易订单号。主订单或子订单均可
        :param current_page: 当前页数
        :param page_size: 每页大小
        """
        return self._top_request(
            "tmall.servicecenter.workcard.query",
            {
                "service_store_id": service_store_id,
                "identify_code": identify_code,
                "id": id,
                "gmt_create_start": gmt_create_start,
                "gmt_create_end": gmt_create_end,
                "biz_order_id": biz_order_id,
                "current_page": current_page,
                "page_size": page_size
            }
        )

    def tmall_servicecenter_servicestore_create(
            self,
            service_store=None
    ):
        """
        创建门店
        用于创建门店/网点。多个业务共用
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42633

        :param service_store: 网点/门店
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.create",
            {
                "service_store": service_store
            }
        )

    def tmall_servicecenter_servicestore_updatestatus(
            self,
            id,
            biz_type,
            status=''
    ):
        """
        网点/门店状态修改
        修改网点/门店状态
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42634

        :param id: 门店id
        :param biz_type: 业务类型。不同业务传不同的值
        :param status: 状态。1 营业，0歇业，-1彻底关店
        """
        return self._top_request(
            "tmall.servicecenter.servicestore.updatestatus",
            {
                "id": id,
                "biz_type": biz_type,
                "status": status
            }
        )

    def tmall_mallitemcenter_supplier_ability_update(
            self,
            param0
    ):
        """
        门店服务能力授权接口
        门店服务能力授权
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42734

        :param param0: 入参
        """
        return self._top_request(
            "tmall.mallitemcenter.supplier.ability.update",
            {
                "param0": param0
            }
        )

    def tmall_mallitemcenter_supplier_price_upload(
            self,
            provider_price_list,
            service_code=''
    ):
        """
        天猫服务商服务报价上传
        天猫服务商上传服务价格
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42735

        :param provider_price_list: 服务商门店价格列表
        :param service_code: 服务code
        """
        return self._top_request(
            "tmall.mallitemcenter.supplier.price.upload",
            {
                "provider_price_list": provider_price_list,
                "service_code": service_code
            }
        )

    def tmall_mallitemcenter_serviceproduct_query(
            self,
            id='',
            status='',
            service_code='',
            service_product_type=''
    ):
        """
        天猫服务商服务产品查询
        查询天猫服务的服务商发布的服务产品
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42736

        :param id: 服务产品id
        :param status: 产品状态
        :param service_code: 服务名称
        :param service_product_type: 产品类型
        """
        return self._top_request(
            "tmall.mallitemcenter.serviceproduct.query",
            {
                "id": id,
                "status": status,
                "service_code": service_code,
                "service_product_type": service_product_type
            }
        )

    def tmall_mallitemcenter_subscribe_query(
            self,
            query=None
    ):
        """
        天猫服务订购信息查询接口
        查询商家服务订购信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=42770

        :param query: 入参query
        """
        return self._top_request(
            "tmall.mallitemcenter.subscribe.query",
            {
                "query": query
            }
        )

    def tmall_servicecenter_workcard_reserve(
            self,
            workcard_id='',
            reserve_time_start='',
            reserve_time_end='',
            worker_mobile='',
            reserve_remark='',
            worker_name=''
    ):
        """
        工单预约
        服务工单更新通用接口
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=43637

        :param workcard_id: 工单id
        :param reserve_time_start: 服务开始时间
        :param reserve_time_end: 服务结束时间
        :param worker_mobile: 工人手机号
        :param reserve_remark: 预约备注信息
        :param worker_name: 工人姓名
        """
        return self._top_request(
            "tmall.servicecenter.workcard.reserve",
            {
                "workcard_id": workcard_id,
                "reserve_time_start": reserve_time_start,
                "reserve_time_end": reserve_time_end,
                "worker_mobile": worker_mobile,
                "reserve_remark": reserve_remark,
                "worker_name": worker_name
            }
        )


class TbTianMaoJingPinKu(DingTalkBaseAPI):
    """
    天猫精品库API
    """

    def tmall_items_extend_search(
            self,
            q='',
            cat='',
            sort='',
            brand='',
            loc='',
            prop='',
            post_fee='',
            start_price='',
            end_price='',
            support_cod='',
            many_points='',
            wwonline='',
            vip='',
            combo='',
            miaosha='',
            nspu='',
            auction_tag='',
            spuid='',
            user_id='',
            page_no='1',
            page_size='40',
            category=''
    ):
        """
        搜索天猫商品
        提供天猫商品搜索结果，需要调用精选商品，请改为调用：tmall.selected.items.search
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21643

        :param q: 表示搜索的关键字，例如搜索query=nike。当输入关键字为中文时，将对他进行URLEncode的UTF-8格式编码，如 耐克，那么q=%E8%80%90%E5%85%8B。
        :param cat: 前台类目id，支持多选过滤，cat=catid1,catid2
        :param sort: 排序类型。类型包括：s: 人气排序p: 价格从低到高;pd: 价格从高到低;d: 月销量从高到低;td: 总销量从高到低;pt: 按发布时间排序.
        :param brand: 品牌的id。支持多选过滤，brand=brand1,brand2
        :param loc: 宝贝卖家所在地，中文gbk编码
        :param prop: 以“属性id：属性值”的形式传入;
        :param post_fee: 是否包邮，-1为包邮
        :param start_price: 在宝贝页面中进行价格筛选的时候，如果填写了最低价格，就会显示该字段。
        :param end_price: 在宝贝页面中进行价格筛选的时候，如果填写了最高价格，就会显示该字段。
        :param support_cod: 是否货到付款，1为货到付款
        :param many_points: 是否多倍积分，1为多倍积分
        :param wwonline: 显示旺旺在线卖家的宝贝时，wwonline=1
        :param vip: 过滤vip宝贝时，vip=1
        :param combo: 过滤搭配减价宝贝时，combo=1
        :param miaosha: 过滤折扣宝贝时，miaosha=1
        :param nspu: 是否需要spu聚合的开关:1为关闭，不传表示遵循后端聚合逻辑。默认不作spu聚合。
        :param auction_tag: 商品标签。支持多选过滤,auction_tag=auction_tag1,auction_tag2,不支持天猫精品库8578
        :param spuid: 可以根据产品Id搜索属于这个spu的商品。
        :param user_id: 可以根据卖家id搜索属于该卖家的商品
        :param page_no: 页码。取值范围：大于零的整数；最大值：100；默认值：1，即默认返回第一页数据。
        :param page_size: 每页条数。取值范围：大于零的整数；最大值：100；默认值：40
        :param category: 后台类目id，category=categoryId
        """
        return self._top_request(
            "tmall.items.extend.search",
            {
                "q": q,
                "cat": cat,
                "sort": sort,
                "brand": brand,
                "loc": loc,
                "prop": prop,
                "post_fee": post_fee,
                "start_price": start_price,
                "end_price": end_price,
                "support_cod": support_cod,
                "many_points": many_points,
                "wwonline": wwonline,
                "vip": vip,
                "combo": combo,
                "miaosha": miaosha,
                "nspu": nspu,
                "auction_tag": auction_tag,
                "spuid": spuid,
                "user_id": user_id,
                "page_no": page_no,
                "page_size": page_size,
                "category": category
            }
        )


class TbJuShiTa(DingTalkBaseAPI):
    """
    聚石塔API
    """

    def taobao_rds_db_create(
            self,
            instance_name,
            db_name,
            account_name=''
    ):
        """
        rds创建数据库
        在rds实例里创建数据库
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11133

        :param instance_name: rds的实例名
        :param db_name: 数据库名
        :param account_name: 已存在账号名
        """
        return self._top_request(
            "taobao.rds.db.create",
            {
                "instance_name": instance_name,
                "db_name": db_name,
                "account_name": account_name
            },
            result_processor=lambda x: x["rds_db_info"]
        )

    def taobao_rds_db_get(
            self,
            instance_name,
            db_status='1'
    ):
        """
        查询rds下的数据库
        查询rds实例下的数据库
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11134

        :param instance_name: rds的实例名
        :param db_status: 数据库状态，默认值1
        """
        return self._top_request(
            "taobao.rds.db.get",
            {
                "instance_name": instance_name,
                "db_status": db_status
            },
            result_processor=lambda x: x["rds_db_infos"]
        )

    def taobao_rds_db_delete(
            self,
            instance_name,
            db_name
    ):
        """
        RDS数据库删除
        通过api删除用户RDS的数据库
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=11135

        :param instance_name: rds的实例名
        :param db_name: 数据库的name，可以通过 taobao.rds.db.get 获取
        """
        return self._top_request(
            "taobao.rds.db.delete",
            {
                "instance_name": instance_name,
                "db_name": db_name
            },
            result_processor=lambda x: x["rds_db_info"]
        )

    def taobao_jushita_jdp_user_delete(
            self,
            nick='',
            user_id=''
    ):
        """
        删除数据推送用户
        删除应用的数据推送用户，用户被删除后，重新添加时会重新同步历史数据。
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21732

        :param nick: 要删除用户的昵称
        :param user_id: 需要删除的用户编号
        """
        return self._top_request(
            "taobao.jushita.jdp.user.delete",
            {
                "nick": nick,
                "user_id": user_id
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_jushita_jdp_user_add(
            self,
            rds_name,
            history_days=''
    ):
        """
        添加数据推送用户
        提供给接入数据推送的应用添加数据推送服务的用户
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=21736

        :param rds_name: RDS实例名称
        :param history_days: 推送历史数据天数，只能为90天内，包含90天。当此参数不填时，表示以页面中应用配置的历史天数为准；如果为0表示这个用户不推送历史数据；其它表示推送的历史天数。
        """
        return self._top_request(
            "taobao.jushita.jdp.user.add",
            {
                "rds_name": rds_name,
                "history_days": history_days
            },
            result_processor=lambda x: x["is_success"]
        )

    def taobao_jushita_jdp_users_get(
            self,
            start_modified='',
            end_modified='',
            page_size='200',
            page_no='1',
            user_id=''
    ):
        """
        获取开通的订单同步服务的用户
        获取开通的订单同步服务的用户，含有rds的路由关系
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22129

        :param start_modified: 此参数一般不用传，用于查询最后更改时间在某个时间段内的用户
        :param end_modified: 此参数一般不用传，用于查询最后更改时间在某个时间段内的用户
        :param page_size: 每页记录数，默认200
        :param page_no: 当前页数
        :param user_id: 如果传了user_id表示单条查询
        """
        return self._top_request(
            "taobao.jushita.jdp.users.get",
            {
                "start_modified": start_modified,
                "end_modified": end_modified,
                "page_size": page_size,
                "page_no": page_no,
                "user_id": user_id
            }
        )

    def taobao_jds_trade_traces_get(
            self,
            tid
    ):
        """
        获取单条订单跟踪详情
        获取聚石塔数据共享的交易全链路信息
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22816

        :param tid: 淘宝的订单tid
        """
        return self._top_request(
            "taobao.jds.trade.traces.get",
            {
                "tid": tid
            }
        )

    def taobao_jds_trades_statistics_diff(
            self,
            date,
            pre_status,
            post_status,
            page_no='1'
    ):
        """
        订单全链路状态统计差异比较
        文档地址：https://open-doc.dingtalk.com/docs/api.htm?apiId=22883

        :param date: 查询的日期，格式如YYYYMMDD的日期对应的数字
        :param pre_status: 需要比较的状态，将会和post_status做比较
        :param post_status: 需要比较的状态
        :param page_no: 页数
        """
        return self._top_request(
            "taobao.jds.trades.statistics.diff",
            {
                "date": date,
                "pre_status": pre_status,
                "post_status": post_status,
                "page