#!/usr/bin/env python
u"""
gesdisc_merra_subset.py
Written by Tyler Sutterley (06/2022)

Subsets monthly MERRA-2 products for specific variables from the
    Goddard Earth Sciences Data and Information Server Center (GES DISC)
    https://gmao.gsfc.nasa.gov/reanalysis/MERRA-2/
    https://wiki.earthdata.nasa.gov/display/EL/How+To+Access+Data+With+Python

Register with NASA Earthdata Login system:
    https://urs.earthdata.nasa.gov

Add "NASA GESDISC DATA ARCHIVE" to Earthdata Applications:
    https://urs.earthdata.nasa.gov/approve_app?client_id=e2WVk8Pw6weeLUKZYOxvTQ

CALLING SEQUENCE:
    python gesdisc_merra_subset.py --user <username> --variables PS
    where <username> is your NASA Earthdata username

COMMAND LINE OPTIONS:
    --help: list the command line options
    -U X, --user X: username for NASA Earthdata Login
    -W X, --password X: password for NASA Earthdata Login
    -N X, --netrc X: path to .netrc file for authentication
    -D X, --directory X: Working data directory
    -s X, --shortname X: MERRA-2 product shortname
    -v X, --version X: MERRA-2 version
    -Y X, --year X: years to sync
    -V X, --variables X: variables to subset
    -t X, --timeout X: Timeout in seconds for blocking operations
    -l, --log: output log of files downloaded
    -C, --clobber: Overwrite existing files
    -M X, --mode X: Local permissions mode of the files created

PYTHON DEPENDENCIES:
    numpy: Scientific Computing Tools For Python
        https://numpy.org
        https://numpy.org/doc/stable/user/numpy-for-matlab-users.html
    dateutil: powerful extensions to datetime
        https://dateutil.readthedocs.io/en/stable/
    lxml: Pythonic XML and HTML processing library using libxml2/libxslt
        https://lxml.de/
        https://github.com/lxml/lxml
    future: Compatibility layer between Python 2 and Python 3
        https://python-future.org/

PROGRAM DEPENDENCIES:
    time.py: Utilities for calculating time operations
    utilities.py: download and management utilities for syncing files

UPDATE HISTORY:
    Written 06/2022
"""
from __future__ import print_function

import sys
import os
import time
import netrc
import shutil
import getpass
import logging
import argparse
import builtins
import gravity_toolkit.time
import model_harmonics.utilities

#-- PURPOSE: subsets MERRA-2 files for specific variables
def gesdisc_merra_subset(base_dir, SHORTNAME, VERSION=None, YEARS=None,
    VARIABLES=None, TIMEOUT=None, LOG=False, CLOBBER=False, MODE=None):
    #-- full path to MERRA-2 directory
    DIRECTORY = os.path.join(base_dir,'MERRA-2')
    #-- output file format
    file_format = '{0}.SUB.nc'
    #-- check if DIRECTORY exists and recursively create if not
    if not os.access(os.path.join(DIRECTORY), os.F_OK):
        os.makedirs(os.path.join(DIRECTORY), mode=MODE, exist_ok=True)

    #-- create log file with list of synchronized files (or print to terminal)
    if LOG:
        #-- format: NASA_GESDISC_MERRA2_subset_2002-04-01.log
        today = time.strftime('%Y-%m-%d',time.localtime())
        LOGFILE = 'NASA_GESDISC_MERRA2_subset_{0}.log'.format(today)
        fid = open(os.path.join(DIRECTORY,LOGFILE),'w')
        logging.basicConfig(stream=fid, level=logging.INFO)
        logging.info('NASA MERRA-2 Sync Log ({0})'.format(today))
        PRODUCT = '{0}.{1}'.format(SHORTNAME,VERSION)
        logging.info('PRODUCT: {0}'.format(PRODUCT))
    else:
        #-- standard output (terminal output)
        fid = sys.stdout
        logging.basicConfig(stream=fid, level=logging.INFO)

    #-- for each unique date
    for YEAR in YEARS:
        dpm = gravity_toolkit.time.calendar_days(YEAR)
        #-- for each month of the year
        for i,days_per_month in enumerate(dpm):
            #-- year and month as strings
            YY = '{0:4d}'.format(YEAR)
            MM = '{0:02d}'.format(i+1)
            #-- start and end date for query
            start_date = '{0}-{1}-{2:02.0f}'.format(YY,MM,1.0)
            end_date = '{0}-{1}-{2:02.0f}'.format(YY,MM,days_per_month)
            #-- query for data
            ids,urls,mtimes = model_harmonics.utilities.cmr(SHORTNAME,
                version=VERSION, start_date=start_date, end_date=end_date,
                provider='GES_DISC', verbose=True)
            #-- skip years and months without any data
            if not ids:
                continue
            #-- for each granule
            for id,url,mtime in zip(ids,urls,mtimes):
                #-- build filename for output
                fileBasename,_ = os.path.splitext(id)
                FILE = file_format.format(fileBasename)
                local_file = os.path.join(DIRECTORY, FILE)
                #-- get subsetting API url for granule
                request_url = model_harmonics.utilities.build_request(
                    SHORTNAME, VERSION, url, variables=VARIABLES,
                    bbox=[-90,-180,90,180], LABEL=FILE)
                #-- copy subsetted file and update modified dates
                http_pull_file(request_url, mtime, local_file,
                    TIMEOUT=TIMEOUT, CLOBBER=CLOBBER, MODE=0o775)

    #-- close log file and set permissions level to MODE
    if LOG:
        fid.close()
        os.chmod(os.path.join(DIRECTORY,LOGFILE), MODE)

#-- PURPOSE: pull file from a remote host checking if file exists locally
#-- and if the remote file is newer than the local file
def http_pull_file(remote_file, remote_mtime, local_file,
    TIMEOUT=None, CLOBBER=False, MODE=0o775):
    #-- if file exists in file system: check if remote file is newer
    TEST = False
    OVERWRITE = ' (clobber)'
    #-- check if local version of file exists
    if os.access(local_file, os.F_OK):
        #-- check last modification time of local file
        local_mtime = os.stat(local_file).st_mtime
        #-- if remote file is newer: overwrite the local file
        if (model_harmonics.utilities.even(remote_mtime) >
            model_harmonics.utilities.even(local_mtime)):
            TEST = True
            OVERWRITE = ' (overwrite)'
    else:
        TEST = True
        OVERWRITE = ' (new)'
    #-- if file does not exist locally, is to be overwritten, or CLOBBER is set
    if TEST or CLOBBER:
        #-- Printing files transferred
        logging.info('{0} --> '.format(remote_file))
        logging.info('\t{0}{1}\n'.format(local_file,OVERWRITE))
        #-- Create and submit request. There are a wide range of exceptions
        #-- that can be thrown here, including HTTPError and URLError.
        request = model_harmonics.utilities.urllib2.Request(remote_file)
        response = model_harmonics.utilities.urllib2.urlopen(request,
            timeout=TIMEOUT)
        #-- chunked transfer encoding size
        CHUNK = 16 * 1024
        #-- copy contents to local file using chunked transfer encoding
        #-- transfer should work properly with ascii and binary data formats
        with open(local_file, 'wb') as f:
            shutil.copyfileobj(response, f, CHUNK)
        #-- keep remote modification time of file and local access time
        os.utime(local_file, (os.stat(local_file).st_atime, remote_mtime))
        os.chmod(local_file, MODE)

#-- PURPOSE: create argument parser
def arguments():
    parser = argparse.ArgumentParser(
        description="""Creates monthly MERRA-2 3D model level
            products syncing data from the Goddard Earth Sciences
            Data and Information Server Center (GES DISC)
            """
    )
    #-- command line parameters
    #-- NASA Earthdata credentials
    parser.add_argument('--user','-U',
        type=str, default=os.environ.get('EARTHDATA_USERNAME'),
        help='Username for NASA Earthdata Login')
    parser.add_argument('--password','-W',
        type=str, default=os.environ.get('EARTHDATA_PASSWORD'),
        help='Password for NASA Earthdata Login')
    parser.add_argument('--netrc','-N',
        type=lambda p: os.path.abspath(os.path.expanduser(p)),
        default=os.path.join(os.path.expanduser('~'),'.netrc'),
        help='Path to .netrc file for authentication')
    #-- working data directory
    parser.add_argument('--directory','-D',
        type=lambda p: os.path.abspath(os.path.expanduser(p)),
        default=os.getcwd(),
        help='Working data directory')
    #-- MERRA-2 product shortname
    parser.add_argument('--shortname','-s',
        type=str, default='M2TMNXSLV',
        help='MERRA-2 product shortname')
    #-- MERRA-2 version
    parser.add_argument('--version','-v',
        type=str, default='5.12.4',
        help='MERRA-2 version')
    #-- years to download
    now = time.gmtime()
    parser.add_argument('--year','-Y',
        type=int, nargs='+', default=range(2000,now.tm_year+1),
        help='Years of model outputs to sync')
    #-- variable names to subset from product
    parser.add_argument('--variables','-V',
        type=str, nargs='+',
        help='Variables to subset from product')
    #-- connection timeout
    parser.add_argument('--timeout','-t',
        type=int, default=360,
        help='Timeout in seconds for blocking operations')
    #-- Output log file in form
    #-- NASA_GESDISC_MERRA2_subset_2002-04-01.log
    parser.add_argument('--log','-l',
        default=False, action='store_true',
        help='Output log file')
    #-- sync options
    parser.add_argument('--clobber','-C',
        default=False, action='store_true',
        help='Overwrite existing data in transfer')
    #-- permissions mode of the directories and files synced (number in octal)
    parser.add_argument('--mode','-M',
        type=lambda x: int(x,base=8), default=0o775,
        help='Permission mode of directories and files synced')
    #-- return the parser
    return parser

#-- This is the main part of the program that calls the individual functions
def main():
    #-- Read the system arguments listed after the program
    parser = arguments()
    args,_ = parser.parse_known_args()

    #-- NASA Earthdata hostname
    URS = 'urs.earthdata.nasa.gov'
    #-- get NASA Earthdata credentials
    try:
        args.user,_,args.password = netrc.netrc(args.netrc).authenticators(URS)
    except:
        #-- check that NASA Earthdata credentials were entered
        if not args.user:
            prompt = 'Username for {0}: '.format(URS)
            args.user = builtins.input(prompt)
        #-- enter password securely from command-line
        if not args.password:
            prompt = 'Password for {0}@{1}: '.format(args.user,URS)
            args.password = getpass.getpass(prompt)

    #-- build a urllib opener for NASA GESDISC
    #-- Add the username and password for NASA Earthdata Login system
    model_harmonics.utilities.build_opener(args.user, args.password,
        password_manager=True, authorization_header=False)

    #-- check internet connection before attempting to run program
    HOST = 'https://goldsmr5.gesdisc.eosdis.nasa.gov/'
    if model_harmonics.utilities.check_credentials(HOST):
        gesdisc_merra_subset(args.directory, args.shortname,
            VERSION=args.version,
            YEARS=args.year,
            VARIABLES=args.variables,
            TIMEOUT=args.timeout,
            LOG=args.log,
            CLOBBER=args.clobber,
            MODE=args.mode)

#-- run main program
if __name__ == '__main__':
    main()
