"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const ec2 = require("@aws-cdk/aws-ec2");
const ecr = require("@aws-cdk/aws-ecr");
const ecs = require("@aws-cdk/aws-ecs");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const batch = require("../lib");
describe('Batch Job Definition', () => {
    let stack;
    let jobDefProps;
    beforeEach(() => {
        stack = new cdk.Stack();
        const role = new iam.Role(stack, 'job-role', {
            assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
        });
        const linuxParams = new ecs.LinuxParameters(stack, 'job-linux-params', {
            initProcessEnabled: true,
            sharedMemorySize: 1,
        });
        jobDefProps = {
            jobDefinitionName: 'test-job',
            container: {
                command: ['echo "Hello World"'],
                environment: {
                    foo: 'bar',
                },
                jobRole: role,
                gpuCount: 1,
                image: ecs.EcrImage.fromRegistry('docker/whalesay'),
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.T2, ec2.InstanceSize.MICRO),
                linuxParams,
                memoryLimitMiB: 1,
                mountPoints: new Array(),
                privileged: true,
                readOnly: true,
                ulimits: new Array(),
                user: 'root',
                vcpus: 2,
                volumes: new Array(),
            },
            nodeProps: {
                count: 2,
                mainNode: 1,
                rangeProps: new Array(),
            },
            parameters: {
                foo: 'bar',
            },
            retryAttempts: 2,
            timeout: cdk.Duration.seconds(30),
        };
    });
    test('renders the correct cloudformation properties', () => {
        // WHEN
        new batch.JobDefinition(stack, 'job-def', jobDefProps);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            JobDefinitionName: jobDefProps.jobDefinitionName,
            ContainerProperties: jobDefProps.container ? {
                Command: jobDefProps.container.command,
                Environment: [
                    {
                        Name: 'foo',
                        Value: 'bar',
                    },
                ],
                InstanceType: jobDefProps.container.instanceType ? jobDefProps.container.instanceType.toString() : '',
                LinuxParameters: {},
                Memory: jobDefProps.container.memoryLimitMiB,
                MountPoints: [],
                Privileged: jobDefProps.container.privileged,
                ReadonlyRootFilesystem: jobDefProps.container.readOnly,
                ResourceRequirements: [{ Type: 'GPU', Value: String(jobDefProps.container.gpuCount) }],
                Ulimits: [],
                User: jobDefProps.container.user,
                Vcpus: jobDefProps.container.vcpus,
                Volumes: [],
            } : undefined,
            NodeProperties: jobDefProps.nodeProps ? {
                MainNode: jobDefProps.nodeProps.mainNode,
                NodeRangeProperties: [],
                NumNodes: jobDefProps.nodeProps.count,
            } : undefined,
            Parameters: {
                foo: 'bar',
            },
            RetryStrategy: {
                Attempts: jobDefProps.retryAttempts,
            },
            Timeout: {
                AttemptDurationSeconds: jobDefProps.timeout ? jobDefProps.timeout.toSeconds() : -1,
            },
            Type: 'container',
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can use an ecr image', () => {
        // WHEN
        const repo = new ecr.Repository(stack, 'image-repo');
        new batch.JobDefinition(stack, 'job-def', {
            container: {
                image: ecs.ContainerImage.fromEcrRepository(repo),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            ContainerProperties: {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    4,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::GetAtt': [
                                                    'imagerepoD116FAF0',
                                                    'Arn',
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            },
                            '.dkr.ecr.',
                            {
                                'Fn::Select': [
                                    3,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::GetAtt': [
                                                    'imagerepoD116FAF0',
                                                    'Arn',
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            },
                            '.',
                            {
                                Ref: 'AWS::URLSuffix',
                            },
                            '/',
                            {
                                Ref: 'imagerepoD116FAF0',
                            },
                            ':latest',
                        ],
                    ],
                },
                Memory: 4,
                Privileged: false,
                ReadonlyRootFilesystem: false,
                Vcpus: 1,
            },
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can use a registry image', () => {
        // WHEN
        new batch.JobDefinition(stack, 'job-def', {
            container: {
                image: ecs.ContainerImage.fromRegistry('docker/whalesay'),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Batch::JobDefinition', {
            ContainerProperties: {
                Image: 'docker/whalesay',
                Memory: 4,
                Privileged: false,
                ReadonlyRootFilesystem: false,
                Vcpus: 1,
            },
        }, have_resource_1.ResourcePart.Properties);
    });
    test('can be imported from an ARN', () => {
        // WHEN
        const importedJob = batch.JobDefinition.fromJobDefinitionArn(stack, 'job-def-clone', 'arn:aws:batch:us-east-1:123456789012:job-definition/job-def-name:1');
        // THEN
        expect(importedJob.jobDefinitionName).toEqual('job-def-name:1');
        expect(importedJob.jobDefinitionArn).toEqual('arn:aws:batch:us-east-1:123456789012:job-definition/job-def-name:1');
    });
});
//# sourceMappingURL=data:application/json;base64,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