"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = void 0;
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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