"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * Property to specify if the compute environment
 * uses On-Demand or SpotFleet compute resources.
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    /**
     * Resources will be EC2 On-Demand resources.
     */
    ComputeResourceType["ON_DEMAND"] = "EC2";
    /**
     * Resources will be EC2 SpotFleet resources.
     */
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * Properties for how to prepare compute resources
 * that are provisioned for a compute environment.
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    /**
     * Batch will use the best fitting instance type will be used
     * when assigning a batch job in this compute environment.
     */
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    /**
     * Batch will select additional instance types that are large enough to
     * meet the requirements of the jobs in the queue, with a preference for
     * instance types with a lower cost per unit vCPU.
     */
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    /**
     * This is only available for Spot Instance compute resources and will select
     * additional instance types that are large enough to meet the requirements of
     * the jobs in the queue, with a preference for instance types that are less
     * likely to be interrupted.
     */
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 */
class ComputeEnvironment extends core_1.Resource {
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole ? spotFleetRole.roleArn : undefined,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole
                ? props.serviceRole.roleArn
                : new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * Fetches an existing batch compute environment by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param computeEnvironmentArn
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed === undefined ? true : props.managed;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29tcHV0ZS1lbnZpcm9ubWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbXB1dGUtZW52aXJvbm1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsd0NBQXdDO0FBQ3hDLHdDQUF3QztBQUN4Qyx3Q0FBMkU7QUFDM0UsdURBQTBEO0FBRTFEOzs7R0FHRztBQUNILElBQVksbUJBVVg7QUFWRCxXQUFZLG1CQUFtQjtJQUM3Qjs7T0FFRztJQUNILHdDQUFpQixDQUFBO0lBRWpCOztPQUVHO0lBQ0gsb0NBQWEsQ0FBQTtBQUNmLENBQUMsRUFWVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQVU5QjtBQUVEOzs7R0FHRztBQUNILElBQVksa0JBcUJYO0FBckJELFdBQVksa0JBQWtCO0lBQzVCOzs7T0FHRztJQUNILDJDQUFxQixDQUFBO0lBRXJCOzs7O09BSUc7SUFDSCxtRUFBNkMsQ0FBQTtJQUU3Qzs7Ozs7T0FLRztJQUNILHlFQUFtRCxDQUFBO0FBQ3JELENBQUMsRUFyQlcsa0JBQWtCLEdBQWxCLDBCQUFrQixLQUFsQiwwQkFBa0IsUUFxQjdCO0FBa1BEOzs7O0dBSUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLGVBQVE7SUFrQzlDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBaUMsRUFBRSxPQUFPLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxJQUFJLEVBQUU7UUFDekcsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtTQUMzQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTFCLE1BQU0sYUFBYSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNuRCxJQUFJLGdCQUE0RSxDQUFDO1FBRWpGLGlFQUFpRTtRQUNqRSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ25ELGdCQUFnQixHQUFHO2dCQUNqQixrQkFBa0IsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCO3VCQUN6RCxDQUNELEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEtBQUssbUJBQW1CLENBQUMsSUFBSTt3QkFDdEQsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLHVCQUF1Qjt3QkFDNUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FDaEM7Z0JBQ0YsYUFBYSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhO2dCQUNuRCxZQUFZLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVk7Z0JBQ2pELFVBQVUsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsVUFBVTtnQkFDN0MsT0FBTyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTztnQkFDNUYsWUFBWSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZO29CQUMvQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLFlBQVk7b0JBQ3JDLENBQUMsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7d0JBQ3JELEtBQUssRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7Z0NBQzlDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztnQ0FDeEQsZUFBZSxFQUFFO29DQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsa0RBQWtELENBQUM7aUNBQy9GOzZCQUNGLENBQUMsQ0FBQyxRQUFRLENBQUM7cUJBQ2IsQ0FBQyxDQUFDLE9BQU87Z0JBQ1osYUFBYSxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxDQUFDO2dCQUM1RSxjQUFjLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGNBQWM7Z0JBQ3JELFFBQVEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxJQUFJLEdBQUc7Z0JBQ2hELFFBQVEsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxJQUFJLENBQUM7Z0JBQzlDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLENBQUM7Z0JBQy9HLGdCQUFnQixFQUFFLGFBQWEsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDbkUsT0FBTyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQyxTQUFTO2dCQUM5RixJQUFJLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLG9CQUFvQjtnQkFDakQsSUFBSSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLElBQUksbUJBQW1CLENBQUMsU0FBUzthQUNuRSxDQUFDO1NBQ0g7UUFFRCxNQUFNLGtCQUFrQixHQUFHLElBQUksdUNBQXFCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNyRSxzQkFBc0IsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN6QyxnQkFBZ0I7WUFDaEIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO2dCQUM1QixDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxPQUFPO2dCQUMzQixDQUFDLENBQUMsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxnQ0FBZ0MsRUFBRTtvQkFDckQsZUFBZSxFQUFFO3dCQUNmLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsa0NBQWtDLENBQUM7cUJBQy9FO29CQUNELFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxxQkFBcUIsQ0FBQztpQkFDM0QsQ0FBQyxDQUFDLE9BQU87WUFDWixLQUFLLEVBQUUsS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztZQUN6RixJQUFJLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxXQUFXO1NBQ3RELENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLGdCQUFnQixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUU7WUFDeEQsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUU7WUFDaEYsT0FBTyxFQUFFLE9BQU87WUFDaEIsUUFBUSxFQUFFLHFCQUFxQjtZQUMvQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLHNCQUFzQixHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0RixDQUFDO0lBdkdEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyx5QkFBeUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxxQkFBNkI7UUFDakcsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM5QixNQUFNLHNCQUFzQixHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMscUJBQXFCLENBQUMsQ0FBQyxZQUFhLENBQUM7UUFFbkYsTUFBTSxNQUFPLFNBQVEsZUFBUTtZQUE3Qjs7Z0JBQ2tCLDBCQUFxQixHQUFHLHFCQUFxQixDQUFDO2dCQUM5QywyQkFBc0IsR0FBRyxzQkFBc0IsQ0FBQztZQUNsRSxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBd0ZPLFNBQVMsQ0FBQyxLQUE4QjtRQUM5QyxPQUFPLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUM7SUFDNUQsQ0FBQztJQUVEOztPQUVHO0lBQ0ssYUFBYSxDQUFDLEtBQThCO1FBQ2xELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUN2QixPQUFPO1NBQ1I7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQ2xFLE1BQU0sSUFBSSxLQUFLLENBQUMsbUZBQW1GLENBQUMsQ0FBQztTQUN0RztRQUVELElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEtBQUssU0FBUyxFQUFFO1lBQ2pFLE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLENBQUMsQ0FBQztTQUM5RjtRQUVELCtEQUErRDtRQUMvRCwwREFBMEQ7UUFDMUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxLQUFLLG1CQUFtQixDQUFDLFNBQVMsRUFBRTtnQkFDakUseUJBQXlCO2dCQUV6QixnQ0FBZ0M7Z0JBQ2hDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7b0JBQ3RELE1BQU0sSUFBSSxLQUFLLENBQUMsbUdBQW1HLENBQUMsQ0FBQztpQkFDdEg7Z0JBRUQsb0RBQW9EO2dCQUNwRCxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLEtBQUssa0JBQWtCLENBQUMsdUJBQXVCLEVBQUU7b0JBQ3pJLE1BQU0sSUFBSSxLQUFLLENBQUMsdUhBQXVILENBQUMsQ0FBQztpQkFDMUk7YUFDRjtpQkFBTTtnQkFDTCxvQkFBb0I7Z0JBRXBCLHNDQUFzQztnQkFDdEMsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxLQUFLLFNBQVM7b0JBQ3BELENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsR0FBRyxDQUFDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsR0FBRyxHQUFHLENBQUMsRUFBRTtvQkFDMUYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO2lCQUN6RTthQUNGO1lBRUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxFQUFFO2dCQUNuQyxpQ0FBaUM7Z0JBQ2pDLElBQUksS0FBSyxDQUFDLGdCQUFnQixDQUFDLFFBQVEsR0FBRyxDQUFDLEVBQUU7b0JBQ3ZDLE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQztpQkFDeEY7Z0JBRUQsbUNBQW1DO2dCQUNuQyxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRO29CQUNqQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUU7b0JBQ25FLE1BQU0sSUFBSSxLQUFLLENBQUMsd0RBQXdELENBQUMsQ0FBQztpQkFDM0U7YUFDRjtTQUNGO0lBQ0gsQ0FBQztJQUVPLGtCQUFrQixDQUFDLGFBQWtDO1FBQzNELElBQUksYUFBYSxLQUFLLFNBQVMsRUFBRTtZQUMvQixPQUFPO2dCQUNMLFNBQVM7YUFDVixDQUFDO1NBQ0g7UUFFRCxPQUFPLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFzQixFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRU8scUJBQXFCLENBQUMsR0FBYSxFQUFFLGNBQXFDO1FBQ2hGLElBQUksY0FBYyxLQUFLLFNBQVMsRUFBRTtZQUNoQyxPQUFPO2dCQUNMLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDLGVBQWU7YUFDaEYsQ0FBQztTQUNIO1FBRUQsT0FBTyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBeUIsRUFBRSxFQUFFLENBQUMsS0FBSyxDQUFDLGVBQWUsQ0FBQyxDQUFDO0lBQ2xGLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNLLGdCQUFnQixDQUFDLEtBQThCOztRQUNyRCxJQUFJLE9BQUEsS0FBSyxDQUFDLGdCQUFnQiwwQ0FBRSxrQkFBa0IsS0FBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLEtBQUssa0JBQWtCLENBQUMsUUFBUSxFQUFFO1lBQzNILE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsYUFBYSxFQUFFO2dCQUN4QyxPQUFPLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxhQUFhLENBQUM7YUFDN0M7aUJBQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxLQUFLLG1CQUFtQixDQUFDLElBQUksRUFBRTtnQkFDbkUsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQ3pELE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLFNBQVMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLDhFQUE4RSxDQUFDLENBQUM7YUFDekk7U0FDRjtRQUVELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Q0FDRjtBQWhORCxnREFnTkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIElSZXNvdXJjZSwgUmVzb3VyY2UsIFN0YWNrLCBUYWcgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENmbkNvbXB1dGVFbnZpcm9ubWVudCB9IGZyb20gJy4vYmF0Y2guZ2VuZXJhdGVkJztcblxuLyoqXG4gKiBQcm9wZXJ0eSB0byBzcGVjaWZ5IGlmIHRoZSBjb21wdXRlIGVudmlyb25tZW50XG4gKiB1c2VzIE9uLURlbWFuZCBvciBTcG90RmxlZXQgY29tcHV0ZSByZXNvdXJjZXMuXG4gKi9cbmV4cG9ydCBlbnVtIENvbXB1dGVSZXNvdXJjZVR5cGUge1xuICAvKipcbiAgICogUmVzb3VyY2VzIHdpbGwgYmUgRUMyIE9uLURlbWFuZCByZXNvdXJjZXMuXG4gICAqL1xuICBPTl9ERU1BTkQgPSAnRUMyJyxcblxuICAvKipcbiAgICogUmVzb3VyY2VzIHdpbGwgYmUgRUMyIFNwb3RGbGVldCByZXNvdXJjZXMuXG4gICAqL1xuICBTUE9UID0gJ1NQT1QnLFxufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGhvdyB0byBwcmVwYXJlIGNvbXB1dGUgcmVzb3VyY2VzXG4gKiB0aGF0IGFyZSBwcm92aXNpb25lZCBmb3IgYSBjb21wdXRlIGVudmlyb25tZW50LlxuICovXG5leHBvcnQgZW51bSBBbGxvY2F0aW9uU3RyYXRlZ3kge1xuICAvKipcbiAgICogQmF0Y2ggd2lsbCB1c2UgdGhlIGJlc3QgZml0dGluZyBpbnN0YW5jZSB0eXBlIHdpbGwgYmUgdXNlZFxuICAgKiB3aGVuIGFzc2lnbmluZyBhIGJhdGNoIGpvYiBpbiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqL1xuICBCRVNUX0ZJVCA9ICdCRVNUX0ZJVCcsXG5cbiAgLyoqXG4gICAqIEJhdGNoIHdpbGwgc2VsZWN0IGFkZGl0aW9uYWwgaW5zdGFuY2UgdHlwZXMgdGhhdCBhcmUgbGFyZ2UgZW5vdWdoIHRvXG4gICAqIG1lZXQgdGhlIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvclxuICAgKiBpbnN0YW5jZSB0eXBlcyB3aXRoIGEgbG93ZXIgY29zdCBwZXIgdW5pdCB2Q1BVLlxuICAgKi9cbiAgQkVTVF9GSVRfUFJPR1JFU1NJVkUgPSAnQkVTVF9GSVRfUFJPR1JFU1NJVkUnLFxuXG4gIC8qKlxuICAgKiBUaGlzIGlzIG9ubHkgYXZhaWxhYmxlIGZvciBTcG90IEluc3RhbmNlIGNvbXB1dGUgcmVzb3VyY2VzIGFuZCB3aWxsIHNlbGVjdFxuICAgKiBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGVzIHRoYXQgYXJlIGxhcmdlIGVub3VnaCB0byBtZWV0IHRoZSByZXF1aXJlbWVudHMgb2ZcbiAgICogdGhlIGpvYnMgaW4gdGhlIHF1ZXVlLCB3aXRoIGEgcHJlZmVyZW5jZSBmb3IgaW5zdGFuY2UgdHlwZXMgdGhhdCBhcmUgbGVzc1xuICAgKiBsaWtlbHkgdG8gYmUgaW50ZXJydXB0ZWQuXG4gICAqL1xuICBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCA9ICdTUE9UX0NBUEFDSVRZX09QVElNSVpFRCcsXG59XG5cbi8qKlxuICogTGF1bmNoIHRlbXBsYXRlIHByb3BlcnR5IHNwZWNpZmljYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb24ge1xuICAvKipcbiAgICogVGhlIExhdW5jaCB0ZW1wbGF0ZSBuYW1lXG4gICAqL1xuICByZWFkb25seSBsYXVuY2hUZW1wbGF0ZU5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBsYXVuY2ggdGVtcGxhdGUgdmVyc2lvbiB0byBiZSB1c2VkIChvcHRpb25hbCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGRlZmF1bHQgdmVyc2lvbiBvZiB0aGUgbGF1bmNoIHRlbXBsYXRlXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIHRoZSBzdHJ1Y3R1cmUgb2YgdGhlIGJhdGNoIGNvbXB1dGUgY2x1c3Rlci5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb21wdXRlUmVzb3VyY2VzIHtcbiAgLyoqXG4gICAqIFRoZSBhbGxvY2F0aW9uIHN0cmF0ZWd5IHRvIHVzZSBmb3IgdGhlIGNvbXB1dGUgcmVzb3VyY2UgaW4gY2FzZSBub3QgZW5vdWdoIGluc3RhbmNlcyBvZiB0aGUgYmVzdFxuICAgKiBmaXR0aW5nIGluc3RhbmNlIHR5cGUgY2FuIGJlIGFsbG9jYXRlZC4gVGhpcyBjb3VsZCBiZSBkdWUgdG8gYXZhaWxhYmlsaXR5IG9mIHRoZSBpbnN0YW5jZSB0eXBlIGluXG4gICAqIHRoZSByZWdpb24gb3IgQW1hem9uIEVDMiBzZXJ2aWNlIGxpbWl0cy4gSWYgdGhpcyBpcyBub3Qgc3BlY2lmaWVkLCB0aGUgZGVmYXVsdCBmb3IgdGhlIEVDMlxuICAgKiBDb21wdXRlUmVzb3VyY2VUeXBlIGlzIEJFU1RfRklULCB3aGljaCB3aWxsIHVzZSBvbmx5IHRoZSBiZXN0IGZpdHRpbmcgaW5zdGFuY2UgdHlwZSwgd2FpdGluZyBmb3JcbiAgICogYWRkaXRpb25hbCBjYXBhY2l0eSBpZiBpdCdzIG5vdCBhdmFpbGFibGUuIFRoaXMgYWxsb2NhdGlvbiBzdHJhdGVneSBrZWVwcyBjb3N0cyBsb3dlciBidXQgY2FuIGxpbWl0XG4gICAqIHNjYWxpbmcuIElmIHlvdSBhcmUgdXNpbmcgU3BvdCBGbGVldHMgd2l0aCBCRVNUX0ZJVCB0aGVuIHRoZSBTcG90IEZsZWV0IElBTSBSb2xlIG11c3QgYmUgc3BlY2lmaWVkLlxuICAgKiBCRVNUX0ZJVF9QUk9HUkVTU0lWRSB3aWxsIHNlbGVjdCBhbiBhZGRpdGlvbmFsIGluc3RhbmNlIHR5cGUgdGhhdCBpcyBsYXJnZSBlbm91Z2ggdG8gbWVldCB0aGVcbiAgICogcmVxdWlyZW1lbnRzIG9mIHRoZSBqb2JzIGluIHRoZSBxdWV1ZSwgd2l0aCBhIHByZWZlcmVuY2UgZm9yIGFuIGluc3RhbmNlIHR5cGUgd2l0aCBhIGxvd2VyIGNvc3QuXG4gICAqIFRoZSBkZWZhdWx0IHZhbHVlIGZvciB0aGUgU1BPVCBpbnN0YW5jZSB0eXBlIGlzIFNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVELCB3aGljaCBpcyBvbmx5IGF2YWlsYWJsZSBmb3JcbiAgICogZm9yIHRoaXMgdHlwZSBvZiBjb21wdXRlIHJlc291cmNlcyBhbmQgd2lsbCBzZWxlY3QgYW4gYWRkaXRpb25hbCBpbnN0YW5jZSB0eXBlIHRoYXQgaXMgbGFyZ2UgZW5vdWdoXG4gICAqIHRvIG1lZXQgdGhlIHJlcXVpcmVtZW50cyBvZiB0aGUgam9icyBpbiB0aGUgcXVldWUsIHdpdGggYSBwcmVmZXJlbmNlIGZvciBhbiBpbnN0YW5jZSB0eXBlIHRoYXQgaXNcbiAgICogbGVzcyBsaWtlbHkgdG8gYmUgaW50ZXJydXB0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IEFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVFxuICAgKi9cbiAgcmVhZG9ubHkgYWxsb2NhdGlvblN0cmF0ZWd5PzogQWxsb2NhdGlvblN0cmF0ZWd5O1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIEVDUyBpbnN0YW5jZSBwcm9maWxlIGFwcGxpZWQgdG8gQW1hem9uIEVDMiBpbnN0YW5jZXMgaW4gYSBjb21wdXRlIGVudmlyb25tZW50LiBZb3UgY2FuIHNwZWNpZnlcbiAgICogdGhlIHNob3J0IG5hbWUgb3IgZnVsbCBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiBhbiBpbnN0YW5jZSBwcm9maWxlLiBGb3IgZXhhbXBsZSwgZWNzSW5zdGFuY2VSb2xlIG9yXG4gICAqIGFybjphd3M6aWFtOjo8YXdzX2FjY291bnRfaWQ+Omluc3RhbmNlLXByb2ZpbGUvZWNzSW5zdGFuY2VSb2xlIC4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBBbWF6b24gRUNTXG4gICAqIEluc3RhbmNlIFJvbGUgaW4gdGhlIEFXUyBCYXRjaCBVc2VyIEd1aWRlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9pbnN0YW5jZV9JQU1fcm9sZS5odG1sXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVJvbGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIGxhdW5jaCB0ZW1wbGF0ZSB0byBhc3NvY2lhdGUgd2l0aCB5b3VyIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIFJFQURNRSBmaWxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGN1c3RvbSBsYXVuY2ggdGVtcGxhdGUgd2lsbCBiZSB1c2VkXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL2xhdW5jaC10ZW1wbGF0ZXMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgbGF1bmNoVGVtcGxhdGU/OiBMYXVuY2hUZW1wbGF0ZVNwZWNpZmljYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSB0eXBlcyBvZiBFQzIgaW5zdGFuY2VzIHRoYXQgbWF5IGJlIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LiBZb3UgY2FuIHNwZWNpZnkgaW5zdGFuY2VcbiAgICogZmFtaWxpZXMgdG8gbGF1bmNoIGFueSBpbnN0YW5jZSB0eXBlIHdpdGhpbiB0aG9zZSBmYW1pbGllcyAoZm9yIGV4YW1wbGUsIGM0IG9yIHAzKSwgb3IgeW91IGNhbiBzcGVjaWZ5XG4gICAqIHNwZWNpZmljIHNpemVzIHdpdGhpbiBhIGZhbWlseSAoc3VjaCBhcyBjNC44eGxhcmdlKS4gWW91IGNhbiBhbHNvIGNob29zZSBvcHRpbWFsIHRvIHBpY2sgaW5zdGFuY2UgdHlwZXNcbiAgICogKGZyb20gdGhlIEMsIE0sIGFuZCBSIGluc3RhbmNlIGZhbWlsaWVzKSBvbiB0aGUgZmx5IHRoYXQgbWF0Y2ggdGhlIGRlbWFuZCBvZiB5b3VyIGpvYiBxdWV1ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG9wdGltYWxcbiAgICovXG4gIHJlYWRvbmx5IGluc3RhbmNlVHlwZXM/OiBlYzIuSW5zdGFuY2VUeXBlW107XG5cbiAgLyoqXG4gICAqIFRoZSBFQzIgc2VjdXJpdHkgZ3JvdXAocykgYXNzb2NpYXRlZCB3aXRoIGluc3RhbmNlcyBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBV1MgZGVmYXVsdCBzZWN1cml0eSBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBzPzogZWMyLklTZWN1cml0eUdyb3VwW107XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgbmV0d29yayB0aGF0IGFsbCBjb21wdXRlIHJlc291cmNlcyB3aWxsIGJlIGNvbm5lY3RlZCB0by5cbiAgICovXG4gIHJlYWRvbmx5IHZwYzogZWMyLklWcGM7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgc3VibmV0cyBpbnRvIHdoaWNoIHRoZSBjb21wdXRlIHJlc291cmNlcyBhcmUgbGF1bmNoZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcHJpdmF0ZSBzdWJuZXRzIG9mIHRoZSBzdXBwbGllZCBWUEMuXG4gICAqL1xuICByZWFkb25seSB2cGNTdWJuZXRzPzogZWMyLlN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVGhlIHR5cGUgb2YgY29tcHV0ZSBlbnZpcm9ubWVudDogT05fREVNQU5EIG9yIFNQT1QuXG4gICAqXG4gICAqIEBkZWZhdWx0IE9OX0RFTUFORFxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZT86IENvbXB1dGVSZXNvdXJjZVR5cGU7XG5cbiAgLyoqXG4gICAqIFRoaXMgcHJvcGVydHkgd2lsbCBiZSBpZ25vcmVkIGlmIHlvdSBzZXQgdGhlIGVudmlyb25tZW50IHR5cGUgdG8gT05fREVNQU5ELlxuICAgKlxuICAgKiBUaGUgbWF4aW11bSBwZXJjZW50YWdlIHRoYXQgYSBTcG90IEluc3RhbmNlIHByaWNlIGNhbiBiZSB3aGVuIGNvbXBhcmVkIHdpdGggdGhlIE9uLURlbWFuZCBwcmljZSBmb3JcbiAgICogdGhhdCBpbnN0YW5jZSB0eXBlIGJlZm9yZSBpbnN0YW5jZXMgYXJlIGxhdW5jaGVkLiBGb3IgZXhhbXBsZSwgaWYgeW91ciBtYXhpbXVtIHBlcmNlbnRhZ2UgaXMgMjAlLFxuICAgKiB0aGVuIHRoZSBTcG90IHByaWNlIG11c3QgYmUgYmVsb3cgMjAlIG9mIHRoZSBjdXJyZW50IE9uLURlbWFuZCBwcmljZSBmb3IgdGhhdCBFQzIgaW5zdGFuY2UuIFlvdSBhbHdheXNcbiAgICogcGF5IHRoZSBsb3dlc3QgKG1hcmtldCkgcHJpY2UgYW5kIG5ldmVyIG1vcmUgdGhhbiB5b3VyIG1heGltdW0gcGVyY2VudGFnZS4gSWYgeW91IGxlYXZlIHRoaXMgZmllbGQgZW1wdHksXG4gICAqIHRoZSBkZWZhdWx0IHZhbHVlIGlzIDEwMCUgb2YgdGhlIE9uLURlbWFuZCBwcmljZS5cbiAgICpcbiAgICogQGRlZmF1bHQgMTAwXG4gICAqL1xuICByZWFkb25seSBiaWRQZXJjZW50YWdlPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgZGVzaXJlZCBudW1iZXIgb2YgRUMyIHZDUFVTIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGRlc2lyZWQgdmNwdSB2YWx1ZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBkZXNpcmVkdkNwdXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIG51bWJlciBvZiBFQzIgdkNQVXMgdGhhdCBhbiBlbnZpcm9ubWVudCBjYW4gcmVhY2guIEVhY2ggdkNQVSBpcyBlcXVpdmFsZW50IHRvXG4gICAqIDEsMDI0IENQVSBzaGFyZXMuIFlvdSBtdXN0IHNwZWNpZnkgYXQgbGVhc3Qgb25lIHZDUFUuXG4gICAqXG4gICAqIEBkZWZhdWx0IDI1NlxuICAgKi9cbiAgcmVhZG9ubHkgbWF4dkNwdXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIG51bWJlciBvZiBFQzIgdkNQVXMgdGhhdCBhbiBlbnZpcm9ubWVudCBzaG91bGQgbWFpbnRhaW4gKGV2ZW4gaWYgdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQgc3RhdGUgaXMgRElTQUJMRUQpLlxuICAgKiBFYWNoIHZDUFUgaXMgZXF1aXZhbGVudCB0byAxLDAyNCBDUFUgc2hhcmVzLiBCeSBrZWVwaW5nIHRoaXMgc2V0IHRvIDAgeW91IHdpbGwgbm90IGhhdmUgaW5zdGFuY2UgdGltZSB3YXN0ZWQgd2hlblxuICAgKiB0aGVyZSBpcyBubyB3b3JrIHRvIGJlIHJ1bi4gSWYgeW91IHNldCB0aGlzIGFib3ZlIHplcm8geW91IHdpbGwgbWFpbnRhaW4gdGhhdCBudW1iZXIgb2YgdkNQVXMgYXQgYWxsIHRpbWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAwXG4gICAqL1xuICByZWFkb25seSBtaW52Q3B1cz86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIEVDMiBrZXkgcGFpciB0aGF0IGlzIHVzZWQgZm9yIGluc3RhbmNlcyBsYXVuY2hlZCBpbiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICogSWYgbm8ga2V5IGlzIGRlZmluZWQsIHRoZW4gU1NIIGFjY2VzcyBpcyBub3QgYWxsb3dlZCB0byBwcm92aXNpb25lZCBjb21wdXRlIHJlc291cmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBTU0ggYWNjZXNzIHdpbGwgYmUgcG9zc2libGUuXG4gICAqL1xuICByZWFkb25seSBlYzJLZXlQYWlyPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIE1hY2hpbmUgSW1hZ2UgKEFNSSkgSUQgdXNlZCBmb3IgaW5zdGFuY2VzIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGltYWdlIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IGltYWdlPzogZWMyLklNYWNoaW5lSW1hZ2U7XG5cbiAgLyoqXG4gICAqIFRoaXMgcHJvcGVydHkgd2lsbCBiZSBpZ25vcmVkIGlmIHlvdSBzZXQgdGhlIGVudmlyb25tZW50IHR5cGUgdG8gT05fREVNQU5ELlxuICAgKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgb2YgdGhlIEFtYXpvbiBFQzIgU3BvdCBGbGVldCBJQU0gcm9sZSBhcHBsaWVkIHRvIGEgU1BPVCBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgc2VlIEFtYXpvbiBFQzIgU3BvdCBGbGVldCBSb2xlIGluIHRoZSBBV1MgQmF0Y2ggVXNlciBHdWlkZS5cbiAgICpcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC91c2VyZ3VpZGUvc3BvdF9mbGVldF9JQU1fcm9sZS5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm8gZmxlZXQgcm9sZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBzcG90RmxlZXRSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBLZXktdmFsdWUgcGFpciB0YWdzIHRvIGJlIGFwcGxpZWQgdG8gcmVzb3VyY2VzIHRoYXQgYXJlIGxhdW5jaGVkIGluIHRoZSBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKiBGb3IgQVdTIEJhdGNoLCB0aGVzZSB0YWtlIHRoZSBmb3JtIG9mIFwiU3RyaW5nMVwiOiBcIlN0cmluZzJcIiwgd2hlcmUgU3RyaW5nMSBpcyB0aGUgdGFnIGtleSBhbmRcbiAgICogU3RyaW5nMiBpcyB0aGUgdGFnIHZhbHVl4oCUZm9yIGV4YW1wbGUsIHsgXCJOYW1lXCI6IFwiQVdTIEJhdGNoIEluc3RhbmNlIC0gQzRPbkRlbWFuZFwiIH0uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gdGFncyB3aWxsIGJlIGFzc2lnbmVkIG9uIGNvbXB1dGUgcmVzb3VyY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZVJlc291cmNlc1RhZ3M/OiBUYWc7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgY3JlYXRpbmcgYSBuZXcgQ29tcHV0ZSBFbnZpcm9ubWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbXB1dGVFbnZpcm9ubWVudFByb3BzIHtcbiAgLyoqXG4gICAqIEEgbmFtZSBmb3IgdGhlIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIFVwIHRvIDEyOCBsZXR0ZXJzICh1cHBlcmNhc2UgYW5kIGxvd2VyY2FzZSksIG51bWJlcnMsIGh5cGhlbnMsIGFuZCB1bmRlcnNjb3JlcyBhcmUgYWxsb3dlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDbG91ZEZvcm1hdGlvbi1nZW5lcmF0ZWQgbmFtZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRldGFpbHMgb2YgdGhlIHJlcXVpcmVkIGNvbXB1dGUgcmVzb3VyY2VzIGZvciB0aGUgbWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBJZiBzcGVjaWZpZWQsIGFuZCB0aGlzIGlzIGFuIHVubWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50LCB3aWxsIHRocm93IGFuIGVycm9yLlxuICAgKlxuICAgKiBCeSBkZWZhdWx0LCBBV1MgQmF0Y2ggbWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50cyB1c2UgYSByZWNlbnQsIGFwcHJvdmVkIHZlcnNpb24gb2YgdGhlXG4gICAqIEFtYXpvbiBFQ1Mtb3B0aW1pemVkIEFNSSBmb3IgY29tcHV0ZSByZXNvdXJjZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQ2xvdWRGb3JtYXRpb24gZGVmYXVsdHNcbiAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtYmF0Y2gtY29tcHV0ZWVudmlyb25tZW50LWNvbXB1dGVyZXNvdXJjZXMuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZVJlc291cmNlcz86IENvbXB1dGVSZXNvdXJjZXM7XG5cbiAgLyoqXG4gICAqIFRoZSBzdGF0ZSBvZiB0aGUgY29tcHV0ZSBlbnZpcm9ubWVudC4gSWYgdGhlIHN0YXRlIGlzIHNldCB0byB0cnVlLCB0aGVuIHRoZSBjb21wdXRlXG4gICAqIGVudmlyb25tZW50IGFjY2VwdHMgam9icyBmcm9tIGEgcXVldWUgYW5kIGNhbiBzY2FsZSBvdXQgYXV0b21hdGljYWxseSBiYXNlZCBvbiBxdWV1ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgdXNlZCBieSBCYXRjaCB0byBtYWtlIGNhbGxzIHRvIG90aGVyIEFXUyBzZXJ2aWNlcyBvbiB5b3VyIGJlaGFsZiBmb3IgbWFuYWdpbmdcbiAgICogdGhlIHJlc291cmNlcyB0aGF0IHlvdSB1c2Ugd2l0aCB0aGUgc2VydmljZS4gQnkgZGVmYXVsdCwgdGhpcyByb2xlIGlzIGNyZWF0ZWQgZm9yIHlvdSB1c2luZ1xuICAgKiB0aGUgQVdTIG1hbmFnZWQgc2VydmljZSBwb2xpY3kgZm9yIEJhdGNoLlxuICAgKlxuICAgKiBAbGluayBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9zZXJ2aWNlX0lBTV9yb2xlLmh0bWxcbiAgICpcbiAgICogQGRlZmF1bHQgLSBSb2xlIHVzaW5nIHRoZSAnc2VydmljZS1yb2xlL0FXU0JhdGNoU2VydmljZVJvbGUnIHBvbGljeS5cbiAgICovXG4gIHJlYWRvbmx5IHNlcnZpY2VSb2xlPzogaWFtLklSb2xlLFxuXG4gIC8qKlxuICAgKiBEZXRlcm1pbmVzIGlmIEFXUyBzaG91bGQgbWFuYWdlIHRoZSBhbGxvY2F0aW9uIG9mIGNvbXB1dGUgcmVzb3VyY2VzIGZvciBwcm9jZXNzaW5nIGpvYnMuXG4gICAqIElmIHNldCB0byBmYWxzZSwgdGhlbiB5b3UgYXJlIGluIGNoYXJnZSBvZiBwcm92aWRpbmcgdGhlIGNvbXB1dGUgcmVzb3VyY2UgZGV0YWlscy5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgbWFuYWdlZD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBvZiBhIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUNvbXB1dGVFbnZpcm9ubWVudCBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50QXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIEJhdGNoIENvbXB1dGUgRW52aXJvbm1lbnQuXG4gKlxuICogRGVmaW5lcyBhIGJhdGNoIGNvbXB1dGUgZW52aXJvbm1lbnQgdG8gcnVuIGJhdGNoIGpvYnMgb24uXG4gKi9cbmV4cG9ydCBjbGFzcyBDb21wdXRlRW52aXJvbm1lbnQgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDb21wdXRlRW52aXJvbm1lbnQge1xuICAvKipcbiAgICogRmV0Y2hlcyBhbiBleGlzdGluZyBiYXRjaCBjb21wdXRlIGVudmlyb25tZW50IGJ5IGl0cyBhbWF6b24gcmVzb3VyY2UgbmFtZS5cbiAgICpcbiAgICogQHBhcmFtIHNjb3BlXG4gICAqIEBwYXJhbSBpZFxuICAgKiBAcGFyYW0gY29tcHV0ZUVudmlyb25tZW50QXJuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Db21wdXRlRW52aXJvbm1lbnRBcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgY29tcHV0ZUVudmlyb25tZW50QXJuOiBzdHJpbmcpOiBJQ29tcHV0ZUVudmlyb25tZW50IHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCBjb21wdXRlRW52aXJvbm1lbnROYW1lID0gc3RhY2sucGFyc2VBcm4oY29tcHV0ZUVudmlyb25tZW50QXJuKS5yZXNvdXJjZU5hbWUhO1xuXG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJQ29tcHV0ZUVudmlyb25tZW50IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnRBcm4gPSBjb21wdXRlRW52aXJvbm1lbnRBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY29tcHV0ZUVudmlyb25tZW50TmFtZSA9IGNvbXB1dGVFbnZpcm9ubWVudE5hbWU7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNvbXB1dGVFbnZpcm9ubWVudEFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGlzIGNvbXB1dGUgZW52aXJvbm1lbnQuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjb21wdXRlRW52aXJvbm1lbnROYW1lOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzID0geyBlbmFibGVkOiB0cnVlLCBtYW5hZ2VkOiB0cnVlIH0pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuY29tcHV0ZUVudmlyb25tZW50TmFtZSxcbiAgICB9KTtcblxuICAgIHRoaXMudmFsaWRhdGVQcm9wcyhwcm9wcyk7XG5cbiAgICBjb25zdCBzcG90RmxlZXRSb2xlID0gdGhpcy5nZXRTcG90RmxlZXRSb2xlKHByb3BzKTtcbiAgICBsZXQgY29tcHV0ZVJlc291cmNlczogQ2ZuQ29tcHV0ZUVudmlyb25tZW50LkNvbXB1dGVSZXNvdXJjZXNQcm9wZXJ0eSB8IHVuZGVmaW5lZDtcblxuICAgIC8vIE9ubHkgYWxsb3cgY29tcHV0ZSByZXNvdXJjZXMgdG8gYmUgc2V0IHdoZW4gdXNpbmcgTUFOQUdFRCB0eXBlXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgJiYgdGhpcy5pc01hbmFnZWQocHJvcHMpKSB7XG4gICAgICBjb21wdXRlUmVzb3VyY2VzID0ge1xuICAgICAgICBhbGxvY2F0aW9uU3RyYXRlZ3k6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5XG4gICAgICAgICB8fCAoXG4gICAgICAgICAgIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudHlwZSA9PT0gQ29tcHV0ZVJlc291cmNlVHlwZS5TUE9UXG4gICAgICAgICAgICAgPyBBbGxvY2F0aW9uU3RyYXRlZ3kuU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRURcbiAgICAgICAgICAgICA6IEFsbG9jYXRpb25TdHJhdGVneS5CRVNUX0ZJVFxuICAgICAgICAgKSxcbiAgICAgICAgYmlkUGVyY2VudGFnZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlLFxuICAgICAgICBkZXNpcmVkdkNwdXM6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuZGVzaXJlZHZDcHVzLFxuICAgICAgICBlYzJLZXlQYWlyOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmVjMktleVBhaXIsXG4gICAgICAgIGltYWdlSWQ6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW1hZ2UgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy5pbWFnZS5nZXRJbWFnZSh0aGlzKS5pbWFnZUlkLFxuICAgICAgICBpbnN0YW5jZVJvbGU6IHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VSb2xlXG4gICAgICAgICAgPyBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmluc3RhbmNlUm9sZVxuICAgICAgICAgIDogbmV3IGlhbS5DZm5JbnN0YW5jZVByb2ZpbGUodGhpcywgJ0luc3RhbmNlLVByb2ZpbGUnLCB7XG4gICAgICAgICAgICByb2xlczogW25ldyBpYW0uUm9sZSh0aGlzLCAnRWNzLUluc3RhbmNlLVJvbGUnLCB7XG4gICAgICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdlYzIuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICAgICAgICBpYW0uTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ3NlcnZpY2Utcm9sZS9BbWF6b25FQzJDb250YWluZXJTZXJ2aWNlZm9yRUMyUm9sZScpLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSkucm9sZU5hbWVdLFxuICAgICAgICAgIH0pLmF0dHJBcm4sXG4gICAgICAgIGluc3RhbmNlVHlwZXM6IHRoaXMuYnVpbGRJbnN0YW5jZVR5cGVzKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuaW5zdGFuY2VUeXBlcyksXG4gICAgICAgIGxhdW5jaFRlbXBsYXRlOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmxhdW5jaFRlbXBsYXRlLFxuICAgICAgICBtYXh2Q3B1czogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5tYXh2Q3B1cyB8fCAyNTYsXG4gICAgICAgIG1pbnZDcHVzOiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzIHx8IDAsXG4gICAgICAgIHNlY3VyaXR5R3JvdXBJZHM6IHRoaXMuYnVpbGRTZWN1cml0eUdyb3VwSWRzKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudnBjLCBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnNlY3VyaXR5R3JvdXBzKSxcbiAgICAgICAgc3BvdElhbUZsZWV0Um9sZTogc3BvdEZsZWV0Um9sZSA/IHNwb3RGbGVldFJvbGUucm9sZUFybiA6IHVuZGVmaW5lZCxcbiAgICAgICAgc3VibmV0czogcHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMuc2VsZWN0U3VibmV0cyhwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnZwY1N1Ym5ldHMpLnN1Ym5ldElkcyxcbiAgICAgICAgdGFnczogcHJvcHMuY29tcHV0ZVJlc291cmNlcy5jb21wdXRlUmVzb3VyY2VzVGFncyxcbiAgICAgICAgdHlwZTogcHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlIHx8IENvbXB1dGVSZXNvdXJjZVR5cGUuT05fREVNQU5ELFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBjb25zdCBjb21wdXRlRW52aXJvbm1lbnQgPSBuZXcgQ2ZuQ29tcHV0ZUVudmlyb25tZW50KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGNvbXB1dGVFbnZpcm9ubWVudE5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgY29tcHV0ZVJlc291cmNlcyxcbiAgICAgIHNlcnZpY2VSb2xlOiBwcm9wcy5zZXJ2aWNlUm9sZVxuICAgICAgICA/IHByb3BzLnNlcnZpY2VSb2xlLnJvbGVBcm5cbiAgICAgICAgOiBuZXcgaWFtLlJvbGUodGhpcywgJ1Jlc291cmNlLVNlcnZpY2UtSW5zdGFuY2UtUm9sZScsIHtcbiAgICAgICAgICBtYW5hZ2VkUG9saWNpZXM6IFtcbiAgICAgICAgICAgIGlhbS5NYW5hZ2VkUG9saWN5LmZyb21Bd3NNYW5hZ2VkUG9saWN5TmFtZSgnc2VydmljZS1yb2xlL0FXU0JhdGNoU2VydmljZVJvbGUnKSxcbiAgICAgICAgICBdLFxuICAgICAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdiYXRjaC5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIH0pLnJvbGVBcm4sXG4gICAgICBzdGF0ZTogcHJvcHMuZW5hYmxlZCA9PT0gdW5kZWZpbmVkID8gJ0VOQUJMRUQnIDogKHByb3BzLmVuYWJsZWQgPyAnRU5BQkxFRCcgOiAnRElTQUJMRUQnKSxcbiAgICAgIHR5cGU6IHRoaXMuaXNNYW5hZ2VkKHByb3BzKSA/ICdNQU5BR0VEJyA6ICdVTk1BTkFHRUQnLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgJiYgcHJvcHMuY29tcHV0ZVJlc291cmNlcy52cGMpIHtcbiAgICAgIHRoaXMubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLmNvbXB1dGVSZXNvdXJjZXMudnBjKTtcbiAgICB9XG5cbiAgICB0aGlzLmNvbXB1dGVFbnZpcm9ubWVudEFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoY29tcHV0ZUVudmlyb25tZW50LnJlZiwge1xuICAgICAgc2VydmljZTogJ2JhdGNoJyxcbiAgICAgIHJlc291cmNlOiAnY29tcHV0ZS1lbnZpcm9ubWVudCcsXG4gICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgIH0pO1xuICAgIHRoaXMuY29tcHV0ZUVudmlyb25tZW50TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKGNvbXB1dGVFbnZpcm9ubWVudC5yZWYpO1xuICB9XG5cbiAgcHJpdmF0ZSBpc01hbmFnZWQocHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHByb3BzLm1hbmFnZWQgPT09IHVuZGVmaW5lZCA/IHRydWUgOiBwcm9wcy5tYW5hZ2VkO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlcyB0aGUgcHJvcGVydGllcyBwcm92aWRlZCBmb3IgYSBuZXcgYmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyhwcm9wczogQ29tcHV0ZUVudmlyb25tZW50UHJvcHMpIHtcbiAgICBpZiAocHJvcHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIGlmICghdGhpcy5pc01hbmFnZWQocHJvcHMpICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJdCBpcyBub3QgYWxsb3dlZCB0byBzZXQgY29tcHV0ZVJlc291cmNlcyBvbiBhbiBBV1MgdW5tYW5hZ2VkIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5pc01hbmFnZWQocHJvcHMpICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdjb21wdXRlUmVzb3VyY2VzIGlzIG1pc3NpbmcgYnV0IHJlcXVpcmVkIG9uIGEgbWFuYWdlZCBjb21wdXRlIGVudmlyb25tZW50Jyk7XG4gICAgfVxuXG4gICAgLy8gU2V0dGluZyBhIGJpZCBwZXJjZW50YWdlIGlzIG9ubHkgYWxsb3dlZCBvbiBTUE9UIHJlc291cmNlcyArXG4gICAgLy8gQ2Fubm90IHVzZSBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCB3aGVuIHVzaW5nIE9OX0RFTUFORFxuICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzKSB7XG4gICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLk9OX0RFTUFORCkge1xuICAgICAgICAvLyBWQUxJREFURSBGT1IgT05fREVNQU5EXG5cbiAgICAgICAgLy8gQmlkIHBlcmNlbnRhZ2UgaXMgbm90IGFsbG93ZWRcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYmlkUGVyY2VudGFnZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdTZXR0aW5nIHRoZSBiaWQgcGVyY2VudGFnZSBpcyBvbmx5IGFsbG93ZWQgZm9yIFNQT1QgdHlwZSByZXNvdXJjZXMgb24gYSBiYXRjaCBjb21wdXRlIGVudmlyb25tZW50Jyk7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBTUE9UX0NBUEFDSVRZX09QVElNSVpFRCBhbGxvY2F0aW9uIGlzIG5vdCBhbGxvd2VkXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmFsbG9jYXRpb25TdHJhdGVneSAmJiBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmFsbG9jYXRpb25TdHJhdGVneSA9PT0gQWxsb2NhdGlvblN0cmF0ZWd5LlNQT1RfQ0FQQUNJVFlfT1BUSU1JWkVEKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgU1BPVF9DQVBBQ0lUWV9PUFRJTUlaRUQgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBvbmx5IGFsbG93ZWQgaWYgdGhlIGVudmlyb25tZW50IGlzIGEgU1BPVCB0eXBlIGNvbXB1dGUgZW52aXJvbm1lbnQnKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gVkFMSURBVEUgRk9SIFNQT1RcblxuICAgICAgICAvLyBCaWQgcGVyY2VudGFnZSBtdXN0IGJlIGZyb20gMCAtIDEwMFxuICAgICAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlICE9PSB1bmRlZmluZWQgJiZcbiAgICAgICAgICAocHJvcHMuY29tcHV0ZVJlc291cmNlcy5iaWRQZXJjZW50YWdlIDwgMCB8fCBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLmJpZFBlcmNlbnRhZ2UgPiAxMDApKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdCaWQgcGVyY2VudGFnZSBjYW4gb25seSBiZSBhIHZhbHVlIGJldHdlZW4gMCBhbmQgMTAwJyk7XG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWludkNwdXMpIHtcbiAgICAgICAgLy8gbWludkNwdXMgY2Fubm90IGJlIGxlc3MgdGhhbiAwXG4gICAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzIDwgMCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWluaW11bSB2Q3B1cyBmb3IgYSBiYXRjaCBjb21wdXRlIGVudmlyb25tZW50IGNhbm5vdCBiZSBsZXNzIHRoYW4gMCcpO1xuICAgICAgICB9XG5cbiAgICAgICAgLy8gbWludkNwdXMgY2Fubm90IGV4Y2VlZCBtYXggdkNwdXNcbiAgICAgICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMubWF4dkNwdXMgJiZcbiAgICAgICAgICBwcm9wcy5jb21wdXRlUmVzb3VyY2VzLm1pbnZDcHVzID4gcHJvcHMuY29tcHV0ZVJlc291cmNlcy5tYXh2Q3B1cykge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTWluaW11bSB2Q3B1cyBjYW5ub3QgYmUgZ3JlYXRlciB0aGFuIHRoZSBtYXhpbXVtIHZDcHVzJyk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkSW5zdGFuY2VUeXBlcyhpbnN0YW5jZVR5cGVzPzogZWMyLkluc3RhbmNlVHlwZVtdKTogc3RyaW5nW10ge1xuICAgIGlmIChpbnN0YW5jZVR5cGVzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiBbXG4gICAgICAgICdvcHRpbWFsJyxcbiAgICAgIF07XG4gICAgfVxuXG4gICAgcmV0dXJuIGluc3RhbmNlVHlwZXMubWFwKCh0eXBlOiBlYzIuSW5zdGFuY2VUeXBlKSA9PiB0eXBlLnRvU3RyaW5nKCkpO1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZFNlY3VyaXR5R3JvdXBJZHModnBjOiBlYzIuSVZwYywgc2VjdXJpdHlHcm91cHM/OiBlYzIuSVNlY3VyaXR5R3JvdXBbXSk6IHN0cmluZ1tdIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoc2VjdXJpdHlHcm91cHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdSZXNvdXJjZS1TZWN1cml0eS1Hcm91cCcsIHsgdnBjIH0pLnNlY3VyaXR5R3JvdXBJZCxcbiAgICAgIF07XG4gICAgfVxuXG4gICAgcmV0dXJuIHNlY3VyaXR5R3JvdXBzLm1hcCgoZ3JvdXA6IGVjMi5JU2VjdXJpdHlHcm91cCkgPT4gZ3JvdXAuc2VjdXJpdHlHcm91cElkKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZXMgYW4gQVdTIElBTSByb2xlIGZvciBwcm92aXNpb25pbmcgc3BvdGZsZWV0IHJlc291cmNlc1xuICAgKiBpZiB0aGUgYWxsb2NhdGlvbiBzdHJhdGVneSBpcyBzZXQgdG8gQkVTVF9GSVQgb3Igbm90IGRlZmluZWQuXG4gICAqXG4gICAqIEBwYXJhbSBwcm9wcyAtIHRoZSBjb21wdXRlIGVudmlyb25tZW50IGNvbnN0cnVjdCBwcm9wZXJ0aWVzXG4gICAqL1xuICBwcml2YXRlIGdldFNwb3RGbGVldFJvbGUocHJvcHM6IENvbXB1dGVFbnZpcm9ubWVudFByb3BzKTogaWFtLklSb2xlIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcz8uYWxsb2NhdGlvblN0cmF0ZWd5ICYmIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuYWxsb2NhdGlvblN0cmF0ZWd5ICE9PSBBbGxvY2F0aW9uU3RyYXRlZ3kuQkVTVF9GSVQpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmNvbXB1dGVSZXNvdXJjZXMpIHtcbiAgICAgIGlmIChwcm9wcy5jb21wdXRlUmVzb3VyY2VzLnNwb3RGbGVldFJvbGUpIHtcbiAgICAgICAgcmV0dXJuIHByb3BzLmNvbXB1dGVSZXNvdXJjZXMuc3BvdEZsZWV0Um9sZTtcbiAgICAgIH0gZWxzZSBpZiAocHJvcHMuY29tcHV0ZVJlc291cmNlcy50eXBlID09PSBDb21wdXRlUmVzb3VyY2VUeXBlLlNQT1QpIHtcbiAgICAgICAgcmV0dXJuIGlhbS5Sb2xlLmZyb21Sb2xlQXJuKHRoaXMsICdSZXNvdXJjZS1TcG90RmxlZXQtUm9sZScsXG4gICAgICAgICAgYGFybjoke3RoaXMuc3RhY2sucGFydGl0aW9ufTppYW06OiR7dGhpcy5zdGFjay5hY2NvdW50fTpyb2xlL2F3cy1zZXJ2aWNlLXJvbGUvc3BvdGZsZWV0LmFtYXpvbmF3cy5jb20vQVdTU2VydmljZVJvbGVGb3JFQzJTcG90RmxlZXRgKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG59XG4iXX0=