"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const string_variable_1 = require("../interface/variables/string-variable");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.documentName = props.documentName ?? id;
        this.description = props.description ?? this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole && string_variable_1.StringVariable.of(props.assumeRole).toString() || undefined;
        this.docOutputs = props.docOutputs ?? [];
        this.docInputs = props.docInputs ?? [];
        if (this.assumeRole && !this.docInputs.map(i => i.name).includes(this.assumeRole)) {
            throw new Error('Assume rold specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        new aws_ssm_1.CfnDocument(this, 'Resource', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            ssmInputs[inp.name] = inp.toSsm();
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(this.buildSsmDocument());
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(this.buildSsmDocument());
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "@cdklabs/cdk-ssm-documents.SsmDocument", version: "0.0.16" };
//# sourceMappingURL=data:application/json;base64,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