"use strict";
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'add stacked property to graphs'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'Test widget',
            stacked: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'Test widget',
                    region: { Ref: 'AWS::Region' },
                    stacked: true,
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add metrics to graphs on either axis'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' }),
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                        ['CDK', 'Tast', { yAxis: 'right' }],
                    ],
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'label and color are respected in constructor'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [new lib_1.Metric({ namespace: 'CDK', metricName: 'Test', label: 'MyMetric', color: '000000' })],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { label: 'MyMetric', color: '000000' }],
                    ],
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'singlevalue widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                },
            }]);
        test.done();
    },
    'query result widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const logGroup = { logGroupName: 'my-log-group' };
        // WHEN
        const widget = new lib_1.LogQueryWidget({
            logGroupNames: [logGroup.logGroupName],
            queryLines: [
                'fields @message',
                'filter @message like /Error/',
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'log',
                width: 6,
                height: 6,
                properties: {
                    view: 'table',
                    region: { Ref: 'AWS::Region' },
                    query: `SOURCE '${logGroup.logGroupName}' | fields @message\n| filter @message like /Error/`,
                },
            }]);
        test.done();
    },
    'query result widget - bar'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const logGroup = { logGroupName: 'my-log-group' };
        // WHEN
        const widget = new lib_1.LogQueryWidget({
            logGroupNames: [logGroup.logGroupName],
            view: lib_1.LogQueryVisualizationType.BAR,
            queryLines: [
                'fields @message',
                'filter @message like /Error/',
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'log',
                width: 6,
                height: 6,
                properties: {
                    view: 'bar',
                    region: { Ref: 'AWS::Region' },
                    query: `SOURCE '${logGroup.logGroupName}' | fields @message\n| filter @message like /Error/`,
                },
            }]);
        test.done();
    },
    'query result widget - pie'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const logGroup = { logGroupName: 'my-log-group' };
        // WHEN
        const widget = new lib_1.LogQueryWidget({
            logGroupNames: [logGroup.logGroupName],
            view: lib_1.LogQueryVisualizationType.PIE,
            queryLines: [
                'fields @message',
                'filter @message like /Error/',
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'log',
                width: 6,
                height: 6,
                properties: {
                    view: 'pie',
                    region: { Ref: 'AWS::Region' },
                    query: `SOURCE '${logGroup.logGroupName}' | fields @message\n| filter @message like /Error/`,
                },
            }]);
        test.done();
    },
    'query result widget - line'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const logGroup = { logGroupName: 'my-log-group' };
        // WHEN
        const widget = new lib_1.LogQueryWidget({
            logGroupNames: [logGroup.logGroupName],
            view: lib_1.LogQueryVisualizationType.LINE,
            queryLines: [
                'fields @message',
                'filter @message like /Error/',
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'log',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    stacked: false,
                    region: { Ref: 'AWS::Region' },
                    query: `SOURCE '${logGroup.logGroupName}' | fields @message\n| filter @message like /Error/`,
                },
            }]);
        test.done();
    },
    'query result widget - stackedarea'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const logGroup = { logGroupName: 'my-log-group' };
        // WHEN
        const widget = new lib_1.LogQueryWidget({
            logGroupNames: [logGroup.logGroupName],
            view: lib_1.LogQueryVisualizationType.STACKEDAREA,
            queryLines: [
                'fields @message',
                'filter @message like /Error/',
            ],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'log',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    stacked: true,
                    region: { Ref: 'AWS::Region' },
                    query: `SOURCE '${logGroup.logGroupName}' | fields @message\n| filter @message like /Error/`,
                },
            }]);
        test.done();
    },
    'alarm widget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const alarm = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }).createAlarm(stack, 'Alarm', {
            evaluationPeriods: 2,
            threshold: 1000,
        });
        // WHEN
        const widget = new lib_1.AlarmWidget({
            alarm,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    annotations: {
                        alarms: [{ 'Fn::GetAtt': ['Alarm7103F465', 'Arn'] }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add annotations to graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            leftAnnotations: [{
                    value: 1000,
                    color: '667788',
                    fill: lib_1.Shading.BELOW,
                    label: 'this is the annotation',
                }],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    annotations: {
                        horizontal: [{
                                yAxis: 'left',
                                value: 1000,
                                color: '667788',
                                fill: 'below',
                                label: 'this is the annotation',
                            }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'convert alarm to annotation'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        const alarm = metric.createAlarm(stack, 'Alarm', {
            evaluationPeriods: 7,
            datapointsToAlarm: 2,
            threshold: 1000,
        });
        // WHEN
        const widget = new lib_1.GraphWidget({
            right: [metric],
            rightAnnotations: [alarm.toAnnotation()],
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test', { yAxis: 'right' }],
                    ],
                    annotations: {
                        horizontal: [{
                                yAxis: 'right',
                                value: 1000,
                                label: 'Test >= 1000 for 2 datapoints within 35 minutes',
                            }],
                    },
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'add yAxis to graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My fancy graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            right: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Tast' }),
            ],
            leftYAxis: ({
                label: 'Left yAxis',
                max: 100,
            }),
            rightYAxis: ({
                label: 'Right yAxis',
                min: 10,
                showUnits: false,
            }),
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My fancy graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                        ['CDK', 'Tast', { yAxis: 'right' }],
                    ],
                    yAxis: {
                        left: { label: 'Left yAxis', max: 100 },
                        right: { label: 'Right yAxis', min: 10, showUnits: false },
                    },
                },
            }]);
        test.done();
    },
    'specify liveData property on graph'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            title: 'My live graph',
            left: [
                new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' }),
            ],
            liveData: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    title: 'My live graph',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    liveData: true,
                    yAxis: {},
                },
            }]);
        test.done();
    },
    'can use imported alarm with graph'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const alarm = lib_1.Alarm.fromAlarmArn(stack, 'Alarm', 'arn:aws:cloudwatch:region:account-id:alarm:alarm-name');
        // WHEN
        new lib_1.AlarmWidget({
            title: 'My fancy graph',
            alarm,
        });
        // THEN: Compiles
        test.done();
    },
    'add setPeriodToTimeRange to singleValueWidget'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.SingleValueWidget({
            metrics: [metric],
            setPeriodToTimeRange: true,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 3,
                properties: {
                    view: 'singleValue',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    setPeriodToTimeRange: true,
                },
            }]);
        test.done();
    },
    'allows overriding custom values of dashboard widgets'(test) {
        class HiddenMetric extends lib_1.Metric {
            toMetricConfig() {
                const ret = super.toMetricConfig();
                // @ts-ignore
                ret.renderingProperties.visible = false;
                return ret;
            }
        }
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [
                new HiddenMetric({ namespace: 'CDK', metricName: 'Test' }),
            ],
        });
        // test.ok(widget.toJson()[0].properties.metrics[0].visible === false);
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.metrics[0], ['CDK', 'Test', { visible: false }]);
        test.done();
    },
    'GraphColor is correctly converted into the correct hexcode'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const metric = new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' });
        // WHEN
        const widget = new lib_1.GraphWidget({
            left: [metric.with({
                    color: lib_1.Color.BLUE,
                })],
            leftAnnotations: [
                { color: lib_1.Color.RED, value: 100 },
            ],
        });
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.metrics[0], ['CDK', 'Test', { color: '#1f77b4' }]);
        test.deepEqual(stack.resolve(widget.toJson())[0].properties.annotations.horizontal[0], { yAxis: 'left', value: 100, color: '#d62728' });
        test.done();
    },
    'legend position is respected in constructor'(test) {
        // WHEN
        const stack = new core_1.Stack();
        const widget = new lib_1.GraphWidget({
            left: [new lib_1.Metric({ namespace: 'CDK', metricName: 'Test' })],
            legendPosition: lib_1.LegendPosition.RIGHT,
        });
        // THEN
        test.deepEqual(stack.resolve(widget.toJson()), [{
                type: 'metric',
                width: 6,
                height: 6,
                properties: {
                    view: 'timeSeries',
                    region: { Ref: 'AWS::Region' },
                    metrics: [
                        ['CDK', 'Test'],
                    ],
                    yAxis: {},
                    legend: {
                        position: 'right',
                    },
                },
            }]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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