"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Dashboard = exports.PeriodOverride = void 0;
const core_1 = require("@aws-cdk/core");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const layout_1 = require("./layout");
/**
 * Specify the period for graphs when the CloudWatch dashboard loads
 */
var PeriodOverride;
(function (PeriodOverride) {
    /**
     * Period of all graphs on the dashboard automatically adapt to the time range of the dashboard.
     */
    PeriodOverride["AUTO"] = "auto";
    /**
     * Period set for each graph will be used
     */
    PeriodOverride["INHERIT"] = "inherit";
})(PeriodOverride = exports.PeriodOverride || (exports.PeriodOverride = {}));
/**
 * A CloudWatch dashboard
 */
class Dashboard extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.dashboardName,
        });
        this.rows = [];
        {
            const { dashboardName } = props;
            if (dashboardName && !core_1.Token.isUnresolved(dashboardName) && !dashboardName.match(/^[\w-]+$/)) {
                throw new Error([
                    `The value ${dashboardName} for field dashboardName contains invalid characters.`,
                    'It can only contain alphanumerics, dash (-) and underscore (_).',
                ].join(' '));
            }
        }
        new cloudwatch_generated_1.CfnDashboard(this, 'Resource', {
            dashboardName: this.physicalName,
            dashboardBody: core_1.Lazy.stringValue({
                produce: () => {
                    const column = new layout_1.Column(...this.rows);
                    column.position(0, 0);
                    return core_1.Stack.of(this).toJsonString({
                        start: props.start,
                        end: props.end,
                        periodOverride: props.periodOverride,
                        widgets: column.toJson(),
                    });
                },
            }),
        });
        (props.widgets || []).forEach(row => {
            this.addWidgets(...row);
        });
    }
    /**
     * Add a widget to the dashboard.
     *
     * Widgets given in multiple calls to add() will be laid out stacked on
     * top of each other.
     *
     * Multiple widgets added in the same call to add() will be laid out next
     * to each other.
     */
    addWidgets(...widgets) {
        if (widgets.length === 0) {
            return;
        }
        const w = widgets.length > 1 ? new layout_1.Row(...widgets) : widgets[0];
        this.rows.push(w);
    }
}
exports.Dashboard = Dashboard;
//# sourceMappingURL=data:application/json;base64,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