"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodeType = exports.ElasticacheAutoMonitor = void 0;
const cdk = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const sns = require("@aws-cdk/aws-sns");
const sns_sub = require("@aws-cdk/aws-sns-subscriptions");
const actions = require("@aws-cdk/aws-cloudwatch-actions");
const cpu_monitor_1 = require("./cpu-monitor");
const swap_usage_monitor_1 = require("./swap-usage-monitor");
const evictions_monitor_1 = require("./evictions-monitor");
const current_connections_monitor_1 = require("./current-connections-monitor");
const freeable_memory_monitor_1 = require("./freeable-memory-monitor");
/**
 * ElasticacheAutoMonitor allows you to send email, sms, slack, or trigger aws sns topic when an alarm occurs.
 * You will get the following monitoring:
 *  1. Cpu Monitor: Should be less than 90%. (See below reference)
 *  2. SwapUsage Monitor: Should be less than 50M.
 *  3. Evictions Monitor: Should not have evictions value.
 *  4. CurrConnections Monitor: According to your business needs, default every 1 vcup is equal to 50 connections.
 *  5. FreeableMemory Monitor: Not less than 10%
 *
 * Reference: https://docs.aws.amazon.com/AmazonElastiCache/latest/red-ug/CacheMetrics.WhichShouldIMonitor.html
 */
class ElasticacheAutoMonitor extends cdk.Construct {
    static setUpWithLambda(scope, cacheClusterId, fn, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-lambda');
        topic.addSubscription(new sns_sub.LambdaSubscription(fn));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSlack(scope, cacheClusterId, props) {
        let channel = props.channel || 'cloudwatch-alarm';
        let username = props.username || 'Webhookbot';
        let icon_emoji = props.iconEmoji || ':scream:';
        const fn = new lambda.Function(scope, 'alarm-' + cacheClusterId + '-to-slack', {
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            code: lambda.Code.fromAsset('./src/asset/lambda'),
            environment: {
                SLACK_WEBHOOK_URL: props.webHookUrl,
                CHANNEL: channel,
                USERNAME: username,
                ICON_EMOJI: icon_emoji,
            },
        });
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-slack');
        topic.addSubscription(new sns_sub.LambdaSubscription(fn));
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithEmail(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-with-email-topic');
        for (let email in props.emails) {
            topic.addSubscription(new sns_sub.EmailSubscription(email));
        }
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setUpWithSms(scope, cacheClusterId, props) {
        const topic = new sns.Topic(scope, 'alarm-' + cacheClusterId + '-with-sms-topic');
        for (let phone in props.phones) {
            topic.addSubscription(new sns_sub.SmsSubscription(phone));
        }
        ElasticacheAutoMonitor.setup(scope, cacheClusterId, topic, props);
    }
    static setup(scope, cacheClusterId, topic, props) {
        let nodeType = NodeType.M5_LARGE;
        if (props && props.nodeType) {
            nodeType = props.nodeType;
        }
        let currConnectionsPeriod = cdk.Duration.minutes(1);
        if (props && props.currConnectionsPeriod) {
            currConnectionsPeriod = props.currConnectionsPeriod;
        }
        let currConnectionsThreshold = nodeType.vcupCount * 50;
        if (props && props.currConnectionsThreshold) {
            currConnectionsThreshold = props.currConnectionsThreshold;
        }
        cpu_monitor_1.CpuMonitor.setup(scope, cacheClusterId, nodeType).addAlarmAction(new actions.SnsAction(topic));
        swap_usage_monitor_1.SwapUsageMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        evictions_monitor_1.EvictionsMomiton.setup(scope, cacheClusterId).addAlarmAction(new actions.SnsAction(topic));
        current_connections_monitor_1.CurrConnectionsMomiton.setup(scope, cacheClusterId, currConnectionsPeriod, currConnectionsThreshold).addAlarmAction(new actions.SnsAction(topic));
        freeable_memory_monitor_1.FreeableMemoryMomiton.setup(scope, cacheClusterId, nodeType).addAlarmAction(new actions.SnsAction(topic));
    }
}
exports.ElasticacheAutoMonitor = ElasticacheAutoMonitor;
class NodeType {
    constructor(name, vcupCount, memory) {
        this.name = name;
        this.vcupCount = vcupCount;
        this.memory = memory;
    }
}
exports.NodeType = NodeType;
NodeType.T2_MIRCO = new NodeType('cache.t2.mirco', 1, 0.555 * 1024 * 1024 * 1024);
NodeType.T2_SMALL = new NodeType('cache.t2.small', 1, 1.55 * 1024 * 1024 * 1024);
NodeType.T2_MEDIUM = new NodeType('cache.t2.medium', 2, 3.22 * 1024 * 1024 * 1024);
NodeType.T3_MICRO = new NodeType('cache.t3.micro', 2, 0.5 * 1024 * 1024 * 1024);
NodeType.T3_SMALL = new NodeType('cache.t3.small', 2, 1.37 * 1024 * 1024 * 1024);
NodeType.T3_MEDIUM = new NodeType('cache.t3.medium', 2, 3.09 * 1024 * 1024 * 1024);
NodeType.M4_LARGE = new NodeType('cache.m4.large', 2, 6.42 * 1024 * 1024 * 1024);
NodeType.M4_XLARGE = new NodeType('cache.m4.xlarge', 4, 14.28 * 1024 * 1024 * 1024);
NodeType.M4_2XLARGE = new NodeType('cache.m4.2xlarge', 8, 29.70 * 1024 * 1024 * 1024);
NodeType.M4_4XLARGE = new NodeType('cache.m4.4xlarge', 16, 60.78 * 1024 * 1024 * 1024);
NodeType.M4_10XLARGE = new NodeType('cache.m4.10xlarge', 40, 154.64 * 1024 * 1024 * 1024);
NodeType.M5_LARGE = new NodeType('cache.m5.large', 2, 6.38 * 1024 * 1024 * 1024);
NodeType.M5_XLARGE = new NodeType('cache.m5.xlarge', 4, 12.93 * 1024 * 1024 * 1024);
NodeType.M5_2XLARGE = new NodeType('cache.m5.2xlarge', 8, 26.04 * 1024 * 1024 * 1024);
NodeType.M5_4XLARGE = new NodeType('cache.m5.4xlarge', 16, 52.26 * 1024 * 1024 * 1024);
NodeType.M5_12XLARGE = new NodeType('cache.m5.12xlarge', 48, 157.12 * 1024 * 1024 * 1024);
NodeType.M5_24XLARGE = new NodeType('cache.m5.12xlarge', 96, 314.32 * 1024 * 1024 * 1024);
NodeType.R4_LARGE = new NodeType('cache.r4.large', 2, 12.3 * 1024 * 1024 * 1024);
NodeType.R4_XLARGE = new NodeType('cache.r4.xlarge', 4, 25.05 * 1024 * 1024 * 1024);
NodeType.R4_2XLARGE = new NodeType('cache.r4.2xlarge', 8, 50.47 * 1024 * 1024 * 1024);
NodeType.R4_4XLARGE = new NodeType('cache.r4.4xlarge', 16, 101.38 * 1024 * 1024 * 1024);
NodeType.R4_8XLARGE = new NodeType('cache.r4.8xlarge', 32, 203.26 * 1024 * 1024 * 1024);
NodeType.R4_16XLARGE = new NodeType('cache.r4.16xlarge', 64, 407 * 1024 * 1024 * 1024);
NodeType.R5_LARGE = new NodeType('cache.r5.large', 2, 13.07 * 1024 * 1024 * 1024);
NodeType.R5_XLARGE = new NodeType('cache.r5.xlarge', 4, 26.32 * 1024 * 1024 * 1024);
NodeType.R5_2XLARGE = new NodeType('cache.r5.2xlarge', 8, 52.82 * 1024 * 1024 * 1024);
NodeType.R5_4XLARGE = new NodeType('cache.r5.4xlarge', 16, 105.81 * 1024 * 1024 * 1024);
NodeType.R5_12XLARGE = new NodeType('cache.r5.12xlarge', 48, 317.77 * 1024 * 1024 * 1024);
NodeType.R5_24LARGE = new NodeType('cache.r5.12xlarge', 96, 635.61 * 1024 * 1024 * 1024);
//# sourceMappingURL=data:application/json;base64,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