# coding: utf-8

"""日志设定文件"""

import logging

from longling.lib.stream import build_dir
from longling.lib.time import get_current_timestamp as default_timestamp

__all__ = ["LogLevel", "config_logging", "default_timestamp"]


class LogLevel(object):
    ERROR = logging.ERROR
    WARN = logging.WARN
    INFO = logging.INFO
    DEBUG = logging.DEBUG
    CRITICAL = logging.CRITICAL


LogLevelDict = {}
for key, value in {
    "error": LogLevel.ERROR,
    "warn": LogLevel.WARN,
    "info": LogLevel.INFO,
    "debug": LogLevel.DEBUG,
    "critical": LogLevel.CRITICAL,
}.items():
    LogLevelDict[key] = value
    LogLevelDict[key.upper()] = value


def config_logging(filename=None,
                   log_format='%(name)s, %(levelname)s %(message)s',
                   level=logging.INFO,
                   logger=None, console_log_level=None, propagate=False,
                   mode='a',
                   file_format=None, encoding: (str, None) = "utf-8"):
    """
    主日志设定文件

    Parameters
    ----------
    filename: str or None
        日志存储文件名，不为空时将创建文件存储日志
    log_format: str
        默认日志输出格式
    level: str or int
        默认日志等级
    logger: str or logging.logger
        日志logger名，可以为空（使用root logger），
        字符串类型（创建对应名logger），logger
    console_log_level: str, int or None
        屏幕日志等级，不为空时，使能屏幕日志输出
    propagate: bool
    mode: str
    file_format: str or None
        文件日志输出格式，为空时，使用log_format
    encoding
    Returns
    -------

    """
    if logger is None:
        logger = logging.getLogger()
    elif isinstance(logger, str):
        logger = logging.getLogger(logger)

    if isinstance(level, str):
        level = LogLevelDict[level]
    if isinstance(console_log_level, str):
        console_log_level = LogLevelDict[console_log_level]

    # need a clean state, for some module may
    # have called logging functions already (i.e. logging.info)
    # in that case, a default handler would been appended,
    # causing undesired output to stderr
    for handler in logger.handlers:
        logger.removeHandler(handler)
    formatter = logging.Formatter(log_format)
    logger.setLevel(level)
    if not propagate:
        logger.propagate = False
    if filename and filename is not None:
        build_dir(filename)
        handler = logging.FileHandler(filename, mode=mode, encoding=encoding)
        file_formatter = formatter
        if file_format:
            file_formatter = logging.Formatter(file_format)
        handler.setFormatter(file_formatter)
        logger.addHandler(handler)
    if console_log_level is not None:
        ch = logging.StreamHandler()
        ch.setFormatter(logging.Formatter(log_format))
        ch.setLevel(console_log_level)
        logger.addHandler(ch)

    return logger
