"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNagApp", version: "0.5.2" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            console.info(`Unable to add Nag Suppression for path: ${path} as it does not exist.`);
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            return `${stack.stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "@aws-prototyping-sdk/pdk-nag.PDKNag", version: "0.5.2" };
//# sourceMappingURL=data:application/json;base64,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