"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../lib/index");
const core_1 = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
test('Create the Runner', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN' });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        UserData: {
            'Fn::Base64': '#!/bin/bash\nyum update -y\nyum install docker -y\nservice docker start\nusermod -aG docker ec2-user\nchmod +x /var/run/docker.sock\nservice docker restart &&  chkconfig docker on\ndocker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner-register gitlab/gitlab-runner:alpine register --non-interactive --url https://gitlab.com/ --registration-token GITLAB_TOKEN --docker-pull-policy if-not-present --docker-volumes \"/var/run/docker.sock:/var/run/docker.sock\" --executor docker --docker-image \"alpine:latest\" --description \"Docker Runner\" --tag-list \"gitlab,awscdk,runner\" --docker-privileged\nsleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner gitlab/gitlab-runner:alpine\nusermod -aG docker ssm-user',
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Role');
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [{
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            }],
    });
});
test('Testing runner tag change ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing-have-type-tag', { gitlabtoken: 'GITLAB_TOKEN', tag1: 'aa', tag2: 'bb', tag3: 'cc' });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        UserData: {
            'Fn::Base64': '#!/bin/bash\nyum update -y\nyum install docker -y\nservice docker start\nusermod -aG docker ec2-user\nchmod +x /var/run/docker.sock\nservice docker restart &&  chkconfig docker on\ndocker run -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner-register gitlab/gitlab-runner:alpine register --non-interactive --url https://gitlab.com/ --registration-token GITLAB_TOKEN --docker-pull-policy if-not-present --docker-volumes \"/var/run/docker.sock:/var/run/docker.sock\" --executor docker --docker-image \"alpine:latest\" --description \"Docker Runner\" --tag-list \"aa,bb,cc\" --docker-privileged\nsleep 2 && docker run --restart always -d -v /home/ec2-user/.gitlab-runner:/etc/gitlab-runner -v /var/run/docker.sock:/var/run/docker.sock --name gitlab-runner gitlab/gitlab-runner:alpine\nusermod -aG docker ssm-user',
        },
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [{
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            }],
    });
});
test('Testing Runner Instance Type Change ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MICRO) });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupEgress: [{
                CidrIp: '0.0.0.0/0',
                Description: 'Allow all outbound traffic by default',
                IpProtocol: '-1',
            }],
    });
});
test('Runner Can Add Ingress ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const runner = new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MICRO), tag1: 'aa', tag2: 'bb', tag3: 'cc' });
    runner.runnerEc2.connections.allowFrom(aws_ec2_1.Peer.ipv4('1.2.3.4/8'), aws_ec2_1.Port.tcp(80));
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '10.0.0.0/16',
    });
    expect(stack).toHaveResource('AWS::EC2::SecurityGroup', {
        SecurityGroupIngress: [{
                CidrIp: '1.2.3.4/8',
                Description: 'from 1.2.3.4/8:80',
                FromPort: 80,
                IpProtocol: 'tcp',
                ToPort: 80,
            }],
    });
});
test('Runner Can Use Self VPC ', () => {
    const mockApp = new core_1.App();
    const stack = new core_1.Stack(mockApp, 'testing-stack');
    const newvpc = new aws_ec2_1.Vpc(stack, 'NEWVPC', {
        cidr: '10.1.0.0/16',
        maxAzs: 2,
        subnetConfiguration: [{
                cidrMask: 26,
                name: 'RunnerVPC',
                subnetType: aws_ec2_1.SubnetType.PUBLIC,
            }],
        natGateways: 0,
    });
    new index_1.GitlabContainerRunner(stack, 'testing', { gitlabtoken: 'GITLAB_TOKEN', ec2type: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MICRO), selfvpc: newvpc });
    expect(stack).toHaveResource('AWS::EC2::Instance', {
        InstanceType: 't2.micro',
    });
    expect(stack).not.toHaveResource('AWS::S3::Bucket');
    expect(stack).toHaveResource('AWS::EC2::VPC', {
        CidrBlock: '10.1.0.0/16',
    });
});
//# sourceMappingURL=data:application/json;base64,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