"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const api = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Check for ApiGateway params
// --------------------------------------------------------------
test('Test for default Params construct props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    const construct = new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-params', props);
    // Assertion
    expect(construct.apiGateway).not.toBeNull();
    expect(construct.apiGatewayCloudWatchRole).not.toBeNull();
    expect(construct.apiGatewayLogGroup).not.toBeNull();
    expect(construct.apiGatewayRole).not.toBeNull();
});
// --------------------------------------------------------------
// Check for Default IAM Role
// --------------------------------------------------------------
test('Test for default IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-default-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "awsapigatewayiotpolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Request Validator
// --------------------------------------------------------------
test('Test for default Params request validator', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-request-validator', props);
    // Assertion
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RequestValidator", {
        ValidateRequestBody: false,
        ValidateRequestParameters: true,
    });
});
// --------------------------------------------------------------
// Check for Integ Props and Method Props
// --------------------------------------------------------------
test('Test for default Params Integ Props and Method Props', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-integpros-methodprops', props);
    // Assertion for {topic-level-7} to ensure all Integration Request Params, Integration Responses,
    // Method Request Params and Method Reponses are intact
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "5\\d{2}",
                    StatusCode: "500"
                },
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    StatusCode: "403"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.path.topic-level-1": "method.request.path.topic-level-1",
                "integration.request.path.topic-level-2": "method.request.path.topic-level-2",
                "integration.request.path.topic-level-3": "method.request.path.topic-level-3",
                "integration.request.path.topic-level-4": "method.request.path.topic-level-4",
                "integration.request.path.topic-level-5": "method.request.path.topic-level-5",
                "integration.request.path.topic-level-6": "method.request.path.topic-level-6",
                "integration.request.path.topic-level-7": "method.request.path.topic-level-7"
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:${props.iotEndpoint}.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}/{topic-level-4}/{topic-level-5}/{topic-level-6}/{topic-level-7}`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200"
            },
            {
                StatusCode: "500"
            },
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true,
            "method.request.path.topic-level-2": true,
            "method.request.path.topic-level-3": true,
            "method.request.path.topic-level-4": true,
            "method.request.path.topic-level-5": true,
            "method.request.path.topic-level-6": true,
            "method.request.path.topic-level-7": true
        }
    });
});
// --------------------------------------------------------------
// Check for valid IoT Endpoint
// --------------------------------------------------------------
test('Test for valid iot enpoint', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: ' '
    };
    const app = () => {
        new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-no-endpoint', props);
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// Check for binaryMediaTypes
// --------------------------------------------------------------
test('Test for Binary Media types', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-binaryMediaTypes', {
        iotEndpoint: 'a1234567890123-ats'
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        BinaryMediaTypes: [
            "application/octet-stream",
        ],
    });
});
// --------------------------------------------------------------
// Check for Api Name and Desc
// --------------------------------------------------------------
test('Test for Api Name and Desc', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        restApiName: 'RestApi-Regional',
        description: 'Description for the Regional Rest Api'
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-name-desc', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        Name: 'RestApi-Regional',
        Description: 'Description for the Regional Rest Api'
    });
});
// --------------------------------------------------------------
// Check for Overriden IAM Role
// --------------------------------------------------------------
test('Test for overriden IAM Role', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const policyJSON = {
        Version: "2012-10-17",
        Statement: [
            {
                Action: [
                    "iot:UpdateThingShadow"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/mything1`,
                Effect: "Allow"
            },
            {
                Action: [
                    "iot:Publish"
                ],
                Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/topic-abc`,
                Effect: "Allow"
            }
        ]
    };
    const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
    const iamRoleProps = {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        path: '/',
        inlinePolicies: { testPolicy: policyDocument }
    };
    // Create a policy that overrides the default policy that gets created with the construct
    const apiGatewayExecutionRole = new iam.Role(stack, 'apigateway-iot-role', iamRoleProps);
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayExecutionRole,
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-overriden-iam-role', props);
    // Check whether default IAM role is creted to access IoT core
    expect(stack).toHaveResource("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "apigateway.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Path: "/",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "iot:UpdateThingShadow",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":thing/mything1"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "iot:Publish",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:iot:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":topic/topic-abc"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "testPolicy"
            }
        ]
    });
});
// --------------------------------------------------------------
// Check for Api Key Source
// --------------------------------------------------------------
test('Test for APi Key Source', () => {
    // Stack
    const stack = new cdk.Stack();
    const apiGatewayProps = {
        apiKeySourceType: api.ApiKeySourceType.AUTHORIZER,
    };
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key-source', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        ApiKeySourceType: "AUTHORIZER"
    });
});
// --------------------------------------------------------------
// Check for Api Key Creation
// --------------------------------------------------------------
test('Test for Api Key Creation', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const props = {
        iotEndpoint: `a1234567890123-ats`,
        apiGatewayCreateApiKey: true
    };
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-api-key', props);
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Properties: {
            ApiKeyRequired: true
        },
        Metadata: {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: "W59"
                    }
                ]
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------
// Test deployment for ApiGateway endPointCongiurationOverride
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-auth-none', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            endpointConfiguration: {
                types: [api.EndpointType.REGIONAL]
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::RestApi", {
        EndpointConfiguration: {
            Types: ["REGIONAL"]
        }
    });
});
// -----------------------------------------------------------------
// Test deployment for override ApiGateway AuthorizationType to NONE
// -----------------------------------------------------------------
test('Test for deployment ApiGateway AuthorizationType override', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "NONE"
    });
});
// -----------------------------------------------------------------
// Test deployment for fully qualified iotEndpoint name
// -----------------------------------------------------------------
test('Test for handling fully qualified iotEndpoint', () => {
    // Stack
    const stack = new cdk.Stack();
    // Helper declaration
    new lib_1.ApiGatewayToIot(stack, 'test-apigateway-iot-override-auth', {
        iotEndpoint: 'a1234567890123-ats.iot.ap-south-1.amazonaws.com',
        apiGatewayProps: {
            defaultMethodOptions: {
                authorizationType: api.AuthorizationType.NONE
            }
        }
    });
    // Assertion 1
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        Integration: {
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        ":a1234567890123-ats.iotdata:path/topics/{topic-level-1}/{topic-level-2}/{topic-level-3}"
                    ]
                ]
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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