"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const semver = require("semver");
const yargs = require("yargs");
const inventory = require("../../inventory");
const logging = require("../../logging");
const option_hints_1 = require("../../option-hints");
const projects_1 = require("../../projects");
const util_1 = require("../../util");
const macros_1 = require("../macros");
const util_2 = require("../util");
class Command {
    constructor() {
        this.command = "new [PROJECT-TYPE-NAME] [OPTIONS]";
        this.describe = "Creates a new projen project";
    }
    builder(args) {
        args.positional("PROJECT-TYPE-NAME", {
            describe: "optional only when --from is used and there is a single project type in the external module",
            type: "string",
        });
        args.option("synth", {
            type: "boolean",
            default: true,
            desc: "Synthesize after creating .projenrc.js",
        });
        args.option("comments", {
            type: "boolean",
            default: true,
            desc: "Include commented out options in .projenrc.js (use --no-comments to disable)",
        });
        args.option("from", {
            type: "string",
            alias: "f",
            desc: 'External jsii npm module to create project from. Supports any package spec supported by npm (such as "my-pack@^2.0")',
        });
        args.option("git", {
            type: "boolean",
            default: true,
            desc: "Run `git init` and create an initial commit (use --no-git to disable)",
        });
        args.example("projen new awscdk-app-ts", 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example("projen new --from projen-vue@^2", 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, type.docs ?? "", {
                builder: (cargs) => {
                    cargs.showHelpOnFail(false);
                    for (const option of type.options ?? []) {
                        if (option.simpleType !== "string" &&
                            option.simpleType !== "number" &&
                            option.simpleType !== "boolean" &&
                            option.kind !== "enum" &&
                            !isPrimitiveArrayOption(option)) {
                            /**
                             * Currently we only support these field types as command line options:
                             * - primitives (string, number, boolean)
                             * - lists of primitives
                             * - enums
                             */
                            continue;
                        }
                        let desc = [option.docs?.replace(/\ *\.$/, "") ?? ""];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== "undefined") {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default
                                    .replace(/^\ *-/, "")
                                    .replace(/\.$/, "")
                                    .trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(process.cwd(), option.default);
                            }
                        }
                        cargs.option(option.switch, {
                            group: required ? "Required:" : "Optional:",
                            type: argType(option),
                            description: desc.join(" "),
                            required,
                            // yargs behaves differently for arrays if the defaultValue property is present or not
                            ...(defaultValue ? { default: defaultValue } : {}),
                        });
                    }
                    return cargs;
                },
                handler: (argv) => initProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return initProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectTypeName) {
            console.log(`Invalid project type ${args.projectTypeName}. Supported types:`);
            for (const pjid of inventory.discover().map((x) => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Returns the yargs option type for a given project option
 */
function argType(option) {
    if (option.kind === "enum") {
        return "string";
    }
    if (isPrimitiveArrayOption(option)) {
        return "array";
    }
    return option.simpleType;
}
/**
 * Checks if the given option is a primitive array
 */
function isPrimitiveArrayOption(option) {
    return Boolean(option.jsonLike &&
        option.fullType.collection?.kind === "array" &&
        option.fullType.collection.elementtype.primitive &&
        ["string", "number"].includes(option.fullType.collection.elementtype.primitive));
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(cwd, value) {
    return macros_1.tryProcessMacro(cwd, value) ?? JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(cwd, type, argv) {
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== "undefined" && !prop.optional) {
            props[prop.name] = renderDefault(cwd, prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        curr[p] = value;
                    }
                    else {
                        curr[p] = curr[p] ?? {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function initProjectFromModule(baseDir, spec, args) {
    const projenVersion = args.projenVersion ?? "latest";
    const installCommand = util_2.renderInstallCommand(baseDir, `projen@${projenVersion}`);
    if (args.projenVersion) {
        util_1.exec(installCommand, { cwd: baseDir });
    }
    else {
        // do not overwrite existing installation
        util_1.exec(`npm ls --prefix="${baseDir}" --depth=0 --pattern projen || ${installCommand}`, { cwd: baseDir });
    }
    const moduleName = util_2.installPackage(baseDir, spec);
    // Find the just installed package and discover the rest recursively from this package folder
    const moduleDir = path.dirname(require.resolve(`${moduleName}/.jsii`, {
        paths: [baseDir],
    }));
    // Only leave projects from the main (requested) package
    const projects = inventory
        .discover(moduleDir)
        .filter((x) => x.moduleName === moduleName); // Only list project types from the requested 'from' module
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectTypeName;
    const types = projects.map((p) => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(",")}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested
        ? projects[0]
        : projects.find((p) => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(",")}`);
    }
    for (const option of type.options ?? []) {
        if (option.simpleType !== "string" &&
            option.simpleType !== "number" &&
            option.simpleType !== "boolean") {
            continue; // we don't support non-primitive fields as command line options
        }
        if (args[option.name] !== undefined) {
            if (option.simpleType === "number") {
                args[option.name] = parseInt(args[option.name]);
                args[option.switch] = args[option.name];
            }
            else if (option.simpleType === "boolean") {
                const raw = args[option.name];
                const safe = typeof raw === "string" ? util_1.isTruthy(raw) : raw;
                args[option.name] = safe;
                args[option.switch] = safe;
            }
            continue; // do not overwrite passed arguments
        }
        if (option.default && option.default !== "undefined") {
            if (!option.optional) {
                const defaultValue = renderDefault(baseDir, option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    args.devDeps = [spec];
    args["dev-deps"] = [spec];
    await initProject(baseDir, type, args);
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function initProject(baseDir, type, args) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(baseDir, type, args);
    projects_1.Projects.createProject({
        dir: props.outdir ?? baseDir,
        projectFqn: type.fqn,
        projectOptions: props,
        optionHints: args.comments
            ? option_hints_1.InitProjectOptionHints.FEATURED
            : option_hints_1.InitProjectOptionHints.NONE,
        synth: args.synth,
        post: args.post,
    });
    if (fs.existsSync(path.join(baseDir, "package.json")) && args.post) {
        util_1.exec("npm run eslint --if-present", { cwd: baseDir });
    }
    if (args.git) {
        const git = (cmd) => util_1.exec(`git ${cmd}`, { cwd: baseDir });
        const gitversion = util_1.getGitVersion(util_1.execCapture("git --version", { cwd: baseDir }).toString());
        logging.debug("system using git version ", gitversion);
        if (gitversion && semver.gte(gitversion, "2.28.0")) {
            git("init -b main");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("default branch name set to main");
        }
        else {
            git("init");
            git("add .");
            git('commit --allow-empty -m "chore: project created with projen"');
            logging.debug("older version of git detected, changed default branch name to main");
            git("branch -M main");
        }
    }
}
exports.default = new Command();
//# sourceMappingURL=data:application/json;base64,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