"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const nag_rules_1 = require("../../nag-rules");
/**
 * EventBridge event bus policies do not allow for open access
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_events_1.CfnEventBusPolicy) {
        if (aws_cdk_lib_1.Stack.of(node).resolve(node.principal) === '*') {
            const condition = aws_cdk_lib_1.Stack.of(node).resolve(node.condition);
            if (condition === undefined ||
                condition.key === undefined ||
                condition.type === undefined ||
                condition.value === undefined) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        const resolvedStatement = aws_cdk_lib_1.Stack.of(node).resolve(node.statement);
        if (resolvedStatement !== undefined) {
            const condition = aws_cdk_lib_1.Stack.of(node).resolve(resolvedStatement?.Condition);
            if (resolvedStatement?.Effect === 'Allow' &&
                checkMatchingPrincipal(resolvedStatement?.Principal) === true &&
                (condition === undefined || JSON.stringify(condition) === '{}')) {
                return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
            }
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
/**
 * Helper function to check whether the event bus policy applies to all principals
 * @param node The CfnEventBusPolicy to check
 * @param principal The principals in the event bus policy
 * @returns Whether the CfnEventBusPolicy applies to all principals
 */
function checkMatchingPrincipal(principals) {
    if (principals === '*') {
        return true;
    }
    const awsPrincipal = principals.AWS;
    if (Array.isArray(awsPrincipal)) {
        for (const account of awsPrincipal) {
            if (account === '*') {
                return true;
            }
        }
    }
    else if (awsPrincipal === '*') {
        return true;
    }
    return false;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiRXZlbnRCdXNPcGVuQWNjZXNzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL3J1bGVzL2V2ZW50YnJpZGdlL0V2ZW50QnVzT3BlbkFjY2Vzcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUFBOzs7RUFHRTtBQUNGLCtCQUE2QjtBQUM3Qiw2Q0FBaUQ7QUFDakQsdURBQTJEO0FBQzNELCtDQUFvRDtBQUVwRDs7O0dBR0c7QUFDSCxrQkFBZSxNQUFNLENBQUMsY0FBYyxDQUNsQyxDQUFDLElBQWlCLEVBQXFCLEVBQUU7SUFDdkMsSUFBSSxJQUFJLFlBQVksOEJBQWlCLEVBQUU7UUFDckMsSUFBSSxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEdBQUcsRUFBRTtZQUNsRCxNQUFNLFNBQVMsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3pELElBQ0UsU0FBUyxLQUFLLFNBQVM7Z0JBQ3ZCLFNBQVMsQ0FBQyxHQUFHLEtBQUssU0FBUztnQkFDM0IsU0FBUyxDQUFDLElBQUksS0FBSyxTQUFTO2dCQUM1QixTQUFTLENBQUMsS0FBSyxLQUFLLFNBQVMsRUFDN0I7Z0JBQ0EsT0FBTyw2QkFBaUIsQ0FBQyxhQUFhLENBQUM7YUFDeEM7U0FDRjtRQUNELE1BQU0saUJBQWlCLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNqRSxJQUFJLGlCQUFpQixLQUFLLFNBQVMsRUFBRTtZQUNuQyxNQUFNLFNBQVMsR0FBRyxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsU0FBUyxDQUFDLENBQUM7WUFDdkUsSUFDRSxpQkFBaUIsRUFBRSxNQUFNLEtBQUssT0FBTztnQkFDckMsc0JBQXNCLENBQUMsaUJBQWlCLEVBQUUsU0FBUyxDQUFDLEtBQUssSUFBSTtnQkFDN0QsQ0FBQyxTQUFTLEtBQUssU0FBUyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLEtBQUssSUFBSSxDQUFDLEVBQy9EO2dCQUNBLE9BQU8sNkJBQWlCLENBQUMsYUFBYSxDQUFDO2FBQ3hDO1NBQ0Y7UUFDRCxPQUFPLDZCQUFpQixDQUFDLFNBQVMsQ0FBQztLQUNwQztTQUFNO1FBQ0wsT0FBTyw2QkFBaUIsQ0FBQyxjQUFjLENBQUM7S0FDekM7QUFDSCxDQUFDLEVBQ0QsTUFBTSxFQUNOLEVBQUUsS0FBSyxFQUFFLFlBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxJQUFJLEVBQUUsQ0FDbEMsQ0FBQztBQUVGOzs7OztHQUtHO0FBQ0gsU0FBUyxzQkFBc0IsQ0FBQyxVQUFlO0lBQzdDLElBQUksVUFBVSxLQUFLLEdBQUcsRUFBRTtRQUN0QixPQUFPLElBQUksQ0FBQztLQUNiO0lBQ0QsTUFBTSxZQUFZLEdBQUcsVUFBVSxDQUFDLEdBQUcsQ0FBQztJQUNwQyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEVBQUU7UUFDL0IsS0FBSyxNQUFNLE9BQU8sSUFBSSxZQUFZLEVBQUU7WUFDbEMsSUFBSSxPQUFPLEtBQUssR0FBRyxFQUFFO2dCQUNuQixPQUFPLElBQUksQ0FBQzthQUNiO1NBQ0Y7S0FDRjtTQUFNLElBQUksWUFBWSxLQUFLLEdBQUcsRUFBRTtRQUMvQixPQUFPLElBQUksQ0FBQztLQUNiO0lBQ0QsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLypcbkNvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiovXG5pbXBvcnQgeyBwYXJzZSB9IGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ2ZuRXZlbnRCdXNQb2xpY3kgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZXZlbnRzJztcbmltcG9ydCB7IE5hZ1J1bGVDb21wbGlhbmNlIH0gZnJvbSAnLi4vLi4vbmFnLXJ1bGVzJztcblxuLyoqXG4gKiBFdmVudEJyaWRnZSBldmVudCBidXMgcG9saWNpZXMgZG8gbm90IGFsbG93IGZvciBvcGVuIGFjY2Vzc1xuICogQHBhcmFtIG5vZGUgdGhlIENmblJlc291cmNlIHRvIGNoZWNrXG4gKi9cbmV4cG9ydCBkZWZhdWx0IE9iamVjdC5kZWZpbmVQcm9wZXJ0eShcbiAgKG5vZGU6IENmblJlc291cmNlKTogTmFnUnVsZUNvbXBsaWFuY2UgPT4ge1xuICAgIGlmIChub2RlIGluc3RhbmNlb2YgQ2ZuRXZlbnRCdXNQb2xpY3kpIHtcbiAgICAgIGlmIChTdGFjay5vZihub2RlKS5yZXNvbHZlKG5vZGUucHJpbmNpcGFsKSA9PT0gJyonKSB7XG4gICAgICAgIGNvbnN0IGNvbmRpdGlvbiA9IFN0YWNrLm9mKG5vZGUpLnJlc29sdmUobm9kZS5jb25kaXRpb24pO1xuICAgICAgICBpZiAoXG4gICAgICAgICAgY29uZGl0aW9uID09PSB1bmRlZmluZWQgfHxcbiAgICAgICAgICBjb25kaXRpb24ua2V5ID09PSB1bmRlZmluZWQgfHxcbiAgICAgICAgICBjb25kaXRpb24udHlwZSA9PT0gdW5kZWZpbmVkIHx8XG4gICAgICAgICAgY29uZGl0aW9uLnZhbHVlID09PSB1bmRlZmluZWRcbiAgICAgICAgKSB7XG4gICAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGNvbnN0IHJlc29sdmVkU3RhdGVtZW50ID0gU3RhY2sub2Yobm9kZSkucmVzb2x2ZShub2RlLnN0YXRlbWVudCk7XG4gICAgICBpZiAocmVzb2x2ZWRTdGF0ZW1lbnQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBjb25zdCBjb25kaXRpb24gPSBTdGFjay5vZihub2RlKS5yZXNvbHZlKHJlc29sdmVkU3RhdGVtZW50Py5Db25kaXRpb24pO1xuICAgICAgICBpZiAoXG4gICAgICAgICAgcmVzb2x2ZWRTdGF0ZW1lbnQ/LkVmZmVjdCA9PT0gJ0FsbG93JyAmJlxuICAgICAgICAgIGNoZWNrTWF0Y2hpbmdQcmluY2lwYWwocmVzb2x2ZWRTdGF0ZW1lbnQ/LlByaW5jaXBhbCkgPT09IHRydWUgJiZcbiAgICAgICAgICAoY29uZGl0aW9uID09PSB1bmRlZmluZWQgfHwgSlNPTi5zdHJpbmdpZnkoY29uZGl0aW9uKSA9PT0gJ3t9JylcbiAgICAgICAgKSB7XG4gICAgICAgICAgcmV0dXJuIE5hZ1J1bGVDb21wbGlhbmNlLk5PTl9DT01QTElBTlQ7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5DT01QTElBTlQ7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBOYWdSdWxlQ29tcGxpYW5jZS5OT1RfQVBQTElDQUJMRTtcbiAgICB9XG4gIH0sXG4gICduYW1lJyxcbiAgeyB2YWx1ZTogcGFyc2UoX19maWxlbmFtZSkubmFtZSB9XG4pO1xuXG4vKipcbiAqIEhlbHBlciBmdW5jdGlvbiB0byBjaGVjayB3aGV0aGVyIHRoZSBldmVudCBidXMgcG9saWN5IGFwcGxpZXMgdG8gYWxsIHByaW5jaXBhbHNcbiAqIEBwYXJhbSBub2RlIFRoZSBDZm5FdmVudEJ1c1BvbGljeSB0byBjaGVja1xuICogQHBhcmFtIHByaW5jaXBhbCBUaGUgcHJpbmNpcGFscyBpbiB0aGUgZXZlbnQgYnVzIHBvbGljeVxuICogQHJldHVybnMgV2hldGhlciB0aGUgQ2ZuRXZlbnRCdXNQb2xpY3kgYXBwbGllcyB0byBhbGwgcHJpbmNpcGFsc1xuICovXG5mdW5jdGlvbiBjaGVja01hdGNoaW5nUHJpbmNpcGFsKHByaW5jaXBhbHM6IGFueSk6IGJvb2xlYW4ge1xuICBpZiAocHJpbmNpcGFscyA9PT0gJyonKSB7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cbiAgY29uc3QgYXdzUHJpbmNpcGFsID0gcHJpbmNpcGFscy5BV1M7XG4gIGlmIChBcnJheS5pc0FycmF5KGF3c1ByaW5jaXBhbCkpIHtcbiAgICBmb3IgKGNvbnN0IGFjY291bnQgb2YgYXdzUHJpbmNpcGFsKSB7XG4gICAgICBpZiAoYWNjb3VudCA9PT0gJyonKSB7XG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuICAgIH1cbiAgfSBlbHNlIGlmIChhd3NQcmluY2lwYWwgPT09ICcqJykge1xuICAgIHJldHVybiB0cnVlO1xuICB9XG4gIHJldHVybiBmYWxzZTtcbn1cbiJdfQ==