"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*
Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0
*/
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_elasticbeanstalk_1 = require("aws-cdk-lib/aws-elasticbeanstalk");
const nag_rules_1 = require("../../nag-rules");
/**
 * Elastic Beanstalk environments are configured to use a specific VPC
 * https://docs.aws.amazon.com/elasticbeanstalk/latest/dg/command-options-general.html#command-options-general-ec2vpc
 * @param node the CfnResource to check
 */
exports.default = Object.defineProperty((node) => {
    if (node instanceof aws_elasticbeanstalk_1.CfnEnvironment) {
        const optionSettings = aws_cdk_lib_1.Stack.of(node).resolve(node.optionSettings);
        if (optionSettings == undefined) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        let foundEnabled = false;
        for (const optionSetting of optionSettings) {
            const resolvedOptionSetting = aws_cdk_lib_1.Stack.of(node).resolve(optionSetting);
            const namespace = resolvedOptionSetting.namespace;
            const optionName = resolvedOptionSetting.optionName;
            const value = resolvedOptionSetting.value;
            if (namespace === 'aws:ec2:vpc' &&
                optionName === 'VPCId' &&
                value !== undefined) {
                foundEnabled = true;
                break;
            }
        }
        if (!foundEnabled) {
            return nag_rules_1.NagRuleCompliance.NON_COMPLIANT;
        }
        return nag_rules_1.NagRuleCompliance.COMPLIANT;
    }
    else {
        return nag_rules_1.NagRuleCompliance.NOT_APPLICABLE;
    }
}, 'name', { value: path_1.parse(__filename).name });
//# sourceMappingURL=data:application/json;base64,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