import { ICertificate } from '../../aws-certificatemanager';
import { IUserPool } from '../../aws-cognito';
import { IRole, Grant, IGrantable } from '../../aws-iam';
import { IFunction } from '../../aws-lambda';
import { ILogGroup, RetentionDays } from '../../aws-logs';
import { CfnResource, Duration, Expiration, IResolvable } from '../../core';
import { Construct } from 'constructs';
import { IGraphqlApi, GraphqlApiBase } from './graphqlapi-base';
import { ISchema } from './schema';
/**
 * enum with all possible values for AppSync authorization type
 */
export declare enum AuthorizationType {
    /**
     * API Key authorization type
     */
    API_KEY = "API_KEY",
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    IAM = "AWS_IAM",
    /**
     * Cognito User Pool authorization type
     */
    USER_POOL = "AMAZON_COGNITO_USER_POOLS",
    /**
     * OpenID Connect authorization type
     */
    OIDC = "OPENID_CONNECT",
    /**
     * Lambda authorization type
     */
    LAMBDA = "AWS_LAMBDA"
}
/**
 * Interface to specify default or additional authorization(s)
 */
export interface AuthorizationMode {
    /**
     * One of possible four values AppSync supports
     *
     * @see https://docs.aws.amazon.com/appsync/latest/devguide/security.html
     *
     * @default - `AuthorizationType.API_KEY`
     */
    readonly authorizationType: AuthorizationType;
    /**
     * If authorizationType is `AuthorizationType.USER_POOL`, this option is required.
     * @default - none
     */
    readonly userPoolConfig?: UserPoolConfig;
    /**
     * If authorizationType is `AuthorizationType.API_KEY`, this option can be configured.
     * @default - name: 'DefaultAPIKey' | description: 'Default API Key created by CDK'
     */
    readonly apiKeyConfig?: ApiKeyConfig;
    /**
     * If authorizationType is `AuthorizationType.OIDC`, this option is required.
     * @default - none
     */
    readonly openIdConnectConfig?: OpenIdConnectConfig;
    /**
     * If authorizationType is `AuthorizationType.LAMBDA`, this option is required.
     * @default - none
     */
    readonly lambdaAuthorizerConfig?: LambdaAuthorizerConfig;
}
/**
 * enum with all possible values for Cognito user-pool default actions
 */
export declare enum UserPoolDefaultAction {
    /**
     * ALLOW access to API
     */
    ALLOW = "ALLOW",
    /**
     * DENY access to API
     */
    DENY = "DENY"
}
/**
 * Configuration for Cognito user-pools in AppSync
 */
export interface UserPoolConfig {
    /**
     * The Cognito user pool to use as identity source
     */
    readonly userPool: IUserPool;
    /**
     * the optional app id regex
     *
     * @default -  None
     */
    readonly appIdClientRegex?: string;
    /**
     * Default auth action
     *
     * @default ALLOW
     */
    readonly defaultAction?: UserPoolDefaultAction;
}
/**
 * Configuration for API Key authorization in AppSync
 */
export interface ApiKeyConfig {
    /**
     * Unique name of the API Key
     * @default - 'DefaultAPIKey'
     */
    readonly name?: string;
    /**
     * Description of API key
     * @default - 'Default API Key created by CDK'
     */
    readonly description?: string;
    /**
     * The time from creation time after which the API key expires.
     * It must be a minimum of 1 day and a maximum of 365 days from date of creation.
     * Rounded down to the nearest hour.
     *
     * @default - 7 days rounded down to nearest hour
     */
    readonly expires?: Expiration;
}
/**
 * Configuration for OpenID Connect authorization in AppSync
 */
export interface OpenIdConnectConfig {
    /**
     * The number of milliseconds an OIDC token is valid after being authenticated by OIDC provider.
     * `auth_time` claim in OIDC token is required for this validation to work.
     * @default - no validation
     */
    readonly tokenExpiryFromAuth?: number;
    /**
     * The number of milliseconds an OIDC token is valid after being issued to a user.
     * This validation uses `iat` claim of OIDC token.
     * @default - no validation
     */
    readonly tokenExpiryFromIssue?: number;
    /**
     * The client identifier of the Relying party at the OpenID identity provider.
     * A regular expression can be specified so AppSync can validate against multiple client identifiers at a time.
     * @example - 'ABCD|CDEF' // where ABCD and CDEF are two different clientId
     * @default - * (All)
     */
    readonly clientId?: string;
    /**
     * The issuer for the OIDC configuration. The issuer returned by discovery must exactly match the value of `iss` in the OIDC token.
     */
    readonly oidcProvider: string;
}
/**
 * Configuration for Lambda authorization in AppSync. Note that you can only have a single AWS Lambda function configured to authorize your API.
 */
export interface LambdaAuthorizerConfig {
    /**
     * The authorizer lambda function.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-appsync-graphqlapi-lambdaauthorizerconfig.html
     */
    readonly handler: IFunction;
    /**
     * How long the results are cached.
     * Disable caching by setting this to 0.
     *
     * @default Duration.minutes(5)
     */
    readonly resultsCacheTtl?: Duration;
    /**
     * A regular expression for validation of tokens before the Lambda function is called.
     *
     * @default - no regex filter will be applied.
     */
    readonly validationRegex?: string;
}
/**
 * Configuration of the API authorization modes.
 */
export interface AuthorizationConfig {
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly defaultAuthorization?: AuthorizationMode;
    /**
     * Additional authorization modes
     *
     * @default - No other modes
     */
    readonly additionalAuthorizationModes?: AuthorizationMode[];
}
/**
 * log-level for fields in AppSync
 */
export declare enum FieldLogLevel {
    /**
     * No logging
     */
    NONE = "NONE",
    /**
     * Error logging
     */
    ERROR = "ERROR",
    /**
     * All logging
     */
    ALL = "ALL"
}
/**
 * Logging configuration for AppSync
 */
export interface LogConfig {
    /**
     * exclude verbose content
     *
     * @default false
     */
    readonly excludeVerboseContent?: boolean | IResolvable;
    /**
     * log level for fields
     *
     * @default - Use AppSync default
     */
    readonly fieldLogLevel?: FieldLogLevel;
    /**
     * The role for CloudWatch Logs
     *
     * @default - None
     */
    readonly role?: IRole;
    /**
    * The number of days log events are kept in CloudWatch Logs.
    * By default AppSync keeps the logs infinitely. When updating this property,
    * unsetting it doesn't remove the log retention policy.
    * To remove the retention policy, set the value to `INFINITE`
    *
    * @default RetentionDays.INFINITE
    */
    readonly retention?: RetentionDays;
}
/**
 * Domain name configuration for AppSync
 */
export interface DomainOptions {
    /**
     * The certificate to use with the domain name.
     */
    readonly certificate: ICertificate;
    /**
     * The actual domain name. For example, `api.example.com`.
     */
    readonly domainName: string;
}
/**
 * Properties for an AppSync GraphQL API
 */
export interface GraphqlApiProps {
    /**
     * the name of the GraphQL API
     */
    readonly name: string;
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly authorizationConfig?: AuthorizationConfig;
    /**
     * Logging configuration for this api
     *
     * @default - None
     */
    readonly logConfig?: LogConfig;
    /**
     * GraphQL schema definition. Specify how you want to define your schema.
     *
     * Schema.fromFile(filePath: string) allows schema definition through schema.graphql file
     *
     * @default - schema will be generated code-first (i.e. addType, addObjectType, etc.)
     *
     */
    readonly schema: ISchema;
    /**
     * A flag indicating whether or not X-Ray tracing is enabled for the GraphQL API.
     *
     * @default - false
     */
    readonly xrayEnabled?: boolean;
    /**
     * The domain name configuration for the GraphQL API
     *
     * The Route 53 hosted zone and CName DNS record must be configured in addition to this setting to
     * enable custom domain URL
     *
     * @default - no domain name
     */
    readonly domainName?: DomainOptions;
}
/**
 * A class used to generate resource arns for AppSync
 */
export declare class IamResource {
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns: string[]): IamResource;
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type: string, ...fields: string[]): IamResource;
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all(): IamResource;
    private arns;
    private constructor();
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api: GraphqlApi): string[];
}
/**
 * Attributes for GraphQL imports
 */
export interface GraphqlApiAttributes {
    /**
     * an unique AWS AppSync GraphQL API identifier
     * i.e. 'lxz775lwdrgcndgz3nurvac7oa'
     */
    readonly graphqlApiId: string;
    /**
     * the arn for the GraphQL Api
     * @default - autogenerated arn
     */
    readonly graphqlApiArn?: string;
}
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
export declare class GraphqlApi extends GraphqlApiBase {
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope: Construct, id: string, attrs: GraphqlApiAttributes): IGraphqlApi;
    /**
     * an unique AWS AppSync GraphQL API identifier
     * i.e. 'lxz775lwdrgcndgz3nurvac7oa'
     */
    readonly apiId: string;
    /**
     * the ARN of the API
     */
    readonly arn: string;
    /**
     * the URL of the endpoint created by AppSync
     *
     * @attribute GraphQlUrl
     */
    readonly graphqlUrl: string;
    /**
     * the name of the API
     */
    readonly name: string;
    /**
     * the schema attached to this api
     */
    readonly schema: ISchema;
    /**
     * The Authorization Types for this GraphQL Api
     */
    readonly modes: AuthorizationType[];
    /**
     * the configured API key, if present
     *
     * @default - no api key
     */
    readonly apiKey?: string;
    /**
     * the CloudWatch Log Group for this API
     */
    readonly logGroup: ILogGroup;
    private schemaResource;
    private api;
    private apiKeyResource?;
    private domainNameResource?;
    constructor(scope: Construct, id: string, props: GraphqlApiProps);
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee: IGrantable, resources: IamResource, ...actions: string[]): Grant;
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee: IGrantable, ...fields: string[]): Grant;
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee: IGrantable, ...fields: string[]): Grant;
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee: IGrantable, ...fields: string[]): Grant;
    private validateAuthorizationProps;
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct: CfnResource): boolean;
    private setupLogConfig;
    private setupOpenIdConnectConfig;
    private setupUserPoolConfig;
    private setupLambdaAuthorizerConfig;
    private setupAdditionalAuthorizationModes;
    private createAPIKey;
    /**
     * The AppSyncDomainName of the associated custom domain
     */
    get appSyncDomainName(): string;
}
