#!/usr/bin/env python
from setuptools import setup
setup(
  name = 'cs.context',
  author = 'Cameron Simpson',
  author_email = 'cs@cskk.id.au',
  version = '20210420',
  url = 'https://bitbucket.org/cameron_simpson/css/commits/all',
  description =
    'Assorted context managers.',
  long_description =
    ('Assorted context managers.\n'    
 '\n'    
 '*Latest release 20210420*:\n'    
 'Docstring corrections and improvements.\n'    
 '\n'    
 '## Function `popattrs(o, attr_names, old_values)`\n'    
 '\n'    
 'The "pop" part of `stackattrs`.\n'    
 'Restore previous attributes of `o`\n'    
 'named by `attr_names` with previous state in `old_values`.\n'    
 '\n'    
 'This can be useful in hooks/signals/callbacks,\n'    
 'where you cannot inline a context manager.\n'    
 '\n'    
 '## Function `popkeys(d, key_names, old_values)`\n'    
 '\n'    
 'The "pop" part of `stackkeys`.\n'    
 'Restore previous key values of `d`\n'    
 'named by `key_names` with previous state in `old_values`.\n'    
 '\n'    
 'This can be useful in hooks/signals/callbacks,\n'    
 'where you cannot inline a context manager.\n'    
 '\n'    
 '## Function `pushattrs(o, **attr_values)`\n'    
 '\n'    
 'The "push" part of `stackattrs`.\n'    
 'Push `attr_values` onto `o` as attributes,\n'    
 'return the previous attribute values in a dict.\n'    
 '\n'    
 'This can be useful in hooks/signals/callbacks,\n'    
 'where you cannot inline a context manager.\n'    
 '\n'    
 '## Function `pushkeys(d, **key_values)`\n'    
 '\n'    
 'The "push" part of `stackkeys`.\n'    
 'Push `key_values` onto `d` as key values.\n'    
 'return the previous key values in a dict.\n'    
 '\n'    
 'This can be useful in hooks/signals/callbacks,\n'    
 'where you cannot inline a context manager.\n'    
 '\n'    
 '## Function `setup_cmgr(cmgr)`\n'    
 '\n'    
 'Run the set up phase of the context manager `cmgr`\n'    
 'and return a callable which runs the tear down phase.\n'    
 '\n'    
 'This is a convenience wrapper for the lower level `twostep()` function\n'    
 'which produces a two iteration generator from a context manager.\n'    
 '\n'    
 "The purpose of `setup_cmgr()` is to split any context manager's operation\n"    
 'across two steps when the set up and teardown phases must operate\n'    
 'in different parts of your code.\n'    
 'A common situation is the `__enter__` and `__exit__` methods\n'    
 'of another context manager class.\n'    
 '\n'    
 'The call to `setup_cmgr()` performs the "enter" phase\n'    
 'and returns the tear down callable.\n'    
 'Calling that performs the tear down phase.\n'    
 '\n'    
 'Example use in a class:\n'    
 '\n'    
 '    class SomeClass:\n'    
 '        def __init__(self, foo)\n'    
 '            self.foo = foo\n'    
 '            self._teardown = None\n'    
 '        def __enter__(self):\n'    
 '            self._teardown = setup_cmgr(stackattrs(o, setting=foo))\n'    
 '        def __exit__(self, *_):\n'    
 '            self._teardown()\n'    
 '            self._teardown = None\n'    
 '\n'    
 '## Class `StackableState(_thread._local)`\n'    
 '\n'    
 'An object which can be called as a context manager\n'    
 'to push changes to its attributes.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 '    >>> state = StackableState(a=1, b=2)\n'    
 '    >>> state.a\n'    
 '    1\n'    
 '    >>> state.b\n'    
 '    2\n'    
 '    >>> state\n'    
 '    StackableState(a=1,b=2)\n'    
 '    >>> with state(a=3, x=4):\n'    
 '    ...     print(state)\n'    
 '    ...     print("a", state.a)\n'    
 '    ...     print("b", state.b)\n'    
 '    ...     print("x", state.x)\n'    
 '    ...\n'    
 '    StackableState(a=3,b=2,x=4)\n'    
 '    a 3\n'    
 '    b 2\n'    
 '    x 4\n'    
 '    >>> state.a\n'    
 '    1\n'    
 '    >>> state\n'    
 '    StackableState(a=1,b=2)\n'    
 '\n'    
 '### Method `StackableState.__call__(self, **kw)`\n'    
 '\n'    
 'Calling an instance is a context manager yielding `self`\n'    
 'with attributes modified by `kw`.\n'    
 '\n'    
 '## Function `stackattrs(o, **attr_values)`\n'    
 '\n'    
 'Context manager to push new values for the attributes of `o`\n'    
 'and to restore them afterward.\n'    
 'Returns a `dict` containing a mapping of the previous attribute values.\n'    
 'Attributes not present are not present in the mapping.\n'    
 '\n'    
 'Restoration includes deleting attributes which were not present\n'    
 'initially.\n'    
 '\n'    
 'This makes it easy to adjust temporarily some shared context object\n'    
 'without having to pass it through the call stack.\n'    
 '\n'    
 'See `stackkeys` for a flavour of this for mappings.\n'    
 '\n'    
 'Example of fiddling a programme\'s "verbose" mode:\n'    
 '\n'    
 '    >>> class RunModes:\n'    
 '    ...     def __init__(self, verbose=False):\n'    
 '    ...         self.verbose = verbose\n'    
 '    ...\n'    
 '    >>> runmode = RunModes()\n'    
 '    >>> if runmode.verbose:\n'    
 '    ...     print("suppressed message")\n'    
 '    ...\n'    
 '    >>> with stackattrs(runmode, verbose=True):\n'    
 '    ...     if runmode.verbose:\n'    
 '    ...         print("revealed message")\n'    
 '    ...\n'    
 '    revealed message\n'    
 '    >>> if runmode.verbose:\n'    
 '    ...     print("another suppressed message")\n'    
 '    ...\n'    
 '\n'    
 'Example exhibiting restoration of absent attributes:\n'    
 '\n'    
 '    >>> class O:\n'    
 '    ...     def __init__(self):\n'    
 '    ...         self.a = 1\n'    
 '    ...\n'    
 '    >>> o = O()\n'    
 '    >>> print(o.a)\n'    
 '    1\n'    
 '    >>> print(o.b)\n'    
 '    Traceback (most recent call last):\n'    
 '      File "<stdin>", line 1, in <module>\n'    
 "    AttributeError: 'O' object has no attribute 'b'\n"    
 '    >>> with stackattrs(o, a=3, b=4):\n'    
 '    ...     print(o.a)\n'    
 '    ...     print(o.b)\n'    
 '    ...     o.b = 5\n'    
 '    ...     print(o.b)\n'    
 "    ...     delattr(o, 'a')\n"    
 '    ...\n'    
 '    3\n'    
 '    4\n'    
 '    5\n'    
 '    >>> print(o.a)\n'    
 '    1\n'    
 '    >>> print(o.b)\n'    
 '    Traceback (most recent call last):\n'    
 '      File "<stdin>", line 1, in <module>\n'    
 "    AttributeError: 'O' object has no attribute 'b'\n"    
 '\n'    
 '## Function `stackkeys(d, **key_values)`\n'    
 '\n'    
 'Context manager to push new values for the key values of `d`\n'    
 'and to restore them afterward.\n'    
 'Returns a `dict` containing a mapping of the previous key values.\n'    
 'Keys not present are not present in the mapping.\n'    
 '\n'    
 'Restoration includes deleting key values which were not present\n'    
 'initially.\n'    
 '\n'    
 'This makes it easy to adjust temporarily some shared context object\n'    
 'without having to pass it through the call stack.\n'    
 '\n'    
 'See `stackattrs` for a flavour of this for object attributes.\n'    
 '\n'    
 'Example of making log entries which may reference\n'    
 'some higher level context log entry:\n'    
 '\n'    
 '    >>> import time\n'    
 '    >>> global_context = {\n'    
 "    ...     'parent': None,\n"    
 '    ... }\n'    
 '    >>> def log_entry(desc, **kw):\n'    
 '    ...     print("log_entry: global_context =", repr(global_context))\n'    
 '    ...     entry = dict(global_context)\n'    
 '    ...     entry.update(desc=desc, when=time.time())\n'    
 '    ...     entry.update(kw)\n'    
 '    ...     return entry\n'    
 '    ...\n'    
 '    >>> log_entry("stand alone entry")    #doctest: +ELLIPSIS\n'    
 "    log_entry: global_context = {'parent': None}\n"    
 "    {'parent': None, 'desc': 'stand alone entry', 'when': ...}\n"    
 '    >>> context_entry = log_entry("high level entry")\n'    
 "    log_entry: global_context = {'parent': None}\n"    
 '    >>> context_entry                     #doctest: +ELLIPSIS\n'    
 "    {'parent': None, 'desc': 'high level entry', 'when': ...}\n"    
 '    >>> with stackkeys(global_context, parent=context_entry): #doctest: '    
 '+ELLIPSIS\n'    
 '    ...     print(repr(log_entry("low level event")))\n'    
 '    ...\n'    
 "    log_entry: global_context = {'parent': {'parent': None, 'desc': 'high "    
 "level entry', 'when': ...}}\n"    
 "    {'parent': {'parent': None, 'desc': 'high level entry', 'when': ...}, "    
 "'desc': 'low level event', 'when': ...}\n"    
 '    >>> log_entry("another standalone entry")    #doctest: +ELLIPSIS\n'    
 "    log_entry: global_context = {'parent': None}\n"    
 "    {'parent': None, 'desc': 'another standalone entry', 'when': ...}\n"    
 '\n'    
 '## Function `twostep(cmgr)`\n'    
 '\n'    
 'Return a generator which operates the context manager `cmgr`.\n'    
 '\n'    
 'See also the `setup_cmgr(cmgr)` function\n'    
 'which is a convenience wrapper for this low level generator.\n'    
 '\n'    
 "The purpose of `twostep()` is to split any context manager's operation\n"    
 'across two steps when the set up and teardown phases must operate\n'    
 'in different parts of your code.\n'    
 'A common situation is the `__enter__` and `__exit__` methods\n'    
 'of another context manager class.\n'    
 '\n'    
 'The first iteration performs the "enter" phase and yields.\n'    
 'The second iteration performs the "exit" phase and yields.\n'    
 '\n'    
 'Example use in a class:\n'    
 '\n'    
 '    class SomeClass:\n'    
 '        def __init__(self, foo)\n'    
 '            self.foo = foo\n'    
 '            self._cmgr = None\n'    
 '        def __enter__(self):\n'    
 '            self._cmgr = next(stackattrs(o, setting=foo))\n'    
 '        def __exit__(self, *_):\n'    
 '            next(self._cmgr)\n'    
 '            self._cmgr = None\n'    
 '\n'    
 '# Release Log\n'    
 '\n'    
 '\n'    
 '\n'    
 '*Release 20210420*:\n'    
 'Docstring corrections and improvements.\n'    
 '\n'    
 '*Release 20210306*:\n'    
 '* New twostep() and setup_cmgr() functions to split a context manager into '    
 'set up and teardown phases for when these must occur in different parts of '    
 'the code.\n'    
 '* New thread local StackableState class which can be called to push '    
 'attribute changes with stackattrs - intended for use as shared global state '    
 'to avoiod passing through deep function call chains.\n'    
 '\n'    
 '*Release 20200725.1*:\n'    
 'Docstring improvements.\n'    
 '\n'    
 '*Release 20200725*:\n'    
 'New stackkeys and components pushkeys and popkeys doing "stackattrs for '    
 'dicts/mappings".\n'    
 '\n'    
 '*Release 20200517*:\n'    
 '* Add `nullcontext` like the one from recent contextlib.\n'    
 '* stackattrs: expose the push and pop parts as pushattrs() and popattrs().\n'    
 '\n'    
 '*Release 20200228.1*:\n'    
 'Initial release with stackattrs context manager.'),
  classifiers = ['Programming Language :: Python', 'Programming Language :: Python :: 2', 'Programming Language :: Python :: 3', 'Development Status :: 4 - Beta', 'Intended Audience :: Developers', 'Operating System :: OS Independent', 'Topic :: Software Development :: Libraries :: Python Modules', 'License :: OSI Approved :: GNU General Public License v3 or later (GPLv3+)'],
  install_requires = [],
  keywords = ['python2', 'python3'],
  license = 'GNU General Public License v3 or later (GPLv3+)',
  long_description_content_type = 'text/markdown',
  package_dir = {'': 'lib/python'},
  py_modules = ['cs.context'],
)
