# -------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
# --------------------------------------------------------------------------

from abc import ABC, abstractmethod

from hummingbird.ml._utils import pandas_installed
from hummingbird.ml.operator_converters import constants

if pandas_installed():
    from pandas import DataFrame
else:
    DataFrame = None


# Abstract containers enabling the Sklearn API.
class SklearnContainer(ABC):
    def __init__(self, model, n_threads=None, batch_size=None, extra_config={}):
        """
        Base container abstract class allowing to mirror the Sklearn API.
        *SklearnContainer* enables the use of `predict`, `predict_proba` etc. API of Sklearn
        also over the models generated by Hummingbird (irrespective of the selected backend).

        Args:
            model: Any Hummingbird supported model
            n_threads: How many threads should be used by the containter to run the model. None means use all threads.
            batch_size: If different than None, split the input into batch_size partitions and score one partition at a time.
            extra_config: Some additional configuration parameter.
        """
        self._model = model
        self._n_threads = n_threads
        self._extra_config = extra_config
        self._batch_size = batch_size
        self._check_dataframe_to_array = constants.TEST_INPUT not in extra_config

    @property
    def model(self):
        return self._model

    @abstractmethod
    def save(self, location):
        """
        Method used to save the container for future use.

        Args:
            location: The location on the file system where to save the model.
        """
        return

    def _run(self, function, *inputs):
        """
        This function scores the full dataset at once. See BatchContainer below for batched scoring.
        """
        if DataFrame is not None and isinstance(inputs[0], DataFrame):
            if self._check_dataframe_to_array:
                # We were expecting a numpy array as input but we got a dataframe: call values() on it to get the array.
                inputs = [inputs[0].values]
            else:
                # Split the dataframe into column ndarrays.
                inputs = inputs[0]
                input_names = list(inputs.columns)
                splits = [inputs[input_names[idx]] for idx in range(len(input_names))]
                inputs = [df.to_numpy().reshape(-1, 1) for df in splits]

        return function(*inputs)


class SklearnContainerTransformer(SklearnContainer):
    """
    Abstract container mirroring Sklearn transformers API.
    """

    @abstractmethod
    def _transform(self, *input):
        """
        This method contains container-specific implementation of transform.
        """
        pass

    def transform(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On data transformers it returns transformed output data
        """
        return self._run(self._transform, *inputs)


class SklearnContainerRegression(SklearnContainer):
    """
    Abstract container mirroring Sklearn regressors API.
    """

    def __init__(
        self, model, n_threads, batch_size, is_regression=True, is_anomaly_detection=False, extra_config={}, **kwargs
    ):
        super(SklearnContainerRegression, self).__init__(model, n_threads, batch_size, extra_config)

        assert not (is_regression and is_anomaly_detection)

        self._is_regression = is_regression
        self._is_anomaly_detection = is_anomaly_detection

    @abstractmethod
    def _predict(self, *input):
        """
        This method contains container-specific implementation of predict.
        """
        pass

    def predict(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On regression returns the predicted values.
        On classification tasks returns the predicted class labels for the input data.
        On anomaly detection (e.g. isolation forest) returns the predicted classes (-1 or 1).
        """
        return self._run(self._predict, *inputs)


class SklearnContainerClassification(SklearnContainerRegression):
    """
    Container mirroring Sklearn classifiers API.
    """

    def __init__(self, model, n_threads, batch_size, extra_config={}):
        super(SklearnContainerClassification, self).__init__(
            model, n_threads, batch_size, is_regression=False, extra_config=extra_config
        )

    @abstractmethod
    def _predict_proba(self, *input):
        """
        This method contains container-specific implementation of predict_proba.
        """
        pass

    def predict_proba(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On classification tasks returns the probability estimates.
        """
        return self._run(self._predict_proba, *inputs)


class SklearnContainerAnomalyDetection(SklearnContainerRegression):
    """
    Container mirroring Sklearn anomaly detection API.
    """

    def __init__(self, model, n_threads, batch_size, extra_config={}):
        super(SklearnContainerAnomalyDetection, self).__init__(
            model, n_threads, batch_size, is_regression=False, is_anomaly_detection=True, extra_config=extra_config
        )

    @abstractmethod
    def _decision_function(self, *inputs):
        """
        This method contains container-specific implementation of decision_function.
        """
        pass

    def decision_function(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On anomaly detection (e.g. isolation forest) returns the decision function scores.
        """
        scores = self._run(self._decision_function, *inputs)

        # Backward compatibility for sklearn <= 0.21
        if constants.IFOREST_THRESHOLD in self._extra_config:
            scores += self._extra_config[constants.IFOREST_THRESHOLD]
        return scores

    def score_samples(self, *inputs):
        """
        Utility functions used to emulate the behavior of the Sklearn API.
        On anomaly detection (e.g. isolation forest) returns the decision_function score plus offset_
        """
        return self.decision_function(*inputs) + self._extra_config[constants.OFFSET]
