from collections import namedtuple
import pytest

import os

import datetime
import numpy
import h5py
try:
    from osgeo import gdal
except ModuleNotFoundError:
    import gdal

from s100py import s111

path_to_current_file = os.path.realpath(__file__)
current_directory = os.path.dirname(path_to_current_file)


def h5py_string_comp(h5py_val, cmp_str):
    # h5py <3.0 returns a string, >3.0 returns bytes
    return h5py_val in (cmp_str, bytes(cmp_str, "utf-8"))


InputData = namedtuple(
    'InputData',
    ['speed_2d_001',
     'direction_2d_001',
     'speed_2d_002',
     'direction_2d_002',
     'speed_1d',
     'direction_1d',
     'lon',
     'lat',
     'grid_2d_properties',
     'grid_1d_properties',
     'metadata',
     'datetime_value',
     'update_2d_meta',
     'update_1d_meta',
     'expected_2d_chunks',
     'expected_1d_chunks',
     'expected_groupf',
     'expected_georeferenced_coordinates'])


@pytest.fixture
def input_data():

    speed_2d_001 = numpy.array([
        [0.34, 0.35, 0.35, 0.36, 0.37, 0.39, 0.4 , 0.4 , 0.4 , 0.39, 0.39,
            0.4 , 0.41, 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.46, 0.47,
            0.48, 0.49, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.52, 0.52, 0.53,
            0.53, 0.54, 0.54, 0.55, 0.56, 0.57, 0.58, 0.58, 0.59, 0.59, 0.59,
            0.59, 0.6 , 0.61, 0.62, 0.63, 0.64, 0.64, 0.65, 0.66, 0.66],
           [0.34, 0.34, 0.35, 0.35, 0.37, 0.38, 0.39, 0.39, 0.39, 0.39, 0.39,
            0.39, 0.4 , 0.41, 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.46,
            0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.52,
            0.52, 0.53, 0.54, 0.55, 0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.58,
            0.59, 0.6 , 0.61, 0.62, 0.63, 0.64, 0.64, 0.65, 0.66, 0.67],
           [0.34, 0.34, 0.34, 0.34, 0.35, 0.36, 0.37, 0.38, 0.38, 0.38, 0.38,
            0.39, 0.39, 0.4 , 0.41, 0.41, 0.42, 0.43, 0.43, 0.44, 0.45, 0.46,
            0.46, 0.47, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.51, 0.52,
            0.52, 0.52, 0.53, 0.54, 0.55, 0.56, 0.57, 0.57, 0.57, 0.57, 0.58,
            0.59, 0.6 , 0.61, 0.62, 0.63, 0.63, 0.64, 0.65, 0.66, 0.66],
           [0.33, 0.33, 0.33, 0.33, 0.34, 0.35, 0.36, 0.37, 0.37, 0.38, 0.38,
            0.39, 0.39, 0.4 , 0.41, 0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.46,
            0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.51,
            0.52, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.58,
            0.59, 0.6 , 0.61, 0.61, 0.61, 0.62, 0.63, 0.64, 0.65, 0.66],
           [0.32, 0.33, 0.33, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39,
            0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.46,
            0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51,
            0.52, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.56, 0.57, 0.58,
            0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.62, 0.63, 0.64],
           [0.32, 0.32, 0.32, 0.32, 0.32, 0.33, 0.34, 0.36, 0.38, 0.38, 0.38,
            0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.46,
            0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.51,
            0.52, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.57, 0.57,
            0.58, 0.58, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.62, 0.63, 0.64],
           [0.32, 0.32, 0.32, 0.32, 0.32, 0.33, 0.34, 0.36, 0.37, 0.38, 0.38,
            0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.45, 0.45,
            0.46, 0.46, 0.47, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.52,
            0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.56, 0.57, 0.57,
            0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.62, 0.63, 0.64],
           [0.31, 0.31, 0.32, 0.32, 0.32, 0.32, 0.34, 0.35, 0.36, 0.37, 0.37,
            0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45,
            0.45, 0.46, 0.47, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.52,
            0.53, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57,
            0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.61, 0.62, 0.64, 0.65],
           [0.31, 0.31, 0.31, 0.32, 0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.36,
            0.37, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.42, 0.43, 0.43, 0.44, 0.45,
            0.46, 0.46, 0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.52, 0.52, 0.53,
            0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.58,
            0.58, 0.59, 0.59, 0.6 , 0.6 , 0.61, 0.61, 0.62, 0.64, 0.65],
           [0.3 , 0.31, 0.31, 0.32, 0.32, 0.32, 0.32, 0.33, 0.34, 0.35, 0.35,
            0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.42, 0.43, 0.44, 0.45,
            0.46, 0.47, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.52, 0.53, 0.53,
            0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.56, 0.56, 0.57, 0.58, 0.58,
            0.59, 0.59, 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.63, 0.64],
           [0.3 , 0.31, 0.31, 0.32, 0.32, 0.32, 0.33, 0.34, 0.34, 0.35, 0.35,
            0.36, 0.37, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.42, 0.43, 0.44, 0.45,
            0.46, 0.47, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.53,
            0.53, 0.54, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.58, 0.58,
            0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.63, 0.64],
           [0.3 , 0.31, 0.32, 0.32, 0.33, 0.33, 0.34, 0.34, 0.35, 0.35, 0.36,
            0.37, 0.38, 0.39, 0.4 , 0.41, 0.41, 0.42, 0.43, 0.43, 0.44, 0.45,
            0.46, 0.47, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.53,
            0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.56, 0.57, 0.58, 0.59, 0.59,
            0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63],
           [0.31, 0.31, 0.32, 0.32, 0.33, 0.33, 0.34, 0.35, 0.35, 0.36, 0.37,
            0.38, 0.39, 0.4 , 0.41, 0.42, 0.43, 0.44, 0.43, 0.43, 0.44, 0.45,
            0.46, 0.47, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.53,
            0.53, 0.54, 0.54, 0.54, 0.55, 0.55, 0.56, 0.57, 0.58, 0.59, 0.6 ,
            0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.63],
           [0.31, 0.32, 0.32, 0.33, 0.33, 0.34, 0.35, 0.35, 0.36, 0.37, 0.38,
            0.4 , 0.4 , 0.42, 0.43, 0.45, 0.45, 0.45, 0.45, 0.45, 0.45, 0.46,
            0.47, 0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.53, 0.53, 0.53, 0.53,
            0.54, 0.54, 0.54, 0.55, 0.55, 0.56, 0.57, 0.58, 0.58, 0.59, 0.6 ,
            0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.63],
           [0.32, 0.32, 0.32, 0.33, 0.34, 0.35, 0.35, 0.36, 0.37, 0.38, 0.39,
            0.41, 0.42, 0.43, 0.45, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54,
            0.54, 0.54, 0.55, 0.55, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59,
            0.59, 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.62, 0.63, 0.62],
           [0.32, 0.32, 0.33, 0.34, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.41,
            0.42, 0.44, 0.45, 0.46, 0.47, 0.47, 0.47, 0.47, 0.47, 0.47, 0.48,
            0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.53, 0.54, 0.54, 0.54, 0.54,
            0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.58,
            0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.62, 0.62, 0.62],
           [0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.41, 0.42,
            0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.47, 0.47, 0.48, 0.48,
            0.49, 0.5 , 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.54, 0.54, 0.55,
            0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58,
            0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61],
           [0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.42, 0.43,
            0.44, 0.45, 0.45, 0.44, 0.45, 0.45, 0.46, 0.46, 0.47, 0.48, 0.49,
            0.49, 0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.55, 0.55,
            0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.57, 0.58, 0.57, 0.57,
            0.57, 0.58, 0.58, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.61],
           [0.32, 0.34, 0.35, 0.35, 0.36, 0.37, 0.39, 0.4 , 0.41, 0.42, 0.43,
            0.43, 0.42, 0.42, 0.42, 0.42, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49,
            0.49, 0.5 , 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.53, 0.54, 0.55, 0.55,
            0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.58, 0.57,
            0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.61],
           [0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.41, 0.42,
            0.42, 0.4 , 0.39, 0.4 , 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48,
            0.49, 0.49, 0.49, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.54, 0.55, 0.56,
            0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.58, 0.58, 0.57,
            0.57, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.59, 0.6 , 0.61],
           [0.33, 0.34, 0.34, 0.35, 0.36, 0.38, 0.39, 0.4 , 0.4 , 0.41, 0.41,
            0.41, 0.4 , 0.4 , 0.4 , 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48,
            0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.52, 0.53, 0.55, 0.55,
            0.56, 0.57, 0.58, 0.58, 0.58, 0.58, 0.58, 0.59, 0.58, 0.58, 0.57,
            0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.61],
           [0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41,
            0.41, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.45, 0.46, 0.46, 0.47, 0.47,
            0.48, 0.48, 0.48, 0.48, 0.48, 0.48, 0.5 , 0.52, 0.53, 0.54, 0.55,
            0.56, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.58, 0.58, 0.57,
            0.57, 0.58, 0.58, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.61],
           [0.31, 0.32, 0.33, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41,
            0.41, 0.41, 0.41, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.47, 0.48,
            0.48, 0.48, 0.48, 0.48, 0.48, 0.48, 0.5 , 0.51, 0.53, 0.54, 0.55,
            0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.57, 0.57, 0.57,
            0.58, 0.58, 0.58, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.61],
           [0.31, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.39, 0.39, 0.4 , 0.4 ,
            0.4 , 0.41, 0.42, 0.42, 0.44, 0.45, 0.46, 0.47, 0.47, 0.48, 0.48,
            0.48, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.52, 0.53, 0.54,
            0.55, 0.55, 0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57,
            0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.61],
           [0.31, 0.32, 0.33, 0.34, 0.34, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.4 ,
            0.41, 0.41, 0.42, 0.43, 0.45, 0.45, 0.46, 0.47, 0.48, 0.48, 0.48,
            0.48, 0.49, 0.49, 0.49, 0.48, 0.48, 0.48, 0.49, 0.51, 0.52, 0.53,
            0.54, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57,
            0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.59, 0.59, 0.6 , 0.61, 0.61],
           [0.31, 0.32, 0.33, 0.34, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 , 0.4 ,
            0.41, 0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.47, 0.48, 0.48, 0.48,
            0.48, 0.49, 0.49, 0.48, 0.48, 0.47, 0.47, 0.48, 0.5 , 0.51, 0.52,
            0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.58,
            0.59, 0.6 , 0.61, 0.61, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 ],
           [0.32, 0.32, 0.33, 0.34, 0.34, 0.35, 0.37, 0.38, 0.4 , 0.4 , 0.41,
            0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.48, 0.48,
            0.48, 0.49, 0.49, 0.48, 0.48, 0.47, 0.46, 0.47, 0.49, 0.5 , 0.52,
            0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.58, 0.58,
            0.6 , 0.61, 0.62, 0.61, 0.61, 0.61, 0.6 , 0.6 , 0.6 , 0.6 ],
           [0.32, 0.33, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.4 , 0.41, 0.41,
            0.42, 0.42, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49,
            0.49, 0.49, 0.49, 0.48, 0.47, 0.47, 0.46, 0.47, 0.48, 0.5 , 0.51,
            0.53, 0.53, 0.54, 0.54, 0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59,
            0.6 , 0.61, 0.62, 0.62, 0.61, 0.61, 0.61, 0.61, 0.6 , 0.6 ],
           [0.32, 0.33, 0.33, 0.34, 0.35, 0.36, 0.37, 0.39, 0.4 , 0.41, 0.41,
            0.41, 0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.49, 0.49,
            0.49, 0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.47, 0.49, 0.51,
            0.52, 0.53, 0.54, 0.54, 0.55, 0.56, 0.56, 0.56, 0.57, 0.58, 0.59,
            0.6 , 0.61, 0.61, 0.62, 0.61, 0.61, 0.61, 0.6 , 0.6 , 0.6 ],
           [0.32, 0.33, 0.34, 0.34, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.4 ,
            0.4 , 0.4 , 0.41, 0.41, 0.42, 0.43, 0.44, 0.46, 0.47, 0.49, 0.49,
            0.49, 0.49, 0.48, 0.48, 0.47, 0.46, 0.46, 0.45, 0.47, 0.49, 0.5 ,
            0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.58,
            0.59, 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.6 , 0.6 , 0.6 , 0.6 ],
           [0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.39, 0.39, 0.39,
            0.39, 0.39, 0.4 , 0.41, 0.42, 0.43, 0.44, 0.46, 0.47, 0.49, 0.49,
            0.49, 0.49, 0.48, 0.48, 0.47, 0.46, 0.46, 0.45, 0.46, 0.47, 0.49,
            0.5 , 0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.56, 0.57,
            0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 ],
           [0.32, 0.33, 0.34, 0.35, 0.36, 0.36, 0.37, 0.37, 0.38, 0.38, 0.38,
            0.38, 0.38, 0.39, 0.4 , 0.42, 0.43, 0.45, 0.46, 0.47, 0.49, 0.5 ,
            0.49, 0.49, 0.48, 0.48, 0.47, 0.46, 0.46, 0.45, 0.46, 0.47, 0.48,
            0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56,
            0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 ],
           [0.31, 0.32, 0.33, 0.34, 0.35, 0.36, 0.36, 0.36, 0.36, 0.36, 0.36,
            0.37, 0.37, 0.38, 0.4 , 0.41, 0.43, 0.45, 0.46, 0.47, 0.48, 0.5 ,
            0.49, 0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55,
            0.56, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 ],
           [0.31, 0.32, 0.33, 0.34, 0.34, 0.34, 0.35, 0.34, 0.34, 0.35, 0.35,
            0.36, 0.37, 0.38, 0.39, 0.41, 0.43, 0.44, 0.46, 0.47, 0.48, 0.49,
            0.49, 0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.55,
            0.56, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.61, 0.61],
           [0.3 , 0.31, 0.32, 0.32, 0.33, 0.33, 0.33, 0.33, 0.33, 0.34, 0.34,
            0.35, 0.36, 0.37, 0.39, 0.4 , 0.42, 0.44, 0.45, 0.47, 0.48, 0.49,
            0.49, 0.48, 0.48, 0.47, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55,
            0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 ],
           [0.29, 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.32, 0.32, 0.33, 0.33,
            0.34, 0.35, 0.36, 0.38, 0.4 , 0.42, 0.43, 0.45, 0.46, 0.47, 0.48,
            0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.55,
            0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.59],
           [0.28, 0.29, 0.29, 0.29, 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.32, 0.33,
            0.33, 0.34, 0.36, 0.37, 0.39, 0.41, 0.43, 0.44, 0.45, 0.46, 0.47,
            0.48, 0.48, 0.47, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55,
            0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59],
           [0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.3 , 0.3 , 0.3 , 0.31, 0.32,
            0.32, 0.34, 0.35, 0.37, 0.38, 0.4 , 0.42, 0.44, 0.45, 0.46, 0.46,
            0.47, 0.47, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.53, 0.54,
            0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.58, 0.58],
           [0.27, 0.28, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.3 , 0.3 , 0.31,
            0.32, 0.33, 0.35, 0.36, 0.38, 0.4 , 0.41, 0.43, 0.44, 0.45, 0.46,
            0.46, 0.47, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.53, 0.54,
            0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.57, 0.57],
           [0.27, 0.27, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.3 , 0.31,
            0.31, 0.33, 0.34, 0.36, 0.37, 0.39, 0.4 , 0.42, 0.43, 0.44, 0.45,
            0.45, 0.46, 0.46, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
            0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.54,
            0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.56],
           [0.27, 0.27, 0.28, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.3 ,
            0.31, 0.32, 0.33, 0.35, 0.36, 0.38, 0.39, 0.41, 0.42, 0.43, 0.44,
            0.44, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46,
            0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53,
            0.54, 0.55, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56],
           [0.27, 0.27, 0.28, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.3 ,
            0.31, 0.31, 0.33, 0.34, 0.35, 0.37, 0.38, 0.4 , 0.41, 0.42, 0.43,
            0.43, 0.44, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46,
            0.47, 0.47, 0.47, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.53,
            0.54, 0.55, 0.55, 0.55, 0.56, 0.56, 0.56, 0.56, 0.55, 0.55],
           [0.26, 0.27, 0.28, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.3 ,
            0.3 , 0.31, 0.32, 0.33, 0.35, 0.36, 0.37, 0.39, 0.4 , 0.41, 0.42,
            0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46,
            0.47, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52,
            0.53, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.26, 0.27, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.3 ,
            0.3 , 0.31, 0.32, 0.33, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 , 0.41,
            0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46,
            0.47, 0.47, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.51, 0.52,
            0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.26, 0.27, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29, 0.29, 0.3 ,
            0.3 , 0.31, 0.31, 0.32, 0.34, 0.35, 0.36, 0.38, 0.39, 0.4 , 0.41,
            0.42, 0.43, 0.44, 0.44, 0.44, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46,
            0.46, 0.46, 0.47, 0.47, 0.47, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.51,
            0.53, 0.54, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.26, 0.27, 0.28, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29,
            0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.37, 0.38, 0.39, 0.4 ,
            0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46,
            0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.51,
            0.52, 0.53, 0.54, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.25, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29,
            0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39,
            0.41, 0.42, 0.43, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46,
            0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.51,
            0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.25, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29,
            0.3 , 0.3 , 0.3 , 0.31, 0.32, 0.34, 0.35, 0.36, 0.37, 0.39, 0.39,
            0.4 , 0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.46, 0.46,
            0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.5 ,
            0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.24, 0.25, 0.26, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29,
            0.29, 0.3 , 0.3 , 0.31, 0.32, 0.34, 0.35, 0.36, 0.37, 0.39, 0.39,
            0.4 , 0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.45, 0.46,
            0.46, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.5 ,
            0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55],
           [0.24, 0.25, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29,
            0.29, 0.3 , 0.3 , 0.31, 0.32, 0.33, 0.35, 0.36, 0.38, 0.38, 0.39,
            0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.45,
            0.46, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49,
            0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55],
           [0.24, 0.25, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29,
            0.29, 0.3 , 0.3 , 0.31, 0.32, 0.34, 0.35, 0.37, 0.38, 0.38, 0.39,
            0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.45,
            0.45, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49,
            0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55],
           [0.24, 0.24, 0.25, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29,
            0.29, 0.29, 0.3 , 0.31, 0.32, 0.34, 0.35, 0.37, 0.38, 0.38, 0.39,
            0.4 , 0.41, 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45,
            0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
            0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55],
           [0.24, 0.24, 0.25, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29, 0.29,
            0.29, 0.29, 0.3 , 0.31, 0.32, 0.34, 0.35, 0.37, 0.38, 0.38, 0.39,
            0.4 , 0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.44, 0.44, 0.45,
            0.45, 0.45, 0.45, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
            0.5 , 0.51, 0.51, 0.52, 0.53, 0.54, 0.55, 0.56, 0.55, 0.56],
           [0.23, 0.24, 0.25, 0.26, 0.26, 0.27, 0.28, 0.28, 0.28, 0.29, 0.29,
            0.29, 0.29, 0.3 , 0.31, 0.32, 0.34, 0.35, 0.37, 0.38, 0.38, 0.39,
            0.4 , 0.4 , 0.41, 0.42, 0.42, 0.43, 0.43, 0.44, 0.44, 0.44, 0.44,
            0.45, 0.45, 0.45, 0.45, 0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
            0.5 , 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.56, 0.56, 0.56]
        ])

    direction_2d_001 = numpy.array([
        [189.1, 188.6, 188.1, 188.8, 189.7, 190.6, 191.5, 192. , 192.3,
            192.6, 192.9, 192.8, 192.9, 193.1, 193.4, 193.5, 193.4, 193.6,
            193.6, 193.8, 193.9, 193.8, 193.6, 193.4, 193.1, 193. , 192.9,
            192.8, 192.7, 192.9, 193.1, 193.3, 193.6, 193.7, 194.1, 194.6,
            195. , 195.4, 195.7, 195.8, 195.7, 195.4, 195.1, 194.8, 194.5,
            194.3, 194.1, 194. , 194. , 194. , 194.1, 194.1, 194.1, 194.2],
           [189. , 188.7, 188.7, 189.2, 190.3, 191.2, 191.8, 192.2, 192.5,
            192.7, 193.2, 193.7, 194. , 194.2, 194.5, 194.7, 194.7, 194.7,
            194.9, 195. , 194.9, 194.7, 194.4, 194. , 193.6, 193.5, 193.4,
            193.4, 193.3, 193.4, 193.6, 193.8, 194. , 194.2, 194.5, 194.9,
            195.3, 195.7, 196.1, 196. , 195.8, 195.4, 195. , 194.6, 194.5,
            194.3, 194.2, 194. , 194. , 194. , 194. , 194.1, 194.1, 194.2],
           [189.6, 189.6, 189.6, 189.7, 190.7, 191.3, 191.8, 192.5, 192.9,
            193.5, 194.2, 195. , 195.4, 195.6, 195.7, 195.7, 195.8, 195.8,
            195.9, 195.9, 195.8, 195.5, 195.1, 194.7, 194.3, 194.1, 194. ,
            193.9, 194. , 193.9, 194.1, 194.2, 194.4, 194.6, 194.9, 195.4,
            195.8, 196.1, 196.3, 196.2, 195.8, 195.4, 195. , 194.7, 194.6,
            194.5, 194.5, 194.2, 194. , 194. , 194.1, 194.1, 194.1, 194.2],
           [190.4, 190.4, 190.4, 190.3, 190.7, 191.4, 192.2, 193.1, 194.1,
            194.8, 195.7, 196.5, 196.6, 196.6, 196.6, 196.6, 196.6, 196.6,
            196.6, 196.6, 196.5, 196.2, 195.8, 195.4, 194.9, 194.6, 194.5,
            194.4, 194.4, 194.3, 194.4, 194.6, 194.8, 195.1, 195.5, 195.8,
            196.1, 196.3, 196.3, 196.2, 195.9, 195.6, 195.3, 195.2, 195.2,
            195. , 194.7, 194.4, 194. , 193.9, 193.9, 194. , 194.1, 194.3],
           [191. , 190.9, 190.8, 190.7, 191. , 192. , 193.2, 194.4, 195.4,
            196.1, 196.7, 197.1, 197.4, 197.2, 197. , 197.2, 197.3, 197.3,
            197.3, 197.2, 197.1, 196.9, 196.4, 195.9, 195.4, 195. , 194.8,
            194.8, 194.7, 194.7, 194.8, 195. , 195.3, 195.6, 195.9, 196.2,
            196.2, 196.2, 196.3, 196.3, 196.1, 196. , 195.8, 195.7, 195.6,
            195.3, 194.9, 194.5, 194.1, 193.9, 194. , 194.2, 194.4, 194.6],
           [191.4, 191.4, 191.4, 191.7, 191.9, 193.2, 194.4, 195.5, 196.5,
            197. , 197.4, 197.5, 197.5, 197.5, 197.5, 197.6, 197.9, 198. ,
            197.9, 197.8, 197.5, 197.3, 196.8, 196.3, 195.8, 195.4, 195.2,
            195.1, 195.1, 195.1, 195.2, 195.4, 195.7, 195.9, 196.1, 196.3,
            196.3, 196.3, 196.4, 196.4, 196.4, 196.3, 196.1, 196. , 195.8,
            195.4, 195. , 194.6, 194.3, 194. , 194.2, 194.3, 194.6, 194.9],
           [192.1, 192.2, 192.4, 192.9, 193.5, 194.5, 195.9, 196.8, 197.3,
            197.7, 197.7, 197.8, 197.7, 197.6, 197.9, 198.2, 198.4, 198.7,
            198.5, 198.2, 197.9, 197.5, 197. , 196.6, 196.1, 195.7, 195.5,
            195.5, 195.5, 195.4, 195.4, 195.5, 195.7, 195.9, 196.1, 196.3,
            196.3, 196.4, 196.5, 196.5, 196.5, 196.4, 196.2, 196. , 195.8,
            195.6, 195.2, 194.9, 194.6, 194.3, 194.3, 194.5, 194.7, 195.1],
           [193. , 193.2, 193.7, 194.4, 195.5, 196.5, 197.3, 198.1, 198.2,
            198.3, 198.2, 198.3, 198.3, 198.3, 198.5, 198.9, 199.1, 199.1,
            199. , 198.6, 198.2, 197.7, 197.1, 196.6, 196.3, 195.9, 195.5,
            195.3, 195.3, 195.4, 195.6, 195.7, 195.9, 196.1, 196.1, 196.2,
            196.3, 196.4, 196.4, 196.5, 196.4, 196.3, 196. , 195.9, 195.9,
            195.8, 195.4, 195. , 194.7, 194.3, 194.2, 194.6, 195.1, 195.6],
           [194.4, 194.8, 195.4, 196.4, 197.4, 198.3, 198.9, 199.1, 199.3,
            199.3, 199.3, 199.3, 199.3, 199.3, 199.5, 199.6, 199.8, 199.8,
            199.5, 199. , 198.4, 197.9, 197. , 196.4, 196. , 195.8, 195.6,
            195.6, 195.9, 195.9, 196. , 195.9, 195.9, 196. , 196.1, 196.2,
            196.3, 196.3, 196.3, 196.2, 196.2, 196.1, 196. , 195.9, 195.8,
            195.7, 195.4, 195.1, 194.8, 194.7, 194.7, 195.2, 195.6, 195.9],
           [196.3, 196.9, 197.4, 198.2, 199.1, 199.8, 200.5, 200.5, 200.4,
            200.5, 200.5, 200.5, 200.6, 200.6, 200.8, 200.9, 200.6, 200.6,
            200. , 199.4, 198.6, 197.8, 197.2, 196.7, 196.6, 196.5, 196.4,
            196.2, 196.1, 196.1, 196. , 196. , 195.9, 196.1, 196.2, 196.1,
            196.1, 196.1, 196.1, 196.1, 196.1, 196. , 195.9, 195.8, 195.7,
            195.7, 195.7, 195.5, 195.5, 195.3, 195.3, 195.6, 195.9, 196. ],
           [198.4, 198.9, 199.2, 199.8, 200.5, 201.1, 201.6, 201.7, 201.7,
            201.7, 202. , 202. , 202.1, 202. , 202.3, 202.3, 201.8, 201.2,
            200.7, 200. , 199.3, 198.7, 197.9, 197.3, 197.1, 196.9, 196.6,
            196.5, 196.3, 196.3, 196.3, 196.2, 196.1, 196.1, 196.1, 196.2,
            196.2, 196.1, 196.1, 196.1, 196.1, 196. , 195.9, 195.9, 195.9,
            195.8, 195.7, 195.6, 195.5, 195.6, 195.7, 195.8, 196. , 196.1],
           [200.4, 200.7, 201. , 201.2, 201.8, 202.4, 202.6, 202.8, 202.9,
            202.9, 203.1, 203.1, 203.3, 203.4, 203.6, 203.5, 202.9, 202.2,
            201.3, 200.5, 199.6, 198.9, 198.3, 197.8, 197.5, 197.3, 197.1,
            196.8, 196.6, 196.5, 196.4, 196.3, 196.3, 196.2, 196.3, 196.3,
            196.3, 196.3, 196.2, 196.2, 196.2, 196.1, 196. , 196. , 195.9,
            195.8, 195.6, 195.5, 195.4, 195.5, 195.7, 195.9, 196. , 196.1],
           [202.2, 202.5, 202.7, 202.8, 203.1, 203.4, 203.5, 203.5, 203.6,
            203.7, 203.9, 204.1, 204.2, 204.3, 204.3, 203.8, 202.9, 202.1,
            201.3, 200.6, 199.8, 199.1, 198.7, 198.3, 197.9, 197.6, 197.3,
            197. , 196.8, 196.7, 196.5, 196.4, 196.5, 196.5, 196.5, 196.5,
            196.5, 196.5, 196.5, 196.5, 196.3, 196.3, 196. , 195.9, 195.8,
            195.5, 195.3, 195.1, 195.2, 195.3, 195.6, 195.8, 196. , 196.1],
           [204. , 204.1, 204.2, 204.2, 204.2, 204.2, 204.3, 204.2, 204.3,
            204.4, 204.5, 204.4, 204.4, 204.1, 203.8, 203.6, 203.1, 202.4,
            201.5, 200.7, 199.8, 199.1, 198.6, 198.3, 198. , 197.6, 197.3,
            197. , 196.8, 196.7, 196.6, 196.6, 196.6, 196.7, 196.7, 196.7,
            196.8, 196.8, 197. , 196.9, 196.7, 196.4, 196.2, 195.8, 195.6,
            195.3, 195. , 194.7, 194.9, 195.2, 195.4, 195.7, 195.8, 195.9],
           [205.3, 205.4, 205.4, 205.3, 205.1, 205.1, 205. , 204.8, 204.6,
            204.6, 204.6, 204.6, 204.7, 204.5, 204.2, 203.8, 203.1, 202.2,
            201.3, 200.3, 199.3, 198.8, 198.4, 198. , 197.7, 197.4, 197. ,
            196.7, 196.6, 196.6, 196.5, 196.6, 196.6, 196.8, 196.9, 197. ,
            197.2, 197.3, 197.4, 197.3, 197. , 196.6, 196.1, 195.7, 195.4,
            195.2, 194.9, 194.6, 194.7, 194.9, 195.2, 195.4, 195.7, 195.8],
           [206.2, 206.5, 206.2, 206. , 205.9, 205.7, 205.5, 205.3, 204.8,
            204.7, 205. , 205.1, 205.1, 204.8, 204.2, 203.5, 202.7, 201.6,
            200.5, 199.7, 198.9, 198.3, 197.9, 197.5, 197.1, 196.8, 196.6,
            196.4, 196.3, 196.3, 196.3, 196.5, 196.7, 196.9, 197.2, 197.4,
            197.5, 197.6, 197.7, 197.6, 197.2, 196.7, 196.2, 195.7, 195.4,
            195.1, 194.9, 194.6, 194.5, 194.8, 195.1, 195.4, 195.6, 195.8],
           [206.7, 207. , 206.9, 206.7, 206.5, 206.3, 205.9, 205.5, 205.2,
            205.1, 205.2, 205.4, 205.4, 204.9, 204.3, 203. , 201.9, 200.7,
            199.7, 198.8, 198.2, 197.6, 197.2, 196.9, 196.5, 196.2, 196. ,
            196. , 196.1, 196.1, 196.2, 196.4, 196.7, 197.1, 197.4, 197.5,
            197.6, 197.7, 197.8, 197.8, 197.3, 196.8, 196.3, 195.9, 195.7,
            195.5, 195.1, 194.7, 194.4, 194.7, 195. , 195.3, 195.6, 195.8],
           [207.2, 207.5, 207.4, 207.1, 206.9, 206.7, 206.3, 206. , 205.9,
            206.2, 206.4, 206.1, 205.9, 205.1, 204.1, 202.6, 200.9, 199.7,
            199.2, 198.7, 197.9, 197.1, 196.7, 196.4, 196.2, 195.9, 195.8,
            195.9, 195.9, 196. , 196. , 196.2, 196.5, 196.9, 197.3, 197.5,
            197.5, 197.6, 197.6, 197.8, 197.4, 197. , 196.7, 196.3, 196. ,
            195.6, 195.3, 194.9, 194.5, 194.7, 195. , 195.4, 195.7, 196. ],
           [207.4, 207.5, 207.6, 207.4, 207.2, 207.1, 206.8, 207.1, 207.2,
            207.5, 207.5, 207. , 206.3, 205.6, 204.2, 202.8, 201.4, 200.1,
            199.3, 198.5, 197.8, 197.1, 196.7, 196.5, 196.2, 195.9, 195.6,
            195.6, 195.6, 195.7, 195.7, 196. , 196.4, 196.9, 197. , 197.2,
            197.2, 197.4, 197.5, 197.6, 197.4, 196.9, 196.6, 196.4, 196.2,
            195.8, 195.5, 195. , 194.7, 194.9, 195.2, 195.5, 195.9, 196.2],
           [207.8, 207.9, 207.8, 207.7, 207.5, 207.5, 207.6, 207.9, 208.1,
            208.2, 207.8, 207. , 206.6, 206.3, 204.8, 203.1, 201.6, 200.4,
            199.4, 198.7, 198. , 197.4, 196.9, 196.6, 196.2, 195.7, 195.4,
            195.6, 195.8, 195.9, 195.9, 196.1, 196.4, 196.6, 196.8, 196.9,
            196.9, 197. , 197.1, 197.1, 197. , 196.8, 196.5, 196.2, 196.1,
            195.7, 195.3, 195.1, 195. , 195.1, 195.4, 195.7, 196.1, 196.3],
           [208.3, 208.2, 208.1, 208. , 208. , 208. , 207.9, 208.1, 207.9,
            207.9, 207.6, 207. , 206.4, 205.6, 204.5, 203. , 201.6, 200.6,
            199.7, 199. , 198.3, 197.9, 197.4, 197.2, 197. , 196.6, 196.2,
            196.1, 196.1, 196. , 196.1, 196.2, 196.5, 196.6, 196.6, 196.6,
            196.6, 196.7, 196.8, 196.7, 196.5, 196.2, 196. , 195.9, 195.8,
            195.7, 195.5, 195.4, 195.3, 195.3, 195.7, 196. , 196.2, 196.4],
           [208.8, 208.8, 208.5, 208.2, 208.2, 208.3, 208.1, 208. , 207.8,
            207.5, 207.3, 206.7, 206. , 205.2, 204.3, 203. , 202. , 201.1,
            200.7, 200.3, 199.9, 199.3, 198.7, 197.9, 197.3, 196.6, 196.2,
            196. , 196. , 196.1, 196.3, 196.4, 196.4, 196.5, 196.5, 196.5,
            196.5, 196.6, 196.3, 196.1, 195.9, 195.8, 195.8, 195.7, 195.7,
            195.8, 195.6, 195.6, 195.6, 195.7, 195.9, 196.1, 196.4, 196.6],
           [209.6, 209.4, 209.2, 209. , 208.9, 208.6, 208.4, 208.1, 207.9,
            207.5, 207.1, 206.6, 205.9, 205.2, 204.7, 203.9, 203.2, 202.8,
            202. , 201. , 200.1, 199.1, 198.1, 197.3, 196.8, 196.5, 196.1,
            195.9, 196. , 196.1, 196.2, 196.3, 196.4, 196.4, 196.4, 196.4,
            196.4, 196.3, 196.2, 196. , 195.8, 195.7, 195.6, 195.7, 195.8,
            195.8, 195.9, 195.9, 196. , 196.2, 196.4, 196.7, 196.9, 196.8],
           [210.7, 210.3, 209.9, 209.7, 209.5, 209.2, 208.8, 208.4, 208. ,
            207.7, 207.4, 207. , 206.4, 205.9, 205.2, 204.4, 203.6, 202.6,
            201.5, 200.5, 199.5, 198.5, 197.6, 196.9, 196.5, 196.2, 196.1,
            195.9, 195.9, 196. , 196.1, 196.2, 196.3, 196.2, 196.2, 196.3,
            196.3, 196.3, 196.1, 195.9, 195.8, 195.7, 195.8, 195.9, 196. ,
            196.1, 196.2, 196.3, 196.5, 196.7, 197. , 197.1, 197.1, 197. ],
           [211. , 210.9, 210.6, 210.3, 210.2, 209.8, 209.3, 208.8, 208.3,
            208. , 207.6, 207.5, 207. , 206.1, 205.2, 204.3, 203.1, 202. ,
            200.9, 199.8, 198.8, 198. , 197.3, 196.6, 196.3, 196.1, 195.9,
            195.7, 195.7, 195.8, 195.9, 196. , 196.1, 196.1, 196.2, 196.3,
            196.4, 196.4, 196.3, 196.2, 196.1, 196.1, 196.2, 196.3, 196.4,
            196.5, 196.5, 196.8, 197. , 197.2, 197.5, 197.5, 197.4, 197.3],
           [211.3, 211.3, 211.1, 210.9, 210.6, 210.1, 209.5, 209. , 208.5,
            208.3, 208. , 207.7, 206.9, 206. , 204.9, 203.7, 202.5, 201.3,
            200.2, 199. , 198.2, 197.4, 196.8, 196.4, 196.1, 195.9, 195.7,
            195.6, 195.5, 195.6, 195.7, 195.8, 196. , 196.2, 196.3, 196.4,
            196.6, 196.7, 196.6, 196.5, 196.5, 196.5, 196.5, 196.5, 196.5,
            196.6, 196.8, 197. , 197.2, 197.5, 197.8, 197.9, 197.7, 197.5],
           [211.5, 211.5, 211.2, 211. , 210.7, 210.3, 209.7, 209.1, 208.7,
            208.4, 207.9, 207.4, 206.7, 205.4, 204.2, 203. , 201.7, 200.5,
            199.4, 198.4, 197.5, 197. , 196.6, 196.1, 195.8, 195.6, 195.4,
            195.3, 195.3, 195.4, 195.5, 195.8, 196.1, 196.3, 196.5, 196.6,
            196.8, 196.9, 196.8, 196.7, 196.6, 196.5, 196.5, 196.5, 196.7,
            196.8, 196.9, 197.2, 197.5, 197.7, 197.9, 197.8, 197.6, 197.3],
           [211.5, 211.5, 211.2, 211. , 210.7, 210.4, 209.8, 209.3, 208.8,
            208.1, 207.6, 206.9, 206. , 204.8, 203.5, 202.1, 200.9, 199.7,
            198.7, 197.6, 196.9, 196.5, 196. , 195.7, 195.3, 195.1, 195.1,
            195.1, 195.1, 195.3, 195.7, 196.1, 196.5, 196.7, 196.9, 197.1,
            197. , 196.9, 196.8, 196.7, 196.6, 196.6, 196.5, 196.6, 196.7,
            196.8, 196.8, 197. , 197.2, 197.3, 197.5, 197.6, 197.5, 197.4],
           [211.4, 211.3, 211.2, 210.9, 210.6, 210.5, 210. , 209.3, 208.5,
            207.8, 207. , 206.1, 205.2, 203.9, 202.5, 201.2, 200. , 198.9,
            197.9, 197.1, 196.4, 195.9, 195.6, 195.2, 194.9, 194.8, 194.8,
            194.8, 195.4, 195.9, 196.4, 196.7, 196.9, 197.2, 197.2, 197.1,
            197.1, 197. , 196.8, 196.8, 196.7, 196.6, 196.5, 196.5, 196.6,
            196.8, 196.9, 197. , 197.3, 197.5, 197.6, 197.7, 197.6, 197.6],
           [211.2, 211.1, 211. , 210.7, 210.6, 210.4, 209.9, 209.1, 208.3,
            207.3, 206.3, 205.4, 204.1, 202.8, 201.4, 200.3, 199.1, 198.2,
            197.3, 196.5, 195.9, 195.4, 195. , 194.8, 194.6, 194.5, 194.8,
            195.4, 196. , 196.6, 196.8, 197.1, 197.3, 197.5, 197.6, 197.4,
            197.3, 197.1, 197. , 197. , 197. , 197. , 196.9, 197. , 197.1,
            197.2, 197.2, 197.3, 197.5, 197.6, 197.8, 197.9, 197.8, 197.7],
           [210.9, 210.8, 210.7, 210.5, 210.4, 210.1, 209.7, 208.8, 207.7,
            206.6, 205.5, 204.2, 202.9, 201.6, 200.4, 199.1, 198.2, 197.3,
            196.5, 195.9, 195.3, 194.8, 194.7, 194.4, 194.2, 194.2, 194.7,
            195.2, 196. , 196.8, 197.1, 197.4, 197.7, 197.9, 198.1, 198. ,
            197.9, 197.8, 197.7, 197.7, 197.6, 197.5, 197.5, 197.6, 197.7,
            197.8, 197.8, 197.6, 197.8, 198. , 198. , 198. , 198. , 197.9],
           [210.5, 210.4, 210.4, 210.3, 210. , 209.7, 209.2, 207.9, 206.7,
            205.4, 204.1, 202.8, 201.5, 200.3, 199.1, 198.2, 197.3, 196.5,
            195.8, 195.2, 194.6, 194. , 193.8, 193.7, 193.7, 193.8, 194.5,
            195.3, 196.1, 197.1, 197.7, 198. , 198.3, 198.6, 198.9, 198.7,
            198.6, 198.5, 198.5, 198.5, 198.5, 198.5, 198.4, 198.5, 198.6,
            198.5, 198.3, 198.1, 198.2, 198.2, 198.2, 198.2, 198.2, 198.2],
           [210.2, 210.2, 210.1, 209.8, 209.5, 208.7, 207.7, 206.5, 205.2,
            203.9, 202.7, 201.5, 200.3, 199.6, 198.7, 197.8, 196.8, 195.8,
            195.1, 194.4, 193.8, 193.3, 193.2, 193.2, 193.5, 194. , 194.6,
            195.5, 196.5, 197.5, 198.3, 198.6, 199. , 199.3, 199.6, 199.6,
            199.4, 199.4, 199.5, 199.5, 199.5, 199.4, 199.4, 199.4, 199.2,
            198.9, 198.7, 198.5, 198.4, 198.4, 198.5, 198.4, 198.4, 198.5],
           [210.1, 209.8, 209.5, 209.1, 208.2, 207.3, 206.2, 204.9, 203.6,
            202.6, 201.7, 200.7, 200. , 199.2, 198.3, 197.2, 196.2, 195.2,
            194.4, 193.8, 193.3, 192.8, 192.8, 193.2, 193.6, 194.1, 195. ,
            195.9, 196.9, 198. , 199. , 199.3, 199.7, 200. , 200.4, 200.4,
            200.4, 200.5, 200.4, 200.2, 200.1, 199.9, 199.8, 199.7, 199.5,
            199.2, 199. , 198.9, 198.8, 198.8, 198.7, 198.7, 198.6, 198.7],
           [209.9, 209.6, 208.9, 208.2, 207.1, 205.8, 204.7, 203.5, 202.6,
            201.6, 200.9, 200.1, 199.3, 198.5, 197.8, 196.7, 195.6, 194.7,
            193.9, 193.4, 192.9, 192.7, 192.7, 193.1, 193.7, 194.5, 195.3,
            196.3, 197.4, 198.5, 199.6, 200.1, 200.4, 200.8, 201. , 201. ,
            200.9, 200.8, 200.6, 200.4, 200.3, 200.1, 200. , 199.9, 199.7,
            199.5, 199.4, 199.2, 199.1, 199.1, 199. , 198.9, 198.8, 198.8],
           [209.8, 209.1, 208.4, 207.5, 206.4, 205.2, 203.9, 202.6, 201.7,
            200.9, 200.2, 199.5, 198.8, 198.1, 197.4, 196.3, 195.3, 194.4,
            193.6, 193.1, 192.9, 192.7, 192.6, 193.2, 194. , 194.9, 195.9,
            197. , 198.2, 199.3, 200.3, 200.9, 201.2, 201.4, 201.4, 201.3,
            201.2, 201. , 200.8, 200.7, 200.5, 200.4, 200.2, 200.1, 200. ,
            199.8, 199.7, 199.5, 199.4, 199.3, 199.3, 199.1, 198.8, 198.7],
           [209.7, 209.1, 208. , 206.9, 205.8, 204.5, 203.3, 202.1, 201.3,
            200.5, 199.9, 199.2, 198.5, 197.7, 197. , 196.1, 195.1, 194.2,
            193.4, 193.1, 193. , 192.8, 192.9, 193.7, 194.5, 195.5, 196.8,
            197.9, 199.1, 200. , 201. , 201.7, 201.8, 201.8, 201.7, 201.7,
            201.5, 201.3, 201.1, 200.9, 200.6, 200.5, 200.3, 200.3, 200.2,
            200. , 199.9, 199.8, 199.7, 199.6, 199.5, 199.2, 198.9, 198.7],
           [209.4, 208.6, 207.5, 206.4, 205.5, 204.2, 203. , 201.9, 201. ,
            200.2, 199.6, 198.9, 198.2, 197.5, 196.9, 196.1, 195.1, 194.2,
            193.7, 193.3, 193.1, 193.3, 193.5, 194.1, 195.2, 196.4, 197.7,
            198.8, 199.7, 200.7, 201.7, 202.3, 202.3, 202.2, 202.2, 202. ,
            201.8, 201.6, 201.4, 201. , 200.8, 200.6, 200.5, 200.5, 200.4,
            200.2, 200.2, 200.1, 200. , 199.9, 199.5, 199.2, 198.9, 198.7],
           [208.7, 208. , 207. , 206. , 205. , 203.9, 202.8, 201.7, 201. ,
            200.3, 199.8, 199.2, 198.5, 197.8, 197. , 196.2, 195.2, 194.5,
            193.9, 193.5, 193.7, 193.9, 194.1, 194.8, 196. , 197.2, 198.4,
            199.5, 200.5, 201.4, 202.2, 202.8, 202.8, 202.7, 202.5, 202.4,
            202.1, 201.9, 201.5, 201.2, 200.9, 200.8, 200.7, 200.7, 200.6,
            200.5, 200.4, 200.3, 200.2, 200. , 199.7, 199.4, 199.1, 199. ],
           [208. , 207.5, 206.6, 205.6, 204.8, 203.9, 202.8, 201.9, 201.3,
            200.6, 200.1, 199.5, 198.9, 197.9, 197.1, 196.3, 195.6, 195. ,
            194.4, 194.1, 194.3, 194.5, 194.8, 195.4, 196.7, 197.9, 199.1,
            200.3, 201.2, 201.9, 202.5, 203.1, 203.3, 203.1, 202.9, 202.7,
            202.4, 202. , 201.6, 201.3, 201. , 200.9, 200.9, 200.8, 200.7,
            200.6, 200.5, 200.4, 200.3, 200.1, 199.8, 199.5, 199.3, 199.2],
           [207.1, 206.9, 206.2, 205.3, 204.6, 203.9, 203. , 202.3, 201.6,
            201. , 200.4, 199.9, 199.5, 198.6, 197.8, 197.1, 196.4, 195.8,
            195.2, 194.8, 194.9, 195.2, 195.7, 196.1, 197.4, 198.6, 199.8,
            201. , 201.7, 202.3, 202.9, 203.5, 203.4, 203.2, 202.9, 202.7,
            202.4, 202. , 201.6, 201.3, 201.2, 201.1, 201. , 201. , 200.9,
            200.8, 200.7, 200.6, 200.4, 200.3, 200. , 199.7, 199.6, 199.5],
           [206.5, 206.3, 205.8, 205.1, 204.4, 203.8, 203.1, 202.4, 201.9,
            201.6, 201. , 200.6, 200.4, 199.4, 198.6, 198. , 197.3, 196.6,
            196.1, 195.6, 195.6, 196.1, 196.6, 197. , 198. , 199.2, 200.4,
            201.5, 202. , 202.6, 203.2, 203.5, 203.4, 203.2, 202.9, 202.7,
            202.4, 202. , 201.7, 201.5, 201.4, 201.3, 201.2, 201.1, 201. ,
            200.9, 200.8, 200.7, 200.5, 200.4, 200.1, 200. , 199.9, 199.8],
           [205.8, 205.6, 205.4, 204.7, 204. , 203.7, 203.2, 202.5, 202.4,
            202.2, 201.6, 201.4, 201.1, 200.4, 199.6, 198.9, 198.1, 197.4,
            196.8, 196.2, 196.4, 196.9, 197.3, 197.8, 198.7, 199.9, 201.1,
            201.8, 202.3, 202.9, 203.5, 203.4, 203.4, 203.1, 202.9, 202.6,
            202.4, 202.1, 201.8, 201.7, 201.6, 201.5, 201.4, 201.3, 201.2,
            201.1, 200.9, 200.8, 200.7, 200.5, 200.4, 200.3, 200.1, 200. ],
           [205.1, 204.9, 204.8, 204.2, 203.9, 203.6, 203.2, 203. , 202.9,
            202.7, 202.2, 201.8, 201.4, 200.8, 199.9, 199. , 198.6, 198. ,
            197.3, 197. , 197.2, 197.6, 198. , 198.6, 199.3, 200.6, 201.5,
            202.1, 202.6, 203.2, 203.4, 203.4, 203.3, 203.1, 202.9, 202.6,
            202.4, 202.1, 202. , 201.9, 201.7, 201.6, 201.5, 201.4, 201.4,
            201.3, 201.1, 201. , 200.8, 200.7, 200.5, 200.4, 200.3, 200.2],
           [204.3, 204.2, 204.1, 204. , 203.7, 203.4, 203.1, 202.9, 202.6,
            202.4, 202.1, 201.7, 201.3, 201. , 200.1, 199.3, 199.1, 198.5,
            197.8, 197.9, 198. , 198.4, 198.9, 199.5, 200. , 201.3, 201.8,
            202.4, 202.9, 203.3, 203.3, 203.3, 203.3, 203.1, 202.9, 202.6,
            202.3, 202.2, 202. , 201.9, 201.8, 201.9, 201.9, 201.8, 201.7,
            201.5, 201.3, 201.1, 201. , 200.8, 200.7, 200.5, 200.4, 200.5],
           [203.8, 203.8, 203.8, 203.5, 203.2, 202.9, 202.8, 202.5, 202.2,
            202.2, 201.9, 201.5, 201.3, 201.2, 200.2, 199.7, 199.5, 199.1,
            198.7, 198.8, 198.9, 199.2, 199.8, 200.3, 200.8, 201.6, 202.1,
            202.7, 203.3, 203.3, 203.3, 203.3, 203.3, 202.8, 202.5, 202.3,
            202. , 202.1, 202. , 201.9, 201.7, 202.2, 202.2, 202.1, 202. ,
            201.7, 201.5, 201.3, 201.2, 201. , 200.8, 200.7, 200.6, 200.8],
           [203.4, 203.4, 203.3, 203.1, 202.8, 202.5, 202.4, 202.1, 201.9,
            201.9, 201.8, 201.4, 201.3, 201.2, 200.5, 200.2, 200. , 199.7,
            199.7, 199.7, 199.7, 200.1, 200.6, 201.1, 201.6, 202. , 202.6,
            203.2, 203.5, 203.3, 203.3, 203.3, 203. , 202.5, 202.2, 201.9,
            201.9, 202.1, 202. , 201.8, 202. , 202.5, 202.6, 202.5, 202.3,
            201.9, 201.7, 201.5, 201.3, 201.1, 200.9, 200.8, 200.9, 201.1],
           [203.1, 203.1, 202.9, 202.6, 202.3, 202.1, 202. , 201.8, 201.7,
            201.6, 201.6, 201.4, 201.3, 201.2, 201. , 200.7, 200.4, 200.3,
            200.4, 200.3, 200.3, 200.8, 201.3, 201.8, 202.2, 202.5, 203.1,
            203.7, 203.6, 203.3, 203.3, 203.3, 202.8, 202.2, 201.9, 201.6,
            201.8, 202. , 201.9, 201.8, 202.2, 202.7, 203. , 202.8, 202.5,
            202.2, 201.9, 201.7, 201.5, 201.3, 201.1, 201. , 201.2, 201.4],
           [202.7, 202.7, 202.4, 202.1, 201.9, 201.7, 201.6, 201.4, 201.4,
            201.4, 201.4, 201.5, 201.4, 201.3, 201.5, 201.2, 200.9, 200.9,
            201. , 200.9, 201. , 201.6, 202.1, 202.5, 202.8, 203. , 203.6,
            204.1, 203.7, 203.3, 203.3, 203.1, 202.5, 201.9, 201.6, 201.5,
            201.7, 201.9, 201.9, 202. , 202.5, 203. , 203.3, 203.1, 202.7,
            202.4, 202.1, 201.9, 201.7, 201.4, 201.2, 201.3, 201.5, 201.8],
           [202.3, 202.2, 202. , 201.8, 201.6, 201.4, 201.3, 201.2, 201.2,
            201.2, 201.4, 201.6, 201.5, 201.5, 201.9, 201.6, 201.3, 201.5,
            201.6, 201.5, 201.8, 202.3, 202.8, 203.2, 203.4, 203.5, 204.1,
            204.2, 203.7, 203.3, 203.3, 202.8, 202.2, 201.6, 201.2, 201.4,
            201.6, 201.8, 201.9, 202.3, 202.8, 203.2, 203.7, 203.4, 203. ,
            202.6, 202.3, 202. , 201.8, 201.6, 201.4, 201.5, 201.7, 202. ],
           [201.9, 202. , 201.8, 201.6, 201.3, 201.2, 201.1, 201. , 201.1,
            201. , 201.3, 201.7, 201.6, 201.8, 202.2, 202.1, 201.9, 202.1,
            202.2, 202.1, 202.6, 203.1, 203.6, 203.8, 204. , 204.1, 204.6,
            204.2, 203.8, 203.4, 203.2, 202.6, 202. , 201.4, 201.1, 201.3,
            201.5, 201.7, 202. , 202.4, 202.9, 203.3, 203.8, 203.5, 203.2,
            202.8, 202.5, 202.2, 202. , 201.8, 201.6, 201.7, 201.9, 202.1],
           [201.5, 201.7, 201.6, 201.4, 201.1, 201. , 200.8, 200.8, 200.9,
            200.9, 201.3, 201.9, 201.8, 202.1, 202.5, 202.5, 202.5, 202.6,
            202.8, 202.9, 203.4, 203.8, 204.3, 204.5, 204.6, 204.7, 204.8,
            204.3, 203.9, 203.4, 202.9, 202.3, 201.6, 201. , 200.9, 201.1,
            201.3, 201.5, 201.9, 202.4, 202.9, 203.3, 203.8, 203.7, 203.3,
            202.9, 202.7, 202.4, 202.2, 202. , 201.9, 201.9, 202.1, 202.3],
           [201.1, 201.3, 201.3, 201.2, 200.9, 200.7, 200.6, 200.6, 200.7,
            200.7, 201.3, 201.8, 201.9, 202.4, 202.8, 203. , 203. , 203.2,
            203.3, 203.6, 204.1, 204.5, 204.8, 204.8, 204.9, 204.9, 204.6,
            204.1, 203.7, 203.2, 202.5, 201.9, 201.3, 200.8, 200.7, 200.9,
            201.1, 201.4, 201.9, 202.4, 202.9, 203.3, 203.9, 203.9, 203.5,
            203.1, 202.9, 202.7, 202.4, 202.2, 202.1, 202. , 202.3, 202.5],
           [200.7, 201. , 201.1, 200.9, 200.7, 200.4, 200.4, 200.4, 200.5,
            200.7, 201.2, 201.8, 202.2, 202.6, 203.1, 203.4, 203.5, 203.6,
            203.7, 203.7, 204.1, 204.5, 204.7, 204.8, 204.8, 204.8, 204.3,
            203.9, 203.4, 202.8, 202.2, 201.5, 200.9, 200.7, 200.6, 200.7,
            200.9, 201.4, 201.9, 202.4, 202.8, 203.3, 203.9, 204. , 203.6,
            203.3, 203.1, 202.9, 202.6, 202.5, 202.3, 202.2, 202.5, 202.6]
        ])

    speed_2d_002 = numpy.array([
        [0.39, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.44, 0.44, 0.43, 0.43, 0.42,
        0.43, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.48, 0.49,
        0.5 , 0.51, 0.51, 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54,
        0.54, 0.55, 0.55, 0.56, 0.57, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 ,
        0.6 , 0.6 , 0.61, 0.62, 0.63, 0.64, 0.65, 0.66, 0.67, 0.67],
       [0.39, 0.39, 0.39, 0.39, 0.4 , 0.42, 0.43, 0.43, 0.42, 0.42, 0.42,
        0.42, 0.43, 0.43, 0.44, 0.45, 0.45, 0.45, 0.46, 0.47, 0.47, 0.49,
        0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.53, 0.53, 0.53, 0.54,
        0.54, 0.54, 0.55, 0.56, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59,
        0.6 , 0.6 , 0.61, 0.63, 0.64, 0.64, 0.65, 0.66, 0.67, 0.67],
       [0.39, 0.39, 0.38, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.41, 0.41, 0.41,
        0.42, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.47, 0.48,
        0.49, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.53,
        0.54, 0.54, 0.55, 0.56, 0.57, 0.58, 0.58, 0.58, 0.58, 0.58, 0.59,
        0.6 , 0.61, 0.62, 0.63, 0.63, 0.64, 0.65, 0.65, 0.66, 0.67],
       [0.39, 0.38, 0.37, 0.37, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.41, 0.42,
        0.42, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.48, 0.48,
        0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.53,
        0.53, 0.54, 0.55, 0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59,
        0.6 , 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.64, 0.65, 0.66],
       [0.38, 0.38, 0.37, 0.36, 0.36, 0.37, 0.38, 0.4 , 0.41, 0.42, 0.42,
        0.42, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.48, 0.49,
        0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.51, 0.52, 0.52, 0.53, 0.53,
        0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.59,
        0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.62, 0.63, 0.64, 0.65],
       [0.38, 0.38, 0.37, 0.36, 0.36, 0.37, 0.38, 0.4 , 0.42, 0.42, 0.42,
        0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.48, 0.48,
        0.49, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.53,
        0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59,
        0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.62, 0.63, 0.64],
       [0.37, 0.37, 0.37, 0.36, 0.36, 0.37, 0.38, 0.4 , 0.41, 0.42, 0.42,
        0.42, 0.43, 0.43, 0.44, 0.44, 0.45, 0.46, 0.47, 0.47, 0.48, 0.48,
        0.48, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.53, 0.53, 0.54,
        0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58,
        0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.62, 0.63, 0.65],
       [0.36, 0.37, 0.37, 0.36, 0.36, 0.36, 0.38, 0.39, 0.4 , 0.41, 0.41,
        0.42, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.47, 0.47, 0.47, 0.48,
        0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54,
        0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.59,
        0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.62, 0.63, 0.64, 0.66],
       [0.36, 0.36, 0.37, 0.37, 0.37, 0.37, 0.37, 0.38, 0.39, 0.4 , 0.4 ,
        0.41, 0.42, 0.43, 0.43, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.48,
        0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.54, 0.55,
        0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59,
        0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.63, 0.65, 0.66],
       [0.36, 0.36, 0.37, 0.37, 0.37, 0.37, 0.37, 0.38, 0.39, 0.39, 0.4 ,
        0.4 , 0.41, 0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.47, 0.48,
        0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55,
        0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 ,
        0.6 , 0.6 , 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.64, 0.65],
       [0.36, 0.36, 0.37, 0.37, 0.37, 0.38, 0.38, 0.39, 0.39, 0.39, 0.39,
        0.4 , 0.41, 0.43, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.48,
        0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55,
        0.56, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 ,
        0.6 , 0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.64, 0.64],
       [0.36, 0.36, 0.37, 0.38, 0.38, 0.39, 0.39, 0.39, 0.39, 0.4 , 0.4 ,
        0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.46, 0.47, 0.48,
        0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55,
        0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.6 , 0.61,
        0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63],
       [0.36, 0.37, 0.38, 0.38, 0.39, 0.39, 0.39, 0.4 , 0.4 , 0.41, 0.41,
        0.43, 0.43, 0.44, 0.45, 0.46, 0.47, 0.47, 0.47, 0.46, 0.47, 0.48,
        0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55,
        0.56, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.6 , 0.61, 0.61,
        0.61, 0.61, 0.62, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63, 0.63],
       [0.36, 0.37, 0.38, 0.39, 0.39, 0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42,
        0.44, 0.44, 0.45, 0.47, 0.48, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49,
        0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.56,
        0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.6 , 0.6 , 0.61, 0.61,
        0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63, 0.63],
       [0.37, 0.38, 0.38, 0.38, 0.39, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.43,
        0.44, 0.46, 0.47, 0.48, 0.5 , 0.5 , 0.49, 0.49, 0.49, 0.49, 0.5 ,
        0.51, 0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56,
        0.56, 0.56, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 ,
        0.61, 0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63],
       [0.37, 0.38, 0.38, 0.38, 0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.44,
        0.46, 0.48, 0.49, 0.5 , 0.5 , 0.5 , 0.49, 0.49, 0.49, 0.5 , 0.5 ,
        0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.56,
        0.56, 0.57, 0.57, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 ,
        0.6 , 0.6 , 0.61, 0.61, 0.61, 0.61, 0.62, 0.62, 0.63, 0.63],
       [0.37, 0.38, 0.38, 0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.45,
        0.48, 0.49, 0.5 , 0.5 , 0.5 , 0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.51,
        0.52, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57,
        0.57, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.59,
        0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.62],
       [0.38, 0.39, 0.39, 0.4 , 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.44, 0.46,
        0.48, 0.49, 0.48, 0.47, 0.48, 0.49, 0.49, 0.49, 0.49, 0.5 , 0.51,
        0.52, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57,
        0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.59, 0.59,
        0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.61, 0.61, 0.62],
       [0.38, 0.4 , 0.41, 0.41, 0.41, 0.41, 0.42, 0.43, 0.44, 0.45, 0.46,
        0.46, 0.45, 0.45, 0.45, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 , 0.51,
        0.52, 0.52, 0.52, 0.53, 0.53, 0.53, 0.54, 0.55, 0.56, 0.57, 0.57,
        0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.59,
        0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.61, 0.61],
       [0.39, 0.4 , 0.41, 0.41, 0.41, 0.42, 0.43, 0.44, 0.44, 0.44, 0.44,
        0.44, 0.43, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 ,
        0.51, 0.51, 0.51, 0.51, 0.51, 0.52, 0.53, 0.54, 0.56, 0.57, 0.58,
        0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.6 , 0.6 , 0.59,
        0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.61, 0.61],
       [0.39, 0.39, 0.4 , 0.41, 0.42, 0.42, 0.44, 0.44, 0.44, 0.44, 0.43,
        0.43, 0.42, 0.42, 0.42, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 ,
        0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.52, 0.54, 0.55, 0.57, 0.58,
        0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.6 , 0.6 , 0.59,
        0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.61, 0.62],
       [0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.44, 0.45, 0.44, 0.43, 0.43,
        0.42, 0.42, 0.43, 0.43, 0.44, 0.46, 0.47, 0.48, 0.49, 0.5 , 0.5 ,
        0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.52, 0.53, 0.55, 0.56, 0.57,
        0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.59, 0.59,
        0.59, 0.59, 0.6 , 0.6 , 0.59, 0.59, 0.6 , 0.61, 0.61, 0.62],
       [0.37, 0.38, 0.39, 0.39, 0.4 , 0.42, 0.44, 0.45, 0.44, 0.43, 0.43,
        0.42, 0.43, 0.43, 0.45, 0.46, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.5 ,
        0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.51, 0.53, 0.54, 0.56, 0.57,
        0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59,
        0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.62, 0.62],
       [0.37, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.43, 0.45, 0.44, 0.44, 0.43,
        0.43, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 , 0.5 , 0.5 , 0.5 ,
        0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.55, 0.56,
        0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 ,
        0.6 , 0.61, 0.61, 0.61, 0.61, 0.6 , 0.6 , 0.61, 0.62, 0.62],
       [0.37, 0.37, 0.38, 0.39, 0.39, 0.41, 0.43, 0.44, 0.45, 0.44, 0.44,
        0.43, 0.44, 0.46, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.5 , 0.5 , 0.5 ,
        0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.49, 0.49, 0.51, 0.52, 0.54, 0.55,
        0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.6 ,
        0.61, 0.62, 0.62, 0.62, 0.61, 0.61, 0.61, 0.61, 0.62, 0.62],
       [0.37, 0.38, 0.38, 0.39, 0.39, 0.41, 0.42, 0.44, 0.45, 0.45, 0.45,
        0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.5 ,
        0.5 , 0.5 , 0.5 , 0.49, 0.49, 0.48, 0.48, 0.5 , 0.51, 0.53, 0.55,
        0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.6 ,
        0.61, 0.62, 0.63, 0.62, 0.62, 0.62, 0.61, 0.62, 0.62, 0.62],
       [0.37, 0.38, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.44, 0.45, 0.45, 0.45,
        0.45, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.5 ,
        0.5 , 0.5 , 0.49, 0.49, 0.48, 0.48, 0.47, 0.48, 0.5 , 0.52, 0.54,
        0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.61,
        0.62, 0.63, 0.63, 0.63, 0.63, 0.62, 0.62, 0.62, 0.62, 0.62],
       [0.37, 0.38, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.43, 0.45, 0.45, 0.45,
        0.45, 0.45, 0.45, 0.46, 0.47, 0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 ,
        0.5 , 0.49, 0.49, 0.49, 0.48, 0.47, 0.47, 0.47, 0.49, 0.51, 0.53,
        0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.61,
        0.62, 0.63, 0.63, 0.63, 0.63, 0.63, 0.62, 0.62, 0.62, 0.61],
       [0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41, 0.43, 0.44, 0.45, 0.45, 0.45,
        0.45, 0.44, 0.45, 0.45, 0.46, 0.46, 0.47, 0.47, 0.49, 0.5 , 0.5 ,
        0.49, 0.49, 0.49, 0.48, 0.47, 0.47, 0.46, 0.47, 0.49, 0.51, 0.53,
        0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.6 , 0.61,
        0.61, 0.62, 0.63, 0.63, 0.63, 0.62, 0.62, 0.62, 0.61, 0.61],
       [0.37, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.44,
        0.44, 0.43, 0.43, 0.44, 0.45, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 ,
        0.49, 0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.48, 0.5 , 0.52,
        0.52, 0.53, 0.53, 0.54, 0.55, 0.56, 0.56, 0.57, 0.58, 0.59, 0.59,
        0.6 , 0.61, 0.62, 0.62, 0.62, 0.62, 0.61, 0.61, 0.61, 0.61],
       [0.37, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.42, 0.43, 0.43, 0.43, 0.43,
        0.43, 0.42, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 ,
        0.49, 0.49, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.48, 0.49,
        0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.56, 0.56, 0.57, 0.58,
        0.59, 0.6 , 0.61, 0.61, 0.61, 0.61, 0.61, 0.61, 0.61, 0.61],
       [0.37, 0.38, 0.39, 0.4 , 0.41, 0.41, 0.42, 0.42, 0.42, 0.42, 0.41,
        0.41, 0.41, 0.41, 0.42, 0.43, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 ,
        0.49, 0.49, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.48, 0.48,
        0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.56, 0.56, 0.57,
        0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.61],
       [0.36, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.4 , 0.4 , 0.4 , 0.4 , 0.4 ,
        0.4 , 0.4 , 0.41, 0.42, 0.43, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 ,
        0.49, 0.48, 0.48, 0.47, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.48,
        0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56, 0.56,
        0.57, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61],
       [0.36, 0.37, 0.38, 0.39, 0.39, 0.39, 0.39, 0.38, 0.38, 0.38, 0.39,
        0.39, 0.39, 0.4 , 0.41, 0.42, 0.44, 0.46, 0.47, 0.48, 0.48, 0.49,
        0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48,
        0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56,
        0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.61, 0.61, 0.61],
       [0.35, 0.36, 0.36, 0.37, 0.37, 0.37, 0.37, 0.37, 0.37, 0.37, 0.37,
        0.38, 0.38, 0.39, 0.4 , 0.42, 0.43, 0.45, 0.46, 0.47, 0.48, 0.48,
        0.49, 0.48, 0.47, 0.47, 0.47, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48,
        0.48, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.56,
        0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.61, 0.61],
       [0.34, 0.34, 0.35, 0.35, 0.35, 0.35, 0.35, 0.35, 0.35, 0.36, 0.36,
        0.37, 0.38, 0.38, 0.39, 0.41, 0.43, 0.44, 0.46, 0.46, 0.47, 0.48,
        0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48,
        0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.56,
        0.56, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.6 ],
       [0.33, 0.33, 0.33, 0.33, 0.33, 0.33, 0.34, 0.34, 0.34, 0.35, 0.35,
        0.36, 0.36, 0.38, 0.39, 0.4 , 0.42, 0.43, 0.45, 0.46, 0.46, 0.47,
        0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48,
        0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55,
        0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59],
       [0.32, 0.32, 0.32, 0.32, 0.32, 0.33, 0.33, 0.33, 0.33, 0.34, 0.34,
        0.35, 0.36, 0.37, 0.38, 0.39, 0.41, 0.43, 0.44, 0.45, 0.46, 0.46,
        0.47, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48,
        0.49, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.55,
        0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58],
       [0.31, 0.32, 0.32, 0.32, 0.31, 0.32, 0.32, 0.32, 0.32, 0.33, 0.33,
        0.34, 0.35, 0.36, 0.37, 0.39, 0.4 , 0.42, 0.43, 0.44, 0.45, 0.45,
        0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48,
        0.49, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55,
        0.56, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.57, 0.57],
       [0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.32, 0.32, 0.33,
        0.34, 0.34, 0.36, 0.37, 0.38, 0.39, 0.41, 0.42, 0.43, 0.44, 0.44,
        0.45, 0.45, 0.45, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48,
        0.48, 0.49, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.54,
        0.55, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.57, 0.57, 0.56],
       [0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.32, 0.32,
        0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.4 , 0.41, 0.42, 0.42, 0.43,
        0.44, 0.45, 0.45, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48,
        0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.53, 0.53,
        0.54, 0.55, 0.55, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56, 0.56],
       [0.3 , 0.31, 0.31, 0.31, 0.31, 0.3 , 0.31, 0.31, 0.31, 0.31, 0.32,
        0.33, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.4 , 0.41, 0.41, 0.42,
        0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48,
        0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.5 , 0.51, 0.52, 0.53,
        0.54, 0.55, 0.55, 0.55, 0.55, 0.55, 0.56, 0.55, 0.55, 0.55],
       [0.3 , 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.32,
        0.32, 0.33, 0.34, 0.34, 0.35, 0.36, 0.38, 0.39, 0.4 , 0.41, 0.42,
        0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.47, 0.47, 0.47, 0.47, 0.48,
        0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.52,
        0.53, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55],
       [0.29, 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
        0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.41,
        0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48,
        0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.51, 0.52,
        0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.54],
       [0.29, 0.3 , 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
        0.32, 0.32, 0.32, 0.33, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 , 0.4 ,
        0.41, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.47,
        0.47, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.51,
        0.53, 0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55],
       [0.28, 0.29, 0.3 , 0.31, 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.31,
        0.31, 0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 ,
        0.41, 0.42, 0.43, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.47,
        0.47, 0.47, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51,
        0.52, 0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.55, 0.54, 0.55],
       [0.28, 0.28, 0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.31,
        0.31, 0.31, 0.32, 0.32, 0.33, 0.34, 0.36, 0.37, 0.37, 0.39, 0.4 ,
        0.41, 0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.47,
        0.47, 0.47, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51,
        0.52, 0.53, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55],
       [0.27, 0.28, 0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 ,
        0.31, 0.31, 0.31, 0.32, 0.33, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 ,
        0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46,
        0.47, 0.47, 0.47, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 ,
        0.51, 0.52, 0.53, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55],
       [0.26, 0.27, 0.28, 0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 ,
        0.3 , 0.31, 0.31, 0.32, 0.33, 0.34, 0.36, 0.37, 0.38, 0.39, 0.39,
        0.41, 0.42, 0.43, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46,
        0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.5 ,
        0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55],
       [0.26, 0.27, 0.28, 0.29, 0.29, 0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 ,
        0.3 , 0.31, 0.31, 0.32, 0.33, 0.34, 0.36, 0.37, 0.38, 0.39, 0.39,
        0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46,
        0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
        0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55],
       [0.26, 0.26, 0.27, 0.28, 0.29, 0.29, 0.29, 0.29, 0.3 , 0.3 , 0.3 ,
        0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.37, 0.38, 0.39, 0.4 ,
        0.4 , 0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.45, 0.46,
        0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
        0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55],
       [0.25, 0.26, 0.27, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29, 0.3 ,
        0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.37, 0.38, 0.39, 0.4 ,
        0.4 , 0.41, 0.42, 0.43, 0.44, 0.44, 0.44, 0.45, 0.45, 0.45, 0.45,
        0.45, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
        0.5 , 0.51, 0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.55],
       [0.25, 0.26, 0.26, 0.27, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29, 0.29,
        0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.38, 0.38, 0.39, 0.4 ,
        0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.44, 0.44, 0.45, 0.45,
        0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49,
        0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.55, 0.55],
       [0.25, 0.25, 0.26, 0.27, 0.27, 0.28, 0.28, 0.29, 0.29, 0.29, 0.29,
        0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.38, 0.38, 0.39, 0.4 ,
        0.4 , 0.41, 0.42, 0.42, 0.43, 0.43, 0.43, 0.44, 0.44, 0.44, 0.44,
        0.45, 0.45, 0.45, 0.45, 0.46, 0.47, 0.48, 0.48, 0.48, 0.48, 0.48,
        0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.55, 0.55]])

    direction_2d_002 = numpy.array([
        [197.7, 197.4, 197.1, 197.7, 198.4, 199.2, 199.9, 200.5, 200.9,
        201.4, 201.8, 201.8, 201.9, 202.1, 202.4, 202.5, 202.5, 202.8,
        202.8, 203.1, 203.1, 202.9, 202.6, 202.3, 202. , 201.8, 201.6,
        201.3, 201.2, 201.3, 201.6, 201.8, 202. , 202.2, 202.6, 203. ,
        203.4, 203.8, 204. , 204. , 203.8, 203.4, 203.2, 202.8, 202.5,
        202.2, 201.9, 201.6, 201.6, 201.7, 201.7, 201.7, 201.8, 202.1],
       [197.6, 197.5, 197.7, 198.2, 199. , 199.8, 200.4, 200.8, 201.2,
        201.6, 202.2, 202.7, 202.9, 203.2, 203.4, 203.5, 203.5, 203.7,
        203.9, 204.1, 203.9, 203.6, 203.2, 202.8, 202.5, 202.2, 202. ,
        201.9, 201.7, 201.7, 202. , 202.2, 202.4, 202.6, 202.9, 203.3,
        203.6, 203.9, 204.2, 204. , 203.8, 203.4, 203. , 202.6, 202.4,
        202.2, 201.9, 201.7, 201.5, 201.6, 201.7, 201.7, 201.8, 202. ],
       [198. , 198.2, 198.4, 198.8, 199.7, 200.2, 200.6, 201.3, 201.9,
        202.5, 203.2, 203.8, 204.1, 204.3, 204.3, 204.2, 204.4, 204.5,
        204.6, 204.7, 204.6, 204.2, 203.8, 203.4, 203. , 202.7, 202.4,
        202.3, 202.2, 202.2, 202.3, 202.5, 202.7, 203. , 203.2, 203.6,
        204. , 204.1, 204.2, 204.2, 203.8, 203.4, 203.1, 202.7, 202.5,
        202.3, 202.2, 201.9, 201.6, 201.6, 201.7, 201.8, 201.8, 202. ],
       [198.7, 199. , 199.3, 199.5, 199.9, 200.5, 201.3, 202.1, 202.9,
        203.5, 204.2, 204.9, 205. , 204.9, 204.9, 204.9, 205. , 205. ,
        205.1, 205. , 205. , 204.7, 204.2, 203.8, 203.3, 202.9, 202.8,
        202.6, 202.6, 202.5, 202.6, 202.8, 203. , 203.4, 203.7, 204. ,
        204.2, 204.3, 204.3, 204.2, 203.9, 203.6, 203.3, 203.2, 203. ,
        202.7, 202.4, 202.1, 201.8, 201.7, 201.8, 201.8, 201.9, 202.2],
       [199.1, 199.2, 199.5, 199.7, 200.2, 201.1, 202.1, 203.1, 203.8,
        204.4, 204.9, 205.2, 205.4, 205.3, 205.1, 205.3, 205.5, 205.5,
        205.5, 205.4, 205.3, 205.1, 204.6, 204.1, 203.6, 203.2, 203. ,
        203. , 202.9, 202.9, 202.9, 203.2, 203.5, 203.8, 204. , 204.3,
        204.3, 204.3, 204.3, 204.3, 204.2, 204. , 203.8, 203.6, 203.4,
        203.1, 202.7, 202.3, 202. , 201.9, 202. , 202.2, 202.4, 202.6],
       [199.4, 199.6, 199.9, 200.4, 200.9, 202.1, 203.1, 203.9, 204.5,
        204.9, 205.4, 205.4, 205.3, 205.3, 205.4, 205.6, 205.9, 206. ,
        205.9, 205.8, 205.5, 205.3, 204.8, 204.3, 203.9, 203.5, 203.3,
        203.3, 203.2, 203.2, 203.3, 203.5, 203.8, 204. , 204.1, 204.3,
        204.3, 204.3, 204.4, 204.4, 204.5, 204.2, 204.1, 203.8, 203.6,
        203.3, 202.9, 202.6, 202.4, 202.1, 202.2, 202.4, 202.7, 202.9],
       [200. , 200.1, 200.7, 201.4, 202.3, 203.3, 204.3, 205. , 205.3,
        205.5, 205.6, 205.7, 205.5, 205.4, 205.7, 206.1, 206.3, 206.5,
        206.3, 206. , 205.8, 205.4, 204.9, 204.5, 204.1, 203.8, 203.6,
        203.5, 203.5, 203.4, 203.5, 203.6, 203.7, 203.9, 204.1, 204.2,
        204.3, 204.3, 204.4, 204.5, 204.6, 204.3, 204.1, 203.9, 203.6,
        203.5, 203.2, 202.9, 202.6, 202.3, 202.4, 202.6, 202.7, 203.1],
       [200.8, 201. , 201.8, 202.7, 204. , 205.2, 205.8, 206.2, 206.3,
        206.3, 206.2, 206.3, 206.2, 206.1, 206.4, 206.7, 206.9, 206.8,
        206.7, 206.3, 205.9, 205.5, 204.9, 204.5, 204.2, 203.9, 203.5,
        203.4, 203.3, 203.4, 203.6, 203.6, 203.8, 203.9, 204. , 204.1,
        204.2, 204.3, 204.4, 204.4, 204.4, 204.2, 203.9, 203.8, 203.7,
        203.7, 203.3, 202.9, 202.6, 202.3, 202.2, 202.6, 203. , 203.4],
       [202.2, 202.6, 203.3, 204.5, 205.6, 206.7, 207.3, 207.4, 207.4,
        207.4, 207.4, 207.3, 207.3, 207.2, 207.4, 207.4, 207.6, 207.4,
        207.1, 206.7, 206.1, 205.6, 204.7, 204.2, 203.9, 203.6, 203.5,
        203.4, 203.6, 203.6, 203.6, 203.6, 203.7, 203.8, 203.9, 204. ,
        204.1, 204.1, 204.2, 204.1, 204.1, 204. , 203.9, 203.8, 203.6,
        203.5, 203.2, 202.9, 202.7, 202.6, 202.7, 203.1, 203.4, 203.7],
       [203.9, 204.4, 205. , 205.9, 207. , 207.9, 208.7, 208.6, 208.4,
        208.6, 208.7, 208.6, 208.6, 208.6, 208.7, 208.7, 208.4, 208.3,
        207.6, 207.1, 206.3, 205.5, 204.8, 204.3, 204.1, 204. , 203.9,
        203.7, 203.6, 203.7, 203.6, 203.6, 203.6, 203.8, 203.9, 203.9,
        203.9, 204. , 204. , 204. , 203.9, 203.9, 203.7, 203.6, 203.5,
        203.5, 203.4, 203.2, 203.2, 203.1, 203.2, 203.4, 203.7, 203.9],
       [205.8, 206.2, 206.5, 207.1, 208. , 208.6, 209.1, 209.3, 209.3,
        209.6, 210.1, 210.1, 210.1, 209.9, 210.2, 210.2, 209.6, 208.9,
        208.3, 207.6, 206.9, 206.1, 205.3, 204.6, 204.5, 204.3, 204. ,
        203.8, 203.8, 203.8, 203.8, 203.8, 203.7, 203.8, 203.9, 203.9,
        204. , 204. , 204. , 204. , 203.9, 203.8, 203.7, 203.6, 203.6,
        203.5, 203.4, 203.2, 203.3, 203.4, 203.5, 203.7, 203.9, 204. ],
       [207.4, 207.7, 207.8, 207.9, 208.5, 209.2, 209.5, 209.7, 210.2,
        210.6, 210.9, 211. , 211.1, 211.2, 211.3, 211.1, 210.5, 209.8,
        208.9, 208.1, 207.2, 206.4, 205.7, 205. , 204.7, 204.5, 204.3,
        204.2, 204. , 203.9, 203.9, 203.9, 203.9, 203.9, 204. , 204.1,
        204.1, 204.2, 204.1, 204. , 204. , 203.9, 203.7, 203.6, 203.5,
        203.4, 203.2, 203.1, 203. , 203.2, 203.5, 203.7, 203.9, 204.1],
       [208.8, 208.9, 208.9, 208.9, 209.3, 209.8, 210.3, 210.6, 210.9,
        211.2, 211.5, 211.6, 211.7, 211.8, 211.8, 211.4, 210.5, 209.7,
        208.9, 208.1, 207.2, 206.4, 205.9, 205.4, 204.9, 204.6, 204.4,
        204.2, 204.1, 204. , 204. , 204. , 204.1, 204.2, 204.2, 204.2,
        204.3, 204.4, 204.4, 204.3, 204.1, 203.9, 203.7, 203.5, 203.4,
        203.1, 202.8, 202.7, 202.8, 203. , 203.2, 203.5, 203.7, 204. ],
       [210.2, 210.1, 210.1, 210.3, 210.5, 210.8, 211.1, 211.3, 211.6,
        211.7, 211.9, 212. , 212. , 211.7, 211.4, 211.1, 210.4, 209.7,
        208.8, 207.8, 206.9, 206.1, 205.5, 205.2, 204.8, 204.5, 204.3,
        204.1, 204. , 204. , 204. , 204. , 204.1, 204.3, 204.3, 204.4,
        204.5, 204.6, 204.7, 204.6, 204.3, 204. , 203.7, 203.3, 203.1,
        202.9, 202.5, 202.2, 202.4, 202.7, 203. , 203.3, 203.5, 203.7],
       [211.1, 211.3, 211.2, 211.3, 211.4, 211.6, 211.9, 212. , 212. ,
        212.1, 212.1, 212.2, 212.2, 212. , 211.5, 210.9, 210.2, 209.2,
        208.2, 207.1, 206.1, 205.5, 205. , 204.7, 204.4, 204.2, 204. ,
        203.8, 203.8, 203.8, 203.9, 204. , 204.1, 204.3, 204.5, 204.6,
        204.7, 204.8, 204.9, 204.8, 204.5, 204.1, 203.6, 203.2, 203. ,
        202.7, 202.4, 202.1, 202.2, 202.5, 202.7, 203. , 203.3, 203.6],
       [211.8, 212.1, 212. , 212.1, 212.3, 212.4, 212.4, 212.4, 212.2,
        212.2, 212.4, 212.5, 212.3, 211.7, 211. , 210.3, 209.4, 208.2,
        207. , 206.2, 205.4, 204.8, 204.3, 203.9, 203.7, 203.5, 203.5,
        203.4, 203.4, 203.5, 203.6, 203.8, 204.1, 204.3, 204.7, 204.8,
        204.9, 205. , 205.1, 205. , 204.6, 204.2, 203.7, 203.3, 203. ,
        202.6, 202.4, 202.1, 202. , 202.3, 202.7, 203. , 203.3, 203.6],
       [212.1, 212.5, 212.6, 212.6, 212.7, 212.9, 212.7, 212.5, 212.5,
        212.6, 212.5, 212.3, 212.1, 211.5, 210.8, 209.5, 208.2, 207. ,
        206.1, 205.2, 204.4, 203.7, 203.6, 203.4, 203.2, 202.9, 202.9,
        203. , 203.1, 203.2, 203.3, 203.5, 203.9, 204.3, 204.7, 204.8,
        204.8, 204.9, 205.1, 205.2, 204.7, 204.2, 203.8, 203.5, 203.2,
        203. , 202.6, 202.3, 202. , 202.3, 202.7, 203. , 203.4, 203.7],
       [212.3, 212.6, 212.6, 212.6, 212.7, 213. , 213. , 212.9, 213.1,
        213.6, 213.6, 212.9, 212.4, 211.8, 211. , 209.3, 207.4, 205.9,
        205.4, 204.9, 204.2, 203.4, 203.2, 203.1, 203. , 202.7, 202.6,
        202.7, 202.8, 202.9, 202.9, 203.2, 203.6, 204. , 204.4, 204.6,
        204.7, 204.7, 204.8, 205. , 204.7, 204.3, 204.1, 203.8, 203.5,
        203.1, 202.8, 202.4, 202.2, 202.4, 202.8, 203.2, 203.5, 203.9],
       [212.3, 212.2, 212.2, 212.5, 212.8, 213.2, 213.3, 213.8, 214.1,
        214.8, 214.5, 214. , 213.5, 212.9, 211.2, 209.5, 208. , 206.5,
        205.8, 205.3, 204.7, 204. , 203.7, 203.4, 203. , 202.7, 202.4,
        202.5, 202.5, 202.6, 202.6, 202.9, 203.4, 203.8, 204. , 204.1,
        204.2, 204.4, 204.6, 204.7, 204.5, 204.1, 203.8, 203.7, 203.6,
        203.3, 203. , 202.6, 202.4, 202.6, 203. , 203.4, 203.7, 204.1],
       [212.4, 212.4, 212.5, 212.7, 213. , 213.4, 213.7, 214.2, 214.7,
        215.1, 215.1, 214.3, 213.9, 213.4, 212.1, 210.3, 209. , 208. ,
        207.3, 206.4, 205.5, 204.6, 203.9, 203.6, 203.2, 202.7, 202.3,
        202.5, 202.7, 202.7, 202.7, 202.8, 203.1, 203.3, 203.6, 203.7,
        203.8, 204. , 204. , 204.1, 204. , 203.9, 203.7, 203.4, 203.4,
        203.1, 202.7, 202.7, 202.7, 202.8, 203.2, 203.6, 203.9, 204.2],
       [212.8, 212.8, 212.7, 212.9, 213.2, 213.4, 213.4, 213.7, 214.2,
        214.7, 215.2, 214.7, 214.2, 213.7, 212.9, 211.3, 209.8, 208.7,
        207.8, 206.8, 205.9, 205. , 204.2, 203.8, 203.4, 203.2, 202.9,
        202.9, 202.8, 202.7, 202.7, 202.8, 203.1, 203.3, 203.3, 203.3,
        203.4, 203.6, 203.7, 203.7, 203.4, 203.2, 203.1, 203.1, 203. ,
        203. , 202.9, 202.9, 202.8, 203. , 203.4, 203.8, 204. , 204.2],
       [213.2, 213.3, 213. , 212.8, 213. , 213.3, 213.3, 213.5, 214. ,
        214.5, 215.1, 214.7, 214.1, 213.3, 212.4, 211.1, 210. , 208.8,
        208. , 207. , 206.1, 205.4, 204.7, 204.1, 203.6, 203.1, 202.9,
        202.7, 202.7, 202.8, 202.9, 202.9, 203. , 203.1, 203.2, 203.3,
        203.3, 203.5, 203.2, 203. , 202.9, 202.8, 202.8, 202.8, 202.9,
        202.9, 202.8, 202.9, 203.1, 203.3, 203.6, 203.8, 204. , 204.3],
       [213.9, 213.8, 213.7, 213.7, 213.9, 213.7, 213.6, 213.3, 213.8,
        214.2, 214.5, 214.4, 213.6, 212.8, 211.7, 210.5, 209.5, 208.7,
        207.8, 206.9, 205.9, 205.1, 204.2, 203.5, 203.2, 203. , 202.7,
        202.6, 202.7, 202.7, 202.8, 202.9, 203. , 203.1, 203.2, 203.2,
        203.3, 203.2, 203.1, 203. , 202.8, 202.6, 202.6, 202.7, 202.7,
        202.8, 202.9, 203.1, 203.3, 203.6, 204. , 204.3, 204.5, 204.4],
       [214.9, 214.7, 214.5, 214.5, 214.5, 214.2, 213.8, 213.4, 213.6,
        214. , 214.1, 214.1, 213. , 212.1, 211.1, 210.2, 209.3, 208.3,
        207.3, 206.3, 205.4, 204.5, 203.7, 203.1, 202.9, 202.7, 202.6,
        202.6, 202.6, 202.7, 202.8, 202.9, 202.9, 203. , 203. , 203.1,
        203.2, 203.2, 203. , 202.8, 202.7, 202.6, 202.7, 202.8, 202.9,
        203. , 203.1, 203.3, 203.6, 204.1, 204.5, 204.6, 204.6, 204.5],
       [215.1, 215.2, 215.1, 215. , 215.1, 214.6, 214.1, 213.5, 213.4,
        213.6, 213.7, 213.8, 213. , 212. , 210.9, 209.9, 208.8, 207.7,
        206.6, 205.6, 204.7, 204. , 203.4, 202.8, 202.7, 202.6, 202.5,
        202.5, 202.6, 202.7, 202.7, 202.7, 202.8, 202.8, 203. , 203.1,
        203.2, 203.3, 203.1, 203. , 202.9, 202.8, 202.9, 203. , 203.1,
        203.1, 203.3, 203.7, 204.1, 204.4, 204.9, 204.8, 204.8, 204.8],
       [215.2, 215.3, 215.3, 215.3, 215.3, 214.8, 214.2, 213.6, 213.2,
        213.3, 213.5, 213.5, 212.7, 211.6, 210.4, 209.2, 208.1, 207. ,
        206. , 204.9, 204.2, 203.5, 202.9, 202.6, 202.4, 202.4, 202.4,
        202.5, 202.5, 202.6, 202.6, 202.7, 202.7, 202.9, 203. , 203.1,
        203.3, 203.4, 203.3, 203.2, 203.2, 203.1, 203.1, 203.1, 203.2,
        203.3, 203.4, 203.8, 204.2, 204.6, 205.1, 205.2, 205.1, 205. ],
       [215.2, 215.3, 215.2, 215.2, 215.2, 214.9, 214.3, 213.7, 213.4,
        213.5, 213.2, 212.8, 212.3, 211. , 209.8, 208.5, 207.4, 206.2,
        205.2, 204.3, 203.5, 203.1, 202.8, 202.4, 202.2, 202.2, 202.1,
        202.2, 202.4, 202.5, 202.6, 202.7, 202.8, 202.9, 203.1, 203.2,
        203.4, 203.5, 203.4, 203.3, 203.2, 203.2, 203.1, 203.2, 203.3,
        203.4, 203.6, 203.9, 204.3, 204.7, 205. , 205. , 204.9, 204.8],
       [215.1, 215.1, 215.1, 215. , 215. , 214.9, 214.4, 214. , 213.6,
        213.1, 212.7, 212.2, 211.5, 210.3, 209. , 207.7, 206.5, 205.5,
        204.6, 203.6, 203. , 202.6, 202.3, 202. , 201.8, 201.8, 201.9,
        202.1, 202.4, 202.5, 202.8, 203. , 203.2, 203.2, 203.5, 203.7,
        203.6, 203.5, 203.4, 203.3, 203.3, 203.2, 203.2, 203.3, 203.4,
        203.5, 203.5, 203.8, 204.1, 204.3, 204.5, 204.7, 204.8, 204.8],
       [214.9, 214.9, 214.8, 214.8, 214.7, 214.8, 214.4, 213.9, 213.3,
        212.7, 212.1, 211.3, 210.6, 209.4, 208.1, 206.8, 205.8, 204.8,
        203.9, 203.2, 202.5, 202.2, 202. , 201.7, 201.5, 201.6, 201.8,
        202. , 202.8, 203.4, 203.6, 203.7, 203.7, 203.8, 203.8, 203.8,
        203.8, 203.8, 203.7, 203.7, 203.6, 203.5, 203.3, 203.4, 203.5,
        203.6, 203.7, 203.8, 204.2, 204.5, 204.7, 204.9, 204.9, 204.9],
       [214.6, 214.6, 214.5, 214.5, 214.5, 214.6, 214.3, 213.6, 212.9,
        212. , 211.3, 210.5, 209.5, 208.4, 207.1, 206.1, 205.1, 204.2,
        203.4, 202.7, 202.1, 201.8, 201.5, 201.4, 201.4, 201.5, 202. ,
        202.7, 203.5, 204.2, 204.2, 204.2, 204.3, 204.6, 204.7, 204.7,
        204.5, 204.4, 204.3, 204.2, 204.1, 204. , 203.9, 204. , 204.1,
        204.1, 204.1, 204.2, 204.4, 204.6, 204.9, 205. , 205. , 205.1],
       [214.3, 214.2, 214.2, 214.3, 214.3, 214.2, 214. , 213.3, 212.3,
        211.3, 210.4, 209.4, 208.3, 207.3, 206.2, 205.1, 204.2, 203.4,
        202.6, 202.2, 201.7, 201.4, 201.4, 201.3, 201.2, 201.4, 202. ,
        202.7, 203.5, 204.4, 204.7, 205. , 205.2, 205.4, 205.6, 205.5,
        205.4, 205.3, 205.1, 205. , 204.9, 204.8, 204.7, 204.7, 204.7,
        204.8, 204.8, 204.6, 204.8, 205.1, 205.1, 205.1, 205.1, 205.2],
       [213.8, 213.9, 213.9, 214. , 213.9, 213.8, 213.4, 212.3, 211.1,
        210.1, 209.1, 208. , 207.1, 206.1, 205.1, 204.2, 203.4, 202.7,
        202.2, 201.7, 201.2, 200.7, 200.6, 200.8, 200.9, 201.1, 202. ,
        202.8, 203.7, 204.7, 205.3, 205.5, 205.9, 206.3, 206.7, 206.5,
        206.3, 206. , 205.9, 205.8, 205.7, 205.6, 205.6, 205.6, 205.6,
        205.5, 205.3, 205.1, 205.2, 205.2, 205.2, 205.2, 205.3, 205.4],
       [213.6, 213.7, 213.7, 213.5, 213.4, 212.7, 211.9, 210.8, 209.8,
        208.7, 207.8, 206.9, 206. , 205.5, 204.9, 204. , 203.1, 202.2,
        201.6, 201.1, 200.6, 200.2, 200.3, 200.4, 200.9, 201.4, 202.2,
        203.1, 204.1, 205.1, 205.9, 206.3, 206.6, 207. , 207.2, 207.1,
        206.8, 206.7, 206.7, 206.6, 206.5, 206.5, 206.5, 206.5, 206.3,
        206. , 205.8, 205.5, 205.5, 205.6, 205.6, 205.5, 205.4, 205.6],
       [213.6, 213.4, 213.2, 212.9, 212.2, 211.4, 210.5, 209.4, 208.4,
        207.6, 206.9, 206.2, 205.8, 205.3, 204.6, 203.6, 202.7, 201.9,
        201.2, 200.8, 200.4, 200. , 200.1, 200.6, 201.1, 201.7, 202.6,
        203.6, 204.6, 205.5, 206.5, 206.8, 207.2, 207.4, 207.7, 207.7,
        207.6, 207.6, 207.4, 207.3, 207.1, 207. , 206.9, 206.8, 206.5,
        206.2, 206. , 206. , 206. , 206. , 205.9, 205.7, 205.6, 205.7],
       [213.5, 213.3, 212.8, 212.2, 211.3, 210.2, 209.2, 208.3, 207.6,
        206.8, 206.2, 205.7, 205.2, 204.8, 204.3, 203.3, 202.4, 201.6,
        201. , 200.6, 200.2, 200. , 200.2, 200.6, 201.3, 202.1, 203. ,
        204. , 205. , 206. , 207. , 207.4, 207.7, 208. , 208.2, 208.1,
        208. , 207.8, 207.6, 207.4, 207.3, 207.1, 207. , 206.9, 206.7,
        206.6, 206.5, 206.4, 206.4, 206.3, 206.1, 206. , 205.8, 205.8],
       [213.5, 213. , 212.4, 211.7, 210.8, 209.8, 208.7, 207.6, 206.9,
        206.3, 205.7, 205.2, 204.8, 204.4, 204. , 203.1, 202.2, 201.5,
        200.9, 200.5, 200.3, 200.2, 200.2, 201. , 201.8, 202.7, 203.7,
        204.7, 205.8, 206.8, 207.6, 208.1, 208.4, 208.5, 208.5, 208.4,
        208.2, 208. , 207.8, 207.6, 207.4, 207.3, 207.2, 207.1, 207. ,
        206.9, 206.8, 206.7, 206.7, 206.5, 206.4, 206.2, 205.9, 205.8],
       [213.5, 213. , 212.1, 211.3, 210.4, 209.3, 208.2, 207.2, 206.5,
        205.9, 205.4, 205. , 204.6, 204.1, 203.7, 202.9, 202.1, 201.4,
        200.8, 200.7, 200.6, 200.5, 200.7, 201.5, 202.4, 203.4, 204.6,
        205.7, 206.7, 207.5, 208.2, 208.8, 208.9, 208.8, 208.7, 208.7,
        208.4, 208.2, 208. , 207.8, 207.5, 207.4, 207.3, 207.3, 207.2,
        207.1, 207. , 207. , 207. , 206.8, 206.6, 206.3, 206. , 205.8],
       [213.1, 212.6, 211.7, 210.9, 210.1, 209. , 208. , 207. , 206.2,
        205.7, 205.2, 204.7, 204.3, 203.9, 203.6, 203. , 202.2, 201.6,
        201.2, 201. , 200.9, 201.1, 201.4, 202.1, 203.2, 204.3, 205.5,
        206.5, 207.3, 208.1, 208.8, 209.3, 209.3, 209.2, 209.2, 208.9,
        208.7, 208.5, 208.2, 207.9, 207.7, 207.6, 207.5, 207.4, 207.4,
        207.3, 207.3, 207.3, 207.2, 207.1, 206.8, 206.4, 206.1, 205.9],
       [212.6, 212.1, 211.3, 210.5, 209.7, 208.8, 207.8, 206.8, 206.2,
        205.8, 205.4, 205. , 204.6, 204.2, 203.7, 203.2, 202.4, 202. ,
        201.6, 201.3, 201.6, 201.9, 202.2, 202.8, 204. , 205.1, 206.2,
        207.2, 208. , 208.7, 209.3, 209.7, 209.7, 209.7, 209.4, 209.2,
        209. , 208.7, 208.4, 208.1, 207.8, 207.7, 207.7, 207.6, 207.6,
        207.5, 207.5, 207.5, 207.5, 207.3, 206.9, 206.6, 206.3, 206.2],
       [211.9, 211.5, 210.8, 210. , 209.4, 208.7, 207.7, 207. , 206.5,
        206. , 205.6, 205.3, 204.9, 204.3, 203.8, 203.3, 202.9, 202.5,
        202.2, 202. , 202.2, 202.5, 202.9, 203.5, 204.6, 205.8, 206.8,
        207.9, 208.6, 209.2, 209.7, 210.1, 210.2, 210. , 209.7, 209.5,
        209.2, 208.9, 208.5, 208.2, 208. , 207.9, 207.9, 207.8, 207.8,
        207.7, 207.7, 207.7, 207.6, 207.5, 207.1, 206.8, 206.6, 206.5],
       [210.9, 210.8, 210.2, 209.6, 209.1, 208.6, 207.8, 207.3, 206.8,
        206.3, 206. , 205.6, 205.4, 204.9, 204.4, 204.1, 203.7, 203.3,
        203. , 202.7, 202.9, 203.3, 203.7, 204.2, 205.3, 206.4, 207.4,
        208.5, 209.1, 209.5, 210. , 210.4, 210.3, 210. , 209.8, 209.5,
        209.3, 208.9, 208.5, 208.2, 208.2, 208.1, 208.1, 208. , 208. ,
        207.9, 207.9, 207.9, 207.8, 207.7, 207.3, 207. , 206.9, 206.8],
       [210.1, 210. , 209.8, 209.3, 208.8, 208.4, 207.9, 207.3, 207. ,
        206.8, 206.4, 206.2, 206.2, 205.7, 205.3, 204.9, 204.6, 204.1,
        203.8, 203.5, 203.7, 204.1, 204.6, 205. , 205.9, 207. , 208. ,
        208.9, 209.4, 209.8, 210.3, 210.4, 210.3, 210.1, 209.8, 209.5,
        209.3, 208.9, 208.6, 208.4, 208.4, 208.4, 208.3, 208.2, 208.2,
        208.1, 208.1, 208. , 207.9, 207.8, 207.5, 207.4, 207.3, 207.2],
       [209.2, 209.1, 209.1, 208.6, 208.1, 208.1, 207.7, 207.3, 207.3,
        207.3, 207. , 206.9, 206.9, 206.5, 206.2, 205.8, 205.4, 204.9,
        204.5, 204. , 204.3, 204.7, 205.1, 205.6, 206.4, 207.5, 208.6,
        209.2, 209.6, 210.1, 210.5, 210.4, 210.3, 210.1, 209.8, 209.5,
        209.3, 209. , 208.7, 208.7, 208.6, 208.6, 208.5, 208.5, 208.4,
        208.3, 208.3, 208.2, 208.1, 208. , 207.8, 207.7, 207.6, 207.5],
       [208.2, 208.2, 208.3, 207.9, 207.8, 207.8, 207.6, 207.6, 207.7,
        207.7, 207.5, 207.3, 207.1, 206.9, 206.4, 205.9, 205.7, 205.3,
        204.8, 204.7, 204.9, 205.4, 205.8, 206.3, 207. , 208.1, 208.9,
        209.4, 209.9, 210.4, 210.5, 210.4, 210.3, 210.1, 209.8, 209.6,
        209.3, 209. , 208.9, 208.9, 208.9, 208.8, 208.8, 208.7, 208.6,
        208.6, 208.4, 208.3, 208.2, 208.1, 208. , 207.9, 207.7, 207.6],
       [207.2, 207.3, 207.4, 207.4, 207.5, 207.5, 207.5, 207.5, 207.4,
        207.5, 207.4, 207.2, 207.1, 207.1, 206.6, 206.1, 206.1, 205.7,
        205.2, 205.4, 205.6, 206. , 206.5, 207. , 207.6, 208.6, 209.2,
        209.7, 210.2, 210.5, 210.4, 210.4, 210.3, 210.1, 209.8, 209.5,
        209.2, 209.1, 209.1, 209. , 208.9, 209.2, 209.1, 209.1, 209. ,
        208.8, 208.6, 208.5, 208.4, 208.3, 208.1, 208. , 207.9, 208. ],
       [206.6, 206.9, 207.1, 207.1, 207.1, 207.1, 207.2, 207.1, 207.1,
        207.3, 207.3, 207.1, 207.1, 207.3, 206.7, 206.5, 206.4, 206.2,
        205.9, 206.1, 206.3, 206.7, 207.2, 207.7, 208.2, 208.9, 209.4,
        210. , 210.5, 210.5, 210.4, 210.3, 210.3, 209.8, 209.5, 209.2,
        208.9, 209.1, 209.1, 209. , 209. , 209.5, 209.5, 209.5, 209.4,
        209. , 208.8, 208.7, 208.6, 208.4, 208.2, 208.1, 208. , 208.3],
       [206.3, 206.6, 206.7, 206.7, 206.7, 206.7, 206.8, 206.8, 206.8,
        207. , 207.1, 207. , 207.1, 207.3, 206.9, 206.8, 206.7, 206.6,
        206.7, 206.9, 207. , 207.4, 207.9, 208.4, 208.9, 209.3, 209.8,
        210.4, 210.7, 210.4, 210.4, 210.3, 210. , 209.5, 209.2, 208.9,
        208.9, 209.1, 209.1, 209. , 209.2, 209.8, 209.9, 209.8, 209.6,
        209.2, 209. , 208.9, 208.7, 208.5, 208.4, 208.2, 208.3, 208.6],
       [206. , 206.3, 206.3, 206.3, 206.3, 206.4, 206.4, 206.5, 206.6,
        206.8, 207. , 207. , 207.2, 207.3, 207.3, 207.2, 207.1, 207.1,
        207.3, 207.4, 207.5, 208.1, 208.6, 209.1, 209.4, 209.7, 210.3,
        210.8, 210.7, 210.4, 210.3, 210.3, 209.8, 209.2, 208.9, 208.6,
        208.8, 209.1, 209.1, 209.1, 209.5, 210.1, 210.3, 210.2, 209.9,
        209.5, 209.1, 209. , 208.9, 208.7, 208.5, 208.4, 208.7, 208.9],
       [205.8, 206. , 205.9, 205.9, 205.9, 206. , 206.1, 206.2, 206.4,
        206.6, 206.8, 207.1, 207.3, 207.4, 207.7, 207.6, 207.5, 207.6,
        207.8, 207.9, 208.1, 208.7, 209.2, 209.7, 209.9, 210.1, 210.7,
        211.1, 210.7, 210.4, 210.3, 210.1, 209.5, 209. , 208.6, 208.5,
        208.8, 209.1, 209.1, 209.3, 209.8, 210.4, 210.7, 210.5, 210.1,
        209.7, 209.3, 209.2, 209. , 208.8, 208.6, 208.7, 209. , 209.3],
       [205.5, 205.6, 205.6, 205.6, 205.6, 205.8, 205.9, 206. , 206.3,
        206.5, 206.8, 207.3, 207.4, 207.6, 208.2, 208. , 207.9, 208.1,
        208.3, 208.3, 208.7, 209.4, 209.9, 210.3, 210.4, 210.6, 211.1,
        211.1, 210.8, 210.4, 210.3, 209.9, 209.3, 208.7, 208.3, 208.5,
        208.8, 209. , 209.2, 209.6, 210.1, 210.7, 211.2, 210.8, 210.4,
        210. , 209.6, 209.3, 209.2, 209. , 208.8, 208.9, 209.2, 209.5],
       [205.1, 205.4, 205.4, 205.5, 205.5, 205.6, 205.7, 205.9, 206.1,
        206.4, 206.9, 207.4, 207.6, 207.9, 208.4, 208.4, 208.4, 208.6,
        208.7, 208.8, 209.4, 210. , 210.5, 210.8, 211. , 211.1, 211.5,
        211.2, 210.8, 210.4, 210.2, 209.6, 209. , 208.5, 208.2, 208.5,
        208.7, 209. , 209.3, 209.8, 210.3, 210.8, 211.4, 211.1, 210.7,
        210.2, 209.9, 209.6, 209.4, 209.2, 209.1, 209.1, 209.4, 209.6],
       [204.7, 205.2, 205.3, 205.3, 205.3, 205.4, 205.5, 205.8, 206. ,
        206.2, 206.9, 207.5, 207.7, 208.1, 208.7, 208.8, 208.9, 209.1,
        209.2, 209.4, 210. , 210.6, 211.2, 211.4, 211.5, 211.6, 211.6,
        211.2, 210.8, 210.4, 209.9, 209.3, 208.7, 208.2, 208.1, 208.3,
        208.6, 208.8, 209.4, 209.9, 210.4, 210.9, 211.5, 211.4, 211. ,
        210.5, 210.1, 209.9, 209.7, 209.5, 209.3, 209.3, 209.5, 209.8],
       [204.3, 204.9, 205.1, 205.1, 205.1, 205.2, 205.3, 205.6, 205.9,
        206.2, 206.9, 207.6, 207.9, 208.4, 208.9, 209.3, 209.4, 209.5,
        209.7, 210.1, 210.7, 211.2, 211.6, 211.7, 211.7, 211.8, 211.5,
        211.1, 210.7, 210.2, 209.6, 209. , 208.4, 207.9, 207.9, 208.1,
        208.4, 208.8, 209.4, 209.9, 210.4, 211. , 211.7, 211.7, 211.3,
        210.8, 210.4, 210.1, 209.9, 209.7, 209.5, 209.4, 209.7, 209.9],
       [204. , 204.5, 204.9, 205. , 205. , 205. , 205.2, 205.5, 205.8,
        206.2, 206.9, 207.6, 208.1, 208.7, 209.2, 209.7, 209.8, 210. ,
        210.1, 210.2, 210.7, 211.2, 211.6, 211.7, 211.7, 211.7, 211.3,
        210.9, 210.5, 209.9, 209.2, 208.6, 208. , 207.8, 207.8, 208. ,
        208.2, 208.8, 209.4, 210. , 210.5, 211.1, 211.8, 212. , 211.6,
        211.1, 210.7, 210.4, 210.2, 210. , 209.8, 209.6, 209.9, 210.1]])

    speed_1d = numpy.array([0.39, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.44, 0.44, 0.43, 0.43, 0.42,
       0.43, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.48, 0.49,
       0.5 , 0.51, 0.51, 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54,
       0.54, 0.55, 0.55, 0.56, 0.57, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 ,
       0.6 , 0.6 , 0.61, 0.62, 0.63, 0.64, 0.65, 0.66, 0.67, 0.67, 0.39,
       0.39, 0.39, 0.39, 0.4 , 0.42, 0.43, 0.43, 0.42, 0.42, 0.42, 0.42,
       0.43, 0.43, 0.44, 0.45, 0.45, 0.45, 0.46, 0.47, 0.47, 0.49, 0.49,
       0.49, 0.5 , 0.5 , 0.51, 0.51, 0.52, 0.53, 0.53, 0.53, 0.54, 0.54,
       0.54, 0.55, 0.56, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 ,
       0.6 , 0.61, 0.63, 0.64, 0.64, 0.65, 0.66, 0.67, 0.67, 0.39, 0.39,
       0.38, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.41, 0.41, 0.41, 0.42, 0.42,
       0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.47, 0.48, 0.49, 0.49,
       0.49, 0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.53, 0.54, 0.54,
       0.55, 0.56, 0.57, 0.58, 0.58, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.61,
       0.62, 0.63, 0.63, 0.64, 0.65, 0.65, 0.66, 0.67, 0.39, 0.38, 0.37,
       0.37, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.41, 0.42, 0.42, 0.42, 0.43,
       0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 ,
       0.5 , 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.53, 0.53, 0.54, 0.55,
       0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.61, 0.61,
       0.62, 0.62, 0.62, 0.63, 0.64, 0.65, 0.66, 0.38, 0.38, 0.37, 0.36,
       0.36, 0.37, 0.38, 0.4 , 0.41, 0.42, 0.42, 0.42, 0.42, 0.43, 0.44,
       0.44, 0.45, 0.45, 0.46, 0.47, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 ,
       0.51, 0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55,
       0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 ,
       0.61, 0.61, 0.62, 0.63, 0.64, 0.65, 0.38, 0.38, 0.37, 0.36, 0.36,
       0.37, 0.38, 0.4 , 0.42, 0.42, 0.42, 0.42, 0.43, 0.43, 0.44, 0.44,
       0.45, 0.46, 0.46, 0.47, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 ,
       0.51, 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56,
       0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 ,
       0.6 , 0.61, 0.62, 0.63, 0.64, 0.37, 0.37, 0.37, 0.36, 0.36, 0.37,
       0.38, 0.4 , 0.41, 0.42, 0.42, 0.42, 0.43, 0.43, 0.44, 0.44, 0.45,
       0.46, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.5 , 0.51,
       0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57,
       0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 ,
       0.61, 0.62, 0.63, 0.65, 0.36, 0.37, 0.37, 0.36, 0.36, 0.36, 0.38,
       0.39, 0.4 , 0.41, 0.41, 0.42, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46,
       0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.52,
       0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.57,
       0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.62,
       0.63, 0.64, 0.66, 0.36, 0.36, 0.37, 0.37, 0.37, 0.37, 0.37, 0.38,
       0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.45, 0.46, 0.46,
       0.47, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.53, 0.54,
       0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.57, 0.58,
       0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.63,
       0.65, 0.66, 0.36, 0.36, 0.37, 0.37, 0.37, 0.37, 0.37, 0.38, 0.39,
       0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46,
       0.47, 0.48, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55,
       0.55, 0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.58, 0.59,
       0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.64,
       0.65, 0.36, 0.36, 0.37, 0.37, 0.37, 0.38, 0.38, 0.39, 0.39, 0.39,
       0.39, 0.4 , 0.41, 0.43, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47,
       0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55,
       0.55, 0.56, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.59,
       0.6 , 0.6 , 0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.63, 0.64, 0.64,
       0.36, 0.36, 0.37, 0.38, 0.38, 0.39, 0.39, 0.39, 0.39, 0.4 , 0.4 ,
       0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.46, 0.47, 0.48,
       0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55,
       0.55, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.58, 0.59, 0.6 , 0.61,
       0.61, 0.61, 0.61, 0.62, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63, 0.36,
       0.37, 0.38, 0.38, 0.39, 0.39, 0.39, 0.4 , 0.4 , 0.41, 0.41, 0.43,
       0.43, 0.44, 0.45, 0.46, 0.47, 0.47, 0.47, 0.46, 0.47, 0.48, 0.49,
       0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.56,
       0.56, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.6 , 0.61, 0.61, 0.61,
       0.61, 0.62, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63, 0.63, 0.36, 0.37,
       0.38, 0.39, 0.39, 0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.44, 0.44,
       0.45, 0.47, 0.48, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.51,
       0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.56, 0.56, 0.56,
       0.56, 0.57, 0.57, 0.58, 0.59, 0.6 , 0.6 , 0.61, 0.61, 0.61, 0.61,
       0.61, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63, 0.63, 0.37, 0.38, 0.38,
       0.38, 0.39, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.43, 0.44, 0.46, 0.47,
       0.48, 0.5 , 0.5 , 0.49, 0.49, 0.49, 0.49, 0.5 , 0.51, 0.51, 0.52,
       0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.56, 0.56, 0.57,
       0.58, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61,
       0.61, 0.62, 0.62, 0.62, 0.63, 0.63, 0.63, 0.37, 0.38, 0.38, 0.38,
       0.39, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.44, 0.46, 0.48, 0.49, 0.5 ,
       0.5 , 0.5 , 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.53,
       0.54, 0.54, 0.55, 0.55, 0.56, 0.56, 0.56, 0.56, 0.57, 0.57, 0.58,
       0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.61,
       0.61, 0.61, 0.62, 0.62, 0.63, 0.63, 0.37, 0.38, 0.38, 0.39, 0.39,
       0.4 , 0.4 , 0.41, 0.42, 0.43, 0.45, 0.48, 0.49, 0.5 , 0.5 , 0.5 ,
       0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.54,
       0.54, 0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.57, 0.58, 0.59, 0.59,
       0.59, 0.59, 0.59, 0.59, 0.6 , 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 ,
       0.61, 0.61, 0.61, 0.62, 0.62, 0.38, 0.39, 0.39, 0.4 , 0.4 , 0.4 ,
       0.41, 0.42, 0.43, 0.44, 0.46, 0.48, 0.49, 0.48, 0.47, 0.48, 0.49,
       0.49, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.54,
       0.55, 0.55, 0.56, 0.56, 0.57, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59,
       0.59, 0.6 , 0.6 , 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 ,
       0.6 , 0.61, 0.61, 0.62, 0.38, 0.4 , 0.41, 0.41, 0.41, 0.41, 0.42,
       0.43, 0.44, 0.45, 0.46, 0.46, 0.45, 0.45, 0.45, 0.45, 0.46, 0.47,
       0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.52, 0.53, 0.53, 0.53, 0.54,
       0.55, 0.56, 0.57, 0.57, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 ,
       0.6 , 0.6 , 0.6 , 0.59, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 ,
       0.6 , 0.61, 0.61, 0.39, 0.4 , 0.41, 0.41, 0.41, 0.42, 0.43, 0.44,
       0.44, 0.44, 0.44, 0.44, 0.43, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47,
       0.48, 0.49, 0.5 , 0.51, 0.51, 0.51, 0.51, 0.51, 0.52, 0.53, 0.54,
       0.56, 0.57, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.61,
       0.6 , 0.6 , 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 ,
       0.61, 0.61, 0.39, 0.39, 0.4 , 0.41, 0.42, 0.42, 0.44, 0.44, 0.44,
       0.44, 0.43, 0.43, 0.42, 0.42, 0.42, 0.44, 0.45, 0.46, 0.47, 0.48,
       0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.52, 0.54, 0.55,
       0.57, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.6 ,
       0.6 , 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.6 , 0.61,
       0.62, 0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.44, 0.45, 0.44, 0.43,
       0.43, 0.42, 0.42, 0.43, 0.43, 0.44, 0.46, 0.47, 0.48, 0.49, 0.5 ,
       0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.52, 0.53, 0.55, 0.56,
       0.57, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.59,
       0.59, 0.59, 0.59, 0.6 , 0.6 , 0.59, 0.59, 0.6 , 0.61, 0.61, 0.62,
       0.37, 0.38, 0.39, 0.39, 0.4 , 0.42, 0.44, 0.45, 0.44, 0.43, 0.43,
       0.42, 0.43, 0.43, 0.45, 0.46, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.5 ,
       0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.51, 0.53, 0.54, 0.56, 0.57,
       0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59, 0.59,
       0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.6 , 0.61, 0.62, 0.62, 0.37,
       0.37, 0.38, 0.39, 0.4 , 0.41, 0.43, 0.45, 0.44, 0.44, 0.43, 0.43,
       0.44, 0.45, 0.46, 0.47, 0.48, 0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.5 ,
       0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.55, 0.56, 0.57,
       0.57, 0.57, 0.58, 0.58, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.6 ,
       0.61, 0.61, 0.61, 0.61, 0.6 , 0.6 , 0.61, 0.62, 0.62, 0.37, 0.37,
       0.38, 0.39, 0.39, 0.41, 0.43, 0.44, 0.45, 0.44, 0.44, 0.43, 0.44,
       0.46, 0.47, 0.48, 0.49, 0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 ,
       0.5 , 0.5 , 0.5 , 0.49, 0.49, 0.51, 0.52, 0.54, 0.55, 0.56, 0.56,
       0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.61, 0.62,
       0.62, 0.62, 0.61, 0.61, 0.61, 0.61, 0.62, 0.62, 0.37, 0.38, 0.38,
       0.39, 0.39, 0.41, 0.42, 0.44, 0.45, 0.45, 0.45, 0.44, 0.45, 0.46,
       0.47, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.5 , 0.5 ,
       0.49, 0.49, 0.48, 0.48, 0.5 , 0.51, 0.53, 0.55, 0.56, 0.56, 0.56,
       0.57, 0.57, 0.58, 0.58, 0.58, 0.59, 0.59, 0.6 , 0.61, 0.62, 0.63,
       0.62, 0.62, 0.62, 0.61, 0.62, 0.62, 0.62, 0.37, 0.38, 0.39, 0.39,
       0.4 , 0.41, 0.42, 0.44, 0.45, 0.45, 0.45, 0.45, 0.46, 0.46, 0.47,
       0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.5 , 0.5 , 0.5 , 0.49, 0.49,
       0.48, 0.48, 0.47, 0.48, 0.5 , 0.52, 0.54, 0.55, 0.56, 0.56, 0.57,
       0.57, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.61, 0.62, 0.63, 0.63, 0.63,
       0.63, 0.62, 0.62, 0.62, 0.62, 0.62, 0.37, 0.38, 0.39, 0.39, 0.4 ,
       0.41, 0.42, 0.43, 0.45, 0.45, 0.45, 0.45, 0.45, 0.45, 0.46, 0.47,
       0.47, 0.48, 0.48, 0.49, 0.5 , 0.5 , 0.5 , 0.49, 0.49, 0.49, 0.48,
       0.47, 0.47, 0.47, 0.49, 0.51, 0.53, 0.55, 0.56, 0.56, 0.57, 0.57,
       0.58, 0.58, 0.59, 0.59, 0.6 , 0.61, 0.62, 0.63, 0.63, 0.63, 0.63,
       0.63, 0.62, 0.62, 0.62, 0.61, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41,
       0.43, 0.44, 0.45, 0.45, 0.45, 0.45, 0.44, 0.45, 0.45, 0.46, 0.46,
       0.47, 0.47, 0.49, 0.5 , 0.5 , 0.49, 0.49, 0.49, 0.48, 0.47, 0.47,
       0.46, 0.47, 0.49, 0.51, 0.53, 0.55, 0.55, 0.56, 0.56, 0.56, 0.57,
       0.57, 0.58, 0.59, 0.6 , 0.61, 0.61, 0.62, 0.63, 0.63, 0.63, 0.62,
       0.62, 0.62, 0.61, 0.61, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.43,
       0.43, 0.44, 0.44, 0.44, 0.44, 0.43, 0.43, 0.44, 0.45, 0.45, 0.46,
       0.47, 0.48, 0.49, 0.5 , 0.49, 0.49, 0.48, 0.48, 0.47, 0.47, 0.46,
       0.46, 0.48, 0.5 , 0.52, 0.52, 0.53, 0.53, 0.54, 0.55, 0.56, 0.56,
       0.57, 0.58, 0.59, 0.59, 0.6 , 0.61, 0.62, 0.62, 0.62, 0.62, 0.61,
       0.61, 0.61, 0.61, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.42, 0.43,
       0.43, 0.43, 0.43, 0.43, 0.42, 0.42, 0.43, 0.44, 0.45, 0.46, 0.47,
       0.48, 0.49, 0.5 , 0.49, 0.49, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46,
       0.47, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.56,
       0.56, 0.57, 0.58, 0.59, 0.6 , 0.61, 0.61, 0.61, 0.61, 0.61, 0.61,
       0.61, 0.61, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.41, 0.42, 0.42, 0.42,
       0.42, 0.41, 0.41, 0.41, 0.41, 0.42, 0.43, 0.45, 0.46, 0.47, 0.48,
       0.49, 0.5 , 0.49, 0.49, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47,
       0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.56,
       0.56, 0.57, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61,
       0.61, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.4 , 0.4 , 0.4 , 0.4 ,
       0.4 , 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.45, 0.46, 0.47, 0.48, 0.49,
       0.5 , 0.49, 0.48, 0.48, 0.47, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47,
       0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56,
       0.56, 0.57, 0.58, 0.59, 0.59, 0.6 , 0.6 , 0.6 , 0.61, 0.61, 0.61,
       0.36, 0.37, 0.38, 0.39, 0.39, 0.39, 0.39, 0.38, 0.38, 0.38, 0.39,
       0.39, 0.39, 0.4 , 0.41, 0.42, 0.44, 0.46, 0.47, 0.48, 0.48, 0.49,
       0.49, 0.48, 0.48, 0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48,
       0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.56,
       0.57, 0.58, 0.58, 0.59, 0.59, 0.59, 0.6 , 0.61, 0.61, 0.61, 0.35,
       0.36, 0.36, 0.37, 0.37, 0.37, 0.37, 0.37, 0.37, 0.37, 0.37, 0.38,
       0.38, 0.39, 0.4 , 0.42, 0.43, 0.45, 0.46, 0.47, 0.48, 0.48, 0.49,
       0.48, 0.47, 0.47, 0.47, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48,
       0.49, 0.5 , 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.56, 0.56,
       0.57, 0.57, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.61, 0.61, 0.34, 0.34,
       0.35, 0.35, 0.35, 0.35, 0.35, 0.35, 0.35, 0.36, 0.36, 0.37, 0.38,
       0.38, 0.39, 0.41, 0.43, 0.44, 0.46, 0.46, 0.47, 0.48, 0.48, 0.48,
       0.47, 0.47, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49,
       0.49, 0.5 , 0.51, 0.52, 0.53, 0.54, 0.54, 0.55, 0.56, 0.56, 0.57,
       0.57, 0.58, 0.58, 0.58, 0.59, 0.6 , 0.6 , 0.6 , 0.33, 0.33, 0.33,
       0.33, 0.33, 0.33, 0.34, 0.34, 0.34, 0.35, 0.35, 0.36, 0.36, 0.38,
       0.39, 0.4 , 0.42, 0.43, 0.45, 0.46, 0.46, 0.47, 0.48, 0.47, 0.47,
       0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49,
       0.5 , 0.51, 0.52, 0.53, 0.53, 0.54, 0.54, 0.55, 0.56, 0.56, 0.57,
       0.57, 0.57, 0.58, 0.59, 0.59, 0.59, 0.59, 0.32, 0.32, 0.32, 0.32,
       0.32, 0.33, 0.33, 0.33, 0.33, 0.34, 0.34, 0.35, 0.36, 0.37, 0.38,
       0.39, 0.41, 0.43, 0.44, 0.45, 0.46, 0.46, 0.47, 0.46, 0.46, 0.46,
       0.46, 0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 ,
       0.51, 0.52, 0.52, 0.53, 0.53, 0.54, 0.55, 0.56, 0.56, 0.57, 0.57,
       0.57, 0.57, 0.57, 0.58, 0.58, 0.58, 0.31, 0.32, 0.32, 0.32, 0.31,
       0.32, 0.32, 0.32, 0.32, 0.33, 0.33, 0.34, 0.35, 0.36, 0.37, 0.39,
       0.4 , 0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46,
       0.46, 0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 ,
       0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.56, 0.56, 0.56, 0.56, 0.57,
       0.57, 0.57, 0.57, 0.57, 0.57, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
       0.31, 0.31, 0.32, 0.32, 0.33, 0.34, 0.34, 0.36, 0.37, 0.38, 0.39,
       0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.45, 0.45, 0.46, 0.46,
       0.47, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.51,
       0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.56, 0.56, 0.56, 0.56, 0.56,
       0.56, 0.57, 0.57, 0.56, 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
       0.31, 0.31, 0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.4 ,
       0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.45, 0.45, 0.46, 0.46, 0.47,
       0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.51,
       0.51, 0.52, 0.53, 0.53, 0.54, 0.55, 0.55, 0.56, 0.56, 0.56, 0.56,
       0.56, 0.56, 0.56, 0.3 , 0.31, 0.31, 0.31, 0.31, 0.3 , 0.31, 0.31,
       0.31, 0.31, 0.32, 0.33, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.4 ,
       0.41, 0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.45, 0.46, 0.47, 0.47,
       0.47, 0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 , 0.5 ,
       0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55, 0.56, 0.55,
       0.55, 0.55, 0.3 , 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
       0.31, 0.32, 0.32, 0.33, 0.34, 0.34, 0.35, 0.36, 0.38, 0.39, 0.4 ,
       0.41, 0.42, 0.42, 0.43, 0.44, 0.45, 0.45, 0.46, 0.47, 0.47, 0.47,
       0.47, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.5 ,
       0.51, 0.52, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55, 0.55,
       0.55, 0.29, 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
       0.31, 0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 ,
       0.41, 0.42, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47,
       0.48, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.51,
       0.52, 0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.54,
       0.29, 0.3 , 0.3 , 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31, 0.31,
       0.32, 0.32, 0.32, 0.33, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 , 0.4 ,
       0.41, 0.43, 0.44, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.47,
       0.47, 0.48, 0.48, 0.48, 0.48, 0.48, 0.49, 0.49, 0.49, 0.5 , 0.51,
       0.53, 0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.28,
       0.29, 0.3 , 0.31, 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.31, 0.31,
       0.32, 0.32, 0.33, 0.34, 0.35, 0.36, 0.37, 0.38, 0.39, 0.4 , 0.41,
       0.42, 0.43, 0.44, 0.45, 0.46, 0.46, 0.47, 0.47, 0.47, 0.47, 0.47,
       0.47, 0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52,
       0.53, 0.54, 0.54, 0.54, 0.54, 0.55, 0.55, 0.54, 0.55, 0.28, 0.28,
       0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.31,
       0.32, 0.32, 0.33, 0.34, 0.36, 0.37, 0.37, 0.39, 0.4 , 0.41, 0.42,
       0.43, 0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47, 0.47,
       0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53,
       0.53, 0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.55, 0.27, 0.28, 0.29,
       0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.31,
       0.32, 0.33, 0.34, 0.35, 0.37, 0.38, 0.39, 0.4 , 0.41, 0.42, 0.43,
       0.44, 0.44, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47, 0.47,
       0.47, 0.47, 0.47, 0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.53,
       0.53, 0.53, 0.54, 0.55, 0.55, 0.55, 0.55, 0.26, 0.27, 0.28, 0.29,
       0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.32,
       0.33, 0.34, 0.36, 0.37, 0.38, 0.39, 0.39, 0.41, 0.42, 0.43, 0.43,
       0.44, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
       0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53,
       0.53, 0.54, 0.54, 0.55, 0.55, 0.55, 0.26, 0.27, 0.28, 0.29, 0.29,
       0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.31, 0.32, 0.33,
       0.34, 0.36, 0.37, 0.38, 0.39, 0.39, 0.41, 0.42, 0.42, 0.43, 0.44,
       0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47,
       0.47, 0.47, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.53, 0.53,
       0.54, 0.54, 0.55, 0.55, 0.55, 0.26, 0.26, 0.27, 0.28, 0.29, 0.29,
       0.29, 0.29, 0.3 , 0.3 , 0.3 , 0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34,
       0.36, 0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.44, 0.44,
       0.45, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47,
       0.47, 0.48, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.53,
       0.54, 0.55, 0.55, 0.55, 0.25, 0.26, 0.27, 0.28, 0.28, 0.29, 0.29,
       0.29, 0.29, 0.29, 0.3 , 0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36,
       0.37, 0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.44, 0.44, 0.44,
       0.45, 0.45, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.46, 0.47, 0.47,
       0.48, 0.48, 0.48, 0.49, 0.5 , 0.51, 0.51, 0.52, 0.53, 0.53, 0.54,
       0.55, 0.55, 0.55, 0.25, 0.26, 0.26, 0.27, 0.28, 0.28, 0.29, 0.29,
       0.29, 0.29, 0.29, 0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.38,
       0.38, 0.39, 0.4 , 0.4 , 0.41, 0.42, 0.43, 0.43, 0.44, 0.44, 0.44,
       0.44, 0.45, 0.45, 0.45, 0.45, 0.46, 0.46, 0.46, 0.47, 0.47, 0.48,
       0.48, 0.48, 0.49, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55,
       0.55, 0.55, 0.25, 0.25, 0.26, 0.27, 0.27, 0.28, 0.28, 0.29, 0.29,
       0.29, 0.29, 0.3 , 0.3 , 0.31, 0.32, 0.33, 0.34, 0.36, 0.38, 0.38,
       0.39, 0.4 , 0.4 , 0.41, 0.42, 0.42, 0.43, 0.43, 0.43, 0.44, 0.44,
       0.44, 0.44, 0.45, 0.45, 0.45, 0.45, 0.46, 0.47, 0.48, 0.48, 0.48,
       0.48, 0.48, 0.49, 0.5 , 0.51, 0.52, 0.52, 0.53, 0.54, 0.55, 0.55,
       0.55])

    direction_1d = numpy.array([197.7, 197.4, 197.1, 197.7, 198.4, 199.2, 199.9, 200.5, 200.9,
       201.4, 201.8, 201.8, 201.9, 202.1, 202.4, 202.5, 202.5, 202.8,
       202.8, 203.1, 203.1, 202.9, 202.6, 202.3, 202. , 201.8, 201.6,
       201.3, 201.2, 201.3, 201.6, 201.8, 202. , 202.2, 202.6, 203. ,
       203.4, 203.8, 204. , 204. , 203.8, 203.4, 203.2, 202.8, 202.5,
       202.2, 201.9, 201.6, 201.6, 201.7, 201.7, 201.7, 201.8, 202.1,
       197.6, 197.5, 197.7, 198.2, 199. , 199.8, 200.4, 200.8, 201.2,
       201.6, 202.2, 202.7, 202.9, 203.2, 203.4, 203.5, 203.5, 203.7,
       203.9, 204.1, 203.9, 203.6, 203.2, 202.8, 202.5, 202.2, 202. ,
       201.9, 201.7, 201.7, 202. , 202.2, 202.4, 202.6, 202.9, 203.3,
       203.6, 203.9, 204.2, 204. , 203.8, 203.4, 203. , 202.6, 202.4,
       202.2, 201.9, 201.7, 201.5, 201.6, 201.7, 201.7, 201.8, 202. ,
       198. , 198.2, 198.4, 198.8, 199.7, 200.2, 200.6, 201.3, 201.9,
       202.5, 203.2, 203.8, 204.1, 204.3, 204.3, 204.2, 204.4, 204.5,
       204.6, 204.7, 204.6, 204.2, 203.8, 203.4, 203. , 202.7, 202.4,
       202.3, 202.2, 202.2, 202.3, 202.5, 202.7, 203. , 203.2, 203.6,
       204. , 204.1, 204.2, 204.2, 203.8, 203.4, 203.1, 202.7, 202.5,
       202.3, 202.2, 201.9, 201.6, 201.6, 201.7, 201.8, 201.8, 202. ,
       198.7, 199. , 199.3, 199.5, 199.9, 200.5, 201.3, 202.1, 202.9,
       203.5, 204.2, 204.9, 205. , 204.9, 204.9, 204.9, 205. , 205. ,
       205.1, 205. , 205. , 204.7, 204.2, 203.8, 203.3, 202.9, 202.8,
       202.6, 202.6, 202.5, 202.6, 202.8, 203. , 203.4, 203.7, 204. ,
       204.2, 204.3, 204.3, 204.2, 203.9, 203.6, 203.3, 203.2, 203. ,
       202.7, 202.4, 202.1, 201.8, 201.7, 201.8, 201.8, 201.9, 202.2,
       199.1, 199.2, 199.5, 199.7, 200.2, 201.1, 202.1, 203.1, 203.8,
       204.4, 204.9, 205.2, 205.4, 205.3, 205.1, 205.3, 205.5, 205.5,
       205.5, 205.4, 205.3, 205.1, 204.6, 204.1, 203.6, 203.2, 203. ,
       203. , 202.9, 202.9, 202.9, 203.2, 203.5, 203.8, 204. , 204.3,
       204.3, 204.3, 204.3, 204.3, 204.2, 204. , 203.8, 203.6, 203.4,
       203.1, 202.7, 202.3, 202. , 201.9, 202. , 202.2, 202.4, 202.6,
       199.4, 199.6, 199.9, 200.4, 200.9, 202.1, 203.1, 203.9, 204.5,
       204.9, 205.4, 205.4, 205.3, 205.3, 205.4, 205.6, 205.9, 206. ,
       205.9, 205.8, 205.5, 205.3, 204.8, 204.3, 203.9, 203.5, 203.3,
       203.3, 203.2, 203.2, 203.3, 203.5, 203.8, 204. , 204.1, 204.3,
       204.3, 204.3, 204.4, 204.4, 204.5, 204.2, 204.1, 203.8, 203.6,
       203.3, 202.9, 202.6, 202.4, 202.1, 202.2, 202.4, 202.7, 202.9,
       200. , 200.1, 200.7, 201.4, 202.3, 203.3, 204.3, 205. , 205.3,
       205.5, 205.6, 205.7, 205.5, 205.4, 205.7, 206.1, 206.3, 206.5,
       206.3, 206. , 205.8, 205.4, 204.9, 204.5, 204.1, 203.8, 203.6,
       203.5, 203.5, 203.4, 203.5, 203.6, 203.7, 203.9, 204.1, 204.2,
       204.3, 204.3, 204.4, 204.5, 204.6, 204.3, 204.1, 203.9, 203.6,
       203.5, 203.2, 202.9, 202.6, 202.3, 202.4, 202.6, 202.7, 203.1,
       200.8, 201. , 201.8, 202.7, 204. , 205.2, 205.8, 206.2, 206.3,
       206.3, 206.2, 206.3, 206.2, 206.1, 206.4, 206.7, 206.9, 206.8,
       206.7, 206.3, 205.9, 205.5, 204.9, 204.5, 204.2, 203.9, 203.5,
       203.4, 203.3, 203.4, 203.6, 203.6, 203.8, 203.9, 204. , 204.1,
       204.2, 204.3, 204.4, 204.4, 204.4, 204.2, 203.9, 203.8, 203.7,
       203.7, 203.3, 202.9, 202.6, 202.3, 202.2, 202.6, 203. , 203.4,
       202.2, 202.6, 203.3, 204.5, 205.6, 206.7, 207.3, 207.4, 207.4,
       207.4, 207.4, 207.3, 207.3, 207.2, 207.4, 207.4, 207.6, 207.4,
       207.1, 206.7, 206.1, 205.6, 204.7, 204.2, 203.9, 203.6, 203.5,
       203.4, 203.6, 203.6, 203.6, 203.6, 203.7, 203.8, 203.9, 204. ,
       204.1, 204.1, 204.2, 204.1, 204.1, 204. , 203.9, 203.8, 203.6,
       203.5, 203.2, 202.9, 202.7, 202.6, 202.7, 203.1, 203.4, 203.7,
       203.9, 204.4, 205. , 205.9, 207. , 207.9, 208.7, 208.6, 208.4,
       208.6, 208.7, 208.6, 208.6, 208.6, 208.7, 208.7, 208.4, 208.3,
       207.6, 207.1, 206.3, 205.5, 204.8, 204.3, 204.1, 204. , 203.9,
       203.7, 203.6, 203.7, 203.6, 203.6, 203.6, 203.8, 203.9, 203.9,
       203.9, 204. , 204. , 204. , 203.9, 203.9, 203.7, 203.6, 203.5,
       203.5, 203.4, 203.2, 203.2, 203.1, 203.2, 203.4, 203.7, 203.9,
       205.8, 206.2, 206.5, 207.1, 208. , 208.6, 209.1, 209.3, 209.3,
       209.6, 210.1, 210.1, 210.1, 209.9, 210.2, 210.2, 209.6, 208.9,
       208.3, 207.6, 206.9, 206.1, 205.3, 204.6, 204.5, 204.3, 204. ,
       203.8, 203.8, 203.8, 203.8, 203.8, 203.7, 203.8, 203.9, 203.9,
       204. , 204. , 204. , 204. , 203.9, 203.8, 203.7, 203.6, 203.6,
       203.5, 203.4, 203.2, 203.3, 203.4, 203.5, 203.7, 203.9, 204. ,
       207.4, 207.7, 207.8, 207.9, 208.5, 209.2, 209.5, 209.7, 210.2,
       210.6, 210.9, 211. , 211.1, 211.2, 211.3, 211.1, 210.5, 209.8,
       208.9, 208.1, 207.2, 206.4, 205.7, 205. , 204.7, 204.5, 204.3,
       204.2, 204. , 203.9, 203.9, 203.9, 203.9, 203.9, 204. , 204.1,
       204.1, 204.2, 204.1, 204. , 204. , 203.9, 203.7, 203.6, 203.5,
       203.4, 203.2, 203.1, 203. , 203.2, 203.5, 203.7, 203.9, 204.1,
       208.8, 208.9, 208.9, 208.9, 209.3, 209.8, 210.3, 210.6, 210.9,
       211.2, 211.5, 211.6, 211.7, 211.8, 211.8, 211.4, 210.5, 209.7,
       208.9, 208.1, 207.2, 206.4, 205.9, 205.4, 204.9, 204.6, 204.4,
       204.2, 204.1, 204. , 204. , 204. , 204.1, 204.2, 204.2, 204.2,
       204.3, 204.4, 204.4, 204.3, 204.1, 203.9, 203.7, 203.5, 203.4,
       203.1, 202.8, 202.7, 202.8, 203. , 203.2, 203.5, 203.7, 204. ,
       210.2, 210.1, 210.1, 210.3, 210.5, 210.8, 211.1, 211.3, 211.6,
       211.7, 211.9, 212. , 212. , 211.7, 211.4, 211.1, 210.4, 209.7,
       208.8, 207.8, 206.9, 206.1, 205.5, 205.2, 204.8, 204.5, 204.3,
       204.1, 204. , 204. , 204. , 204. , 204.1, 204.3, 204.3, 204.4,
       204.5, 204.6, 204.7, 204.6, 204.3, 204. , 203.7, 203.3, 203.1,
       202.9, 202.5, 202.2, 202.4, 202.7, 203. , 203.3, 203.5, 203.7,
       211.1, 211.3, 211.2, 211.3, 211.4, 211.6, 211.9, 212. , 212. ,
       212.1, 212.1, 212.2, 212.2, 212. , 211.5, 210.9, 210.2, 209.2,
       208.2, 207.1, 206.1, 205.5, 205. , 204.7, 204.4, 204.2, 204. ,
       203.8, 203.8, 203.8, 203.9, 204. , 204.1, 204.3, 204.5, 204.6,
       204.7, 204.8, 204.9, 204.8, 204.5, 204.1, 203.6, 203.2, 203. ,
       202.7, 202.4, 202.1, 202.2, 202.5, 202.7, 203. , 203.3, 203.6,
       211.8, 212.1, 212. , 212.1, 212.3, 212.4, 212.4, 212.4, 212.2,
       212.2, 212.4, 212.5, 212.3, 211.7, 211. , 210.3, 209.4, 208.2,
       207. , 206.2, 205.4, 204.8, 204.3, 203.9, 203.7, 203.5, 203.5,
       203.4, 203.4, 203.5, 203.6, 203.8, 204.1, 204.3, 204.7, 204.8,
       204.9, 205. , 205.1, 205. , 204.6, 204.2, 203.7, 203.3, 203. ,
       202.6, 202.4, 202.1, 202. , 202.3, 202.7, 203. , 203.3, 203.6,
       212.1, 212.5, 212.6, 212.6, 212.7, 212.9, 212.7, 212.5, 212.5,
       212.6, 212.5, 212.3, 212.1, 211.5, 210.8, 209.5, 208.2, 207. ,
       206.1, 205.2, 204.4, 203.7, 203.6, 203.4, 203.2, 202.9, 202.9,
       203. , 203.1, 203.2, 203.3, 203.5, 203.9, 204.3, 204.7, 204.8,
       204.8, 204.9, 205.1, 205.2, 204.7, 204.2, 203.8, 203.5, 203.2,
       203. , 202.6, 202.3, 202. , 202.3, 202.7, 203. , 203.4, 203.7,
       212.3, 212.6, 212.6, 212.6, 212.7, 213. , 213. , 212.9, 213.1,
       213.6, 213.6, 212.9, 212.4, 211.8, 211. , 209.3, 207.4, 205.9,
       205.4, 204.9, 204.2, 203.4, 203.2, 203.1, 203. , 202.7, 202.6,
       202.7, 202.8, 202.9, 202.9, 203.2, 203.6, 204. , 204.4, 204.6,
       204.7, 204.7, 204.8, 205. , 204.7, 204.3, 204.1, 203.8, 203.5,
       203.1, 202.8, 202.4, 202.2, 202.4, 202.8, 203.2, 203.5, 203.9,
       212.3, 212.2, 212.2, 212.5, 212.8, 213.2, 213.3, 213.8, 214.1,
       214.8, 214.5, 214. , 213.5, 212.9, 211.2, 209.5, 208. , 206.5,
       205.8, 205.3, 204.7, 204. , 203.7, 203.4, 203. , 202.7, 202.4,
       202.5, 202.5, 202.6, 202.6, 202.9, 203.4, 203.8, 204. , 204.1,
       204.2, 204.4, 204.6, 204.7, 204.5, 204.1, 203.8, 203.7, 203.6,
       203.3, 203. , 202.6, 202.4, 202.6, 203. , 203.4, 203.7, 204.1,
       212.4, 212.4, 212.5, 212.7, 213. , 213.4, 213.7, 214.2, 214.7,
       215.1, 215.1, 214.3, 213.9, 213.4, 212.1, 210.3, 209. , 208. ,
       207.3, 206.4, 205.5, 204.6, 203.9, 203.6, 203.2, 202.7, 202.3,
       202.5, 202.7, 202.7, 202.7, 202.8, 203.1, 203.3, 203.6, 203.7,
       203.8, 204. , 204. , 204.1, 204. , 203.9, 203.7, 203.4, 203.4,
       203.1, 202.7, 202.7, 202.7, 202.8, 203.2, 203.6, 203.9, 204.2,
       212.8, 212.8, 212.7, 212.9, 213.2, 213.4, 213.4, 213.7, 214.2,
       214.7, 215.2, 214.7, 214.2, 213.7, 212.9, 211.3, 209.8, 208.7,
       207.8, 206.8, 205.9, 205. , 204.2, 203.8, 203.4, 203.2, 202.9,
       202.9, 202.8, 202.7, 202.7, 202.8, 203.1, 203.3, 203.3, 203.3,
       203.4, 203.6, 203.7, 203.7, 203.4, 203.2, 203.1, 203.1, 203. ,
       203. , 202.9, 202.9, 202.8, 203. , 203.4, 203.8, 204. , 204.2,
       213.2, 213.3, 213. , 212.8, 213. , 213.3, 213.3, 213.5, 214. ,
       214.5, 215.1, 214.7, 214.1, 213.3, 212.4, 211.1, 210. , 208.8,
       208. , 207. , 206.1, 205.4, 204.7, 204.1, 203.6, 203.1, 202.9,
       202.7, 202.7, 202.8, 202.9, 202.9, 203. , 203.1, 203.2, 203.3,
       203.3, 203.5, 203.2, 203. , 202.9, 202.8, 202.8, 202.8, 202.9,
       202.9, 202.8, 202.9, 203.1, 203.3, 203.6, 203.8, 204. , 204.3,
       213.9, 213.8, 213.7, 213.7, 213.9, 213.7, 213.6, 213.3, 213.8,
       214.2, 214.5, 214.4, 213.6, 212.8, 211.7, 210.5, 209.5, 208.7,
       207.8, 206.9, 205.9, 205.1, 204.2, 203.5, 203.2, 203. , 202.7,
       202.6, 202.7, 202.7, 202.8, 202.9, 203. , 203.1, 203.2, 203.2,
       203.3, 203.2, 203.1, 203. , 202.8, 202.6, 202.6, 202.7, 202.7,
       202.8, 202.9, 203.1, 203.3, 203.6, 204. , 204.3, 204.5, 204.4,
       214.9, 214.7, 214.5, 214.5, 214.5, 214.2, 213.8, 213.4, 213.6,
       214. , 214.1, 214.1, 213. , 212.1, 211.1, 210.2, 209.3, 208.3,
       207.3, 206.3, 205.4, 204.5, 203.7, 203.1, 202.9, 202.7, 202.6,
       202.6, 202.6, 202.7, 202.8, 202.9, 202.9, 203. , 203. , 203.1,
       203.2, 203.2, 203. , 202.8, 202.7, 202.6, 202.7, 202.8, 202.9,
       203. , 203.1, 203.3, 203.6, 204.1, 204.5, 204.6, 204.6, 204.5,
       215.1, 215.2, 215.1, 215. , 215.1, 214.6, 214.1, 213.5, 213.4,
       213.6, 213.7, 213.8, 213. , 212. , 210.9, 209.9, 208.8, 207.7,
       206.6, 205.6, 204.7, 204. , 203.4, 202.8, 202.7, 202.6, 202.5,
       202.5, 202.6, 202.7, 202.7, 202.7, 202.8, 202.8, 203. , 203.1,
       203.2, 203.3, 203.1, 203. , 202.9, 202.8, 202.9, 203. , 203.1,
       203.1, 203.3, 203.7, 204.1, 204.4, 204.9, 204.8, 204.8, 204.8,
       215.2, 215.3, 215.3, 215.3, 215.3, 214.8, 214.2, 213.6, 213.2,
       213.3, 213.5, 213.5, 212.7, 211.6, 210.4, 209.2, 208.1, 207. ,
       206. , 204.9, 204.2, 203.5, 202.9, 202.6, 202.4, 202.4, 202.4,
       202.5, 202.5, 202.6, 202.6, 202.7, 202.7, 202.9, 203. , 203.1,
       203.3, 203.4, 203.3, 203.2, 203.2, 203.1, 203.1, 203.1, 203.2,
       203.3, 203.4, 203.8, 204.2, 204.6, 205.1, 205.2, 205.1, 205. ,
       215.2, 215.3, 215.2, 215.2, 215.2, 214.9, 214.3, 213.7, 213.4,
       213.5, 213.2, 212.8, 212.3, 211. , 209.8, 208.5, 207.4, 206.2,
       205.2, 204.3, 203.5, 203.1, 202.8, 202.4, 202.2, 202.2, 202.1,
       202.2, 202.4, 202.5, 202.6, 202.7, 202.8, 202.9, 203.1, 203.2,
       203.4, 203.5, 203.4, 203.3, 203.2, 203.2, 203.1, 203.2, 203.3,
       203.4, 203.6, 203.9, 204.3, 204.7, 205. , 205. , 204.9, 204.8,
       215.1, 215.1, 215.1, 215. , 215. , 214.9, 214.4, 214. , 213.6,
       213.1, 212.7, 212.2, 211.5, 210.3, 209. , 207.7, 206.5, 205.5,
       204.6, 203.6, 203. , 202.6, 202.3, 202. , 201.8, 201.8, 201.9,
       202.1, 202.4, 202.5, 202.8, 203. , 203.2, 203.2, 203.5, 203.7,
       203.6, 203.5, 203.4, 203.3, 203.3, 203.2, 203.2, 203.3, 203.4,
       203.5, 203.5, 203.8, 204.1, 204.3, 204.5, 204.7, 204.8, 204.8,
       214.9, 214.9, 214.8, 214.8, 214.7, 214.8, 214.4, 213.9, 213.3,
       212.7, 212.1, 211.3, 210.6, 209.4, 208.1, 206.8, 205.8, 204.8,
       203.9, 203.2, 202.5, 202.2, 202. , 201.7, 201.5, 201.6, 201.8,
       202. , 202.8, 203.4, 203.6, 203.7, 203.7, 203.8, 203.8, 203.8,
       203.8, 203.8, 203.7, 203.7, 203.6, 203.5, 203.3, 203.4, 203.5,
       203.6, 203.7, 203.8, 204.2, 204.5, 204.7, 204.9, 204.9, 204.9,
       214.6, 214.6, 214.5, 214.5, 214.5, 214.6, 214.3, 213.6, 212.9,
       212. , 211.3, 210.5, 209.5, 208.4, 207.1, 206.1, 205.1, 204.2,
       203.4, 202.7, 202.1, 201.8, 201.5, 201.4, 201.4, 201.5, 202. ,
       202.7, 203.5, 204.2, 204.2, 204.2, 204.3, 204.6, 204.7, 204.7,
       204.5, 204.4, 204.3, 204.2, 204.1, 204. , 203.9, 204. , 204.1,
       204.1, 204.1, 204.2, 204.4, 204.6, 204.9, 205. , 205. , 205.1,
       214.3, 214.2, 214.2, 214.3, 214.3, 214.2, 214. , 213.3, 212.3,
       211.3, 210.4, 209.4, 208.3, 207.3, 206.2, 205.1, 204.2, 203.4,
       202.6, 202.2, 201.7, 201.4, 201.4, 201.3, 201.2, 201.4, 202. ,
       202.7, 203.5, 204.4, 204.7, 205. , 205.2, 205.4, 205.6, 205.5,
       205.4, 205.3, 205.1, 205. , 204.9, 204.8, 204.7, 204.7, 204.7,
       204.8, 204.8, 204.6, 204.8, 205.1, 205.1, 205.1, 205.1, 205.2,
       213.8, 213.9, 213.9, 214. , 213.9, 213.8, 213.4, 212.3, 211.1,
       210.1, 209.1, 208. , 207.1, 206.1, 205.1, 204.2, 203.4, 202.7,
       202.2, 201.7, 201.2, 200.7, 200.6, 200.8, 200.9, 201.1, 202. ,
       202.8, 203.7, 204.7, 205.3, 205.5, 205.9, 206.3, 206.7, 206.5,
       206.3, 206. , 205.9, 205.8, 205.7, 205.6, 205.6, 205.6, 205.6,
       205.5, 205.3, 205.1, 205.2, 205.2, 205.2, 205.2, 205.3, 205.4,
       213.6, 213.7, 213.7, 213.5, 213.4, 212.7, 211.9, 210.8, 209.8,
       208.7, 207.8, 206.9, 206. , 205.5, 204.9, 204. , 203.1, 202.2,
       201.6, 201.1, 200.6, 200.2, 200.3, 200.4, 200.9, 201.4, 202.2,
       203.1, 204.1, 205.1, 205.9, 206.3, 206.6, 207. , 207.2, 207.1,
       206.8, 206.7, 206.7, 206.6, 206.5, 206.5, 206.5, 206.5, 206.3,
       206. , 205.8, 205.5, 205.5, 205.6, 205.6, 205.5, 205.4, 205.6,
       213.6, 213.4, 213.2, 212.9, 212.2, 211.4, 210.5, 209.4, 208.4,
       207.6, 206.9, 206.2, 205.8, 205.3, 204.6, 203.6, 202.7, 201.9,
       201.2, 200.8, 200.4, 200. , 200.1, 200.6, 201.1, 201.7, 202.6,
       203.6, 204.6, 205.5, 206.5, 206.8, 207.2, 207.4, 207.7, 207.7,
       207.6, 207.6, 207.4, 207.3, 207.1, 207. , 206.9, 206.8, 206.5,
       206.2, 206. , 206. , 206. , 206. , 205.9, 205.7, 205.6, 205.7,
       213.5, 213.3, 212.8, 212.2, 211.3, 210.2, 209.2, 208.3, 207.6,
       206.8, 206.2, 205.7, 205.2, 204.8, 204.3, 203.3, 202.4, 201.6,
       201. , 200.6, 200.2, 200. , 200.2, 200.6, 201.3, 202.1, 203. ,
       204. , 205. , 206. , 207. , 207.4, 207.7, 208. , 208.2, 208.1,
       208. , 207.8, 207.6, 207.4, 207.3, 207.1, 207. , 206.9, 206.7,
       206.6, 206.5, 206.4, 206.4, 206.3, 206.1, 206. , 205.8, 205.8,
       213.5, 213. , 212.4, 211.7, 210.8, 209.8, 208.7, 207.6, 206.9,
       206.3, 205.7, 205.2, 204.8, 204.4, 204. , 203.1, 202.2, 201.5,
       200.9, 200.5, 200.3, 200.2, 200.2, 201. , 201.8, 202.7, 203.7,
       204.7, 205.8, 206.8, 207.6, 208.1, 208.4, 208.5, 208.5, 208.4,
       208.2, 208. , 207.8, 207.6, 207.4, 207.3, 207.2, 207.1, 207. ,
       206.9, 206.8, 206.7, 206.7, 206.5, 206.4, 206.2, 205.9, 205.8,
       213.5, 213. , 212.1, 211.3, 210.4, 209.3, 208.2, 207.2, 206.5,
       205.9, 205.4, 205. , 204.6, 204.1, 203.7, 202.9, 202.1, 201.4,
       200.8, 200.7, 200.6, 200.5, 200.7, 201.5, 202.4, 203.4, 204.6,
       205.7, 206.7, 207.5, 208.2, 208.8, 208.9, 208.8, 208.7, 208.7,
       208.4, 208.2, 208. , 207.8, 207.5, 207.4, 207.3, 207.3, 207.2,
       207.1, 207. , 207. , 207. , 206.8, 206.6, 206.3, 206. , 205.8,
       213.1, 212.6, 211.7, 210.9, 210.1, 209. , 208. , 207. , 206.2,
       205.7, 205.2, 204.7, 204.3, 203.9, 203.6, 203. , 202.2, 201.6,
       201.2, 201. , 200.9, 201.1, 201.4, 202.1, 203.2, 204.3, 205.5,
       206.5, 207.3, 208.1, 208.8, 209.3, 209.3, 209.2, 209.2, 208.9,
       208.7, 208.5, 208.2, 207.9, 207.7, 207.6, 207.5, 207.4, 207.4,
       207.3, 207.3, 207.3, 207.2, 207.1, 206.8, 206.4, 206.1, 205.9,
       212.6, 212.1, 211.3, 210.5, 209.7, 208.8, 207.8, 206.8, 206.2,
       205.8, 205.4, 205. , 204.6, 204.2, 203.7, 203.2, 202.4, 202. ,
       201.6, 201.3, 201.6, 201.9, 202.2, 202.8, 204. , 205.1, 206.2,
       207.2, 208. , 208.7, 209.3, 209.7, 209.7, 209.7, 209.4, 209.2,
       209. , 208.7, 208.4, 208.1, 207.8, 207.7, 207.7, 207.6, 207.6,
       207.5, 207.5, 207.5, 207.5, 207.3, 206.9, 206.6, 206.3, 206.2,
       211.9, 211.5, 210.8, 210. , 209.4, 208.7, 207.7, 207. , 206.5,
       206. , 205.6, 205.3, 204.9, 204.3, 203.8, 203.3, 202.9, 202.5,
       202.2, 202. , 202.2, 202.5, 202.9, 203.5, 204.6, 205.8, 206.8,
       207.9, 208.6, 209.2, 209.7, 210.1, 210.2, 210. , 209.7, 209.5,
       209.2, 208.9, 208.5, 208.2, 208. , 207.9, 207.9, 207.8, 207.8,
       207.7, 207.7, 207.7, 207.6, 207.5, 207.1, 206.8, 206.6, 206.5,
       210.9, 210.8, 210.2, 209.6, 209.1, 208.6, 207.8, 207.3, 206.8,
       206.3, 206. , 205.6, 205.4, 204.9, 204.4, 204.1, 203.7, 203.3,
       203. , 202.7, 202.9, 203.3, 203.7, 204.2, 205.3, 206.4, 207.4,
       208.5, 209.1, 209.5, 210. , 210.4, 210.3, 210. , 209.8, 209.5,
       209.3, 208.9, 208.5, 208.2, 208.2, 208.1, 208.1, 208. , 208. ,
       207.9, 207.9, 207.9, 207.8, 207.7, 207.3, 207. , 206.9, 206.8,
       210.1, 210. , 209.8, 209.3, 208.8, 208.4, 207.9, 207.3, 207. ,
       206.8, 206.4, 206.2, 206.2, 205.7, 205.3, 204.9, 204.6, 204.1,
       203.8, 203.5, 203.7, 204.1, 204.6, 205. , 205.9, 207. , 208. ,
       208.9, 209.4, 209.8, 210.3, 210.4, 210.3, 210.1, 209.8, 209.5,
       209.3, 208.9, 208.6, 208.4, 208.4, 208.4, 208.3, 208.2, 208.2,
       208.1, 208.1, 208. , 207.9, 207.8, 207.5, 207.4, 207.3, 207.2,
       209.2, 209.1, 209.1, 208.6, 208.1, 208.1, 207.7, 207.3, 207.3,
       207.3, 207. , 206.9, 206.9, 206.5, 206.2, 205.8, 205.4, 204.9,
       204.5, 204. , 204.3, 204.7, 205.1, 205.6, 206.4, 207.5, 208.6,
       209.2, 209.6, 210.1, 210.5, 210.4, 210.3, 210.1, 209.8, 209.5,
       209.3, 209. , 208.7, 208.7, 208.6, 208.6, 208.5, 208.5, 208.4,
       208.3, 208.3, 208.2, 208.1, 208. , 207.8, 207.7, 207.6, 207.5,
       208.2, 208.2, 208.3, 207.9, 207.8, 207.8, 207.6, 207.6, 207.7,
       207.7, 207.5, 207.3, 207.1, 206.9, 206.4, 205.9, 205.7, 205.3,
       204.8, 204.7, 204.9, 205.4, 205.8, 206.3, 207. , 208.1, 208.9,
       209.4, 209.9, 210.4, 210.5, 210.4, 210.3, 210.1, 209.8, 209.6,
       209.3, 209. , 208.9, 208.9, 208.9, 208.8, 208.8, 208.7, 208.6,
       208.6, 208.4, 208.3, 208.2, 208.1, 208. , 207.9, 207.7, 207.6,
       207.2, 207.3, 207.4, 207.4, 207.5, 207.5, 207.5, 207.5, 207.4,
       207.5, 207.4, 207.2, 207.1, 207.1, 206.6, 206.1, 206.1, 205.7,
       205.2, 205.4, 205.6, 206. , 206.5, 207. , 207.6, 208.6, 209.2,
       209.7, 210.2, 210.5, 210.4, 210.4, 210.3, 210.1, 209.8, 209.5,
       209.2, 209.1, 209.1, 209. , 208.9, 209.2, 209.1, 209.1, 209. ,
       208.8, 208.6, 208.5, 208.4, 208.3, 208.1, 208. , 207.9, 208. ,
       206.6, 206.9, 207.1, 207.1, 207.1, 207.1, 207.2, 207.1, 207.1,
       207.3, 207.3, 207.1, 207.1, 207.3, 206.7, 206.5, 206.4, 206.2,
       205.9, 206.1, 206.3, 206.7, 207.2, 207.7, 208.2, 208.9, 209.4,
       210. , 210.5, 210.5, 210.4, 210.3, 210.3, 209.8, 209.5, 209.2,
       208.9, 209.1, 209.1, 209. , 209. , 209.5, 209.5, 209.5, 209.4,
       209. , 208.8, 208.7, 208.6, 208.4, 208.2, 208.1, 208. , 208.3,
       206.3, 206.6, 206.7, 206.7, 206.7, 206.7, 206.8, 206.8, 206.8,
       207. , 207.1, 207. , 207.1, 207.3, 206.9, 206.8, 206.7, 206.6,
       206.7, 206.9, 207. , 207.4, 207.9, 208.4, 208.9, 209.3, 209.8,
       210.4, 210.7, 210.4, 210.4, 210.3, 210. , 209.5, 209.2, 208.9,
       208.9, 209.1, 209.1, 209. , 209.2, 209.8, 209.9, 209.8, 209.6,
       209.2, 209. , 208.9, 208.7, 208.5, 208.4, 208.2, 208.3, 208.6,
       206. , 206.3, 206.3, 206.3, 206.3, 206.4, 206.4, 206.5, 206.6,
       206.8, 207. , 207. , 207.2, 207.3, 207.3, 207.2, 207.1, 207.1,
       207.3, 207.4, 207.5, 208.1, 208.6, 209.1, 209.4, 209.7, 210.3,
       210.8, 210.7, 210.4, 210.3, 210.3, 209.8, 209.2, 208.9, 208.6,
       208.8, 209.1, 209.1, 209.1, 209.5, 210.1, 210.3, 210.2, 209.9,
       209.5, 209.1, 209. , 208.9, 208.7, 208.5, 208.4, 208.7, 208.9,
       205.8, 206. , 205.9, 205.9, 205.9, 206. , 206.1, 206.2, 206.4,
       206.6, 206.8, 207.1, 207.3, 207.4, 207.7, 207.6, 207.5, 207.6,
       207.8, 207.9, 208.1, 208.7, 209.2, 209.7, 209.9, 210.1, 210.7,
       211.1, 210.7, 210.4, 210.3, 210.1, 209.5, 209. , 208.6, 208.5,
       208.8, 209.1, 209.1, 209.3, 209.8, 210.4, 210.7, 210.5, 210.1,
       209.7, 209.3, 209.2, 209. , 208.8, 208.6, 208.7, 209. , 209.3,
       205.5, 205.6, 205.6, 205.6, 205.6, 205.8, 205.9, 206. , 206.3,
       206.5, 206.8, 207.3, 207.4, 207.6, 208.2, 208. , 207.9, 208.1,
       208.3, 208.3, 208.7, 209.4, 209.9, 210.3, 210.4, 210.6, 211.1,
       211.1, 210.8, 210.4, 210.3, 209.9, 209.3, 208.7, 208.3, 208.5,
       208.8, 209. , 209.2, 209.6, 210.1, 210.7, 211.2, 210.8, 210.4,
       210. , 209.6, 209.3, 209.2, 209. , 208.8, 208.9, 209.2, 209.5,
       205.1, 205.4, 205.4, 205.5, 205.5, 205.6, 205.7, 205.9, 206.1,
       206.4, 206.9, 207.4, 207.6, 207.9, 208.4, 208.4, 208.4, 208.6,
       208.7, 208.8, 209.4, 210. , 210.5, 210.8, 211. , 211.1, 211.5,
       211.2, 210.8, 210.4, 210.2, 209.6, 209. , 208.5, 208.2, 208.5,
       208.7, 209. , 209.3, 209.8, 210.3, 210.8, 211.4, 211.1, 210.7,
       210.2, 209.9, 209.6, 209.4, 209.2, 209.1, 209.1, 209.4, 209.6,
       204.7, 205.2, 205.3, 205.3, 205.3, 205.4, 205.5, 205.8, 206. ,
       206.2, 206.9, 207.5, 207.7, 208.1, 208.7, 208.8, 208.9, 209.1,
       209.2, 209.4, 210. , 210.6, 211.2, 211.4, 211.5, 211.6, 211.6,
       211.2, 210.8, 210.4, 209.9, 209.3, 208.7, 208.2, 208.1, 208.3,
       208.6, 208.8, 209.4, 209.9, 210.4, 210.9, 211.5, 211.4, 211. ,
       210.5, 210.1, 209.9, 209.7, 209.5, 209.3, 209.3, 209.5, 209.8,
       204.3, 204.9, 205.1, 205.1, 205.1, 205.2, 205.3, 205.6, 205.9,
       206.2, 206.9, 207.6, 207.9, 208.4, 208.9, 209.3, 209.4, 209.5,
       209.7, 210.1, 210.7, 211.2, 211.6, 211.7, 211.7, 211.8, 211.5,
       211.1, 210.7, 210.2, 209.6, 209. , 208.4, 207.9, 207.9, 208.1,
       208.4, 208.8, 209.4, 209.9, 210.4, 211. , 211.7, 211.7, 211.3,
       210.8, 210.4, 210.1, 209.9, 209.7, 209.5, 209.4, 209.7, 209.9,
       204. , 204.5, 204.9, 205. , 205. , 205. , 205.2, 205.5, 205.8,
       206.2, 206.9, 207.6, 208.1, 208.7, 209.2, 209.7, 209.8, 210. ,
       210.1, 210.2, 210.7, 211.2, 211.6, 211.7, 211.7, 211.7, 211.3,
       210.9, 210.5, 209.9, 209.2, 208.6, 208. , 207.8, 207.8, 208. ,
       208.2, 208.8, 209.4, 210. , 210.5, 211.1, 211.8, 212. , 211.6,
       211.1, 210.7, 210.4, 210.2, 210. , 209.8, 209.6, 209.9, 210.1])

    lon = numpy.array([-75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745,
       -75.59722   , -75.5916658 , -75.5861116 , -75.5805574 ,
       -75.5750032 , -75.569449  , -75.56389481, -75.55834061,
       -75.55278641, -75.54723221, -75.54167801, -75.53612381,
       -75.53056961, -75.52501541, -75.51946121, -75.51390701,
       -75.50835282, -75.50279862, -75.49724442, -75.49169022,
       -75.48613602, -75.48058182, -75.47502762, -75.46947342,
       -75.46391922, -75.45836502, -75.45281083, -75.44725663,
       -75.44170243, -75.43614823, -75.43059403, -75.42503983,
       -75.41948563, -75.41393143, -75.40837723, -75.40282303,
       -75.39726884, -75.39171464, -75.38616044, -75.38060624,
       -75.37505204, -75.36949784, -75.36394364, -75.35838944,
       -75.35283524, -75.34728104, -75.34172685, -75.33617265,
       -75.33061845, -75.32506425, -75.31951005, -75.31395585,
       -75.30840165, -75.30284745, -75.59722   , -75.5916658 ,
       -75.5861116 , -75.5805574 , -75.5750032 , -75.569449  ,
       -75.56389481, -75.55834061, -75.55278641, -75.54723221,
       -75.54167801, -75.53612381, -75.53056961, -75.52501541,
       -75.51946121, -75.51390701, -75.50835282, -75.50279862,
       -75.49724442, -75.49169022, -75.48613602, -75.48058182,
       -75.47502762, -75.46947342, -75.46391922, -75.45836502,
       -75.45281083, -75.44725663, -75.44170243, -75.43614823,
       -75.43059403, -75.42503983, -75.41948563, -75.41393143,
       -75.40837723, -75.40282303, -75.39726884, -75.39171464,
       -75.38616044, -75.38060624, -75.37505204, -75.36949784,
       -75.36394364, -75.35838944, -75.35283524, -75.34728104,
       -75.34172685, -75.33617265, -75.33061845, -75.32506425,
       -75.31951005, -75.31395585, -75.30840165, -75.30284745])

    lat = numpy.array([37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.202778  ,
       37.202778  , 37.202778  , 37.202778  , 37.202778  , 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.20833601, 37.20833601,
       37.20833601, 37.20833601, 37.20833601, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21389403, 37.21389403, 37.21389403,
       37.21389403, 37.21389403, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.21945204, 37.21945204, 37.21945204, 37.21945204,
       37.21945204, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.22501006, 37.22501006, 37.22501006, 37.22501006, 37.22501006,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23056807,
       37.23056807, 37.23056807, 37.23056807, 37.23056807, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.23612608, 37.23612608,
       37.23612608, 37.23612608, 37.23612608, 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 , 37.2416841 ,
       37.2416841 , 37.2416841 , 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.24724211, 37.24724211, 37.24724211, 37.24724211,
       37.24724211, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25280013, 37.25280013, 37.25280013, 37.25280013, 37.25280013,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.25835814,
       37.25835814, 37.25835814, 37.25835814, 37.25835814, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26391615, 37.26391615,
       37.26391615, 37.26391615, 37.26391615, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.26947417, 37.26947417, 37.26947417,
       37.26947417, 37.26947417, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.27503218, 37.27503218, 37.27503218, 37.27503218,
       37.27503218, 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 , 37.2805902 ,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.28614821,
       37.28614821, 37.28614821, 37.28614821, 37.28614821, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29170622, 37.29170622,
       37.29170622, 37.29170622, 37.29170622, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.29726424, 37.29726424, 37.29726424,
       37.29726424, 37.29726424, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30282225, 37.30282225, 37.30282225, 37.30282225,
       37.30282225, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.30838027, 37.30838027, 37.30838027, 37.30838027, 37.30838027,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31393828,
       37.31393828, 37.31393828, 37.31393828, 37.31393828, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.31949629, 37.31949629,
       37.31949629, 37.31949629, 37.31949629, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.32505431, 37.32505431, 37.32505431,
       37.32505431, 37.32505431, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33061232, 37.33061232, 37.33061232, 37.33061232,
       37.33061232, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.33617034, 37.33617034, 37.33617034, 37.33617034, 37.33617034,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34172835,
       37.34172835, 37.34172835, 37.34172835, 37.34172835, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.34728636, 37.34728636,
       37.34728636, 37.34728636, 37.34728636, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35284438, 37.35284438, 37.35284438,
       37.35284438, 37.35284438, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.35840239, 37.35840239, 37.35840239, 37.35840239,
       37.35840239, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36396041, 37.36396041, 37.36396041, 37.36396041, 37.36396041,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.36951842,
       37.36951842, 37.36951842, 37.36951842, 37.36951842, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.37507643, 37.37507643,
       37.37507643, 37.37507643, 37.37507643, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38063445, 37.38063445, 37.38063445,
       37.38063445, 37.38063445, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.38619246, 37.38619246, 37.38619246, 37.38619246,
       37.38619246, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39175048, 37.39175048, 37.39175048, 37.39175048, 37.39175048,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.39730849,
       37.39730849, 37.39730849, 37.39730849, 37.39730849, 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 , 37.4028665 ,
       37.4028665 , 37.4028665 , 37.4028665 , 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.40842452, 37.40842452, 37.40842452,
       37.40842452, 37.40842452, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41398253, 37.41398253, 37.41398253, 37.41398253,
       37.41398253, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.41954055, 37.41954055, 37.41954055, 37.41954055, 37.41954055,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.42509856,
       37.42509856, 37.42509856, 37.42509856, 37.42509856, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43065657, 37.43065657,
       37.43065657, 37.43065657, 37.43065657, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.43621459, 37.43621459, 37.43621459,
       37.43621459, 37.43621459, 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 , 37.4417726 ,
       37.4417726 , 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.44733062, 37.44733062, 37.44733062, 37.44733062, 37.44733062,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45288863,
       37.45288863, 37.45288863, 37.45288863, 37.45288863, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.45844664, 37.45844664,
       37.45844664, 37.45844664, 37.45844664, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46400466, 37.46400466, 37.46400466,
       37.46400466, 37.46400466, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.46956267, 37.46956267, 37.46956267, 37.46956267,
       37.46956267, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.47512069, 37.47512069, 37.47512069, 37.47512069, 37.47512069,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 ,
       37.4806787 , 37.4806787 , 37.4806787 , 37.4806787 , 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.48623671, 37.48623671,
       37.48623671, 37.48623671, 37.48623671, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49179473, 37.49179473, 37.49179473,
       37.49179473, 37.49179473, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274, 37.49735274, 37.49735274, 37.49735274, 37.49735274,
       37.49735274])

    grid_2d_properties = {
        'maxx': -75.30278,
        'minx': -75.59722,
        'miny': 37.202778,
        'maxy': 37.497223,
        'cellsize_x': 0.005554199,
        'cellsize_y': 0.005558014,
        'nx': 54,
        'ny': 54
    }

    grid_1d_properties = {
        'maxx': -75.30278,
        'minx': -75.59722,
        'miny': 37.202778,
        'maxy': 37.497223,
        'latitude': lat,
        'longitude': lon,
        'nodes': 2916
    }

    metadata = {
        'horizontalDatumReference': 'EPSG',
        'horizontalDatumValue': 4326,
        'metadata': f'MD_test_s111.XML',
        'epoch': 'G1762',
        'geographicIdentifier': 'Chesapeake_Bay',
        'speedUncertainty': -1.0,
        'directionUncertainty': -1.0,
        'verticalUncertainty': -1.0,
        'horizontalPositionUncertainty': -1.0,
        'timeUncertainty': -1.0,
        'surfaceCurrentDepth': -4.5,
        'depthTypeIndex': 2,
        'commonPointRule': 3,
        'interpolationType': 10,
        'typeOfCurrentData': 6,
        'methodCurrentsProduct': 'ROMS_Hydrodynamic_Model_Forecasts',
        'datetimeOfFirstRecord': '2021-01-07T13:00:00'

    }

    datetime_value = datetime.datetime(2021, 1, 7, 12, 0, 0)

    update_2d_meta = {
        'dateTimeOfLastRecord': '2021-01-07T14:00:00',
        'numberOfGroups': 2,
        'numberOfTimes': 2,
        'timeRecordInterval': 3600,
        'num_instances': 1
    }

    update_1d_meta = {
        'dateTimeOfLastRecord': '2021-01-07T13:00:00',
        'numberOfGroups': 1,
        'numberOfTimes': 1,
        'timeRecordInterval': 0,
        'num_instances': 1
    }

    expected_2d_chunks = '27,54'

    expected_1d_chunks = '1458'

    expected_groupf = numpy.array([
        ('surfaceCurrentSpeed', 'Surface current speed', 'knots', '-9999', 'H5T_FLOAT', '0', '', 'geSemiInterval'),
        ('surfaceCurrentDirection', 'Surface current direction', 'arc-degrees', '-9999', 'H5T_FLOAT', '0', '360', 'geLtInterval')],
      dtype=[('code', 'O'), ('name', 'O'), ('uom.name', 'O'), ('fillValue', 'O'), ('datatype', 'O'), ('lower', 'O'), ('upper', 'O'), ('closure', 'O')])

    expected_georeferenced_coordinates = (-75.30278, 0.005554199, 0.0, 37.202778, 0.0, 0.005558014)

    return InputData(speed_2d_001, direction_2d_001, speed_2d_002, direction_2d_002, speed_1d, direction_1d, lon, lat, grid_2d_properties, grid_1d_properties, metadata, datetime_value, update_2d_meta, update_1d_meta, expected_2d_chunks, expected_1d_chunks, expected_groupf, expected_georeferenced_coordinates)


def test_create_s111_dcf2(input_data):
    data_file = s111.utils.create_s111(f"{current_directory}/test_s111_dcf2.h5")

    s111.utils.add_metadata(input_data.metadata, data_file)
    s111.utils.add_data_from_arrays(input_data.speed_2d_001, input_data.direction_2d_001, data_file, input_data.grid_2d_properties, input_data.datetime_value, 2)
    s111.utils.add_data_from_arrays(input_data.speed_2d_002, input_data.direction_2d_002, data_file, input_data.grid_2d_properties, input_data.datetime_value, 2)
    s111.utils.update_metadata(data_file, input_data.grid_2d_properties, input_data.update_2d_meta)

    s111.utils.write_data_file(data_file)

    assert os.path.isfile(f"{current_directory}/test_s111_dcf2.h5")
    h5_file = h5py.File(f"{current_directory}/test_s111_dcf2.h5", "r")

    assert 'Group_F/SurfaceCurrent' in h5_file
    assert 'Group_F/featureCode' in h5_file
    assert 'SurfaceCurrent/SurfaceCurrent.01/uncertainty' in h5_file
    assert 'SurfaceCurrent/axisNames' in h5_file
    assert h5_file['Group_F/SurfaceCurrent'].attrs['chunking'] == input_data.expected_2d_chunks
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Group_001/values']['surfaceCurrentSpeed'], input_data.speed_2d_001)
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Group_001/values']['surfaceCurrentDirection'], input_data.direction_2d_001)
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Group_002/values']['surfaceCurrentSpeed'], input_data.speed_2d_002)
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Group_002/values']['surfaceCurrentDirection'], input_data.direction_2d_002)
    assert h5_file['SurfaceCurrent/SurfaceCurrent.01/'].attrs['numPointsLongitudinal'] == input_data.speed_2d_001.shape[0]
    assert h5_file['SurfaceCurrent/SurfaceCurrent.01/'].attrs['numPointsLatitudinal'] == input_data.speed_2d_001.shape[1]
    assert all([h5py_string_comp(actual, expected) for actual, expected in zip(h5_file['Group_F/SurfaceCurrent'][()][0],
                                                                               input_data.expected_groupf[0])])
    assert all([h5py_string_comp(actual, expected) for actual, expected in zip(h5_file['Group_F/SurfaceCurrent'][()][1],
                                                                               input_data.expected_groupf[1])])


def test_create_s111_dcf3(input_data):
    data_file = s111.utils.create_s111(f"{current_directory}/test_s111_dcf3.h5")

    s111.utils.add_metadata(input_data.metadata, data_file)
    s111.utils.add_data_from_arrays(input_data.speed_1d, input_data.direction_1d, data_file, input_data.grid_1d_properties, input_data.datetime_value, 3)
    s111.utils.update_metadata(data_file, input_data.grid_1d_properties, input_data.update_1d_meta)

    s111.utils.write_data_file(data_file)

    assert os.path.isfile(f"{current_directory}/test_s111_dcf3.h5")
    h5_file = h5py.File(f"{current_directory}/test_s111_dcf3.h5", "r")

    assert 'Group_F/SurfaceCurrent' in h5_file
    assert 'Group_F/featureCode' in h5_file
    assert 'SurfaceCurrent/SurfaceCurrent.01/uncertainty' in h5_file
    assert 'SurfaceCurrent/axisNames' in h5_file
    assert h5_file['Group_F/SurfaceCurrent'].attrs['chunking'] == input_data.expected_1d_chunks
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Group_001/values']['surfaceCurrentSpeed'], input_data.speed_1d)
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Group_001/values']['surfaceCurrentDirection'], input_data.direction_1d)
    assert h5_file['SurfaceCurrent/SurfaceCurrent.01/'].attrs['numberOfNodes'] == input_data.speed_1d.size
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Positioning/geometryValues']['longitude'], input_data.lon)
    assert numpy.allclose(h5_file['SurfaceCurrent/SurfaceCurrent.01/Positioning/geometryValues']['latitude'], input_data.lat)


def test_to_geotiff(input_data):

    s111.utils.to_geotiff(f"{current_directory}/test_s111_dcf2.h5", current_directory)

    assert os.path.isfile(f"{current_directory}/test_s111_dcf2_20210107T120000Z.tif")

    tif_file = gdal.Open(f"{current_directory}/test_s111_dcf2_20210107T120000Z.tif")
    h5_file = h5py.File(f"{current_directory}/test_s111_dcf2.h5", "r")
    feature_instance = h5_file['/SurfaceCurrent/SurfaceCurrent.01/']

    assert tif_file.GetGeoTransform() == input_data.expected_georeferenced_coordinates
    assert tif_file.RasterXSize == feature_instance.attrs['numPointsLongitudinal']
    assert tif_file.RasterYSize == feature_instance.attrs['numPointsLatitudinal']
    assert tif_file.GetGeoTransform() == (feature_instance.attrs['gridOriginLongitude'], feature_instance.attrs['gridSpacingLongitudinal'], 0, feature_instance.attrs['gridOriginLatitude'], 0, feature_instance.attrs['gridSpacingLatitudinal'])
    assert tif_file.GetRasterBand(1).ComputeRasterMinMax()[0] >= 0
    assert tif_file.GetRasterBand(1).ComputeRasterMinMax()[1] <= 100
    assert tif_file.GetRasterBand(2).ComputeRasterMinMax()[0] >= 0
    assert tif_file.GetRasterBand(2).ComputeRasterMinMax()[1] <= 360



