"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const YAML = require("yaml");
const aws_credentials_1 = require("./private/aws-credentials");
const github = require("./workflows-model");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = (_b = props.preSynthed) !== null && _b !== void 0 ? _b : false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = (_c = props.preBuildSteps) !== null && _c !== void 0 ? _c : [];
        this.postBuildSteps = (_d = props.postBuildSteps) !== null && _d !== void 0 ? _d : [];
        this.gitHubActionRoleArn = props.gitHubActionRoleArn;
        this.useGitHubActionRole = this.gitHubActionRoleArn ? true : false;
        this.awsCredentials = (_e = props.awsCredentials) !== null && _e !== void 0 ? _e : {
            accessKeyId: 'AWS_ACCESS_KEY_ID',
            secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
        };
        this.dockerCredentials = (_f = props.dockerCredentials) !== null && _f !== void 0 ? _f : [];
        this.workflowPath = (_g = props.workflowPath) !== null && _g !== void 0 ? _g : '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowName = (_h = props.workflowName) !== null && _h !== void 0 ? _h : 'deploy';
        this.workflowTriggers = (_j = props.workflowTriggers) !== null && _j !== void 0 ? _j : {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = (_k = props.runner) !== null && _k !== void 0 ? _k : github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = (_l = props.publishAssetsAuthRegion) !== null && _l !== void 0 ? _l : 'us-west-2';
    }
    doBuildPipeline() {
        var _b;
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        const yaml = YAML.stringify(workflow, {
            indent: 2,
        });
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const diffProtection = (_b = this.node.tryGetContext('cdk-pipelines-github:diffProtection')) !== null && _b !== void 0 ? _b : true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || yaml !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        // eslint-disable-next-line no-console
        console.log(`writing ${this.workflowPath}`);
        fs_1.writeFileSync(this.workflowPath, yaml);
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        var _b, _c;
        switch ((_b = node.data) === null || _b === void 0 ? void 0 : _b.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${(_c = node.data) === null || _c === void 0 ? void 0 : _c.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Worflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -x'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '::set-output name=${ASSET_HASH_NAME}::${assetId}'`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region: this.publishAssetsAuthRegion }),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            var _b;
            const hash = path.parse((_b = template.split('/').pop()) !== null && _b !== void 0 ? _b : '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(this.useGitHubActionRole, { region, assumeRoleArn }),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.useGitHubActionRole ? github.JobPermission.WRITE : github.JobPermission.NONE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        var _b;
        for (const dep of node.allDeps) {
            if (((_b = dep.data) === null || _b === void 0 ? void 0 : _b.type) === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v2',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    stepsToConfigureAws(openId, { region, assumeRoleArn }) {
        function getDeployRole(arn) {
            return arn.replace('cfn-exec', 'deploy');
        }
        let steps = [];
        if (openId) {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via OIDC Role', {
                region,
                gitHubActionRoleArn: this.gitHubActionRoleArn,
            }));
            if (assumeRoleArn) {
                // Result of initial credentials with GitHub Action role are these environment variables
                steps.push(aws_credentials_1.awsCredentialStep('Assume CDK Deploy Role', {
                    region,
                    accessKeyId: '${{ env.AWS_ACCESS_KEY_ID }}',
                    secretAccessKey: '${{ env.AWS_SECRET_ACCESS_KEY }}',
                    sessionToken: '${{ env.AWS_SESSION_TOKEN }}',
                    roleToAssume: getDeployRole(assumeRoleArn),
                }));
            }
        }
        else {
            steps.push(aws_credentials_1.awsCredentialStep('Authenticate Via GitHub Secrets', {
                region,
                accessKeyId: `\${{ secrets.${this.awsCredentials.accessKeyId} }}`,
                secretAccessKey: `\${{ secrets.${this.awsCredentials.secretAccessKey} }}`,
                sessionToken: `\${{ secrets.${this.awsCredentials.sessionToken} }}`,
                roleToAssume: assumeRoleArn,
            }));
        }
        return steps;
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v1',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v2',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [{
                name: 'Checkout',
                uses: 'actions/checkout@v2',
            }];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v2.1.1',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.2.50" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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