# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/01_data_processing.ipynb (unless otherwise specified).

__all__ = ['query_db', 'postouts', 'outs_per_inning', 'batters_faced', 'get_games_pitchers_years', 'preliminary_clean',
           'last', 'aggregate_at_bats', 'at_bat_aggs', 'create_indicator', 'accumulate', 'feature_engineering']

# Internal Cell
import pandas as pd
import sqlite3
import numpy as np

# Cell


def query_db(db_path: str = "../data/raw/statcast_pitches.db",
             year: str = "2019",
             columns: str = "*",
             limit=None,
             verbose=True):
    if verbose:
        print(f"querying db at {db_path} now.")
    conn = sqlite3.connect(db_path)
    query = f"""select {columns}
                from statcast_{year}"""
    if limit:
        query += f" limit {limit}"
    df = pd.read_sql_query(query, conn)
    conn.close()
    return df

# Cell

# utility functions for identifying openers


def postouts(df):
    """assumes sorted game pitcher df"""
    # put assert here to ensure that the df is sorted and is all of the same pitcher
    df["postouts"] = df["outs_when_up"].shift(-1).fillna(method="ffill")

    # if the inning changed, then the postouts is 3
    df.loc[(df["inning"] != df["inning"].shift(-1)), "postouts"] = 3
    return df


def outs_per_inning(x: pd.Series):
    """assumes df came straight out of postouts()"""
    # use should be: t.groupby(["inning"]).agg({"postouts": outs_per_inning})
    return (x - x.shift(1).fillna(0)).sum()


def batters_faced(at_bats: pd.Series):
    return len(at_bats.unique())

# Cell

def get_games_pitchers_years(df: pd.DataFrame, verbose: bool):
    """
    Filter out openers to get all game-pitcher combinations that qualify
    """
    # get unique game ids from regular season games w/ AL starting pitchers
    games = np.sort(df.loc[(df["game_type"]=="R"), "game_pk"].unique())
    if verbose:
        print(f"In this dataset, there are {len(games)} total games.")

    # This will be list of tuples for each game and pitcher to analyze
    games_pitchers_years = []

    # identifying "opener" candidates
    for game in games:
        # getting df of game data and saving year
        game_df = df.loc[(df["game_pk"]==game)]
        year = int(game_df["game_year"].iloc[0])

        # first pitcher for each team is throwing at min(at_bat_number)
        home_pitcher_first_ab = game_df.loc[(game_df["inning_topbot"]=="Bot"), "at_bat_number"].min()
        home_pitcher = game_df.loc[(game_df["at_bat_number"]==home_pitcher_first_ab), "pitcher"].head(1).item()

        away_pitcher_first_ab = game_df.loc[(game_df["inning_topbot"]=="Top", "at_bat_number")].min()
        away_pitcher = game_df.loc[(game_df["at_bat_number"]==away_pitcher_first_ab), "pitcher"].head(1).item()

        # check if either are "openers"
        for pitcher in (home_pitcher, away_pitcher):

            # getting sorted df for specific game and specific pitcher
            game_pitcher_df = game_df.loc[(game_df["pitcher"]==pitcher)].sort_values("at_bat_number", ascending=True)

            # adding postouts
            game_pitcher_df = postouts(game_pitcher_df)
            outs = game_pitcher_df.groupby(["inning"]).agg({"postouts": outs_per_inning}).sum().item()
            n_batters = batters_faced(game_pitcher_df["at_bat_number"])
            opener = outs < 7 or n_batters < 10
            if not opener:
                games_pitchers_years.append((game, pitcher, year))

    if verbose:
        print(f"There are {(len(games)*2) - len(games_pitchers_years)} 'openers' in the dataset.")
        print(f"There are {len(games_pitchers_years)} total eligible game-pitcher combinations in this dataset.")

    return games_pitchers_years

# Cell

def preliminary_clean(df: pd.DataFrame, g: int, p: int):
    """
    Before aggregating, perform a preliminary cleaning of dataset
    """
    temp = df.loc[(df["game_pk"]==g) & (df["pitcher"]==p)].sort_values("at_bat_number", ascending=True)

    # adding postouts as a column
    temp = postouts(temp)

    # filling missing events with empty string so can aggregate easily
    temp["events"] = temp["events"].fillna("")

    # post_bat_score is not actually score after at-bat, needs to be lagged
    temp["post_bat_score"] = temp["post_bat_score"].shift(-1).fillna(method="ffill")

    # post runners on (need to lag -> this info is known in between at-bats)
    temp["post_on_1b"] = temp["on_1b"].fillna(0).apply(lambda x: 1 if x>0 else 0).shift(-1).fillna(method="ffill")
    temp["post_on_2b"] = temp["on_2b"].fillna(0).apply(lambda x: 1 if x>0 else 0).shift(-1).fillna(method="ffill")
    temp["post_on_3b"] = temp["on_3b"].fillna(0).apply(lambda x: 1 if x>0 else 0).shift(-1).fillna(method="ffill")

    # if next batter opposite handed
    temp["post_opposite_hand"] = (temp["stand"]!=temp["p_throws"]).astype(int).shift(-1).fillna(method="ffill")

    return temp

# Cell

def last(x: pd.Series):
    """
    Used for getting last value in df.groupby.agg
    """
    return x.iloc[-1]

at_bat_aggs = {"balls": "max",
                "strikes": "max",
                "pitch_number": "max",
                "post_bat_score": last,
                "post_fld_score": last,
                "events": "max",
                "postouts": last,
                "post_on_1b": last,
                "post_on_2b": last,
                "post_on_3b": last,
                "game_type": last,
                "home_team": last,
                "away_team": last,
                "inning": last,
                "inning_topbot": last,
               "post_opposite_hand": last,
               "game_year": last}

def aggregate_at_bats(df: pd.DataFrame):
    """
    assumes the df has come straight out of preliminary clean
    """
    agged_df = df.groupby(by=["game_pk", "pitcher", "batter", "at_bat_number"]).agg(at_bat_aggs).sort_values(by="at_bat_number")
    return agged_df

# Cell

# helper feature engineering funcs

def create_indicator(df, col="events", indicators=[], indicator_col_names=[]):
    """

    :param col: string, column from which to create a new indicator column
    :param indicators: each time indicator occurs in col, assign a 1 to the indicator column
    :param indicator_col_names: name of new indicator columns
    :return:
    """
    if not indicator_col_names:
        indicator_col_names = indicators
    for indicator, indicator_col_name in zip(indicators, indicator_col_names):
        df[indicator_col_name] = 0
        df.loc[(df[col] == indicator), indicator_col_name] = 1
    return df

def accumulate(df, col, agg_func="cumsum"):
    if not agg_func.startswith("cum"):
        raise Warning("Are you sure you want to accumulate with a non-cumulative aggregation function?")

    df[f"{col}_{agg_func}"] = df[col].agg([agg_func])
    return df

# Cell

def feature_engineering(t: pd.DataFrame):
    """
    Assuming df came straight out of aggregate_at_bats
    """
    # strike-ball ratio
    t["cum_balls"] = t["balls"].cumsum()
    t["cum_strikes"] = t["strikes"].cumsum()
    t["cum_sb_ratio"] = t["cum_strikes"] / (t["cum_balls"] + 1)

    # end of inning
    t["end_inning"] = t["postouts"].apply(lambda x: 1 if (x==3) else 0)

    # times through order
    t["times_thru_order"] = [1/9*i for i in range(1, len(t)+1)]

    # score diff
    t["score_diff"] = t["post_fld_score"] - t["post_bat_score"]

    # post total runners
    t["post_total_runners"] = t[["post_on_1b", "post_on_2b", "post_on_3b"]].sum(axis=1)

    # tying run or leading run on base
    t["tying_run_on"] = ((t["score_diff"].isin((0,1))) & (t["post_total_runners"]>=1)).astype(int)

    # pitch total
    t["pitch_total"] = t["pitch_number"].cumsum()

    # getting pitcher's team
    if t["inning_topbot"].iloc[0] == "Bot":
        t["pitcher_team"] = t["away_team"]
    else:
        t["pitcher_team"] = t["home_team"]

    # collapsing grouped multi-index of pd dataframe
    t = t.reset_index()

    # subsetting dataset to only get AL starting pitchers
    AL_teams = ['MIN', 'CLE', 'DET', 'HOU',
                'BOS',  'TOR', 'LAA',  'BAL', 'KC',
                'NYY', 'CWS',  'TEX',  'TB','OAK',
                'SEA']
    t = t.loc[(t["pitcher_team"].isin(AL_teams))]

    # adding unique category for each team-year combo (for embeddings later)
    t["pitcher_team_year"] = t["pitcher_team"] + "_" + t["game_year"].astype(int).astype(str)

    # creating indicator cols for different events
    t = create_indicator(t, col="events", indicators=["strikeout", "walk", "single", "double", "triple", "home_run"])

    # cumulative stats
    cum_cols = [("strikeout", "cumsum"),
                ("walk", "cumsum"),
                ("single", "cumsum"),
                ("double", "cumsum"),
                ("triple", "cumsum"),
                ("home_run", "cumsum")]
    for col, agg_func in cum_cols:
        t = accumulate(t, col=col, agg_func=agg_func)

    # cumulative bases
    t["bases_cumsum"] = t["walk_cumsum"] + t["single_cumsum"] + (2 * t["double_cumsum"]) + \
                         (3 * t["triple_cumsum"]) + (4 * t["home_run_cumsum"])

    return t