"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const lambda_layer_awscli_1 = require("@aws-cdk/lambda-layer-awscli");
const case_1 = require("case");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * `BucketDeployment` populates an S3 bucket with the contents of .zip files from other S3 buckets or from local disk.
 *
 * @stability stable
 */
class BucketDeployment extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _d, _e;
        super(scope, id);
        if (props.distributionPaths) {
            if (!props.distribution) {
                throw new Error('Distribution must be specified if distribution paths are specified');
            }
            if (!cdk.Token.isUnresolved(props.distributionPaths)) {
                if (!props.distributionPaths.every(distributionPath => cdk.Token.isUnresolved(distributionPath) || distributionPath.startsWith('/'))) {
                    throw new Error('Distribution paths must start with "/"');
                }
            }
        }
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(path.join(__dirname, 'lambda')),
            layers: [new lambda_layer_awscli_1.AwsCliLayer(this, 'AwsCliLayer')],
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
            vpc: props.vpc,
            vpcSubnets: props.vpcSubnets,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_d = props.prune) !== null && _d !== void 0 ? _d : true,
                Exclude: props.exclude,
                Include: props.include,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: (_e = props.distribution) === null || _e === void 0 ? void 0 : _e.distributionId,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
_a = JSII_RTTI_SYMBOL_1;
BucketDeployment[_a] = { fqn: "@aws-cdk/aws-s3-deployment.BucketDeployment", version: "1.122.0" };
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase();
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    if (metadata.accessControl) {
        res.acl = case_1.kebab(metadata.accessControl.toString());
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
class CacheControl {
    constructor(
    /**
     * The raw cache control setting.
     */
    value) {
        this.value = value;
    }
    /**
     * Sets 'must-revalidate'.
     *
     * @stability stable
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * Sets 'no-cache'.
     *
     * @stability stable
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * Sets 'no-transform'.
     *
     * @stability stable
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * Sets 'public'.
     *
     * @stability stable
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * Sets 'private'.
     *
     * @stability stable
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * Sets 'proxy-revalidate'.
     *
     * @stability stable
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * Sets 'max-age=<duration-in-seconds>'.
     *
     * @stability stable
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * Sets 's-maxage=<duration-in-seconds>'.
     *
     * @stability stable
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * Constructs a custom cache control key from the literal value.
     *
     * @stability stable
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
_b = JSII_RTTI_SYMBOL_1;
CacheControl[_b] = { fqn: "@aws-cdk/aws-s3-deployment.CacheControl", version: "1.122.0" };
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @stability stable
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * (deprecated) Used for HTTP expires header, which influences downstream caches.
 *
 * Does NOT influence deletion of the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(
    /**
     * The raw expiration date expression.
     */
    value) {
        this.value = value;
    }
    /**
     * (deprecated) Expire at the specified date.
     *
     * @param d date to expire at.
     * @deprecated
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * (deprecated) Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @deprecated
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * (deprecated) Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @deprecated
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * (deprecated) Create an expiration date from a raw date string.
     *
     * @deprecated
     */
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
_c = JSII_RTTI_SYMBOL_1;
Expires[_c] = { fqn: "@aws-cdk/aws-s3-deployment.Expires", version: "1.122.0" };
//# sourceMappingURL=data:application/json;base64,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