# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['crim_intervals']

package_data = \
{'': ['*']}

install_requires = \
['httpx>=0.16.1,<0.17.0', 'music21>=5.7.2,<6.0.0', 'requests>=2.24.0,<3.0.0']

setup_kwargs = {
    'name': 'crim-intervals',
    'version': '0.3.7',
    'description': 'A python based, interactive music similarity engine',
    'long_description': '# An interval-analysis based music similarity engine\n\n### Find the project on Github and PyPI\n#### Current Version: 0.3.3\n- [Github](https://github.com/HCDigitalScholarship/intervals)\n- [PyPI](https://pypi.org/project/crim-intervals/)\n- [Streamlet](https://crim-intervals-search.herokuapp.com/)\n\n## Project Capabilities\n- Find exact and close melodic matches given an mei file\n- Find occurences of matching soggetto given a "vectorized" pattern\n- Output a 0-1 similarity score between two musical works\n- Classify melodic matches into periodic entries, imitative duos, fuga\n- Output match data in a variety of ways: command line output, csv, pandas, python data types\n\n## Getting Started\nTo download the project via the Python Packagae Index, use ```pip install crim-intervals``` and in a python shell enter ```from crim_intervals import *```\nTo use the project via github, clone the repository and in a python shell in the directory of the repository enter ```from main import *```  \n\n## Method, Class help\nThe project is now documented with docstrings, for help using/understanding methods or classes use ```help(method_or_class_name)```\n\n## Assisted Usage\nFor a guided way to get results for the basic intended usages of the project, simply enter:\n```\nfrom crim_intervals import *\nassisted_interface()\n```\nwherever you are writing your code. The assisted interface will return an array of matches.\n\n## User-inputted parameters\nEach parameter listed has its own section below detailing configuration.\n- Whether to input one score at a time, or a entire corpus at once with more limited selection ability, as well as what notes are to be analyzed, and the variety of ways in which they can be grouped (Detailed under "Note List Selection- Corpus" and "Note List Selection- Single Score")\n- Whether to create generic or semitone intervals (Detailed under "Creating vectorized representations and selecting their types")\n- The size of pattern to be analyzed (Detailed under "Grouping the vectors into patterns")\n- The minimum number of matches needed to be displayed, and optionally, the cumulative difference threshold for a two patterns to be considered closely matched (Detailed under "Finding close and exact matches")\n\n### Note List Selection- Corpus\nThis section covers the capabilities falling under the CorpusBase object, which has the capability to import multiple pieces at once. To begin, import your scores using either as a list of urls and/or file paths. File paths must begin with a \'/\', otherwise they will be processed as urls.\n```\ncorpus = CorpusBase([\'url_to_mei_file1.mei\', \'url_to_mei_file2.mei\', \'path/to/mei/file1.mei\', \'path/to/mei/file2.mei\'])\n```\nAfter, the first decision to be made is how you want to analyze the imported pieces:\n- Get the whole piece ```corpus.note_list_whole_piece()```\n- Get the whole piece combining unisons into one note ```corpus.note_list_no_unisons()```\n- Get the whole piece only at selected offset within a measure ```corpus.note_list_selected_offset([offset1, offset2, offset3, etc.])```\n- Get the note sounding at every regular offset ```corpus.note_list_incremental_offset(offset_increment)```\n*For more information on each method, use help(method name), for example: help(note_list_incremental_offset)*\n\n### Note List Selection- Single Score\nThis section covers the capabilities falling under the ScoreBase object, which can give more precise note lists, but only for a single piece at a time. To begin, import your score using either\n```score1 = ScoreBase(\'https://url_to_mei_file.mei\')``` for a file url or\n```score2 = ScoreBase(\'/path/to/file.mei\')``` for a file path (this path MUST start with a \'/\', otherwise it will be read as a url\nAfter, decide on how you want to analyze or deconstruct your imported piece:\n- Get the whole piece ```score1.note_list_whole_piece()```\n- Get a note list from a selected measure range within a single voice ```score1.note_list_single_part(part_number, measure_start, measures_until_end)```\n- Get a note list from a selected measure range over all voices ```score1.note_list_all_parts(measure_start, measures_until_end)```\n- Get the whole piece combining unisons into one note ```score1.note_list_no_unisons()```\n- Get the whole piece only at selected offset within a measure ```score1.note_list_selected_offset([offset1, offset2, offset3, etc.])```\n- Get the note sounding at every regular offset ```score1.note_list_incremental_offset(offset_increment)```\n- Get a note list from the whole piece, going by provided beats ```score1.note_list_selected_beat([beat1, beat2, etc.])```\n*For more information on each method, use help(method name), for example: help(note_list_incremental_offset)\n\n### Creating vectorized representations and selecting their types\nAt this point you should have constructed a note list from the methods of a CorpusBase or ScoreBase object. The next step is to group those notes into intervals using the IntervalBase object, which accepts note lists as a list, in case you want to analyze multiple ScoreBase note lists.\n- Multiple note lists: ```vectors = IntervalBase([score1.note_list_whole_piece(), score2.note_list_incremental_offset(2), corpus.note_list_whole_piece()]```\n- Just one: ```vectors = IntervalBase([corpus.note_list_whole_piece()]```\nThe IntervalBase object\'s methods turn the note list given into the vectors with which we do pattern comparisons. To get those vectors, we must decide whether to use generic or semitone intervals:\n- Semitone intervals: ```vectors.semitone_intervals()```\n- Generic intervals: ```vectors.generic_intervals()```\n\n### Grouping the vectors into patterns\nNow that we have a list of vectors (or intervals between notes), we can begin to place them into patterns to be analyzed for similarity. To do so we must select the size of pattern to be used for our analysis:\n```patterns = into_patterns(vectors.generic_intervals, pattern_size)```\n*As always, for information on methods and their parameters, use the help() function- help(into_patterns)*\n\n### Finding close and exact matches\nNow that we have patterns, it is time to analyze them for similarity, which can be either in the form of exact matches, or "close" matches- which gauge similarity based on a cumulative difference threshold (for more on that, see [this example notebook](https://colab.research.google.com/drive/10YmmjOCt2xvkqaJYbBbE5Wu29_sF7mV3?authuser=3#scrollTo=Py-Q9TjiHAfC)). To find only exact matches- or those that follow the same melodic pattern (with potential for transposition across pitches), we bring in the ```patterns``` variable from the previous section:\n```exact_matches = find_exact_matches(patterns, min_matches)```\nwhere the parameter ```min_matches``` determines the minimum number of matches a pattern needs to be considered relevant and displayed. To print information about all matches found, use a simple for loop and another method:\n```\nfor item in exact_matches:\n    item.print_exact_matches()\n```\nAlternatively, if we want to look for "close" matches, we follow a similar stucture, but must provide the threshold detailed above and print slightly differently:\n```\nclose_matches = find_close_matches(patterns, min_matches, threshold)\nfor item in close_matches:\n    item.print_close_matches()\n```\n\n### Accessing information about matches\n There are a few ways information about matches can be accessed.\n- To get information on the command line, use the for loop specified above, using the ```print_exact_matches``` or ```print_close_matches``` methods\n- To export the matches information to a csv file use: ```export_to_csv(exact_matches)``` or ```export_to_csv(close_matches)``` where the parameter for the method is the return value from the match finding functions detailed above.\n- To export the matches information to a pandas dataframe use: ```export_pandas(exact_matches)``` or ```export_pandas(close_matches)``` where the parameter for the method is the return value from the match finding functions detailed above.\n- For more programming-oriented users: The methods ```find_exact_matches``` and ```find_close_matches``` return an array of PatternMatches objects, each of which contain a list of Match object under the parameter ```pattern_match_obj.matches```. Each match object has information about its pattern and the notes which make it up, which can be useful for data analysis. Using the help function is always recommended if parameters/attributes are unclear.\n\n\n### Additional Features\n- Get a similarity "score" between 0 to 1, comparing the motifs shared between two pieces: ```similarity_score(first piece note list, second piece note list)```. The note lists are gathered from the methods of either a ScoreBase or CorpusBase object.\n- Find a desired motif/soggetto within a corpus. Your soggetto must be specified as a list of intervals between notes. For example, the soggetto C-D-E-D-C would be vectorized in generic intervals as [2,2,-2,-2].```find_motif(corpus, soggetto_vector_list)```. If instead you wish to search in terms of semitone intervals, you have to specify an additional parameter as False: ```find_motif(corpus, soggetto_vector_list, False)```\n- Classify Matches into periodic entries, imitative duos, and fuga. Using the return value from ```find_exact_matches``` or ```find_close_matches```, you can classify matches using ```classify_matches(exact_matches)``` or ```classify_matches(exact_matches, 2)``` where the second parameter is an optional cumulative duration difference threshold. The return value of this function is a list of ClassifiedMatch objects, with Match object data inside the parameter matches. Use ```help(ClassifiedMatch)``` for more information.\n  - Additionally, in addition to the printed terminal output, this information can be exported to a csv file using the return value of the function:\n  ```\n  classified_matches = classify_matches(exact_matches)\n  export_to_csv(classified_matches)\n  ```\n',
    'author': 'Freddie Gould',
    'author_email': 'fgould@haverford.edu',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
