"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const tasks = require("../lib");
let stack;
let child;
beforeEach(() => {
    stack = new core_1.Stack();
    child = new sfn.StateMachine(stack, 'ChildStateMachine', {
        definition: sfn.Chain.start(new sfn.Pass(stack, 'PassState')),
    });
});
test('Execute State Machine - Default - Fire and Forget', () => {
    const task = new sfn.Task(stack, 'ChildTask', {
        task: new tasks.StartExecution(child, {
            input: {
                foo: 'bar',
            },
            name: 'myExecutionName',
        }),
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::states:startExecution',
                ],
            ],
        },
        End: true,
        Parameters: {
            Input: {
                foo: 'bar',
            },
            Name: 'myExecutionName',
            StateMachineArn: {
                Ref: 'ChildStateMachine9133117F',
            },
        },
    });
});
test('Execute State Machine - Sync', () => {
    const task = new sfn.Task(stack, 'ChildTask', {
        task: new tasks.StartExecution(child, {
            integrationPattern: sfn.ServiceIntegrationPattern.SYNC,
        }),
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::states:startExecution.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            StateMachineArn: {
                Ref: 'ChildStateMachine9133117F',
            },
        },
    });
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'states:StartExecution',
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ChildStateMachine9133117F',
                    },
                },
                {
                    Action: [
                        'states:DescribeExecution',
                        'states:StopExecution',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':states:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':execution:',
                                {
                                    'Fn::Select': [
                                        6,
                                        {
                                            'Fn::Split': [
                                                ':',
                                                {
                                                    Ref: 'ChildStateMachine9133117F',
                                                },
                                            ],
                                        },
                                    ],
                                },
                                '*',
                            ],
                        ],
                    },
                },
                {
                    Action: [
                        'events:PutTargets',
                        'events:PutRule',
                        'events:DescribeRule',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':events:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':rule/StepFunctionsGetEventsForStepFunctionsExecutionRule',
                            ],
                        ],
                    },
                },
            ],
            Version: '2012-10-17',
        },
        Roles: [
            {
                Ref: 'ParentStateMachineRoleE902D002',
            },
        ],
    });
});
test('Execute State Machine - Wait For Task Token', () => {
    const task = new sfn.Task(stack, 'ChildTask', {
        task: new tasks.StartExecution(child, {
            integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            input: {
                token: sfn.JsonPath.taskToken,
            },
        }),
    });
    new sfn.StateMachine(stack, 'ParentStateMachine', {
        definition: task,
    });
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::states:startExecution.waitForTaskToken',
                ],
            ],
        },
        End: true,
        Parameters: {
            Input: {
                'token.$': '$$.Task.Token',
            },
            StateMachineArn: {
                Ref: 'ChildStateMachine9133117F',
            },
        },
    });
});
test('Execute State Machine - Wait For Task Token - Missing Task Token', () => {
    expect(() => {
        new sfn.Task(stack, 'ChildTask', {
            task: new tasks.StartExecution(child, {
                integrationPattern: sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
            }),
        });
    }).toThrow('Task Token is missing in input (pass JsonPath.taskToken somewhere in input');
});
//# sourceMappingURL=data:application/json;base64,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