"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sns = require("@aws-cdk/aws-sns");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const publish_1 = require("../../lib/sns/publish");
describe('Publish', () => {
    test('default settings', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        const task = new publish_1.SnsPublish(stack, 'Publish', {
            topic,
            message: sfn.TaskInput.fromText('Publish this message'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sns:publish',
                    ],
                ],
            },
            End: true,
            Parameters: {
                TopicArn: { Ref: 'TopicBFC7AF6E' },
                Message: 'Publish this message',
            },
        });
    });
    test('publish SNS message and wait for task token', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        const task = new publish_1.SnsPublish(stack, 'Publish', {
            topic,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            message: sfn.TaskInput.fromObject({
                Input: 'Publish this message',
                Token: sfn.JsonPath.taskToken,
            }),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sns:publish.waitForTaskToken',
                    ],
                ],
            },
            End: true,
            Parameters: {
                TopicArn: { Ref: 'TopicBFC7AF6E' },
                Message: {
                    'Input': 'Publish this message',
                    'Token.$': '$$.Task.Token',
                },
            },
        });
    });
    test('publish different message per subscription type', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        // WHEN
        const task = new publish_1.SnsPublish(stack, 'Publish', {
            topic,
            integrationPattern: sfn.IntegrationPattern.REQUEST_RESPONSE,
            message: sfn.TaskInput.fromObject({
                default: 'A message',
                sqs: 'A message for Amazon SQS',
                email: 'A message for email',
            }),
            messagePerSubscriptionType: true,
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            End: true,
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sns:publish',
                    ],
                ],
            },
            Parameters: {
                TopicArn: { Ref: 'TopicBFC7AF6E' },
                Message: {
                    default: 'A message',
                    sqs: 'A message for Amazon SQS',
                    email: 'A message for email',
                },
                MessageStructure: 'json',
            },
        });
    });
    test('topic ARN supplied through the task input', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = sns.Topic.fromTopicArn(stack, 'Topic', sfn.JsonPath.stringAt('$.topicArn'));
        // WHEN
        const task = new publish_1.SnsPublish(stack, 'Publish', {
            topic,
            message: sfn.TaskInput.fromText('Publish this message'),
        });
        // THEN
        expect(stack.resolve(task.toStateJson())).toEqual({
            Type: 'Task',
            Resource: {
                'Fn::Join': [
                    '',
                    [
                        'arn:',
                        {
                            Ref: 'AWS::Partition',
                        },
                        ':states:::sns:publish',
                    ],
                ],
            },
            End: true,
            Parameters: {
                'TopicArn.$': '$.topicArn',
                'Message': 'Publish this message',
            },
        });
    });
    test('fails when WAIT_FOR_TASK_TOKEN integration pattern is used without supplying a task token in message', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        expect(() => {
            // WHEN
            new publish_1.SnsPublish(stack, 'Publish', {
                topic,
                integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
                message: sfn.TaskInput.fromText('Publish this message'),
            });
            // THEN
        }).toThrow(/Task Token is required in `message` Use JsonPath.taskToken to set the token./);
    });
    test('fails when RUN_JOB integration pattern is used', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const topic = new sns.Topic(stack, 'Topic');
        expect(() => {
            new publish_1.SnsPublish(stack, 'Publish', {
                topic,
                integrationPattern: sfn.IntegrationPattern.RUN_JOB,
                message: sfn.TaskInput.fromText('Publish this message'),
            });
        }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,WAIT_FOR_TASK_TOKEN. Received: RUN_JOB/);
    });
});
//# sourceMappingURL=data:application/json;base64,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