"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
const kms = require("@aws-cdk/aws-kms");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const tasks = require("../../lib");
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
test('create basic endpoint config', () => {
    // WHEN
    const task = new tasks.SageMakerCreateEndpointConfig(stack, 'SagemakerEndpointConfig', {
        endpointConfigName: 'MyEndpointConfig',
        productionVariants: [{
                initialInstanceCount: 2,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
                modelName: 'MyModel',
                variantName: 'awesome-variant',
            }],
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createEndpointConfig',
                ],
            ],
        },
        End: true,
        Parameters: {
            EndpointConfigName: 'MyEndpointConfig',
            ProductionVariants: [{
                    InitialInstanceCount: 2,
                    InstanceType: 'ml.m5.xlarge',
                    ModelName: 'MyModel',
                    VariantName: 'awesome-variant',
                }],
        },
    });
});
test('create complex endpoint config', () => {
    // WHEN
    const key = new kms.Key(stack, 'Key');
    const task = new tasks.SageMakerCreateEndpointConfig(stack, 'SagemakerEndpointConfig', {
        endpointConfigName: sfn.JsonPath.stringAt('$.Endpoint.EndpointConfig'),
        kmsKey: key,
        productionVariants: [{
                initialInstanceCount: 1,
                initialVariantWeight: 0.8,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
                modelName: 'MyModel',
                variantName: 'awesome-variant',
            },
            {
                initialInstanceCount: 1,
                initialVariantWeight: 0.2,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
                modelName: sfn.JsonPath.stringAt('$.Endpoint.Model'),
                variantName: 'awesome-variant-2',
            }],
        tags: sfn.TaskInput.fromObject([{
                Key: 'Project',
                Value: 'ML',
            }]),
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::sagemaker:createEndpointConfig',
                ],
            ],
        },
        End: true,
        Parameters: {
            'EndpointConfigName.$': '$.Endpoint.EndpointConfig',
            'KmsKeyId': {
                Ref: 'Key961B73FD',
            },
            'ProductionVariants': [{
                    InitialInstanceCount: 1,
                    InitialVariantWeight: 0.8,
                    InstanceType: 'ml.m5.xlarge',
                    ModelName: 'MyModel',
                    VariantName: 'awesome-variant',
                },
                {
                    'InitialInstanceCount': 1,
                    'InitialVariantWeight': 0.2,
                    'InstanceType': 'ml.m4.xlarge',
                    'ModelName.$': '$.Endpoint.Model',
                    'VariantName': 'awesome-variant-2',
                }],
            'Tags': [
                {
                    Key: 'Project',
                    Value: 'ML',
                },
            ],
        },
    });
});
test('Cannot create a SageMaker create enpoint config task with empty production variant', () => {
    expect(() => new tasks.SageMakerCreateEndpointConfig(stack, 'EndpointConfig', {
        endpointConfigName: 'MyEndpointConfig',
        productionVariants: [],
    }))
        .toThrowError(/Must specify from 1 to 10 production variants per endpoint configuration/);
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => new tasks.SageMakerCreateEndpointConfig(stack, 'EndpointConfig', {
        endpointConfigName: 'MyEndpointConfig',
        integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        productionVariants: [{
                initialInstanceCount: 2,
                instanceType: ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.XLARGE),
                modelName: 'MyModel',
                variantName: 'awesome-variant',
            }],
    }))
        .toThrowError(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE. Received: WAIT_FOR_TASK_TOKEN/i);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlLWVuZHBvaW50LWNvbmZpZy50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3JlYXRlLWVuZHBvaW50LWNvbmZpZy50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLHdDQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsa0RBQWtEO0FBQ2xELHFDQUFxQztBQUNyQyxtQ0FBbUM7QUFFbkMsSUFBSSxLQUFnQixDQUFDO0FBRXJCLFVBQVUsQ0FBQyxHQUFHLEVBQUU7SUFDZCxRQUFRO0lBQ1IsS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0FBQzFCLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhCQUE4QixFQUFFLEdBQUcsRUFBRTtJQUN4QyxPQUFPO0lBQ1AsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLENBQUMsNkJBQTZCLENBQUMsS0FBSyxFQUFFLHlCQUF5QixFQUFFO1FBQ3JGLGtCQUFrQixFQUFFLGtCQUFrQjtRQUN0QyxrQkFBa0IsRUFBRSxDQUFDO2dCQUNuQixvQkFBb0IsRUFBRSxDQUFDO2dCQUN2QixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7Z0JBQ2hGLFNBQVMsRUFBRSxTQUFTO2dCQUNwQixXQUFXLEVBQUUsaUJBQWlCO2FBQy9CLENBQUM7S0FDSCxDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDBDQUEwQztpQkFDM0M7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixrQkFBa0IsRUFBRSxrQkFBa0I7WUFDdEMsa0JBQWtCLEVBQUUsQ0FBQztvQkFDbkIsb0JBQW9CLEVBQUUsQ0FBQztvQkFDdkIsWUFBWSxFQUFFLGNBQWM7b0JBQzVCLFNBQVMsRUFBRSxTQUFTO29CQUNwQixXQUFXLEVBQUUsaUJBQWlCO2lCQUMvQixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7SUFDMUMsT0FBTztJQUNQLE1BQU0sR0FBRyxHQUFHLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFdEMsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLENBQUMsNkJBQTZCLENBQUMsS0FBSyxFQUFFLHlCQUF5QixFQUFFO1FBQ3JGLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLDJCQUEyQixDQUFDO1FBQ3RFLE1BQU0sRUFBRSxHQUFHO1FBQ1gsa0JBQWtCLEVBQUUsQ0FBQztnQkFDbkIsb0JBQW9CLEVBQUUsQ0FBQztnQkFDdkIsb0JBQW9CLEVBQUUsR0FBRztnQkFDekIsWUFBWSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUNoRixTQUFTLEVBQUUsU0FBUztnQkFDcEIsV0FBVyxFQUFFLGlCQUFpQjthQUMvQjtZQUNEO2dCQUNFLG9CQUFvQixFQUFFLENBQUM7Z0JBQ3ZCLG9CQUFvQixFQUFFLEdBQUc7Z0JBQ3pCLFlBQVksRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDLEVBQUUsRUFBRSxHQUFHLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQztnQkFDaEYsU0FBUyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDO2dCQUNwRCxXQUFXLEVBQUUsbUJBQW1CO2FBQ2pDLENBQUM7UUFDRixJQUFJLEVBQUUsR0FBRyxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDOUIsR0FBRyxFQUFFLFNBQVM7Z0JBQ2QsS0FBSyxFQUFFLElBQUk7YUFDWixDQUFDLENBQUM7S0FDSixDQUFDLENBQUM7SUFFSCxPQUFPO0lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDaEQsSUFBSSxFQUFFLE1BQU07UUFDWixRQUFRLEVBQUU7WUFDUixVQUFVLEVBQUU7Z0JBQ1YsRUFBRTtnQkFDRjtvQkFDRSxNQUFNO29CQUNOO3dCQUNFLEdBQUcsRUFBRSxnQkFBZ0I7cUJBQ3RCO29CQUNELDBDQUEwQztpQkFDM0M7YUFDRjtTQUNGO1FBQ0QsR0FBRyxFQUFFLElBQUk7UUFDVCxVQUFVLEVBQUU7WUFDVixzQkFBc0IsRUFBRSwyQkFBMkI7WUFDbkQsVUFBVSxFQUFFO2dCQUNWLEdBQUcsRUFBRSxhQUFhO2FBQ25CO1lBQ0Qsb0JBQW9CLEVBQUUsQ0FBQztvQkFDckIsb0JBQW9CLEVBQUUsQ0FBQztvQkFDdkIsb0JBQW9CLEVBQUUsR0FBRztvQkFDekIsWUFBWSxFQUFFLGNBQWM7b0JBQzVCLFNBQVMsRUFBRSxTQUFTO29CQUNwQixXQUFXLEVBQUUsaUJBQWlCO2lCQUMvQjtnQkFDRDtvQkFDRSxzQkFBc0IsRUFBRSxDQUFDO29CQUN6QixzQkFBc0IsRUFBRSxHQUFHO29CQUMzQixjQUFjLEVBQUUsY0FBYztvQkFDOUIsYUFBYSxFQUFFLGtCQUFrQjtvQkFDakMsYUFBYSxFQUFFLG1CQUFtQjtpQkFDbkMsQ0FBQztZQUNGLE1BQU0sRUFBRTtnQkFDTjtvQkFDRSxHQUFHLEVBQUUsU0FBUztvQkFDZCxLQUFLLEVBQUUsSUFBSTtpQkFDWjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxvRkFBb0YsRUFBRSxHQUFHLEVBQUU7SUFDOUYsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksS0FBSyxDQUFDLDZCQUE2QixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUM1RSxrQkFBa0IsRUFBRSxrQkFBa0I7UUFDdEMsa0JBQWtCLEVBQUUsRUFBRTtLQUN2QixDQUFDLENBQUM7U0FDQSxZQUFZLENBQUMsMEVBQTBFLENBQUMsQ0FBQztBQUM5RixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrRUFBK0UsRUFBRSxHQUFHLEVBQUU7SUFDekYsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksS0FBSyxDQUFDLDZCQUE2QixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUM1RSxrQkFBa0IsRUFBRSxrQkFBa0I7UUFDdEMsa0JBQWtCLEVBQUUsR0FBRyxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQjtRQUM5RCxrQkFBa0IsRUFBRSxDQUFDO2dCQUNuQixvQkFBb0IsRUFBRSxDQUFDO2dCQUN2QixZQUFZLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUM7Z0JBQ2hGLFNBQVMsRUFBRSxTQUFTO2dCQUNwQixXQUFXLEVBQUUsaUJBQWlCO2FBQy9CLENBQUM7S0FDSCxDQUFDLENBQUM7U0FDQSxZQUFZLENBQUMsK0dBQStHLENBQUMsQ0FBQztBQUNuSSxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMga21zIGZyb20gJ0Bhd3MtY2RrL2F3cy1rbXMnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJy4uLy4uL2xpYic7XG5cbmxldCBzdGFjazogY2RrLlN0YWNrO1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgLy8gR0lWRU5cbiAgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG59KTtcblxudGVzdCgnY3JlYXRlIGJhc2ljIGVuZHBvaW50IGNvbmZpZycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCB0YXNrID0gbmV3IHRhc2tzLlNhZ2VNYWtlckNyZWF0ZUVuZHBvaW50Q29uZmlnKHN0YWNrLCAnU2FnZW1ha2VyRW5kcG9pbnRDb25maWcnLCB7XG4gICAgZW5kcG9pbnRDb25maWdOYW1lOiAnTXlFbmRwb2ludENvbmZpZycsXG4gICAgcHJvZHVjdGlvblZhcmlhbnRzOiBbe1xuICAgICAgaW5pdGlhbEluc3RhbmNlQ291bnQ6IDIsXG4gICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuTTUsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFKSxcbiAgICAgIG1vZGVsTmFtZTogJ015TW9kZWwnLFxuICAgICAgdmFyaWFudE5hbWU6ICdhd2Vzb21lLXZhcmlhbnQnLFxuICAgIH1dLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OnNhZ2VtYWtlcjpjcmVhdGVFbmRwb2ludENvbmZpZycsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgIEVuZHBvaW50Q29uZmlnTmFtZTogJ015RW5kcG9pbnRDb25maWcnLFxuICAgICAgUHJvZHVjdGlvblZhcmlhbnRzOiBbe1xuICAgICAgICBJbml0aWFsSW5zdGFuY2VDb3VudDogMixcbiAgICAgICAgSW5zdGFuY2VUeXBlOiAnbWwubTUueGxhcmdlJyxcbiAgICAgICAgTW9kZWxOYW1lOiAnTXlNb2RlbCcsXG4gICAgICAgIFZhcmlhbnROYW1lOiAnYXdlc29tZS12YXJpYW50JyxcbiAgICAgIH1dLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2NyZWF0ZSBjb21wbGV4IGVuZHBvaW50IGNvbmZpZycsICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCBrZXkgPSBuZXcga21zLktleShzdGFjaywgJ0tleScpO1xuXG4gIGNvbnN0IHRhc2sgPSBuZXcgdGFza3MuU2FnZU1ha2VyQ3JlYXRlRW5kcG9pbnRDb25maWcoc3RhY2ssICdTYWdlbWFrZXJFbmRwb2ludENvbmZpZycsIHtcbiAgICBlbmRwb2ludENvbmZpZ05hbWU6IHNmbi5Kc29uUGF0aC5zdHJpbmdBdCgnJC5FbmRwb2ludC5FbmRwb2ludENvbmZpZycpLFxuICAgIGttc0tleToga2V5LFxuICAgIHByb2R1Y3Rpb25WYXJpYW50czogW3tcbiAgICAgIGluaXRpYWxJbnN0YW5jZUNvdW50OiAxLFxuICAgICAgaW5pdGlhbFZhcmlhbnRXZWlnaHQ6IDAuOCxcbiAgICAgIGluc3RhbmNlVHlwZTogZWMyLkluc3RhbmNlVHlwZS5vZihlYzIuSW5zdGFuY2VDbGFzcy5NNSwgZWMyLkluc3RhbmNlU2l6ZS5YTEFSR0UpLFxuICAgICAgbW9kZWxOYW1lOiAnTXlNb2RlbCcsXG4gICAgICB2YXJpYW50TmFtZTogJ2F3ZXNvbWUtdmFyaWFudCcsXG4gICAgfSxcbiAgICB7XG4gICAgICBpbml0aWFsSW5zdGFuY2VDb3VudDogMSxcbiAgICAgIGluaXRpYWxWYXJpYW50V2VpZ2h0OiAwLjIsXG4gICAgICBpbnN0YW5jZVR5cGU6IGVjMi5JbnN0YW5jZVR5cGUub2YoZWMyLkluc3RhbmNlQ2xhc3MuTTQsIGVjMi5JbnN0YW5jZVNpemUuWExBUkdFKSxcbiAgICAgIG1vZGVsTmFtZTogc2ZuLkpzb25QYXRoLnN0cmluZ0F0KCckLkVuZHBvaW50Lk1vZGVsJyksXG4gICAgICB2YXJpYW50TmFtZTogJ2F3ZXNvbWUtdmFyaWFudC0yJyxcbiAgICB9XSxcbiAgICB0YWdzOiBzZm4uVGFza0lucHV0LmZyb21PYmplY3QoW3tcbiAgICAgIEtleTogJ1Byb2plY3QnLFxuICAgICAgVmFsdWU6ICdNTCcsXG4gICAgfV0pLFxuICB9KTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChzdGFjay5yZXNvbHZlKHRhc2sudG9TdGF0ZUpzb24oKSkpLnRvRXF1YWwoe1xuICAgIFR5cGU6ICdUYXNrJyxcbiAgICBSZXNvdXJjZToge1xuICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAnJyxcbiAgICAgICAgW1xuICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBSZWY6ICdBV1M6OlBhcnRpdGlvbicsXG4gICAgICAgICAgfSxcbiAgICAgICAgICAnOnN0YXRlczo6OnNhZ2VtYWtlcjpjcmVhdGVFbmRwb2ludENvbmZpZycsXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0sXG4gICAgRW5kOiB0cnVlLFxuICAgIFBhcmFtZXRlcnM6IHtcbiAgICAgICdFbmRwb2ludENvbmZpZ05hbWUuJCc6ICckLkVuZHBvaW50LkVuZHBvaW50Q29uZmlnJyxcbiAgICAgICdLbXNLZXlJZCc6IHtcbiAgICAgICAgUmVmOiAnS2V5OTYxQjczRkQnLFxuICAgICAgfSxcbiAgICAgICdQcm9kdWN0aW9uVmFyaWFudHMnOiBbe1xuICAgICAgICBJbml0aWFsSW5zdGFuY2VDb3VudDogMSxcbiAgICAgICAgSW5pdGlhbFZhcmlhbnRXZWlnaHQ6IDAuOCxcbiAgICAgICAgSW5zdGFuY2VUeXBlOiAnbWwubTUueGxhcmdlJyxcbiAgICAgICAgTW9kZWxOYW1lOiAnTXlNb2RlbCcsXG4gICAgICAgIFZhcmlhbnROYW1lOiAnYXdlc29tZS12YXJpYW50JyxcbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgICdJbml0aWFsSW5zdGFuY2VDb3VudCc6IDEsXG4gICAgICAgICdJbml0aWFsVmFyaWFudFdlaWdodCc6IDAuMixcbiAgICAgICAgJ0luc3RhbmNlVHlwZSc6ICdtbC5tNC54bGFyZ2UnLFxuICAgICAgICAnTW9kZWxOYW1lLiQnOiAnJC5FbmRwb2ludC5Nb2RlbCcsXG4gICAgICAgICdWYXJpYW50TmFtZSc6ICdhd2Vzb21lLXZhcmlhbnQtMicsXG4gICAgICB9XSxcbiAgICAgICdUYWdzJzogW1xuICAgICAgICB7XG4gICAgICAgICAgS2V5OiAnUHJvamVjdCcsXG4gICAgICAgICAgVmFsdWU6ICdNTCcsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0Nhbm5vdCBjcmVhdGUgYSBTYWdlTWFrZXIgY3JlYXRlIGVucG9pbnQgY29uZmlnIHRhc2sgd2l0aCBlbXB0eSBwcm9kdWN0aW9uIHZhcmlhbnQnLCAoKSA9PiB7XG4gIGV4cGVjdCgoKSA9PiBuZXcgdGFza3MuU2FnZU1ha2VyQ3JlYXRlRW5kcG9pbnRDb25maWcoc3RhY2ssICdFbmRwb2ludENvbmZpZycsIHtcbiAgICBlbmRwb2ludENvbmZpZ05hbWU6ICdNeUVuZHBvaW50Q29uZmlnJyxcbiAgICBwcm9kdWN0aW9uVmFyaWFudHM6IFtdLFxuICB9KSlcbiAgICAudG9UaHJvd0Vycm9yKC9NdXN0IHNwZWNpZnkgZnJvbSAxIHRvIDEwIHByb2R1Y3Rpb24gdmFyaWFudHMgcGVyIGVuZHBvaW50IGNvbmZpZ3VyYXRpb24vKTtcbn0pO1xuXG50ZXN0KCdUYXNrIHRocm93cyBpZiBXQUlUX0ZPUl9UQVNLX1RPS0VOIGlzIHN1cHBsaWVkIGFzIHNlcnZpY2UgaW50ZWdyYXRpb24gcGF0dGVybicsICgpID0+IHtcbiAgZXhwZWN0KCgpID0+IG5ldyB0YXNrcy5TYWdlTWFrZXJDcmVhdGVFbmRwb2ludENvbmZpZyhzdGFjaywgJ0VuZHBvaW50Q29uZmlnJywge1xuICAgIGVuZHBvaW50Q29uZmlnTmFtZTogJ015RW5kcG9pbnRDb25maWcnLFxuICAgIGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgIHByb2R1Y3Rpb25WYXJpYW50czogW3tcbiAgICAgIGluaXRpYWxJbnN0YW5jZUNvdW50OiAyLFxuICAgICAgaW5zdGFuY2VUeXBlOiBlYzIuSW5zdGFuY2VUeXBlLm9mKGVjMi5JbnN0YW5jZUNsYXNzLk01LCBlYzIuSW5zdGFuY2VTaXplLlhMQVJHRSksXG4gICAgICBtb2RlbE5hbWU6ICdNeU1vZGVsJyxcbiAgICAgIHZhcmlhbnROYW1lOiAnYXdlc29tZS12YXJpYW50JyxcbiAgICB9XSxcbiAgfSkpXG4gICAgLnRvVGhyb3dFcnJvcigvVW5zdXBwb3J0ZWQgc2VydmljZSBpbnRlZ3JhdGlvbiBwYXR0ZXJuLiBTdXBwb3J0ZWQgUGF0dGVybnM6IFJFUVVFU1RfUkVTUE9OU0UuIFJlY2VpdmVkOiBXQUlUX0ZPUl9UQVNLX1RPS0VOL2kpO1xufSk7XG5cblxuIl19