"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const sfn = require("@aws-cdk/aws-stepfunctions");
const tasks = require("../../lib");
describe('DynamoProjectionExpression', () => {
    test('should correctly configure projectionExpression', () => {
        expect(new tasks.DynamoProjectionExpression()
            .withAttribute('Messages')
            .atIndex(1)
            .atIndex(10)
            .withAttribute('Tags')
            .withAttribute('Items')
            .atIndex(0)
            .toString()).toEqual('Messages[1][10].Tags.Items[0]');
    });
    test('should throw if expression starts with atIndex', () => {
        expect(() => new tasks.DynamoProjectionExpression().atIndex(1).withAttribute('Messages').toString()).toThrow(/Expression must start with an attribute/);
    });
});
describe('DynamoAttributeValue', () => {
    test('from string with a string literal', () => {
        // GIVEN
        const s = 'my-string';
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromString(s);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                S: s,
            },
        });
    });
    test('from string with a json path', () => {
        // GIVEN
        const s = '$.string';
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromString(sfn.JsonPath.stringAt(s));
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                'S.$': s,
            },
        });
    });
    test('from number', () => {
        // GIVEN
        const n = 9;
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromNumber(n);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                N: `${n}`,
            },
        });
    });
    test('number from string', () => {
        // GIVEN
        const n = '9';
        // WHEN
        const attribute = tasks.DynamoAttributeValue.numberFromString(n);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                N: n,
            },
        });
    });
    test('from binary', () => {
        // GIVEN
        const b = 'ejBtZ3d0ZmJicQ==';
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromBinary(b);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                B: b,
            },
        });
    });
    test('from string set', () => {
        // GIVEN
        const ss = ['apple', 'banana'];
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromStringSet(ss);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                SS: ss,
            },
        });
    });
    test('from number set', () => {
        // GIVEN
        const ns = [1, 2];
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromNumberSet(ns);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                NS: ['1', '2'],
            },
        });
    });
    test('number set from strings', () => {
        // GIVEN
        const ns = ['1', '2'];
        // WHEN
        const attribute = tasks.DynamoAttributeValue.numberSetFromStrings(ns);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                NS: ns,
            },
        });
    });
    test('from binary set', () => {
        // GIVEN
        const bs = ['Y2RrIGlzIGF3ZXNvbWU=', 'ejBtZ3d0ZmJicQ=='];
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromBinarySet(bs);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                BS: bs,
            },
        });
    });
    test('from map', () => {
        // GIVEN
        const m = { cdk: tasks.DynamoAttributeValue.fromString('is-cool') };
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromMap(m);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                M: {
                    cdk: { S: 'is-cool' },
                },
            },
        });
    });
    test('map from json path', () => {
        // GIVEN
        const m = '$.path';
        // WHEN
        const attribute = tasks.DynamoAttributeValue.mapFromJsonPath(m);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                'M.$': m,
            },
        });
    });
    test('map from invalid json path throws', () => {
        // GIVEN
        const m = 'invalid';
        // WHEN / THEN
        expect(() => {
            tasks.DynamoAttributeValue.mapFromJsonPath(m);
        }).toThrow("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    });
    test('from list', () => {
        // GIVEN
        const l = [tasks.DynamoAttributeValue.fromString('a string'), tasks.DynamoAttributeValue.fromNumber(7)];
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromList(l);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                L: [
                    {
                        S: 'a string',
                    },
                    {
                        N: '7',
                    },
                ],
            },
        });
    });
    test('from null', () => {
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromNull(true);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                NULL: true,
            },
        });
    });
    test('from invalid boolean with json path', () => {
        // GIVEN
        const m = 'invalid';
        // WHEN / THEN
        expect(() => {
            tasks.DynamoAttributeValue.booleanFromJsonPath(m);
        }).toThrow("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    });
    test('from boolean with json path', () => {
        // GIVEN
        const m = '$.path';
        // WHEN
        const attribute = tasks.DynamoAttributeValue.booleanFromJsonPath(sfn.JsonPath.stringAt(m));
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                'BOOL.$': m,
            },
        });
    });
    test('from boolean', () => {
        // WHEN
        const attribute = tasks.DynamoAttributeValue.fromBoolean(true);
        // THEN
        expect(sfn.FieldUtils.renderObject(attribute)).toEqual({
            attributeValue: {
                BOOL: true,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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