"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const codebuild = require("@aws-cdk/aws-codebuild");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const lib_1 = require("../../lib");
let stack;
let codebuildProject;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    codebuildProject = new codebuild.Project(stack, 'Project', {
        projectName: 'MyTestProject',
        buildSpec: codebuild.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                build: {
                    commands: [
                        'echo "Hello, CodeBuild!"',
                    ],
                },
            },
        }),
    });
});
test('Task with only the required parameters', () => {
    // WHEN
    const task = new lib_1.CodeBuildStartBuild(stack, 'Task', {
        project: codebuildProject,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::codebuild:startBuild.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ProjectName: {
                Ref: 'ProjectC78D97AD',
            },
        },
    });
});
test('Task with all the parameters', () => {
    // WHEN
    const task = new lib_1.CodeBuildStartBuild(stack, 'Task', {
        project: codebuildProject,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        environmentVariablesOverride: {
            env: {
                type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                value: 'prod',
            },
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::codebuild:startBuild.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ProjectName: {
                Ref: 'ProjectC78D97AD',
            },
            EnvironmentVariablesOverride: [
                {
                    Name: 'env',
                    Type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    Value: 'prod',
                },
            ],
        },
    });
});
test('supports tokens', () => {
    // WHEN
    const task = new lib_1.CodeBuildStartBuild(stack, 'Task', {
        project: codebuildProject,
        integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        environmentVariablesOverride: {
            ZONE: {
                type: codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                value: sfn.JsonPath.stringAt('$.envVariables.zone'),
            },
        },
    });
    // THEN
    expect(stack.resolve(task.toStateJson())).toEqual({
        Type: 'Task',
        Resource: {
            'Fn::Join': [
                '',
                [
                    'arn:',
                    {
                        Ref: 'AWS::Partition',
                    },
                    ':states:::codebuild:startBuild.sync',
                ],
            ],
        },
        End: true,
        Parameters: {
            ProjectName: {
                Ref: 'ProjectC78D97AD',
            },
            EnvironmentVariablesOverride: [
                {
                    'Name': 'ZONE',
                    'Type': codebuild.BuildEnvironmentVariableType.PLAINTEXT,
                    'Value.$': '$.envVariables.zone',
                },
            ],
        },
    });
});
test('Task throws if WAIT_FOR_TASK_TOKEN is supplied as service integration pattern', () => {
    expect(() => {
        new lib_1.CodeBuildStartBuild(stack, 'Task', {
            project: codebuildProject,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
        });
    }).toThrow(/Unsupported service integration pattern. Supported Patterns: REQUEST_RESPONSE,RUN_JOB. Received: WAIT_FOR_TASK_TOKEN/);
});
//# sourceMappingURL=data:application/json;base64,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