"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateModel = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * A Step Functions Task to create a SageMaker model
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-sagemaker.html
 * @experimental
 */
class SageMakerCreateModel extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS);
        // add the security groups to the connections object
        if (props.vpc) {
            this.vpc = props.vpc;
            this.subnets = props.subnetSelection ? this.vpc.selectSubnets(props.subnetSelection).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.role = this.props.role || this.createSagemakerRole();
        this.grantPrincipal = this.role;
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createModel', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        var _a, _b;
        return {
            EnableNetworkIsolation: this.props.enableNetworkIsolation,
            ExecutionRoleArn: this.role.roleArn,
            ModelName: this.props.modelName,
            Tags: (_a = this.props.tags) === null || _a === void 0 ? void 0 : _a.value,
            PrimaryContainer: this.props.primaryContainer.bind(this).parameters,
            Containers: (_b = this.props.containers) === null || _b === void 0 ? void 0 : _b.map(container => (container.bind(this))),
            ...this.renderVpcConfig(),
        };
    }
    makePolicyStatements() {
        const stack = cdk.Stack.of(this);
        return [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateModel'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'model',
                        // If the model name comes from input, we cannot target the policy to a particular ARN prefix reliably.
                        // SageMaker uses lowercase for resource name in the arn
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.modelName) ? '*' : `${this.props.modelName.toLowerCase()}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
    }
    createSagemakerRole() {
        // https://docs.aws.amazon.com/sagemaker/latest/dg/sagemaker-roles.html
        const role = new iam.Role(this, 'SagemakerRole', {
            assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            inlinePolicies: {
                CreateModel: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: [
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatch.html
                                'cloudwatch:PutMetricData',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncloudwatchlogs.html
                                'logs:CreateLogStream',
                                'logs:CreateLogGroup',
                                'logs:PutLogEvents',
                                'logs:DescribeLogStreams',
                                // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerregistry.html
                                'ecr:GetAuthorizationToken',
                            ],
                            resources: ['*'],
                        }),
                    ],
                }),
            },
        });
        if (this.props.vpc) {
            role.addToPrincipalPolicy(
            // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            new iam.PolicyStatement({
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DeleteNetworkInterface',
                    'ec2:DeleteNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DescribeVpcs',
                    'ec2:DescribeDhcpOptions',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                ],
                resources: ['*'],
            }));
        }
        return role;
    }
    renderVpcConfig() {
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'ModelSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        return this.vpc
            ? {
                VpcConfig: {
                    SecurityGroupIds: cdk.Lazy.listValue({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
}
exports.SageMakerCreateModel = SageMakerCreateModel;
SageMakerCreateModel.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
];
//# sourceMappingURL=data:application/json;base64,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