# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/hierarchical.ipynb (unless otherwise specified).

__all__ = ['Labour', 'Tourism', 'TourismLarge', 'TourismSmall', 'Traffic', 'Wiki2', 'HierarchicalInfo',
           'HierarchicalData']

# Cell
from dataclasses import dataclass
from pathlib import Path
from typing import Tuple

import pandas as pd

from .utils import download_file, Info

# Cell
@dataclass
class Labour:
    freq = 'M'

# Cell
@dataclass
class Tourism:
    freq = 'Q'

# Cell
@dataclass
class TourismLarge:
    freq = 'M'

# Cell
@dataclass
class TourismSmall:
    freq = 'Q'

# Cell
@dataclass
class Traffic:
    freq = 'D'

# Cell
@dataclass
class Wiki2:
    freq = 'D'

# Cell
HierarchicalInfo = Info(
    (
        Labour, Tourism, TourismLarge,
        TourismSmall,
        Traffic, Wiki2
    )
)

# Cell
class HierarchicalData:

    source_url: str = 'https://nixtla-public.s3.amazonaws.com/hierarchical-data/datasets.zip'

    @staticmethod
    def load(directory: str,
             group: str,
             cache: bool = True) -> Tuple[pd.DataFrame, pd.DataFrame]:
        """
        Downloads hierarchical forecasting benchmark datasets.

            Parameters
            ----------
            directory: str
                Directory where data will be downloaded.
            group: str
                Group name.
            cache: bool
                If `True` saves and loads

            Returns
            -------
            Y_df: pd.DataFrame
                Target time series with columns ['unique_id', 'ds', 'y'].
                Containes the base time series.
            S: pd.DataFrame
                Summing matrix of size (hierarchies, bottom).
        """
        if group not in HierarchicalInfo.groups:
            raise Exception(f'group not found {group}')

        path = f'{directory}/hierarchical/'
        file_cache = Path(f'{path}/{group}.p')

        if file_cache.is_file() and cache:
            Y_df, S = pd.read_pickle(file_cache)

            return Y_df, S

        HierarchicalData.download(directory)
        path = Path(f'{path}/{group}')
        S = pd.read_csv(path / 'agg_mat.csv', index_col=0)
        Y_df = pd.read_csv(path / 'data.csv', index_col=0).T
        Y_df = Y_df.stack()
        Y_df.name = 'y'
        Y_df.index = Y_df.index.set_names(['unique_id', 'ds'])
        Y_df = Y_df.reset_index()

        if not all(Y_df['unique_id'].unique() == S.index):
            raise Exception('mismatch order between `Y_df` and `S`')

        if cache:
            pd.to_pickle((Y_df, S), file_cache)

        return Y_df, S


    @staticmethod
    def download(directory: str) -> None:
        """
        Download Hierarchical Datasets.

            Parameters
            ----------
            directory: str
                Directory path to download dataset.
        """
        path = f'{directory}/hierarchical/'
        if not Path(path).exists():
             download_file(path, HierarchicalData.source_url, decompress=True)